/*****************************************************************************
 *  Memory Test and Utility Programs
 *
 *  FILE: mem.c
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential 
 * proprietary information distributed solely pursuant to a confidentiality 
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2004-2005 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <common.h>
#include <mem.h>
#include <util.h>


#undef CMD_MEM_DEBUG

#ifdef	CMD_MEM_DEBUG
#define	PRINTF(fmt,args...)	printf (fmt ,##args)
#else
#define PRINTF(fmt,args...)
#endif

#define DISP_LINE_LEN     16

/* maximum tolerated error for memory test */
#define MAX_TOLERATED_ERROR  3

#define MODULO_TEST_ARRAY_SIZE  71
static ulong primedata[MODULO_TEST_ARRAY_SIZE] = 
{   0x00000000, 0xffffffff, 0x11111111, 0xeeeeeeee,
    0x22222222, 0xdddddddd, 0x44444444, 0xbbbbbbbb,
    0x88888888, 0x77777777, 0x33333333, 0xcccccccc,
    0x55555555, 0xaaaaaaaa, 0x66666666, 0x99999999,

    0x0f0f0f0f, 0xf0f0f0f0, 0x1e1e1e1e, 0xe1e1e1e1,
    0x2d2d2d2d, 0xd2d2d2d2, 0x4b4b4b4b, 0xb4b4b4b4,
    0x87878787, 0x78787878, 0x3c3c3c3c, 0xc3c3c3c3,
    0x5a5a5a5a, 0xa5a5a5a5, 0x69696969, 0x96969696,

    0x00ff00ff, 0xff00ff00, 0x11ee11ee, 0xee11ee11,
    0x22dd22dd, 0xdd22dd22, 0x44bb44bb, 0xbb44bb44,
    0x88778877, 0x77887788, 0x33cc33cc, 0xcc33cc33,
    0x55aa55aa, 0xaa55aa55, 0x66996699, 0x99669966,

    0x0ff00ff0, 0xf00ff00f, 0x1ee11ee1, 0xe11ee11e,
    0x2dd22dd2, 0xd22dd22d, 0x4bb44bb4, 0xb44bb44b,
    0x87788778, 0x78877887, 0x3cc33cc3, 0xc33cc33c,
    0x5aa55aa5, 0xa55aa55a, 0x69966996, 0x96699669,

    0x01234567, 0x76543210, 0x89abcdef, 0xfedcba98,
    0xbabeface, 0xfacebabe, 0xdeadbeef
};

static char memtest_print_enabled = 1;


void mem_dump( unsigned long start, unsigned long length, char size )
{
    ulong addr;
    ulong i, nbytes, linebytes;
    u_char *cp;
    int count = 0;
    int num_char;

    /* Print the lines.
     *
     * We buffer all read data, so we can make sure data is read only
     * once, and all accesses are with the specified bus width.
     */
    addr = start;
    nbytes = length; /*length * size;*/
    while (nbytes > 0) {
       char	linebuf[DISP_LINE_LEN];
       uint	*uip = (uint   *)linebuf;
       ushort	*usp = (ushort *)linebuf;
       u_char	*ucp = (u_char *)linebuf;

       printf("%08lx:", addr);
       linebytes = (nbytes>DISP_LINE_LEN)?DISP_LINE_LEN:nbytes;
       for (i=0; i<linebytes; i+= size) {
          if (size == 4) {
             printf(" %08x", (*uip++ = *((uint *)addr)));
             count += 4;
          } else if (size == 2) {
             printf(" %04x", (*usp++ = *((ushort *)addr)));
             count += 2;
          } else {
             printf(" %02x", (*ucp++ = *((u_char *)addr)));
             count += 1;
          }
          addr += size;
       }
       printf("    ");
       cp = linebuf;
       for (i=0; i<linebytes; i++) {
          if ((*cp < 0x20) || (*cp > 0x7e))
             printf(".");
          else
             printf("%c", *cp);
          cp++;
       }
       printf("\n");
       nbytes -= linebytes;
       if((count%0x100) == 0 && nbytes > 0) {
          num_char = print_pause();
          if (num_char == -1) {
             /* ctrl-C is pressed */
             break;
          }
          printf("\n");
          count = 0;
       }
    }

    return;
}

void mem_fill( ulong addr, ulong length, char size, ulong writeval )
{
    while (length-- > 0) {
       if (size == 4)
          *((ulong  *)addr) = (ulong )writeval;
       else if (size == 2)
          *((ushort *)addr) = (ushort)writeval;
       else
          *((u_char *)addr) = (u_char)writeval;
       addr += size;
    }

    return;
}

int mem_test ( ulong s_addr, ulong e_addr, int continuous )
{
    vu_long	*addr, *start, *end;
    ulong	val;
    ulong	readback;

    vu_long	addr_mask;
    vu_long	offset;
    vu_long	test_offset;
    vu_long	pattern;
    vu_long	temp;
    vu_long	anti_pattern;
    vu_long	num_words;
    vu_long *dummy = NULL;
    int	j;
    int iterations = 1;

    static const ulong bitpattern[] = {
       0x00000001,	/* single bit */
       0x00000003,	/* two adjacent bits */
       0x00000007,	/* three adjacent bits */
       0x0000000F,	/* four adjacent bits */
       0x00000005,	/* two non-adjacent bits */
       0x00000015,	/* three non-adjacent bits */
       0x00000055,	/* four non-adjacent bits */
       0xaaaaaaaa,	/* alternating 1/0 */
    };
    
    start = (ulong *)s_addr;
    end = (ulong *)e_addr;

    if (continuous) {
       printf ("\nContinuous SDRAM Test on addr 0x%08x - 0x%08x\n",
               (uint)start, (uint)end - sizeof(ulong));
    }
    else {
       printf ("\nSingle-Pass SDRAM Test on addr 0x%08x - 0x%08x......",
               (uint)start, (uint)end - sizeof(ulong));
    }

    do {
       if (ctrlc()) {
          return 1;
       }

       if (continuous) {
          printf("    Iteration: %6d\r", iterations);
       }
       iterations++;

       /*
        * Data line test: write a pattern to the first
        * location, write the 1's complement to a 'parking'
        * address (changes the state of the data bus so a
        * floating bus doen't give a false OK), and then
        * read the value back. Note that we read it back
        * into a variable because the next time we read it,
        * it might be right (been there, tough to explain to
        * the quality guys why it prints a failure when the
        * "is" and "should be" are obviously the same in the
        * error message).
        *
        * Rather than exhaustively testing, we test some
        * patterns by shifting '1' bits through a field of
        * '0's and '0' bits through a field of '1's (i.e.
        * pattern and ~pattern).
        */
       addr = start;
       for (j = 0; j < sizeof(bitpattern)/sizeof(bitpattern[0]); j++) {
          val = bitpattern[j];
          for(; val != 0; val <<= 1) {
             *addr  = val;
             *dummy  = ~val; /* clear the test data off of the bus */
             readback = *addr;
             if(readback != val) {
                printf ("\nFAILURE (data line): "
                        "expected %08lx, actual %08lx\n",
                        val, readback);
                return -1;
             }
             *addr  = ~val;
             *dummy  = val;
             readback = *addr;
             if(readback != ~val) {
                printf ("\nFAILURE (data line): "
                        "Is %08lx, should be %08lx\n",
                        val, readback);
                return -1;
             }
          }
       }

       PRINTF("Address line test\n");
       /*
        * Based on code whose Original Author and Copyright
        * information follows: Copyright (c) 1998 by Michael
        * Barr. This software is placed into the public
        * domain and may be used for any purpose. However,
        * this notice must not be changed or removed and no
        * warranty is either expressed or implied by its
        * publication or distribution.
        */

       /*
        * Address line test
        *
        * Description: Test the address bus wiring in a
        *              memory region by performing a walking
        *              1's test on the relevant bits of the
        *              address and checking for aliasing.
        *              This test will find single-bit
        *              address failures such as stuck -high,
        *              stuck-low, and shorted pins. The base
        *              address and size of the region are
        *              selected by the caller.
        *
        * Notes:	For best results, the selected base
        *              address should have enough LSB 0's to
        *              guarantee single address bit changes.
        *              For example, to test a 64-Kbyte
        *              region, select a base address on a
        *              64-Kbyte boundary. Also, select the
        *              region size as a power-of-two if at
        *              all possible.
        *
        * Returns:     0 if the test succeeds, 1 if the test fails.
        *
        * ## NOTE ##	Be sure to specify start and end
        *              addresses such that addr_mask has
        *              lots of bits set. For example an
        *              address range of 01000000 02000000 is
        *              bad while a range of 01000000
        *              01ffffff is perfect.
        */
       addr_mask = ((ulong)end - (ulong)start)/sizeof(vu_long);
       pattern = (vu_long) 0xaaaaaaaa;
       anti_pattern = (vu_long) 0x55555555;

       PRINTF("%s:%d: addr mask = 0x%.8lx\n",
              __FUNCTION__, __LINE__,
              addr_mask);
       /*
        * Write the default pattern at each of the
        * power-of-two offsets.
        */
       for (offset = 1; (offset & addr_mask) != 0; offset <<= 1) {
          start[offset] = pattern;
       }

       PRINTF("Check for address bits stuck high\n");
       /*
        * Check for address bits stuck high.
        */
       test_offset = 0;
       start[test_offset] = anti_pattern;
       
       for (offset = 1; (offset & addr_mask) != 0; offset <<= 1) {
          temp = start[offset];
          if (temp != pattern) {
             printf ("\nFAILURE: Address bit stuck high @ 0x%.8lx:"
                     " expected 0x%.8lx, actual 0x%.8lx\n",
                     (ulong)&start[offset], pattern, temp);
             return -1;
          }
       }
       start[test_offset] = pattern;

       PRINTF("Check for address bits stuck low or shorted\n");
       /*
        * Check for addr bits stuck low or shorted.
        */
       for (test_offset = 1; (test_offset & addr_mask) != 0; test_offset <<= 1) {
          start[test_offset] = anti_pattern;

          for (offset = 1; (offset & addr_mask) != 0; offset <<= 1) {
             temp = start[offset];
             if ((temp != pattern) && (offset != test_offset)) {
                printf ("\nFAILURE: Address bit stuck low or shorted @"
                        " 0x%.8lx: expected 0x%.8lx, actual 0x%.8lx\n",
                        (ulong)&start[offset], pattern, temp);
                return -1;
             }
          }
          start[test_offset] = pattern;
       }

       if (ctrlc()) {
          return 1;
       }

       PRINTF("Test physical memory device integrity\n");
       /*
        * Description: Test the integrity of a physical
        *              memory device by performing an
        *              increment/decrement test over the
        *              entire region. In the process every
        *              storage bit in the device is tested
        *              as a zero and a one. The base address
        *              and the size of the region are
        *              selected by the caller.
        *
        * Returns:     0 if the test succeeds, 1 if the test fails.
        */
       num_words = ((ulong)end - (ulong)start)/sizeof(vu_long) + 1;

       /*
        * Fill memory with a known pattern.
        */
       for (pattern = 1, offset = 0; offset < num_words; pattern++, offset++) {
          start[offset] = pattern;
          dotik(20, 0);
       }
       
       if (ctrlc()) {
          return 1;
       }

       PRINTF("Check each location and invert it for the 2nd pass\n");
       /*
        * Check each location and invert it for the second pass.
        */
       for (pattern = 1, offset = 0; offset < num_words; pattern++, offset++) {
          temp = start[offset];
          if (temp != pattern) {
             printf ("\nFAILURE (read/write) @ 0x%.8lx:"
                     " expected 0x%.8lx, actual 0x%.8lx)\n",
                     (ulong)&start[offset], pattern, temp);
             return -1;
          }

          anti_pattern = ~pattern;
          start[offset] = anti_pattern;
          dotik(30, 0);
       }

       if (ctrlc()) {
          return 1;
       }

       PRINTF("Check each location for the inverted pattern and zero it\n");
       /*
        * Check each location for the inverted pattern and zero it.
        */
       for (pattern = 1, offset = 0; offset < num_words; pattern++, offset++) {
          anti_pattern = ~pattern;
          temp = start[offset];
          if (temp != anti_pattern) {
             printf ("\nFAILURE (read/write): @ 0x%.8lx:"
                     " expected 0x%.8lx, actual 0x%.8lx)\n",
                     (ulong)&start[offset], anti_pattern, temp);
             return -1;
          }
          start[offset] = 0;
          dotik(30, 0);
       }
    } while (continuous);

    return 0;
}

int memtest_walking_ones( uint addr, uint size )
{
    int i, j, err = 0, temp;
    uint w, r;
    volatile uint *p;

    PRINTF("%s: addr = 0x%08x size = 0x%x\n", __FUNCTION__, addr, size);
    printf("\nWalking Ones on addr 0x%08x - 0x%08x......", addr, addr+size-4);

    for (p = (volatile uint *)addr, i=0; i < size ; i+=sizeof(uint), p++) {
       w = 1;
       for (j = 0; j < 32; j++) {
          *p = w;
          temp = 0;
          r = *p;

          if (r != w) {
             if (err == 0) printf("\bFAILED\n\n");
             else printf("\r");
             err++;
             printf("[ERROR: addr=%p read=%08x expected=%08x]\n", p,r,w);
             if (err >= MAX_TOLERATED_ERROR) {
                return -err;
             }
             printf("Continuing test......");
          }
          w <<= 1;
       }

       if (ctrlc()) {
          return 1;
       }

       dotik(100, 0);
    }

    PRINTF("%s: last addr = 0x%p\n", __FUNCTION__, p);

    return err;
}

int memtest_address( uint addr, uint size )
{
    int i, err = 0;
    uint r;
    volatile uint *p;

    PRINTF("%s: addr = 0x%08x size = 0x%x\n", __FUNCTION__, addr, size);
    printf("\nAddress Test on addr 0x%08x - 0x%08x......", addr, addr+size-4);

    /* store the address in each address */
    for (p = (volatile uint *)addr, i=0; i < size; i += sizeof(uint), p++) {
       *p = (unsigned int)p;
       dotik(10,0);
    }

    if (ctrlc()) {
       return 1;
    }

    /* check that each address contains its address */
    for (p = (volatile uint *)addr, i=0; i < size; i += sizeof(uint), p++) {
       r = *p;
       if (r != (unsigned int)p) {
           if (err == 0) printf("\bFAILED\n\n");
           else printf("\r");
           err++;
           printf("[ERROR: addr=%p read=%08x expected=%p]\n", p,r,p);
           if (err >= MAX_TOLERATED_ERROR) {
              return -err;
           }
           printf("Continuing test......");
       }
       dotik(10,0);
    }

    PRINTF("%s: last addr = 0x%p\n", __FUNCTION__, p);

    return err;
}

int memtest_fill( uint addr, uint size, uint pattern )
{
    int i, err = 0;
    uint r;
    volatile uint *p;

    if( memtest_print_enabled ) {
        printf("\nFill 0x%08x Test on addr 0x%08x - 0x%08x......",
               pattern, addr, addr+size-4);
    }

    /* store the pattern in each address */
    for (p = (volatile uint *)addr, i=0; i < size; i += sizeof(uint), p++) {
       *p = (unsigned int)pattern;
       if (ctrlc()) {
           return 1;
       }
       if( memtest_print_enabled ) {
           dotik(10,0);
       }
    }

    /* check that each address contains the pattern */
    for (p = (volatile uint *)addr, i=0; i < size; i += sizeof(uint), p++) {
       r = *p;
       if (r != pattern) {
           if (err == 0) printf("\bFAILED\n\n");
           else printf("\r");
           err++;
           printf("[ERROR: addr=%p read=%08x expected=%08x]\n", p, r, pattern);
           if (err >= MAX_TOLERATED_ERROR) {
              return -err;
           }
           printf("Continuing test......");
       }
       if (ctrlc()) {
           return 1;
       }
       if( memtest_print_enabled ) {
           dotik(10,0);
       }
    }

    PRINTF("%s: last addr = 0x%p\n", __FUNCTION__, p);

    return err;
}

int memtest_modfill( uint addr, uint size )
{
    int i, j, err = 0;
    uint r;
    volatile uint *p;

    printf("\nFill Modulo Test on addr 0x%08x - 0x%08x......", addr,
           addr+size-4);

    /* store the pattern in each address */
    j = 0;
    for (p = (volatile uint *)addr, i=0, j=0; i < size; i += sizeof(uint),
         p++, j++) {
       *p = primedata[j % MODULO_TEST_ARRAY_SIZE];
       dotik(10,0);
    }

    if (ctrlc()) {
       return 1;
    }

    /* check that each address contains the pattern */
    for (p = (volatile uint *)addr, i=0, j=0; i < size; i += sizeof(uint),
         p++, j++) {
       r = *p;
       if (r != primedata[j % MODULO_TEST_ARRAY_SIZE]) {
           if (err == 0) printf("\bFAILED\n\n");
           else printf("\r");
           err++;
           printf("[ERROR: addr=%p read=%08x expected=%08lx]\n", p,r,
                  primedata[j % MODULO_TEST_ARRAY_SIZE]);
           if (err >= MAX_TOLERATED_ERROR) {
              return -err;
           }
           printf("Continuing test......");
       }
       dotik(10,0);
    }

    PRINTF("%s: last addr = 0x%p\n", __FUNCTION__, p);

    return err;
}

void memtest_config_print( char enabled )
{
    memtest_print_enabled = enabled;
    return;
}
