/*****************************************************************************
 *  Manufacturing Test
 *
 *  FILE: mfg_test.c
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential 
 * proprietary information distributed solely pursuant to a confidentiality 
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2004-2005 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <common.h>
#include <command.h>
#include <flash.h>
#include <i2c.h>
#include <net.h>
#include <rtc.h>
#include <post.h>
#include <mfg_test.h>
#include <mem.h>
#include <util.h>
#include <ad9888.h>
#include <pca9543a.h>
#include <x1226.h>
#include <lcd_raritan.h>
#include <led.h>
#include <speaker.h>
#include <flash_partition.h>
#include <ether_menu.h>
#include <rtc_menu.h>

#define SDRAM_TEST                  0x00000001
#define FLASH_TEST                  0x00000002
#define LED_TEST                    0x00000004
#define LAN_TEST                    0x00000008
#define LCD_TEST                    0x00000010
#define RTC_TEST                    0x00000020
#define BUZZER_TEST                 0x00000040
#define POWER_TEST                  0x00000080
#define ALL_TESTS                   (SDRAM_TEST | FLASH_TEST | LED_TEST | \
                                     LAN_TEST | LCD_TEST | RTC_TEST | \
                                     BUZZER_TEST | POWER_TEST)
#define MEMTEST_FILL_5              0x55555555
#define MEMTEST_FILL_A              0xaaaaaaaa
#define FILL_5_ADDR                 0x05000000
#define FILL_A_ADDR                 0x05200000
#define FLASH_IMAGE_MIRROR_ADDR     0x01000000
#define MAX_TOLERATED_ERROR         1

/* for injecting errors */
#undef INJECT_ERRORS

/* defined in board\raritan\common\strataflash.c */
extern void flash_set_enable_print( char enable );

static unsigned long cfg_test = ALL_TESTS;
static unsigned long flash_testaddr[8] = { 0xFC020000,   /* JFFS2   */
                                           0xFC040000,   /* JFFS2   */
                                           0xFC080000,   /* JFFS2   */
                                           0xFC100000,   /* JFFS2   */
                                           0xFC200000,   /* JFFS2   */
                                           0xFC400000,   /* JFFS2   */
                                           0xFC800000,   /* JFFS2   */
                                           0xFD000000 }; /* JFFS2   */


static void print_mfg_test( void )
{
    printf("\n+--------------------------------------------------------+");
    printf("\n+                   MANUFACTURING TEST                   +");
    printf("\n+--------------------------------------------------------+\n\n");
    return;
}

static int memory_test( void )
{
    int result;
    extern int  raritan_memory_post_test( int flags );
    extern void raritan_in_mfg_test( int answer );
    extern mfg_test_memstage_t raritan_get_memory_stage( void );

    /* put the POST memory test in MFG TEST mode */
    raritan_in_mfg_test(1);

    printf("Testing SDRAM..............................");
    result = raritan_memory_post_test(POST_SLOWTEST);
    if( result != 0 ) {
        /* get the memory test stage */
        switch( raritan_get_memory_stage() ) {
            case MEMORY_TEST_DATA:
                result = MFGTST_SDRAM_FAIL_DATA_LINES;
                break;
            case MEMORY_TEST_ADDR:
                result = MFGTST_SDRAM_FAIL_ADDR_LINES;
                break;
            case MEMORY_TEST_00_PATTERN:
                result = MFGTST_SDRAM_FAIL_00_PATTERN;
                break;
            case MEMORY_TEST_FF_PATTERN:
                result = MFGTST_SDRAM_FAIL_FF_PATTERN;
                break;
            case MEMORY_TEST_55_PATTERN:
                result = MFGTST_SDRAM_FAIL_55_PATTERN;
                break;
            case MEMORY_TEST_AA_PATTERN:
                result = MFGTST_SDRAM_FAIL_AA_PATTERN;
                break;
            case MEMORY_TEST_BIT_FLIP_PATTERN:
                result = MFGTST_SDRAM_FAIL_BIT_FLIP_PATTERN;
                break;
            case MEMORY_TEST_ADDR_OFFSET_PATTERN:
                result = MFGTST_SDRAM_FAIL_ADDR_OFFSET_PATTERN;
                break;
            case MEMORY_TEST_ADDR_OFFSET_COMP_PATTERN:
                result = MFGTST_SDRAM_FAIL_ADDR_OFFSET_COMP_PATTERN;
                break;
            default:
                result = MFGTST_SDRAM_FAIL_DATA_LINES;
                break;
        }
    }
    else {
        printf("passed.\n");
        result = MFGTST_PASS;
    }

    /* put the POST memory test in default mode */
    raritan_in_mfg_test(0);

    return result;
}

static mfg_test_result_t
flashtest_fill( uint addr, uint size, uint pattern )
{
    int i, err = 0;
    uint r;
    volatile uint *p;

    /* store the pattern in each address */
    if(pattern == MEMTEST_FILL_5) {
        err = flash_write((unsigned char *)FILL_5_ADDR, addr, size);
    }
    else if(pattern == MEMTEST_FILL_A) {
        err = flash_write((unsigned char *)FILL_A_ADDR, addr, size);
    }
    else {
        printf("unspecified pattern");
    }

    if(err != ERR_OK) {
        printf("FAILED.\n\n");
        printf("ERROR: ");
        flash_perror(err);
        printf("\n");
        return MFGTST_FLASH_FAIL_WRITE;
    }

    /* check that each address contains the pattern */
    for (p = (volatile uint *)addr, i=0; i < size; i += sizeof(uint), p++) {
        r = *p;
        if (r != pattern) {
            if (err == 0) printf("FAILED\n\n");
            else printf("\r");
            err++;
            printf("[ERROR: addr=%p read=%08x expected=%08x]\n\n",
                   p, r, pattern);
            if (err >= MAX_TOLERATED_ERROR) {
                return MFGTST_FLASH_FAIL_FILL_PATTERN;
            }
        }

        if (ctrlc()) {
            return MFGTST_HALTED;
        }
    }

    if( err ) {
        return MFGTST_FLASH_FAIL_FILL_PATTERN;
    }
    else {
        /* test passed */
        return MFGTST_PASS;
    }
}

static mfg_test_result_t
flashtest_address( flash_info_t *pinfo, int bank, int sector, uint size )
{
    int i, err = 0;
    ulong rdata, wdata;
    unsigned long addr;
    volatile uint *p;

    flash_protect_off(bank, sector, sector);

    /* retrieve the address */
    addr = pinfo->start[sector];

    /* erase flash sector */
    if(flash_erase(pinfo, sector, sector) != 0) {
        printf("FAILED\n\n");
        printf("ERROR: Failed to erase flash sector %d.\n\n", sector);
        return MFGTST_FLASH_FAIL_ERASE;
    }

    /* verify that sector is erased */
    for(i = 0; i < size; i += sizeof(uint)) {
        rdata = *(unsigned long *)(addr + i);
        if( rdata != 0xFFFFFFFF ) {
            printf("FAILED!\n\n");
            printf("ERROR: Sector is not fully erased.\n");
            printf("       Address 0x%08lx still contains 0x%08lx\n\n",
                   addr+i, rdata);
            return MFGTST_FLASH_FAIL_ERASE_VERIFY;
        }
    }

    /* store the address in each address */
    for(i=0; i < size; i += sizeof(uint)) {
        wdata = (ulong)(addr + i);
        err = flash_write((unsigned char *)&wdata, wdata, sizeof(ulong));
        if(err != ERR_OK) {
            printf("FAILED.\n\n");
            printf("ERROR: ");
            flash_perror(err);
            printf("\n");
            return MFGTST_FLASH_FAIL_WRITE;
        }

        if (ctrlc()) {
            return MFGTST_HALTED;
        }
    }

    /* check that each address contains its address */
    for(p = (volatile uint *)addr, i=0; i < size; i += sizeof(uint), p++) {
        rdata = *p;
        if (rdata != (unsigned int)p) {
            if (err == 0) printf("FAILED\n\n");
            else printf("\r");
            err++;
            printf("[ERROR: addr=%p read=%08lx expected=%p]\n\n", p, rdata, p);
            if (err >= MAX_TOLERATED_ERROR) {
                return MFGTST_FLASH_FAIL_ADDR;
            }
        }

        if (ctrlc()) {
            return MFGTST_HALTED;
        }
    }

    if( err ) {
        return MFGTST_FLASH_FAIL_ADDR;
    }
    else {
        /* test passed */
        return MFGTST_PASS;
    }
}

static mfg_test_result_t
flashtest_address_a7_a14( flash_info_t *pinfo, int bank )
{
    int sector;
    unsigned long dst, offset;
    volatile uint *p;
    uint rdata;
    int id;
    int rc;

    for( id = 0; id < 8; id++ ) {
        dst    = flash_testaddr[id] + (0x02000000 * bank);
        offset = dst - (0x02000000 * bank) - CFG_FLASH_BASE;
        sector = (offset/FLASH_SECTOR_SIZE) + 3;

        flash_protect_off(bank, sector, sector);

        if( flash_erase(pinfo, sector, sector) != 0 ) {
            printf("FAILED\n\n");
            printf("ERROR: Failed to erase flash sector %d.\n\n", sector);
            return MFGTST_FLASH_FAIL_ERASE;
        }

        rc = flash_write((uchar *)&dst, dst, sizeof(unsigned long));
        if (rc != 0) {
            printf("FAILED.\n\n");
            printf("ERROR: ");
            flash_perror(rc);
            printf("\n");
            return MFGTST_FLASH_FAIL_WRITE;
        }
    }

    /* verify the contents */
    for( id = 0; id < 8; id++ ) {
        p = (volatile uint *)flash_testaddr[id];
        rdata = *p;
        if (rdata != (unsigned int)p) {
            printf("FAILED\n\n");
            printf("[ERROR: addr=%p read=%08x expected=%p]\n\n", p, rdata, p);
            return MFGTST_FLASH_FAIL_ADDR;
        }
    }

    return MFGTST_PASS;
}

static mfg_test_result_t
flashtest_walking_ones( flash_info_t *pinfo, int bank, int sector, int sector_size )
{
    int j, err = 0, temp, offset;
    unsigned long wdata, rdata;
    unsigned long addr;
    static int first_check = 1;
    volatile unsigned long *p;

    flash_protect_off(bank, sector, sector);

    addr = pinfo->start[sector];

    p = (volatile unsigned long *)addr;
    wdata = 1;
    /* 16-bit data lines */
    for(j = 0; j < 16; j++) {
        /* erase flash sector */
        if(flash_erase(pinfo, sector, sector) != 0) {
            printf("FAILED\n\n");
            printf("ERROR: Failed to erase flash sector %d.\n\n", sector);
            return MFGTST_FLASH_FAIL_ERASE;
        }

        /* verify erased sector */
        if( !first_check ) {
            /* just check the specific test address */
            rdata = *(volatile unsigned long *)(addr);
            if( rdata != 0xFFFFFFFF ) {
                printf("FAILED!\n\n");
                printf("ERROR: Sector is not fully erased.\n");
                printf("       Address 0x%08lx still contains 0x%08lx\n\n",
                       addr, rdata);
                return MFGTST_FLASH_FAIL_ERASE_VERIFY;
            }
        }
        else {
            /* check the whole sector */
            for(offset = 0; offset < sector_size; offset += 4) {
                rdata = *(volatile unsigned long *)(addr + offset);
                if( rdata != 0xFFFFFFFF ) {
                    printf("FAILED!\n\n");
                    printf("ERROR: Sector is not fully erased.\n");
                    printf("       Address 0x%08lx still contains 0x%08lx\n\n",
                           addr+offset, rdata);
                    return MFGTST_FLASH_FAIL_ERASE_VERIFY;
                }
            }
            first_check = 0;
        }

        /* write data to flash */
        err = flash_write((unsigned char *)&wdata, addr, sizeof(ulong));
        if(err != ERR_OK) {
            printf("FAILED.\n\n");
            printf("ERROR: ");
            flash_perror(err);
            printf("\n");
            return MFGTST_FLASH_FAIL_WRITE;
        }

        /* in case data lines are floating, write some pattern to some addr */
        temp = 0xbabeface;

        /* read back the flash data */
        rdata = *p;

#ifdef INJECT_ERRORS
        if( wdata == 0x00001000 ) {
            rdata |= 0x00002000;
        }
#endif

        /* verify contents */
        if (rdata != wdata) {
            if (err == 0) printf("FAILED\n\n");
            else printf("\r");
            err++;
            printf("[ERROR: addr=%p read=%08lx expected=%08lx]\n\n",
                   p, rdata, wdata);
            if (err >= MAX_TOLERATED_ERROR) {
                return MFGTST_FLASH_FAIL_WALK_ONES;
            }
        }

        wdata <<= 1;
    }

    if( err ) {
        return MFGTST_FLASH_FAIL_WALK_ONES;
    }
    else {
        /* test passed */
        return MFGTST_PASS;
    }
}

static mfg_test_result_t
flash_recover_image( int bank, int recover_all )
{
    int sector;
    int size;
    unsigned int addr;
    unsigned long dst, offset;
    int id;
    int rc;

    /* info for FLASH chips */
    extern flash_info_t flash_info[];

    flash_info_t *pinfo = &flash_info[bank];

    /**************************/
    /* recover sector 0 image */
    /**************************/
    for( sector = 0; sector <= 4; sector+= 4 ) {
        dst    = pinfo->start[sector];
        offset = dst - (0x02000000 * bank) - CFG_FLASH_BASE;
        size   = pinfo->start[sector+1] - pinfo->start[sector];
        addr   = FLASH_IMAGE_MIRROR_ADDR + offset;

        if( flash_erase(pinfo, sector, sector) != 0 ) {
            printf("FAILED\n\n");
            printf("ERROR: Failed to erase flash sector %d.\n\n", sector);
            return MFGTST_FLASH_FAIL_ERASE;
        }

        rc = flash_write((uchar *)addr, dst, size);
        if (rc != 0) {
            printf("FAILED.\n\n");
            printf("ERROR: ");
            flash_perror(rc);
            printf("\n");
            return MFGTST_FLASH_FAIL_JFFS2_RECOVERY;
        }
    }
    
    if( !recover_all ) {
        /* test failed before doing A7-A14 address test */
        return MFGTST_PASS;
    }

    /*******************************************************/
    /* recover sectors 5, 7, 11, 19, 35, 67 and 131 images */
    /*******************************************************/
    for( id = 1; id < 8; id++ ) {
        dst    = flash_testaddr[id] + (0x02000000 * bank);
        offset = dst - (0x02000000 * bank) - CFG_FLASH_BASE;
        sector = (offset/FLASH_SECTOR_SIZE) + 3;
        addr   = FLASH_IMAGE_MIRROR_ADDR + offset;

        if( flash_erase(pinfo, sector, sector) != 0 ) {
            printf("FAILED\n\n");
            printf("ERROR: Failed to erase flash sector %d.\n\n", sector);
            return MFGTST_FLASH_FAIL_ERASE;
        }

        rc = flash_write((uchar *)addr, dst, size);
        if (rc != 0) {
            printf("FAILED.\n\n");
            printf("ERROR: ");
            flash_perror(rc);
            printf("\n");
            return MFGTST_FLASH_FAIL_JFFS2_RECOVERY;
        }
    }

    return MFGTST_PASS;
}

static mfg_test_result_t
flash_test_cleanup( int bank, mfg_test_flashstage_t test_point, mfg_test_result_t error )
{
    mfg_test_result_t result = MFGTST_PASS;
    int recover_all_image;
    int need_img_recovery;

    /* enable memory test print statements */
    memtest_config_print(1);

    if( bank == 1 ) {
        /*
         * Bank 0 passed all tests but we hit a snag in bank 1.
         * Need to recover all of bank 0 sectors that got tested.
         */
        printf("Recovering KX2.0 flash bank0 image.........");
        result = flash_recover_image(0, 1);
        if( result == MFGTST_PASS ) {
            printf("passed.\n");
        }
    }

    switch( test_point ) {
        case FLASH_TEST_WALK_ONES:
        case FLASH_TEST_FILL_PATTERN:
        case FLASH_TEST_ADDR_A15_A31:
            need_img_recovery = 1;
            recover_all_image = 0;
            break;
        case FLASH_TEST_ADDR_A7_A14:
        case FLASH_TEST_COMPLETE:
            need_img_recovery = 1;
            recover_all_image = 1;
            break;
        default:
            need_img_recovery = 0;
            recover_all_image = 0;
            break;
    }

    if( need_img_recovery ) {
        printf("Recovering KX2.0 flash bank%d image.........", bank);
        result = flash_recover_image(bank, recover_all_image);
        if( result == MFGTST_PASS ) {
            printf("passed.\n");
        }
    }

    /* enable flash print statements */
    flash_set_enable_print(1);

    if( error == MFGTST_PASS ) {
        return( result );
    }
    else {
        /*
         * Return the first cause of failure, even if the recovery has its
         * own failure.
         */
        return( error );
    }
}

/*
 * Flash test consists of the following test:
 *  - Walking ones on address 0xFF000000 (sector 0) (D0 - D31)
 *  - Fill 55555555 on address 0xFF000000-0xFF01FFFF
 *  - Fill aaaaaaaa on address 0xFF000000-0xFF01FFFF
 *  - Address test on address 0xFF000000-0xFF01FFFF (A15 - A31)
 *  - Address test on addresses: 0xFE040000
 *                               0xFE080000
 *                               0xFE100000
 *                               0xFE200000
 *                               0xFE400000
 *                               0xFE800000
 *                               0xFF040000 (A8 - A14)
 *
 * NOTE: To be extra careful, this test does not touch the bootloader!
 */
static int flash_test( void )
{
    unsigned long bank, sector, size = 0;
    unsigned long offset, addr, val;
    int start_sector, last_sector;
    int retval, iter;
    mfg_test_result_t result;
    mfg_test_flashstage_t testpt;

    /* info for FLASH chips */
    extern flash_info_t flash_info[];

    /* disable flash print statements */
    flash_set_enable_print(0);

    /* disable memory test print statements */
    memtest_config_print(0);

    /* copy flash image to a pre-defined SDRAM location */
    testpt = FLASH_TEST_MIRROR;
    printf("Mirroring flash image to SDRAM address.....");
    for( bank = 0; bank < CFG_MAX_FLASH_BANKS; bank++ ) {
        size += flash_info[bank].size;
    }
    memcpy((void *)FLASH_IMAGE_MIRROR_ADDR, (void *)CFG_FLASH_BASE, size);
    retval = memcmp ((void *)FLASH_IMAGE_MIRROR_ADDR,
                     (void *)CFG_FLASH_BASE,
                     size);
    if(retval != 0) {
        int j;
        unsigned long *srcbuf, *dstbuf;

        printf("FAILED!\n\n");
        printf("ERROR: memcmp(0x%08x, 0x%08x, 0x%08lx) = %d\n",
               FLASH_IMAGE_MIRROR_ADDR, CFG_FLASH_BASE, size, retval);

        srcbuf = (unsigned long *)CFG_FLASH_BASE;
        dstbuf = (unsigned long *)FLASH_IMAGE_MIRROR_ADDR;

        for(j = 0; j < (size/sizeof(unsigned long)); j++) {
            if(*srcbuf != *dstbuf) {
                printf("       Flash Address 0x%p contains 0x%08lx\n",
                       srcbuf, *srcbuf);
                printf("       SDRAM Address 0x%p contains 0x%08lx\n",
                       dstbuf, *dstbuf);
                break;
            }
            ++srcbuf;
            ++dstbuf;
        }

        return( flash_test_cleanup(0, testpt, MFGTST_FLASH_MIRROR_FAIL) );
    }
    else {
        printf("passed.\n");
    }

    for( bank = 0; bank < CFG_MAX_FLASH_BANKS; bank++ ) {
        /*
         * Most of the test will use two sectors of JFFS2 partition 
         *(1st and 4th flash sector of each bank) to fully test the flash
         * device (data and some address lines).
         */
        sector = 0;
        size = flash_info[bank].start[sector+1] - flash_info[bank].start[sector];
        addr = flash_info[bank].start[sector];
        start_sector = last_sector = sector;

        /* test data lines by walking ones on a flash address */
        testpt = FLASH_TEST_WALK_ONES;
        printf("Testing flash Bank%d (walking ones).........", bank);
        result = flashtest_walking_ones(&flash_info[bank], bank, start_sector, size);
        if( result == MFGTST_PASS ) {
            printf("passed.\n");
        }
        else {
            return( flash_test_cleanup(bank, testpt, result) );
        }
    
        /* Fill 55 and aa pattern test */
        testpt = FLASH_TEST_FILL_PATTERN;
        for(iter = 0; iter < 2; iter++) {
            /*
             * store the pattern in SDRAM first so that we can just use flash_copy 
             * to write to the flash bank with the specific data.
             */
            if(iter) {
                printf("Testing flash B%d (fill aaaaaaaa pattern)...", bank);
                retval = memtest_fill(FILL_A_ADDR, size, MEMTEST_FILL_A);
            }
            else {
                printf("Testing flash B%d (fill 55555555 pattern)...", bank);
                retval = memtest_fill(FILL_5_ADDR, size, MEMTEST_FILL_5);
            }

            /* check the result */
            if(retval != 0) {
                if(retval == 1) {
                    printf("halted\n");
                    return( flash_test_cleanup(bank, testpt, MFGTST_HALTED) );
                }
                else {
                    printf("\nERROR: Failed to store the pattern to SDRAM.\n");
                    printf("       Pattern will be used to copy to flash ");
                    printf("later on.\n\n");
                    if(iter) {
                        result = MFGTST_SDRAM_FAIL_AA_PATTERN;
                    }
                    else {
                        result = MFGTST_SDRAM_FAIL_55_PATTERN;
                    }
                    return( flash_test_cleanup(bank, testpt, result) );
                }
            }

            /* erase flash sector to be tested */
            if(flash_erase(&flash_info[bank], start_sector, last_sector) != 0) {
                printf("FAILED\n\n");
                printf("ERROR: Failed to erase flash sector %d.\n\n",start_sector);
                return( flash_test_cleanup(bank, testpt, MFGTST_FLASH_FAIL_ERASE) );
            }

            /* verify that sector is erased */
            for(offset = 0; offset < size; offset += 4) {
                val = *(unsigned long *)(addr + offset);
                if( val != 0xFFFFFFFF ) {
                    printf("FAILED!\n\n");
                    printf("ERROR: Sector is not fully erased.\n");
                    printf("       Address 0x%08lx still contains 0x%08lx\n\n",
                           addr+offset, val);
                    return( flash_test_cleanup(bank, testpt, MFGTST_FLASH_FAIL_ERASE_VERIFY) );
                }
            }

            if(iter == 0) {
                result = flashtest_fill(addr, size, MEMTEST_FILL_5);
            }
            else {
                result = flashtest_fill(addr, size, MEMTEST_FILL_A);
            }

            /* check result */
            if(result == MFGTST_PASS) {
                printf("passed.\n");
            }
            else if(result == MFGTST_HALTED) {
                printf("halted\n");
                return( flash_test_cleanup(bank, testpt, result) );
            }
            else {
                /* test failed */
                return( flash_test_cleanup(bank, testpt, result) );
            }
        }

        /* address test (A15 - A31) */
        sector = 4;
        size = flash_info[bank].start[sector+1] - flash_info[bank].start[sector];
        addr = flash_info[bank].start[sector];

        testpt = FLASH_TEST_ADDR_A15_A31;
        printf("Testing flash B%d (address A15 - A31 test)..", bank);
        result = flashtest_address(&flash_info[bank], bank, sector, size);
        if(result == MFGTST_PASS) {
            printf("passed.\n");
        }
        else if(result == MFGTST_HALTED) {
            printf("halted\n");
            return( flash_test_cleanup(bank, testpt, result) );
        }
        else {
            /* test failed */
            return( flash_test_cleanup(bank, testpt, result) );
        }

        /* address test (A7 - A14) */
        testpt = FLASH_TEST_ADDR_A7_A14;
        printf("Testing flash B%d (address A7 - A14 test)...", bank);
        result = flashtest_address_a7_a14(&flash_info[bank], bank);
        if (result == MFGTST_PASS) {
            printf("passed.\n");
        }
        else {
            /* test failed */
            return( flash_test_cleanup(bank, testpt, result) );
        }
    }

    testpt = FLASH_TEST_COMPLETE;

    return( flash_test_cleanup(CFG_MAX_FLASH_BANKS - 1, testpt, MFGTST_PASS) );
}

static mfg_test_result_t
led_test( void )
{
    int remote_led[4] = {IP_USER1, IP_USER2, IP_USER3, IP_USER4};
    int nleds;
    int id;
    int test_case;
    int num_char;
    enum test{ LED_OFF = 0, LED_ON };
    mfg_test_result_t result = MFGTST_PASS;

    printf("Please verify LED ON/OFF mode visually (for this part only).\n");
    num_char = readline("Hit ENTER key to continue [CTRL-C to exit] ");
    if (num_char == -1) {
        /* ctrl-C is pressed */
        return( MFGTST_HALTED );
    }

    led_off(POWER_RED);
    led_off(POWER_BLUE);

    /* determine the correct number of remote user LEDs */
    nleds = 4;

    /* test the front-panel remote user LEDs */
    for( id = 0; id < nleds; id++) {
        for(test_case = LED_OFF; test_case <= LED_ON; test_case++) {
            printf("Turning %s Remote %d LED...................",
                   test_case?"on ":"off", id+1);
            if(test_case == LED_OFF)
                led_off(remote_led[id]);
            else
                led_on(remote_led[id]);
            printf("done\n");

            if (ctrlc()) {
                result = MFGTST_HALTED;
                goto led_test_cleanup;
            }

            udelay(1000000);
        }
    }

    /* test the front-panel LAN LEDs */

    /* test Power LED */
    for( id = POWER_RED; id <= POWER_BLUE; id++ ) {
        for(test_case = LED_ON; test_case >= LED_OFF; test_case--) {
            printf("Turning %s Power %s LED.................",
                   test_case?"on ":"off", (id == POWER_RED)?"RED ":"BLUE");
            if(test_case == LED_OFF)
                led_off(id);
            else
                led_on(id);
            printf("done\n");

            if (ctrlc()) {
                result = MFGTST_HALTED;
                goto led_test_cleanup;
            }

            udelay(1000000);
        }
    }

led_test_cleanup:
    led_on(POWER_BLUE);

    if( result == MFGTST_PASS ) {
        printf("LED Visual Test............................completed.\n");
    }

    return result;
}

static mfg_test_result_t
buzzer_test( void )
{
    int num_char;

    printf("Please verify buzzer ON/OFF mode audibly.\n");
    num_char = readline("Hit ENTER key to continue [CTRL-C to exit] ");
    if (num_char == -1) {
        /* ctrl-C is pressed */
        return( MFGTST_HALTED );
    }

    printf("Turning on buzzer..........................");
    printf("done\n");
    speaker_on(1);

    udelay(1000000);

    printf("Turning off buzzer.........................");
    printf("done\n");
    speaker_off();

    return MFGTST_PASS;
}

static int lan_test( int index )
{
    cmd_tbl_t *cmdtp;
    char tmp[10];
    char buf[128];
    char *argv[2];
    int num_chars, rc = MFGTST_PASS;
    u32 gpio0_or, index_save;

    /* read and save current GE selection */
    gpio0_or = in32(GPIO0_OR);
    index_save = LAN_GPIO6 & gpio0_or;

    if ( index == 1 ) {
	gpio0_or &= (~LAN_GPIO6);
    }
    else if ( index == 2 ) {
	gpio0_or |= LAN_GPIO6;
    }
    else {
        return MFGTST_LAN_NO_MAC_PORT;
    }

    /* pick the GE for testing */
    out32(GPIO0_OR, gpio0_or);
    udelay(3000000);

    printf("Verifying LAN settings.....................");

    /* verify that we have a valid MAC address */
    memset(buf, 0, 128);
    num_chars = getenv_r("ethaddr", buf, 128);
    if(num_chars <= 0) {
        printf("FAILED.\n\n");
        printf("ERROR: No valid MAC 1 address specified!\n");
        rc = MFGTST_LAN_NO_MAC_ADDR1;
	goto lan_test_cleanup;
    }

    /* verify that we have a valid IP address */
    memset(buf, 0, 128);
    num_chars = getenv_r("ipaddr", buf, 128);
    if(num_chars <= 0) {
        printf("FAILED.\n\n");
        printf("ERROR: No valid IP address specified!\n");
        rc = MFGTST_LAN_NO_IPADDR;
	goto lan_test_cleanup;
    }

    /* verify that we have a valid server IP address */
    memset(buf, 0, 128);
    num_chars = getenv_r("serverip", buf, 128);
    if(num_chars <= 0) {
        printf("FAILED.\n\n");
        printf("ERROR: No valid server IP address specified!\n");
        rc = MFGTST_LAN_NO_SERVERIP;
	goto lan_test_cleanup;
    }

    argv[0] = tmp;
    strcpy(argv[0], "ping");
    argv[1] = buf;
    if ((cmdtp = find_cmd(argv[0])) == NULL) {
        printf("FAILED.\n\n");
        printf("ERROR: No ping cmd!\n");
        rc = MFGTST_LAN_NO_PINGCMD;
	goto lan_test_cleanup;
    }

    /* necessary LAN settings are present */
    printf("passed.\n");

    printf("Trying to ping server from LAN %d...........", index);
    if((cmdtp->cmd) (cmdtp, 0, 2, argv) != 0) {
        rc = MFGTST_LAN_FAIL_PING;
	goto lan_test_cleanup;
    }

    printf("Testing tftp using LAN %d...................in-progress\n\n", index);
    /* passed parameter preventing user-input */
    if(eth_tftp_test(0) != 0) {
        printf("\n\nTftp test using LAN %d......................FAILED.\n", index);
        printf("\nERROR: Failed to TFTP file.\n");
        rc = MFGTST_LAN_FAIL_TFTP;
	goto lan_test_cleanup;
    }
    else {
        printf("\nTftp test using LAN %d......................passed.\n", index);
    }

lan_test_cleanup:
    /* restore the pre test GE selection */
    gpio0_or = in32(GPIO0_OR);
    if ( index_save ) {
	gpio0_or |= index_save;
	out32(GPIO0_OR, gpio0_or);
    }
    else {
	gpio0_or &= (~LAN_GPIO6);
	out32(GPIO0_OR, gpio0_or);
    }

    gpio0_or = in32(GPIO0_OR);
    return rc;
}


static int rtc_detect_and_set( void )
{
    struct rtc_time tm;
    int num_chars;

    printf("Detecting real-time clock..................");
    if (i2c_probe(CFG_I2C_RTC_ADDR) == 0) {
        printf("found\n");
    }
    else {
        printf("NOT FOUND.\n");
        return MFGTST_RTC_NOT_FOUND;
    }

    /* set the time */
    printf("Please set the correct time.");
    num_chars = get_rtc_args(&tm, 0);
    if (num_chars > 0) {
        rtc_set(&tm);
    }
    else {
        printf("\n************************************************");
        printf("\n* WARNING: Tester is bypassing RTC settings!!! *");
        printf("\n************************************************\n");
        return MFGTST_RTC_BYPASS_SET_TIME;
    }

    return MFGTST_PASS;
}

static int rtc_test( void )
{
    extern int rtc_post_test (int flags);

    printf("Testing real-time clock....................");
    if( rtc_post_test(POST_MANUAL) < 0 ) {
        return MFGTST_RTC_FAIL_OPERATION;
    }
    printf("passed.\n");

    return MFGTST_PASS;
}

static int lcd_test( int lcd_cnt )
{
    int chan;
    uchar i2c_addr[4] = {CFG_I2C_LCD0_ADDR, CFG_I2C_LCD1_ADDR,
                         CFG_I2C_LCD2_ADDR, CFG_I2C_LCD3_ADDR};

    for(chan = 0; chan < lcd_cnt; chan++) {
        printf("Detecting LCD controller %d.................", chan);

        switch( chan ) {
            case 0:
            case 1:
                pca9543a_select_channel(0);
                break;

            case 2:
            case 3:
                pca9543a_select_channel(1);
                break;

            default:
                break;
        }

        if (i2c_probe(i2c_addr[chan]) == 0) {
            printf("found.\n");
        }
        else {
            printf("NOT FOUND.\n");
            return MFGTST_LCD_NOT_FOUND;
        }

        /* test LCD access */
        printf("Testing LCD controller %d register access...", chan);
        if( ad9888_test(chan) == LCD_SUCCESS ) {
            printf("passed.\n");
        }
        else {
            return MFGTST_LCD_FAIL_REG_ACCESS;
        }
    }

    return MFGTST_PASS;
}

static void process_error( mfg_test_result_t error )
{
    printf("\nCause of Failure:");
    printf("\n-----------------\n");

    switch( error ) {
        case MFGTST_LCD_FAIL_REG_ACCESS:
            printf("LCD register access test failure.\n");
            break;
        case MFGTST_LCD_NOT_FOUND:
            printf("Unable to find LCD controller (SAA6713) device.\n");
            break;
        case MFGTST_RTC_FAIL_OPERATION:
            printf("RTC is not operating correctly.\n");
            break;
        case MFGTST_RTC_BYPASS_SET_TIME:
            printf("RTC settings was bypassed and tester halted test.\n");
            break;
        case MFGTST_RTC_FAIL_SET_TIME:
            printf("RTC is currently stopped.\n");
            printf("RTC must be set to continue test.\n");
            break;
        case MFGTST_RTC_NOT_FOUND:
            printf("Unable to find RTC (M41ST84) device.\n");
            break;
        case MFGTST_LAN_FAIL_TFTP:
        case MFGTST_LAN_FAIL_PING:
            switch( error ) {
                case MFGTST_LAN_FAIL_TFTP:
                    printf("Unable to tftp fg_plus_a.bin file from server.\n");
                    break;
                default:
                    printf("Unable to ping server.\n");
                    break;
            }
            printf("Please check:\n");
            printf(" 1. RJ45 cable connection\n");
            printf(" 2. server availability\n");
            printf(" 3. on-board MII PHY device\n");
            printf(" 4. MPC8270 FCC signals\n");
            if( error == MFGTST_LAN_FAIL_TFTP ) {
                printf(" 5. fg_plus_a.bin is present in server /tftpboot ");
                printf("directory\n");
            }
            break;
        case MFGTST_LAN_NO_PINGCMD:
            printf("Ping command is not supported.\n");
            break;
        case MFGTST_LAN_NO_SERVERIP:
            printf("No server IP address assigned.\n");
            break;
        case MFGTST_LAN_NO_IPADDR:
            printf("No IP address assigned.\n");
            break;
        case MFGTST_LAN_NO_MAC_PORT:
            printf("Invalid G Ethernet port selection.\n");
            break;
        case MFGTST_LAN_NO_MAC_ADDR1:
            printf("No MAC address assigned for ethernet 1.\n");
            break;
        case MFGTST_FLASH_FAIL_JFFS2_RECOVERY:
        case MFGTST_FLASH_FAIL_CONFIG2_RECOVERY:
        case MFGTST_FLASH_FAIL_CONFIG1_RECOVERY:
            printf("Failed to recover ");
            switch( error ) {
                case MFGTST_FLASH_FAIL_CONFIG2_RECOVERY:
                    printf("CONFIG2");
                    break;
                case MFGTST_FLASH_FAIL_CONFIG1_RECOVERY:
                    printf("CONFIG1");
                    break;
                default:
                    printf("JFFS2");
            }
            printf(" image properly.\n");
            printf("Use Update Images menu to load the image again.\n");
            break;
        case MFGTST_FLASH_FAIL_ADDR:
            printf("Flash Address Test failure.\n");
            break;
        case MFGTST_FLASH_FAIL_FILL_PATTERN:
            printf("Flash Fill Pattern Test failure.\n");
            break;
        case MFGTST_FLASH_FAIL_WALK_ONES:
            printf("Flash Walking Ones Test failure.\n");
            break;
        case MFGTST_FLASH_FAIL_WRITE:
            printf("Flash Programming (Write) failure.\n");
            break;
        case MFGTST_FLASH_FAIL_ERASE_VERIFY:
            printf("Flash did not get erased correctly.\n");
            break;
        case MFGTST_FLASH_FAIL_ERASE:
            printf("Flash Erase command failure.\n");
            break;
        case MFGTST_FLASH_MIRROR_FAIL:
            printf("Failed to copy 32MB flash image to SDRAM.\n");
            printf("Please verify SDRAM and Flash address and data lines.\n");
            break;
        case MFGTST_SDRAM_FAIL_ADDR_OFFSET_COMP_PATTERN:
            printf("SDRAM address pattern offset complement test failure.\n");
            break;
        case MFGTST_SDRAM_FAIL_ADDR_OFFSET_PATTERN:
            printf("SDRAM address pattern offset test failure.\n");
            break;
        case MFGTST_SDRAM_FAIL_BIT_FLIP_PATTERN:
            printf("SDRAM bit-flip pattern test failure.\n");
            break;
        case MFGTST_SDRAM_FAIL_AA_PATTERN:
            printf("SDRAM pattern aaaaaaaa test failure.\n");
            break;
        case MFGTST_SDRAM_FAIL_55_PATTERN:
            printf("SDRAM pattern 55555555 test failure.\n");
            break;
        case MFGTST_SDRAM_FAIL_FF_PATTERN:
            printf("SDRAM pattern ffffffff test failure.\n");
            break;
        case MFGTST_SDRAM_FAIL_00_PATTERN:
            printf("SDRAM pattern 00000000 test failure.\n");
            break;
        case MFGTST_SDRAM_FAIL_ADDR_LINES:
            printf("SDRAM address test failure.\n");
            printf("Please verify SDRAM address lines.\n");
            break;
        case MFGTST_SDRAM_FAIL_DATA_LINES:
            printf("SDRAM data test failure.\n");
            printf("Please verify SDRAM data lines.\n");
            break;
        default:
            break;
    }

    return;
}

static int mfg_test_cleanup( mfg_test_result_t result )
{
    struct rtc_time tm;

    rtc_get(&tm);

    if( result == MFGTST_PASS ) {
        /* all tests passed */
        printf("\nPASSED all tests.\n");
        printf("\nManufacturing Test End Time: "
               "%4d-%02d-%02d  %2d:%02d:%02d UTC\n",
               tm.tm_year, tm.tm_mon, tm.tm_mday,
               tm.tm_hour, tm.tm_min, tm.tm_sec);
        printf("\nPower-down the system and power it back up to verify system "
               "retains the time.\n");
    }
    else if( result == MFGTST_HALTED ) {
        printf("\nTEST WAS HALTED! (%4d-%02d-%02d  %2d:%02d:%02d UTC)\n",
               tm.tm_year, tm.tm_mon, tm.tm_mday,
               tm.tm_hour, tm.tm_min, tm.tm_sec);
    }
    else {
        process_error(result);
        printf("\nKX2.0 FAILED!!! (%4d-%02d-%02d  %2d:%02d:%02d UTC)\n",
               tm.tm_year, tm.tm_mon, tm.tm_mday,
               tm.tm_hour, tm.tm_min, tm.tm_sec);
    }

    return result;
}

int mfg_test( void )
{
    mfg_test_result_t result = 0;
    struct rtc_time tm;

    print_mfg_test();

    /* detect RTC and set the time */
    if( cfg_test & RTC_TEST ) {
        result = rtc_detect_and_set();
        if( result != MFGTST_PASS ) {
            if( result == MFGTST_RTC_BYPASS_SET_TIME ) {
                int num_chars;
                char tmp;

                /* query the user to proceed or not */
                num_chars = readline("\nDo you want to continue test (y/n): ");
                if (num_chars > 0) {
                    strncpy(&tmp, console_buffer, 1);
                    if((tmp != 'y') && (tmp != 'Y')) {
                        return( mfg_test_cleanup(result) );
                    }
                }
                else {
                    /* user change mind and hit Ctrl-C */
                    return( mfg_test_cleanup(result) );
                }
            }
            else {
                return( mfg_test_cleanup(result) );
            }
        }
    }

    rtc_get(&tm);
    printf("\nManufacturing Test Start Time: %4d-%02d-%02d  %2d:%02d:%02d UTC\n\n",
           tm.tm_year, tm.tm_mon, tm.tm_mday,
           tm.tm_hour, tm.tm_min, tm.tm_sec);

    /* test front-panel LEDs */
    if( cfg_test & LED_TEST ) {
        result = led_test();
        if( result != MFGTST_PASS ) {
            return( mfg_test_cleanup(result) );
        }
    }

    /* test buzzer */
    if( cfg_test & BUZZER_TEST ) {
        result = buzzer_test();
        if( result != MFGTST_PASS ) {
            return( mfg_test_cleanup(result) );
        }
    }

    /* test power-supply */

    /* test SDRAM */
    if( cfg_test & SDRAM_TEST ) {
        result = memory_test();
        if( result != MFGTST_PASS ) {
            return( mfg_test_cleanup(result) );
        }
    }

    /* test network */
    if( cfg_test & LAN_TEST ) {
        result = lan_test(1);
        if( result != MFGTST_PASS ) {
            return( mfg_test_cleanup(result) );
        }
        result = lan_test(2);
        if( result != MFGTST_PASS ) {
            return( mfg_test_cleanup(result) );
        }
    }

    /* test flash device */
    if( cfg_test & FLASH_TEST ) {
        result = flash_test();
        if( result != MFGTST_PASS ) {
            return( mfg_test_cleanup(result) );
        }
    }

    /* quick verification of LCD controllers */
    if( cfg_test & LCD_TEST ) {
        result = lcd_test( lcd_get_ndev() );
        if( result != MFGTST_PASS ) {
            return( mfg_test_cleanup(result) );
        }
    }

    /* test RTC */
    if( cfg_test & RTC_TEST ) {
        result = rtc_test();
        if( result != MFGTST_PASS ) {
            return( mfg_test_cleanup(result) );
        }
    }

    return( mfg_test_cleanup(MFGTST_PASS) );
}

static int mfg_test_check_user_input( ulong opt )
{
    if (opt > 1) {
       printf("\nERROR: Invalid option specified %d\n", (uint)opt);
       return -1;
    }
    return 0;
}

static int mfg_test_parse_input( unsigned long test )
{
    int nchars = -1;
    unsigned long opt;

    switch(test) {
        case SDRAM_TEST:
            nchars = print_prompt("SDRAM : ", &opt, 10);
            break;
        case FLASH_TEST:
            nchars = print_prompt("FLASH : ", &opt, 10);
            break;
        case LED_TEST:
            nchars = print_prompt("LED   : ", &opt, 10);
            break;
        case LAN_TEST:
            nchars = print_prompt("LAN   : ", &opt, 10);
            break;
        case LCD_TEST:
            nchars = print_prompt("LCD   : ", &opt, 10);
            break;
        case RTC_TEST:
            nchars = print_prompt("RTC   : ", &opt, 10);
            break;
        case BUZZER_TEST:
            nchars = print_prompt("BUZZER: ", &opt, 10);
            break;
        case POWER_TEST:
            nchars = print_prompt("POWER : ", &opt, 10);
            break;
        default:
            return -1;
    }

    /* parse the input and return appropriate result */
    if (nchars > 0 && mfg_test_check_user_input(opt) == 0) {
        if (opt) {
            return 1;
        }
        else {
            return 0;
        }
    }
    else if (nchars == 0) {
        /* default is enabled */
            return 1;
    }
    else {
        /* wrong input -> exit */
        return -1;
    }
}

int mfg_test_options( void )
{
    int result;
    unsigned long test;
    unsigned long new_cfg = 0;

    printf("\nFor each option, enter 0 to disable test and 1 to enable it.\n");

    for (test = SDRAM_TEST; test <= POWER_TEST; test = test << 1) {
        result = mfg_test_parse_input(test);

        switch(result) {
            case 0:
                break;
            case 1:
                new_cfg += test;
                break;
            default:
                return -1;
        }
    }

    /* set the new manufacturing test configuration */
    printf("\nNew Manufacturing Test Configuration: 0x%lx\n", new_cfg);
    cfg_test = new_cfg;

    return 0;
}
