/* U-Boot driver for SSP module of KIRA100
   needed for accessing a SPI flash chip
*/

#include <asm/u-boot.h>
#include <asm/string.h>
#include <kira/kira100.h>
#include <kira/kira_ssp.h>

static unsigned char tx_dma_buf[512], rx_dma_buf[512];
static ssp_dev_t ssp_dev;

int ssp_init(void)
{
    ssp_dev_t *dev = &ssp_dev;
    unsigned int clkdiv;
    
    /* init dev's private data */
    memset(dev, 0, sizeof(ssp_dev_t));
    
    dev->base = (ulong*) CPE_SSP_BASE;
    dev->txdma_base = (ulong*) CPE_APBDMA_CHANNEL0_BASE;
    dev->rxdma_base = (ulong*) CPE_APBDMA_CHANNEL2_BASE;
    
    /* initialize SPI master mode */
    dev->base[SSPCR0] = (SSP_FRFORMAT_SPI << SSPCR0_FRFORMAT_SHIFT) | (SSP_OPM_MASTERMONO << SSPCR0_OPM_SHIFT);
    clkdiv = 0;//(50/(2*1))-1;
    dev->base[SSPCR1] = (7 << SSPCR1_SDL_SHIFT) | clkdiv;
    /* reset SSP core */
    dev->base[SSPCR2] = SSPCR2_SSPRST | SSPCR2_TXFCLR | SSPCR2_RXFCLR;
    /* enable SSP core */
    dev->base[SSPCR2] = SSPCR2_SSPEN;
    dev->base[SSPICR] = (SSP_DMA_TX_THRESHOLD << SSPICR_TFTHOD_SHIFT) | (SSP_DMA_RX_THRESHOLD << SSPICR_RFTHOD_SHIFT) |
            SSPICR_TFDMAEN | SSPICR_RFDMAEN;
    
    return 0;
}

void ssp_uninit(void)
{
}

int addr_spi(ulong addr)
{
    return ((addr >= CFG_FLASH_BASE) && (addr <= (CFG_FLASH_BASE + PHYS_FLASH_SIZE)));
}

int ssp_txrx(char *inbuf, char*outbuf, unsigned int count)
{
    ssp_dev_t *dev = &ssp_dev;
    if ((count > 512) || (inbuf == NULL)) return -1;
    
    memcpy(tx_dma_buf, inbuf, count);
    
    /* program APB DMA channels */
    dev->txdma_base[APBDMA_SOURCE] = (ulong) tx_dma_buf;
    dev->txdma_base[APBDMA_DEST] = (ulong)(dev->base + SSPDR);
    dev->txdma_base[APBDMA_CYCLES] = count;
    // from AHB to APB, bytewise, bytewise source increment, no dest increment, HW handshake pair 6
    dev->txdma_base[APBDMA_CONTROL] = 0x260140; 
            
    dev->rxdma_base[APBDMA_SOURCE] = (ulong)(dev->base + SSPDR);
    dev->rxdma_base[APBDMA_DEST] = (ulong) rx_dma_buf;
    dev->rxdma_base[APBDMA_CYCLES] = count;
    // from APB to AHB, bytewise, no source increment, bytewise dest increment, HW handshake pair 7
    dev->rxdma_base[APBDMA_CONTROL] = 0x271080;
    
    /* start APB DMA channels and wait for finish */
    dev->txdma_base[APBDMA_CONTROL] |= 0x1;
    dev->rxdma_base[APBDMA_CONTROL] |= 0x1;
    while (!(dev->rxdma_base[APBDMA_CONTROL] & 0x2));
    
    if (outbuf) memcpy(outbuf, rx_dma_buf, count);
            
    return count;
}
