/*
 * (C) Copyright 2002
 * Sysgo Real-Time Solutions, GmbH <www.elinos.com>
 * Marius Groeger <mgroeger@sysgo.de>
 *
 * Copyright (C) 2001  Erik Mouw (J.A.K.Mouw@its.tudelft.nl)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <common.h>
#include <command.h>
#include <image.h>
#include <zlib.h>
#include <asm/byteorder.h>

#ifdef CONFIG_HAS_DATAFLASH
#include <dataflash.h>
#endif

#ifdef PP_FEAT_SPI_BOOT
#include <flash.h>
#include <kira/kira_ssp.h>
#endif

#include <asm/setup.h>
#define tag_size(type)  ((sizeof(struct tag_header) + sizeof(struct type)) >> 2)
#define tag_next(t)     ((struct tag *)((u32 *)(t) + (t)->hdr.size))

/*cmd_boot.c*/
extern int do_reset (cmd_tbl_t *cmdtp, int flag, int argc, char *argv[]);

#if defined (CONFIG_SETUP_MEMORY_TAGS) || \
    defined (CONFIG_CMDLINE_TAG) || \
    defined (CONFIG_INITRD_TAG) || \
    defined (CONFIG_VFD)
static void setup_start_tag(bd_t *bd);
# ifdef CONFIG_SETUP_MEMORY_TAGS
static void setup_memory_tags(bd_t *bd);
# endif
static void setup_commandline_tag(bd_t *bd, char *commandline);
#if 0
static void setup_ramdisk_tag(bd_t *bd);
#endif
# ifdef CONFIG_INITRD_TAG
static void setup_initrd_tag(bd_t *bd, ulong initrd_start, ulong initrd_end);
# endif
static void setup_end_tag(bd_t *bd);
# if defined (CONFIG_VFD)
static void setup_videolfb_tag(gd_t *gd);
# endif
# ifdef CONFIG_PEPPERCON
static void setup_peppercon_tags(gd_t *gd);
# endif

static struct tag *params;
#endif /* CONFIG_SETUP_MEMORY_TAGS || CONFIG_CMDLINE_TAG || CONFIG_INITRD_TAG */

#ifdef CONFIG_SHOW_BOOT_PROGRESS
# include <status_led.h>
# define SHOW_BOOT_PROGRESS(arg)	show_boot_progress(arg)
#else
# define SHOW_BOOT_PROGRESS(arg)
#endif

extern image_header_t header;           /* from cmd_bootm.c */

/*
 * Some systems (for example LWMON) have very short watchdog periods;
 * we must make sure to split long operations like memmove() or
 * crc32() into reasonable chunks.
 */
#if defined(CONFIG_HW_WATCHDOG) || defined(CONFIG_WATCHDOG)
# define CHUNKSZ (64 * 1024)
#endif

void do_bootm_linux(cmd_tbl_t *cmdtp, int flag, int argc, char *argv[],
		ulong addr, ulong *len_ptr, int   verify)
{
	DECLARE_GLOBAL_DATA_PTR;

    ulong len = 0, checksum;
    ulong initrd_start, initrd_end, initrd_high ;
    ulong data;
    char *s;
    int	initrd_copy_to_ram = 1;
    void (*theKernel)(int zero, int arch);
    image_header_t *hdr = &header;
    bd_t *bd = gd->bd;
#ifdef CONFIG_CMDLINE_TAG
    char *commandline = getenv("bootargs");
#endif

    /* check for user defined max address */
    if ((s = getenv ("initrd_high")) != NULL) {
	/* a value of "no" or a similar string will act like 0,
	 * turning the "load high" feature off. This is intentional.
	 */
	initrd_high = simple_strtoul(s, NULL, 16);
	if (initrd_high == ~0)
	    initrd_copy_to_ram = 0;
    } else {	/* not set, no restrictions to load high */
	initrd_high = 0;
    }
    
    theKernel = (void (*)(int, int))ntohl(hdr->ih_ep);

    /*
     * Check if there is an initrd image
     */
    if (argc >= 3) {
	SHOW_BOOT_PROGRESS (9);

	addr = simple_strtoul(argv[2], NULL, 16);

	printf ("## Loading Ramdisk Image at %08lx ...\n", addr);

	/* Copy header so we can blank CRC field for re-calculation */
#ifdef CONFIG_HAS_DATAFLASH
	if (addr_dataflash(addr)){
		read_dataflash(addr, sizeof(image_header_t), (char *)&header);
	} else
#endif
        flash_read((uchar *)&header, addr, sizeof(image_header_t));

	if (ntohl(hdr->ih_magic) != IH_MAGIC) {
	    printf ("Bad Magic Number\n");
	    SHOW_BOOT_PROGRESS (-10);
	    do_reset (cmdtp, flag, argc, argv);
	}

	data = (ulong)&header;
	len  = sizeof(image_header_t);

	checksum = ntohl(hdr->ih_hcrc);
	hdr->ih_hcrc = 0;

	if (crc32 (0, (char *)data, len) != checksum) {
	    printf ("Bad Header Checksum\n");
	    SHOW_BOOT_PROGRESS (-11);
	    do_reset (cmdtp, flag, argc, argv);
	}

	SHOW_BOOT_PROGRESS (10);

	print_image_hdr (hdr);

	data = addr + sizeof(image_header_t);
	len  = ntohl(hdr->ih_size);

#ifdef CONFIG_HAS_DATAFLASH
	if (addr_dataflash(addr)){
		read_dataflash(data, len, (char *)CFG_LOAD_ADDR);
		data = CFG_LOAD_ADDR;
	}
#endif
#ifdef PP_FEAT_SPI_BOOT
        if (addr_spi(data)) {
            ulong initrd_addr = (initrd_high) ? (initrd_high - len) : (bd->bi_dram[0].start + bd->bi_dram[0].size) - len;	
            initrd_addr &= ~(4096 - 1);	/* align on page */
    
            flash_read((uchar *)initrd_addr, data, len);
            data = initrd_addr;
            /* we do not need to copy the initrd a second time */
            initrd_copy_to_ram = 0;
        }
#endif

	if (verify) {
	    ulong csum = 0;

	    printf ("   Verifying Checksum ... ");
	    csum = crc32 (0, (char *)data, len);
	    if (csum != ntohl(hdr->ih_dcrc)) {
		printf ("Bad Data CRC\n");
		SHOW_BOOT_PROGRESS (-12);
		do_reset (cmdtp, flag, argc, argv);
	    }
	    printf ("OK\n");
	}

	SHOW_BOOT_PROGRESS (11);

	if ((hdr->ih_os   != IH_OS_LINUX)	||
	    (hdr->ih_arch != IH_CPU_ARM)	||
	    (hdr->ih_type != IH_TYPE_RAMDISK)	) {
	    printf ("No Linux ARM Ramdisk Image\n");
	    SHOW_BOOT_PROGRESS (-13);
	    do_reset (cmdtp, flag, argc, argv);
	}

	/*
	 * Now check if we have a multifile image
	 */
    } else if ((hdr->ih_type==IH_TYPE_MULTI) && (len_ptr[1])) {
	ulong tail    = ntohl(len_ptr[0]) % 4;
	int i;

	SHOW_BOOT_PROGRESS (13);

	/* skip kernel length and terminator */
	data = (ulong)(&len_ptr[2]);
	/* skip any additional image length fields */
	for (i=1; len_ptr[i]; ++i)
	  data += 4;
	/* add kernel length, and align */
	data += ntohl(len_ptr[0]);
	if (tail) {
	    data += 4 - tail;
	}

	len   = ntohl(len_ptr[1]);

    } else {
	/*
	 * no initrd image
	 */
	SHOW_BOOT_PROGRESS (14);

	data = 0;
    }

#ifdef	DEBUG
    if (!data) {
	printf ("No initrd\n");
    }
#endif

    if (data) {
	if (!initrd_copy_to_ram) {	/* zero-copy ramdisk support */
	    initrd_start = data;
	    initrd_end = initrd_start + len;
	} else {
	    if (initrd_high) {
		/* if specified, use configured initrd end address */
		initrd_start  = initrd_high - len;
	    } else {
		/* loading the ramdisk at the end of the first SDRAM bank */
		initrd_start  = (bd->bi_dram[0].start + bd->bi_dram[0].size) - len;	
	    }
	    initrd_start &= ~(4096 - 1);	/* align on page */

	    SHOW_BOOT_PROGRESS (12);
	    
	    debug ("## initrd at 0x%08lX ... 0x%08lX (len=%ld=0x%lX)\n",
		   data, data + len - 1, len, len);
	    
	    initrd_end    = initrd_start + len;
	    printf ("   Loading Ramdisk to %08lx, end %08lx ... ",
		    initrd_start, initrd_end);

#if defined(CONFIG_HW_WATCHDOG) || defined(CONFIG_WATCHDOG)
		{
			size_t l = len;
			void *to = (void *)initrd_start;
			void *from = (void *)data;

			while (l > 0) {
				size_t tail = (l > CHUNKSZ) ? CHUNKSZ : l;
				WATCHDOG_RESET();
				memmove (to, from, tail);
				to += tail;
				from += tail;
				l -= tail;
			}
		}
#else	/* !(CONFIG_HW_WATCHDOG || CONFIG_WATCHDOG) */   
	    memmove ((void *)initrd_start, (void *)data, len);
#endif	/* CONFIG_HW_WATCHDOG || CONFIG_WATCHDOG */
	    
	    printf ("OK\n");
	}
    } else {
	initrd_start = 0;
	initrd_end = 0;
    }

    SHOW_BOOT_PROGRESS (15);

#ifdef DEBUG
    printf ("## Transferring control to Linux (at address %08lx) ...\n",
	    (ulong)theKernel);
#endif

#if defined (CONFIG_SETUP_MEMORY_TAGS) || \
    defined (CONFIG_CMDLINE_TAG) || \
    defined (CONFIG_INITRD_TAG) || \
    defined (CONFIG_VFD)
    setup_start_tag(bd);
#ifdef CONFIG_SETUP_MEMORY_TAGS
    setup_memory_tags(bd);
#endif
#ifdef CONFIG_CMDLINE_TAG
    setup_commandline_tag(bd, commandline);
#endif
#ifdef CONFIG_INITRD_TAG
    setup_initrd_tag(bd, initrd_start, initrd_end);
#endif
#if 0
    setup_ramdisk_tag(bd);
#endif
#ifdef CONFIG_PEPPERCON
    setup_peppercon_tags((gd_t*)gd);
#endif	
#if defined (CONFIG_VFD)
    setup_videolfb_tag((gd_t *)gd);
#endif
    setup_end_tag(bd);
#endif

    /* we assume that the kernel is in place */
    printf("\nStarting kernel ...\n\n");

    cleanup_before_linux();

    theKernel(0, bd->bi_arch_number);
}


#if defined (CONFIG_SETUP_MEMORY_TAGS) || \
    defined (CONFIG_CMDLINE_TAG) || \
    defined (CONFIG_INITRD_TAG) || \
    defined (CONFIG_VFD)
static void setup_start_tag(bd_t *bd)
{
    params = (struct tag *)bd->bi_boot_params;

    params->hdr.tag = ATAG_CORE;
    params->hdr.size = tag_size(tag_core);

    params->u.core.flags = 0;
    params->u.core.pagesize = 0;
    params->u.core.rootdev = 0;

    params = tag_next(params);
}


#ifdef CONFIG_SETUP_MEMORY_TAGS
static void setup_memory_tags(bd_t *bd)
{
    int i;

    for(i = 0; i < CONFIG_NR_DRAM_BANKS; i++) {
	params->hdr.tag = ATAG_MEM;
	params->hdr.size = tag_size(tag_mem32);

	params->u.mem.start = bd->bi_dram[i].start;
	params->u.mem.size = bd->bi_dram[i].size;

	params = tag_next(params);
    }
}
#endif	/* CONFIG_SETUP_MEMORY_TAGS */


static void setup_commandline_tag(bd_t *bd, char *commandline)
{
    char *p;

    /* eat leading white space */
    for(p = commandline; *p == ' '; p++)
      ;

    /* skip non-existent command lines so the kernel will still
     * use its default command line.
     */
    if(*p == '\0')
      return;

    params->hdr.tag = ATAG_CMDLINE;
    params->hdr.size = (sizeof(struct tag_header) + strlen(p) + 1 + 4) >> 2;

    strcpy(params->u.cmdline.cmdline, p);

    params = tag_next(params);
}


#ifndef ATAG_INITRD2
#define ATAG_INITRD2    0x54420005
#endif

#ifdef CONFIG_INITRD_TAG
static void setup_initrd_tag(bd_t *bd, ulong initrd_start, ulong initrd_end)
{
    /* an ATAG_INITRD node tells the kernel where the compressed
     * ramdisk can be found. ATAG_RDIMG is a better name, actually.
     */
    params->hdr.tag = ATAG_INITRD2;
    params->hdr.size = tag_size(tag_initrd);

    params->u.initrd.start = initrd_start;
    params->u.initrd.size = initrd_end - initrd_start;

    params = tag_next(params);
}
#endif	/* CONFIG_INITRD_TAG */


#if 0
static void setup_ramdisk_tag(bd_t *bd)
{
    /* an ATAG_RAMDISK node tells the kernel how large the
     * decompressed ramdisk will become.
     */
    params->hdr.tag = ATAG_RAMDISK;
    params->hdr.size = tag_size(tag_ramdisk);

    params->u.ramdisk.start = 0;
    /*params->u.ramdisk.size = RAMDISK_SIZE; */
    params->u.ramdisk.flags = 1;	/* automatically load ramdisk */

    params = tag_next(params);
}
#endif /* 0 */

#if defined (CONFIG_VFD)
static void setup_videolfb_tag(gd_t *gd)
{
    /* An ATAG_VIDEOLFB node tells the kernel where and how large
     * the framebuffer for video was allocated (among other things).
     * Note that a _physical_ address is passed !
     *
     * We only use it to pass the address and size, the other entries
     * in the tag_videolfb are not of interest.
     */
    params->hdr.tag = ATAG_VIDEOLFB;
    params->hdr.size = tag_size(tag_videolfb);

    params->u.videolfb.lfb_base = (u32)gd->fb_base;
    /* 7168 = 256*4*56/8 - actually 2 pages (8192 bytes) are allocated */
    params->u.videolfb.lfb_size = 7168;

    params = tag_next(params);
}
#endif


static void setup_end_tag(bd_t *bd)
{
    params->hdr.tag = ATAG_NONE;
    params->hdr.size = 0;
}

#if defined (CONFIG_PEPPERCON)
static void setup_peppercon_tags(gd_t *gd)
{
    {
	params->hdr.tag = ATAG_FLASH;
	params->hdr.size = tag_size(tag_flash);
	memcpy(&(params->u.flash.flash_info), &gd->bd->flash_info, sizeof(bd_flash_info_t));
	params = tag_next(params);
    }
    {
	bd_pp_info_t pp_info;

	pp_info.bi_def_jumper_set = gd->pp.def_jumper_set;
	pp_info.bi_debug_mode = gd->pp.serial_debug;
	pp_info.bi_config_mode = gd->pp.config_mode;
	pp_info.bi_hwid = gd->pp.hwid;
	pp_info.bi_power_cim = gd->pp.power_cim;
	
	params->hdr.tag = ATAG_PP;
	params->hdr.size = tag_size(tag_pp);
	memcpy(&params->u.pp.pp_info, &pp_info, sizeof(bd_pp_info_t));
	params = tag_next(params);
    }

    params->hdr.tag = ATAG_CLOCK;
    params->hdr.size = tag_size(tag_clock);
    memcpy(&params->u.clock.clock_info, &gd->bd->clock_info, sizeof(bd_arm_clock_info_t));
    params = tag_next(params);

    {
	bd_net_info_t net_info;
	
	memcpy(net_info.bi_enetaddr, gd->bd->bi_enetaddr, 6);

	params->hdr.tag = ATAG_NET;
	params->hdr.size = tag_size(tag_net);
	memcpy(&params->u.net.net_info, &net_info, sizeof(bd_net_info_t));
	params = tag_next(params);
    }
}
#endif

#endif /* CONFIG_SETUP_MEMORY_TAGS || CONFIG_CMDLINE_TAG || CONFIG_INITRD_TAG */

