#!/usr/bin/perl -w
###############################################################################
#
# patchcrc   - Calculates and patches a CRC checksum into a binary at a specific
#              offset
#

use strict;

#
###############################################################################
# imports
#

use Getopt::Std;

my $opts  = {};
my $usage = "Usage: \n\n" .
  "  patchcrc -f <binary filename> -a <start address> -l <data length>\n" .
  "           [ -e <ELF filename> ] [ -s <ELF section> ] -c [ crc tool ] \n\n" .

  "           -f <binary filename> - filename to generate checksum from\n" .
  "           -e <ELF filename>    - ELF file to get the section offset and to patch CRC\n" .
  "           -s <ELF section>     - section name in the ELF file to get offset\n" .
  "           -a <start address>   - hex start address for generating the CRC\n" .
  "           -l <data length>     - length of the data portion for generating CRC\n" .
  "           -c <crc tool>        - tool which prints CRC on STDOUT for data on STDIN\n" .
  "                                  if omitted or missing, Perl module String::CRC32 is used\n" .
  "\n";

getopts('f:e:s:o:a:l:c:', $opts);

sub get_offset_for_section($$$);
sub calculate_crc($$$$);
sub patch_crc($$$);

my $objdumps = {
		"arm" => "armv4-pp001-linux-uclibc-objdump",
		"ppc" => "powerpc-pp001-linux-uclibc-objdump",
};

# -----------------------------------------------------------------------------
# check command line options and get required information
# -----------------------------------------------------------------------------

my $filename;
my $elf_filename;
my $start;
my $length;
my $offset;
my $crc;
my $tool;

defined $opts->{"f"} or die "Target filename missing.\n\n$usage";
$filename = $opts->{"f"};

defined $opts->{"e"} or die "ELF filename missing, need ELF file and section name\n\n$usage";
defined $opts->{"s"} or die "Section name missing, need ELF file and section name\n\n$usage";

$offset = get_offset_for_section($opts->{"e"}, $opts->{"s"}, "file");
$elf_filename = $opts->{"e"};

if (defined $opts->{"a"}) {
  $start = $opts->{"a"};
} else {
  $start = 0;
}

if (defined $opts->{"l"}) {
  $length = $opts->{"l"};
} else {
  $length = get_offset_for_section($opts->{"e"}, $opts->{"s"}, "VMA");
}

if (defined $opts->{"c"} && -x $opts->{"c"}) {
  use FileHandle;
  use IPC::Open2;
  $tool = $opts->{"c"};
} else {
  require String::CRC32;
  $tool = "";
}

# -----------------------------------------------------------------------------
# calculate crc and patch it
# -----------------------------------------------------------------------------
$crc = calculate_crc($filename, $start, $length, $tool);
patch_crc($elf_filename, $offset, $crc);

printf("CRC generated (0x%08x, len 0x%08x), patched in $elf_filename @ 0x%08x\n", 
       $start, $length, $offset);

# -----------------------------------------------------------------------------
# subs
# -----------------------------------------------------------------------------
sub calculate_crc($$$$) {
  my $filename = shift;
  my $start = shift;
  my $length = shift;
  my $tool = shift;
  my $filedata;
  my $crc;
  my $line;
  my $result;
  my $pid;

  open(FILE, "<$filename") or die "Could not open binary file '$filename' for reading.\n";
  while (<FILE>) {
    $filedata .= $_;
  }
  close (FILE);

  if ($tool ne "") {
    $pid = open2(\*Reader, \*Writer, "$tool") or die "Could not execute CRC tool.\n";
    Writer->autoflush();
    print Writer substr($filedata, $start, $length);
    close(Writer);
    while ($line = <Reader>) {
      if ($line =~ /^CRC=(\w+)/) {
	$crc = hex($1);
      }
    }
    close(Reader);
    print "Calculated CRC with external tool '$tool'.\n";
  } else {
    $crc = String::CRC32::crc32(substr($filedata, $start, $length), 0);
    print "Calculated CRC internally.\n";
  }

  return $crc;
}

sub patch_crc($$$) {
  my $filename = shift;
  my $offset = shift;
  my $crc = shift;
  my $filedata;

  open(FILE, "<$filename") or die "Could not open binary file '$filename' for reading.\n";
  while (<FILE>) {
    $filedata .= $_;
  }
  close (FILE);

  open(FILE, ">$filename") or die "Could not open binary file '$filename' for writing.\n";
  print FILE substr($filedata, 0, $offset);
  print FILE pack("L", $crc);
  print FILE substr($filedata, $offset + 4);
  close (FILE);
}

sub get_offset_for_section( $$$ ) {
  my $filename = shift;
  my $section = shift;
  my $offset_type = shift; # file or VMA
  my $offset;
  my $line;
  my $objdump;

  # recognize file type to choose appropriate objdump
  open FILE, "file $filename|" or die "Could not determine ELF file type (file check failed)\n";
  while ($line = <FILE>) {
    if ($line =~ /ARM/) {
      $objdump = $objdumps->{"arm"};
      last;
    }
    if ($line =~ /PowerPC/) {
      $objdump = $objdumps->{"ppc"};
      last;
    }
  }
  close FILE;

  unless (defined $objdump) {
    die "Could not determine ELF file type (unknown elf filetype)\n";
  }

  open FILE, "$objdump -h $filename|" or die "Could not determine ELF file type (file check failed)\n";
  while ($line = <FILE>) {
    if ($line =~ /$section\s+(\w+)\s+(\w+)\s+(\w+)\s+(\w+)\s*/) {
      if ($offset_type eq "file") {
	$offset = hex($4);
      } elsif ($offset_type eq "VMA") {
	$offset = hex($2);
      } else {
	die "Unknown offset type $offset_type";
      }
    }
  }

  return $offset;
}
