/* Splash control program.
 * 
 * (c) 2001, 2002, 2003 by Stefan Reinauer <stepan@suse.de>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 *
 */

#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/select.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <libgen.h>
#include <getopt.h>
#include <signal.h>
#include "libmng.h"

#define PROCSPLASH "/proc/splash"
#define BUFSIZE 4096

long unit = 0;
long version = 1;
long state = -1;
long fgcolor = -1;
long bgcolor = -1;
long tx = -1;
long ty = -1;
long tw = -1;
long th = -1;
long px = 0;
long py = 0;
long pw = 0;
long ph = 0;
long pr = 0;
long pg = 0;
long pb = 0;
long pa = 160;
long pf = 1;
long pc = 0x141464;
long ptl = 0xffff;
long pbr = 0x00;
long ax = 0;
long ay = 0;
long progress = 0;
long overpaintok = 0;
long percent = 0;
long text = 0;
char *jpgfile = 0;
char *silentjpgfile = 0;
char splashdscname[512];


int splashfd;
int splashstate = -1;	/* -1:unknown 0:verbose 1:silent */

int configready;

#define TRIGGER_QUIT		0
#define TRIGGER_PLAY		1
#define TRIGGER_SYSTEM		2
#define TRIGGER_SYSTEMBG	3
#define TRIGGER_TOSILENT	4
#define TRIGGER_TOVERBOSE	5
#define TRIGGER_STOPANIM	6

struct boxanim;

struct trigger {
  struct trigger *next;
  char *name;
  int action;
  struct boxanim *anim;
  char *tagstr;
  int tag;
  char *shellcmd;
};
struct trigger *triggers;

#define FRAME_STOP 1
#define FRAME_JUMP 2

struct frame {
  unsigned char *boxes;
  unsigned char *sboxes;
  int boxcount;
  int sboxcount;
  char *name;
  int action;
  int tag;
  int delay;
};

struct frame mainframe;

struct boxanim {
  struct boxanim *next;
  char *file;
  int id;
  int nframes;
  int cframe;
  struct frame *frames;
  char *name;
  int silent;
  int posx, posy;
  int posmode;
  int scale1, scale2;
  int sizex, sizey;
  int hasrecolor;
  unsigned char recolor[8];
  int boxcount;
  int sboxcount;
  int boxstart;
  int sboxstart;
  char *initframe;
  int init;
  int defaultdelay;
};

struct boxanim *boxanims;
int nextanimid = 0;

struct event {
  struct event *next;
  int type;
  int due;
  struct boxanim *ba;
};

#define EVENT_ANIM    0
#define EVENT_PERCENT 1

struct event *events;

int now = 0;

#define POSMODE_UPPERLEFT 0
#define POSMODE_CENTER    1

struct parstab {
	char *name;
	long *val;
	char **str;
} parstab[] = {
	{ "unit", &unit }, 
	{ "version", &version }, 
	{ "state", &state }, 
	{ "fgcolor", &fgcolor }, 
	{ "bgcolor", &bgcolor },
	    /* v1+ */
	{ "tx", &tx }, 
	{ "ty", &ty }, 
	{ "tw", &tw }, 
	{ "th", &th },
	    /* v2+ */
	{ "px", &px }, 
	{ "py", &py }, 
	{ "pw", &pw }, 
	{ "ph", &ph }, 
	{ "pr", &pr }, 
	{ "pg", &pg }, 
	{ "pb", &pb }, 
	{ "ax", &ax }, 
	{ "ay", &ay },
	    /* v3+ */
	    /* independant */
	{ "progress_enable", &progress }, 
	{ "progress_x", &progress }, 
	{ "progress_y", &progress }, 
	{ "progress_width", &progress }, 
	{ "progress_height", &progress }, 
	{ "progress_fgcolor", &progress }, 
	{ "progress_bgcolor", &progress }, 
	{ "text_size", &text }, 
	{ "text_color", &text }, 
	{ "text_x", &text }, 
	{ "text_y", &text }, 
	{ "text_font", (long *) 0 }, 
	{ "silentjpeg", (long *) 0, &silentjpgfile }, 
	{ "percent", &percent }, 
	{ "overpaintok", &overpaintok }, 
	{ "jpeg", (long *) 0, &jpgfile }, 
	{ (char *) 0, (long *) 0 }
};

/*******************************************************************/

void *xmalloc(int len)
{
  void *r = malloc(len ? len : 1);
  if (r)
    return r;
  fprintf(stderr, "Out of memory allocating %d bytes!\n", len);
  exit(1);
}

void *xrealloc(void *old, int len)
{
  if (!old)
    return xmalloc(len);
  old = realloc(old, len ? len : 1);
  if (old)
    return old;
  fprintf(stderr, "Out of memory reallocating %d bytes!\n", len);
  exit(1);
}

int splitargs(char *p, char **args, int maxargs)
{
  int n = 0;

  for(;;)
    {
      while (*p == ' ' || *p == '\t')
        p++;
      if (!*p)
	return n;
      if (n == maxargs)
	{
	  fprintf(stderr, "Too many args\n");
	  exit(1);
	}
      if (*p == '"')
	{
	  args[n++] = p + 1;
	  p = strchr(p + 1, '"');
	  if (!p)
	    {
	      fprintf(stderr, "missing quote\n");
	      exit(1);
	    }
	}
      else
	{
	  args[n++] = p;
	  while (*p && *p != ' ' && *p != '\t')
	    p++;
	  if (!*p)
	    return n;
	}
      *p++ = 0;
    }
}


/*******************************************************************/

unsigned char *scalepic(unsigned char *opic, int ow, int oh, int w, int h)
{
  unsigned char *pic, *p, *op, *op2, *oop;
  int x, y;
  int dx, dy;
  int fx, fy;
  int d;

  pic = (unsigned char *)malloc(w * h);
  y = 0;
  dy = h / 2;
  op = opic;
  p = pic;
  for (y = h; y-- > 0; )
    {
      while (dy < 0)
	{
	  op += ow;
	  dy += h;
	}
      fy = 255 * dy / h;
      dy -= oh;
      dx = w / 2;
      oop = op;
      op2 = y == 0 ? op : op + ow;
      for (x = w - 1; x-- > 0 ; )
	{
	  while (dx < 0)
	    {
	      op++;
	      op2++;
	      dx += w;
	    }
	  fx = 255 * dx / w;
	  dx -= ow;
	  d  = (op[0] * fx + op[1] * (255 - fx)) * fy;
	  d += (op2[0] * fx + op2[1] * (255 - fx)) * (255 - fy);
	  *p++ = d / (255 * 255);
	}
      while (dx < 0)
	{
	  op++;
	  op2++;
	  dx += w;
	}
      fx = 255 * dx / w;
      *p++ = (op[0] * fy + op2[0] * (255 - fy)) / 255;
      op = oop;
    }
  free(opic);
  return pic;
}


/*******************************************************************/
/********************                            *******************/
/********************        MNG Support         *******************/
/********************                            *******************/
/*******************************************************************/

/* as libmng is much too big we'll probably parse the mng stream
 * ourself in the future */

/* globals */
FILE *mngf;
int mngwidth, mngheight;
unsigned char *mngimage;
int mngstopit;
mng_uint32 mngtick, mngtickwait;
int mngpic;
unsigned char **mngframes;
char **mnganno;
int *mngdelay;

mng_ptr mngalloc(mng_size_t s)
{
  return (mng_ptr)calloc(1, (size_t)s);
}

void mngfree(mng_ptr p, mng_size_t s)
{
  free(p);
}

mng_bool mngopenstream(mng_handle hMNG)
{
  return MNG_TRUE;
}

mng_bool mngclosestream(mng_handle hMNG)
{
  return MNG_TRUE;
}

mng_bool mngreaddata (mng_handle hMNG, mng_ptr buf, mng_uint32 s, mng_uint32 *sp)
{
  *sp = fread (buf, 1, s, mngf);
  return MNG_TRUE;
}

mng_bool mngprocessheader(mng_handle hMNG, mng_uint32 iWidth, mng_uint32 iHeight)
{
  mngwidth = iWidth;
  mngheight = iHeight;
  mngimage = malloc(iWidth * iHeight * 3);
  mng_set_canvasstyle(hMNG, MNG_CANVAS_RGB8);
  return MNG_TRUE;
}

mng_bool mngprocessmend(mng_handle hMNG, mng_uint32  iIterationsdone, mng_uint32  iIterationsleft)
{
  mngstopit++;
  return mngstopit < 2 ? MNG_TRUE : MNG_FALSE;
}

mng_ptr mnggetcanvasline(mng_handle hMNG, mng_uint32 iLinenr)
{
  return (mng_ptr)(mngimage + mngwidth * 3 * iLinenr);
}

mng_bool mngrefresh(mng_handle hMNG, mng_uint32 iX, mng_uint32 iY, mng_uint32 iWidth, mng_uint32 iHeight)
{
  unsigned char *o, *i;
  int j;

  if (mngpic % 16 == 0)
    {
      char **oldmnganno = mnganno;
      mngframes = xrealloc(mngframes, sizeof(unsigned char *) * (mngpic + 16));
      mnganno = xrealloc(mnganno, sizeof(char *) * (mngpic + 17));
      mngdelay = xrealloc(mngdelay, sizeof(int) * (mngpic + 16));
      if (!oldmnganno)
	*mnganno = 0;
    }
  i = mngimage + 1;
  j = mngwidth *mngheight;
  o = xmalloc(j);
  mngframes[mngpic] = o;
  mngdelay[mngpic] = 0;
  while (j-- > 0)
    {
      *o++ = *i;
      i += 3;
    }
  mngpic++;
  mnganno[mngpic] = 0;
  return MNG_TRUE;
}

mng_uint32 mnggettickcount(mng_handle hMNG)
{
  return mngtick;
}

mng_bool mngsettimer (mng_handle hHandle, mng_uint32 iMsecs)
{
  mngtickwait = iMsecs;
  return MNG_TRUE;
}

mng_bool mngprocesstext(mng_handle hHandle,
                            mng_uint8  iType,
                            mng_pchar  zKeyword,
                            mng_pchar  zText,
                            mng_pchar  zLanguage,
                            mng_pchar  zTranslation)
{
  if (iType != 0 || strcmp(zKeyword, "comment"))
    return MNG_TRUE;
  if (!mnganno)
    mnganno = xmalloc(sizeof(char *));
  mnganno[mngpic] = strdup(zText);
  return MNG_TRUE;
}

#define GETPIC(x, y) ((x) < w && (y) < h ? (c = pic[(y) * w + (x)], (opic && c == opic[(y) * w + (x)]) ? -1 : c) : -1)

unsigned char *
makeboxes(int *cntp, unsigned char *opic, unsigned char *pic, int w, int h, int size1, int size2)
{
  int c, x, y;
  int col, colw, cole, colew, col2;
  int xs, ys, xe, ye;
  unsigned char *box, *bp;
  int bcnt, bf;

  box = bp = 0;
  bcnt = 0;
  bf = 0;
  for (y = 0; y < h; y++)
    {
      for (x = 0; x < w; x++)
        {
          if (GETPIC(x, y) != -1)
            break;
        }
      if (x == w)
        continue;
      x = 0;
      while (x < w)
        {
          xs = x;
          col = GETPIC(x, y);
          colw = GETPIC(x, y + 1);
          for (; x < w; x++)
            {
              col2  = GETPIC(x, y);
              if (col != col2)
                break;
              col2  = GETPIC(x, y + 1);
              if (colw != col2)
                break;
            }
          if (x == xs)
            x++;
          if (col == -1 && colw == -1)
            continue;
          if (x == xs + 1 && x < w)
            x++;
          ys = y;
          xe = x - 1;
          ye = y + 1;
          cole = GETPIC(xe, ys);
          colw = GETPIC(xs, ye);
          colew = GETPIC(xe, ye);
          if (cole == -1 && colew == -1)
            {
              xe--;
              cole = col;
              colew = colw;
            }
          if (col == -1 && cole == -1 && ys < ye)
            {
              ys++;
              col = colw;
              cole = colew;
            }
          else if (colw == -1 && colew == -1 && ys < ye)
            {
              ye--;
              colw = col;
              colew = cole;
            }
          if (col == cole && col == colw && col == colew && xs == 0 && xe == w - 1 && ye < h - 1)
	    {
	      while (ye < h - 1)
		{
		  for (x = 0; x < w; x++)
		    if (col != GETPIC(x, ye + 1))
		      break;
		  if (x < w)
		   break;
		  ye++;
		}
	      x = w;
	    }
	  if (bf < 2)
	    {
	      box = realloc(box, 12 * (16 + bcnt));
	      bf = 16;
	      bp = box + 12 * bcnt;
	    }
	  bp[0] = xs;
	  bp[1] = xs >> 8;
	  bp[2] = ys;
	  bp[3] = ys >> 8;
	  bp[4] = xe;
	  bp[5] = xe >> 8;
	  bp[6] = ye;
	  bp[7] = ye >> 8;
          // printf("box %d %d %d %d", xs, ys, xe, ye);
          if (col == cole && col == colw && col == colew)
            {
	      bp[8] = bp[9] = bp[10] = col;
	      bp[11] = 255;
              // printf(" #%02x%02x%02xff", col, col, col);
            }
          else
            {
	      bp[2] ^= 255;
	      bp[3] ^= 255;
	      bp[8] = bp[9] = bp[10] = (col == -1 ? 0 : col);
	      bp[11] = (col == -1) ? 0 : 255;
	      bp[12] = bp[13] = bp[14] = (cole == -1 ? 0 : cole);
	      bp[15] = (cole == -1) ? 0 : 255;
	      bp[16] = bp[17] = bp[18] = (colw == -1 ? 0 : colw);
	      bp[19] = (colw == -1) ? 0 : 255;
	      bp[20] = bp[21] = bp[22] = (colew == -1 ? 0 : colew);
	      bp[23] = (colew == -1) ? 0 : 255;
	      bcnt++;
	      bf--;
	      bp += 12;
#if 0
              if (col == -1)
                printf(" #00000000");
              else
                printf(" #%02x%02x%02xff", col, col, col);
              if (cole == -1)
                printf(" #00000000");
              else
                printf(" #%02x%02x%02xff", cole, cole, cole);
              if (colw == -1)
                printf(" #00000000");
              else
                printf(" #%02x%02x%02xff", colw, colw, colw);
              if (colew == -1)
                printf(" #00000000");
              else
                printf(" #%02x%02x%02xff", colew, colew, colew);
#endif
            }
          // printf("\n");
	  bcnt++;
	  bf--;
	  bp += 12;
        }
      if (y < h - 1)
        y++;
    }
  *cntp = bcnt;
  // printf("--------------------------\n");
  return box;
}

void mngtoba(struct boxanim *ba, unsigned char **fra, char **ano, int *del, int loop)
{
  struct frame *fr;
  int i, j, cf;
  char *args[256];
  int nargs;
  int nodelta;

  ba->frames = xmalloc(ba->nframes * sizeof(struct frame));
  memset(ba->frames, 0, ba->nframes * sizeof(struct frame));
  for (cf = 0; cf < ba->nframes; cf++)
    {
      fr = ba->frames + cf;
      fr->delay = del[cf];
      nodelta = 0;
      if (ano[cf])
	{
	  nargs = splitargs(ano[cf], args, 256);
	  for (i = 0; i < nargs; )
	    {
	      if (!strcmp(args[i], "id") && i + 1 < nargs)
		{
		  fr->name = strdup(args[i + 1]);
		  i += 2;
		  continue;
		}
	      if (!strcmp(args[i], "delay") && i + 1 < nargs)
		{
		  fr->delay = atoi(args[i + 1]);
		  i += 2;
		  continue;
		}
	      if (!strcmp(args[i], "stop"))
		{
		  fr->action = FRAME_STOP;
		  fr->tag = 0;
		  i++;
		  continue;
		}
	      if (!strcmp(args[i], "nodelta"))
		{
		  nodelta = 1;
		  i++;
		  continue;
		}
	      if (!strcmp(args[i], "loop"))
		{
		  fr->action = FRAME_JUMP;
		  fr->tag = 0;
		  if (i + 1 < nargs)
		    {
		      for (j = 0; j < cf; j++)
			if (ba->frames[j].name && !strcmp(ba->frames[j].
name, args[i + 1]))
			  break;
		      if (j == cf)
			{
			  fprintf(stderr, "Bad loop tag\n");
			  exit(1);
			}
		      fr->tag = j;
		      i++;
		    }
		  i++;
		  continue;
		}
	      fprintf(stderr, "frame: unknown subcommand '%s'\n", args[i
]);
	      exit(1);
	    }
	}
      if (ba->silent)
        fr->sboxes = makeboxes(&fr->sboxcount, nodelta ? 0 : fra[cf == 0 && ba->nframes > 1 ? ba->nframes - 1 : cf - 1], fra[cf], ba->sizex, ba->sizey, ba->scale1, ba->scale2);
      else
        fr->boxes = makeboxes(&fr->boxcount, nodelta ? 0 : fra[cf == 0 && ba->nframes > 1 ? ba->nframes - 1 : cf - 1], fra[cf], ba->sizex, ba->sizey, ba->scale1, ba->scale2);
      if (loop && cf == ba->nframes - 1 && !fr->action)
	{
	  fr->action = FRAME_JUMP;
	  fr->tag = 0;
	}
    }
}

void readmnganim(struct boxanim *ba, char *path)
{
  mng_handle hMNG;
  mng_retcode r;
  int i;

  if ((mngf = fopen(path, "r")) == 0)
    {
      perror(path);
      exit(1);
    }
  mngwidth = mngheight = 0;
  mngimage = 0;
  mngstopit = 0;
  mngtick = mngtickwait = 0;
  mngpic = 0;
  hMNG = mng_initialize((mng_ptr)0, mngalloc, mngfree, MNG_NULL);
  if (!hMNG)
    {
      fprintf(stderr, "Cannot initialize libmng.\n");
      exit(1);
    }
  mng_setcb_openstream(hMNG, mngopenstream);
  mng_setcb_closestream(hMNG, mngclosestream);
  mng_setcb_readdata(hMNG, mngreaddata);
  mng_setcb_processheader(hMNG, mngprocessheader);
  mng_setcb_processmend(hMNG, mngprocessmend);
  mng_setcb_processtext(hMNG, mngprocesstext);
  mng_setcb_settimer(hMNG, mngsettimer);
  mng_setcb_gettickcount(hMNG, mnggettickcount);
  mng_setcb_refresh(hMNG, mngrefresh);
  mng_setcb_getcanvasline(hMNG, mnggetcanvasline);
  r = mng_readdisplay(hMNG);
  while (r == MNG_NEEDTIMERWAIT)
    {
      if (mngpic)
        mngdelay[mngpic - 1] = mngtickwait;
      if (mngstopit)
	break;
      mngtick += mngtickwait;
      r = mng_display_resume(hMNG);
    }
  if (!mngstopit && r != MNG_NOERROR)
    {
      fprintf(stderr, "libmng layer dump error: %d\n", r);
      exit(1);
    }
  mng_cleanup (&hMNG);
  fclose(mngf);
  if (mngimage)
    free(mngimage);
  mngimage = 0;
  ba->sizex = mngwidth;
  ba->sizey = mngheight;
  ba->nframes = mngpic;
  ba->cframe = ba->nframes;
  if (ba->scale1 != ba->scale2)
    {
      int nw, nh;
      nw = (ba->sizex * ba->scale1) / ba->scale2;
      nh = (ba->sizey * ba->scale1) / ba->scale2;
      for (i = 0; i < mngpic; i++)
        mngframes[i] = scalepic(mngframes[i], ba->sizex, ba->sizey, nw, nh);
      ba->sizex = nw;
      ba->sizey = nh;
    }
  if (mngpic)
    mngtoba(ba, mngframes, mnganno, mngdelay, mngstopit);
  for (i = 0; i < mngpic; i++)
    free(mngframes[i]);
  for (i = 0; i < mngpic; i++)
    if (mnganno[i])
      free(mnganno[i]);
  if (mngframes)
    free(mngframes);
  if (mnganno)
    free(mnganno);
  if (mngdelay)
    free(mngdelay);
  mngframes = 0;
  mnganno = 0;
  mngdelay = 0;
}


/*******************************************************************/

void fixboxcopy(unsigned char *buf, int cnt)
{
	int *boxoff;
	unsigned char *bp;
	int bc = 0, bc2 = 0;
	int i, ni;

        if (!cnt)
	  return;
        boxoff = xmalloc(sizeof(int) * cnt);
	for (i = 0; i < cnt; i = ni) {
		ni = i + 1;
		bp = buf + 12 * i;
		if (bp[3] > 0x7f)
			ni++;
		if (bp[1] > 0x7f)
			ni += buf[12 * ni + 3] >= 0x7f ? 2 : 1;
		if (bp[0] == 0xff && bp[1] == 0x7f)
			continue;
		boxoff[bc++] = i;
	}
	for (i = 0; i < cnt; i = ni) {
		ni = i + 1;
		bp = buf + 12 * i;
		if (bp[3] > 0x7f)
			ni++;
		if (bp[1] > 0x7f)
			ni += buf[12 * ni + 3] >= 0x7f ? 2 : 1;
		if (bp[0] == 0xff && bp[1] == 0x7f && bp[2] == 0xfe && bp[3] == 0x7f) {
			i = bp[6] | bp[7] << 8;
			if (i > 32767)
				i -= 65536;
			i += bc2;
			if (i < 0 || i >= bc) {
				fprintf(stderr, "box copy out of range\n");
				exit(1);
			}
			i = boxoff[i] - ni;
			bp[6] = i & 0xff;
			bp[7] = (i >> 8) & 0xff;
			continue;
		}
		if (bp[0] == 0xff && bp[1] == 0x7f)
			continue;
		bc2++;
	}
        free(boxoff);
}

void fixposandcolor(struct boxanim *ba, unsigned char *bp, int cnt)
{
  int addx, addy;
  int i, j, d;

  if (!cnt)
    return;
  addx = ba->posx;
  addy = ba->posy;
  if (ba->posmode == POSMODE_CENTER)
    {
      addx -= ba->sizex / 2;
      addy -= ba->sizey / 2;
    }
  if (addx == 0 && addy == 0 && !ba->hasrecolor)
    return;
  while (cnt-- > 0)
    {
      if (bp[0] == 0xff && bp[1] == 0x7f)
	{
	  if (bp[3] & 0x80)
	    {
	      cnt--;
	      bp += 12;
	    }
	  bp += 12;
	  continue;
	}
      for (i = 0; i < 4; i++)
	{
	  d = bp[0] | (bp[1] << 8);
	  if (d & 32768)
	    d ^= 65535;
	  if (i & 1)
	    d += addy;
	  else
	    d += addx;
	  if (d < 0)
	    d = 0;
	  if (d > 32767)
	    d = 32767;
	  if (bp[1] & 0x80)
	    d = ~d;
	  bp[0] = d & 255;
	  bp[1] = d >> 8;
	  bp += 2;
	}
      bp -= 8;
      if (ba->hasrecolor)
	{
	  for (j = 0; j < 4; j++)
	    {
	      for (i = 0; i < 4; i++)
		{
		  d = bp[8 + i + 4 * j];
		  d = ((255 - d) * ba->recolor[i] + d * ba->recolor[4 + i]) / 255;
		  bp[8 + i + 4 * j] = d;
		}
	      if (j == 0 && !(bp[3] & 0x80))
		break;
	    }
	}
      if (bp[3] & 0x80)
	{
	  cnt--;
	  bp += 12;
	}
      bp += 12;
    }
}

void parsebox(char *pl, struct frame *fr)
{
	int co[4];
	int i, j, c, d = 0;
	char cols[16];
	unsigned char *bp;
	int boxcnt;
	int isibox = 0;
	int *boxcntp;
	unsigned char *boxptr, **boxptrp;
	char *p = pl + 3;
	int noover = 0;
	int hidemax = 0;
	int isstip = 0;
	int iscopy = 0;
	int range = 0;
	int anchor = -1;

	boxcntp = &fr->boxcount;
	boxptrp = &fr->boxes;
	for (i = 0; i < 4; i++) {
		if (*p != ' ' && *p != '\t' && *p != '=') {
			fprintf(stderr, "syntax error: '%c' %s\n", *p, pl);
			exit(1);
		}
		while (*p == ' ' || *p == '\t' || *p == '=')
			p++;
		if (i == 0 && !strncmp(p, "copy", 4)) {
			i = 3 - 1;
			iscopy = 1;
			co[0] = 0x7fff;
			co[1] = 0x7ffe;
			co[2] = co[3] = 0;
			cols[0] = cols[1] = cols[2] = cols[3] = 0;
			p += 4;
			continue;
		}
		if (iscopy && *p == '-') {
			iscopy = -1;
			p++;
		}
		if (i == 0 && !strncmp(p, "stipple", 7)) {
			i = 2 - 1;
			co[0] = co[1] = 0x7fff;
			co[2] = co[3] = 0;
			isstip = 1;
			p += 7;
			continue;
		}
		if (isstip && i == 2 && !strncmp(p, "range", 5)) {
			i--;
			range = 1;
			p += 5;
			continue;
		}
		if (isstip && i == 2 && !strncmp(p, "anchorabs", 9)) {
			i--;
			anchor = 0;
			p += 9;
			continue;
		}
		if (isstip && i == 2 && !strncmp(p, "anchorstart", 11)) {
			i--;
			anchor = -1;
			p += 11;
			continue;
		}
		if (isstip && i == 2 && !strncmp(p, "anchorend", 9)) {
			i--;
			anchor = 1;
			p += 9;
			continue;
		}
		if (i == 0 && !strncmp(p, "silent", 6)) {
			i--;
			boxcntp = &fr->sboxcount;
			boxptrp = &fr->sboxes;
			p += 6;
			continue;
		}
		if (i == 0 && !strncmp(p, "inter", 5)) {
			i--;
			isibox = 1;
			p += 5;
			continue;
		}
		if (i == 0 && !strncmp(p, "noover", 6)) {
			i--;
			noover = 1;
			p += 6;
			continue;
		}
		if (i == 0 && !strncmp(p, "hidemax", 7)) {
			i--;
			hidemax = 1;
			p += 7;
			continue;
		}
		if (isstip && !range) {
			p--;
			break;
		}
		co[i] = strtol(p, (char **) NULL, 0);
		while (*p == 'x' || (*p >= '0' && *p <= '9')
		       || (*p >= 'a' && *p <= 'f') || (*p >= 'A'
						       && *p <= 'F'))
			p++;
	}
	if (iscopy < 0)
		co[3] = -co[3];
	if (isstip && anchor < 0)
		co[2] = ~co[2];
	if (isstip && anchor > 0)
		co[3] = ~co[3];
	for (i = 0; !iscopy && i < 4; i++) {
		if (*p != ' ' && *p != '\t') {
			fprintf(stderr, "syntax error 1: %s\n", pl);
			exit(1);
		}
		while (*p == ' ' || *p == '\t')
			p++;
		if (i == 1 && *p == 0)
			break;
		if (*p++ != '#') {
			fprintf(stderr, "syntax error 2: %s\n", pl);
			exit(1);
		}
		for (j = 0; j < 8; j++) {
			c = *p++;
			if (c >= '0' && c <= '9')
				c -= '0';
			else if (c >= 'a' && c <= 'f')
				c -= 'a' - 10;
			else if (c >= 'A' && c <= 'F')
				c -= 'A' - 10;
			else {
				fprintf(stderr, "syntax error 3: %s\n", pl);
				exit(1);
			}
			if (!(j & 1)) {
				d = c;
				continue;
			}
			c += d << 4;
			cols[i * 4 + (j >> 1)] = c;
			if (!isstip && j == 5 && (*p == 0 || *p == ' ' || *p == '\t')) {
				cols[i * 4 + 3] = 255;
				break;
			}
		}
		if (*p == 0 && i == 0) {
			i = 1;
			break;
		}
	}
	while (*p == ' ' || *p == '\t')
		p++;
	if (*p) {
		fprintf(stderr, "syntax error 4: %s\n", pl);
		exit(1);
	}
	boxcnt = *boxcntp;
	boxptr = *boxptrp;
	if (boxcnt % 64 == 0)
	  boxptr = *boxptrp = xrealloc(boxptr, 12 * (boxcnt + 64));
	if (i > 1 && (boxcnt + 1) % 64 == 0)
	  boxptr = *boxptrp = xrealloc(boxptr, 12 * (boxcnt + 65));
	if (isstip || iscopy) {
		isibox = noover = hidemax = 0;
	}
	bp = boxptr + 12 * boxcnt++;
	if (isibox)
		co[0] = ~co[0];
	if (i > 1) {
		co[1] = ~co[1];
		boxcnt++;
	}
	if (noover)
		co[2] = ~co[2];
	if (hidemax)
		co[3] = ~co[3];
	*bp++ = co[0] & 255;
	*bp++ = co[0] >> 8;
	*bp++ = co[1] & 255;
	*bp++ = co[1] >> 8;
	*bp++ = co[2] & 255;
	*bp++ = co[2] >> 8;
	*bp++ = co[3] & 255;
	*bp++ = co[3] >> 8;
	for (j = 0; j < i * 4; j++)
		*bp++ = cols[j];
	*boxcntp = boxcnt;
}

void parsefile(char *cfgfile, FILE *fp, int depth)
{
  char buf[256], *p;
  struct parstab *pt;
  int l;
  FILE *ifp;
  struct frame *curfr;
  char *args[256 + 1];
  int nargs;

  curfr = &mainframe;
  while (fgets(buf, sizeof(buf), fp))
    {
      l = strlen(buf);
      if (l == 0)
	continue;
      if (l == sizeof(buf) - 1)
	{
	  fprintf(stderr, "line too long\n");
	  exit(1);
	}
      buf[--l] = 0;
      while (l && (buf[l - 1] == ' ' || (buf[l - 1] == '\t')))
	buf[--l] = 0;
      for (p = buf; *p == ' ' || *p == '\t'; p++)
	;
      if (*p == 0 || *p == '#')
	continue;
      if (!strncmp(p, "includeprog", 11))
	{
	  p += 11;
	  if (depth > 16)
	    {
	      fprintf(stderr, "recursion limit reached\n");
	      exit(1);
	    }
	  while (*p == ' ' || *p == '\t')
	    p++;
	  if ((ifp = popen(p, "r")) == 0)
	    {
	      perror(p);
	      exit(1);
	    }
	  parsefile(p, ifp, depth + 1);
	  if ((l = pclose(ifp)))
	    {
	      fprintf(stderr, "%s: exit %d\n", p, l > 255 ? (l >> 8) : -l);
	      exit(1);
	    }
	  continue;
	}
      if (!strncmp(p, "include", 7))
	{
	  nargs = splitargs(p + 7, args, 256);
	  if (depth > 16)
	    {
	      fprintf(stderr, "recursion limit reached\n");
	      exit(1);
	    }
	  if ((ifp = fopen(args[0], "r")) == 0)
	    {
	      perror(args[0]);
	      exit(1);
	    }
	  parsefile(args[0], ifp, depth + 1);
	  fclose(ifp);
	  continue;
	}
      if (!strncmp(p, "mnganim", 7))
	{
	  struct boxanim *nba;
	  int i;

	  nba = xmalloc(sizeof(*nba));
	  memset(nba, 0, sizeof(*nba));
	  nba->init = -1;
	  nba->scale1 = nba->scale2 = 1;
	  nargs = splitargs(p + 7, args, 256);
	  if (nargs < 2)
	    {
	      fprintf(stderr, "mnganim: no filename or anim name\n");
	      exit(1);
	    }
	  nba->name = strdup(args[0]);
	  for (i = 2; i < nargs; )
	    {
	      if (!strcmp(args[i], "silent"))
		{
		  nba->silent = 1;
		  i++;
		  continue;
		}
	      if (!strcmp(args[i], "initframe") && i + 1 < nargs)
		{
		  nba->initframe = strdup(args[i + 1]);
		  i += 2;
		  continue;
		}
	      if (!strcmp(args[i], "center") && i + 2 < nargs)
		{
		  nba->posx = atoi(args[i + 1]);
		  nba->posy = atoi(args[i + 2]);
		  nba->posmode = POSMODE_CENTER;
		  i += 3;
		  continue;
		}
	      if (!strcmp(args[i], "scale") && i + 1 < nargs)
		{
		  p = strchr(args[i + 1], ':');
		  if (p)
		    *p = 0;
		  if (!p || (nba->scale1 = atoi(args[i + 1])) <= 0 || (nba->scale2 = atoi(p + 1)) <= 0) 
		    {
		      fprintf(stderr, "bad mnganim scale\n");
		      exit(1);
		    }
		  i += 2;
		  continue;
		}
	      if (!strcmp(args[i], "recolor") && i + 2 < nargs)
		{
		  int j, k, rc;
		  nba->hasrecolor = 1;
		  for (j = 0; j < 2; j++)
		    {
		      p = args[i + 1 + j];
		      if (*p++ != '#' || strlen(p) != 8)
			p = "";
		      rc = 0;
		      for (k = 0; k < 8; k++, p++)
			{
			  if (*p >= '0' && *p <= '9')
			    rc = (rc << 4) | (*p - '0');
			  else if (*p >= 'a' && *p <= 'f')
			    rc = (rc << 4) | (*p - ('a' - 10));
			  else if (*p >= 'A' && *p <= 'F')
			    rc = (rc << 4) | (*p - ('A' - 10));
			  else
			    {
			      fprintf(stderr, "bad recolor arg\n");
			      exit(1);
			    }
			  if (k % 2 == 1)
			    nba->recolor[k / 2 + 4 * j] = rc;
			}
		    }
		  i += 3;
		  continue;
		}
	      fprintf(stderr, "unknown mnganim subcommand: %s\n", args[i]);
	      exit(1);
	    }
	  nba->file = strdup(args[1]);
	  nba->id = nextanimid++;;
	  nba->next = boxanims;
	  boxanims = nba;
	  continue;
	}
      if (!strncmp(p, "trigger", 7))
	{
	  struct trigger *tr, *tr2;
	  nargs = splitargs(p + 8, args, 256);
	  if (nargs < 2)
	    {
	      fprintf(stderr, "too few arguments for trigger\n");
	      exit(1);
	    }
	  tr = xmalloc(sizeof(*tr));
	  memset(tr, 0, sizeof(*tr));
	  tr->name = strdup(args[0]);
	  if (!strcmp(args[1], "quit"))
	    tr->action = TRIGGER_QUIT;
	  else if (!strcmp(args[1], "stopanim") && nargs > 2)
	    {
	      struct boxanim *nba;
	      for (nba = boxanims; nba; nba = nba->next)
		if (!strcmp(nba->name, args[2]))
		  break;
	      if (!nba)
		{
		  fprintf(stderr, "trigger stopanim: unknown animation %s\n", args[2]);
		  exit(1);
		}
	      tr->anim = nba;
	      tr->action = TRIGGER_STOPANIM;
	    }
	  else if (!strcmp(args[1], "play") && nargs > 2)
	    {
	      struct boxanim *nba;
	      for (nba = boxanims; nba; nba = nba->next)
		if (!strcmp(nba->name, args[2]))
		  break;
	      if (!nba)
		{
		  fprintf(stderr, "trigger play: unknown animation %s\n", args[2]);
		  exit(1);
		}
	      tr->anim = nba;
	      tr->tag = 0;
	      if (nargs > 3)
		tr->tagstr = strdup(args[3]);
	      tr->action = TRIGGER_PLAY;
	    }
	  else if ((!strcmp(args[1], "system") || !strcmp(args[1], "systembg")) && nargs > 2)
	    {
	      tr->action = !strcmp(args[1], "system") ? TRIGGER_SYSTEM : TRIGGER_SYSTEMBG;
	      tr->shellcmd = strdup(args[2]);
	    }
	  else if (!strcmp(args[1], "tosilent"))
	    {
	      tr->action = TRIGGER_TOSILENT;
	    }
	  else if (!strcmp(args[1], "toverbose"))
	    {
	      tr->action = TRIGGER_TOVERBOSE;
	    }
	  else
	    {
	      fprintf(stderr, "trigger: unknown action %s\n", args[1]);
	      exit(1);
	    }
	  tr->next = 0;
	  if (!triggers)
	    triggers = tr;
	  else
	    for (tr2 = triggers; tr2; tr2 = tr2->next)
	      if (tr2->next == 0)
		{
		  tr2->next = tr;
		  break;
		}
	  continue;
	}
      if (!strncmp(p, "box", 3) && (!p[3] || p[3] == ' ' || p[3] == '\t'))
	{
	  parsebox(p, curfr);
	  continue;
	}
      for (pt = parstab; pt->name; pt++)
	{
	  l = strlen(pt->name);
	  if (!strncmp(p, pt->name, l) && (p[l] == ' ' || p[l] == '\t' || p[l] == '='))
	    break;
        }
      if (!pt->name)
	{
	  fprintf(stderr, "syntax error (unknown command): %s\n", p);
	  exit(1);
	}
      while (p[l] == ' ' || p[l] == '\t')
	l++;
      if (p[l++] != '=') 
	{
	  fprintf(stderr, "syntax error (missing '='): %s\n", p);
	  exit(1);
	}
      while (p[l] == ' ' || p[l] == '\t')
	l++;
      if (pt->val)
	*pt->val = strtol(p + l, (char **) NULL, 0);
      else
	*pt->str = strdup(p + l);
    }
}

void readanims()
{
  struct boxanim *ba;
  struct trigger *tr;
  int i;

  for (ba = boxanims; ba; ba = ba->next)
    {
      readmnganim(ba, ba->file);
      if (ba->initframe)
	{
	  for (i = 0; i < ba->nframes; i++)
	    if (ba->frames[i].name && !strcmp(ba->frames[i].name, ba->initframe))
	      break;
	  if (i == ba->nframes)
	    {
	      fprintf(stderr, "mnganim: bad initframe '%s'\n", ba->initframe);
	      exit(1);
	    }
	  ba->init = i;
	}
      ba->boxcount = 0;
      ba->sboxcount = 0;
      for (i = 0; i < ba->nframes; i++)
	{
	  fixposandcolor(ba, ba->frames[i].boxes, ba->frames[i].boxcount);
	  fixposandcolor(ba, ba->frames[i].sboxes, ba->frames[i].sboxcount);
	  if (ba->frames[i].boxcount > ba->boxcount)
	    ba->boxcount = ba->frames[i].boxcount;
	  if (ba->frames[i].sboxcount > ba->sboxcount)
	    ba->sboxcount = ba->frames[i].sboxcount;
	}
      /* reserve room for animbox marker */
      if (ba->boxcount)
	ba->boxcount++;
      if (ba->sboxcount)
	ba->sboxcount++;
    }
  for (tr = triggers; tr; tr = tr->next)
    {
      if (tr->action != TRIGGER_PLAY)
	continue;
      ba = tr->anim;
      if (tr->tagstr)
	{
	  for (i = 0; i < ba->nframes; i++)
	    if (ba->frames[i].name && !strcmp(ba->frames[i].name, tr->tagstr))
	      break;
	  if (i == ba->nframes)
	    {
	      fprintf(stderr, "trigger play: bad frame '%s'\n", tr->tagstr);
	      exit(1);
	    }
	  tr->tag = i;
	}
    }
}

void fillempty(unsigned char *b, int s, int c)
{
  b += s * 12;
  for (; c > 0; c--)
    {
      *b++ = 0xff;
      *b++ = 0x7f;
      *b++ = 0;
      *b++ = 0;
      *b++ = 0xff;
      *b++ = 0x7f;
      *b++ = 0;
      *b++ = 0;
      *b++ = 0;
      *b++ = 0;
      *b++ = 0;
      *b++ = 0;
    }
}

void
mkanimbox(unsigned char *bp, int off, int id, int len)
{
  bp += off * 12;
  *bp++ = 0xff;
  *bp++ = 0x7f;
  *bp++ = 0xfd;
  *bp++ = 0x7f;
  *bp++ = id & 255;
  *bp++ = id >> 8;
  *bp++ = len & 255;
  *bp++ = len >> 8;
  *bp++ = 0;
  *bp++ = 0;
  *bp++ = 0;
  *bp = 0;
}

void readconfig(char *cfgfile)
{
  FILE *fp;

  if (!cfgfile)
    {
      fprintf(stderr, "Please supply a config file\n");
      exit(1);
    }
  if ((fp = fopen(cfgfile, "r")) == 0)
    {
      perror(cfgfile);
      exit(1);
    }
  parsefile(cfgfile, fp, 0);
  fclose(fp);
}

void finishconfig()
{
  struct boxanim *ba;
  struct frame *fr;
  int i;

  readanims();
  fixboxcopy(mainframe.boxes, mainframe.boxcount);
  fixboxcopy(mainframe.sboxes, mainframe.sboxcount);
  for (ba = boxanims; ba; ba = ba->next)
    {
      fr = ba->init != -1 ? ba->frames + ba->init : 0;
      ba->boxstart = mainframe.boxcount;
      /* boxcount/sboxcount already include room for the animbox */
      if (ba->boxcount)
	{
	  i = mainframe.boxcount + ba->boxcount;
	  mainframe.boxes = xrealloc(mainframe.boxes, ((i + 63) & ~63) * 12);
	  mainframe.boxcount = i;
	  mkanimbox(mainframe.sboxes, ba->boxstart, ba->id, ba->boxcount - 1);
	  if (fr && fr->boxcount)
	    {
	      memcpy(mainframe.boxes + 12 * (ba->boxstart + 1), fr->boxes, fr->boxcount * 12);
	      fillempty(mainframe.boxes, ba->boxstart + 1 + fr->boxcount, ba->boxcount - fr->boxcount - 1);
	    }
	  else
	    fillempty(mainframe.boxes, ba->boxstart + 1, ba->boxcount - 1);
	  // fprintf(stderr, "anim %s: reserved %d boxes\n", ba->name, ba->boxcount);
	}
      ba->sboxstart = mainframe.sboxcount;
      if (ba->sboxcount)
	{
	  i = mainframe.sboxcount + ba->sboxcount;
	  mainframe.sboxes = xrealloc(mainframe.sboxes, ((i + 63) & ~63) * 12);
	  mainframe.sboxcount = i;
	  mkanimbox(mainframe.sboxes, ba->sboxstart, ba->id, ba->sboxcount - 1);
	  if (fr && fr->sboxcount)
	    {
	      memcpy(mainframe.sboxes + 12 * (ba->sboxstart + 1), fr->sboxes, fr->sboxcount * 12);
	      fillempty(mainframe.sboxes, ba->sboxstart + 1 + fr->sboxcount, ba->sboxcount - fr->sboxcount - 1);
	    }
	  else
	    fillempty(mainframe.sboxes, ba->sboxstart + 1, ba->sboxcount - 1);
	  // fprintf(stderr, "anim %s: reserved %d sboxes\n", ba->name, ba->sboxcount);
	}
    }
  configready = 1;
}

void setsplash(char *cfgfile, int filterflag, int verbose_only)
{
	struct stat stb;
	char *p;
	int l, size, fd;
	FILE *fp = 0;
	FILE *sfp = 0;
	unsigned char *pic;
	int silentl = 0;

	if (cfgfile) {
		readconfig(cfgfile);
		finishconfig();
		if (!jpgfile) {
			l = strlen(cfgfile);
			if (l > 4 && !strcmp(cfgfile + l - 4, ".cfg")) {
				jpgfile = strdup(cfgfile);
				strcpy(jpgfile + l - 4, ".jpg");
				p = strstr(jpgfile, "config");
				if (p)
					strncpy(p, "images", 6);
			}
		}
		if (tx == 0 && ty == 0 && tw == 0 && th == 0) {
			tx = px + 8;
			ty = py + 4;
			tw = pw - 16;
			th = ph - 8;
		}

		if (jpgfile
		    && (*jpgfile == 0 || tx < 0 || ty < 0 || tw <= 0
			|| th <= 0)) {
			free(jpgfile);
			jpgfile = 0;
		}
	}

	if (verbose_only)
		silentjpgfile = NULL;


	if (silentjpgfile && jpgfile) {
		if ((sfp = fopen(silentjpgfile, "r")) == 0) {
			perror(silentjpgfile);
			exit(1);
		}
		if (fstat(fileno(sfp), &stb) == -1) {
			perror("fstat");
			exit(1);
		}
		silentl = stb.st_size;
	}

	if (jpgfile) {
		if ((fp = fopen(jpgfile, "r")) == 0) {
			perror(jpgfile);
			exit(1);
		}
		if (fstat(fileno(fp), &stb) == -1) {
			perror("fstat");
			exit(1);
		}
		l = stb.st_size;
		pic = calloc(47 + l + silentl + mainframe.boxcount * 12 + mainframe.sboxcount * 12, 1);
		if (pic == 0) {
			fprintf(stderr, "Out of memory.\n");
			exit(1);
		}
		if (fread(pic + 47 + mainframe.boxcount * 12, l, 1, fp) != 1) {
			perror("fread");
			exit(1);
		}
		if (silentjpgfile) {
			if (fread
			    (pic + 47 + l + mainframe.boxcount * 12 + mainframe.sboxcount * 12,
			     silentl, 1, sfp) != 1) {
				perror("fread");
				exit(1);
			}
			fclose(sfp);
			if (mainframe.sboxcount)
				memmove(pic + 47 + l + mainframe.boxcount * 12,
					mainframe.sboxes, mainframe.sboxcount * 12);
			silentl += mainframe.sboxcount * 12;
			l += silentl;
		}
		fclose(fp);
	} else {
		l = 0;
		pic = calloc(47, 1);
		if (pic == 0) {
			fprintf(stderr, "Out of memory.\n");
			exit(1);
		}
	}
	size = l;
	if (version < 1 || version > 3) {
		fprintf(stderr, "Illegal version: %ld\n", version);
		exit(1);
	}
	if (version == 1 && unit != 0) {
		/* convert to version 2 */
		tx *= 8;
		ty *= 16;
		tw *= 8;
		th *= 16;
		px = tx + 10;
		py = ty + 10;
		pw = tw - 20;
		ph = th - 20;
		pr = pg = pb = 240;
		state = 1;
		fgcolor = 0;
		bgcolor = 15;
		version = 2;
	}
	if (version == 1) {
		/* write version 1 file */
		strcpy(pic, "BOOTSPL1");
		pic[8] = tx;
		pic[9] = tx >> 8;
		pic[10] = ty;
		pic[11] = ty >> 8;
		pic[12] = tw;
		pic[13] = tw >> 8;
		pic[14] = th;
		pic[15] = th >> 8;
		pic[16] = size;
		pic[17] = size >> 8;
		pic[18] = size >> 16;
		pic[19] = size >> 24;
		if (l > 0)
			memmove(pic + 20, pic + 47, l);
		l += 20;
	} else if (version == 2) {
		/* write version 2 file */
		strcpy(pic, "BOOTSPL2");
		pic[8] = unit;
		pic[9] = state;
		pic[10] = fgcolor;
		pic[11] = bgcolor;
		if (cfgfile && size == 0
		    && (state != -1 || fgcolor != -1 || bgcolor != -1))
			size = -1;
		pic[12] = size;
		pic[13] = size >> 8;
		pic[14] = size >> 16;
		pic[15] = size >> 24;
		pic[16] = tx;
		pic[17] = tx >> 8;
		pic[18] = ty;
		pic[19] = ty >> 8;
		pic[20] = tw;
		pic[21] = tw >> 8;
		pic[22] = th;
		pic[23] = th >> 8;
		pic[24] = px;
		pic[25] = px >> 8;
		pic[26] = py;
		pic[27] = py >> 8;
		pic[28] = pw;
		pic[29] = pw >> 8;
		pic[30] = ph;
		pic[31] = ph >> 8;
		pic[32] = pr;
		pic[33] = pg;
		pic[34] = pb;
		if (l > 0)
			memmove(pic + 35, pic + 47, l);
		l += 35;
	} else {		/* version 3 */
		strcpy(pic, "BOOTSPL3");
		pic[8] = unit;
		pic[9] = state;
		pic[10] = fgcolor;
		pic[11] = bgcolor;
		size += mainframe.boxcount * 12;
		if (cfgfile && size == 0
		    && (state != -1 || fgcolor != -1 || bgcolor != -1))
			size = -1;
		pic[12] = size;
		pic[13] = size >> 8;
		pic[14] = size >> 16;
		pic[15] = size >> 24;
		pic[16] = tx;
		pic[17] = tx >> 8;
		pic[18] = ty;
		pic[19] = ty >> 8;
		pic[20] = tw;
		pic[21] = tw >> 8;
		pic[22] = th;
		pic[23] = th >> 8;
		pic[24] = mainframe.boxcount;
		pic[25] = mainframe.boxcount >> 8;
		pic[28] = silentl;
		pic[29] = silentl >> 8;
		pic[30] = silentl >> 16;
		pic[31] = silentl >> 24;
		pic[32] = mainframe.sboxcount;
		pic[33] = mainframe.sboxcount >> 8;
		pic[34] = percent;
		pic[35] = percent >> 8;
		pic[36] = overpaintok;
		pic[37] = 0;	/* numpalette */
		if (l > 0)
			memmove(pic + 38 + mainframe.boxcount * 12,
				pic + 47 + mainframe.boxcount * 12, l);
		if (mainframe.boxcount)
			memmove(pic + 38, mainframe.boxes, mainframe.boxcount * 12);
		l += 38 + mainframe.boxcount * 12;
	}

	if (filterflag)
		fd = 1;
	else if ((fd = open(PROCSPLASH, O_WRONLY)) == -1) {
		perror(PROCSPLASH);
		exit(1);
	}
	if (write(fd, pic, l) != l) {
		perror("write");
		exit(1);
	}
	if (!filterflag)
		close(fd);
	free(pic);
}

void setframe(struct boxanim *ba, struct frame *fr)
{
  unsigned char *splashme;
  unsigned char *p;

  if (!ba->boxcount && !ba->sboxcount)
    return;
  splashme = xmalloc(38 + 4 + (ba->boxcount + ba->sboxcount) * 12);
  memset(splashme, 0, 38);
  strcpy(splashme, "BOOTSPL3");
  splashme[8] = 0;
  splashme[9] = 255;
  splashme[10] = 255;
  splashme[11] = 255;

  splashme[12] = 255;
  splashme[13] = 255;
  splashme[14] = 255;
  splashme[15] = 255;

  splashme[24] = ba->boxcount & 255;
  splashme[25] = ba->boxcount >> 8;

  splashme[32] = ba->sboxcount & 255;
  splashme[33] = ba->sboxcount >> 8;

  splashme[36] = 1;
  p = splashme + 38;
  /* boxcount/sboxcount already include room for the animbox */
  if (ba->boxcount)
    {
      *p++ = ba->boxstart & 255;
      *p++ = ba->boxstart >> 8;
      mkanimbox(p, 0, ba->id, ba->boxcount - 1);
      if (fr->boxcount)
        memcpy(p + 12, fr->boxes, fr->boxcount * 12);
      fillempty(p + 12, fr->boxcount, ba->boxcount - fr->boxcount - 1);
      p += ba->boxcount * 12;
    }
  if (ba->sboxcount)
    {
      *p++ = ba->sboxstart & 255;
      *p++ = ba->sboxstart >> 8;
      mkanimbox(p, 0, ba->id, ba->sboxcount - 1);
      if (fr->sboxcount)
        memcpy(p + 12, fr->sboxes, fr->sboxcount * 12);
      fillempty(p + 12, fr->sboxcount, ba->sboxcount - fr->sboxcount - 1);
      p += ba->sboxcount * 12;
    }
  write(splashfd, splashme, p - splashme);
  free(splashme);
}

void nextframe(struct boxanim *ba)
{
  struct frame *fr;
  struct event *ev;

  for (;;)
    {
      fr = &ba->frames[ba->cframe];
      setframe(ba, fr);
      ba->cframe = ba->cframe + 1;
      if (fr->action)
	{
	  if (fr->action == FRAME_STOP)
	    {
	      if (ba->cframe == ba->nframes)
		ba->cframe = 0;
	      return;
	    }
	  if (fr->action == FRAME_JUMP)
	    ba->cframe = fr->tag;
	}
      if (ba->cframe == ba->nframes)
	{
	  ba->cframe = 0;
	  return;
	}
      if (fr->delay)
	break;
    }
  ev = xmalloc(sizeof(struct event));
  ev->type = EVENT_ANIM;
  ev->ba = ba;
  ev->due = now + fr->delay;
  ev->next = events;
  events = ev;
}

int getsplashstate(void)
{
  char readbuffer[BUFSIZE];

  memset(readbuffer, 0, sizeof(readbuffer));
  lseek(splashfd, 0, SEEK_SET);
  if (read(splashfd, readbuffer, sizeof(readbuffer) - 1) < 0 || !strstr(readbuffer, "silent"))
    return 0;
  return 1;
}

int percentcur;
int percentcurset;
int percentgoal;
int percentnum;
int percenthaveev;
int percentdelay = 1000 / 20;

void percentset(int percent)
{
  char buf[80];
  sprintf(buf, "show %d\n", percent);
  write(splashfd, buf, strlen(buf));
  percentcur = percent;
  percentcurset = 1;
}

void percentsetgoal(int percent)
{
  struct event *ev;
  if (percent < 0 || percent > 65535)
    return;
  percentgoal = percent;
  percentnum = 20;
  if (!percenthaveev)
    {
      ev = xmalloc(sizeof(struct event));
      ev->type = EVENT_PERCENT;
      ev->ba = 0;
      ev->due = now + percentdelay;
      ev->next = events;
      events = ev;
      percenthaveev = 1;
    }
}

void nextpercent()
{
  struct event *ev;
  int step;

  if (percentnum <= 1)
    percentnum = 1;
  step = (percentgoal - percentcur) / percentnum;
  percentcur += step;
  percentnum--;
  percentset(percentcur);
  if (percentcur != percentgoal)
    {
      ev = xmalloc(sizeof(struct event));
      ev->type = EVENT_PERCENT;
      ev->ba = 0;
      ev->due = now + percentdelay;
      ev->next = events;
      events = ev;
      percenthaveev = 1;
    }
  else
    percenthaveev = 0;
}

char *parsepercent(char *trigger)
{
  char *p, *s;
  char *perstr;

  if (strncmp(trigger, "percent ", 8))
    return trigger;
  p = trigger + 8;
  while (*p == ' ' || *p == '\t')
    p++;
  if (!*p)
    return p;
  perstr = p;
  while (*p && *p != ' ' && *p != '\t')
    p++;
  if (*p)
    *p++ = 0;
  while (*p == ' ' || *p == '\t')
    p++;
  if ((s = strchr(perstr, ':')) != 0)
    {
      int cur;
      cur = atoi(perstr);
      if (cur < 0 || cur > 65535)
	cur = 0;
      if (!percentcurset)
	{
	  percentcur = cur;
	  percentcurset = 1;
	}
      percentsetgoal(cur + atoi(s + 1));
    }
  else
    {
      percentgoal = atoi(perstr);
      percentset(percentgoal);
    }
  return p;
}

void runtrigger(char *trname)
{
  struct trigger *tr;
  if (!trname || *trname == 0)
    return;
  // printf("runtrigger %s\n", trname);
  if (!strcmp(trname, "quit"))
    {
      if (percenthaveev && percentcur != percentgoal)
        percentset(percentgoal);
      exit(0);
    }
  for (tr = triggers; tr; tr = tr->next)
    {
      if (strcmp(tr->name, trname))
	continue;
      // printf("found, action = %d\n", tr->action);
      if (tr->action == TRIGGER_QUIT)
	{
	  if (percenthaveev && percentcur != percentgoal)
	    percentset(percentgoal);
	  exit(0);
	}
      if (tr->action == TRIGGER_PLAY)
	{
	  struct boxanim *ba = tr->anim;
	  int state = splashstate;
	  if (state == -1)
	    state = getsplashstate();
	  if (ba->silent == state)
	    {
	      if (!configready)
		finishconfig();
	      ba->cframe = tr->tag;
	      nextframe(ba);
	    }
	}
      else if (tr->action == TRIGGER_STOPANIM)
	{
	  struct event *ev, **evp;
	  for (evp = &events; (ev = *evp) != 0; )
	    {
	      if (ev->type != EVENT_ANIM || ev->ba != tr->anim)
		{
		  evp = &ev->next;
		  continue;
		}
	      *evp = ev->next;
	      ev->next = 0;
	      free(ev);
	    }
	}
      else if (tr->action == TRIGGER_SYSTEM)
	system(tr->shellcmd);
      else if (tr->action == TRIGGER_SYSTEMBG)
	{
	  if (fork() == 0)
	    {
	      setsid();
	      system(tr->shellcmd);
	      _exit(0);
	    }
	}
      else if (tr->action == TRIGGER_TOSILENT)
	{
	  write(splashfd, "silent\n", 7);
	}
      else if (tr->action == TRIGGER_TOVERBOSE)
	{
	  write(splashfd, "verbose\n", 8);
	}
    }
}

void checksilent(void)
{
  int state;
  struct boxanim *ba;
  struct event *ev, **evp;

  state = getsplashstate();
  if (splashstate == state)
    return;
  splashstate = state;
  runtrigger(splashstate ? "silent" : "verbose");

  /* stop all animations that are on the wrong plane */
  for (evp = &events; (ev = *evp) != 0; )
    {
      if (ev->type == EVENT_ANIM)
	{
	  ba = ev->ba;
	  if (ba->silent != splashstate)
	    {
	      *evp = ev->next;
	      ev->next = 0;
	      free(ev);
	      continue;
	    }
	}
      evp = &ev->next;
    }
}

void server(int s, char *cfg, char *trigger, int oneshot)
{
  struct event *ev, *next, **evp;
  int i, w;
  fd_set r;
  unsigned char p[257];
  int watchsilent = 0;
  struct trigger *tr;
  struct timeval tv;

  splashfd = open(PROCSPLASH, O_RDWR);
  if (splashfd == -1)
    {
      perror(PROCSPLASH);
      exit(1);
    }
  if (trigger)
    trigger = parsepercent(trigger);

  if (trigger && !strcmp(trigger, "quit"))
    runtrigger(trigger);

  readconfig(cfg);

  for (tr = triggers; tr; tr = tr->next)
    if (!strcmp(tr->name, "silent") || !strcmp(tr->name, "verbose"))
      watchsilent = 1;
  for (tr = triggers; tr; tr = tr->next)
    if (tr->action == TRIGGER_PLAY)
      watchsilent = 1;
  /* do this first before calling checksilent */
  for (tr = triggers; tr; tr = tr->next)
    if (trigger && !strcmp(tr->name, trigger) && (tr->action == TRIGGER_TOSILENT || tr->action == TRIGGER_TOVERBOSE))
      {
	runtrigger(trigger);
	trigger = 0;
	break;
      }
  if (!oneshot)
    checksilent();
  if (trigger)
    runtrigger(trigger);
  if (oneshot)
    runtrigger("quit");
  for (;;)
    {
      if (watchsilent)
	checksilent();
      next = 0;
      for (ev = events; ev; ev = ev->next)
	{
	  if (!next)
	    next = ev;
	  else if (ev->due < next->due)
	    next = ev;
	}
      if (next)
	{
	  if (next->due <= now)
	    {
	      for (evp = &events; (ev = *evp) != 0; evp = &ev->next)
		if (ev == next)
		  break;
	      *evp = ev->next;
	      ev->next = 0;
	      if (ev->type == EVENT_ANIM)
	        nextframe(ev->ba);
	      else
		nextpercent();
	      free(ev);
	      continue;
	    }
          tv.tv_sec = (next->due - now) / 1000;
          tv.tv_usec = ((next->due - now) % 1000) * 1000;
	}
      else
	{
	  tv.tv_sec = tv.tv_usec = 0;
	}
      FD_ZERO(&r);
      FD_SET(s, &r);
      if ((i = select(s + 1, &r, 0, 0, &tv)) == -1)
	{
	  if (errno == EINTR)
	    continue;
	  perror("select");
	  exit(1);
	}
      if (i == 0 || !FD_ISSET(s, &r))
	{
	  if (!next)
	    break;
	  now = next->due;
	  continue;
	}
      if (next)
        {
          w = tv.tv_sec * 1000 + (tv.tv_usec + 500) / 1000;
	  if (next->due - w > now)
	    now = next->due - w;
        }
      if (read(s, p, 1) != 1)
	continue;
      if (read(s, p + 1, *p + 1) != *p + 1 || p[*p + 1] != 0)
	continue;
      trigger = p + 1;
      trigger = parsepercent(trigger);
      runtrigger(trigger);
    }
  exit(0);
}

void sendtoserver(char *cfg, char *trigger, int oneshot, int nosetsid)
{
  int s;
  int l;
  char p[257];
  struct flock fl;
  pid_t pid;
  int waitpipe[2];

  l = trigger ? strlen(trigger) : 0;
  if (l > 255)
    {
      fprintf(stderr, "trigger length too big\n");
      exit(1);
    }
  if (!oneshot)
    {
      if (pipe(waitpipe) == -1)
	{
	  perror("pipe");
	  exit(1);
	}
      pid = fork();
      if (pid == (pid_t)-1)
	{
	  perror("fork");
	  exit(1);
	}
      if (pid)
	{
	  char wbuf;
	  close(waitpipe[1]);
	  read(waitpipe[0], &wbuf, 1);
	  close(waitpipe[0]);
	  exit(0);
	}
      close(waitpipe[0]);
      if (!nosetsid)
        setsid();
    }
  if ((s = open("/var/run/bootsplashctl", O_RDWR)) == -1)
    {
      perror("/var/run/bootsplashctl");
      exit(1);
    }
  memset(&fl, 0, sizeof(fl));
  fl.l_type = F_WRLCK;
  fl.l_whence = SEEK_SET;
  fl.l_start = 0;
  fl.l_len = 0;
  if (fcntl(s, F_SETLK, &fl) == 0)
    {
      if (!oneshot)
        close(waitpipe[1]);
      server(s, cfg, trigger, oneshot);
    }
  if (errno != EAGAIN)
    {
      perror("fcntl");
      exit(1);
    }
  if (trigger)
    {
      p[0] = l;
      strcpy(p + 1, trigger);
      signal(SIGPIPE, SIG_IGN);
      if (write(s, p, l + 2) != l + 2)
	{
	  perror("write");
	  exit(1);
	}
    }
  if (!oneshot)
    close(waitpipe[1]);
  else
    {
      if (write(s, "\004quit", 6) != 6)
	{
	  perror("write");
	  exit(1);
	}
      fcntl(s, F_SETLKW, &fl);
    }
  close(s);
  exit(0);
}

void usage(int x)
{
  if (x)
    fprintf(stderr, "Usage: splash -s [-u unit] -n [cfgfile]\n");
  else
    printf("Usage: splash -s [-u unit] -n [cfgfile]\n");
  exit(x);
}

int main(int argc, char *argv[])
{
  int c;
  int setflag = 0;
  int filterflag = 0;
  int verbose_only = 0;
  int oneshot  = 0;
  char *trigger = 0;
  char readbuffer[BUFSIZE];
  char *perstr = 0;
  char *cfg = 0;
  int nosetsid = 0;

  while ((c = getopt(argc, argv, "hp:t:fsu:nqS")) != EOF)
    {
      switch (c)
	{
	case 'S':
	  nosetsid = 1;
	  break;
	case 's':
	  setflag = 1;
	  break;
	case 'f':
	  filterflag = 1;
	  break;
	case 'q':
	  oneshot = 1;
	  break;
	case 'u':
	  unit = atoi(optarg);
	  break;
	case 't':
	  trigger = optarg;
	  break;
	case 'p':
	  perstr = optarg;
	  break;
	case 'n':
	  verbose_only = 1;
	  break;
	case 'h':
	  usage(0);
	  break;
	default:
	  usage(1);
	  break;
	}
    }
  if (argc == optind)
    {
      if (!setflag && (trigger || perstr || oneshot))
	cfg = getenv("SPLASHCFG");
      if (!setflag && !cfg)
	{
	  fprintf(stderr, "Please supply a config file\n");
	  exit(1);
	}
    }
  else if (argc == optind + 1)
    cfg = argv[optind];
  else
    usage(1);
  if (perstr)
    {
#if 0
      char *ps;
      int percent;
      if ((ps = strchr(perstr, ':')) != 0)
        percent = atoi(perstr) + atoi(ps + 1);
      else
        percent = atoi(perstr);
      splashfd = open(PROCSPLASH, O_RDWR);
      if (splashfd < 0)
	{
	  perror(PROCSPLASH);
	  exit(1);
	}
      percentset(percent);
      close(splashfd);
      if (!trigger)
	exit(0);
#else
      char *newtrigger;
      newtrigger = xmalloc(8 + strlen(perstr) + (trigger ? strlen(trigger) + 1 : 0) + 1);
      if (trigger)
	sprintf(newtrigger, "percent %s %s", perstr, trigger);
      else
	sprintf(newtrigger, "percent %s", perstr);
      trigger = newtrigger;
#endif
    }
  if (trigger || oneshot)
    {
      sendtoserver(cfg, trigger, oneshot, nosetsid);
      /* NOTREACHED */
    }
  setflag = 1;	/* nowadays default */
  if (!filterflag) 
    {
      splashfd = open(PROCSPLASH, O_RDWR);
      if (splashfd  < 0)
	{
	  perror(PROCSPLASH);
	  exit(1);
	}
      if (read(splashfd, readbuffer, sizeof(readbuffer)) < 0)
	{
	  perror("read");
	  exit(1);
        }
      if (!strncmp(readbuffer, "Splash screen v0", 16))
	{
	  fprintf(stderr, "Your kernel support for the bootsplash is too old. Get the new version from http://www.bootsplash.org/\n");
	  exit(1);
        }
    }
  setsplash(cfg, filterflag, verbose_only);
  exit(0);
}
