/*
 * Compatiblity Header for compilation working across multiple kernels
 *
 * This code is based on drivers/scsi/mpt2sas/mpt2sas_compatibility.h
 * Copyright (C) 2008-2010  LSI Corporation
 *  (mailto:DL-MPTFusionLinux@lsi.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * NO WARRANTY
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is
 * solely responsible for determining the appropriateness of using and
 * distributing the Program and assumes all risks associated with its
 * exercise of rights under this Agreement, including but not limited to
 * the risks and costs of program errors, damage to or loss of data,
 * programs or equipment, and unavailability or interruption of operations.

 * DISCLAIMER OF LIABILITY
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
 * USA.
 */

#ifndef FUSION_LINUX_COMPAT_H
#define FUSION_LINUX_COMPAT_H

#include <linux/version.h>
#include <scsi/scsi_device.h>
#include <scsi/scsi_cmnd.h>

extern int scsi_internal_device_block(struct scsi_device *sdev);
extern int scsi_internal_device_unblock(struct scsi_device *sdev);

#ifndef DID_TRANSPORT_DISRUPTED
#define DID_TRANSPORT_DISRUPTED DID_BUS_BUSY
#endif

#ifndef ULLONG_MAX
#define ULLONG_MAX      (~0ULL)
#endif

#ifndef USHORT_MAX
#define USHORT_MAX      ((u16)(~0U))
#endif

#define sdev_printk(prefix, sdev, fmt, ...) \
	printk(prefix " sd %d:%d:%d:%d: " fmt, sdev->host->host_no, \
		sdev->channel, sdev->id, sdev->lun, ##__VA_ARGS__)

static inline void *shost_private(struct Scsi_Host *shost)
{
	return (void *)shost->hostdata;
}

static inline sector_t scsi_get_lba(struct scsi_cmnd *scmd)
{
	return scmd->request->sector;
}

/**
 * mpt_scsi_build_sense_buffer - build sense data in a buffer
 * @desc:	Sense format (non zero == descriptor format,
 * 		0 == fixed format)
 * @buf:	Where to build sense data
 * @key:	Sense key
 * @asc:	Additional sense code
 * @ascq:	Additional sense code qualifier
 *
 **/
static inline void mpt_scsi_build_sense_buffer(int desc, u8 *buf, u8 key,
    u8 asc, u8 ascq)
{
	if (desc) {
		buf[0] = 0x72;	/* descriptor, current */
		buf[1] = key;
		buf[2] = asc;
		buf[3] = ascq;
		buf[7] = 0;
	} else {
		buf[0] = 0x70;	/* fixed, current */
		buf[2] = key;
		buf[7] = 0xa;
		buf[12] = asc;
		buf[13] = ascq;
	}
}

/**
 * _scsilun_to_int: convert a scsi_lun to an int
 * @scsilun:    struct scsi_lun to be converted.
 *
 * Description:
 *     Convert @scsilun from a struct scsi_lun to a four byte host byte-ordered
 *     integer, and return the result. The caller must check for
 *     truncation before using this function.
 *
 * Notes:
 *     The struct scsi_lun is assumed to be four levels, with each level
 *     effectively containing a SCSI byte-ordered (big endian) short; the
 *     addressing bits of each level are ignored (the highest two bits).
 *     For a description of the LUN format, post SCSI-3 see the SCSI
 *     Architecture Model, for SCSI-3 see the SCSI Controller Commands.
 *
 *     Given a struct scsi_lun of: 0a 04 0b 03 00 00 00 00, this function
 *     returns the integer: 0x0b030a04
 **/
static inline int _scsilun_to_int(struct scsi_lun *scsilun)
{
	int i;
	unsigned int lun;

	lun = 0;
	for (i = 0; i < sizeof(lun); i += 2)
		lun = lun | (((scsilun->scsi_lun[i] << 8) |
			scsilun->scsi_lun[i + 1]) << (i * 8));
	return lun;
}

/**
 * _int_to_scsilun: reverts an int into a scsi_lun
 * @lun:        integer to be reverted
 * @scsilun:    struct scsi_lun to be set.
 *
 * Description:
 *     Reverts the functionality of the scsilun_to_int, which packed
 *     an 8-byte lun value into an int. This routine unpacks the int
 *     back into the lun value.
 *     Note: the scsilun_to_int() routine does not truly handle all
 *     8bytes of the lun value. This functions restores only as much
 *     as was set by the routine.
 *
 * Notes:
 *     Given an integer : 0x0b030a04,  this function returns a
 *     scsi_lun of : struct scsi_lun of: 0a 04 0b 03 00 00 00 00
 *
 **/
static inline void _int_to_scsilun(unsigned int lun, struct scsi_lun *scsilun)
{
	int i;

	memset(scsilun->scsi_lun, 0, sizeof(scsilun->scsi_lun));

	for (i = 0; i < sizeof(lun); i += 2) {
		scsilun->scsi_lun[i] = (lun >> 8) & 0xFF;
		scsilun->scsi_lun[i+1] = lun & 0xFF;
		lun = lun >> 16;
	}
}

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,8))
extern void print_Scsi_Cmnd(struct scsi_cmnd *cmd);
#else
extern void scsi_print_command(struct scsi_cmnd *cmd);
#endif
static inline void mpt_scsi_print_command(struct scsi_cmnd *cmd)
{
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,8))
	print_Scsi_Cmnd(cmd);
#else
	scsi_print_command(cmd);
#endif
}

#ifndef ARCH_HAS_DMA_GET_REQUIRED_MASK
static inline u64 _dma_get_required_mask(struct device *dev)
{
	struct sysinfo	system_data;
	unsigned long totalram;		/* Total usable main memory size */
#define GB_4_in_Kb	(4 * 1024 * 1024)

	si_meminfo(&system_data);
	totalram = system_data.totalram << (PAGE_SHIFT - 10);

	if (totalram <= GB_4_in_Kb)
		return DMA_32BIT_MASK;
	else
		return DMA_64BIT_MASK;
}
#endif

/* define wait_event_timeout which came in lk 2.6.9
 * to be backward compatible to older variants of lk 2.6
 */
#ifndef wait_event_timeout
#define __wait_event_timeout(wq, condition, ret) 			\
do {									\
	DEFINE_WAIT(__wait);						\
									\
	for (;;) {							\
		prepare_to_wait(&wq, &__wait, TASK_UNINTERRUPTIBLE);	\
		if (condition)						\
			break;						\
		ret = schedule_timeout(ret);				\
		if (!ret)						\
			break;						\
	}								\
	finish_wait(&wq, &__wait);					\
} while (0)

#define wait_event_timeout(wq, condition, timeout)			\
({									\
	long __ret = timeout;						\
	if (!(condition)) 						\
		__wait_event_timeout(wq, condition, __ret);		\
	__ret;								\
})
#endif


#endif /* FUSION_LINUX_COMPAT_H */
