/*
 * Device Mapping API
 *
 * This code is based on drivers/scsi/mpt2sas/mpt2sas_mapping.c
 * Copyright (C) 2008-2010  LSI Corporation
 *  (mailto:DL-MPTFusionLinux@lsi.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * NO WARRANTY
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is
 * solely responsible for determining the appropriateness of using and
 * distributing the Program and assumes all risks associated with its
 * exercise of rights under this Agreement, including but not limited to
 * the risks and costs of program errors, damage to or loss of data,
 * programs or equipment, and unavailability or interruption of operations.

 * DISCLAIMER OF LIABILITY
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
 * USA.
 */

#include <linux/version.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/blkdev.h>
#include <linux/sched.h>
#include <linux/workqueue.h>
#include <linux/delay.h>
#include <linux/pci.h>
#include <linux/interrupt.h>

#include "mpt2sas_base.h"

/**
 * struct _map_phy_change - PHY entries recieved in Topology change list
 * @physical_id: SAS address of the device attached with the associate PHY
 * @device_info: bitfield provides detailed info about the device
 * @dev_handle: device handle for the device pointed by this entry
 * @slot: slot ID
 * @is_processed: Flag to indicate whether this entry is processed or not
 */
struct _map_phy_change {
	u64	physical_id;
	u32	device_info;
	u16	dev_handle;
	u16	slot;
	u8	reason;
	u8	is_processed;
};

/**
 * struct _map_topology_change - entries to be removed from mapping table
 * @dpm_entry_num: index of this device in device persistent map table
 * @dev_handle: device handle for the device pointed by this entry
 */
struct _map_topology_change {
	u16	enc_handle;
	u16	exp_handle;
	u8	num_entries;
	u8	start_phy_num;
	u8	num_phys;
	u8	exp_status;
	struct _map_phy_change *phy_details;
};

/**
 * _mapping_clear_entry - Clear a particular mapping entry.
 * @map_entry: map table entry
 *
 * Returns nothing.
 */
static inline void
_mapping_clear_map_entry(struct dev_mapping_table *map_entry)
{
	map_entry->physical_id = 0;
	map_entry->device_info = 0;
	map_entry->phy_bits = 0;
	map_entry->dpm_entry_num = MPT2SAS_DPM_BAD_IDX;
	map_entry->dev_handle = 0;
	map_entry->channel = -1;
	map_entry->id = -1;
	map_entry->missing_count = 0;
	map_entry->init_complete = 0;
}

/**
 * _mapping_clear_enc_entry - Clear a particular enclosure table entry.
 * @enc_entry: enclosure table entry
 *
 * Returns nothing.
 */
static inline void
_mapping_clear_enc_entry(struct enc_mapping_table *enc_entry)
{
	enc_entry->enclosure_id = 0;
	enc_entry->start_index = MPT2SAS_MAPTABLE_BAD_IDX;
	enc_entry->phy_bits = 0;
	enc_entry->dpm_entry_num = MPT2SAS_DPM_BAD_IDX;
	enc_entry->enc_handle = 0;
	enc_entry->num_slots = 0;
	enc_entry->start_slot = 0;
	enc_entry->missing_count = 0;
	enc_entry->removal_flag = 0;
	enc_entry->skip_search = 0;
	enc_entry->init_complete = 0;
}

/**
 * _mapping_commit_enc_entry - write a particular enc entry in DPM page0.
 * @ioc: per adapter object
 * @enc_entry: enclosure table entry
 *
 * Returns 0 for success, non-zero for failure.
 */
static int
_mapping_commit_enc_entry(struct MPT2SAS_ADAPTER *ioc,
    struct enc_mapping_table *et_entry)
{
	Mpi2DriverMap0Entry_t *dpm_entry;
	struct dev_mapping_table *mt_entry;
	Mpi2ConfigReply_t mpi_reply;
	Mpi2DriverMappingPage0_t config_page;

	if (!ioc->is_dpm_enable)
		return 0;

	memset(&config_page, 0, sizeof(Mpi2DriverMappingPage0_t));
	memcpy(&config_page.Header, (u8 *) ioc->dpm_pg0,
	    sizeof(MPI2_CONFIG_EXTENDED_PAGE_HEADER));
	dpm_entry = (Mpi2DriverMap0Entry_t *)((u8 *)ioc->dpm_pg0 +
	    sizeof(MPI2_CONFIG_EXTENDED_PAGE_HEADER));
	dpm_entry += et_entry->dpm_entry_num;
	dpm_entry->PhysicalIdentifier =
	    cpu_to_le64(et_entry->enclosure_id);
	mt_entry = &ioc->mapping_table[et_entry->start_index];
	dpm_entry->DeviceIndex = cpu_to_le16(mt_entry->id);
	dpm_entry->MappingInformation = et_entry->num_slots;
	dpm_entry->MappingInformation <<= MPI2_DRVMAP0_MAPINFO_SLOT_SHIFT;
	dpm_entry->MappingInformation |= et_entry->missing_count;
	dpm_entry->MappingInformation = cpu_to_le16(dpm_entry->
	    MappingInformation);
	dpm_entry->PhysicalBitsMapping = cpu_to_le32(et_entry->phy_bits);
	dpm_entry->Reserved1 = 0;

	memcpy(&config_page.Entry, (u8 *)dpm_entry,
	    sizeof(Mpi2DriverMap0Entry_t));
	if (mpt2sas_config_set_dpm_pg0(ioc, &mpi_reply, &config_page,
	    et_entry->dpm_entry_num)) {
		dhmprintk(ioc, printk(MPT2SAS_WARN_FMT "%s: "
		    "Writing dpm entry %d for enclosure failed\n",
		    ioc->name, __func__, et_entry->dpm_entry_num));
		dpm_entry->MappingInformation = le16_to_cpu(dpm_entry->
		    MappingInformation);
		dpm_entry->DeviceIndex = le16_to_cpu(dpm_entry->DeviceIndex);
		dpm_entry->PhysicalBitsMapping =
		    le32_to_cpu(dpm_entry->PhysicalBitsMapping);
		return -1;
	}
	dpm_entry->MappingInformation = le16_to_cpu(dpm_entry->
	    MappingInformation);
	dpm_entry->DeviceIndex = le16_to_cpu(dpm_entry->DeviceIndex);
	dpm_entry->PhysicalBitsMapping =
	    le32_to_cpu(dpm_entry->PhysicalBitsMapping);
	return 0;
}

/**
 * _mapping_commit_map_entry - write a particular map table entry in DPM page0.
 * @ioc: per adapter object
 * @enc_entry: enclosure table entry
 *
 * Returns 0 for success, non-zero for failure.
 */
static int
_mapping_commit_map_entry(struct MPT2SAS_ADAPTER *ioc,
    struct dev_mapping_table *mt_entry)
{
	Mpi2DriverMap0Entry_t *dpm_entry;
	Mpi2ConfigReply_t mpi_reply;
	Mpi2DriverMappingPage0_t config_page;

	if (!ioc->is_dpm_enable)
		return 0;

	memset(&config_page, 0, sizeof(Mpi2DriverMappingPage0_t));
	memcpy(&config_page.Header, (u8 *) ioc->dpm_pg0,
	    sizeof(MPI2_CONFIG_EXTENDED_PAGE_HEADER));
	dpm_entry = (Mpi2DriverMap0Entry_t *)((u8 *) ioc->dpm_pg0 +
	    sizeof(MPI2_CONFIG_EXTENDED_PAGE_HEADER));
	dpm_entry += mt_entry->dpm_entry_num;
	dpm_entry->PhysicalIdentifier =
	    cpu_to_le64(mt_entry->physical_id);
	dpm_entry->DeviceIndex = cpu_to_le16(mt_entry->id);
	dpm_entry->MappingInformation = mt_entry->missing_count;
	dpm_entry->PhysicalBitsMapping = 0;
	dpm_entry->Reserved1 = 0;
	dpm_entry->MappingInformation = cpu_to_le16(dpm_entry->
	    MappingInformation);

	memcpy(&config_page.Entry, (u8 *)dpm_entry,
	    sizeof(Mpi2DriverMap0Entry_t));
	if (mpt2sas_config_set_dpm_pg0(ioc, &mpi_reply, &config_page,
	    mt_entry->dpm_entry_num)) {
		dhmprintk(ioc, printk(MPT2SAS_WARN_FMT "%s: "
		    "Writing dpm entry %d for device failed\n",
		    ioc->name, __func__, mt_entry->dpm_entry_num));
		dpm_entry->MappingInformation = le16_to_cpu(dpm_entry->
		    MappingInformation);
		dpm_entry->DeviceIndex = le16_to_cpu(dpm_entry->DeviceIndex);
		return -1;
	}

	dpm_entry->MappingInformation = le16_to_cpu(dpm_entry->
	    MappingInformation);
	dpm_entry->DeviceIndex = le16_to_cpu(dpm_entry->DeviceIndex);
	return 0;
}

/**
 * _mapping_get_ir_maprange - get start and end index for IR map range.
 * @ioc: per adapter object
 * @start_idx: place holder for start index
 * @end_idx: place holder for end index
 *
 * The IR volumes can be mapped either at start or end of the mapping table
 * this function gets the detail of where IR volume mapping starts and ends
 * in the device mapping table
 *
 * Returns nothing.
 */
static void
_mapping_get_ir_maprange(struct MPT2SAS_ADAPTER *ioc, u32 *start_idx,
    u32 *end_idx)
{
	u16 volume_mapping_flags;
	u16 ioc_pg8_flags = le16_to_cpu(ioc->ioc_pg8.Flags);

	volume_mapping_flags = le16_to_cpu(ioc->ioc_pg8.IRVolumeMappingFlags) &
	    MPI2_IOCPAGE8_IRFLAGS_MASK_VOLUME_MAPPING_MODE;
	if (volume_mapping_flags == MPI2_IOCPAGE8_IRFLAGS_LOW_VOLUME_MAPPING) {
		*start_idx = 0;
		if (ioc_pg8_flags & MPI2_IOCPAGE8_FLAGS_RESERVED_TARGETID_0)
			*start_idx = 1;
	} else
		*start_idx = ioc->max_devices - ioc->max_volumes;
	*end_idx = *start_idx + ioc->max_volumes - 1;
}

/**
 * _mapping_get_enc_idx_from_id - get enclosure index from enclosure ID
 * @ioc: per adapter object
 * @enc_id: enclosure logical identifier
 *
 * Returns the index of enclosure entry on success or bad index.
 */
static u8
_mapping_get_enc_idx_from_id(struct MPT2SAS_ADAPTER *ioc, __le64 enc_id,
    __le64 phy_bits)
{
	struct enc_mapping_table *et_entry;
	u8 enc_idx = 0;
	for (enc_idx = 0; enc_idx < ioc->num_enc_table_entries; enc_idx++) {
		et_entry = &ioc->enclosure_table[enc_idx];
		if ((et_entry->enclosure_id == le64_to_cpu(enc_id)) &&
		    (!et_entry->phy_bits || (et_entry->phy_bits &
		     le32_to_cpu(phy_bits))))
			return enc_idx;
	}
	return MPT2SAS_ENCTABLE_BAD_IDX;
}

/**
 * _mapping_get_enc_idx_from_handle - get enclosure index from handle
 * @ioc: per adapter object
 * @enc_id: enclosure handle
 *
 * Returns the index of enclosure entry on success or bad index.
 */
static u8
_mapping_get_enc_idx_from_handle(struct MPT2SAS_ADAPTER *ioc, u16 handle)
{
	struct enc_mapping_table *et_entry;
	u8 enc_idx = 0;
	for (enc_idx = 0; enc_idx < ioc->num_enc_table_entries; enc_idx++) {
		et_entry = &ioc->enclosure_table[enc_idx];
		if (et_entry->missing_count)
			continue;
		if (et_entry->enc_handle == handle)
			return enc_idx;
	}
	return MPT2SAS_ENCTABLE_BAD_IDX;
}

/**
 * _mapping_get_high_missing_et_idx - get missing enclosure index
 * @ioc: per adapter object
 *
 * Search through the enclosure table and identifies the enclosure entry
 * with high missing count and returns it's index
 *
 * Returns the index of enclosure entry on success or bad index.
 */
static u8
_mapping_get_high_missing_et_idx(struct MPT2SAS_ADAPTER *ioc)
{
	struct enc_mapping_table *et_entry;
	u8 high_missing_count = 0;
	u8 enc_idx, high_idx = MPT2SAS_ENCTABLE_BAD_IDX;
	for (enc_idx = 0; enc_idx < ioc->num_enc_table_entries; enc_idx++) {
		et_entry = &ioc->enclosure_table[enc_idx];
		if ((et_entry->missing_count > high_missing_count) &&
		    !et_entry->skip_search) {
			high_missing_count =  et_entry->missing_count;
			high_idx = enc_idx;
		}
	}
	return high_idx;
}

/**
 * _mapping_get_high_missing_mt_idx - get missing map table index
 * @ioc: per adapter object
 *
 * Search through the map table and identifies the device entry
 * with high missing count and returns it's index
 *
 * Returns the index of map table entry on success or bad index.
 */
static u32
_mapping_get_high_missing_mt_idx(struct MPT2SAS_ADAPTER *ioc)
{
	u32 map_idx, high_idx = MPT2SAS_ENCTABLE_BAD_IDX;
	u8 high_missing_count = 0;
	u32 start_idx, end_idx, start_idx_ir, end_idx_ir;
	struct dev_mapping_table *mt_entry;
	u16 ioc_pg8_flags = le16_to_cpu(ioc->ioc_pg8.Flags);

	start_idx = 0;
	end_idx = ioc->max_devices;
	if (ioc_pg8_flags & MPI2_IOCPAGE8_FLAGS_RESERVED_TARGETID_0)
		start_idx = 1;
	if (ioc->ir_firmware)
		_mapping_get_ir_maprange(ioc, &start_idx_ir, &end_idx_ir);
	if (start_idx == start_idx_ir)
		start_idx = end_idx_ir + 1;
	else
		end_idx = start_idx_ir;
	mt_entry = &ioc->mapping_table[start_idx];
	for (map_idx = start_idx; map_idx < end_idx; map_idx++, mt_entry++) {
		if (mt_entry->missing_count > high_missing_count) {
			high_missing_count =  mt_entry->missing_count;
			high_idx = map_idx;
		}
	}
	return high_idx;
}

/**
 * _mapping_get_ir_mt_idx_from_wwid - get map table index from volume WWID
 * @ioc: per adapter object
 * @wwid: world wide unique ID of the volume
 *
 * Returns the index of map table entry on success or bad index.
 */
static u32
_mapping_get_ir_mt_idx_from_wwid(struct MPT2SAS_ADAPTER *ioc, u64 wwid)
{
	u32 start_idx, end_idx, map_idx;
	struct dev_mapping_table *mt_entry;
	_mapping_get_ir_maprange(ioc, &start_idx, &end_idx);
	mt_entry = &ioc->mapping_table[start_idx];
	for (map_idx  = start_idx; map_idx <= end_idx; map_idx++, mt_entry++)
		if (mt_entry->physical_id == wwid)
			return map_idx;
	return MPT2SAS_MAPTABLE_BAD_IDX;
}

/**
 * _mapping_get_mt_idx_from_id - get map table index from a device ID
 * @ioc: per adapter object
 * @dev_id: device identifer (SAS Address)
 *
 * Returns the index of map table entry on success or bad index.
 */
static u32
_mapping_get_mt_idx_from_id(struct MPT2SAS_ADAPTER *ioc, u64 dev_id)
{
	u32 map_idx;
	struct dev_mapping_table *mt_entry;
	for (map_idx = 0; map_idx < ioc->max_devices; map_idx++) {
		mt_entry = &ioc->mapping_table[map_idx];
		if (mt_entry->physical_id == dev_id)
			return map_idx;
	}
	return MPT2SAS_MAPTABLE_BAD_IDX;
}

/**
 * _mapping_get_mt_idx_from_handle - get map table index from handle
 * @ioc: per adapter object
 * @dev_id: device handle
 *
 * Returns the index of map table entry on success or bad index.
 */
static u32
_mapping_get_mt_idx_from_handle(struct MPT2SAS_ADAPTER *ioc, u16 handle)
{
	u32 map_idx;
	struct dev_mapping_table *mt_entry;
	for (map_idx = 0; map_idx < ioc->max_devices; map_idx++) {
		mt_entry = &ioc->mapping_table[map_idx];
		if (mt_entry->dev_handle == handle)
			return map_idx;
	}
	return MPT2SAS_MAPTABLE_BAD_IDX;
}

/**
 * _mapping_get_free_ir_mt_idx - get first free index for a volume
 * @ioc: per adapter object
 *
 * Search through mapping table for free index for a volume and if no free
 * index then looks for a volume with high mapping index
 *
 * Returns the index of map table entry on success or bad index.
 */
static u32
_mapping_get_free_ir_mt_idx(struct MPT2SAS_ADAPTER *ioc)
{
	u8 high_missing_count = 0;
	u32 start_idx, end_idx, map_idx;
	u32 high_idx = MPT2SAS_MAPTABLE_BAD_IDX;
	struct dev_mapping_table *mt_entry;

	_mapping_get_ir_maprange(ioc, &start_idx, &end_idx);

	mt_entry = &ioc->mapping_table[start_idx];
	for (map_idx  = start_idx; map_idx <= end_idx; map_idx++, mt_entry++)
		if (!(mt_entry->device_info & MPT2SAS_MAP_IN_USE))
			return map_idx;

	mt_entry = &ioc->mapping_table[start_idx];
	for (map_idx  = start_idx; map_idx <= end_idx; map_idx++, mt_entry++) {
		if (mt_entry->missing_count > high_missing_count) {
			high_missing_count = mt_entry->missing_count;
			high_idx = map_idx;
		}
	}
	return high_idx;
}

/**
 * _mapping_get_free_mt_idx - get first free index for a device
 * @ioc: per adapter object
 * @start_idx: offset in the table to start search
 *
 * Returns the index of map table entry on success or bad index.
 */
static u32
_mapping_get_free_mt_idx(struct MPT2SAS_ADAPTER *ioc, u32 start_idx)
{
	u32 map_idx, max_idx = ioc->max_devices;
	struct dev_mapping_table *mt_entry = &ioc->mapping_table[start_idx];
	u16 volume_mapping_flags;
	volume_mapping_flags = le16_to_cpu(ioc->ioc_pg8.IRVolumeMappingFlags) &
	    MPI2_IOCPAGE8_IRFLAGS_MASK_VOLUME_MAPPING_MODE;
	if (ioc->ir_firmware && (volume_mapping_flags ==
	    MPI2_IOCPAGE8_IRFLAGS_HIGH_VOLUME_MAPPING))
		max_idx -= ioc->max_volumes;
	for (map_idx  = start_idx; map_idx < max_idx; map_idx++, mt_entry++)
		if (!(mt_entry->device_info & (MPT2SAS_MAP_IN_USE |
		    MPT2SAS_DEV_RESERVED)))
			return map_idx;
	return MPT2SAS_MAPTABLE_BAD_IDX;
}

/**
 * _mapping_get_dpm_idx_from_id - get DPM index from ID
 * @ioc: per adapter object
 * @id: volume WWID or enclosure ID or device ID
 *
 * Returns the index of DPM entry on success or bad index.
 */
static u16
_mapping_get_dpm_idx_from_id(struct MPT2SAS_ADAPTER *ioc, u64 id, u32 phy_bits)
{
	u16 entry_num ;
	Mpi2DriverMap0Entry_t *dpm_entry;

	dpm_entry = (Mpi2DriverMap0Entry_t *)((u8 *)ioc->dpm_pg0 +
	    sizeof(MPI2_CONFIG_EXTENDED_PAGE_HEADER));

	for (entry_num = 0; entry_num < ioc->max_dpm_entries; entry_num++,
	    dpm_entry++)
		if ((id == le64_to_cpu(dpm_entry->PhysicalIdentifier)) &&
		    (!phy_bits || !dpm_entry->PhysicalBitsMapping ||
		    (phy_bits & dpm_entry->PhysicalBitsMapping)))
			return entry_num;

	return MPT2SAS_DPM_BAD_IDX;
}

/**
 * _mapping_get_free_dpm_idx - get first available DPM index
 * @ioc: per adapter object
 *
 * Returns the index of DPM entry on success or bad index.
 */
static u32
_mapping_get_free_dpm_idx(struct MPT2SAS_ADAPTER *ioc)
{
	u16 entry_num ;

	for (entry_num = 0; entry_num < ioc->max_dpm_entries; entry_num++) {
		if (!ioc->dpm_entry_used[entry_num])
			return entry_num;
	}
	return MPT2SAS_DPM_BAD_IDX;
}

/**
 * _mapping_update_ir_missing_cnt - Updates missing count for a volume
 * @ioc: per adapter object
 * @map_idx: map table index of the volume
 * @element: IR configuration change element
 * @wwid: IR volume ID.
 *
 * Updates the missing count in the map table and in the DPM entry for a volume
 *
 * Returns nothing.
 */
static void
_mapping_update_ir_missing_cnt(struct MPT2SAS_ADAPTER *ioc, u32 map_idx,
    Mpi2EventIrConfigElement_t *element, u64 wwid)
{
	struct dev_mapping_table *mt_entry;
	u8 missing_cnt, reason = element->ReasonCode;
	u16 dpm_idx ;
	Mpi2DriverMap0Entry_t *dpm_entry;

	if (!ioc->is_dpm_enable)
		return;
	mt_entry = &ioc->mapping_table[map_idx];
	if (reason == MPI2_EVENT_IR_CHANGE_RC_ADDED) {
		mt_entry->missing_count = 0;
	} else if (reason == MPI2_EVENT_IR_CHANGE_RC_VOLUME_CREATED) {
		mt_entry->missing_count = 0;
		mt_entry->init_complete = 0;
	} else if ((reason == MPI2_EVENT_IR_CHANGE_RC_REMOVED) ||
	    (reason == MPI2_EVENT_IR_CHANGE_RC_VOLUME_DELETED)) {
		if (!mt_entry->init_complete) {
			if (mt_entry->missing_count < MPT2SAS_MAX_MISSING_COUNT)
				mt_entry->missing_count++;
			else
				mt_entry->init_complete = 1;
		}
		if (!mt_entry->missing_count)
			mt_entry->missing_count++;
		mt_entry->dev_handle = 0;
	}

	dpm_idx = mt_entry->dpm_entry_num;
	if (dpm_idx == MPT2SAS_DPM_BAD_IDX) {
		if ((reason == MPI2_EVENT_IR_CHANGE_RC_ADDED) ||
		    (reason == MPI2_EVENT_IR_CHANGE_RC_REMOVED))
			dpm_idx = _mapping_get_dpm_idx_from_id(ioc,
			    mt_entry->physical_id, 0);
		else if (reason == MPI2_EVENT_IR_CHANGE_RC_VOLUME_DELETED)
			return;
	}
	if (dpm_idx != MPT2SAS_DPM_BAD_IDX) {
		dpm_entry = (Mpi2DriverMap0Entry_t *)((u8 *)ioc->dpm_pg0 +
		    sizeof(MPI2_CONFIG_EXTENDED_PAGE_HEADER));
		dpm_entry += dpm_idx;
		missing_cnt = dpm_entry->MappingInformation &
		    MPI2_DRVMAP0_MAPINFO_MISSING_MASK;
		if ((mt_entry->physical_id ==
		    le64_to_cpu(dpm_entry->PhysicalIdentifier))
		    && (missing_cnt == mt_entry->missing_count))
			mt_entry->init_complete = 1;
	} else {
		dpm_idx = _mapping_get_free_dpm_idx(ioc);
		mt_entry->init_complete = 0;
	}

	if ((dpm_idx != MPT2SAS_DPM_BAD_IDX) && !mt_entry->init_complete) {
		mt_entry->init_complete = 1;
		mt_entry->dpm_entry_num = dpm_idx;
		dpm_entry = (Mpi2DriverMap0Entry_t *)((u8 *)ioc->dpm_pg0 +
		    sizeof(MPI2_CONFIG_EXTENDED_PAGE_HEADER));
		dpm_entry += dpm_idx;
		dpm_entry->PhysicalIdentifier =
		    cpu_to_le64(mt_entry->physical_id);
		dpm_entry->DeviceIndex = map_idx;
		dpm_entry->MappingInformation = mt_entry->missing_count;
		dpm_entry->PhysicalBitsMapping = 0;
		dpm_entry->Reserved1 = 0;
		ioc->dpm_flush_entry[dpm_idx] = 1;
		ioc->dpm_entry_used[dpm_idx] = 1;
	} else if (dpm_idx == MPT2SAS_DPM_BAD_IDX) {
		printk(MPT2SAS_ERR_FMT "%s No space to add entry in DPM table\n"
		    , ioc->name, __func__);
		mt_entry->init_complete = 1;
	}
}

/**
 * _mapping_add_to_removal_table - mark an entry for removal
 * @ioc: per adapter object
 * @handle: Handle of enclosures/device/volume
 *
 * Adds the handle or DPM entry number in removal table.
 *
 * Returns nothing.
 */
static void
_mapping_add_to_removal_table(struct MPT2SAS_ADAPTER *ioc, u16 handle,
    u16 dpm_idx)
{
	struct map_removal_table *remove_entry;
	u32 i;
	u16 ioc_pg8_flags = le16_to_cpu(ioc->ioc_pg8.Flags);

	remove_entry = ioc->removal_table;

	for (i = 0; i < ioc->max_devices; i++, remove_entry++) {
		if (remove_entry->dev_handle || remove_entry->dpm_entry_num
		    != MPT2SAS_DPM_BAD_IDX)
			continue;
		if ((ioc_pg8_flags & MPI2_IOCPAGE8_FLAGS_MASK_MAPPING_MODE)
		    == MPI2_IOCPAGE8_FLAGS_ENCLOSURE_SLOT_MAPPING) {
			if (dpm_idx)
				remove_entry->dpm_entry_num = dpm_idx;
			if (remove_entry->dpm_entry_num == MPT2SAS_DPM_BAD_IDX)
				remove_entry->dev_handle = handle;
		} else if ((ioc_pg8_flags &
		    MPI2_IOCPAGE8_FLAGS_MASK_MAPPING_MODE)
		    == MPI2_IOCPAGE8_FLAGS_DEVICE_PERSISTENCE_MAPPING)
			remove_entry->dev_handle = handle;
		break;
	}

}

/**
 * _mapping_update_missing_count - Update missing count for a device
 * @ioc: per adapter object
 * @topo_change: Topology change event entry
 *
 * Search through the topology change list and if any device is found not
 * responding it's associated map table entry and DPM entry is updated
 *
 * Returns nothing.
 */
static void
_mapping_update_missing_count(struct MPT2SAS_ADAPTER *ioc,
    struct _map_topology_change *topo_change)
{
	u16 ioc_pg8_flags = le16_to_cpu(ioc->ioc_pg8.Flags);
	u8 entry;
	struct _map_phy_change *phy_change;
	u32 map_idx;
	struct dev_mapping_table *mt_entry;
	Mpi2DriverMap0Entry_t *dpm_entry;


	for (entry = 0; entry < topo_change->num_entries; entry++) {
		phy_change = &topo_change->phy_details[entry];
		if (!phy_change->dev_handle || (phy_change->reason !=
		    MPI2_EVENT_SAS_TOPO_RC_TARG_NOT_RESPONDING))
			continue;
		map_idx = _mapping_get_mt_idx_from_handle(ioc, phy_change->
		    dev_handle);
		phy_change->is_processed = 1;
		if (map_idx == MPT2SAS_MAPTABLE_BAD_IDX) {
			printk(MPT2SAS_ERR_FMT "%s Trying to remove the device"
			    "which is already removed from map table\n",
			    ioc->name, __func__);
			continue;
		}
		mt_entry = &ioc->mapping_table[map_idx];
		if (!mt_entry->init_complete) {
			if (mt_entry->missing_count < MPT2SAS_MAX_MISSING_COUNT)
				mt_entry->missing_count++;
			else
				mt_entry->init_complete = 1;
		}
		if (!mt_entry->missing_count)
			mt_entry->missing_count++;
		_mapping_add_to_removal_table(ioc, mt_entry->dev_handle, 0);
		mt_entry->dev_handle = 0;

		if (((ioc_pg8_flags & MPI2_IOCPAGE8_FLAGS_MASK_MAPPING_MODE)
		    == MPI2_IOCPAGE8_FLAGS_DEVICE_PERSISTENCE_MAPPING) &&
		    ioc->is_dpm_enable && !mt_entry->init_complete &&
		    mt_entry->dpm_entry_num != MPT2SAS_DPM_BAD_IDX) {
			dpm_entry = (Mpi2DriverMap0Entry_t *)
			    ((u8 *)ioc->dpm_pg0 +
			    sizeof(MPI2_CONFIG_EXTENDED_PAGE_HEADER));
			dpm_entry += mt_entry->dpm_entry_num;
			dpm_entry->MappingInformation = mt_entry->missing_count;
			ioc->dpm_flush_entry[mt_entry->dpm_entry_num] = 1;
		}
		mt_entry->init_complete = 1;
	}
}

/**
 * _mapping_find_enc_map_space -find map table entries for enclosure
 * @ioc: per adapter object
 * @et_entry: enclosure entry
 *
 * Search through the mapping table defragment it and provide contiguous
 * space in map table for a particular enclosure entry
 *
 * Returns start index in map table or bad index.
 */
static u32
_mapping_find_enc_map_space(struct MPT2SAS_ADAPTER *ioc,
    struct enc_mapping_table *et_entry)
{
	u16 vol_mapping_flags;
	u32 skip_count, end_of_table, map_idx, enc_idx;
	u16 num_found;
	u32 start_idx = MPT2SAS_MAPTABLE_BAD_IDX;
	struct dev_mapping_table *mt_entry;
	struct enc_mapping_table *enc_entry;
	unsigned char done_flag = 0, found_space;
	u16 max_num_phy_ids = le16_to_cpu(ioc->ioc_pg8.MaxNumPhysicalMappedIDs);


	skip_count = ioc->num_rsvd_entries;
	num_found = 0;

	vol_mapping_flags = le16_to_cpu(ioc->ioc_pg8.IRVolumeMappingFlags) &
	    MPI2_IOCPAGE8_IRFLAGS_MASK_VOLUME_MAPPING_MODE;

	if (!ioc->ir_firmware)
		end_of_table = ioc->max_devices;
	else if (vol_mapping_flags == MPI2_IOCPAGE8_IRFLAGS_LOW_VOLUME_MAPPING)
		end_of_table = ioc->max_devices;
	else
		end_of_table = ioc->max_devices - ioc->max_volumes;

	for (map_idx = (max_num_phy_ids + skip_count);
	    map_idx < end_of_table; map_idx++) {
		mt_entry = &ioc->mapping_table[map_idx];
		if ((et_entry->enclosure_id == mt_entry->physical_id) &&
		    (!mt_entry->phy_bits || (mt_entry->phy_bits &
		     et_entry->phy_bits))) {
			num_found += 1;
			if (num_found == et_entry->num_slots) {
				start_idx = (map_idx - num_found) + 1;
				return start_idx;
			}
		} else
			num_found = 0;
	}
	for (map_idx = (max_num_phy_ids + skip_count);
	    map_idx < end_of_table; map_idx++) {
		mt_entry = &ioc->mapping_table[map_idx];
		if (!(mt_entry->device_info & MPT2SAS_DEV_RESERVED)) {
			num_found += 1;
			if (num_found == et_entry->num_slots) {
				start_idx = (map_idx - num_found) + 1;
				return start_idx;
			}
		} else
			num_found = 0;
	}

	while (!done_flag) {
		enc_idx = _mapping_get_high_missing_et_idx(ioc);
		if (enc_idx == MPT2SAS_ENCTABLE_BAD_IDX)
			return MPT2SAS_MAPTABLE_BAD_IDX;
		enc_entry = &ioc->enclosure_table[enc_idx];
		/*VSP FIXME*/
		enc_entry->skip_search = 1;
		mt_entry = &ioc->mapping_table[enc_entry->start_index];
		for (map_idx = enc_entry->start_index; map_idx <
		    (enc_entry->start_index + enc_entry->num_slots); map_idx++,
		    mt_entry++)
			mt_entry->device_info  &= ~MPT2SAS_DEV_RESERVED;
		found_space = 0;
		for (map_idx = (max_num_phy_ids +
		    skip_count); map_idx < end_of_table; map_idx++) {
			mt_entry = &ioc->mapping_table[map_idx];
			if (!(mt_entry->device_info & MPT2SAS_DEV_RESERVED)) {
				num_found += 1;
				if (num_found == et_entry->num_slots) {
					start_idx = (map_idx - num_found) + 1;
					found_space = 1;
				}
			} else
				num_found = 0;
		}

		if (!found_space)
			continue;
		for (map_idx = start_idx; map_idx < (start_idx + num_found);
		    map_idx++) {
			enc_entry = ioc->enclosure_table;
			for (enc_idx = 0; enc_idx < ioc->num_enc_table_entries;
			    enc_idx++, enc_entry++) {
				if (map_idx < enc_entry->start_index ||
				    map_idx > (enc_entry->start_index +
				    enc_entry->num_slots))
					continue;
				if (!enc_entry->removal_flag) {
					enc_entry->removal_flag = 1;
					_mapping_add_to_removal_table(ioc, 0,
					    enc_entry->dpm_entry_num);
				}
				mt_entry = &ioc->mapping_table[map_idx];
				if (mt_entry->device_info &
				    MPT2SAS_MAP_IN_USE) {
					_mapping_add_to_removal_table(ioc,
					    mt_entry->dev_handle, 0);
					_mapping_clear_map_entry(mt_entry);
				}
				if (map_idx == (enc_entry->start_index +
				    enc_entry->num_slots - 1))
					_mapping_clear_enc_entry(et_entry);
			}
		}
		enc_entry = ioc->enclosure_table;
		for (enc_idx = 0; enc_idx < ioc->num_enc_table_entries;
		    enc_idx++, enc_entry++) {
			if (!enc_entry->removal_flag) {
				mt_entry = &ioc->mapping_table[enc_entry->
				    start_index];
				for (map_idx = enc_entry->start_index; map_idx <
				    (enc_entry->start_index +
				    enc_entry->num_slots); map_idx++,
				    mt_entry++)
					mt_entry->device_info |=
					    MPT2SAS_DEV_RESERVED;
				et_entry->skip_search = 0;
			}
		}
		done_flag = 1;
	}
	return start_idx;
}

/**
 * _mapping_get_dev_info -get information about newly added devices
 * @ioc: per adapter object
 * @topo_change: Topology change event entry
 *
 * Search through the topology change event list and issues sas device pg0
 * requests for the newly added device and reserved entries in tables
 *
 * Returns nothing
 */
static void
_mapping_get_dev_info(struct MPT2SAS_ADAPTER *ioc,
    struct _map_topology_change *topo_change)
{
	u16 ioc_pg8_flags = le16_to_cpu(ioc->ioc_pg8.Flags);
	Mpi2ConfigReply_t mpi_reply;
	Mpi2SasDevicePage0_t sas_device_pg0;
	u8 entry, enc_idx, phy_idx;
	u32 map_idx, index, device_info;
	struct _map_phy_change *phy_change, *tmp_phy_change;
	__le64 sas_address;
	struct enc_mapping_table *et_entry;
	struct dev_mapping_table *mt_entry;
	u8 add_code = MPI2_EVENT_SAS_TOPO_RC_TARG_ADDED;
	int rc;

	for (entry = 0; entry < topo_change->num_entries; entry++) {
		phy_change = &topo_change->phy_details[entry];
		if (phy_change->is_processed || !phy_change->dev_handle ||
		    phy_change->reason !=
		    MPI2_EVENT_SAS_TOPO_RC_TARG_ADDED)
			continue;
		if (mpt2sas_config_get_sas_device_pg0(ioc, &mpi_reply,
		    &sas_device_pg0, MPI2_SAS_DEVICE_PGAD_FORM_HANDLE,
		    phy_change->dev_handle)) {
			phy_change->is_processed = 1;
			continue;
		}

		device_info =
		    le32_to_cpu(sas_device_pg0.DeviceInfo);
		if ((ioc_pg8_flags & MPI2_IOCPAGE8_FLAGS_MASK_MAPPING_MODE)
		    == MPI2_IOCPAGE8_FLAGS_DEVICE_PERSISTENCE_MAPPING) {
			if ((device_info & MPI2_SAS_DEVICE_INFO_END_DEVICE) &&
			    (device_info & MPI2_SAS_DEVICE_INFO_SATA_DEVICE)) {
				rc =
				    mpt2sas_scsih_get_sas_address_for_sata_disk
				    (ioc, &sas_address, phy_change->dev_handle,
				     device_info);
				if (rc) {
					printk(MPT2SAS_ERR_FMT
					    "%s Failed computing the Hashed SAS"
					    " address for SATA device with"
					    " handle (0x%04x)\n", ioc->name,
					    __func__, phy_change->dev_handle);
					sas_address = le64_to_cpu
						(sas_device_pg0.SASAddress);
				}
			} else
				sas_address =
					le64_to_cpu(sas_device_pg0.SASAddress);
		} else
			sas_address = le64_to_cpu(sas_device_pg0.SASAddress);
		phy_change->physical_id = sas_address;
		phy_change->slot = le16_to_cpu(sas_device_pg0.Slot);
		phy_change->device_info =
		    le32_to_cpu(sas_device_pg0.DeviceInfo);

		if ((ioc_pg8_flags & MPI2_IOCPAGE8_FLAGS_MASK_MAPPING_MODE)
		    == MPI2_IOCPAGE8_FLAGS_ENCLOSURE_SLOT_MAPPING) {
			enc_idx = _mapping_get_enc_idx_from_handle(ioc,
			    topo_change->enc_handle);
			if (enc_idx == MPT2SAS_ENCTABLE_BAD_IDX) {
				phy_change->is_processed = 1;
				printk(MPT2SAS_ERR_FMT
				    "%s Failed adding the device with handle"
				    "(0x%04x) since the enclosure is not"
				    " present in the map table\n",
				    ioc->name, __func__,
				    phy_change->dev_handle);
				continue;
			}
			if (!((phy_change->device_info &
			    MPI2_SAS_DEVICE_INFO_END_DEVICE) &&
			    (phy_change->device_info &
			    (MPI2_SAS_DEVICE_INFO_SSP_TARGET |
			    MPI2_SAS_DEVICE_INFO_STP_TARGET |
			    MPI2_SAS_DEVICE_INFO_SATA_DEVICE)))) {
				phy_change->is_processed = 1;
				continue;
			}
			et_entry = &ioc->enclosure_table[enc_idx];
			if (et_entry->start_index != MPT2SAS_MAPTABLE_BAD_IDX)
				continue;
			if (!topo_change->exp_handle) {
				map_idx	= ioc->num_rsvd_entries;
				et_entry->start_index = map_idx;
			} else {
				map_idx = _mapping_find_enc_map_space(ioc,
				    et_entry);
				et_entry->start_index = map_idx;
				if (et_entry->start_index ==
				    MPT2SAS_MAPTABLE_BAD_IDX) {
					phy_change->is_processed = 1;
					for (phy_idx = 0; phy_idx <
					    topo_change->num_entries;
					    phy_idx++) {
						tmp_phy_change =
						    &topo_change->phy_details
						    [phy_idx];
						if (tmp_phy_change->reason ==
						    add_code)
							tmp_phy_change->
							    is_processed = 1;
					}
					break;
				}
			}
			mt_entry = &ioc->mapping_table[map_idx];
			for (index = map_idx; index < (et_entry->num_slots
			    + map_idx); index++, mt_entry++) {
				mt_entry->device_info = MPT2SAS_DEV_RESERVED;
				mt_entry->physical_id = et_entry->enclosure_id;
				mt_entry->phy_bits = et_entry->phy_bits;
			}
		}
	}
}

/**
 * _mapping_set_mid_to_eid -set map table data from enclosure table
 * @ioc: per adapter object
 * @et_entry: enclosure entry
 *
 * Returns nothing
 */
static inline void
_mapping_set_mid_to_eid(struct MPT2SAS_ADAPTER *ioc,
    struct enc_mapping_table *et_entry)
{
	struct dev_mapping_table *mt_entry;
	u16 slots = et_entry->num_slots, map_idx;
	u32 start_idx = et_entry->start_index;
	if (start_idx != MPT2SAS_MAPTABLE_BAD_IDX) {
		mt_entry = &ioc->mapping_table[start_idx];
		for (map_idx = 0; map_idx < slots; map_idx++, mt_entry++)
			mt_entry->physical_id = et_entry->enclosure_id;
	}
}

/**
 * _mapping_clear_removed_entries - mark the entries to be cleared
 * @ioc: per adapter object
 *
 * Search through the removal table and mark the entries which needs to be
 * flushed to DPM and also updates the map table and enclosure table by
 * clearing the corresponding entries.
 *
 * Returns nothing
 */
static void
_mapping_clear_removed_entries(struct MPT2SAS_ADAPTER *ioc)
{
	u32 remove_idx;
	struct map_removal_table *remove_entry;
	Mpi2DriverMap0Entry_t *dpm_entry;
	u8 done_flag = 0, num_entries, m, i;
	struct enc_mapping_table *et_entry, *from, *to;
	u16 ioc_pg8_flags = le16_to_cpu(ioc->ioc_pg8.Flags);

	if (ioc->is_dpm_enable) {
		remove_entry = ioc->removal_table;
		for (remove_idx = 0; remove_idx < ioc->max_devices;
		    remove_idx++, remove_entry++) {
			if (remove_entry->dpm_entry_num
			    != MPT2SAS_DPM_BAD_IDX) {
				dpm_entry = (Mpi2DriverMap0Entry_t *)
				    ((u8 *) ioc->dpm_pg0 +
				     sizeof(MPI2_CONFIG_EXTENDED_PAGE_HEADER));
				dpm_entry += remove_entry->dpm_entry_num;
				dpm_entry->PhysicalIdentifier = 0;
				dpm_entry->DeviceIndex = 0;
				dpm_entry->MappingInformation = 0;
				dpm_entry->PhysicalBitsMapping = 0;
				ioc->dpm_flush_entry[remove_entry->
				    dpm_entry_num] = 1;
				ioc->dpm_entry_used[remove_entry->dpm_entry_num]
				    = 0;
				remove_entry->dpm_entry_num =
				    MPT2SAS_DPM_BAD_IDX;
			}
		}
	}
	if ((ioc_pg8_flags & MPI2_IOCPAGE8_FLAGS_MASK_MAPPING_MODE)
	    == MPI2_IOCPAGE8_FLAGS_ENCLOSURE_SLOT_MAPPING) {
		num_entries = ioc->num_enc_table_entries;
		while (!done_flag) {
			done_flag = 1;
			et_entry = ioc->enclosure_table;
			for (i = 0; i < num_entries; i++, et_entry++) {
				if (!et_entry->enc_handle && et_entry->
				    init_complete) {
					done_flag = 0;
					if (i != (num_entries - 1)) {
						from = &ioc->enclosure_table
						    [i+1];
						to = &ioc->enclosure_table[i];
						for (m = i; m < (num_entries -
						    1); m++, from++, to++) {
							_mapping_set_mid_to_eid
							    (ioc, to);
							*to = *from;
						}
						_mapping_clear_enc_entry(to);
						ioc->num_enc_table_entries--;
						num_entries =
						    ioc->num_enc_table_entries;
					} else {
						_mapping_clear_enc_entry
						    (et_entry);
						ioc->num_enc_table_entries--;
						num_entries =
						    ioc->num_enc_table_entries;
					}
				}
			}
		}
	}
}

/**
 * _mapping_add_new_device -Add the new device into mapping table
 * @ioc: per adapter object
 * @topo_change: Topology change event entry
 *
 * Search through the topology change event list and updates map table,
 * enclosure table and DPM pages for for the newly added devices.
 *
 * Returns nothing
 */
static void
_mapping_add_new_device(struct MPT2SAS_ADAPTER *ioc,
    struct _map_topology_change *topo_change)
{
	u8 enc_idx, missing_cnt, is_removed = 0;
	u16 dpm_idx;
	u32 search_idx, map_idx;
	u32 entry;
	struct dev_mapping_table *mt_entry;
	struct enc_mapping_table *et_entry;
	struct _map_phy_change *phy_change;
	u16 ioc_pg8_flags = le16_to_cpu(ioc->ioc_pg8.Flags);
	Mpi2DriverMap0Entry_t *dpm_entry;
	u8 map_shift = MPI2_DRVMAP0_MAPINFO_SLOT_SHIFT;
	u8 hdr_sz = sizeof(MPI2_CONFIG_EXTENDED_PAGE_HEADER);
	u16 max_num_phy_ids = le16_to_cpu(ioc->ioc_pg8.MaxNumPhysicalMappedIDs);


	for (entry = 0; entry < topo_change->num_entries; entry++) {
		phy_change = &topo_change->phy_details[entry];
		if (phy_change->is_processed)
			continue;
		if (phy_change->reason != MPI2_EVENT_SAS_TOPO_RC_TARG_ADDED ||
		    !phy_change->dev_handle) {
			phy_change->is_processed = 1;
			continue;
		}
		if ((ioc_pg8_flags & MPI2_IOCPAGE8_FLAGS_MASK_MAPPING_MODE)
		    == MPI2_IOCPAGE8_FLAGS_ENCLOSURE_SLOT_MAPPING) {
			enc_idx = _mapping_get_enc_idx_from_handle
			    (ioc, topo_change->enc_handle);
			if (enc_idx == MPT2SAS_ENCTABLE_BAD_IDX) {
				phy_change->is_processed = 1;
				printk(MPT2SAS_ERR_FMT
				    "%s Failed adding the device with handle"
				    "(0x%04x) since the enclosure is not"
				    " present in the map table\n",
				    ioc->name, __func__,
				    phy_change->dev_handle);
				continue;
			}
			et_entry = &ioc->enclosure_table[enc_idx];
			if (et_entry->start_index == MPT2SAS_MAPTABLE_BAD_IDX) {
				phy_change->is_processed = 1;
				if (!ioc->mt_full_retry) {
					ioc->mt_add_device_failed = 1;
					continue;
				}
				printk(MPT2SAS_ERR_FMT
				    "%s Failed adding the device with handle"
				    "(0x%04x) since there is no free space"
				    " available in the map table\n",
				    ioc->name, __func__,
				    phy_change->dev_handle);
				continue;
			}
			map_idx = et_entry->start_index + phy_change->slot -
			    et_entry->start_slot;
			mt_entry = &ioc->mapping_table[map_idx];
			mt_entry->physical_id = phy_change->physical_id;
			mt_entry->channel = 0;
			mt_entry->id = map_idx;
			mt_entry->dev_handle = phy_change->dev_handle;
			mt_entry->missing_count = 0;
			mt_entry->dpm_entry_num = et_entry->dpm_entry_num;
			mt_entry->device_info = phy_change->device_info |
			    (MPT2SAS_DEV_RESERVED | MPT2SAS_MAP_IN_USE);
			if (ioc->is_dpm_enable) {
				dpm_idx = et_entry->dpm_entry_num;
				if (dpm_idx == MPT2SAS_DPM_BAD_IDX)
					dpm_idx = _mapping_get_dpm_idx_from_id
					    (ioc, et_entry->enclosure_id,
					     et_entry->phy_bits);
				if (dpm_idx == MPT2SAS_DPM_BAD_IDX) {
					dpm_idx = _mapping_get_free_dpm_idx
					    (ioc);
					if (dpm_idx != MPT2SAS_DPM_BAD_IDX) {
						dpm_entry =
						    (Mpi2DriverMap0Entry_t *)
						    ((u8 *) ioc->dpm_pg0 +
						     hdr_sz);
						dpm_entry += dpm_idx;
						dpm_entry->
						    PhysicalIdentifier =
						    cpu_to_le64
						    (et_entry->enclosure_id);
						dpm_entry->DeviceIndex =
						    (U16)et_entry->start_index;
						dpm_entry->MappingInformation =
							et_entry->num_slots;
						dpm_entry->MappingInformation
						    <<= map_shift;
						dpm_entry->PhysicalBitsMapping
						    = et_entry->phy_bits;
						et_entry->dpm_entry_num =
						    dpm_idx;
		/* FIXME Do I need to set the dpm_idxin mt_entry too */
						ioc->dpm_entry_used[dpm_idx] =
						    1;
						ioc->dpm_flush_entry[dpm_idx] =
						    1;
						phy_change->is_processed = 1;
					} else {
						phy_change->is_processed = 1;
						printk(MPT2SAS_ERR_FMT
						    "%s Failed adding the dev"
						    "with handle(0x%04x) to "
						    "persistent table since "
						    "there is no free space "
						    "available\n",
						    ioc->name, __func__,
						    phy_change->dev_handle);
					}
				} else {
					et_entry->dpm_entry_num = dpm_idx;
					mt_entry->dpm_entry_num = dpm_idx;
				}
			}
			/* FIXME Why not mt_entry too? */
			et_entry->init_complete = 1;
		} else if ((ioc_pg8_flags &
		    MPI2_IOCPAGE8_FLAGS_MASK_MAPPING_MODE) ==
		    MPI2_IOCPAGE8_FLAGS_DEVICE_PERSISTENCE_MAPPING) {
			map_idx = _mapping_get_mt_idx_from_id
			    (ioc, phy_change->physical_id);
			if (map_idx == MPT2SAS_MAPTABLE_BAD_IDX) {
				search_idx = ioc->num_rsvd_entries;
				if (topo_change->exp_handle)
					search_idx += max_num_phy_ids;
				map_idx = _mapping_get_free_mt_idx(ioc,
				    search_idx);
			}
			if (map_idx == MPT2SAS_MAPTABLE_BAD_IDX) {
				map_idx = _mapping_get_high_missing_mt_idx(ioc);
				if (map_idx != MPT2SAS_MAPTABLE_BAD_IDX) {
					mt_entry = &ioc->mapping_table[map_idx];
					if (mt_entry->dev_handle) {
						_mapping_add_to_removal_table
						    (ioc, mt_entry->dev_handle,
						     0);
						is_removed = 1;
					}
					mt_entry->init_complete = 0;
				}
			}
			if (map_idx != MPT2SAS_MAPTABLE_BAD_IDX) {
				mt_entry = &ioc->mapping_table[map_idx];
				mt_entry->physical_id = phy_change->physical_id;
				mt_entry->channel = 0;
				mt_entry->id = map_idx;
				mt_entry->dev_handle = phy_change->dev_handle;
				mt_entry->missing_count = 0;
				mt_entry->device_info = phy_change->device_info
				    | (MPT2SAS_DEV_RESERVED |
					MPT2SAS_MAP_IN_USE);
			} else {
				phy_change->is_processed = 1;
				if (!ioc->mt_full_retry) {
					ioc->mt_add_device_failed = 1;
					continue;
				}
				printk(MPT2SAS_ERR_FMT
				    "%s Failed adding the device with handle"
				    "(0x%04x) since there is no free space"
				    " available in the map table\n",
				    ioc->name, __func__,
				    phy_change->dev_handle);
				continue;
			}
			if (ioc->is_dpm_enable) {
				if (mt_entry->dpm_entry_num !=
				    MPT2SAS_DPM_BAD_IDX) {
					dpm_idx = mt_entry->dpm_entry_num;
					dpm_entry = (Mpi2DriverMap0Entry_t *)
					    ((u8 *)ioc->dpm_pg0 + hdr_sz);
					dpm_entry += dpm_idx;
					missing_cnt = dpm_entry->
					    MappingInformation &
					    MPI2_DRVMAP0_MAPINFO_MISSING_MASK;

					if ((mt_entry->physical_id ==
					    le64_to_cpu(dpm_entry->
					    PhysicalIdentifier)) &&
					    !missing_cnt)
						mt_entry->init_complete = 1;
				} else {
					dpm_idx = _mapping_get_free_dpm_idx
					    (ioc);
					mt_entry->init_complete = 0;
				}
				if (dpm_idx != MPT2SAS_DPM_BAD_IDX &&
				    !mt_entry->init_complete) {
					mt_entry->init_complete = 1;
					mt_entry->dpm_entry_num = dpm_idx;
					dpm_entry = (Mpi2DriverMap0Entry_t *)
					    ((u8 *)ioc->dpm_pg0 + hdr_sz);
					dpm_entry += dpm_idx;
					dpm_entry->PhysicalIdentifier =
					    cpu_to_le64(mt_entry->physical_id);
					dpm_entry->DeviceIndex = (U16) map_idx;
					dpm_entry->MappingInformation = 0;
					dpm_entry->PhysicalBitsMapping = 0;
					ioc->dpm_entry_used[dpm_idx] = 1;
					ioc->dpm_flush_entry[dpm_idx] = 1;
					phy_change->is_processed = 1;
				} else if (dpm_idx == MPT2SAS_DPM_BAD_IDX) {
						phy_change->is_processed = 1;
						printk(MPT2SAS_ERR_FMT
						   "%s Failed adding the dev "
						   " with handle(0x%04x) to "
						   "persistent table since "
						   "there is no free space "
						   "available\n", ioc->name,
						   __func__, phy_change->
						   dev_handle);
				}
			}
			mt_entry->init_complete = 1;
		}

		phy_change->is_processed = 1;
	}
	if (is_removed)
		_mapping_clear_removed_entries(ioc);
}

/**
 * _mapping_flush_dpm_pages -Flush the DPM pages to NVRAM
 * @ioc: per adapter object
 *
 * Returns nothing
 */
static void
_mapping_flush_dpm_pages(struct MPT2SAS_ADAPTER *ioc)
{
	Mpi2DriverMap0Entry_t *dpm_entry;
	Mpi2ConfigReply_t mpi_reply;
	Mpi2DriverMappingPage0_t config_page;
	u16 entry_num ;

	for (entry_num = 0; entry_num < ioc->max_dpm_entries; entry_num++) {
		if (!ioc->dpm_flush_entry[entry_num])
			continue;
		memset(&config_page, 0, sizeof(Mpi2DriverMappingPage0_t));
		memcpy(&config_page.Header, (u8 *) ioc->dpm_pg0,
			    sizeof(MPI2_CONFIG_EXTENDED_PAGE_HEADER));
		dpm_entry = (Mpi2DriverMap0Entry_t *) ((u8 *) ioc->dpm_pg0 +
			    sizeof(MPI2_CONFIG_EXTENDED_PAGE_HEADER));
		dpm_entry += entry_num;
		dpm_entry->MappingInformation = cpu_to_le16(dpm_entry->
		    MappingInformation);
		dpm_entry->DeviceIndex = cpu_to_le16(dpm_entry->DeviceIndex);
		dpm_entry->PhysicalBitsMapping = cpu_to_le32(dpm_entry->
		    PhysicalBitsMapping);
		memcpy(&config_page.Entry, (u8 *)dpm_entry,
		    sizeof(Mpi2DriverMap0Entry_t));
		/* TODO-How to handle failed writes? */
		if (mpt2sas_config_set_dpm_pg0(ioc, &mpi_reply, &config_page,
		    entry_num)) {
			dhmprintk(ioc, printk(MPT2SAS_WARN_FMT "%s: "
			    "Writing dpm entry %d for device failed\n",
			    ioc->name, __func__, entry_num));
		} else
			ioc->dpm_flush_entry[entry_num] = 0;
		dpm_entry->MappingInformation = le16_to_cpu(dpm_entry->
		    MappingInformation);
		dpm_entry->DeviceIndex = le16_to_cpu(dpm_entry->DeviceIndex);
		dpm_entry->PhysicalBitsMapping = le32_to_cpu(dpm_entry->
		    PhysicalBitsMapping);
	}
}

/**
 * _mapping_allocate_memory- allocates the memory required for mapping tables
 * @ioc: per adapter object
 *
 * Allocates the memory for all the tables required for host mapping
 *
 * Return 0 on success or non-zero on failure.
 */
static int
_mapping_allocate_memory(struct MPT2SAS_ADAPTER *ioc)
{
	u32 dpm_pg0_sz;

	ioc->mapping_table = kzalloc((sizeof(struct dev_mapping_table) *
	    ioc->max_devices), GFP_KERNEL);
	if (!ioc->mapping_table)
		goto free_resources;

	ioc->removal_table = kzalloc((sizeof(struct map_removal_table) *
	    ioc->max_devices), GFP_KERNEL);
	if (!ioc->removal_table)
		goto free_resources;

	ioc->enclosure_table = kzalloc((sizeof(struct enc_mapping_table) *
	    ioc->max_enclosures), GFP_KERNEL);
	if (!ioc->enclosure_table)
		goto free_resources;

	ioc->dpm_entry_used = kzalloc((sizeof(u8) * ioc->max_dpm_entries),
	    GFP_KERNEL);
	if (!ioc->dpm_entry_used)
		goto free_resources;

	ioc->dpm_flush_entry = kzalloc((sizeof(u8) * ioc->max_dpm_entries),
	    GFP_KERNEL);
	if (!ioc->dpm_flush_entry)
		goto free_resources;

	dpm_pg0_sz = sizeof(MPI2_CONFIG_EXTENDED_PAGE_HEADER) +
	    (ioc->max_dpm_entries * sizeof(MPI2_CONFIG_PAGE_DRIVER_MAP0_ENTRY));

	ioc->dpm_pg0 = kzalloc(dpm_pg0_sz, GFP_KERNEL);

	if (!ioc->dpm_pg0) {
		dhmprintk(ioc, printk(MPT2SAS_WARN_FMT "%s: "
		    "Memory allocation failed for dpm page disabling dpm\n",
		    ioc->name, __func__));
		ioc->is_dpm_enable = 0;
	}

	return 0;

free_resources:
	kfree(ioc->mapping_table);
	kfree(ioc->removal_table);
	kfree(ioc->enclosure_table);
	kfree(ioc->dpm_entry_used);
	kfree(ioc->dpm_flush_entry);
	kfree(ioc->dpm_pg0);
	printk(MPT2SAS_ERR_FMT "%s device initialization failed due to failure"
	    " in mapping table memory allocation\n", ioc->name, __func__);
	return -1;
}

/**
 * mpt2sas_mapping_free_memory- frees the memory allocated for mapping tables
 * @ioc: per adapter object
 *
 * Returns nothing.
 */
void
mpt2sas_mapping_free_memory(struct MPT2SAS_ADAPTER *ioc)
{
	kfree(ioc->mapping_table);
	kfree(ioc->removal_table);
	kfree(ioc->enclosure_table);
	kfree(ioc->dpm_entry_used);
	kfree(ioc->dpm_flush_entry);
	kfree(ioc->dpm_pg0);
}

/**
 * _mapping_process_dpm_pg0- partially initialize the mapping table from dpmpg0
 * @ioc: per adapter object
 *
 * Process the driver persistent page 0 read from the firmware
 *
 * Returns nothing.
 */
static void
_mapping_process_dpm_pg0(struct MPT2SAS_ADAPTER *ioc)
{
	u8 missing_cnt, enc_idx;
	u16 slot_id, entry_num, num_slots;
	u32 map_idx, dev_idx, start_idx, end_idx;
	struct dev_mapping_table *mt_entry;
	Mpi2DriverMap0Entry_t *dpm_entry;
	u16 ioc_pg8_flags = le16_to_cpu(ioc->ioc_pg8.Flags);
	u16 max_num_phy_ids = le16_to_cpu(ioc->ioc_pg8.MaxNumPhysicalMappedIDs);
	struct enc_mapping_table *et_entry;
	u32 phy_bits = 0;

	if (ioc->ir_firmware)
		_mapping_get_ir_maprange(ioc, &start_idx, &end_idx);

	dpm_entry = (Mpi2DriverMap0Entry_t *) ((u8 *) ioc->dpm_pg0 +
	    sizeof(MPI2_CONFIG_EXTENDED_PAGE_HEADER));

	for (entry_num = 0; entry_num < ioc->max_dpm_entries; entry_num++,
	    dpm_entry++) {
		if (!dpm_entry->PhysicalIdentifier) {
			ioc->dpm_entry_used[entry_num] = 0;
			continue;
		}
		ioc->dpm_entry_used[entry_num] = 1;
		dpm_entry->MappingInformation = le16_to_cpu(dpm_entry->
		    MappingInformation);
		missing_cnt = dpm_entry->MappingInformation &
		    MPI2_DRVMAP0_MAPINFO_MISSING_MASK;
		dev_idx = le16_to_cpu(dpm_entry->DeviceIndex);
		phy_bits = le32_to_cpu(dpm_entry->PhysicalBitsMapping);

		if (ioc->ir_firmware && (dev_idx >= start_idx) &&
		    (dev_idx <= end_idx)) {
			mt_entry = &ioc->mapping_table[dev_idx];
			mt_entry->physical_id = le64_to_cpu(dpm_entry->
			    PhysicalIdentifier);
			mt_entry->channel = MPT2SAS_RAID_CHANNEL;
			mt_entry->id = dev_idx;
			mt_entry->missing_count = missing_cnt;
			mt_entry->dpm_entry_num = entry_num;
			mt_entry->device_info = MPT2SAS_DEV_RESERVED;
			continue;
		}
		if ((ioc_pg8_flags & MPI2_IOCPAGE8_FLAGS_MASK_MAPPING_MODE)
			== MPI2_IOCPAGE8_FLAGS_ENCLOSURE_SLOT_MAPPING) {

			if (dev_idx <  (ioc->num_rsvd_entries +
				max_num_phy_ids)) {
				slot_id = 0;
				if (ioc_pg8_flags &
				    MPI2_IOCPAGE8_FLAGS_DA_START_SLOT_1)
					slot_id = 1;
				num_slots = max_num_phy_ids;
			} else {
				slot_id = 0;
				num_slots = dpm_entry->MappingInformation &
				    MPI2_DRVMAP0_MAPINFO_SLOT_MASK;
				num_slots >>= MPI2_DRVMAP0_MAPINFO_SLOT_SHIFT;
			}
			enc_idx = ioc->num_enc_table_entries;
			if (enc_idx >= ioc->max_enclosures) {
				printk(MPT2SAS_ERR_FMT "%s enclosure entries "
				    "exceed max enclosures %d\n", ioc->name,
				    __func__, ioc->max_enclosures);
				break;
			}
			ioc->num_enc_table_entries++;
			et_entry = &ioc->enclosure_table[enc_idx];
			et_entry->enclosure_id = le64_to_cpu(dpm_entry->
			    PhysicalIdentifier);
			et_entry->start_index = dev_idx;
			et_entry->dpm_entry_num = entry_num;
			et_entry->num_slots = num_slots;
			et_entry->start_slot = slot_id;
			et_entry->missing_count = missing_cnt;
			et_entry->phy_bits = phy_bits;

			mt_entry = &ioc->mapping_table[dev_idx];
			for (map_idx = dev_idx; map_idx < (dev_idx + num_slots);
			    map_idx++, mt_entry++) {
				if (mt_entry->dpm_entry_num !=
				    MPT2SAS_DPM_BAD_IDX) {
					printk(MPT2SAS_ERR_FMT
					    "%s conflict in mapping table for "
					    "enclosure %d\n", ioc->name,
					    __func__, enc_idx);
					break;
				}
				mt_entry->physical_id = le64_to_cpu(dpm_entry->
				    PhysicalIdentifier);
				mt_entry->phy_bits = phy_bits;
				mt_entry->channel = 0;
				mt_entry->id = dev_idx;
				mt_entry->dpm_entry_num = entry_num;
				mt_entry->missing_count = missing_cnt;
				mt_entry->device_info = MPT2SAS_DEV_RESERVED;
			}

		} else if ((ioc_pg8_flags &
			MPI2_IOCPAGE8_FLAGS_MASK_MAPPING_MODE)
			== MPI2_IOCPAGE8_FLAGS_DEVICE_PERSISTENCE_MAPPING) {
			map_idx = dev_idx;
			mt_entry = &ioc->mapping_table[map_idx];
			if (mt_entry->dpm_entry_num != MPT2SAS_DPM_BAD_IDX) {
				printk(MPT2SAS_ERR_FMT "%s conflict in mapping"
				   " table for device %d\n", ioc->name,
				   __func__, map_idx);
				break;
			}
			mt_entry->physical_id = le64_to_cpu(dpm_entry->
			    PhysicalIdentifier);
			mt_entry->phy_bits = phy_bits;
			mt_entry->channel = 0;
			mt_entry->id = dev_idx;
			mt_entry->missing_count = missing_cnt;
			mt_entry->dpm_entry_num = entry_num;
			mt_entry->device_info = MPT2SAS_DEV_RESERVED;
		}
	} /*close the loop for DPM table */
}

/**
 * mpt2sas_mapping_check_devices - start of the day check for device availabilty
 * @ioc: per adapter object
 * @sleep_flag: Flag indicating whether this function can sleep or not
 *
 * Returns nothing.
 */
void
mpt2sas_mapping_check_devices(struct MPT2SAS_ADAPTER *ioc, int sleep_flag)
{

	u32 cntdn, i;
	u32 timeout = 60;
	struct dev_mapping_table *mt_entry;
	u16 ioc_pg8_flags = le16_to_cpu(ioc->ioc_pg8.Flags);
	struct enc_mapping_table *et_entry;
	u32 start_idx, end_idx;


	ioc->track_mapping_events = 0;
	cntdn = (sleep_flag == CAN_SLEEP) ? 1000*timeout : 2000*timeout;
	do {
		if (!ioc->pending_map_events)
			break;
		if (sleep_flag == CAN_SLEEP)
			msleep(1);
		else
			udelay(500);
	} while (--cntdn);

	if (!cntdn)
		dhmprintk(ioc, printk(MPT2SAS_WARN_FMT "%s: there are %d"
		    " pending events after %d seconds of delay\n",
		    ioc->name, __func__, ioc->pending_map_events, timeout));
	ioc->pending_map_events = 0;

	if ((ioc_pg8_flags & MPI2_IOCPAGE8_FLAGS_MASK_MAPPING_MODE)
	    == MPI2_IOCPAGE8_FLAGS_ENCLOSURE_SLOT_MAPPING) {
		et_entry = ioc->enclosure_table;
		for (i = 0; i < ioc->num_enc_table_entries; i++, et_entry++) {
			if (!et_entry->init_complete) {
				if (et_entry->missing_count <
				    MPT2SAS_MAX_MISSING_COUNT) {
					et_entry->missing_count++;
					if (et_entry->dpm_entry_num !=
					    MPT2SAS_DPM_BAD_IDX)
						_mapping_commit_enc_entry(ioc,
						    et_entry);
				}
				et_entry->init_complete = 1;
			}
		}
		if (!ioc->ir_firmware)
			return;
		_mapping_get_ir_maprange(ioc, &start_idx, &end_idx);
		mt_entry = &ioc->mapping_table[start_idx];
		for (i = start_idx; i < (end_idx + 1); i++, mt_entry++) {
			if (mt_entry->device_info & MPT2SAS_DEV_RESERVED
			    && !mt_entry->physical_id)
				mt_entry->init_complete = 1;
			else if (mt_entry->device_info & MPT2SAS_DEV_RESERVED) {
				if (!mt_entry->init_complete) {
					if (mt_entry->missing_count <
					    MPT2SAS_MAX_MISSING_COUNT) {
						mt_entry->missing_count++;
						if (mt_entry->dpm_entry_num !=
						    MPT2SAS_DPM_BAD_IDX)
						_mapping_commit_map_entry(ioc,
						    mt_entry);
					}
					mt_entry->init_complete = 1;
				}
			}
		}

	} else if ((ioc_pg8_flags & MPI2_IOCPAGE8_FLAGS_MASK_MAPPING_MODE)
	    == MPI2_IOCPAGE8_FLAGS_DEVICE_PERSISTENCE_MAPPING) {
		mt_entry = ioc->mapping_table;
		for (i = 0; i < ioc->max_devices; i++, mt_entry++) {
			if (mt_entry->device_info & MPT2SAS_DEV_RESERVED
			    && !mt_entry->physical_id)
				mt_entry->init_complete = 1;
			else if (mt_entry->device_info & MPT2SAS_DEV_RESERVED) {
				if (!mt_entry->init_complete) {
					if (mt_entry->missing_count <
					    MPT2SAS_MAX_MISSING_COUNT) {
						mt_entry->missing_count++;
						if (mt_entry->dpm_entry_num !=
						    MPT2SAS_DPM_BAD_IDX)
						_mapping_commit_map_entry(ioc,
						    mt_entry);
					}
					mt_entry->init_complete = 1;
				}
			}
		}
	}

}

/**
 * mpt2sas_mapping_is_reinit_required - check whether event replay required
 * @ioc: per adapter object
 *
 * Checks the per ioc flags and decide whether reinit of events required
 *
 * Returns 1 for reinit of ioc 0 for not.
 */
int mpt2sas_mapping_is_reinit_required(struct MPT2SAS_ADAPTER *ioc)
{
	if (!ioc->mt_full_retry && ioc->mt_add_device_failed) {
		ioc->mt_full_retry = 1;
		ioc->mt_add_device_failed = 0;
		_mapping_flush_dpm_pages(ioc);
		return 1;
	}
	ioc->mt_full_retry = 1;
	return 0;
}

/**
 * mpt2sas_mapping_initialize - initialize mapping tables
 * @ioc: per adapter object
 *
 * Read controller persitant mapping tables into internal data area.
 *
 * Return 0 for success or non-zero for failure.
 */
int
mpt2sas_mapping_initialize(struct MPT2SAS_ADAPTER *ioc)
{
	u16 volume_mapping_flags, dpm_pg0_sz;
	u32 i;
	Mpi2ConfigReply_t mpi_reply;
	int r;
	u8 retry_count;
	u16 ioc_pg8_flags = le16_to_cpu(ioc->ioc_pg8.Flags);

	/* The additional 1 accounts for the virtual enclosure
	 * created for the controller
	 */
	ioc->max_enclosures = ioc->facts.MaxEnclosures + 1;
	ioc->max_expanders = ioc->facts.MaxSasExpanders;
	ioc->max_volumes = ioc->facts.MaxVolumes;
	ioc->max_devices = ioc->facts.MaxTargets + ioc->max_volumes;
	ioc->pending_map_events = 0;
	ioc->num_enc_table_entries = 0;
	ioc->num_rsvd_entries = 0;
	ioc->num_channels = 1;
	ioc->max_dpm_entries = le16_to_cpu(ioc->ioc_pg8.MaxPersistentEntries);
	ioc->is_dpm_enable = (ioc->max_dpm_entries) ? 1 : 0;
	ioc->track_mapping_events = 0;


	if (ioc_pg8_flags & MPI2_IOCPAGE8_FLAGS_DISABLE_PERSISTENT_MAPPING)
		ioc->is_dpm_enable = 0;

	if (ioc_pg8_flags & MPI2_IOCPAGE8_FLAGS_RESERVED_TARGETID_0)
		ioc->num_rsvd_entries = 1;

	volume_mapping_flags = le16_to_cpu(ioc->ioc_pg8.IRVolumeMappingFlags) &
	    MPI2_IOCPAGE8_IRFLAGS_MASK_VOLUME_MAPPING_MODE;
	if (ioc->ir_firmware && (volume_mapping_flags ==
	    MPI2_IOCPAGE8_IRFLAGS_LOW_VOLUME_MAPPING))
		ioc->num_rsvd_entries += ioc->max_volumes;

	if (ioc->wait_for_port_enable_to_complete) {
		r = _mapping_allocate_memory(ioc);
		if (r)
			return r;
	}

	for (i = 0; i < ioc->max_devices; i++)
		_mapping_clear_map_entry(ioc->mapping_table + i);

	for (i = 0; i < ioc->max_enclosures; i++)
		_mapping_clear_enc_entry(ioc->enclosure_table + i);

	for (i = 0; i < ioc->max_devices; i++) {
		ioc->removal_table[i].dev_handle = 0;
		ioc->removal_table[i].dpm_entry_num = MPT2SAS_DPM_BAD_IDX;
	}

	memset(ioc->dpm_entry_used, 0, ioc->max_dpm_entries);
	memset(ioc->dpm_flush_entry, 0, ioc->max_dpm_entries);

	if (ioc->is_dpm_enable) {
		dpm_pg0_sz = sizeof(MPI2_CONFIG_EXTENDED_PAGE_HEADER) +
		    (ioc->max_dpm_entries *
		     sizeof(MPI2_CONFIG_PAGE_DRIVER_MAP0_ENTRY));
		retry_count = 0;
retry_read_dpm:
		if (mpt2sas_config_get_dpm_pg0(ioc, &mpi_reply, ioc->dpm_pg0,
		    dpm_pg0_sz)) {
			dhmprintk(ioc, printk(MPT2SAS_WARN_FMT "%s: "
			    "dpm page read failed disabling dpm\n", ioc->name,
			    __func__));
			if (retry_count < 3) {
				retry_count++;
				goto retry_read_dpm;
			}
			ioc->is_dpm_enable = 0;
		}
	}

	if (ioc->is_dpm_enable)
		_mapping_process_dpm_pg0(ioc);

	ioc->track_mapping_events = 1;
	return 0;
}

/**
 * mpt2sas_mapping_exit - clear mapping table and associated memory
 * @ioc: per adapter object
 *
 * Returns nothing.
 */
void
mpt2sas_mapping_exit(struct MPT2SAS_ADAPTER *ioc)
{
	_mapping_flush_dpm_pages(ioc);
	mpt2sas_mapping_free_memory(ioc);
}

/**
 * mpt2sas_mapping_get_sas_id - assign a target id for sas device
 * @ioc: per adapter object
 * @sas_address: sas address of the device
 * @handle: device handle
 *
 * Returns valid ID on success or BAD_ID.
 */
unsigned int
mpt2sas_mapping_get_sas_id(struct MPT2SAS_ADAPTER *ioc,
    __le64 sas_address, u16 handle)
{
	u32 map_idx;
	struct dev_mapping_table *mt_entry;

	for (map_idx = 0; map_idx < ioc->max_devices; map_idx++) {
		mt_entry = &ioc->mapping_table[map_idx];
		if (mt_entry->dev_handle == handle &&
		  mt_entry->physical_id == sas_address)
			return mt_entry->id;
	}

	return MPT2SAS_MAP_BAD_ID;
}

/**
 * mpt2sas_mapping_get_raid_id - assign a target id for raid device
 * @ioc: per adapter object
 * @wwid: world wide identifier for raid volume
 * @handle: device handle
 *
 * Returns valid ID on success or BAD_ID.
 */
unsigned int
mpt2sas_mapping_get_raid_id(struct MPT2SAS_ADAPTER *ioc, u64 wwid, u16 handle)
{
	u32 map_idx;
	struct dev_mapping_table *mt_entry;

	for (map_idx = 0; map_idx < ioc->max_devices; map_idx++) {
		mt_entry = &ioc->mapping_table[map_idx];
		if (mt_entry->dev_handle == handle &&
		  mt_entry->physical_id == wwid)
			return mt_entry->id;
	}

	return MPT2SAS_MAP_BAD_ID;
}


/**
 * mpt2sas_mapping_enclosure_dev_status_change_event - handle enclosure events
 * @ioc: per adapter object
 * @event_data: event data payload
 *
 * Return nothing.
 */
void
mpt2sas_mapping_enclosure_dev_status_change_event(struct MPT2SAS_ADAPTER *ioc,
    Mpi2EventDataSasEnclDevStatusChange_t *event_data)
{
	u8 enc_idx, missing_count;
	struct enc_mapping_table *et_entry;
	Mpi2DriverMap0Entry_t *dpm_entry;
	u16 ioc_pg8_flags = le16_to_cpu(ioc->ioc_pg8.Flags);
	u8 map_shift = MPI2_DRVMAP0_MAPINFO_SLOT_SHIFT;
	u8 update_phy_bits = 0;
	u32 saved_phy_bits;


	if ((ioc_pg8_flags & MPI2_IOCPAGE8_FLAGS_MASK_MAPPING_MODE)
	    != MPI2_IOCPAGE8_FLAGS_ENCLOSURE_SLOT_MAPPING)
		goto out;

	dpm_entry = (Mpi2DriverMap0Entry_t *)((u8 *)ioc->dpm_pg0 +
	    sizeof(MPI2_CONFIG_EXTENDED_PAGE_HEADER));

	if (event_data->ReasonCode == MPI2_EVENT_SAS_ENCL_RC_ADDED) {
		if (!event_data->NumSlots) {
			printk(MPT2SAS_ERR_FMT "%s: Enclosure with handle"
			    " =0x%x\n reported with 0 slots", ioc->name,
			    __func__, le16_to_cpu(event_data->EnclosureHandle));
			goto out;
		}
		enc_idx = _mapping_get_enc_idx_from_id(ioc,
		    event_data->EnclosureLogicalID, event_data->PhyBits);
		if (enc_idx != MPT2SAS_ENCTABLE_BAD_IDX) {
			et_entry = &ioc->enclosure_table[enc_idx];
			if (et_entry->init_complete &&
			    !et_entry->missing_count) {
				printk(MPT2SAS_ERR_FMT "%s:"
				    "Enclosure %d is already present with"
				    " handle=0x%x\n", ioc->name, __func__,
				    enc_idx, et_entry->enc_handle);
				goto out;
			}
			et_entry->enc_handle = le16_to_cpu(event_data->
			    EnclosureHandle);
			et_entry->start_slot = le16_to_cpu(event_data->
			    StartSlot);
			saved_phy_bits = et_entry->phy_bits;
			et_entry->phy_bits |= le32_to_cpu(event_data->PhyBits);
			if (saved_phy_bits != et_entry->phy_bits)
				update_phy_bits = 1;
			if (et_entry->missing_count || update_phy_bits) {
				et_entry->missing_count = 0;
				if (ioc->is_dpm_enable &&
				    et_entry->dpm_entry_num
				    != MPT2SAS_DPM_BAD_IDX) {
					dpm_entry += et_entry->dpm_entry_num;
					missing_count =
					    (u8)(dpm_entry->MappingInformation &
					    MPI2_DRVMAP0_MAPINFO_MISSING_MASK);
					if (!et_entry->init_complete && (
					    missing_count || update_phy_bits)) {
						dpm_entry->MappingInformation
						    = et_entry->num_slots;
						dpm_entry->MappingInformation
						    <<= map_shift;
						dpm_entry->PhysicalBitsMapping
						    = et_entry->phy_bits;
						ioc->dpm_flush_entry[et_entry->
						    dpm_entry_num] = 1;
					}
				}
			}

		} else {
			enc_idx = ioc->num_enc_table_entries;
			if (enc_idx >= ioc->max_enclosures) {
				printk(MPT2SAS_ERR_FMT "%s:"
				    "Enclosure can not be added as the table"
				    "reached maximum size", ioc->name,
				    __func__);
				goto out;
			}
			ioc->num_enc_table_entries++;
			et_entry = &ioc->enclosure_table[enc_idx];
			et_entry->enc_handle = le16_to_cpu(event_data->
			    EnclosureHandle);
			et_entry->enclosure_id = le64_to_cpu(event_data->
			    EnclosureLogicalID);
			et_entry->start_index = MPT2SAS_MAPTABLE_BAD_IDX;
			et_entry->dpm_entry_num = MPT2SAS_DPM_BAD_IDX;
			et_entry->num_slots = le16_to_cpu(event_data->NumSlots);
			et_entry->start_slot = le16_to_cpu(event_data->
			    StartSlot);
			et_entry->phy_bits = le32_to_cpu(event_data->PhyBits);
		}
		et_entry->init_complete = 1;
	} else if (event_data->ReasonCode ==
	    MPI2_EVENT_SAS_ENCL_RC_NOT_RESPONDING) {
		enc_idx = _mapping_get_enc_idx_from_handle(ioc,
		    le16_to_cpu(event_data->EnclosureHandle));
		if (enc_idx == MPT2SAS_ENCTABLE_BAD_IDX) {
			printk(MPT2SAS_ERR_FMT "%s: Can not "
			    "unmap the enclosure with enc_idx %d as it is "
			    "already deleted", ioc->name, __func__, enc_idx);
			goto out;
		}
		et_entry = &ioc->enclosure_table[enc_idx];
		if (!et_entry->init_complete) {
			if (et_entry->missing_count < MPT2SAS_MAX_MISSING_COUNT)
				et_entry->missing_count++;
			else
				et_entry->init_complete = 1;
		}
		if (!et_entry->missing_count)
			et_entry->missing_count++;
		if (ioc->is_dpm_enable && !et_entry->init_complete &&
		    et_entry->dpm_entry_num != MPT2SAS_DPM_BAD_IDX) {
			dpm_entry += et_entry->dpm_entry_num;
			dpm_entry->MappingInformation = et_entry->num_slots;
			dpm_entry->MappingInformation <<= map_shift;
			dpm_entry->MappingInformation |=
			    et_entry->missing_count;
			ioc->dpm_flush_entry[et_entry->dpm_entry_num] = 1;
		}
		et_entry->init_complete = 1;
	}

out:
	_mapping_flush_dpm_pages(ioc);
	if (ioc->pending_map_events)
		ioc->pending_map_events--;
}

/**
 * mpt2sas_mapping_topology_change_event - handle topology change events
 * @ioc: per adapter object
 * @event_data: event data payload
 *
 * Returns nothing.
 */
void
mpt2sas_mapping_topology_change_event(struct MPT2SAS_ADAPTER *ioc,
    Mpi2EventDataSasTopologyChangeList_t *event_data)
{
	struct _map_topology_change topo_change;
	struct _map_phy_change *phy_change;
	Mpi2EventSasTopoPhyEntry_t *event_phy_change;
	u8 i, num_entries;

	topo_change.enc_handle = le16_to_cpu(event_data->EnclosureHandle);
	topo_change.exp_handle = le16_to_cpu(event_data->ExpanderDevHandle);
	num_entries = event_data->NumEntries;
	topo_change.num_entries = num_entries;
	topo_change.start_phy_num = event_data->StartPhyNum;
	topo_change.num_phys = event_data->NumPhys;
	topo_change.exp_status = event_data->ExpStatus;
	event_phy_change = event_data->PHY;

	if (!num_entries)
		goto out;
	phy_change = kzalloc(sizeof(struct _map_phy_change) * num_entries,
	    GFP_KERNEL);
	topo_change.phy_details = phy_change;
	if (!phy_change)
		goto out;
	for (i = 0; i < num_entries; i++, event_phy_change++, phy_change++) {
		phy_change->dev_handle = le16_to_cpu(event_phy_change->
		    AttachedDevHandle);
		phy_change->reason = event_phy_change->PhyStatus &
		    MPI2_EVENT_SAS_TOPO_RC_MASK;
	}
	_mapping_update_missing_count(ioc, &topo_change);
	_mapping_get_dev_info(ioc, &topo_change);
	_mapping_clear_removed_entries(ioc);
	_mapping_add_new_device(ioc, &topo_change);

out:
	kfree(topo_change.phy_details);
	_mapping_flush_dpm_pages(ioc);
	if (ioc->pending_map_events)
		ioc->pending_map_events--;
}
void
mpt2sas_mapping_update_table_for_sata(struct MPT2SAS_ADAPTER *ioc, u16 handle,
    u16 parent_handle)
{
	struct _map_topology_change topo_change;
	struct _map_phy_change *phy_change;
	u16 ioc_pg8_flags = le16_to_cpu(ioc->ioc_pg8.Flags);

	if ((ioc_pg8_flags & MPI2_IOCPAGE8_FLAGS_MASK_MAPPING_MODE)
	    == MPI2_IOCPAGE8_FLAGS_ENCLOSURE_SLOT_MAPPING)
		return;

	if (parent_handle <= ioc->sas_hba.num_phys)
		topo_change.exp_handle = 0;
	else
		topo_change.exp_handle = parent_handle;
	topo_change.enc_handle = 0;
	topo_change.num_entries = 2;
	topo_change.start_phy_num = 0;
	topo_change.num_phys = 0;
	topo_change.exp_status = 0;

	phy_change = kzalloc(sizeof(struct _map_phy_change) * 2,
	    GFP_KERNEL);
	if (!phy_change)
		return;
	topo_change.phy_details = phy_change;
	phy_change->dev_handle = handle;
	phy_change->reason = MPI2_EVENT_SAS_TOPO_RC_TARG_NOT_RESPONDING;
	phy_change++;
	phy_change->dev_handle = handle;
	phy_change->reason = MPI2_EVENT_SAS_TOPO_RC_TARG_ADDED;
	_mapping_update_missing_count(ioc, &topo_change);
	_mapping_get_dev_info(ioc, &topo_change);
	_mapping_clear_removed_entries(ioc);
	_mapping_add_new_device(ioc, &topo_change);
	_mapping_flush_dpm_pages(ioc);
	kfree(topo_change.phy_details);
	return;
}

/**
 * _mapping_check_update_ir_mt_idx - Check and update IR map table index
 * @ioc: per adapter object
 * @event_data: event data payload
 * @evt_idx: current event index
 * @map_idx: current index and the place holder for new map table index
 * @wwid_table: world wide name for volumes in the element table
 *
 * pass through IR events and find whether any events matches and if so
 * tries to find new index if not returns failure
 *
 * Returns 0 on success and 1 on failure
 */
static int
_mapping_check_update_ir_mt_idx(struct MPT2SAS_ADAPTER *ioc,
    Mpi2EventDataIrConfigChangeList_t *event_data, int evt_idx, u32 *map_idx,
    u64 *wwid_table)
{
	struct dev_mapping_table *mt_entry;
	u32 st_idx, end_idx, mt_idx = *map_idx;
	u8 match = 0;
	Mpi2EventIrConfigElement_t *element;
	u16 element_flags;
	int i;

	mt_entry = &ioc->mapping_table[mt_idx];
	_mapping_get_ir_maprange(ioc, &st_idx, &end_idx);
search_again:
	match = 0;
	for (i = evt_idx + 1; i < event_data->NumElements; i++) {
		element = (Mpi2EventIrConfigElement_t *)
		    &event_data->ConfigElement[i];
		element_flags = le16_to_cpu(element->ElementFlags);
		if ((element_flags &
		    MPI2_EVENT_IR_CHANGE_EFLAGS_ELEMENT_TYPE_MASK) !=
		    MPI2_EVENT_IR_CHANGE_EFLAGS_VOLUME_ELEMENT)
			continue;
		if (element->ReasonCode == MPI2_EVENT_IR_CHANGE_RC_ADDED ||
		    element->ReasonCode ==
		    MPI2_EVENT_IR_CHANGE_RC_VOLUME_CREATED) {
			if (mt_entry->physical_id == wwid_table[i]) {
				match = 1;
				break;
			}
		}
	}

	if (match) {
		do {
			mt_idx++;
			if (mt_idx > end_idx)
				return 1;
			mt_entry = &ioc->mapping_table[mt_idx];
		} while (mt_entry->device_info & MPT2SAS_MAP_IN_USE);
		goto search_again;
	}
	*map_idx = mt_idx;
	return 0;
}

/**
 * mpt2sas_mapping_ir_config_change_event - handle IR config change list events
 * @ioc: per adapter object
 * @event_data: event data payload
 *
 * Returns nothing.
 */
void
mpt2sas_mapping_ir_config_change_event(struct MPT2SAS_ADAPTER *ioc,
    Mpi2EventDataIrConfigChangeList_t *event_data)
{
	Mpi2EventIrConfigElement_t *element;
	int i;
	u64 *wwid_table;
	u32 map_idx, flags;
	struct dev_mapping_table *mt_entry;
	u16 element_flags;
	u8 log_full_error = 0;

	wwid_table = kzalloc(sizeof(u64) * event_data->NumElements, GFP_KERNEL);
	if (!wwid_table)
		goto out;
	element = (Mpi2EventIrConfigElement_t *)&event_data->ConfigElement[0];
	flags = le32_to_cpu(event_data->Flags);
	for (i = 0; i < event_data->NumElements; i++, element++) {
		element_flags = le16_to_cpu(element->ElementFlags);
		if ((element->ReasonCode != MPI2_EVENT_IR_CHANGE_RC_ADDED) &&
		    (element->ReasonCode != MPI2_EVENT_IR_CHANGE_RC_REMOVED) &&
		    (element->ReasonCode != MPI2_EVENT_IR_CHANGE_RC_NO_CHANGE)
		    && (element->ReasonCode !=
			MPI2_EVENT_IR_CHANGE_RC_VOLUME_CREATED))
			continue;
		if ((element_flags &
		    MPI2_EVENT_IR_CHANGE_EFLAGS_ELEMENT_TYPE_MASK) ==
		    MPI2_EVENT_IR_CHANGE_EFLAGS_VOLUME_ELEMENT) {
			mpt2sas_config_get_volume_wwid(ioc,
			    le16_to_cpu(element->VolDevHandle), &wwid_table[i]);
			map_idx = _mapping_get_ir_mt_idx_from_wwid(ioc,
			    wwid_table[i]);
			if (map_idx != MPT2SAS_MAPTABLE_BAD_IDX) {
				mt_entry = &ioc->mapping_table[map_idx];
				mt_entry->device_info |= MPT2SAS_MAP_IN_USE;
			}
		}
	}
	if (flags == MPI2_EVENT_IR_CHANGE_FLAGS_FOREIGN_CONFIG)
		goto out;
	else {
		element = (Mpi2EventIrConfigElement_t *)&event_data->
		    ConfigElement[0];
		for (i = 0; i < event_data->NumElements; i++, element++) {
			if (element->ReasonCode ==
			    MPI2_EVENT_IR_CHANGE_RC_ADDED ||
			    element->ReasonCode ==
			    MPI2_EVENT_IR_CHANGE_RC_VOLUME_CREATED) {
				map_idx = _mapping_get_ir_mt_idx_from_wwid
				    (ioc, wwid_table[i]);
				if (map_idx != MPT2SAS_MAPTABLE_BAD_IDX) {
					mt_entry = &ioc->mapping_table[map_idx];
					mt_entry->channel =
					    MPT2SAS_RAID_CHANNEL;
					mt_entry->id = map_idx;
					mt_entry->dev_handle = le16_to_cpu
					    (element->VolDevHandle);
					mt_entry->device_info =
					    MPT2SAS_DEV_RESERVED |
					    MPT2SAS_MAP_IN_USE;
					_mapping_update_ir_missing_cnt(ioc,
					    map_idx, element, wwid_table[i]);
					continue;
				}
				map_idx = _mapping_get_free_ir_mt_idx(ioc);
				if (map_idx == MPT2SAS_MAPTABLE_BAD_IDX)
					log_full_error = 1;
				else if (i < (event_data->NumElements - 1)) {
					log_full_error =
					    _mapping_check_update_ir_mt_idx
					    (ioc, event_data, i, &map_idx,
					     wwid_table);
				}
				if (log_full_error) {
					printk(MPT2SAS_ERR_FMT
					    "%s No space to add the RAID volume"
					    "handle(0x%04x) in map table\n",
					    ioc->name, __func__, le16_to_cpu
					    (element->VolDevHandle));
					continue;
				}
				mt_entry = &ioc->mapping_table[map_idx];
				mt_entry->physical_id = wwid_table[i];
				mt_entry->channel = MPT2SAS_RAID_CHANNEL;
				mt_entry->id = map_idx;
				mt_entry->dev_handle = le16_to_cpu(element->
				    VolDevHandle);
				mt_entry->device_info = MPT2SAS_DEV_RESERVED |
				    MPT2SAS_MAP_IN_USE;
				mt_entry->init_complete = 0;
				_mapping_update_ir_missing_cnt(ioc, map_idx,
				    element, wwid_table[i]);
			} else if (element->ReasonCode ==
			    MPI2_EVENT_IR_CHANGE_RC_REMOVED) {
				map_idx = _mapping_get_ir_mt_idx_from_wwid
				    (ioc, wwid_table[i]);
				if (map_idx == MPT2SAS_MAPTABLE_BAD_IDX) {
					printk(MPT2SAS_ERR_FMT
					    "%s Trying to remove the volume "
					    "which is already removed\n",
					    ioc->name, __func__);
					continue;
				}
				_mapping_update_ir_missing_cnt(ioc, map_idx,
				    element, wwid_table[i]);
			} else if (element->ReasonCode ==
			    MPI2_EVENT_IR_CHANGE_RC_VOLUME_DELETED) {
				map_idx = _mapping_get_mt_idx_from_handle(ioc,
				    le16_to_cpu(element->VolDevHandle));
				if (map_idx == MPT2SAS_MAPTABLE_BAD_IDX) {
					printk(MPT2SAS_ERR_FMT
					    "%s Trying to remove the volume "
					    "handle(0x%04x) which is already "
					    "removed\n", ioc->name, __func__,
					    le16_to_cpu(element->VolDevHandle));
					continue;
				}
				mt_entry = &ioc->mapping_table[map_idx];
				_mapping_update_ir_missing_cnt(ioc, map_idx,
				    element, mt_entry->physical_id);
			}
		}
	}

out:
	_mapping_flush_dpm_pages(ioc);
	kfree(wwid_table);
	if (ioc->pending_map_events)
		ioc->pending_map_events--;
}
