// =============================================================================
// Copyright (c) 2008, Intel Corporation
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without 
// modification, are permitted provided that the following conditions are met:
//
//     * Redistributions of source code must retain the above copyright notice, 
//       this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright 
//       notice, this list of conditions and the following disclaimer in the 
//       documentation and/or other materials provided with the distribution.
//     * Neither the name of Intel Corporation nor the names of its contributors 
//       may be used to endorse or promote products derived from this software 
//       without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
// POSSIBILITY OF SUCH DAMAGE.
// =============================================================================
using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Diagnostics;
using System.IO;
using System.Net;
using System.Net.NetworkInformation;
using System.Net.Security;
using System.Net.Sockets;
using System.Reflection;
using System.Security.Cryptography.X509Certificates;
using System.Text;
using System.Threading;
using System.Xml;
using System.Xml.Serialization;

using Intel.CmdLineArgs;

namespace RemoteControl
{
    #region Enumerations

    public enum ErrorCode
    {
        NoError = 0,
        InvalidParameters = -1,
        InvalidHostname = -2,
        InvalidAmtSystem = -3,
        ExceptionOccurred = -4,
        InvalidCommand = -5,
    }

    public enum PT_STATUS
    {
        PT_STATUS_SUCCESS = 0x0,
        PT_STATUS_INTERNAL_ERROR = 0x1,
        PT_STATUS_INVALID_PT_MODE = 0x3,
        PT_STATUS_INVALID_NAME = 0xC,
        PT_STATUS_INVALID_BYTE_COUNT = 0xF,
        PT_STATUS_NOT_PERMITTED = 0x10,
        PT_STATUS_MAX_LIMIT_REACHED = 0x17,
        PT_STATUS_INVALID_AUTH_TYPE = 0x18,
        PT_STATUS_INVALID_DHCP_MODE = 0x1A,
        PT_STATUS_INVALID_IP_ADDRESS = 0x1B,
        PT_STATUS_INVALID_DOMAIN_NAME = 0x1C,
        PT_STATUS_INVALID_PROVISIONING_STATE = 0x20,
        PT_STATUS_INVALID_TIME = 0x22,
        PT_STATUS_INVALID_INDEX = 0x23,
        PT_STATUS_INVALID_PARAMETER = 0x24,
        PT_STATUS_INVALID_NETMASK = 0x25,
        PT_STATUS_FLASH_WRITE_LIMIT_EXCEEDED = 0x26,
        PT_STATUS_NETWORK_IF_ERROR_BASE = 0x800,
        PT_STATUS_UNSUPPORTED_OEM_NUMBER = 0x801,
        PT_STATUS_UNSUPPORTED_BOOT_OPTION = 0x802,
        PT_STATUS_INVALID_COMMAND = 0x803,
        PT_STATUS_INVALID_SPECIAL_COMMAND = 0x804,
        PT_STATUS_INVALID_HANDLE = 0x805,
        PT_STATUS_INVALIDpassword = 0x806,
        PT_STATUS_INVALID_REALM = 0x807,
        PT_STATUS_STORAGE_ACL_ENTRY_IN_USE = 0x808,
        PT_STATUS_DATA_MISSING = 0x809,
        PT_STATUS_DUPLICATE = 0x80A,
        PT_STATUS_EVENTLOG_FROZEN = 0x80B,
        PT_STATUS_PKI_MISSING_KEYS = 0x80C,
        PT_STATUS_PKI_GENERATING_KEYS = 0x80D,
        PT_STATUS_INVALID_KEY = 0x80E,
        PT_STATUS_INVALID_CERT = 0x80F,
        PT_STATUS_CERT_KEY_NOT_MATCH = 0x810,
        PT_STATUS_MAX_KERB_DOMAIN_REACHED = 0x811,
        PT_STATUS_UNSUPPORTED = 0x812,
        PT_STATUS_INVALID_PRIORITY = 0x813,
        PT_STATUS_NOT_FOUND = 0x814,
        PT_STATUS_INVALID_CREDENTIALS = 0x815,
        PT_STATUS_INVALID_PASSPHRASE = 0x816,
        PT_STATUS_NO_ASSOCIATION = 0x818
    }

    public enum AmtPowerStates
    {
        S0_POWERED = 0,
        S1_SLEEPING_WITH_CONTEXT = 1,
        S2_SLEEPING_WITHOUT_CONTEXT = 2,
        S3_SLEEPING_WITH_MEMORY_CONTEXT_ONLY = 3,
        S4_SUSPENDED_TO_DISK = 4,
        S5_SOFT_OFF = 5,
        S4_OR_S5 = 6,
        MECHANICAL_OFF = 7,
        S1_OR_S2_OR_S3 = 8,
        S1_OR_S2_OR_S3_OR_S4 = 9,
        S5_OVERRIDE = 10,
        LEGACY_ON = 11,
        LEGACY_OFF = 12,
        UNKNOWN13 = 13,
        UNKNOWN14 = 14,
        UNKNOWN15 = 15,
        ERROR_GETTING_STATE = 16,
        UNKNOWN = 254,
        DISCONNECTED = 255,
    }

    #endregion Enumerations

    class Program
    {
        /// <summary>
        /// Entry point for the application.
        /// </summary>
        /// <param name="args">Zero-indexed command line arguments.</param>
        /// <returns>
        /// If successful, returns 0.
        /// If invalid command line options are provided, the return is -1.
        /// If the AMT hostname (IP address or FQDN) is invalid, the return is -2.
        /// If the AMT device is invalid, the return is -3.
        /// If an exception occurred, the return is -4.
        /// If the specified command switch is invalid or missing, the return 
        /// is -5.
        /// All other non-zero return codes map to PT_STATUS codes, specific to 
        /// the AMT command sent. Refer to the Intel(R) AMT SDK documentation 
        /// for more information regarding the possible PT_STATUS values.
        /// </returns>
        /// <remarks>
        /// The valid command line arguments are:
        /// SWITCHES
        ///     -hostname -     The IP address or FQDN of the Intel(R) AMT 
        ///                     system.
        ///     -user -         [OPTIONAL] Username of the Intel(R) AMT account.
        ///     -pass -         [OPTIONAL] Password for the Intel(R) AMT user 
        ///                     account.
        ///                     NOTE: If username and password are not 
        ///                     specified, then Kerberos authentication will be 
        ///                     used.
        ///     -useProxy -     [OPTIONAL] Uses the host OS proxy settings.
        ///     -tls -          [OPTIONAL] Uses TLS when connecting to the 
        ///                     system.
        ///     -cert -         [OPTIONAL] Common Name of the certificate.
        ///     -certFile -     [OPTIONAL] Full path and filename of the 
        ///                     certificate (.cer) file.
        ///     -ignoreCert -   [OPTIONAL] Ignore invalid server certificates.
        ///     -timeout -      [OPTIONAL] Timeout, in milliseconds, for 
        ///                     commands.
        /// COMMANDS
        ///     -powerDown -    Sends the power down command to the system.
        ///     -powerUp -      Sends the power up command to the system.
        ///     -reset -        Sends the reset command to the system.
        ///     -query -        Query for the current power state of the system.
        /// Manageability Presence Server (MPS)
        ///     -httpproxy -    [OPTIONAL] URI of the MPS.
        ///     -httpport -     [OPTIONAL] Port number used to access the MPS.
        /// </remarks>
        static int Main(string[] args)
        {
            ShowLogo();

            //
            // Retrieve the command line arguments.
            //
            CmdLineArgs cmdLineArgs = new CmdLineArgs(args);
            if ((cmdLineArgs["h"] == "true") ||
                (cmdLineArgs["help"] == "true") ||
                (cmdLineArgs["?"] == "true") ||
                (cmdLineArgs.Count == 0))
            {
                // Display the command line help.
                ShowHelp();
                return (int)ErrorCode.NoError;
            }

            string amtHostname = cmdLineArgs["hostname"];
            string username = cmdLineArgs["user"];
            string password = cmdLineArgs["pass"];
            bool useProxy = (cmdLineArgs["useProxy"] == "true") ? true : false;
            bool useTls = (cmdLineArgs["tls"] == "true") ? true : false;
            string certName = cmdLineArgs["cert"];
            string certFilename = cmdLineArgs["certFile"];
            string httpproxy = cmdLineArgs["httpproxy"];
            string httpport = cmdLineArgs["httpport"];

            int cmdTimeout = 30000;
            if (!String.IsNullOrEmpty(cmdLineArgs["timeout"]))
                cmdTimeout = Convert.ToInt32(cmdLineArgs["timeout"]);

            if (String.IsNullOrEmpty(amtHostname))
            {
                Console.WriteLine("The IP address or FQDN of the Intel(R) AMT system must be specified.\n");
                ShowHelp();
                return (int)ErrorCode.InvalidParameters;
            }

            // Determine if the user wants to ignore potentially invalid 
            // certificates.
            if (cmdLineArgs["ignoreCert"] == "true")
                ServicePointManager.ServerCertificateValidationCallback = new RemoteCertificateValidationCallback(IgnoreCertificateErrorHandler);

            // Define the proxy information for the network connection.
            if (!useProxy)
                WebRequest.DefaultWebProxy = null;
            if (!String.IsNullOrEmpty(httpproxy) &&
                !String.IsNullOrEmpty(httpport))
            {
                // Define the proxy information for the MPS.
                try
                {
                    WebRequest.DefaultWebProxy = new WebProxy(httpproxy, Convert.ToInt32(httpport));
                }
                catch (UriFormatException)
                {
                    Console.WriteLine("The given MPS information is not a valid URI. Host = {0}, Port = {1}",
                        httpproxy, httpport);
                    return (int)ErrorCode.InvalidParameters;
                }
            }

            if (String.IsNullOrEmpty(httpproxy) || String.IsNullOrEmpty(httpport))
            {
                //
                // Since this is not a connection attempt through a MPS, verify 
                // the AMT client system hostname and that the system is 
                // reachable.
                // NOTE: These actions may not successfully complete through a 
                // MPS.
                //
                if (!IsHostnameValid(amtHostname))
                {
                    Console.WriteLine("The Intel(R) AMT device, {0}, is unreachable.\n", amtHostname);
                    return (int)ErrorCode.InvalidHostname;
                }
                if (!IsAmtValid(amtHostname, useTls, cmdTimeout))
                {
                    Console.WriteLine("The Intel(R) AMT device, {0}, is invalid.\n", amtHostname);
                    return (int)ErrorCode.InvalidAmtSystem;
                }
            }

            // TODO: Consider using Remoting to a host app for shutting down the OS gracefully.

            // Create and initialize the Remote Control Service interface.
            RemoteControlService rcs = ConfigureRCService(amtHostname, useTls,
                username, password, cmdTimeout, useProxy, certName, certFilename,
                (!String.IsNullOrEmpty(httpproxy) && !String.IsNullOrEmpty(httpport)));

            if (cmdLineArgs["powerDown"] == "true")
            {
                return PowerDownSystem(rcs);
            }
            else if (cmdLineArgs["powerUp"] == "true")
            {
                return PowerUpSystem(rcs);
            }
            else if (cmdLineArgs["reset"] == "true")
            {
                return ResetSystem(rcs);
            }
            else if (cmdLineArgs["query"] == "true")
            {
                try
                {
                    AmtPowerStates curPwrState = GetSystemPowerState(rcs);
                    Console.WriteLine("The current power state is: {0}", curPwrState.ToString());
                    return (int)ErrorCode.NoError;
                }
                catch (AmtException e)
                {
                    Debug.Print("Exception: {0} (type {1})", e.Message, e.GetType());
                    Console.WriteLine("An exception occurred while attempting to get the system power state. {0}", e.Message);
                    return (int)e.StatusCode;
                }
                catch (Exception e)
                {
                    Debug.Print("Exception: {0} (type {1})", e.Message, e.GetType());
                    Console.WriteLine("An exception occurred while attempting to get the system power state. {0}", e.Message);
                    return (int)ErrorCode.ExceptionOccurred;
                }
            }
            else
            {
                Console.WriteLine("Missing or invalid command switch");
                return (int)ErrorCode.InvalidCommand;
            }
        }

        #region Private Methods

        /// <summary>
        /// Displays the logo header for the application.
        /// </summary>
        private static void ShowLogo()
        {
            Console.WriteLine("{0} {1}", AssemblyTitle, AssemblyVersion);
            Console.WriteLine("[{0}]", AssemblyProduct);
            Console.WriteLine("{0}", AssemblyCopyright);
            Console.WriteLine();
        }

        /// <summary>
        /// Displays the command line help information.
        /// </summary>
        private static void ShowHelp()
        {
            Console.WriteLine("Syntax:  \"{0}\" -hostname <FQDN> [-user <username>] [-pass <password>] " +
                "[-useProxy] [-tls] [-cert <name>] [-certFile <filename>] [-timeout <milliseconds>] " +
                "[-powerDown | -powerUp | -reset] [-httpproxy <MPS URI> -httpport <MPS port number>]",
                Path.GetFileName(Assembly.GetExecutingAssembly().Location));
            Console.WriteLine();
            Console.WriteLine("Switches:");
            Console.WriteLine("   -hostname     IP address or FQDN of the Intel(R) AMT system.");
            Console.WriteLine("   -user         Login name for the Intel(R) AMT system user account.");
            Console.WriteLine("   -pass         Password for the Intel(R) AMT system user account.");
            Console.WriteLine("                 NOTE: If neither the username or password is specified, ");
            Console.WriteLine("                 then Kerberos authentication will be used.");
            Console.WriteLine("   -useProxy     [OPTIONAL] Uses the operating system proxy settings.");
            Console.WriteLine("                 The default behavior is to bypass the system proxy settings.");
            Console.WriteLine("   -tls          [OPTIONAL] Uses TLS when connecting to the system.");
            Console.WriteLine("   -cert         [OPTIONAL] Common Name of the certificate to use.");
            Console.WriteLine("   -certFile     [OPTIONAL] Full path and filename of the certificate.");
            Console.WriteLine("   -ignoreCert   [OPTIONAL] Ignore invalid server certificates.");
            Console.WriteLine("   -timeout      [OPTIONAL] Timeout, in milliseconds, for commands.");
            Console.WriteLine("Commands: (only one command must be used)");
            Console.WriteLine("   -powerDown    Sends the power down command to the system.");
            Console.WriteLine("   -powerUp      Sends the power up command to the system.");
            Console.WriteLine("   -reset        Sends the reset command to the system.");
            Console.WriteLine("   -query        Query for the current power state of the system.");
            Console.WriteLine("Manageability Presence Server (MPS):");
            Console.WriteLine("   -httpproxy -  [OPTIONAL] URI of the MPS.");
            Console.WriteLine("   -httpport -   [OPTIONAL] Port number used to access the MPS.");
            Console.WriteLine("Examples:");
            Console.WriteLine("   RemoteControl.exe -hostname vPro1 -user admin -pass P@ssw0rd -powerDown");
            Console.WriteLine("   RemoteControl.exe -hostname vPro1 -user admin -pass P@ssw0rd -tls -powerDown");
            Console.WriteLine("   RemoteControl.exe -hostname vPro1 -user admin -pass P@ssw0rd -query");
            Console.WriteLine("   RemoteControl.exe -hostname vPro1 -user admin -pass P@ssw0rd -query -httpproxy \"192.168.1.1\" -httpport 8080");
            Console.WriteLine();
        }

        /// <summary>
        /// Simply "approves" the remote Secure Sockets Layer (SSL) certificate 
        /// used for authentication by always returning true.
        /// </summary>
        /// <param name="sender">An object that contains state information for this validation.</param>
        /// <param name="certificate">The certificate used to authenticate the remote party.</param>
        /// <param name="chain">The chain of certificate authorities associated with the remote certificate.</param>
        /// <param name="sslPolicyErrors">One or more errors associated with the remote certificate.</param>
        /// <returns>Always returns true to allow the use of the certificate.</returns>
        private static bool IgnoreCertificateErrorHandler(Object sender,
            X509Certificate certificate, X509Chain chain, SslPolicyErrors sslPolicyErrors)
        {
            return true;
        }

        /// <summary>
        /// Determines if the defined AMT hostname is valid 
        /// (responds to pings).
        /// </summary>
        /// <param name="hostname">The IP address or FQDN of the target system.</param>
        /// <returns>Returns true if the address is valid. Otherwise, 
        /// the return is false.</returns>
        private static bool IsHostnameValid(string hostname)
        {
            try
            {
                for (int iteration = 0; iteration < 3; iteration++)
                {
                    Ping pingSender = new Ping();
                    PingReply reply = pingSender.Send(hostname);
                    if (reply.Status == IPStatus.Success)
                        return true;
                    //else
                    //    return false;
                } // END for (int iteration = 0; ...)

                return false;
            }
            catch (Exception e)
            {
                Console.WriteLine("ERROR: Unable to ping the Intel(R) AMT device. {0}", e.Message);
                Debug.Print("Exception: {0} (type {1})",
                    e.Message, e.GetType().ToString());
                return false;
            }
        }

        /// <summary>
        /// Determines if the AMT device is reachable. If the AMT device is not 
        /// accessible, it may have been unprovisioned.
        /// </summary>
        /// <param name="amtHostname">The IP address or FQDN of the target AMT system.</param>
        /// <param name="useTls">Flag indicating the use of TLS.</param>
        /// <param name="cmdTimeout">Time, in milliseconds, to use for command timeouts.</param>
        /// <returns>Returns true if the AMT device is available and 
        /// responsive. Otherwise, the return is false.</returns>
        private static bool IsAmtValid(string amtHostname, bool useTls, int cmdTimeout)
        {
            try
            {
                // Determine if the AMT device is istening and responsive.
                Socket s = new Socket(AddressFamily.InterNetwork, SocketType.Stream, ProtocolType.Tcp);

                if (useTls)
                {
                    // Connect using the Enterprise port.
                    s.Connect(amtHostname, 16993);
                }
                else
                {
                    // Connect using the SMB port.
                    s.Connect(amtHostname, 16992);
                }
                s.Close();
            }
            catch (Exception e)
            {
                Console.WriteLine("ERROR: Unable to connect with the AMT device. {0}", e.Message);
                Debug.Print("Exception: {0} (type {1})",
                    e.Message, e.GetType().ToString());
                return false;
            }

            if (!useTls)
            {
                //
                // Determine if the device is properly configured for SMB.
                //
                try
                {
                    //
                    // Determine if the AMT device is present at the given port.
                    // This is ONLY concerned with SMB mode.
                    // (The following code derived from the Intel(R) AMT Commander 
                    // included in the Intel(R) AMT DTK).
                    //
                    // (sample from AMT Commander - Intel AMT Stack: AmtDiscovery.cs - LaunchOneSocket (line 226)
                    //
                    string str = null;
                    TcpClient client = new TcpClient(amtHostname, 16992);
                    NetworkStream ns = client.GetStream();
                    ns.ReadTimeout = cmdTimeout;
                    ns.WriteTimeout = cmdTimeout;
                    DateTime failTime = DateTime.Now.AddSeconds(5);
                    byte[] b = UTF8Encoding.UTF8.GetBytes("HEAD / HTTP/1.1\r\nHost: RemoteControl\r\n\r\n");
                    ns.Write(b, 0, b.Length);
                    byte[] buf = new byte[2000];
                    int ptr = 0;
                    string head = "";
                    try
                    {
                        while (DateTime.Now.CompareTo(failTime) < 0 && head.IndexOf("\r\n\r\n") < 0)
                        {
                            Thread.Sleep(100);
                            ptr = ns.Read(buf, ptr, 2000 - ptr);
                            head = UTF8Encoding.UTF8.GetString(buf, 0, ptr);
                        }
                    }
                    catch (Exception e)
                    {
                        Debug.Print("Exception {0} (type {1})", e.Message, e.GetType().ToString());
                        return false; // Assuming an error with the AMT device.
                    }
                    finally
                    {
                        ns.Close();
                        client.Close();
                        client = null;
                    }

                    if (head.IndexOf("\r\n\r\n") > 0)
                    {
                        int i = head.IndexOf("\r\nServer: ");
                        if (i >= 0)
                        {
                            str = head.Substring(i + 10);
                            i = str.IndexOf("\r\n");
                            str = str.Substring(0, i);
                        }
                        if (String.IsNullOrEmpty(str))
                        {
                            return false;
                        }
                        else if (str.StartsWith("Intel(R)") == true)
                        {
                            return true;
                        }
                        else
                        {
                            // At this point, we're going to assume that the 
                            // target system is AMT enabled. So, why keep this 
                            // code to check the SMB HTTP headers that are 
                            // returned from it? Nostalgia? Basically, a sanity 
                            // check to ensure that the system listening at 
                            // this address/port is geniunely AMT.
                            Debug.Print("Returned AMT server header: {0}", str);
                            return true;
                        }
                    }
                    else
                    {
                        return false;
                    }
                }
                catch (Exception e)
                {
                    Console.WriteLine("ERROR: Unable to connect with the AMT device. {0}", e.Message);
                    Debug.Print("Exception: {0} (type {1})",
                        e.Message, e.GetType().ToString());
                    return false;
                }
            } // END if (!useTls)

            return true;
        }

        /// <summary>
        /// Creates and configures an AMT RemoteControlService object.
        /// </summary>
        /// <param name="amtHostname">The IP address or FQDN of the target AMT system.</param>
        /// <param name="useTls">Flag to indicate the use of TLS.</param>
        /// <param name="username">Username for an AMT account that has privledges for accessing the RemoteControlService interface.</param>
        /// <param name="password">Password for the AMT account specified by the username parameter.</param>
        /// <param name="cmdTimeout">Time, in milliseconds, to have AMT commands timeout.</param>
        /// <param name="useProxy">Flag to indicate the use of the host OS proxy settings.</param>
        /// <param name="certName">The Common Name of the certificate to use.</param>
        /// <param name="certFilename">Full path and filename of the certificate to use.</param>
        /// <param name="useMPS">Flag to indicate the use of a MPS (proxy server).</param>
        /// <returns>A valid and configured AMT RemoteControlService object.</returns>
        private static RemoteControlService ConfigureRCService(string amtHostname, bool useTls, string username,
            string password, int cmdTimeout, bool useProxy, string certName, string certFilename, bool useMPS)
        {
            Debug.Assert(!String.IsNullOrEmpty(amtHostname));

            //
            // Create and initialize the Remote Control Service interface.
            //
            RemoteControlService rcs = new RemoteControlService();
            rcs.PreAuthenticate = true;
            rcs.ConnectionGroupName = amtHostname; 

            // Define the URL of the AMT device's Remote Control service.
            if (useTls)
                rcs.Url = String.Format("https://{0}:16993/RemoteControlService", amtHostname);
            else
                rcs.Url = String.Format("http://{0}:16992/RemoteControlService", amtHostname);

            //
            // Define the credentials to use when connecting to the AMT device.
            //
            if (String.IsNullOrEmpty(username) && String.IsNullOrEmpty(password))
            {
                // Attempt to connect and login using Kerberos with the local credentials.
                //
                // Create a new instance of CredentialCache.
                CredentialCache credentialCache = new CredentialCache();

                // Add the NetworkCredential to the CredentialCache.
                credentialCache.Add(new Uri(rcs.Url), "Negotiate", CredentialCache.DefaultNetworkCredentials);

                // Add the CredentialCache to the proxy class credentials.
                rcs.Credentials = credentialCache;

                // Setup the authentication manager.
                if (AuthenticationManager.CustomTargetNameDictionary.ContainsKey(rcs.Url) == false)
                {
                    string SPN = "HTTP/" + amtHostname + ":16993";
                    if (rcs.Url.StartsWith("https", StringComparison.CurrentCultureIgnoreCase) == true)
                        SPN = "HTTP/" + amtHostname + ":16992";
                    AuthenticationManager.CustomTargetNameDictionary.Add(rcs.Url, SPN);
                }
            }
            else if (username.Contains("\\\\"))
            {
                // Attempt to connect and login using Kerberos.
                int splitIndex = username.IndexOf('\\');
                string domain = username.Substring(0, splitIndex);
                username = username.Substring(splitIndex + 1);

                // Create a new instance of CredentialCache.
                CredentialCache credentialCache = new CredentialCache();

                // Create a new instance of NetworkCredential using the client
                // credentials.
                NetworkCredential credentials = new NetworkCredential(username, password, domain);

                // Add the NetworkCredential to the CredentialCache.
                credentialCache.Add(new Uri(rcs.Url), "Negotiate", credentials);

                // Add the CredentialCache to the proxy class credentials.
                rcs.Credentials = credentialCache;

                // Setup the authentication manager.
                if (AuthenticationManager.CustomTargetNameDictionary.ContainsKey(rcs.Url) == false)
                {
                    string SPN = "HTTP/" + amtHostname + ":16993";
                    if (rcs.Url.StartsWith("https", StringComparison.CurrentCultureIgnoreCase) == true)
                        SPN = "HTTP/" + amtHostname + ":16992";
                    AuthenticationManager.CustomTargetNameDictionary.Add(rcs.Url, SPN);
                }
            }
            else
            {
                // Attempt to connect and login using Digest Authentication.
                // Create a new instance of CredentialCache.
                CredentialCache credentialCache = new CredentialCache();

                // Create a new instance of NetworkCredential using the client
                // credentials.
                NetworkCredential credentials = new NetworkCredential(username, password);

                // Add the NetworkCredential to the CredentialCache.
                credentialCache.Add(new Uri(rcs.Url), "Digest", credentials);

                // Add the CredentialCache to the proxy class credentials.
                rcs.Credentials = credentialCache;
            }

            rcs.Timeout = cmdTimeout;

            if (!useProxy)
                rcs.Proxy = null;
            else if (useMPS)
                rcs.Proxy = WebRequest.DefaultWebProxy;

            if (useTls)
            {
                //
                // Retrieve the certificates for TLS communications.
                //
                try
                {
                    if (!String.IsNullOrEmpty(certFilename))
                    {
                        rcs.ClientCertificates.Add(X509Certificate2.CreateFromCertFile(certFilename));
                    }
                    else if (!String.IsNullOrEmpty(certName))
                    {
                        //
                        // Search the CurrentUser store for matching certificates.
                        //
                        X509Store certStore = new X509Store(StoreLocation.CurrentUser);
                        certStore.Open(OpenFlags.ReadOnly | OpenFlags.OpenExistingOnly);
                        foreach (X509Certificate2 cert in certStore.Certificates)
                        {
                            StringCollection commonNames = new StringCollection();
                            string certSubject = cert.Subject;
                            int cnIndex = certSubject.IndexOf("CN=");
                            while (cnIndex >= 0)
                            {
                                certSubject = certSubject.Substring(cnIndex + 3);
                                cnIndex = certSubject.IndexOf(",");
                                if (cnIndex > 0)
                                    commonNames.Add(certSubject.Substring(0, cnIndex));
                                else
                                    commonNames.Add(certSubject);
                                cnIndex = certSubject.IndexOf("CN=");
                            }

                            foreach (string commonName in commonNames)
                            {
                                if (String.Compare(certName, commonName, true) == 0)
                                {
                                    rcs.ClientCertificates.Add(cert);
                                    break;
                                }
                            } // END foreach (string commonName in CommonNames)
                        } // END foreach (X509Certificate2 cert in certStore.Certificates)

                        //
                        // Search the LocalMachine store for matching certificates.
                        //
                        certStore = new X509Store(StoreLocation.LocalMachine);
                        certStore.Open(OpenFlags.ReadOnly | OpenFlags.OpenExistingOnly);
                        foreach (X509Certificate2 cert in certStore.Certificates)
                        {
                            StringCollection commonNames = new StringCollection();
                            string certSubject = cert.Subject;
                            int cnIndex = certSubject.IndexOf("CN=");
                            while (cnIndex >= 0)
                            {
                                certSubject = certSubject.Substring(cnIndex + 3);
                                cnIndex = certSubject.IndexOf(",");
                                if (cnIndex > 0)
                                    commonNames.Add(certSubject.Substring(0, cnIndex));
                                else
                                    commonNames.Add(certSubject);
                                cnIndex = certSubject.IndexOf("CN=");
                            }

                            foreach (string commonName in commonNames)
                            {
                                if (String.Compare(certName, commonName, true) == 0)
                                {
                                    rcs.ClientCertificates.Add(cert);
                                    break;
                                }
                            } // END foreach (string commonName in CommonNames)
                        } // END foreach (X509Certificate2 cert in certStore.Certificates)
                    }
                }
                catch (Exception e)
                {
                    Debug.Print("Exception: {0} (type {1})", e.Message, e.GetType());
                    Console.WriteLine("An exception occurred while retrieving the certificate information. {0}", e.Message);
                }
            } // END if (useTls)

            return rcs;
        }

        /// <summary>
        /// Resets the target AMT system.
        /// </summary>
        /// <param name="rcs">A properly configured RemoteControlService object for the system.</param>
        /// <returns>If the return is zero, then the operation was successful. 
        /// If the return value is greater than zero, then use PT_STATUS for 
        /// the appropriate error code mapping. If the return value is less 
        /// than zero, then use ErrorCode for the appropriate error code mapping.</returns>
        private static int ResetSystem(RemoteControlService rcs)
        {
            Debug.Assert(rcs != null);

            try
            {
                uint IanaOemNumber;
                if (SystemCanReset(rcs, out IanaOemNumber))
                {
                    PT_STATUS status = (PT_STATUS)rcs.RemoteControl(0x10, IanaOemNumber, 0x0, false, 0x0, false, 0x0, false, 0x0, false);
                    if (status == PT_STATUS.PT_STATUS_UNSUPPORTED_OEM_NUMBER)
                    {
                        // HACK: Some systems appear to return an invalid 
                        // IanaOemNumber, preventing the RemoteControl command 
                        // from successfully executing.
                        // This will force the ASF IANA number, 4542 (0x11BE).
                        status = (PT_STATUS)rcs.RemoteControl(0x10, 0x11BE, 0x0, false, 0x0, false, 0x0, false, 0x0, false);
                    }
                    if (status == PT_STATUS.PT_STATUS_UNSUPPORTED_OEM_NUMBER)
                    {
                        // HACK: Some systems appear to return an invalid 
                        // IanaOemNumber, preventing the RemoteControl command 
                        // from successfully executing.
                        // This will force the Intel IANA number, 343 (0x157).
                        status = (PT_STATUS)rcs.RemoteControl(0x10, 0x157, 0x0, false, 0x0, false, 0x0, false, 0x0, false);
                    }
                    if (status != PT_STATUS.PT_STATUS_SUCCESS)
                    {
                        Debug.Print("Unable to reset the system. STATUS: {0}", status);
                        Console.WriteLine("Unable to reset the system. STATUS: {0}", status);
                        return (int)status;
                    }
                }
                else
                {
                    Debug.Print("System reset is not supported. STATUS: {0}", PT_STATUS.PT_STATUS_UNSUPPORTED);
                    Console.WriteLine("System reset is not supported. STATUS: {0}", PT_STATUS.PT_STATUS_UNSUPPORTED);
                    return (int)PT_STATUS.PT_STATUS_UNSUPPORTED;
                }
            }
            catch (AmtException e)
            {
                Debug.Print("Exception: {0} (type {1})", e.Message, e.GetType());
                Console.WriteLine("An exception occurred while attempting to reset the system. {0}", e.Message);
                return (int)e.StatusCode;
            }
            catch (Exception e)
            {
                Debug.Print("Exception: {0} (type {1})", e.Message, e.GetType());
                Console.WriteLine("An exception occurred while attempting to reset the system. {0}", e.Message);
                return (int)ErrorCode.ExceptionOccurred;
            }

            return (int)ErrorCode.NoError;
        }

        /// <summary>
        /// Powers up the target AMT system.
        /// </summary>
        /// <param name="rcs">A properly configured RemoteControlService object for the system.</param>
        /// <returns>If the return is zero, then the operation was successful. 
        /// If the return value is greater than zero, then use PT_STATUS for 
        /// the appropriate error code mapping. If the return value is less 
        /// than zero, then use ErrorCode for the appropriate error code mapping.</returns>
        private static int PowerUpSystem(RemoteControlService rcs)
        {
            Debug.Assert(rcs != null);

            try
            {
                uint IanaOemNumber;
                if (SystemCanPowerUp(rcs, out IanaOemNumber))
                {
                    PT_STATUS status = (PT_STATUS)rcs.RemoteControl(0x11, IanaOemNumber, 0x0, false, 0x0, false, 0x0, false, 0x0, false);
                    if (status == PT_STATUS.PT_STATUS_UNSUPPORTED_OEM_NUMBER)
                    {
                        // HACK: Some systems appear to return an invalid 
                        // IanaOemNumber, preventing the RemoteControl command 
                        // from successfully executing.
                        // This will force the ASF IANA number, 4542 (0x11BE).
                        status = (PT_STATUS)rcs.RemoteControl(0x11, 0x11BE, 0x0, false, 0x0, false, 0x0, false, 0x0, false);
                    }
                    if (status == PT_STATUS.PT_STATUS_UNSUPPORTED_OEM_NUMBER)
                    {
                        // HACK: Some systems appear to return an invalid 
                        // IanaOemNumber, preventing the RemoteControl command 
                        // from successfully executing.
                        // This will force the Intel IANA number, 343 (0x157).
                        status = (PT_STATUS)rcs.RemoteControl(0x11, 0x157, 0x0, false, 0x0, false, 0x0, false, 0x0, false);
                    }
                    if (status != PT_STATUS.PT_STATUS_SUCCESS)
                    {
                        Debug.Print("Unable to power up the system. STATUS: {0}", status);
                        Console.WriteLine("Unable to power up the system. STATUS: {0}", status);
                        return (int)status;
                    }
                }
                else
                {
                    Debug.Print("System power up is not supported. STATUS: {0}", PT_STATUS.PT_STATUS_UNSUPPORTED);
                    Console.WriteLine("System power up is not supported. STATUS: {0}", PT_STATUS.PT_STATUS_UNSUPPORTED);
                    return (int)PT_STATUS.PT_STATUS_UNSUPPORTED;
                }
            }
            catch (AmtException e)
            {
                Debug.Print("Exception: {0} (type {1})", e.Message, e.GetType());
                Console.WriteLine("An exception occurred while attempting to reset the system. {0}", e.Message);
                return (int)e.StatusCode;
            }
            catch (Exception e)
            {
                Debug.Print("Exception: {0} (type {1})", e.Message, e.GetType());
                Console.WriteLine("An exception occurred while attempting to power up the system. {0}", e.Message);
                return (int)ErrorCode.ExceptionOccurred;
            }

            return (int)ErrorCode.NoError;
        }

        /// <summary>
        /// Powers down the target AMT system.
        /// </summary>
        /// <param name="rcs">A properly configured RemoteControlService object for the system.</param>
        /// <returns>If the return is zero, then the operation was successful. 
        /// If the return value is greater than zero, then use PT_STATUS for 
        /// the appropriate error code mapping. If the return value is less 
        /// than zero, then use ErrorCode for the appropriate error code mapping.</returns>
        private static int PowerDownSystem(RemoteControlService rcs)
        {
            Debug.Assert(rcs != null);

            try
            {
                uint IanaOemNumber;
                if (SystemCanPowerDown(rcs, out IanaOemNumber))
                {
                    PT_STATUS status = (PT_STATUS)rcs.RemoteControl(0x12, IanaOemNumber, 0x0, false, 0x0, false, 0x0, false, 0x0, false);
                    if (status == PT_STATUS.PT_STATUS_UNSUPPORTED_OEM_NUMBER)
                    {
                        // HACK: Some systems appear to return an invalid 
                        // IanaOemNumber, preventing the RemoteControl command 
                        // from successfully executing.
                        // This will force the ASF IANA number, 4542 (0x11BE).
                        status = (PT_STATUS)rcs.RemoteControl(0x12, 0x11BE, 0x0, false, 0x0, false, 0x0, false, 0x0, false);
                    }
                    if (status == PT_STATUS.PT_STATUS_UNSUPPORTED_OEM_NUMBER)
                    {
                        // HACK: Some systems appear to return an invalid 
                        // IanaOemNumber, preventing the RemoteControl command 
                        // from successfully executing.
                        // This will force the Intel IANA number, 343 (0x157).
                        status = (PT_STATUS)rcs.RemoteControl(0x12, 0x157, 0x0, false, 0x0, false, 0x0, false, 0x0, false);
                    }
                    if (status != PT_STATUS.PT_STATUS_SUCCESS)
                    {
                        Debug.Print("Unable to power down the system. STATUS: {0}", status);
                        Console.WriteLine("Unable to power down the system. STATUS: {0}", status); 
                        return (int)status;
                    }
                }
                else
                {
                    Debug.Print("System power down is not supported. STATUS: {0}", PT_STATUS.PT_STATUS_UNSUPPORTED);
                    Console.WriteLine("System power down is not supported. STATUS: {0}", PT_STATUS.PT_STATUS_UNSUPPORTED);
                    return (int)PT_STATUS.PT_STATUS_UNSUPPORTED;
                }
            }
            catch (AmtException e)
            {
                Debug.Print("Exception: {0} (type {1})", e.Message, e.GetType());
                Console.WriteLine("An exception occurred while attempting to reset the system. {0}", e.Message);
                return (int)e.StatusCode;
            }
            catch (Exception e)
            {
                Debug.Print("Exception: {0} (type {1})", e.Message, e.GetType());
                Console.WriteLine("An exception occurred while attempting to power down the system. {0}", e.Message);
                return (int)ErrorCode.ExceptionOccurred;
            }

            return (int)ErrorCode.NoError;
        }

        /// <summary>
        /// Retrieve the remote power control capabilities of the AMT system.
        /// </summary>
        /// <param name="rcs">A properly configured RemoteControlService object for the system.</param>
        /// <param name="IanaOemNumber">The IANA-assigned Enterprise Number.</param>
        /// <returns>Returns the raw system capabilities flags that indicate the supported power operations.</returns>
        private static byte GetSystemCapabilities(RemoteControlService rcs, out uint IanaOemNumber)
        {
            Debug.Assert(rcs != null);

            PT_STATUS status;
            uint OemDefinedCapabilities;
            ushort SpecialCommandsSupported;
            byte SystemCapabilitiesSupported;
            uint SystemFirmwareCapabilities;
            status = (PT_STATUS)rcs.GetRemoteControlCapabilities(out IanaOemNumber, out OemDefinedCapabilities,
                out SpecialCommandsSupported, out SystemCapabilitiesSupported, out SystemFirmwareCapabilities);
            if (status != PT_STATUS.PT_STATUS_SUCCESS)
            {
                Debug.Print("Unable to get the system capabilities. STATUS: {0}", status);
                throw new AmtException("Unable to get the system capabilities", status);
            }

            return SystemCapabilitiesSupported;
        }

        /// <summary>
        /// Retrieves the current power state of the AMT system.
        /// </summary>
        /// <param name="rcs">A properly configured RemoteControlService object for the system.</param>
        /// <returns>The current power state of the AMT system.</returns>
        private static AmtPowerStates GetSystemPowerState(RemoteControlService rcs)
        {
            Debug.Assert(rcs != null);

            PT_STATUS status;
            uint systemPowerState;
            status = (PT_STATUS)rcs.GetSystemPowerState(out systemPowerState);
            if (status != PT_STATUS.PT_STATUS_SUCCESS)
            {
                Debug.Print("Unable to get the system power state. STATUS: {0}", status);
                throw new AmtException("Unable to get the system power state.", status);
            }

            AmtPowerStates powerState = (AmtPowerStates)(systemPowerState & (uint)0x000F);
            return powerState;
        }

        /// <summary>
        /// Determines if the AMT system can be reset based on its supported 
        /// capabilities and current power state.
        /// </summary>
        /// <param name="rcs">A properly configured RemoteControlService object for the system.</param>
        /// <param name="IanaOemNumber">The IANA-assigned Enterprise Number.</param>
        /// <returns>Returns true if the AMT system can be reset. 
        /// Otherwise, the return is false.</returns>
        private static bool SystemCanReset(RemoteControlService rcs, out uint IanaOemNumber)
        {
            Debug.Assert(rcs != null);
            byte SystemCapabilities = GetSystemCapabilities(rcs, out IanaOemNumber);
            bool supportsReset = Convert.ToBoolean(((SystemCapabilities & (uint)0x08) >> 3));

            //
            // NOTE: No need to determine if the system is already on.
            //
            //AmtPowerStates curPowerState = GetSystemPowerState(rcs);

            //bool systemCanReset = ((curPowerState == AmtPowerStates.S0_POWERED) ||
            //                       (curPowerState == AmtPowerStates.LEGACY_ON)) &&
            //                      supportsReset;

            //return systemCanReset;

            return supportsReset;
        }

        /// <summary>
        /// Determines if the AMT system can be powered up based on its 
        /// supported capabilities and current power state.
        /// </summary>
        /// <param name="rcs">A properly configured RemoteControlService object for the system.</param>
        /// <param name="IanaOemNumber">The IANA-assigned Enterprise Number.</param>
        /// <returns>Returns true if the AMT system can be powered up. 
        /// Otherwise, the return is false.</returns>
        private static bool SystemCanPowerUp(RemoteControlService rcs, out uint IanaOemNumber)
        {
            Debug.Assert(rcs != null);
            byte SystemCapabilities = GetSystemCapabilities(rcs, out IanaOemNumber);
            bool supportsPowerUp = Convert.ToBoolean(((SystemCapabilities & (uint)0x04) >> 2));

            //
            // NOTE: No need to determine if the system is already off.
            //
            //AmtPowerStates curPowerState = GetSystemPowerState(rcs);

            //bool systemCanPowerUp = ((curPowerState == AmtPowerStates.S5_SOFT_OFF) ||
            //                         (curPowerState == AmtPowerStates.S4_OR_S5) ||
            //                         (curPowerState == AmtPowerStates.MECHANICAL_OFF) ||
            //                         (curPowerState == AmtPowerStates.S5_OVERRIDE) ||
            //                         (curPowerState == AmtPowerStates.LEGACY_OFF)) &&
            //                        supportsPowerUp;

            //return systemCanPowerUp;

            return supportsPowerUp;
        }

        /// <summary>
        /// Determines if the AMT system can be powered down based on its 
        /// supported capabilities and current power state.
        /// </summary>
        /// <param name="rcs">A properly configured RemoteControlService object for the system.</param>
        /// <param name="IanaOemNumber">The IANA-assigned Enterprise Number.</param>
        /// <returns>Returns true if the AMT system can be powered down. 
        /// Otherwise, the return is false.</returns>
        private static bool SystemCanPowerDown(RemoteControlService rcs, out uint IanaOemNumber)
        {
            Debug.Assert(rcs != null);
            byte SystemCapabilities = GetSystemCapabilities(rcs, out IanaOemNumber);
            bool supportsPowerDown = Convert.ToBoolean(((SystemCapabilities & (uint)0x02) >> 1));

            //
            // NOTE: No need to determine if the system is already on.
            //
            //AmtPowerStates curPowerState = GetSystemPowerState(rcs);

            //bool systemCanPowerDown = ((curPowerState == AmtPowerStates.S0_POWERED) ||
            //                           (curPowerState == AmtPowerStates.LEGACY_ON)) &&
            //                          supportsPowerDown;

            //return systemCanPowerDown;

            return supportsPowerDown;
        }

        #region Assembly Attribute Accessors

        /// <summary>
        /// Retrieve the title of the assembly.
        /// </summary>
        private static string AssemblyTitle
        {
            get
            {
                object[] attributes = Assembly.GetExecutingAssembly().GetCustomAttributes(typeof(AssemblyTitleAttribute), false);
                if (attributes.Length > 0)
                {
                    AssemblyTitleAttribute titleAttribute = (AssemblyTitleAttribute)attributes[0];
                    if (titleAttribute.Title != "")
                    {
                        return titleAttribute.Title;
                    }
                }
                return System.IO.Path.GetFileNameWithoutExtension(Assembly.GetExecutingAssembly().CodeBase);
            }
        }

        /// <summary>
        /// Retrieve the version of the assembly.
        /// </summary>
        private static string AssemblyVersion
        {
            get
            {
                return Assembly.GetExecutingAssembly().GetName().Version.ToString();
            }
        }

        /// <summary>
        /// Retrieve the description of the assembly.
        /// </summary>
        private static string AssemblyDescription
        {
            get
            {
                object[] attributes = Assembly.GetExecutingAssembly().GetCustomAttributes(typeof(AssemblyDescriptionAttribute), false);
                if (attributes.Length == 0)
                {
                    return "";
                }
                return ((AssemblyDescriptionAttribute)attributes[0]).Description;
            }
        }

        /// <summary>
        /// Retrieve the product name of the assembly.
        /// </summary>
        private static string AssemblyProduct
        {
            get
            {
                object[] attributes = Assembly.GetExecutingAssembly().GetCustomAttributes(typeof(AssemblyProductAttribute), false);
                if (attributes.Length == 0)
                {
                    return "";
                }
                return ((AssemblyProductAttribute)attributes[0]).Product;
            }
        }

        /// <summary>
        /// Retrieve the copyright text of the assembly.
        /// </summary>
        private static string AssemblyCopyright
        {
            get
            {
                object[] attributes = Assembly.GetExecutingAssembly().GetCustomAttributes(typeof(AssemblyCopyrightAttribute), false);
                if (attributes.Length == 0)
                {
                    return "";
                }
                return ((AssemblyCopyrightAttribute)attributes[0]).Copyright;
            }
        }

        /// <summary>
        /// Retrieve the company name of the assembly.
        /// </summary>
        private static string AssemblyCompany
        {
            get
            {
                object[] attributes = Assembly.GetExecutingAssembly().GetCustomAttributes(typeof(AssemblyCompanyAttribute), false);
                if (attributes.Length == 0)
                {
                    return "";
                }
                return ((AssemblyCompanyAttribute)attributes[0]).Company;
            }
        }

        #endregion Assembly Attribute Accessors

        #endregion
    }

    /// <summary>
    /// Custom exception class for AMT errors.
    /// </summary>
    public class AmtException : Exception
    {
        /// <summary>
        /// The PT_STATUS code for the exception.
        /// </summary>
        private PT_STATUS _statusCode;

        /// <summary>
        /// Default constructor.
        /// </summary>
        public AmtException() : base()
        {
        }

        /// <summary>
        /// Constructs a new AmtException object with the given message.
        /// </summary>
        /// <param name="message">The message to use for the exception.</param>
        public AmtException(string message) : base(message)
        {
        }

        /// <summary>
        /// Constructs a new AmtException object with the given message and status code.
        /// </summary>
        /// <param name="message">The message to use for the exception.</param>
        /// <param name="statusCode">The status code to use for the exception.</param>
        public AmtException(string message, PT_STATUS statusCode) : base(message)
        {
            this.StatusCode = statusCode;
        }

        /// <summary>
        /// Gets or sets the PT_STATUS code for the exception.
        /// </summary>
        public PT_STATUS StatusCode
        {
            get { return this._statusCode; }
            set { this._statusCode = value; }
        }
    }
}
