// =============================================================================
// Copyright (c) 2008, Intel Corporation
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without 
// modification, are permitted provided that the following conditions are met:
//
//     * Redistributions of source code must retain the above copyright notice, 
//       this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright 
//       notice, this list of conditions and the following disclaimer in the 
//       documentation and/or other materials provided with the distribution.
//     * Neither the name of Intel Corporation nor the names of its contributors 
//       may be used to endorse or promote products derived from this software 
//       without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
// POSSIBILITY OF SUCH DAMAGE.
// =============================================================================
using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Text;
using System.Text.RegularExpressions;

namespace Intel.CmdLineArgs
{
    /// <summary>
    /// Parses command line arguments into parameter and value pairs. 
    /// Parameters can start with -, -- or /. and all the values linked. 
    /// Values can be separated from a parameter with a space, or a =.
    /// </summary>
    /// <remarks>
    /// Valid argument forms:
    ///    {-,/,--}param{ ,=}((",')value(",'))
    /// Examples: 
    ///    -param1 value1 --param2 /param3="Test-:-work" 
    ///    /param4=happy -param5 '--=nice=--'
    /// </remarks>
    class CmdLineArgs
    {
        private StringDictionary _params;

        public CmdLineArgs(string[] args)
        {
            _params = new StringDictionary();
            // Split the argument into its parameter and value.
            Regex splitter = new Regex(@"^-{1,2}|^/|=",
                RegexOptions.Compiled | RegexOptions.IgnoreCase);
            // Remove all starting and trailing ' or " characters from a value.
            Regex remover = new Regex(@"^['""]?(.*?)['""]?$", 
                RegexOptions.Compiled | RegexOptions.IgnoreCase);
            string param = null;
            string[] parts;

            foreach (string arg in args)
            {
                parts = splitter.Split(arg, 3);
                switch (parts.Length)
                {
                    case 1:
                        // Value for last parameter.
                        if (param != null)
                        {
                            if (!_params.ContainsKey(param))
                            {
                                parts[0] = remover.Replace(parts[0], "$1");
                                _params.Add(param, parts[0]);
                            }

                            param = null;
                        }
                        break;
                    case 2:
                        // Parameter name
                        if (param != null)
                        {
                            // Value-less parameter. Value will be "true".
                            if (!_params.ContainsKey(param))
                                _params.Add(param, "true");
                        }

                        param = parts[1];
                        break;
                    case 3:
                        // Parameter-value pair.
                        if (param != null)
                        {
                            // Value-less parameter. Value will be "true".
                            if (!_params.ContainsKey(param))
                                _params.Add(param, "true");
                        }

                        param = parts[1];

                        if (!_params.ContainsKey(param))
                        {
                            // Remove quotes from value.
                            parts[2] = remover.Replace(parts[2], "$1");
                            _params.Add(param, parts[2]);
                        }

                        param = null;
                        break;
                } // END switch (parts.Length)
            } // END foreach (string arg in args)

            if (param != null)
            {
                // Value-less parameter. Value will be "true".
                if (!_params.ContainsKey(param))
                    _params.Add(param, "true");
            }
        }

        /// <summary>
        /// Retrieve the specified parameter's value.
        /// </summary>
        /// <param name="param">The name of the parameter.</param>
        /// <returns>Value of the specified parameter, if it exists.</returns>
        public string this[string param]
        {
            get { return (_params[param]); }
        }

        public int Count
        {
            get { return _params.Count; }
        }
    }
}
