/**
* @file Ioh_1588_common.h
*
* @author Wipro Technologies 
* @date 27 July 2009
*
* @brief IEEE1588 common file used both by application and driver
*
* This file contains the various driver loading related
*files
*
* 
 * @par
 * IOH_Windows Driver Release version 1.0 
 * 
 * -- Copyright Notice --
 * 
 * @par
 * Copyright (c) 2009-2012, LAPIS semiconductor .
 * All rights reserved.
 * 
 * @par
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the LAPIS semiconductor  nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * 
 * @par
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * 
 * 
 * @par
 * -- End of Copyright Notice --
*/

#ifndef __IOH_1588_COMMON_H__
#define __IOH_1588_COMMON_H__

/**
 * @ingroup InterfaceLayer
 * @define  GUID_DEVINTERFACE_IOHIEEE1588
 * @brief Guid Device Inteface used from the user application to control the device
 */
DEFINE_GUID(GUID_DEVINTERFACE_IOHIEEE1588, 
0x1f2c728c, 0x5c50, 0x4b72, 0x9a, 0x2e, 0x2e, 0x2b, 0x23, 0xa4, 0xbb, 0x41);

/**
 * @ingroup InterfaceLayer
 * @enum IOH1588PTPPORT
 * @brief IEEE 1588 PTP Communication Port(Channel)
 */
typedef enum _IOH1588PTPPORT
{
    IOH_1588_GBE_0_1588PTP_PORT,     /**< PTP Communication Port on GBE-0 */
    IOH_1588_CAN_0_1588PTP_PORT,     /**< PTP Communication Port on CAN-0 */
    IOH_1588_PORT_INVALID            /**< Invalid PTP Communication Port */ 
} IOH1588PTPPORT,*PIOH1588PTPPORT;

/**
 * @ingroup InterfaceLayer
 * @enum IOH1588PTPPORTMODE
 * @brief PTP Port mode - Master or Slave or any
 */
typedef enum _IOH1588PTPPORTMODE
{
    IOH_1588PTP_PORT_MASTER,       /**< Master Mode */
    IOH_1588PTP_PORT_SLAVE,        /**< Slave Mode */
    IOH_1588PTP_PORT_ANYMODE,      /**< Timestamp all messages */
    IOH_1588PTP_PORT_MODE_INVALID  /**< Invalid PTP Port Mode */
}IOH1588PTPPORTMODE,*PIOH1588PTPPORTMODE;

/**
 * @ingroup InterfaceLayer
 * @struct IOH1588PORTCFGIOCTL
 * @brief Struct to pass port config data for ioctl call
 * @see IOCTL_1588_PORT_CONFIG_SET
 */
typedef struct _IOH1588PORTCFGIOCTL
{
    IOH1588PTPPORT ptpPort;   /**< IEEE 1588 PTP Communication Port */
    IOH1588PTPPORTMODE ptpPortMode; /**< Master, Slave, or Any mode */
}IOH1588PORTCFGIOCTL,*PIOH1588PORTCFGIOCTL;

/**
 * @ingroup InterfaceLayer
 * @enum IOH1588PTPMSGTYPE
 * @brief PTP Messages types that can be detected on communication port
 */
typedef enum  _IOH1588PTPMSGTYPE
{
    IOH_1588PTP_MSGTYPE_SYNC,     /**< PTP Sync message sent by Master or received by Slave */
    IOH_1588PTP_MSGTYPE_DELAYREQ, /**< PTP Delay_Req message sent by Slave or received by Master */
    IOH_1588PTP_MSGTYPE_UNKNOWN   /**< Other PTP and non-PTP message sent or received by both Master and/or Slave */
} IOH1588PTPMSGTYPE,*PIOH1588PTPMSGTYPE;

/**
 * @ingroup InterfaceLayer
 * @struct IOH1588TIMEVALUE
 * @brief Struct to hold 64 bit SystemTime and TimeStamp values
 */
typedef struct  _IOH1588TIMEVALUE
{
    ULONG timeValueLowWord;     /**< Lower 32 bits of time value */
    ULONG timeValueHighWord;    /**< Upper 32 bits of time value */  
} IOH1588TIMEVALUE,*PIOH1588TIMEVALUE;

/**
 * @ingroup InterfaceLayer
 * @struct IOH1588UUID
 * @brief Struct to hold 48 bit UUID values captured in Sync or Delay_Req messages
 */
typedef struct  _IOH1588UUID
{
    ULONG uuidValueLowWord;        /**< The lower 32 bits of UUID */
    ULONG uuidValueHighHalfword;   /**< The upper 16 bits of UUID */  
} IOH1588UUID,*PIOH1588UUID;

/**
 * @ingroup InterfaceLayer
 * @struct IOH1588PTPMSGDATA
 * @brief Struct for data from the PTP message returned when TimeStamp available
 */
typedef struct  _IOH1588PTPMSGDATA
{
    IOH1588PTPMSGTYPE   ptpMsgType;     /**< PTP Messages type */
    IOH1588TIMEVALUE    ptpTimeStamp;   /**< 64 bit TimeStamp value from PTP Message */
    IOH1588UUID         ptpUuid;          /**< 48 bit UUID value from the PTP Message */
    UINT16              ptpSequenceNumber;  /**< 16 bit Sequence Number from PTP Message */
} IOH1588PTPMSGDATA,*PIOH1588PTPMSGDATA;

/**
 * @ingroup InterfaceLayer
 * @struct IOH1588RXTXPOLLIOCTL
 * @brief Struct to pass PTP message data for ioctl call
 */
typedef struct  _IOH1588RXTXPOLLIOCTL
{
    IOH1588PTPPORT      ptpPort;   /**< IEEE 1588 PTP Communication Port */
    IOH1588PTPMSGDATA   ptpMsgData; /**< PTP message data */
} IOH1588RXTXPOLLIOCTL,*PIOH1588RXTXPOLLIOCTL;

/**
 * @ingroup InterfaceLayer
 * @struct IOH1588CANPOLLIOCTL
 * @brief Struct to pass CAN timestamp data for ioctl call
 */
typedef struct  _IOH1588CANPOLLIOCTL
{
    IOH1588PTPPORT      ptpPort;   /**< IEEE 1588 PTP Communication Port */
    IOH1588TIMEVALUE    ptpTimeStamp; /**< CAN PTP timestamp */
} IOH1588CANPOLLIOCTL,*PIOH1588CANPOLLIOCTL;

/**
 * @ingroup InterfaceLayer
 * @enum IOH1588AUXMODE
 * @brief Master or Slave Auxiliary Time Stamp (Snap Shot)
 */
typedef enum _IOH1588AUXMODE
{
    IOH_1588_AUXMODE_MASTER,          /**< Auxiliary Master Mode */
    IOH_1588_AUXMODE_SLAVE,           /**< Auxiliary Slave Mode */
    IOH_1588_AUXMODE_INVALID          /**< Invalid Auxiliary Mode */
}IOH1588AUXMODE,*PIOH1588AUXMODE;

/**
 * @ingroup InterfaceLayer
 * @struct IOH1588TIMEPOLLIOCTL
 * @brief Struct to pass timestamp data for ioctl call
 */
typedef struct  _IOH1588TIMEPOLLIOCTL
{
    ULONG pollFlag;   /**< time event */
    IOH1588TIMEVALUE timeVal; /**< timestamp value */
    IOH1588AUXMODE auxMode;   /**< Master or Slave mode */
} IOH1588TIMEPOLLIOCTL,*PIOH1588TIMEPOLLIOCTL;

/**
 * @ingroup InterfaceLayer
 * @struct IOH1588STATS
 * @brief Provides the number of times timestamps are locked for rx and tx PTP
 * messages. The counters are reinitialized when the module is reset.
 */
typedef struct  _IOH1588STATS
{
    ULONG rxMsgs; /**< Count of timestamps for received PTP Msgs */
    ULONG txMsgs; /**< Count of timestamps for transmitted PTP Msgs */
} IOH1588STATS,*PIOH1588STATS;

/**
 * @ingroup InterfaceLayer
 * @struct IOH1588AUXTIMEIOCTL
 * @brief Struct to pass aux time data for ioctl call
 */
typedef struct  _IOH1588AUXTIMEIOCTL
{
    IOH1588AUXMODE      auxMode;   /**< aux mode: master or slave */
    IOH1588TIMEVALUE    auxTime; /**< aux time snapshot */
} IOH1588AUXTIMEIOCTL,*PIOH1588AUXTIMEIOCTL;

/**
 * @ingroup InterfaceLayer
 * @enum IOH1588PTPVERSION
 * @brief 1588 PTP version value that can be detected on communication port
 */
typedef enum  _IOH1588PTPVERSION
{
    IOH_1588_PTP_VERSION_0,     /**< support version 1 only */
    IOH_1588_PTP_VERSION_1,     /**< support both version 1 and version 2 */
    IOH_1588_PTP_VERSION_INVALID  /**< Invalid version */
} IOH1588PTPVERSION,*PIOH1588PTPVERSION;

/**
 * @ingroup InterfaceLayer
 * @struct IOH1588VERSIONIOCTL
 * @brief Struct to pass timestamp data for ioctl call
 */
typedef struct  _IOH1588VERSIONIOCTL
{
    IOH1588PTPPORT    ptpPort;      /**< IEEE 1588 PTP Communication Port */
    IOH1588PTPVERSION ptpVersion;   /**< version value */
} IOH1588VERSIONIOCTL,*PIOH1588VERSIONIOCTL;

/**
 * @ingroup InterfaceLayer
 * @enum IOH1588VERSIONIOCTL
 * @brief 1588 PTP operation mode value that can be detected on communication port
 */
typedef enum  _IOH1588PTPOPERATIONMODE
{
    IOH_1588PTP_OP_MODE_SYNC_DELAYREQ_MSGS=0,
            /**< timestamp version 1 SYNC and DELAYED_REQ only */
    IOH_1588PTP_OP_MODE_V1_ALL_MSGS=1, 
            /**< timestamp version 1 all messages */
    IOH_1588PTP_OP_MODE_V1_V2_EVENT_MSGS=2,
            /**< timestamp version 1 and 2 event messages only */
    IOH_1588PTP_OP_MODE_V1_V2_ALL_MSGS=3,
            /**< timestamp version 1 and 2 all messages */
    IOH_1588PTP_OP_MODE_INVALID   /**< Invalid mode */
} IOH1588PTPOPERATIONMODE,*PIOH1588PTPOPERATIONMODE;

/**
 * @ingroup InterfaceLayer
 * @struct IOH1588OPERATIONMODEIOCTL
 * @brief Struct to pass timestamp data for ioctl call
 */
typedef struct  _IOH1588OPERATIONMODEIOCTL
{
    IOH1588PTPPORT           ptpPort;   /**< IEEE 1588 PTP Communication Port*/
    IOH1588PTPOPERATIONMODE  ptpOpMode; /**< IEEE 1588 operation mode */
} IOH1588OPERATIONMODEIOCTL,*PIOH1588OPERATIONMODEIOCTL;

/**
 * @ingroup Global
 * @enum ioh_status
 * @brief The status as returned from the HAL
 */
typedef enum _IOH_STATUS
{
    IOH_1588_SUCCESS,           /**< operation successful */
    IOH_1588_INVALIDPARAM,         /**< parameter passed is invalid */
    IOH_1588_NOTIMESTAMP,          /**< no time stamp available when polled */
    IOH_1588_INTERRUPTMODEINUSE,   /**< while operating in interrupt mode, polling not permitted */
    IOH_1588_FAILED,               /**< Internal error in driver */
    IOH_1588_UNSUPPORTED,    /**< Implementation does not support this feature */
}IOH_STATUS,*PIOH_STATUS;
#endif