/**
* @file ioh_dma_common.h
*
* @author Wipro Technologies 
* @date 21 Aug 2009
*
* @brief dma common file used both by application and driver
*
* This file contains the various driver loading related
*files
*
* 
 * @par
 * IOH_Windows Driver Release version 1.0 
 * 
 * -- Copyright Notice --
 * 
 * @par
 * Copyright (C) 2009-2012 LAPIS SEMICONDUCTOR Co., LTD.
 * All rights reserved.
 * 
 * @par
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the LAPIS SEMICONDUCTOR  nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * 
 * @par
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * 
 * 
 * @par
 * -- End of Copyright Notice --
*/

#ifndef __IOH_DMA_COMMON_H__
#define __IOH_DMA_COMMON_H__

/**
 * @ingroup InterfaceLayer
 * @define  GUID_DEVINTERFACE_IOHDMA
 * @brief Guid Device Inteface used from the user application to control the device
 */
DEFINE_GUID(GUID_DEVINTERFACE_IOHDMA, 
0xf8870157, 0x1f08, 0x48bc, 0x9d, 0x11, 0x5f, 0x5a, 0x45, 0xb1, 0xa3, 0xfd);

typedef enum _IOH_DMA_REQ_DEVICES
{
    IOH_UART0 = 0,
    IOH_UART1,
    IOH_UART2,
    IOH_UART3,
    IOH_SPI0,
}IOH_DMA_REQ_DEVICES;



typedef enum _IOH_DMA_CHANNELS
{
    IOH_DMA_CHANNEL0 = 0,
    IOH_DMA_CHANNEL1,
    IOH_DMA_CHANNEL2,
    IOH_DMA_CHANNEL3,
    IOH_DMA_CHANNEL4,
    IOH_DMA_CHANNEL5,
    IOH_DMA_CHANNEL6,
    IOH_DMA_CHANNEL7,
    IOH_DMA_CHANNEL_RESERVED
}IOH_DMA_CHANNELS;

/*!    @ingroup    InterfaceLayerFacilitators
    @enum        ioh_channel_request_id
    @brief        Constant used to denote the channel request type.
    @note        These constants are used by other modules to make the
                DMA module aware of the channel type it requires.
*/
typedef enum _IOH_CHANNEL_REQUEST_ID
{
    IOH_DMA_TX_DATA_REQ0 = 0,        /**< Transmission channel .     */
    IOH_DMA_RX_DATA_REQ0,        /**< Reception channel .        */
}IOH_CHANNEL_REQUEST_ID;


typedef struct _IOH_REQUEST_DMA
{
    IOH_DMA_CHANNELS        uChannel;/*!< channel to be reserved*/
    IOH_CHANNEL_REQUEST_ID  uDreq;   /*!< DMA request Rx/Tx*/
    IOH_DMA_REQ_DEVICES     uDevID;  /*!< Requesting Device Information */

}IOH_REQUEST_DMA,*PIOH_REQUEST_DMA;


typedef struct _IOH_ADDRESS_PAIR
{
    ULONG   uPhysicalAddress;
    PVOID   pVirtualAddress;

}IOH_ADDRESS_PAIR,*PIOH_ADDRESS_PAIR;

/*! @ingroup    InterfaceLayerFacilitators
    @struct     _ioh_dma_desc
    @brief      Format for specifying the descriptors.
    @note       This structure is used by other modules to make the
                DMA module aware of the channel descriptors in
                SCATTER_GATHER_MODE.
*/
typedef struct _IOH_DMA_ADDRESS_PAIR_DESC
{
    IOH_ADDRESS_PAIR iohInsideAddress;
    IOH_ADDRESS_PAIR iohOutsideAddress;
    ULONG uSize;                  /* Size */
    IOH_ADDRESS_PAIR iohNextDescAddress;
}IOH_DMA_ADDRESS_PAIR_DESC,*PIOH_DMA_ADDRESS_PAIR_DESC; 


/*! @ingroup    InterfaceLayerFacilitators
    @struct     _ioh_dma_desc
    @brief      Format for specifying the descriptors.
    @note       This structure is used by other modules to make the
                DMA module aware of the channel descriptors in
                SCATTER_GATHER_MODE.
*/
typedef struct _IOH_DMA_DESC
{
    ULONG uInsideAddress;         /* Inside address  */
    ULONG uOutsideAddress;        /* Outside address */
    ULONG uSize;                  /* Size */
    ULONG uNextDesc;              /* Next Descriptor address */
    
}IOH_DMA_DESC,*PIOH_DMA_DESC; 


/*!    @ingroup    InterfaceLayerFacilitators
    @struct        __ioh_dma_mode_param
    @brief        Format for specifying the mode characteristics of
                a channel.
    @note        This structure is used by other modules to make the
                DMA module aware of the channel mode characteristics.
*/
//
typedef struct _IOH_DMA_MODE_PARAM
{
    UINT16  TransferDirection;      /**< Direction of Transfer(IN to OUT or OUT to IN).             */
    UINT16  DMASizeType;            /**< Type of DMA Transfer size (8bit, 16bit or 32bit).            */
    UINT16  DMATransferMode;        /**< Mode of Transfer (ONE_SHOT_MODE or SCATTER_GATHER_MODE).    */
}IOH_DMA_MODE_PARAM,*PIOH_DMA_MODE_PARAM;

typedef struct _IOH_SCATTER_GATTHER_MODE
{
    PIOH_DMA_ADDRESS_PAIR_DESC pStart;
    PIOH_DMA_ADDRESS_PAIR_DESC pEnd;
}IOH_SCATTER_GATHER_MODE,*PIOH_SCATTER_GATHER_MODE;


typedef struct _IOH_ONE_SHOT_MODE
{
    IOH_ADDRESS_PAIR iohInsideAddress;
    IOH_ADDRESS_PAIR iohOutsideAddress;
    //ULONG insideAddress;  /* inside bridge address*/
    //ULONG outsideAddress; /*outside bridge address*/
    int count;
}IOH_ONE_SHOT_MODE,*PIOH_ONE_SHOT_MODE;


typedef struct _IOH_SET_DMA
{
    ULONG uChannel; /*Handle to the opened channel*/

    int priority; /* Channel priority*/

    IOH_DMA_MODE_PARAM iohModeParam; /*Transfer mode details*/

    union 
    {
        IOH_ONE_SHOT_MODE iohOneShotMode; /*data for ONE SHOT MODE*/
        IOH_SCATTER_GATHER_MODE iohScatterGatherMode; /*data for SCATTER-GATHER MODE*/
    };
}IOH_SET_DMA,*PIOH_SET_DMA;

/*! @ingroup    InterfaceLayer
    @def        DMA_ONE_SHOT_MODE
    @brief      Constant used to denote the mode as ONE_SHOT.
    @note       This constant is used by other modules to make the
                DMA module aware of the mode it requires.
*/
#define DMA_ONE_SHOT_MODE                   (UINT16)(0x2)

/*!    @ingroup    InterfaceLayer
    @def        DMA_SCATTER_GATHER_MODE
    @brief        Constant used to denote the mode as SCATTER_GATHER.
    @note        This constant is used by other modules to make the
                DMA module aware of the mode it requires.
*/
#define DMA_SCATTER_GATHER_MODE             (UINT16)(0x1)

/*!    @ingroup    InterfaceLayer
    @def        IOH_DMA_SIZE_TYPE_8BIT
    @brief        Constant used to denote the access size as 8BIT.
    @note        This constant is used by other modules to make the
                DMA module aware of the access size it requires.
*/
#define IOH_DMA_SIZE_TYPE_8BIT                  ((0x3 << 12))

/*!    @ingroup    InterfaceLayer
    @def        IOH_DMA_SIZE_TYPE_16BIT
    @brief        Constant used to denote the access size as 16BIT.
    @note        This constant is used by other modules to make the
                DMA module aware of the access size it requires.
*/
#define IOH_DMA_SIZE_TYPE_16BIT                 ((0x2 << 12))

/*!    @ingroup    InterfaceLayer
    @def        IOH_DMA_SIZE_TYPE_32BIT
    @brief        Constant used to denote the access size as 32BIT.
    @note        This constant is used by other modules to make the
                DMA module aware of the access size it requires.
*/
#define IOH_DMA_SIZE_TYPE_32BIT                 (0x0)


/*!    @ingroup    InterfaceLayer
    @def        IOH_DMA_DIR_OUT_TO_IN
    @brief        Constant used to denote the transfer direction as OUT_TO_IN.
    @note        This constant is used by other modules to make the
                DMA module aware of the transfer direction it requires.
*/
#define IOH_DMA_DIR_OUT_TO_IN                (0x4)

/*!    @ingroup    InterfaceLayer
    @def        IOH_DMA_DIR_IN_TO_OUT
    @brief        Constant used to denote the transfer direction as IN_TO_OUT.
    @note        This constant is used by other modules to make the
                DMA module aware of the transfer direction it requires.
*/
#define IOH_DMA_DIR_IN_TO_OUT                (0x0)


/*
    DMA EXPORT Functions
*/

NTSTATUS ioh_request_dma(PIOH_REQUEST_DMA pRequestDMA,PULONG pChannel);

NTSTATUS ioh_free_dma(ULONG uChannel);

NTSTATUS ioh_set_dma_channel(PIOH_SET_DMA pSetDMA);

NTSTATUS ioh_enable_dma_channel(ULONG uChannel);

NTSTATUS ioh_disable_dma_channel(ULONG uChannel);

NTSTATUS ioh_direct_start_dma(ULONG uChannel);

NTSTATUS ioh_add_dma_desc(PIOH_SET_DMA pSetDMA);

#endif