 //*-- Intel Copyright Notice --
 //* 
 //* 
 //* Copyright (c) 2002-2010 Intel Corporation All Rights Reserved.
 //* 
 //*
 //* The source code contained or described herein and all documents
 //* related to the source code ("Material") are owned by Intel Corporation
 //* or its suppliers or licensors.  Title to the Material remains with
 //* Intel Corporation or its suppliers and licensors.
 //* 
 //*
 //* The Material is protected by worldwide copyright and trade secret laws
 //* and treaty provisions. No part of the Material may be used, copied,
 //* reproduced, modified, published, uploaded, posted, transmitted,
 //* distributed, or disclosed in any way except in accordance with the
 //* applicable license agreement .
 //* 
 //*
 //* No license under any patent, copyright, trade secret or other
 //* intellectual property right is granted to or conferred upon you by
 //* disclosure or delivery of the Materials, either expressly, by
 //* implication, inducement, estoppel, except in accordance with the
 //* applicable license agreement.
 //* 
 //*
 //* Unless otherwise agreed by Intel in writing, you may not remove or
 //* alter this notice or any other notice embedded in Materials by Intel
 //* or Intel's suppliers or licensors in any way.
 //* 
 //* 
 //* For further details, please see the file README.TXT distributed with
 //* this software.
 //* 
 //*
 //* -- End Intel Copyright Notice --/
 //


//                              -*- Mode: Verilog -*-
// Filename        : I2C_EEPROM.v
// Description     : I2C CPLD switching trigger.



/* 
---TO ACCESS THE CPLD--- 
 Device Address :8'b1100 1010 
 The last bit is a READ(1) or WRITE(0)
  
---TO PROGRAM THE EEPROM---
 
 Data Reg Address : 8'b1111 0010 -- register0
 Data : The last bit [0] must have either 1 (ON), or 0 (OFF)

---TO SWITCH PCIE---
 There are 2 switches, therefore a total of 3 registers are required -- 1 for control, 2 for data 
 
 Control Register Address : 8'b1111 1010 -- register4
 Data : Bit[0] controls Switch 1, Bit[1] controls Switch 2
  
 PCIE Switch 1
 Data Reg Address : 8'b1111 0110 -- register2 
 Data : The last bit [0] must have either 1 (ON), or 0 (OFF)
 
 PCIE Switch 2
 Data Reg Address : 8'b1111 1000 -- register3
 Data : The last bit [0] must have either 1 (ON), or 0 (OFF)
 
---TO READ THE VERSION---
 Reg Address : 8'b1111 0100 -- register1
*/

`timescale 1ns/1ns
 
module i2c_eeprom
  (
	 input CLK,
	 input RST_N,
	
	// I2C signals
	 input SCL,
	 inout SDA,
	 input HEADER1, // -- AND with register 
	 input HEADER2,
	 output SWITCH_OUT,
	 output PCIE_OUT1,
	 output PCIE_OUT2
	);
	
	parameter IDLE = 4'h0,
				 DEV_ADDR = 4'h1,
				 ACK = 4'h2,
				 REG_ADDR = 4'h3,
				 WRITE_MODE = 4'h4,
				 READ_MODE = 4'h5;

	reg 		 rw_mode; // if 1 read, if 0 write
	reg 		 scl_dly1;
	reg 		 sda_dly1, sda_dly2;
	reg 		 sda_en;
	reg 		 addr_mismatch;
	reg 		 addr_cycle_done, addr_cycle_done_dly1 = 0, addr_cycle_done_dly2 = 0;
	reg 		 data_cycle_done, data_cycle_done_dly1 = 0;
	reg [3:0] cnt_8;
	reg [3:0] n_state, n_state_falledge;
	reg [7:0] temp_reg_data;
	reg 		 tc_data_in_muxed;
	reg 		 started;
	reg 		 addr_cycle_done_rise_pulse_dly1 = 0;

	
	wire 		 scl_clk;
	wire 		 addr_cycle_done_rise_pulse, data_cycle_done_fall_pulse;
	wire 		 stop_pulse, start_pulse;
		
	reg [7:0] register0, register1 = 8'b00100001;// 4 bits for year,2 bits for version
	reg [7:0] register3 = 8'b1111_0000, register2 = 8'b1111_0000; // -- by default
	reg [7:0] register4 = 8'b1111_0011;
	wire 		 tc_reg_write;
	reg [7:0] reg_data; //Data that needs to be stored in the CPLD (Version)
	reg [5:0] reg_add;
		
	assign 	 SWITCH_OUT = register0[0];
	assign 	 PCIE_OUT1 = HEADER1 ? 1'bz : 0;
	assign 	 PCIE_OUT2 = HEADER2 ? 1'bz : 0;
	
	// Flop all I2C signals
	always @ (posedge CLK, negedge RST_N)
	  if (!RST_N)
		 begin
			 scl_dly1 <= 1;
			 sda_dly1 <= 1;
			 sda_dly2 <= 1;
		 end
	  else
		 begin
			 scl_dly1 <= SCL;
			 sda_dly1 <= SDA;
			 sda_dly2 <= sda_dly1;
		 end

	// Falling edge of SDA when SCL is high is a START condition
	assign start_pulse = !sda_dly1 & sda_dly2 & scl_dly1;

	// Rising edge of SDA when SCL is high is a STOP condition
	assign stop_pulse = sda_dly1 & !sda_dly2 & scl_dly1;
	
	// Add global clock buffer to SCL
	assign scl_clk = SCL;
	
	// Indicate a valid cycle ORDERING IS IMPORTANT!
	always @ (start_pulse, stop_pulse, RST_N, data_cycle_done,addr_cycle_done, addr_mismatch)
	  begin
		  if (!RST_N)
			 started <= 0;
		  else if (data_cycle_done & addr_cycle_done)  // When all is done
			 started <= 0;
		  else if (start_pulse)   // Sees a START condition
			 started <= 1;
		  else if (stop_pulse)    // Sees a STOP condition
			 started <= 0;
		  else if (addr_mismatch)  // Sees invalid i2c address
			 started <= 0;
		  else
			 started <= started;
	  end // always @ (sda_dly1, RST_N)


	// Shift SDA into a 8 bit shift register
	always @ (posedge scl_clk, negedge RST_N)
	  begin
		  if (!RST_N)
			 temp_reg_data <= 8'h00;
		  else if ((started == 1) && (n_state != ACK))
			 temp_reg_data <= {temp_reg_data[6:0], sda_dly1};
		  else
			 temp_reg_data <= temp_reg_data;
	  end

	// Clock state at negedge to adjust timing
	always @ (negedge scl_clk, negedge RST_N)
	  begin
		  if (!RST_N)
			 n_state_falledge[3:0] <= 4'h0;
		  else
			 n_state_falledge[3:0] <= n_state[3:0];
	  end
	
	// SDA's tristate buffer controlled here
	assign 	 SDA = sda_en ? ((n_state_falledge[3:0] == READ_MODE) ? tc_data_in_muxed : 1'b0) : 1'bZ;
	
	// Device sends data to master
	always @ (negedge scl_clk, negedge RST_N)
	  begin
		  if (!RST_N)
			 sda_en <= 0;
		  else if ((n_state == ACK) || (n_state == READ_MODE))
			 sda_en <= 1;
		  else
			 sda_en <= 0;
	  end

	// State Machine
	always @ (posedge scl_clk, negedge RST_N)
	  begin
		  if (!RST_N)
			 begin
				 n_state <= IDLE;
				 addr_mismatch <= 0;
				 data_cycle_done <= 0;
				 addr_cycle_done <= 0;
				 cnt_8 <= 0;
				 rw_mode <= 0;
			 end
		  
		  else
			 begin
				 case (n_state)
					IDLE:  
					  begin
						  addr_mismatch <= 0;
						  data_cycle_done <= 0;
						  addr_cycle_done <= 0;
						  						  
						  if (started)
							 begin
								 n_state <= DEV_ADDR;
								 cnt_8 <= cnt_8 + 1'b1;
							 end
						  else
							 begin
								 n_state <= IDLE;
								 cnt_8 <= 0;
							 end
					  end

					// Reads in i2c address
					DEV_ADDR:
					  begin
						  cnt_8 <= cnt_8 + 1'b1;
						  data_cycle_done <= 0;
						  addr_cycle_done <= 0;
						  
						  if (cnt_8 == 3'h7)
							 begin
								 if (temp_reg_data[6:0] == 7'b1100101)// device address matches and then sends acknowledge
									begin
										n_state <= ACK;
										addr_mismatch <= 0;
									end
								 else
									begin
										n_state <= IDLE;
										addr_mismatch <= 1;
									end
							 end
						  else
							 begin
								 n_state <= DEV_ADDR;
								 addr_mismatch <= 0;
							 end // else: !if(cnt_8 == 3'h7)
					  end

					// Sends acknowledge 
					ACK:
					  begin
						  cnt_8 <= 4'h0;
						  addr_mismatch <= addr_mismatch;
						  addr_cycle_done <= addr_cycle_done;

						  if (!addr_cycle_done)
							 rw_mode <= temp_reg_data[0];
						  else
							 rw_mode <= rw_mode;
						  						  
						  if (data_cycle_done)
							 begin
								 n_state <= IDLE;
								 cnt_8 <= 0;
								 data_cycle_done <= 0;
							 end								 
						  else if (addr_cycle_done)
							 begin
								 data_cycle_done <= data_cycle_done;
								 
								 if (!rw_mode)
									n_state <= WRITE_MODE;
								 else
									n_state <= READ_MODE;
							 end 
						  else
							 begin
								 n_state <= REG_ADDR;
								 data_cycle_done <= data_cycle_done;
							 end // else: !if(addr_cycle_done)
					  end // case: ack

					// Reads in register address
					REG_ADDR:
					  begin
						  addr_mismatch <= addr_mismatch;
						  cnt_8 <= cnt_8 + 1'b1;
						  data_cycle_done <= 0;
						  
						  if (cnt_8 == 4'h7)
							 begin
								 if ((temp_reg_data[6:0] == 7'b1111_001) || (temp_reg_data[6:0] == 7'b1111_010) || (temp_reg_data[6:0] == 7'b1111_011) || (temp_reg_data[6:0] == 7'b1111_100) ||(temp_reg_data[6:0] == 7'b1111_101))
									begin
										n_state <= ACK;
										addr_cycle_done <= 1'b1;
									end
								 else
									begin
										n_state <= IDLE;
										addr_mismatch <= 1;
									end
							 end // if (cnt_8 == 4'h7)
						  else
							 begin
								 n_state <= REG_ADDR;
								 addr_cycle_done <= 1'b0;
							 end
					  end // case: second_addr

					// Write into local registers
					WRITE_MODE:
					  begin
						  addr_mismatch <= addr_mismatch;
						  cnt_8 <= cnt_8 + 1'b1;
						  addr_cycle_done <= addr_cycle_done;
						  
						  if (cnt_8 == 4'h7)
							 begin
								 n_state <= ACK;
								 data_cycle_done <= 1;
							 end
						  else
							 begin
								 n_state <= WRITE_MODE;
								 data_cycle_done <= 0;
							 end
					  end // case: write_mode

					// Read from local registers or TestChip
					READ_MODE:
					  begin
						  addr_mismatch <= addr_mismatch;
						  cnt_8 <= cnt_8 + 1'b1;
						  addr_cycle_done <= addr_cycle_done;
						  
						  if (cnt_8 == 4'h7)
							 begin
								 n_state <= IDLE;
								 data_cycle_done <= 1;
							 end
						  else
							 begin
								 n_state <= READ_MODE;
								 data_cycle_done <= 0;
							 end
					  end // case: read_mode

					default:
					  begin
						  addr_mismatch <= 1;
						  cnt_8 <= 4'h0;
						  data_cycle_done <= 1;
						  addr_cycle_done <= 1;
						  rw_mode <= 0;
						  n_state <= IDLE;
					  end
					
				 endcase // case(n_state)
			 end // else: !if(!RST_N)
	  end // always @ (posedge scl_clk, negedge RST_N)

	// Flop signals to create a pulse
	always @ (posedge CLK)
	  begin
		  addr_cycle_done_dly1 <= addr_cycle_done;
		  addr_cycle_done_dly2 <= addr_cycle_done_dly1;
		  data_cycle_done_dly1 <= data_cycle_done;
	  end
	
	// Indicate the address is ready to be latched
	assign addr_cycle_done_rise_pulse = addr_cycle_done_dly1 && !addr_cycle_done_dly2;

	always @ (posedge CLK)
		  addr_cycle_done_rise_pulse_dly1 <= addr_cycle_done_rise_pulse;
		
	// Indicate the data has been latched or sent.
	assign data_cycle_done_fall_pulse = !data_cycle_done && data_cycle_done_dly1;

	//Flopping the address in a register for comparison purpose
	always @ (posedge CLK, negedge RST_N)
	  begin
		  if (!RST_N)
			 reg_add[5:0] <= 6'h00;
		  else if (addr_cycle_done_rise_pulse)
			 reg_add[5:0] <= temp_reg_data[5:0];
	  end
	
	// Write enable
	
	assign tc_reg_write = !rw_mode && data_cycle_done_fall_pulse; 

	always @ (posedge CLK, negedge RST_N)
	  begin
		  if (!RST_N)
			 register0 <= 8'h00;
		  else
			 begin
				 if ((reg_add == 6'b11_0010) && tc_reg_write)
					register0 <= temp_reg_data;
				 else
					register0 <= register0;
			 end
	  end // always @ (posedge CLK, negedge RST_N)

	always @ (posedge CLK, negedge RST_N)
	  begin
		  if (!RST_N)
			 register1 <= register1;
		  else
			 begin				 
				 if ((reg_add == 6'b11_0100) && tc_reg_write)
					register1 <= register1;
				 else
					register1 <= register1;
			 end			
	  end // always @ (posedge CLK, negedge RST_N)
	
	always @ (posedge CLK, negedge RST_N)
	  begin
		  if (!RST_N)
			 register2 <= 8'h00;
		  else
			 begin
				 if ((reg_add == 6'b11_0110) && tc_reg_write)
					register2 <= temp_reg_data;
				 else
					register2 <= register2;
			 end
	  end // always @ (posedge CLK, negedge RST_N)

	always @ (posedge CLK, negedge RST_N)
	  begin
		  if (!RST_N)
			 register3 <= 8'h00;
		  else
			 begin
				 if ((reg_add == 6'b11_1000) && tc_reg_write)
					register3 <= temp_reg_data;
				 else
					register3 <= register3;
			 end
	  end // always @ (posedge CLK, negedge RST_N)

	always @ (posedge CLK, negedge RST_N)
	  begin
		  if (!RST_N)
			 register4 <= 8'h00;
		  else
			 begin
				 if ((reg_add == 6'b11_1010) && tc_reg_write)
					register4 <= temp_reg_data;
				 else
					register4 <= register4;
			 end
	  end // always @ (posedge CLK, negedge RST_N)
	
	//Read enable
	
	always @ (posedge CLK, negedge RST_N)
	  begin
		  if (!RST_N)
			 begin
				 reg_data[7:0] <= 8'h00;
			 end
		  else if (rw_mode & addr_cycle_done_rise_pulse_dly1)
			 begin
				 case (reg_add)
					6'b11_0010: reg_data <= register0[7:0];
					6'b11_0100: reg_data <= register1[7:0];
					6'b11_0110: reg_data <= register2[7:0];
					6'b11_1000: reg_data <= register3[7:0];
					6'b11_1010: reg_data <= register4[7:0];
					default: reg_data <= 8'hff;
				 endcase // case (reg_add)
			 end
	  end

	always @ (negedge scl_clk, negedge RST_N)
	  begin
		  if (!RST_N)
			 tc_data_in_muxed <= 0;
		  else
			 case (cnt_8)
				4'h0: tc_data_in_muxed <= reg_data[7];
				4'h1: tc_data_in_muxed <= reg_data[6];
				4'h2: tc_data_in_muxed <= reg_data[5];
				4'h3: tc_data_in_muxed <= reg_data[4];
				4'h4: tc_data_in_muxed <= reg_data[3];
				4'h5: tc_data_in_muxed <= reg_data[2];
				4'h6: tc_data_in_muxed <= reg_data[1];
				4'h7: tc_data_in_muxed <= reg_data[0];
				default : tc_data_in_muxed <= 1'b1;
			 endcase // case(cnt8)
	  end // always @ (negedge scl_clk, negedge RST_N)
	
endmodule // i2c_intf


	
				  
						  
					
					

						  
					

						 
					 
			
					  
					
				  
			 
					 
					 
					 