/*
 * The POST code decoder for beta version of Shell Bay Fab2
 *
 * File: Post_Decorder_fab2.v
 * Rev.: 1.00
 * Date: 09 Feb 2010
 *
*-- Intel Copyright Notice --
* Copyright (c) 2002-2009 Intel Corporation All Rights Reserved.
* The source code contained or described herein and all documents
* related to the source code ("Material") are owned by Intel Corporation
* or its suppliers or licensors.  Title to the Material remains with
* Intel Corporation or its suppliers and licensors.
* The Material is protected by worldwide copyright and trade secret laws
* and treaty provisions. No part of the Material may be used, copied,
* reproduced, modified, published, uploaded, posted, transmitted,
* distributed, or disclosed in any way except in accordance with the
* applicable license agreement .
* No license under any patent, copyright, trade secret or other
* intellectual property right is granted to or conferred upon you by
* disclosure or delivery of the Materials, either expressly, by
* implication, inducement, estoppel, except in accordance with the
* applicable license agreement.
* Unless otherwise agreed by Intel in writing, you may not remove or
* alter this notice or any other notice embedded in Materials by Intel
* or Intel's suppliers or licensors in any way.
* For further details, please see the file README.TXT distributed with
* this software.
* -- End Intel Copyright Notice --
 */


//-------------------------------------------------------------------------
// Define for state matchine
`define START   4'b0000 // 0
`define ADDR3   4'b0010 // 2
`define ADDR2   4'b0011 // 3
`define ADDR1   4'b0100 // 4
`define ADDR0   4'b0101 // 5
`define DATA1   4'b0110 // 6
`define DATA0   4'b0111 // 7
`define TAR00   4'b1000 // 8
`define TAR01   4'b1001 // 9
`define SYNC0   4'b1010 // A
`define TAR10   4'b1100 // C
`define IDLE0   4'b1111 // F


//-------------------------------------------------------------------------
// modlue POST_Decoder
//-------------------------------------------------------------------------
module POST_Decoder_Gold (
    LAD,         // Multiplexed Command, Address, and Data:
    LFRAMEN,     // Frame: Indicates start of a new cycle, termination of broken cycle.
    LRESETN,     // Reset: Same as PCI Reset on the host.
    LCLK,        // Clock: Same 33MHz clock as PCI clock on the host.

    SEVENSEGL,   // Output to seven seg LED for low bibble
    SEVENSEGH,   // Output to seven seg LED for high bibble

    PWRBTN,      // Input from push-swtich 5
    SUS_S5,      // Input from COMe
    WAKE0_PCIE,  // Input from IOH
    CPLD_PWRBTN, // Output to COMe

    KBD_RST,     // Input from SIO
    CPLD_KBD_RST // Output to IOH(IOPLB_RST#)
);


//-------------------------------------------------------------------------
// Signals
inout   [3:0] LAD;          // Multiplexed Command, Address, and Data:
input   LFRAMEN;            // Frame: Indicates start of a new cycle, termination of broken cycle.
input   LRESETN;            // Reset: Same as PCI Reset on the host.
input   LCLK;               // Clock: Same 33MHz clock as PCI clock on the host.

output  [7:0] SEVENSEGL;    // Output to seven seg LED for low bibble
output  [7:0] SEVENSEGH;    // Output to seven seg LED for high bibble

input   PWRBTN;             //
input   SUS_S5;             //
input   WAKE0_PCIE;         //
output  CPLD_PWRBTN;        //

input   KBD_RST;
output  CPLD_KBD_RST;


//-------------------------------------------------------------------------
// Registers
reg     lframe_nreg;
reg     [3:0] lad_rin;
reg     [3:0] lpc_state;
reg     [7:0] data;
reg     [7:0] seven_seg_L;  // Output to seven seg LED for low bibble
reg     [7:0] seven_seg_H;  // Output to seven seg LED for high bibble


//-------------------------------------------------------------------------
// Interface
always @(posedge LCLK)
begin
    lad_rin <=LAD;
    lframe_nreg <= LFRAMEN;
end

assign SEVENSEGL = seven_seg_L;
assign SEVENSEGH = seven_seg_H;

//-------------------------------------------------------------------------
// LPC state machine
always @(posedge LCLK or negedge LRESETN)
begin
    if (LRESETN == 1'b0)
    begin
        lpc_state <= `IDLE0;
    end

    else
    begin
        case(lpc_state)
        `IDLE0: // Looking for a START condition
        begin
            if((lframe_nreg == 1'b0) && (lad_rin == 4'b0000))
                lpc_state <= `START;    // START condition detected
        end

        `START: // Check for I/O write transaction
        begin
            if(lframe_nreg == 1'b0)
            begin
                if(lad_rin != 4'b0000)
                    lpc_state <= `IDLE0;    // unsupported start code
            end
            else
            begin
                if(lad_rin[3:1] == 3'b001)
                begin
                    lpc_state <= `ADDR3;    // I/O write detected
                end
                else
                    lpc_state <= `IDLE0;    //unsupported command
            end
        end

        // --------------------------------
        // I/O write transaction processing
        // --------------------------------
        `ADDR3: // Write Data Address Nibble 3
        begin
            if((lframe_nreg == 1'b0) || (lad_rin != 4'b0000))
                lpc_state <= `IDLE0;    // ab||t cycle, bad frame or address mismatch
            else
            begin
                lpc_state <= `ADDR2;
            end
        end

        `ADDR2: // Write Data Address Nibble 2
        begin
            if((lframe_nreg == 1'b0) || (lad_rin != 4'b0000))
                lpc_state <= `IDLE0;    // ab||t cycle, bad frame or address mismatch
            else
            begin
                lpc_state <= `ADDR1;
            end
        end

        `ADDR1: // Write Data Address Nibble 1
        begin
            if((lframe_nreg == 1'b0) || (lad_rin != 4'b1000))
                lpc_state <= `IDLE0; // ab||t cycle, bad frame or address mismatch
            else 
            begin
                lpc_state <= `ADDR0;
            end
        end

        `ADDR0: // Write Data Address Nibble 0
        begin
            if((lframe_nreg == 1'b0) || (lad_rin != 4'b0000))
                lpc_state <= `IDLE0;    // ab||t cycle, bad frame or address mismatch
            else
                lpc_state <= `DATA0;
        end

        `DATA0: // Data LSN (Least Significant Nibble)
        begin
            data[3:0] <= lad_rin[3:0];  //latch data (LSN)
            if(lframe_nreg == 1'b1)
                lpc_state <= `DATA1;
            else
                lpc_state <= `IDLE0;
        end

        `DATA1: // Data MSN (Most Significant Nibble)
        begin
            data[7:4] <= lad_rin[3:0];  // latch data (MSN)
            if(lframe_nreg == 1'b1)
                lpc_state <= `TAR00;
            else
                lpc_state <= `IDLE0;
        end

        `TAR00: // Write Data Turn Around Cycle 0
        begin
            if((lframe_nreg == 1'b1) && (lad_rin == 4'b1111))
                lpc_state <= `TAR01;
            else
                lpc_state <= `IDLE0;
        end

        `TAR01: // Write Data Turn Around Cycle 1
        begin
            if(lframe_nreg == 1'b1)
                lpc_state <= `SYNC0;
            else
                lpc_state <= `IDLE0;
        end

        `SYNC0: // Write Data Sync Cycle
        begin
            if(lframe_nreg == 1'b1)
                lpc_state <= `TAR10;
            else
                lpc_state <= `IDLE0;
        end

        `TAR10: // Write Data Final Turn Around Cycle
        begin
            lpc_state <= `IDLE0;        // I/O write cycle end
        end

        default:
        begin
            lpc_state <= `IDLE0;        // all other cases
        end
        endcase
    end
end


//-------------------------------------------------------------------------
// Output signals to seven seg LED
always @(posedge LCLK or negedge LRESETN)
begin
    if (LRESETN == 1'b0)
    begin
        seven_seg_H <= 8'b00000000;
    end

    else
    begin
        case(data[7:4])
        4'b0000:    seven_seg_H <= 8'b00111111; // displays a 0
        4'b0001:    seven_seg_H <= 8'b00000110; // displays a 1
        4'b0010:    seven_seg_H <= 8'b01011011; // displays a 2
        4'b0011:    seven_seg_H <= 8'b01001111; // displays a 3
        4'b0100:    seven_seg_H <= 8'b01100110; // displays a 4
        4'b0101:    seven_seg_H <= 8'b01101101; // displays a 5
        4'b0110:    seven_seg_H <= 8'b01111101; // displays a 6
        4'b0111:    seven_seg_H <= 8'b00100111; // displays a 7
        4'b1000:    seven_seg_H <= 8'b01111111; // displays a 8
        4'b1001:    seven_seg_H <= 8'b01101111; // displays a 9
        4'b1010:    seven_seg_H <= 8'b01110111; // displays a A
        4'b1011:    seven_seg_H <= 8'b01111100; // displays a b
        4'b1100:    seven_seg_H <= 8'b00111001; // displays a C
        4'b1101:    seven_seg_H <= 8'b01011110; // displays a d
        4'b1110:    seven_seg_H <= 8'b01111001; // displays a E
        4'b1111:    seven_seg_H <= 8'b01110001; // displays a F
        default:    seven_seg_H <= 8'b01111111; // displays a 8
        endcase
    end
end

always @(posedge LCLK or negedge LRESETN)
begin
    if (LRESETN == 1'b0)
    begin
        seven_seg_L <= 8'b00000000;
    end

    else
    begin
        case(data[3:0])
        4'b0000:    seven_seg_L <= 8'b10111111; // displays a 0
        4'b0001:    seven_seg_L <= 8'b10000110; // displays a 1
        4'b0010:    seven_seg_L <= 8'b11011011; // displays a 2
        4'b0011:    seven_seg_L <= 8'b11001111; // displays a 3
        4'b0100:    seven_seg_L <= 8'b11100110; // displays a 4
        4'b0101:    seven_seg_L <= 8'b11101101; // displays a 5
        4'b0110:    seven_seg_L <= 8'b11111101; // displays a 6
        4'b0111:    seven_seg_L <= 8'b10100111; // displays a 7
        4'b1000:    seven_seg_L <= 8'b11111111; // displays a 8
        4'b1001:    seven_seg_L <= 8'b11101111; // displays a 9
        4'b1010:    seven_seg_L <= 8'b11110111; // displays a A
        4'b1011:    seven_seg_L <= 8'b11111100; // displays a b
        4'b1100:    seven_seg_L <= 8'b10111001; // displays a C
        4'b1101:    seven_seg_L <= 8'b11011110; // displays a d
        4'b1110:    seven_seg_L <= 8'b11111001; // displays a E
        4'b1111:    seven_seg_L <= 8'b11110001; // displays a F
        default:    seven_seg_L <= 8'b11111111; // displays a 8
        endcase
    end
end

// Output signals to Power Swich
assign CPLD_PWRBTN = PWRBTN & ((WAKE0_PCIE) | (SUS_S5)) ;
assign CPLD_KBD_RST = SUS_S5 ? KBD_RST : 1'b0 ;

endmodule   // module POST_Decoder
