 //*-- Intel Copyright Notice --
 //* 
 //* 
 //* Copyright (c) 2002-2011 Intel Corporation All Rights Reserved.
 //* 
 //*
 //* The source code contained or described herein and all documents
 //* related to the source code ("Material") are owned by Intel Corporation
 //* or its suppliers or licensors.  Title to the Material remains with
 //* Intel Corporation or its suppliers and licensors.
 //* 
 //*
 //* The Material is protected by worldwide copyright and trade secret laws
 //* and treaty provisions. No part of the Material may be used, copied,
 //* reproduced, modified, published, uploaded, posted, transmitted,
 //* distributed, or disclosed in any way except in accordance with the
 //* applicable license agreement .
 //* 
 //*
 //* No license under any patent, copyright, trade secret or other
 //* intellectual property right is granted to or conferred upon you by
 //* disclosure or delivery of the Materials, either expressly, by
 //* implication, inducement, estoppel, except in accordance with the
 //* applicable license agreement.
 //* 
 //*
 //* Unless otherwise agreed by Intel in writing, you may not remove or
 //* alter this notice or any other notice embedded in Materials by Intel
 //* or Intel's suppliers or licensors in any way.
 //* 
 //* 
 //* For further details, please see the file README.TXT distributed with
 //* this software.
 //* 
 //*
 //* -- End Intel Copyright Notice --/
 //

//                              -*- Mode: Verilog -*-
// Filename        : pwr_mgt.v
// Description     : Power management
// Version         : 0.47
// Description     : Power management module. Boots up without stalling in any states. This version is for Fab D1 Little Bay only.
`timescale 1ps/ 1ps



module pwr_mgt
  (
	 input   CLK,	    //Clock for EC 5.56MHz on Little Bay
	 input 	RST_BTN_N,
	 input 	PWR_BTN_N,
	 input   SLPMODE,
	 input 	SLPRDY_N,
	 input 	RSTRDY_N,
	 output reg [2:0] PWRMODE = 0,
	 output reg 		PWROK = 0,
	 output reg 		CPU_RST_N = 0,
	 output reg 		RSMRST_N = 0,
	 output reg 		RSTWARN = 0,
	 output reg 		SLP_S3_N = 0,
	 input 				S0_PWRGD,
	 output reg 		SLP_S5_N = 0,
	 input 				S3_PWRGD,
	 output reg 		CK505_EN = 0,
//	 output reg 		HIGHZ_N,
//	 input 				XDP_PRSNT_N,  // Asserted when ITP cable is plugged in
	 output reg 		LVDS_EN_N = 0,
	 input 				THERMTRIP,
	 input 				KBD_RST_N,
	 output reg 		WAKE_N = 0
   );
   
	parameter 			M0 = 3'h0;
	parameter 			M1 = 3'h1;
	parameter 			M2 = 3'h2;
	parameter 			M3 = 3'h3;
	parameter 			M5 = 3'h5;


	
	parameter 			IN_S5				= 5'b00000;
	parameter 			COLD_BOOT_0			= 5'b00010;
	parameter 			COLD_BOOT_1			= 5'b00110;
	parameter 			WAIT_FOR_BSEL		= 5'b00100;
	parameter 			COLD_BOOT_3			= 5'b01100;
	parameter 			COLD_BOOT_4			= 5'b01110;
	parameter 			COLD_BOOT_PWROK_0	= 5'b01010;
	parameter 			COLD_BOOT_6			= 5'b01000;
	parameter 			COLD_BOOT_7			= 5'b11000;
	parameter 			COLD_BOOT_8			= 5'b11010;
	parameter 			COLD_BOOT_9			= 5'b11110;
	parameter 			COLD_BOOT_10		= 5'b11100;
	parameter 			IN_S0				= 5'b10100;
	parameter 			S3_REQUEST_1		= 5'b10110;
	parameter 			S3_REQUEST_2		= 5'b10010;
	parameter 			S3_ENTRY_1			= 5'b10000;
	parameter 			IN_S3				= 5'b10001;
	parameter 			SHUTDOWN_REQUEST_1	= 5'b10011;
	parameter 			SHUTDOWN_REQUEST_2	= 5'b10111;
	parameter 			SHUTDOWN_REQUEST_3	= 5'b10101;
	parameter 			COLDRST_REQUEST_1	= 5'b11101;
	parameter 			COLDRST_REQUEST_2	= 5'b11111;
	parameter 			COLDRST_REQUEST_3	= 5'b11011;
	parameter 			IN_COLD_RESET		= 5'b11001;
	parameter 			WARMRST_REQUEST_1	= 5'b01001;
	parameter 			IN_WARM_RESET		= 5'b01011;
	parameter 			WARM_RESET_EXIT_1	= 5'b01111;
	parameter 			WARM_RESET_EXIT_2	= 5'b01101;
	
	
	
	
	// Delay values for each PWRMODE state
	//---------------------------------------
	//Delays are based of 25MHz clock (40ns)
	//parameter M0_DELAY = 32'h249F0;	// Delay 6ms (>5ms) in M0
	//parameter M1_DELAY = 32'h61A8 * 4'd5;  // Delay 1ms in M1
	//parameter M2_DELAY = 32'h61A8;	// Delay 1ms in M2
	//parameter M3_DELAY = 32'hF4240; // Delay 40ms in M3
	//parameter RTCRST_DELAY = 32'h384; //Delay 36us for RTCRST#
	//parameter RSMRST_DELAY = 32'h640; // Delay 64us for RTCRST# to RSMRST#
	//parameter RESET_DELAY = 32'h9C4; // Delay 100us for PWROK to RESET#
	//parameter PWROK_DELAY = 32'h9C4; //Delay 100us for RSMRST# to PWROK 
	//parameter WARM_RESET_DELAY = 32'h61A8; //Delay 1ms in warm reset.
	
	`define defT_50US 3'b010
        `define defT_PRIME   3'b011
        `define defT_1MS 3'b100
        `define defT_2MS 3'b101   
        `define defT_6MS 3'b000
        `define defT_3S  3'b001

	//1 CLK = 179ns (5.56MHz)
//	parameter [24:0] 	T_183US = 25'h0000400;  //183.3us
//	parameter [24:0] 	T_1MS	    = 25'h0002000; //1.466ms
	parameter [24:0] 	T_50US    = 25'h0000116;
//	parameter [24:0] 	T_500US   = 25'h0000adc;
	parameter [24:0] 	T_PRIME   = 25'd2777;  // This has to be a prime number. Approx 500US
	parameter [24:0] 	T_1MS     = 25'h00015b8;
	parameter [24:0] 	T_2MS     = 25'h0002b70;
	parameter [24:0] 	T_6MS     = 25'h0008250;
//	parameter [24:0] 	T_47MS    = 25'h0040000; //46.8ms
//	parameter [24:0] 	T_188MS    = 25'h0100000; //187.7ms
	parameter [24:0] 	T_3S      = 25'h1000000; //3S


	// Counter
	reg [25:0] 			cntr = 0;
	wire 					cntr_done;
		
	// Main state machine
	reg 					mgt_cntr_load;
	reg [2:0] 			mgt_cntr_value;
	
	reg [4:0] 			n_state;
	
	// Buttons
	reg 					pwr_off_event;
	reg 					pwr_up_event;
//	reg 					ext_wake_event;
	reg 					warm_rst_event;
	reg 					btn_cntr_load;
	reg [1:0] 			btn_cntr_value;
	reg 					rst_btn_n_d1, rst_btn_n_d2;
	reg 					pwr_btn_n_d1;
	reg 					pwr_btn_n_d2;
	wire 					pwr_btn_fedge, rst_btn_fedge;
	reg 					btn_n_state;
	parameter 			BTN_IDLE = 1'b0;
	parameter 			PWR_BTN_SELECT = 1'b1;
	reg [4:0] 			pwrok_lp_cntr = 0;

	reg                                     reset_warm_rst_cntr;
        `define      W_RST_CNTR_LEN             23
        reg [`W_RST_CNTR_LEN -1:0]              warm_rst_cntr;  

	
	// 5.56mhz 179ns
	// [1] = 358ns
	// [2] = 716ns
	// [3] = 1.432us
	// [4] = 2.864us
	// [5] = 5.728us
	// [6] = 11.45us
	// [7] = 22.91us
	// [8] = 45.82us
	// [9] = 91.65us
	//[10] = 183.3us
	//[11] = 366.6us
	//[12] = 733.2us
	//[13] = 1.466ms
	//[14] = 2.933ms
	//[15] = 5.865ms
	//[16] = 11.73ms
	//[17] = 23.46ms
	//[18] = 46.92ms
	//[19] = 93.84ms
	//[20] = 187.7ms
	//[21] = 375.4ms
	//[22] = 750.7ms
	//[23] = 1.501s
	//[24] = 3.003s
	//[25] = 6.006s
	//[26] = 12.01s

	
	// Push buttons behavior
	// RESET push button : Only valid during IN_S0.
	//                   : 1ms assertion triggers event.
	// PWR push button   : Only valid during IN_S0, IN_S3, IN_S5.
	//                   : 1st 1ms assertion triggers wake event during IN_S3.
	//                   : 1st 1ms assertion triggers pwr up during IN_S5.
	//                   : 2nd 4s assertion triggers pwr off during IN_S0.    
	
	// Double latch
	always @ (posedge CLK)
	  begin
		  rst_btn_n_d1 <= RST_BTN_N;
		  rst_btn_n_d2 <= rst_btn_n_d1;
		  pwr_btn_n_d1 <= PWR_BTN_N;
		  pwr_btn_n_d2 <= pwr_btn_n_d1;
	  end

	assign pwr_btn_fedge = !pwr_btn_n_d1 & pwr_btn_n_d2;
	assign rst_btn_fedge = !rst_btn_n_d1 & rst_btn_n_d2;

	always @ (posedge CLK)
	  begin
		  pwr_off_event <= 0;
		  pwr_up_event <= 0;
//		  ext_wake_event <= 0;
		  warm_rst_event <= 0;
		  btn_n_state <= BTN_IDLE;
		  WAKE_N <= 1;
		  
		  case (btn_n_state)
			 BTN_IDLE:
				begin
					if (n_state == IN_S0)
					  WAKE_N <= 1;
					else
					  WAKE_N <= WAKE_N;
										
					if ((!KBD_RST_N || rst_btn_fedge) && (n_state == IN_S0)) //Rst button, kbd_rst_n causes ext warm reset
					  begin
						  warm_rst_event <= 1;
						  btn_cntr_load <= 0;
						  btn_n_state <= btn_n_state;
					  end
					else if (pwr_btn_fedge) //Pwr button is pressed
					  begin
						  warm_rst_event <= 0;
						  case (n_state)
							 IN_S0: // If in S0, see if it's pressed for 3 seconds.
								begin
									btn_cntr_load <= 1;
									btn_cntr_value <= `defT_3S;
									btn_n_state <= PWR_BTN_SELECT;
								end
							 
							 IN_S3: // If in S3, wake the system
								begin
									btn_cntr_load <= 1;
									btn_cntr_value <= `defT_6MS;
									btn_n_state <= PWR_BTN_SELECT;
								end
							 																 
							 IN_S5: // If in S5, power up the system
								begin
									btn_cntr_load <= 1;
									btn_cntr_value <= `defT_6MS;
									btn_n_state <= PWR_BTN_SELECT;
								end

							 default:
								begin
									btn_cntr_load <= 0;
									btn_n_state <= BTN_IDLE;
								end
							 							 
						  endcase // case (n_state)
					  end // if (!pwr_btn_n_d1)
					else
					  begin
						  warm_rst_event <= 0;
						  btn_cntr_load <= 0;
						  btn_n_state <= btn_n_state;
					  end
				end // case: BTN_IDLE

			 PWR_BTN_SELECT:
				begin
					btn_cntr_load <= 0;
					pwr_off_event <= 0;
					pwr_up_event <= 0;
					//ext_wake_event <= 0;
					WAKE_N <= 1;
					
					if (cntr_done)
					  begin
						  case (n_state)
							 IN_S0: // Trigger a shutdown sequence
								begin
									pwr_off_event <= 1;
									btn_n_state <= BTN_IDLE;
								end
							 
							 IN_S3: // Trigger a wake event
								begin
									WAKE_N <= 0;  // Asserts WAKE# signal to TNC
									btn_n_state <= BTN_IDLE;
								end
							 																 
							 IN_S5: // Trigger a cold boot event
								begin
									pwr_up_event <= 1;
									btn_n_state <= BTN_IDLE;
								end

							 default:
								begin
									pwr_off_event <= 0;
									pwr_up_event <= 0;
									//ext_wake_event <= 0;
									btn_n_state <= BTN_IDLE;
								end
						  endcase // case (n_state)
					  end // if (cntr_done)
					else if (pwr_btn_n_d1)
					  btn_n_state <= BTN_IDLE;
					else
					  btn_n_state <= btn_n_state;
				end // case: PWR_BTN_SELECT
			 
			 default:
				begin
					pwr_off_event <= 0;
					pwr_up_event <= 0;
					//ext_wake_event <= 0;
					warm_rst_event <= 0;
					btn_cntr_value <= 25'h0;
					btn_cntr_load <= 0;
					btn_n_state <= BTN_IDLE;
				end
			 					
		  endcase // case (btn_n_state)
	  end // always @ (posedge CLK)
			 
	// Down counter.
	always @ (posedge CLK)
     begin
		  if (btn_cntr_load)
		    case (btn_cntr_value)
		      `defT_6MS : cntr[24:0] <= T_6MS;
		      `defT_3S  : cntr[24:0] <=  T_3S;
		      default   : cntr[24:0] <= 25'h0;
		    endcase // case (btn_cntr_value)
		  else if (mgt_cntr_load)
			 //cntr[24:0] <= mgt_cntr_value[24:0];
			case (mgt_cntr_value) 
			  `defT_50US  : cntr[24:0] <= T_50US;
			  `defT_PRIME : cntr[24:0] <= T_PRIME;
			  `defT_1MS   : cntr[24:0] <= T_1MS;
			  `defT_2MS   : cntr[24:0] <= T_2MS;
			  `defT_6MS   : cntr[24:0] <= T_6MS;
			  `defT_3S    : cntr[24:0] <= T_3S;
                          default    : cntr[24:0] <= 25'b0;
			endcase // case (mgt_cntr_value
		  else if (cntr != 26'h0000000)
 			 cntr[25:0] <= cntr[24:0] - 25'h0000001;
	  end // always @ (posedge CLK)

	assign cntr_done = (cntr == 25'h0000001);  // When reach 1, it's done.
	
	// Sleep state transition state machine
	always @ (posedge CLK)
	  begin
		  RSMRST_N <= RSMRST_N;
		  CPU_RST_N <= 0;
		  mgt_cntr_load <= 0;
		  mgt_cntr_value <= mgt_cntr_value;
                  reset_warm_rst_cntr <= 1'b1;      // -- Special Warm Reset Counter is reset and disabled.
		  
		  case (n_state)
			 // Cold reset/boot sequence. Turn on S3 rails.
			 COLD_BOOT_0:
				begin
					SLP_S5_N <= 1;  // Turn on S3 rails
					SLP_S3_N <= 0;
					RSTWARN <= 0;  // Turn off RSTWARN in S5
					CPU_RST_N <= 0;
					CK505_EN <= 0;
					PWRMODE <= M0;
					RSMRST_N <= 0;
//					HIGHZ_N <= 0;
					LVDS_EN_N <= 1;
										
					// When S3 rails are up, board is in S3
					if (S3_PWRGD && RSTRDY_N && !SLPMODE && SLPRDY_N)
					  begin
						  mgt_cntr_load <= 1;
						  mgt_cntr_value <= `defT_6MS;  // Assert RSMRST# for >5ms
						  n_state <= COLD_BOOT_1;
					  end
					else
					  n_state <= COLD_BOOT_0;
				end

			 // De-asserts RSMRST_N
			 COLD_BOOT_1:
				begin
					SLP_S5_N <= 1;
					SLP_S3_N <= 0;
					RSTWARN <= 1;
					CPU_RST_N <= 0;
					CK505_EN <= 0;
					PWRMODE <= M0;
					RSMRST_N <= 0;
//					HIGHZ_N <= 0;
					LVDS_EN_N <= 1;

					if (cntr_done)
					  begin
						  RSMRST_N <= 1;
						  n_state <= IN_S3;
					  end
					else
					  n_state <= n_state;
				end // case: COLD_BOOT_1
			 
			 
			 // In S3 state. SUSCLK should be toggling by now. 
			 // Wait for wake event or continue boot if it's in middle of cold boot.			 
			 IN_S3:
				begin
					SLP_S5_N <= 1;
					RSTWARN <= 1;
					CPU_RST_N <= 0;
					RSMRST_N <= 1;
					CK505_EN <= 0;
					PWRMODE <= M0;
					SLP_S3_N <= 0;
//					HIGHZ_N <= 0;
					LVDS_EN_N <= 1;
										
					// Although no delay here, but S0 VRs ramp should be >5ms
					if (SLPRDY_N)
					  n_state <= COLD_BOOT_3;
					else
					  n_state <= n_state;
				end
			 			 
			 // Turn on S0 rails. Goes to M2.
			 COLD_BOOT_3:
				begin
					SLP_S5_N <= 1;
					SLP_S3_N <= 1;
					RSTWARN <= 1;
					CPU_RST_N <= 0;
					RSMRST_N <= 1;
					CK505_EN <= 0;
//					HIGHZ_N <= 0;
					LVDS_EN_N <= 1;
										
					if (S0_PWRGD)
					  begin
						  PWRMODE <= M2;   // TC will sense fuse
						  mgt_cntr_load <= 1;
						  mgt_cntr_value <= `defT_6MS;  // To hold M2 for >10us
						  n_state <= COLD_BOOT_4;
					  end
					else
					  n_state <= n_state;
					
				end // case: COLD_BOOT_3
			 				
			 // Goes to M3.
			 COLD_BOOT_4:
				begin
					SLP_S5_N <= 1;
					SLP_S3_N <= 1;
					RSTWARN <= 1;
					CPU_RST_N <= 0;
					RSMRST_N <= 1;
					CK505_EN <= 0;
//					HIGHZ_N <= 0;
					LVDS_EN_N <= 1;
																		
					if (cntr_done)
					  begin
						  PWRMODE <= M3;   // M2 -> M3. M3 will have valid BSEL.
						  mgt_cntr_load <= 1;
						  mgt_cntr_value <= `defT_6MS; // Wait for 6ms to get valid BSEL
						  n_state <= WAIT_FOR_BSEL;
					  end
					else
					  begin
						  PWRMODE <= M2;
						  n_state <= COLD_BOOT_4;
					  end // else: !if(cntr_done)
				end // case: COLD_BOOT_4

			 // Turns on Main Clock
			 WAIT_FOR_BSEL:  // Enters M3 here
				begin
					SLP_S5_N <= 1;
					SLP_S3_N <= 1;
					RSTWARN <= 1;
					CPU_RST_N <= 0;
					RSMRST_N <= 1;
					PWRMODE <= M3;
//					HIGHZ_N <= 0;
					LVDS_EN_N <= 1;

					if (cntr_done)
					  begin
						  CK505_EN <= 1;
						  mgt_cntr_load <= 1;
						  mgt_cntr_value <= `defT_6MS; // Wait CK505 to lock
						  n_state <= COLD_BOOT_6;
						  
//						  if (XDP_PRSNT_N)
//							 n_state <= COLD_BOOT_6;  
//						  else
//							 n_state <= COLD_BOOT_6A; // Branches off to a different sequence for ITP boot
					  end
					else
					  begin
						  CK505_EN <= 0;
						  n_state <= WAIT_FOR_BSEL;
					  end // else: !if(cntr_done)
				end // case: WAIT_FOR_BSEL

			 ////////////////////// Alternate sequence for ITP boot /////////////////////
			 // Goes to M1.
/*			 COLD_BOOT_6A:
				begin
					SLP_S5_N <= 1;
					SLP_S3_N <= 1;
					CPU_RST_N <= 0;
					RSMRST_N <= 1;
					CK505_EN <= 1;
					PWROK <= 0;
					RSTWARN <= 1;
//					HIGHZ_N <= 0;
					LVDS_EN_N <= 1;
										
					if (cntr_done)
					  begin
						  PWRMODE <= M1; // Enters M1 here.
						  mgt_cntr_load <= 1;
						  mgt_cntr_value <= T_6MS;  // Suppose to be 250us
						  n_state <= COLD_BOOT_7A;
					  end
					else
					  begin
						  PWRMODE <= M3;
						  n_state <= n_state;
					  end
				end // case: COLD_BOOT_6

			 // Goes to M5
			 COLD_BOOT_7A:
				begin
					SLP_S5_N <= 1;
					SLP_S3_N <= 1;
					RSTWARN <= 1;
					CPU_RST_N <= 0;
					RSMRST_N <= 1;
					CK505_EN <= 1;
					PWROK <= 0;
//					HIGHZ_N <= 0;
					LVDS_EN_N <= 1;
					
					if (cntr_done)
					  begin
						  PWRMODE <= M5;  // Enters M5 here.
						  mgt_cntr_load <= 1;
						  mgt_cntr_value <= T_6MS;  
						  n_state <= COLD_BOOT_8A;
					  end
					else
					  begin
						  PWRMODE <= M1;
						  n_state <= n_state;
					  end // else: !if(cntr_done)
				end

			 // Holds at M5. Wait for button press.
			 // User performs HPLL_ENABLE fix here.
			 COLD_BOOT_8A:
				begin
					SLP_S5_N <= 1;
					SLP_S3_N <= 1;
					RSTWARN <= 0;
					CPU_RST_N <= 0;
					RSMRST_N <= 1;
					CK505_EN <= 1;
					PWRMODE <= M5;
//					HIGHZ_N <= 1;   // HIGHZ# deasserts here!
					LVDS_EN_N <= 1;
										
					if (rst_btn_fedge) // Button pressed.
					  begin
						  PWROK <= 1;   // PWROK asserts here!
						  RSTWARN <= 0;  // RSTWARN deasserts here!
						  mgt_cntr_load <= 1;
						  mgt_cntr_value <= T_6MS;  // Suppose to be 250us
						  n_state <= COLD_BOOT_9A;
					  end
					else
					  begin
						  PWROK <= 0;
						  RSTWARN <= 1;
						  n_state <= n_state;
					  end
				end 

			 // Release CPU reset signal.
			 COLD_BOOT_9A:
				begin
					SLP_S5_N <= 1;
					SLP_S3_N <= 1;
					RSTWARN <= 0;
					RSMRST_N <= 1;
					CK505_EN <= 1;
					PWRMODE <= M5;
					PWROK <= 1;
//					HIGHZ_N <= 1;
					LVDS_EN_N <= 1;
					
					if (cntr_done)
					  begin
						  CPU_RST_N <= 1;  // RESET# deasserted here!
						  n_state <= IN_S0;
					  end
					else
					  begin
						  CPU_RST_N <= 0;
						  n_state <= n_state;
					  end
				end // case: COLD_BOOT_9A
*/
			 /////////////////////// Ends Alternate ITP boot sequence ////////////////////////////

			 // Asserts PWROK
			 COLD_BOOT_6:
				begin
					SLP_S5_N <= 1;
					SLP_S3_N <= 1;
					RSTWARN <= 1;
					CPU_RST_N <= 0;
					RSMRST_N <= 1;
					CK505_EN <= 1;
					PWRMODE <= M3;
//					HIGHZ_N <= 1;
					LVDS_EN_N <= 1;
										
					if (cntr_done)
					  begin
						  if (pwrok_lp_cntr == 5'd10) // After PWROK loops 10 times
							 begin
								 PWROK <= 1;
								 n_state <= COLD_BOOT_7;
							 end
						  else
							 begin
								 PWROK <= 1;    // Overlay will capture strap values
								 mgt_cntr_load <= 1;
								 mgt_cntr_value <= `defT_PRIME;  // Hold PWROK high for 500us
								 n_state <= COLD_BOOT_PWROK_0;
							 end // else: !if(pwrok_lp_cntr == 4'd11)
					  end
					else
					  begin
						  PWROK <= 0;
						  n_state <= COLD_BOOT_6;
					  end
											
				end // case: COLD_BOOT_6

			 COLD_BOOT_PWROK_0:
				begin
					SLP_S5_N <= 1;
					SLP_S3_N <= 1;
					RSTWARN <= 1;
					CPU_RST_N <= 0;
					RSMRST_N <= 1;
					CK505_EN <= 1;
					PWRMODE <= M3;
					LVDS_EN_N <= 1;

					if (cntr_done)
					  begin
						  PWROK <= 0;
						  mgt_cntr_load <= 1;
						  mgt_cntr_value <= `defT_50US;  // Hold PWROK low for 50us
						  pwrok_lp_cntr[4:0] <= pwrok_lp_cntr[3:0] + 1'b1;
						  n_state <= COLD_BOOT_6;
					  end
					else
					  begin
						  PWROK <= 1;
						  n_state <= COLD_BOOT_PWROK_0;
					  end // else: !if(cntr_done)
				end // case: COLD_BOOT_PWROK_0
					

			 // Deasserts RSTWARN
			 COLD_BOOT_7:
				begin
					SLP_S5_N <= 1;
					SLP_S3_N <= 1;
					RSTWARN <= 0; // This is deasserted.
					CPU_RST_N <= 0;
					RSMRST_N <= 1;
					CK505_EN <= 1;
					PWRMODE <= M3;
//					HIGHZ_N <= 1;
					LVDS_EN_N <= 1;
					mgt_cntr_load <= 1;
					mgt_cntr_value <= `defT_2MS;  // Suppose to be 250us
					n_state <= COLD_BOOT_8;
					pwrok_lp_cntr <= 5'h0;
				end
			 
			 // Release CPU reset		
			 COLD_BOOT_8:
				begin
					SLP_S5_N <= 1;
					SLP_S3_N <= 1;
					RSTWARN <= 0;
					CPU_RST_N <= 0;
					RSMRST_N <= 1;
					CK505_EN <= 1;
					PWRMODE <= M3;
//					HIGHZ_N <= 1;
					LVDS_EN_N <= 1;
					
					if (cntr_done)
					  begin
						  CPU_RST_N <= 1;   // CPU reset pin is released!
						  mgt_cntr_load <= 1;
						  mgt_cntr_value <= `defT_1MS;
						  n_state <= COLD_BOOT_9;
					  end
					else
					  begin
						  CPU_RST_N <= 0;
						  n_state <= COLD_BOOT_8;
					  end
				end // case: COLD_BOOT_8

			 // Goes to M1
			 COLD_BOOT_9: 
				begin
					SLP_S5_N <= 1;
					SLP_S3_N <= 1;
					RSTWARN <= 0;
					CPU_RST_N <= 1;
					RSMRST_N <= 1;
					CK505_EN <= 1;
					PWRMODE <= M1;
//					HIGHZ_N <= 1;
					LVDS_EN_N <= 1;
					mgt_cntr_load <= 1;			
					mgt_cntr_value <= `defT_6MS;  // Hold M1 for 1ms
					n_state <= COLD_BOOT_10;
				end // case: COLD_BOOT_9

			 // Goes to M5
			 COLD_BOOT_10:
				begin
					SLP_S5_N <= 1;
					SLP_S3_N <= 1;
					RSTWARN <= 0;
					CPU_RST_N <= 1;
					RSMRST_N <= 1;
					CK505_EN <= 1;
//					HIGHZ_N <= 1;
					LVDS_EN_N <= 0;   // LVDS controls enabled!
					
					if (cntr_done)
					  begin
						  PWRMODE <= M5;
						  n_state <= IN_S0;
					  end
					else
					  begin
						  PWRMODE <= M1;
						  n_state <= COLD_BOOT_10;
					  end
				end
			 
			 // In S0 state. Waits for events.
			 IN_S0:  
				begin
	        	                reset_warm_rst_cntr <= 1'b0;      // -- Special Warm Reset Counter is enabled.
					CPU_RST_N <= 1;
					SLP_S5_N <= 1;
					SLP_S3_N <= 1;
					RSTWARN <= 0;
					CK505_EN <= 1;
					PWRMODE <= M5;
//					HIGHZ_N <= 1;
					LVDS_EN_N <= 0;   // LVDS controls enabled!
										
					if ((!SLPRDY_N && !RSTRDY_N && !SLPMODE) || pwr_off_event || THERMTRIP) // Cat shutdown
					  begin
						  RSTWARN <= 1;
						  n_state <= SHUTDOWN_REQUEST_1;
					  end
					else if (!RSTRDY_N && !SLPMODE && SLPRDY_N) // Cold reset
					  begin
						  RSTWARN <= 1;
						  n_state <= COLDRST_REQUEST_1;
					  end
					else if ((!RSTRDY_N && SLPMODE && SLPRDY_N) || warm_rst_event)  // Warm reset 
					  begin
						  RSTWARN <= 1;
                                	          if ( warm_rst_cntr[`W_RST_CNTR_LEN -2:`W_RST_CNTR_LEN -3] == 2'b11)
				         	       n_state <= WARMRST_REQUEST_1;
        					  else
					               n_state <= IN_S3;
					  end
					else if (!SLPRDY_N && RSTRDY_N) // TNC wants to go to sleep
					  begin
						  RSTWARN <= 1;
						  n_state <= S3_REQUEST_1;
					  end
					else
					  begin
						  RSTWARN <= 0;
						  n_state <= n_state;
					  end
				end // case: IN_S0

			 // Do S3 entry
			 S3_REQUEST_1:
				begin
					CPU_RST_N <= 1;
					SLP_S5_N <= 1;
					SLP_S3_N <= 1;
					RSTWARN <= 1;
 					CK505_EN <= 1;
					PWRMODE <= M5;
					LVDS_EN_N <= 1;
					
					if (!RSTRDY_N) 
					  begin
						  CPU_RST_N <= 0;
						  PWRMODE <= M1;
						  mgt_cntr_load <= 1;
						  mgt_cntr_value <= `defT_6MS;
						  n_state <= S3_REQUEST_2;
					  end
					else
					  n_state <= n_state;
				end
			 					
			 S3_REQUEST_2:
				begin 
					SLP_S5_N <= 1;
					SLP_S3_N <= 1;
					RSTWARN <= 1;
					CK505_EN <= 1;
					PWRMODE <= M1;
					
					if (cntr_done) 
					  begin
						  PWRMODE <= M0;
						  PWROK <= 0;
						  SLP_S3_N <= 0;
						  LVDS_EN_N <= 1;
						  CK505_EN <= 0;
						  mgt_cntr_load <= 1;
//						  mgt_cntr_value <= T_1MS;
						  mgt_cntr_value <= `defT_6MS;
						  n_state <= S3_ENTRY_1;
					  end
					else
					  begin
						  n_state <= S3_REQUEST_2;
						  LVDS_EN_N <= 0;
					  end // else: !if(cntr_done)
				end // case: S3_REQUEST_2
			 
			 // Almost in S3 entry.
			 S3_ENTRY_1:
				begin
					SLP_S5_N <= 1;
					SLP_S3_N <= 0;
					RSTWARN <= 1;
					CK505_EN <= 0;
					PWRMODE <= M0;
					LVDS_EN_N <= 1;
					
					if (cntr_done)
					  begin
						  if (SLPMODE) // TNC wants to go to S3
							 n_state <= IN_S3;
						  else // TNS wants to go to S5
							 begin
								 mgt_cntr_load <= 1;
//								 mgt_cntr_value <= T_1MS;  
								 mgt_cntr_value <= `defT_6MS;  
								 n_state <= SHUTDOWN_REQUEST_3;
							 end
					  end // if (cntr_done)
					else
					  n_state <= n_state;
				end
			 
			 // Catastrophic shutdown
			 SHUTDOWN_REQUEST_1:
				begin
					SLP_S5_N <= 1;
					SLP_S3_N <= 1;
					RSTWARN <= 1;
					CK505_EN <= 1;
					CPU_RST_N <= 0;
					PWRMODE <= M1;
					LVDS_EN_N <= 0;
					n_state <= SHUTDOWN_REQUEST_2;
				end
									
			 SHUTDOWN_REQUEST_2:
				begin
					SLP_S5_N <= 1;
					SLP_S3_N <= 0;  // Shutting down rails
					RSTWARN <= 1;
					CK505_EN <= 0;
					PWRMODE <= M0;
					PWROK <= 0;
					LVDS_EN_N <= 1;
					n_state <= SHUTDOWN_REQUEST_3;
				end // case: SHUTDOWN_REQUEST_2
			 
			 SHUTDOWN_REQUEST_3:
				begin 
					SLP_S5_N <= 0;
					SLP_S3_N <= 0;
					RSTWARN <= 1;
					CK505_EN <= 0;
					PWRMODE <= M0;
					RSMRST_N <= 0;
					LVDS_EN_N <= 1;
					n_state <= IN_S5;
				end
	
			 IN_S5:  // CPLD starts in this state.
				begin
					CPU_RST_N <= 0;
					SLP_S5_N <= 0;
					SLP_S3_N <= 0;
					PWRMODE <= M0;
					RSTWARN <= 0;  // Turn off RSTWARN in S5
					CK505_EN <= 0;
//					HIGHZ_N <= 0;
					LVDS_EN_N <= 1;
					
					if (pwr_up_event)
					  n_state <= COLD_BOOT_0;
					else
					  n_state <= n_state;
				end
			 
			 
			 COLDRST_REQUEST_1: 
				begin
					SLP_S5_N <= 1;
					SLP_S3_N <= 1;
					CK505_EN <= 1;
					CPU_RST_N <= 0;
					RSTWARN <= 1;
					PWRMODE <= M1;
					LVDS_EN_N <= 0;
					mgt_cntr_load <= 1;
					mgt_cntr_value <= `defT_6MS; //Delay to PWROK deassertion
					n_state <= COLDRST_REQUEST_2;
				end
				
			 COLDRST_REQUEST_2: 
				begin
					SLP_S5_N <= 1;
					SLP_S3_N <= 1;
					RSTWARN <= 1;
					CK505_EN <= 1;
					PWRMODE <= M1;
					LVDS_EN_N <= 0;					
					if (cntr_done) 
					  begin
						  PWROK <= 0;
						  PWRMODE <= M0;
						  SLP_S3_N <= 0;
						  CK505_EN <= 0;
						  mgt_cntr_load <= 1;
//						  mgt_cntr_value <= T_1MS;
						  mgt_cntr_value <= `defT_6MS;
						  n_state <= COLDRST_REQUEST_3;
					  end
					else
					  n_state <= n_state;
				end
			 
			 COLDRST_REQUEST_3:
				begin
					SLP_S5_N <= 1;
					SLP_S3_N <= 0;
					RSTWARN <= 1;
					CK505_EN <= 0;
					PWRMODE <= M0;
					LVDS_EN_N <= 1;
					if (cntr_done)
					  begin
						  RSMRST_N <= 0;
						  SLP_S5_N <= 0;							 
						  mgt_cntr_load <= 1;
						  mgt_cntr_value <= `defT_3S;
						  n_state <= IN_COLD_RESET;
					  end
					else
					  n_state <= n_state;
				  end

			 // Momentarily in cold reset.
			 IN_COLD_RESET:
				begin
					SLP_S5_N <= 0;
					SLP_S3_N <= 0;
					RSTWARN <= 0;  // Turn off RSTWARN in S5
					CK505_EN <= 0;
					PWRMODE <= M0;
					LVDS_EN_N <= 1;
					if (cntr_done)
					  begin
						  n_state <= COLD_BOOT_0;
					  end
					else
					  begin
						  mgt_cntr_load <= 0;
						  mgt_cntr_value <= 25'h0;
						  n_state <= n_state;
					  end
				end // case: IN_COLD_RESET
			 

			 // Performing warm reset. But this is not the correct sequence as it ignores SLPMODE, SLPRDY#
			 // Sighting is still under debug.
			 WARMRST_REQUEST_1:
				begin
					CPU_RST_N <= 0;  // CPU RESET asserted! 
					SLP_S5_N <= 1;
					SLP_S3_N <= 1;
					RSTWARN <= 1;
					CK505_EN <= 1;
					PWRMODE <= M1;  // M5 -> M1
					LVDS_EN_N <= 0;
					mgt_cntr_load <= 1;
					mgt_cntr_value <= `defT_6MS; //Delay in WARM reset
					n_state <= IN_WARM_RESET;
				end

			 IN_WARM_RESET:
				begin
					SLP_S5_N <= 1;
					SLP_S3_N <= 1;
 					RSTWARN <= 1;   
					CK505_EN <= 1;
					CPU_RST_N <= 0;
					PWRMODE <= M1;
					LVDS_EN_N <= 1;
					if (cntr_done) 
					  begin
						  RSTWARN <= 0;   // RSTWARN deasserted!
						  n_state <= WARM_RESET_EXIT_1;
					  end
					else
					  n_state <= n_state;
				end // case: IN_WARM_RESET
			 
			 WARM_RESET_EXIT_1: 
				begin
					SLP_S5_N <= 1;
					SLP_S3_N <= 1;
					RSTWARN <= 0;
					CK505_EN <= 1;
					PWRMODE <= M1;
					LVDS_EN_N <= 1;

					if (RSTRDY_N && !SLPMODE)
					  begin
						  CPU_RST_N <= 1;
						  mgt_cntr_load <= 1;
						  mgt_cntr_value <= `defT_6MS;
						  n_state <= WARM_RESET_EXIT_2;
					  end
					else
					  n_state <= n_state;
				end // case: WARM_RESET_EXIT_1
			 
			 					
			 WARM_RESET_EXIT_2: 
				begin
					CPU_RST_N <= 1;
					SLP_S5_N <= 1;
					SLP_S3_N <= 1;
					RSTWARN <= 0;
					CK505_EN <= 1;
					PWRMODE <= M1;
					LVDS_EN_N <= 1;
					
					if (cntr_done) 
					  begin
						  PWRMODE <= M5;  // M1 -> M5
						  n_state <= IN_S0;
					  end
					else
					  n_state <= n_state;
				end
			 
			 default:
				begin
					SLP_S5_N <= 0;
					SLP_S3_N <= 0;
					RSTWARN <= RSTWARN;
					CK505_EN <= 0;
					n_state <= IN_S5;
					CPU_RST_N <= 0;
					mgt_cntr_load <= 0;
					mgt_cntr_value <= 3'h0;
				end
			 
		  endcase // case (n_state)
	  end // always @ (posedge CLK)

	always @ (posedge CLK)
	  begin
             if ( reset_warm_rst_cntr )
	       warm_rst_cntr[`W_RST_CNTR_LEN -1:0] <= `W_RST_CNTR_LEN'b0;
	     else
	       if (warm_rst_cntr[`W_RST_CNTR_LEN -2:`W_RST_CNTR_LEN -3] != 2'b11)
		 warm_rst_cntr[`W_RST_CNTR_LEN -1:0] <= warm_rst_cntr[`W_RST_CNTR_LEN -2:0] + 1'b1;
	     else
		 warm_rst_cntr <= warm_rst_cntr ;
	  end
					
endmodule
	
	
