param (
    [Parameter(Mandatory = $true)][String[]] $MPSAddress,
    [Parameter(Mandatory = $true)][String[]] $MPSSOAPPort
)

function Invoke-SOAPCommand
(
    [String] $URL,
    [String] $SOAPAction,
    [Xml] $SOAPPayload
)

{
    <#
    .SYNOPSIS
    Invoke a SOAP action against a service URL and receive any response
    #>


    # Write-Host "Sending SOAP request to $URL..."
    $soapWebRequest = [System.Net.WebRequest]::Create($URL)
    $soapWebRequest.Headers.Add("SOAPAction", $SOAPAction)

    $soapWebRequest.ContentType = "text/xml;charset=`"utf-8`""
    $soapWebRequest.Accept = "text/xml"
    $soapWebRequest.Method = "POST"

    try
    {
        # Write-Host "Initiating send..."
        $requestStream = $soapWebRequest.GetRequestStream()

        $SOAPPayload.Save($requestStream)
        $requestStream.Close()

        # Write-Host "Send complete, waiting for response..."
        $response = $soapWebRequest.GetResponse()
        $responseStream = $response.GetResponseStream()
        $soapReader = [System.IO.StreamReader]($responseStream)
        $returnXml = [Xml] $soapReader.ReadToEnd()
        $responseStream.Close()

        # Write-Host "Response received"

        return ($returnXml)
    }
    catch
    {
        # Write non-terminating error
        Write-Error "Error invoking SOAP command to $URL"

        return ($null)
    }
}


function Get-AMTFCFHConnection
(
    [String[]] $MPSAddress,
    [String] $MPSSOAPPort,
    [String] $Username,
    [String] $Password
)

{
    <#
    .SYNOPSIS
    Enumerate Intel vPro clients currently connected to one or more Intel vPro Gateways
    #>


    # Initialize results
    $Results = @()


    foreach ($MPS in $MPSAddress)
    {
        $MPSSOAPURL = "http://" + $MPS + ":" + $MPSSOAPPort + "/"

        # Construct SOAP command and payload
        $SOAPAction = "`"http://schemas.intel.com/platform/client/MPSInterface/2010/11/EnumerateConnectedMachines`""

        [Xml] $SOAPPayload = "<?xml version=`"1.0`" encoding=`"utf-8`"?><soap:Envelope xmlns:soap=`"http://schemas.xmlsoap.org/soap/envelope/`" xmlns:xsi=`"http://www.w3.org/2001/XMLSchema-instance`" xmlns:xsd=`"http://www.w3.org/2001/XMLSchema`"><soap:Body><EnumerateConnectedMachines xmlns=`"http://schemas.intel.com/platform/client/MPSInterface/2010/11`" /></soap:Body></soap:Envelope>"


        $returnXML = Invoke-SOAPCommand $MPSSOAPURL $SOAPAction $SOAPPayload

        if ($returnXML -ne $null)
        {
            $Expires = $returnXML.Envelope.Body.EnumerateConnectedMachinesResponse.Expires.ToString()
            $EnumerationContext = $returnXML.Envelope.Body.EnumerateConnectedMachinesResponse.EnumerationContext."#text".ToString()

            # Construct SOAP command and payload
            $SOAPAction = "`"http://schemas.intel.com/platform/client/MPSInterface/2010/11/Pull`""

            [Xml] $SOAPPayload = "<?xml version=`"1.0`" encoding=`"utf-8`"?><soap:Envelope xmlns:soap=`"http://schemas.xmlsoap.org/soap/envelope/`" xmlns:xsi=`"http://www.w3.org/2001/XMLSchema-instance`" xmlns:xsd=`"http://www.w3.org/2001/XMLSchema`"><soap:Body><Pull xmlns=`"http://schemas.intel.com/platform/client/MPSInterface/2010/11`"><EnumerationContext xmlns=`"http://schemas.intel.com/platform/client/MPSInterface/2010/11`">" + $EnumerationContext + "</EnumerationContext><MaxElements>1</MaxElements></Pull></soap:Body></soap:Envelope>"


            do
            {
                $returnXML = Invoke-SOAPCommand $MPSSOAPURL $SOAPAction $SOAPPayload

                $items = $returnXML.Envelope.Body.PullResponse.items

                if ($items -ne $null)
                {
                    $systemName = $returnXML.Envelope.Body.PullResponse.Items.SystemName.ToString()

                    # Create results object
                    $resultsObject = New-Object PSObject

                    # Add Computer and MPS members to result object
                    $resultsObject | Add-Member -MemberType NoteProperty -Name SystemName -Value $systemName
                    $resultsObject | Add-Member -MemberType NoteProperty -Name MPS -Value $MPS

                    $Results += $resultsObject
                }
            }
            while ($returnXML.Envelope.Body.PullResponse.endofsequence -eq $null)
        }
    }


    # Write results
    Write-Output $Results
}



Get-AMTFCFHConnection $MPSAddress $MPSSOAPPort | Format-Table
