/******************************************************************************
*
*                   INTEL CORPORATION PROPRIETARY INFORMATION
*       This software is supplied under the terms of a license agreement or
*       nondisclosure agreement with Intel Corporation and may not be copied
*       or disclosed except in accordance with the terms of that agreement.
*
*            Copyright (c) 2008-2009 Intel Corporation. All Rights Reserved.
*
*       All rights reserved.  No part of this program or publication may be
*       reproduced, transmitted, transcribed, stored in a retrieval system,
*       or translated into any language or computer language, in any form or
*       by any means, electronic, mechanical, magnetic, optical, chemical,
*       manual, or otherwise, without the prior written permission of Intel
*       Corporation.
*
*
*******************************************************************************/
#ifndef __CR_CSTRING_H_
#define __CR_CSTRING_H_

#include <stdlib.h>
#include <string.h>

extern "C" {
static size_t strnlen(const char *c, const size_t s) {
      return strlen(c);
}
}

namespace osal_stl {

#define NULL 0

class cr_cstring {

public:
    const static int MAX_CR_STRING_LEN=1073741820;
    const static int MIN_CR_STRING_LEN=16;
    const static unsigned int NPOS_LENGTH=4294967295U;

	typedef char value_type;
	typedef value_type *pointer;
	typedef const value_type *const_pointer;
	typedef value_type & reference;
	typedef const value_type & const_reference;
	typedef size_t size_type;
	typedef size_t difference_type;

	typedef const value_type *const_iterator;
	typedef value_type* iterator;

	typedef iterator InputIterator;

	typedef class reverse_iterator {
    public:
        typedef  cr_cstring::iterator        iterator;
        typedef  const iterator              const_iterator;
        typedef  cr_cstring::value_type&     reference;
		typedef  const cr_cstring::value_type& const_reference;
        typedef  cr_cstring::size_type       size_type;
        typedef  cr_cstring::difference_type difference_type;

        CR_NO_THROW reverse_iterator() {}

        explicit reverse_iterator(iterator& itr) : i(itr) {}

        CR_NO_THROW reverse_iterator(const reverse_iterator& ritr) {
            const iterator itr = ritr.get_itr();
            i = itr;
        }

        CR_NO_THROW ~reverse_iterator() {
        }

        CR_NO_THROW inline reverse_iterator& operator= (const reverse_iterator& ritr) {
            const iterator itr = ritr.get_itr();
			if(itr != i)
				i = itr;
            return *this;
        }
        
        CR_NO_THROW inline reference operator*() const {
            iterator itr = i;
            return *(--itr);
        }

        CR_NO_THROW inline reverse_iterator& operator++() {
            --i;
            return *this;
        }

        CR_NO_THROW inline reverse_iterator operator++(int) {
            reverse_iterator tmp = *this;
            --i;
            return tmp;
        }

        CR_NO_THROW inline reverse_iterator& operator--() {
            ++i;
            return *this;
        }

        CR_NO_THROW inline reverse_iterator operator--(int) {
            reverse_iterator tmp = *this;
            ++i;
            return tmp;
        }

        CR_NO_THROW inline reverse_iterator operator+(difference_type n) const {
            iterator tmp = (i - n);
            return reverse_iterator(tmp); 
        }

        CR_NO_THROW inline reverse_iterator& operator+=(difference_type n) {
            i -= n;
            return *this;
        }

        CR_NO_THROW inline reverse_iterator operator-(difference_type n) const {
            iterator tmp = (i+n);
            return reverse_iterator(tmp);
        }

        CR_NO_THROW inline reverse_iterator& operator-=(difference_type n) {
            i += n;
            return *this;
        }

        CR_NO_THROW inline reference operator[](difference_type n) const {
            return *(i+n);
        }

        CR_NO_THROW inline const iterator get_itr(void) const {
            return (const iterator) i;
        }

    private:
        iterator i;
    }reverse_iterator;

	typedef const reverse_iterator const_reverse_iterator;

	static const size_type npos = (size_type) cr_cstring::NPOS_LENGTH;

	/* contructor */
	 CR_NO_THROW cr_cstring() {
		 //printf("cr_cstring: calling default constructor.\n");
	    c_string = NULL;
	    aloc_capacity = 0;
        copyable = false;
	}			//default constructor

	/* contructor */ CR_NO_THROW cr_cstring(const char c) {
		//printf("cr_cstring: const char c constructor\n");
	    c_string = NULL;
	    aloc_capacity = 0;
        copyable = false;
	    *this = c;
	}			//end of constructor

	/* constructor */
    CR_NO_THROW cr_cstring(const cr_cstring & s, size_t pos = 0, size_t n = npos)
	{
		//printf("cr_cstring: const cr_cstring & s = %s, size_t pos = 0, size_t n = npos constructor.\n", s.c_str());
	    c_string = NULL;
	    aloc_capacity = 0;
	    const char *t = s.c_str();
        copyable = false;
	    cat_str(t, pos, n);
	}

	/* constructor */
	CR_NO_THROW cr_cstring(const char *s, size_t pos = 0, size_t n = npos) {
		//printf("cr_cstring: const char *s = %s, size_t pos = 0, size_t n = npos constructor\n", s);
	    c_string = NULL;
	    aloc_capacity = 0;
        copyable = false;
	    cat_str(s, pos, n);
	}

	/* operator += overloading */
	CR_NO_THROW inline cr_cstring & operator+=(const cr_cstring & cstr1) {
	    cat_str(cstr1.c_str(), 0, cstr1.length());
	    return *this;
	}			//end of += overloading

	/* operator += overloading */
	CR_NO_THROW inline cr_cstring & operator+=(const char *cstr) {
	    cat_str(cstr, 0, npos);
	    return *this;
	}			//end of += overloading

	/* operator += overloading */
	CR_NO_THROW inline cr_cstring & operator+=(const char c) {
	    char t[4] = { 0 };
	    t[0] = c;
	    cat_str(t, 0, 1);
	    return *this;
	}			//end of += overloading

	/* operator = overloading */
	CR_NO_THROW inline cr_cstring & operator=(char c) {
	    if (c_string)
		c_string[0] = 0;

	    char t[4] = { 0 };
	    t[0] = c;
	    cat_str(t, 0, npos);

	    return *this;
	}			//end of = overloading

	/* operator = overloading */
	CR_NO_THROW inline cr_cstring & operator=(const char* cstr) {
		//printf("%s function called.\n", __FUNCTION__);
	    if (c_string)
		c_string[0] = 0;

	    cat_str(cstr, 0, npos);

	    return *this;
	}			//end of = overloading

	/* operator = overloading */
	CR_NO_THROW inline cr_cstring & operator=(const cr_cstring& cstr) {
        if (cstr.get_copyable()) {

            const char* tmp = cstr.c_str();
            this->set_string(tmp);

        } else  {
			//check if the pointer being assigned to is not the same pointer
			if( this != &cstr)
			{
				if (c_string)
					c_string[0] = 0;

				cat_str(cstr.c_str(), 0, npos);
			}
		}

	    return *this;
	}			//end of = overloading

	CR_NO_THROW inline reference operator[] (size_t n) {
	    return at(n);
	}

	CR_NO_THROW inline const_reference operator[] (size_type n) const {
		const_iterator ci = begin();
		if (n < length())
			return (const_reference) *(ci + n);

		return (const_reference)*ci;
	}

	CR_NO_THROW ~cr_cstring() {
		//printf("In cr_cstring destructor:\n");
	    if (c_string) {
            /* if its used then dont free */
            if (!copyable)
			{
				//printf("In cr_cstring destructor: freeing string.\n");
                free(c_string);
			}

            c_string = NULL;
            aloc_capacity = 0;
	    }
	}

	CR_NO_THROW inline reference at(size_type n) {
		if (n >= length()) {
			resize(n+1);
		}

		iterator ci = begin();
		return (reference) * (ci + n);
	}

	CR_NO_THROW inline const char *c_str(void) const {
	    return c_string;
	} 
    
    CR_NO_THROW inline const char *data(void) const {
	    return c_string;
	} 
    
    CR_NO_THROW inline size_t size(void) const {
	    return length();
	}

	CR_NO_THROW inline size_t length(void) const {
	    size_t len = 0;
	    if (c_string) {
            len = strnlen(c_string, cr_cstring::MAX_CR_STRING_LEN);
	    }

	    return len;
	}

	CR_NO_THROW inline bool empty(void) const {
	    bool res = false;
	    if (length() == 0) {
		res = true;
	    }

	    return res;
	}

	CR_NO_THROW inline const size_t max_size(void) const {
	    return (size_t) cr_cstring::MAX_CR_STRING_LEN;
	}

	CR_NO_THROW inline const size_t capacity(void) const {
	    return (size_t) ((aloc_capacity == 0) ? 0 : aloc_capacity - 1);
	}
	
    CR_NO_THROW inline cr_cstring substr(size_t pos = 0, size_t n = npos) const {
	    return cr_cstring(*this, pos, n);
	}

	CR_NO_THROW inline size_t copy(char *buf, size_t n = npos, size_t pos = 0) const {
	    size_t tn = 0;
	    if (buf != NULL && c_string != NULL) {
		size_t len = strnlen(c_string, cr_cstring::MAX_CR_STRING_LEN);
		if (pos <= len) {
		    if ((pos + n) <= len)
			tn = n;
		    else
			tn = len - pos;
		    memcpy(buf, c_string + pos, tn);
		}
	    }

	    return tn;
	}

	CR_NO_THROW inline int compare(const char *s) const {
	    if (s != NULL && c_string != NULL) {
		return strncmp(c_string, s, cr_cstring::MAX_CR_STRING_LEN);
	    }

	    return -1;
	}

	CR_NO_THROW inline int compare(const cr_cstring & s) const {
	    return compare(s.c_str());
	}

	CR_NO_THROW inline int compare(size_type pos, size_type n, const cr_cstring & s) const {
	    cr_cstring t;
	     t.assign(*this, pos, n);
	     return t.compare(s);
	}

	CR_NO_THROW inline int compare(size_type pos, size_type n,
			   const cr_cstring & s, size_type pos1,
			   size_type n1) const {
	    cr_cstring t;
	     t.assign(*this, pos, n);
	    cr_cstring t1;
	     t1.assign(s, pos1, n1);
	     return t.compare(t1);
	}

	CR_NO_THROW inline int compare(size_type pos, size_type n, const char *s,
			   size_type len = npos) const {
	    cr_cstring t;
	     t.assign(*this, pos, n);
	    cr_cstring t1(s);
	    if (len > t1.length())
		 len = t1.length();
	     t1.resize(len);
	     return t.compare(t1);
	} 
    
    CR_NO_THROW inline cr_cstring & erase(size_t pos = 0, size_t n = npos) {
		if (c_string != NULL) {	
			size_t len = 0;

			len = strnlen(c_string, cr_cstring::MAX_CR_STRING_LEN);
	    
			size_t rem=0;

			if (pos <= len) {
				if ((pos + n) > len)
					n = len - pos;

				rem = len - (pos + n);

				if (rem) {
					memmove(c_string + pos, c_string + pos + n, rem);
				}

				c_string[pos + rem] = 0;
			}
		}

	    return *this;
	}

	CR_NO_THROW inline void clear(void) {
	    erase(0, npos);
	}

	CR_NO_THROW inline size_t find(const char *str, size_t pos = 0) const {
	    size_t ret = npos;
	    if (str != NULL && c_string != NULL) {
			//printf("c_string+pos = %s\n", (c_string+pos));
			//printf("str = %s\n", str);
		char *rptr = strstr(c_string + pos, str);
		//printf("rptr = %x\n", (int)rptr);

		if (rptr != NULL) {
			//printf("rptr was not NULL\n");
			//printf("rptr = %x c_string = %x\n", (int)rptr, (int)c_string);
		     ret = (rptr - c_string);
		}
	    }
		//printf("In find: returning size_t = %d\n", ret);
	    return ret;
	}

	CR_NO_THROW inline size_t find(const cr_cstring & s, size_t pos = 0) const {
	    return find(s.c_str(), pos);
	} 
    
    CR_NO_THROW inline size_type find(const char *s, size_type pos, size_type n) const {
	    cr_cstring t(s);
	     t.resize(n);
	     return find(t.c_str(), pos);
	} 
    
    CR_NO_THROW inline size_type find(char c, size_type pos = 0) const {
	    cr_cstring t(c);
	     return find(t.c_str(), pos);
	} 

    CR_NO_THROW inline size_type rfind(const char* s, size_type pos, size_type n) const{
        cr_cstring t;
        for (const_iterator i = end() - 1; i != begin() - 1; i--) {
            t += *i;
        } 
        
        if (pos > t.size()) {
            pos = 0;
        } else {
            pos = t.size() - pos;
        }

        cr_cstring ts;
        size_type sn = strnlen(s, cr_cstring::MAX_CR_STRING_LEN);
        sn = sn - (sn-n);
        for(size_type i = sn; i != -1; i--) {
            ts += s[i];
        }

        size_type pos1 = t.find(ts.c_str(), pos);
        if (pos1 != npos) {
            pos1 = length() - pos1 - sn;
        }

        return pos1;
    }

    CR_NO_THROW inline size_type rfind(const cr_cstring& s, size_type pos = npos) const {
        return rfind(s.c_str(), pos, s.length());
    }

    CR_NO_THROW inline size_type rfind(const char* s, size_type pos = npos) const {
        return rfind(s, pos, strnlen(s, cr_cstring::MAX_CR_STRING_LEN));
    }

    CR_NO_THROW inline size_type rfind(char c, size_type pos = npos) const {
        cr_cstring t = c;
        return rfind(t.c_str(), pos, t.length());
    }
    
    CR_NO_THROW inline size_type find_first_of(const char *s, size_type pos, size_type n) const {
	    size_type pos1 = npos;
		//printf("%s function called with pos = %d\n", __FUNCTION__, pos);
	    if (c_string != NULL && pos <= length()) {
			cr_cstring t;
			t.assign(s, n);

			char *c = strpbrk(c_string + pos, t.c_str());

			if (c != NULL) {
				pos1 = (c - begin());
			}
	    }
	    return pos1;
	}

	CR_NO_THROW inline size_type find_first_of(const cr_cstring & s, size_type pos =
				       0) const {
	    return find_first_of(s.c_str(), pos, s.length());
	} 
    
    CR_NO_THROW inline size_type find_first_of(const char *s, size_type pos = 0) const {
	    return find_first_of(s, pos, strnlen(s, cr_cstring::MAX_CR_STRING_LEN));
	} 
    
    CR_NO_THROW inline size_type find_first_of(char c, size_type pos = 0) const {
	    size_type pos1 = npos;
	    if (c_string != NULL && pos <= length()) {
		char *t = strchr(c_string + pos, c);
		if (t != NULL) {
		    pos1 = (t - begin());
		}
	    }
	    return pos1;
	}

	CR_NO_THROW inline size_type find_first_not_of(const char *s, size_type pos,
					   size_type n) const {
	    size_type pos1 = npos;
	    if (c_string != NULL && pos <= length()) {
			cr_cstring t;
			t.assign(s, n);
			size_type i;
			for(i=0; i < (length()-pos);i++)
			{
				bool matched = false;
				for(size_type j=0; j < strlen(s); j++)
				{
					if(*(c_string+pos+i) != *(s+j))
						continue;

					matched = true;
					break;
				}

				if (matched == false)
					break;
			}

			if (i != (length()-pos))
				pos1 = pos + i;
	    }
	    return pos1;
	}

	CR_NO_THROW inline size_type find_first_not_of(const char *s, size_type pos =
					   0) const {
	    return find_first_not_of(s, pos,
				     strnlen(s, cr_cstring::MAX_CR_STRING_LEN));
	} 
    
    CR_NO_THROW inline size_type find_first_not_of(const cr_cstring & s,
					     size_type pos = 0) const {
	    return find_first_not_of(s.c_str(), pos, s.length());
	} 
    
    CR_NO_THROW inline size_type find_first_not_of(char c, size_type pos = 0) const {
	    cr_cstring t;
	     t = c;
	     return find_first_not_of(t.c_str(), pos, 1);
	} 
    
    CR_NO_THROW inline size_type find_last_of(const char *s, size_type pos,
					size_type n) const {
	    cr_cstring t;
	    for (const_iterator i = end() - 1; i != begin() - 1; i--) {
		t += *i;
	    } if (pos > t.size()) {
		pos = 0;
	    } else {
		pos = t.size() - pos;
	    }

	    size_type pos1 = t.find_first_of(s, pos, n);
	    if (pos1 != npos) {
		pos1 = length() - (pos1 + 1);
	    }
	    return pos1;
	}

	CR_NO_THROW inline size_type find_last_of(const char *s, size_type pos = npos) const {
	    return find_last_of(s, pos, strnlen(s, cr_cstring::MAX_CR_STRING_LEN));
	} 
    
    CR_NO_THROW inline size_type find_last_of(char c, size_type pos = npos) const {
	    cr_cstring t;
	     t = c;
	     return find_last_of(t.c_str(), pos, 1);
	} 
    
    CR_NO_THROW inline size_type find_last_of(const cr_cstring & s,
					size_type pos = npos) const {
	    return find_last_of(s.c_str(), pos, s.length());
	} 
    
    CR_NO_THROW inline size_type find_last_not_of(const char *s, size_type pos,
					    size_type n) const {
	    cr_cstring t;
	    for (const_iterator i = end() - 1; i != begin() - 1; i--) {
		t += *i;
	    } if (pos > t.size()) {
		pos = 0;
	    } else {
		pos = t.size() - pos;
	    }

	    size_type pos1 = t.find_first_not_of(s, pos, n);
	    if (pos1 != npos) {
		pos1 = length() - (pos1 + 1);
	    }
	    return pos1;
	}

	CR_NO_THROW inline size_type find_last_not_of(const char *s, size_type pos =
					  npos) const {
	    return find_last_not_of(s, pos, strnlen(s, cr_cstring::MAX_CR_STRING_LEN));
	} 
    
    CR_NO_THROW inline size_type find_last_not_of(char c, size_type pos = npos) const {
	    cr_cstring t;
	     t = c;
	     return find_last_not_of(t.c_str(), pos, 1);
	} 
    
    CR_NO_THROW inline size_type find_last_not_of(const cr_cstring & s,
					    size_type pos = npos) const {
	    return find_last_not_of(s.c_str(), pos, s.length());
	} 
    
    CR_NO_THROW inline iterator begin(void) {
	    if (c_string)
		return &c_string[0];
	    else
		return NULL;
	}

	CR_NO_THROW inline iterator end(void) {
	    if (c_string)
		return &c_string[length()];
	    else
		return NULL;
	}

	CR_NO_THROW inline const_iterator begin(void) const {
	    if (c_string)
		return (const_iterator) & c_string[0];
	    else
		return (const_iterator) NULL;
	} 
    
    CR_NO_THROW inline const_iterator end(void) const {
	    if (c_string)
		return (const_iterator) & c_string[length() + 1];
	    else
		return (const_iterator) NULL;
	} 

    reverse_iterator rbegin(void) {
        iterator tmp = end();
        return reverse_iterator(tmp);
    }

    reverse_iterator rend(void) {
        iterator tmp = begin();
        return reverse_iterator(tmp);
    }

    CR_NO_THROW inline void reserve(size_t n) {

	    if ((n != this->size()) && (n < MAX_CR_STRING_LEN)) {
    
            n = n<MIN_CR_STRING_LEN? MIN_CR_STRING_LEN: n;

            if (n < this->size())
                n = this->size();

            if ( (n >= this->capacity()) && (n < (2 * this->capacity())) )
                n = 2 * this->capacity();

            c_string = (char *) realloc(c_string, n + 1);
		if(c_string == (char*)0)
		{
			printf("ERROR:: Unable to Malloc memory in file %s at line %d\n",__FILE__,__LINE__);
		}


            if (aloc_capacity == 0)
                c_string[0] = 0;

            aloc_capacity = n + 1;
	    } else {
		//cout << "Error:: n value asked is > than max string size."
		//    << endl;
	    }
	}

	CR_NO_THROW inline void swap(cr_cstring & s) {
	    cr_cstring t = s;
	    s = c_string;
	    *this = t;
	}

	CR_NO_THROW inline void push_back(value_type c) {
	    *this += c;
	}

	CR_NO_THROW inline void resize(size_type n, char c = char ()) {
        size_type len = size();

        if ((len!=n) && (n <= cr_cstring::MAX_CR_STRING_LEN)) {

            if (len < n) {
                //size_type aloc = n+1;

                //c_string = (pointer) realloc(c_string, aloc);
                //aloc_capacity = aloc;
                reserve(n);

                pointer tmp = c_string+len;
                for (size_type i = len; i < n; i++) {
                    memcpy(tmp, &c, sizeof(value_type));
                    tmp++;
                }

                c_string[n] = (value_type)0;
            } else {

                if (n == 0) {
                    if(c_string) {
                        free(c_string);
                    }

                    c_string = NULL;
                    aloc_capacity = 0;
                }
                else {
                    c_string[n] = (value_type)0;
                }

            }
        }
	}

	CR_NO_THROW inline cr_cstring & assign(const cr_cstring & s) {
	    *this = s;

	    return *this;
	}

	CR_NO_THROW inline cr_cstring & assign(const cr_cstring & s, size_type pos,
				  size_type n) {
	    cr_cstring t(s, pos, n);
	    *this = t;

	    return *this;
	}

	CR_NO_THROW inline cr_cstring & assign(const char *s, size_type n) {
	    cr_cstring t(s, 0, n);
	    *this = t;
	    return *this;
	}

	CR_NO_THROW inline cr_cstring & assign(const char *s) {
	    cr_cstring t(s);
	    *this = t;
	    return *this;
	}

	CR_NO_THROW inline cr_cstring & assign(const wchar_t *s) {
		int length = wcslen(s);
		char *m = NULL;
		m = (char *)realloc(m, length + 1);
		wcstombs(m,s,length+1);
		cr_cstring t(m);
	    *this = t;
		free(m);
	    return *this;
	}
	CR_NO_THROW inline cr_cstring & assign(size_type n, char c) {
	    resize(0);
	    resize(n, c);
	    return *this;
	}

	CR_NO_THROW inline cr_cstring & assign(InputIterator first, InputIterator last) {
	    resize(0);
	    for (InputIterator i = first; i != last; i++) {
		*this += *i;
	    }

	    return *this;
	}

	CR_NO_THROW inline cr_cstring & insert(size_type pos, const char *s) {
	    if (pos >= 0 && pos <= length()) {
              size_type len = size();
              size_type n = strnlen(s, cr_cstring::MAX_CR_STRING_LEN);

              len += n;

              if (len >= capacity()) reserve(len);

              /* move string from pos n char away */
		      memmove(c_string + pos + n, c_string + pos, size()-pos + 1);

              //for (size_type i = 0; i < n; i++)
                //c_string[pos+i] = s[i];
              memcpy(c_string + pos, s, n);
 
            //cr_cstring t;
            //t.assign(begin(), begin() + pos);
            //t += s;

            //if (pos < length()) {
            //    t.cat_str(c_string + pos, 0, (length() - pos));
            //}
            //*this = t;
	    }

	    return *this;
	}

	CR_NO_THROW inline cr_cstring & insert(size_type pos, const char *s,
				  size_type n) {

	    if (pos >= 0 && pos <= length()) {
              size_type len = size();
              len += n;

              if (len >= capacity()) reserve(len);

              /* move string from pos n char away */
		      memmove(c_string + pos + n, c_string + pos, size()-pos + 1);

              //for (size_type i = 0; i < n; i++)
                //c_string[pos+i] = s[i];
              memcpy(c_string + pos, s, n);
            
//            cr_cstring t;
//            t.assign(begin(), begin() + pos);

//            t.cat_str(s, 0, n);

//            if (pos < length()) {
//                t.cat_str(begin() + pos, 0, (length() - pos));
//            }

//            *this = t;
	    }

	    return *this;
	}

	CR_NO_THROW inline cr_cstring & insert(size_type pos, const cr_cstring & s) {
	    return insert(pos, s.c_str());
	}

	CR_NO_THROW inline cr_cstring & insert(size_type pos, const cr_cstring & s,
				  size_type pos1, size_type n) {
	    if (pos >= 0 && pos <= length()) {
              size_type len = size();
              len += n;

              if (len >= capacity()) reserve(len);

              /* move string from pos n char away */
		      memmove(c_string + pos + n, c_string + pos, size()-pos+1);

              //for (size_type i = 0; i < n; i++)
              //  c_string[pos+i] = s[pos1+i];
              memcpy(c_string + pos, s.c_str() + pos1, n);
 
            //cr_cstring t;
            //t.assign(begin(), begin() + pos);

            //t.cat_str(s.begin() + pos1, 0, n);

            //if (pos < length()) {
            //    t.cat_str(begin() + pos, 0, (length() - pos));
            //}
            //*this = t;
	    }

	    return *this;
	}

	CR_NO_THROW inline cr_cstring & insert(size_type pos, size_type n, char c) {
	    cr_cstring t;
	    t.resize(n, c);
	    return insert(pos, t.c_str());
	}

	CR_NO_THROW inline cr_cstring & append(const char *s) {
	    *this += s;
	    return *this;
	}

	CR_NO_THROW inline cr_cstring & append(const char *s, size_type n) {
	    size_type len = length();
	    *this += s;
	    resize(len + n);
	    return *this;
	}

	CR_NO_THROW inline cr_cstring & append(const cr_cstring & s) {
	    return append(s.c_str());
	}

	CR_NO_THROW inline cr_cstring & append(const cr_cstring & s, size_type pos,
				  size_type n) {
	    return append((s.c_str() + pos), n);
	}

	CR_NO_THROW inline cr_cstring & append(size_type n, char c) {
	    cr_cstring t;
	    t.resize(n, c);
	    return append(t.c_str());
	}

	CR_NO_THROW inline cr_cstring & append(InputIterator first, InputIterator last) {
	    cr_cstring t;
	    t.assign(first, last);
	    return append(t.c_str());
	}

	CR_NO_THROW inline cr_cstring & replace(size_type pos, size_type n,
				   const char *s, size_type n1) {

	    size_type len = (n > n1 ? (n - n1) : (n1 - n));
        if ((len+size()) >= capacity()) reserve(len+size());


        if (n <= n1)
        {
            /* first replace till n */
            memcpy(c_string+pos, s, n);

            /* some chars need to be inserted */
            len = n1 - n;
            if (len)
            {
                insert(pos+n, s+n, len);
            }
        }
        else
        {
            /* first replace till n1 */
            memcpy(c_string+pos, s, n1);

            /* some chars need to be erased */
            len = n - n1;
            erase(pos+n1, len);
        }

	    return *this;
	}

	CR_NO_THROW inline cr_cstring & replace(size_type pos, size_type n,
				   const cr_cstring & s) {
	    return replace(pos, n, s.c_str(), s.length());
	}

	CR_NO_THROW inline cr_cstring & replace(size_type pos, size_type n,
				   const cr_cstring & s, size_type pos1,
				   size_type n1) {
        return replace(pos, n, s.c_str()+pos1, n1);
	}

	CR_NO_THROW inline cr_cstring & replace(size_type pos, size_type n,
				   const char *s) {
	    return replace(pos, n, s, strnlen(s, cr_cstring::MAX_CR_STRING_LEN));
	}

	CR_NO_THROW inline cr_cstring & replace(size_type pos, size_type n,
				   size_type n1, char c) {
	    cr_cstring t;
	    t.resize(n1, c);
	    return replace(pos, n, t);
	}

	CR_NO_THROW inline cr_cstring & replace(iterator first, iterator last,
				   const char *s, size_type n) {
        size_type len = (size_type)(first - c_string);
        size_type n1 = (size_type)(last - first);
        return replace(len, n1, s, n);
	}

	CR_NO_THROW inline cr_cstring & replace(iterator first, iterator last,
				   const cr_cstring & s) {
	    return replace(first, last, s.c_str(), s.length());
	}

	CR_NO_THROW inline cr_cstring & replace(iterator first, iterator last,
				   const char *s) {
	    return replace(first, last, s, strnlen(s, cr_cstring::MAX_CR_STRING_LEN));
	}

	CR_NO_THROW inline cr_cstring & replace(iterator first, iterator last,
				   size_type n, char c) {
	    cr_cstring t;
	    t.resize(n, c);
	    return replace(first, last, t);
	}

private:

	CR_NO_THROW inline size_t copy_str(char *c, size_t pos, size_t n) {
	    size_t tn = 0;
	    size_t len = 0;
	    if (c != NULL && c_string != NULL) {
		len = strnlen(c_string, cr_cstring::MAX_CR_STRING_LEN);

		if (pos <= len) {
		    if ((pos + n) <= len)
			tn = n;
		    else
			tn = len - pos;
		    memcpy(c, c_string + pos, tn);
		}
	    }

	    return tn;
	}			//end of copy_str()

	CR_NO_THROW inline void cat_str(const char *c, size_t pos, size_t n) {
	    if (c != NULL) {
            size_t oldlen = size();
            size_t len1 = strnlen(c, cr_cstring::MAX_CR_STRING_LEN)+1;

            if (n == npos) {
                n = len1;
            }

            if (pos <= len1) {
                size_t tn;
                if ((pos + n) <= len1)
                    tn = n;
                else
                    tn = len1 - pos;

                if (tn > 0) {
                    size_t alloc = oldlen + tn;

                    /* check for >= because we need to consider null character */
                    if (alloc >= this->capacity())
                        reserve(alloc);

                    memcpy(c_string + oldlen, c + pos, tn);
                    c_string[oldlen + tn] = 0;
                }
            }
	    }
	}			//end of cat_str()

public:
     bool get_copyable(void) const { return copyable; }
     void set_copyable(void) { copyable = true; }
     const char* get_string() const { return c_string; }
     void  set_string(const char* s) const {
        if (this->c_string) free(this->c_string);
        /* force a copy */
        memcpy((void*)&c_string, (void*)&s, sizeof(char*));
     }
     bool copyable;

private:
	char *c_string;
	size_type aloc_capacity;

};

typedef class cr_cstring cr_cstring;

//static inline ostream& operator<<(ostream & os, const cr_cstring & s) {
//return os << s.c_str();
//}

//static inline istream& operator>>(istream & is, cr_cstring & s) {
//char c[cr_cstring::MAX_CR_LINE_LEN] = { 0 };
//s = c;
//cr_cstring::iterator cstr = s.begin();
//is >> cstr;
//return is;
//}

//static inline istream & getline(istream & is, cr_cstring & s) {
//char c[cr_cstring::MAX_CR_LINE_LEN] = { 0 };

//is.getline(c, cr_cstring::MAX_CR_LINE_LEN);
//s = c;

//return is;
//}

static CR_NO_THROW inline cr_cstring operator+(const cr_cstring & s1,
                  const cr_cstring & s2) {
cr_cstring t(s1);
t += s2;
#if 0
cr_cstring t;
cr_cstring::size_type len = s1.size() + s2.size();
t.reserve(len);
t.append(s1);
t.append(s2);
t.set_copyable();

/* check if the passed string is copyable */
if (s1.get_copyable())
{
    const char* t1 = s1.get_string();
    /* force free this string */
    s1.set_string(t1);    
}
#endif

return t;
}

static CR_NO_THROW inline cr_cstring operator+(const char *s1, const cr_cstring & s2) {
cr_cstring t(s1);
t += s2;
#if 0
cr_cstring t;
cr_cstring::size_type len = strnlen(s1, cr_cstring::MAX_CR_STRING_LEN);
len += s2.size();
t.reserve(len);
t.append(s1);
t.append(s2);
#endif
return t;
}

static CR_NO_THROW inline cr_cstring operator+(const cr_cstring & s1, const char *s2) {
cr_cstring t(s1);
t += s2;
#if 0
cr_cstring t;
cr_cstring::size_type len = strnlen(s2, cr_cstring::MAX_CR_STRING_LEN);
len += s1.size();
t.reserve(len);
t.append(s1);
t.append(s2);
t.set_copyable();

/* check if the passed string is copyable */
if (s1.get_copyable())
{
    const char* t1 = s1.get_string();
    s1.set_string(t1);    
}
#endif

return t;
}

static CR_NO_THROW inline cr_cstring operator+(char c, const cr_cstring & s2) {
char tt[4] = { 0 };
tt[0] = c;
cr_cstring::size_type len = s2.size();
cr_cstring t;
t.reserve(len+1);
t.append(tt);
t.append(s2);
return t;
}

static CR_NO_THROW inline cr_cstring operator+(const cr_cstring & s1, char c) {
char tt[4] = { 0 };
tt[0] = c;
cr_cstring::size_type len = s1.size();
cr_cstring t;
t.reserve(len+1);
t.append(s1);
t.append(tt);
return t;
}

static CR_NO_THROW inline bool operator==(const cr_cstring & s1,
              const cr_cstring & s2) {
return ((strncmp(s1.c_str(), s2.c_str(), cr_cstring::MAX_CR_STRING_LEN) ==
     0) ? true : false);
}

static CR_NO_THROW inline bool operator==(const char *s1, const cr_cstring & s2) {
return ((strncmp(s1, s2.c_str(), cr_cstring::MAX_CR_STRING_LEN) ==
     0) ? true : false);
}

static CR_NO_THROW inline bool operator==(const cr_cstring & s1, const char *s2) {
return ((strncmp(s1.c_str(), s2, cr_cstring::MAX_CR_STRING_LEN) ==
     0) ? true : false);
}

static CR_NO_THROW inline bool operator!=(const cr_cstring & s1, const cr_cstring & s2) {
return ((strncmp(s1.c_str(), s2.c_str(), cr_cstring::MAX_CR_STRING_LEN) !=
     0) ? true : false);
}

static CR_NO_THROW inline bool operator!=(const char *s1, const cr_cstring & s2) {
return ((strncmp(s1, s2.c_str(), cr_cstring::MAX_CR_STRING_LEN) !=
     0) ? true : false);
}

static CR_NO_THROW inline bool operator!=(const cr_cstring & s1, const char *s2) {
return ((strncmp(s1.c_str(), s2, cr_cstring::MAX_CR_STRING_LEN) !=
     0) ? true : false);
}

static CR_NO_THROW inline bool operator<(const cr_cstring & s1,
             const cr_cstring & s2) {
return ((strncmp(s1.c_str(), s2.c_str(), cr_cstring::MAX_CR_STRING_LEN) <
     0) ? true : false);
}

static CR_NO_THROW inline bool operator<(const char *s1, const cr_cstring & s2) {
return ((strncmp(s1, s2.c_str(), cr_cstring::MAX_CR_STRING_LEN) <
     0) ? true : false);
}

static CR_NO_THROW inline bool operator<(const cr_cstring & s1, const char *s2) {
return ((strncmp(s1.c_str(), s2, cr_cstring::MAX_CR_STRING_LEN) <
     0) ? true : false);
}

static CR_NO_THROW inline bool operator<=(const cr_cstring & s1,
             const cr_cstring & s2) {
return ((strncmp(s1.c_str(), s2.c_str(), cr_cstring::MAX_CR_STRING_LEN) <=
     0) ? true : false);
}

static CR_NO_THROW inline bool operator<=(const char *s1, const cr_cstring & s2) {
return ((strncmp(s1, s2.c_str(), cr_cstring::MAX_CR_STRING_LEN) <=
     0) ? true : false);
}

static CR_NO_THROW inline bool operator<=(const cr_cstring & s1, const char *s2) {
return ((strncmp(s1.c_str(), s2, cr_cstring::MAX_CR_STRING_LEN) <=
     0) ? true : false);
}

static CR_NO_THROW inline bool operator>(const cr_cstring & s1,
             const cr_cstring & s2) {
return ((strncmp(s1.c_str(), s2.c_str(), cr_cstring::MAX_CR_STRING_LEN) >
     0) ? true : false);
}

static CR_NO_THROW inline bool operator>(const char *s1, const cr_cstring & s2) {
return ((strncmp(s1, s2.c_str(), cr_cstring::MAX_CR_STRING_LEN) >
     0) ? true : false);
}

static CR_NO_THROW inline bool operator>(const cr_cstring & s1, const char *s2) {
return ((strncmp(s1.c_str(), s2, cr_cstring::MAX_CR_STRING_LEN) >
     0) ? true : false);
}

static CR_NO_THROW inline bool operator>=(const cr_cstring & s1,
             const cr_cstring & s2) {
return ((strncmp(s1.c_str(), s2.c_str(), cr_cstring::MAX_CR_STRING_LEN) >=
     0) ? true : false);
}

static CR_NO_THROW inline bool operator>=(const char *s1, const cr_cstring & s2) {
return ((strncmp(s1, s2.c_str(), cr_cstring::MAX_CR_STRING_LEN) >=
     0) ? true : false);
}

static CR_NO_THROW inline bool operator>=(const cr_cstring & s1, const char *s2) {
return ((strncmp(s1.c_str(), s2, cr_cstring::MAX_CR_STRING_LEN) >=
     0) ? true : false);
}

static CR_NO_THROW inline bool operator==(const cr_cstring::reverse_iterator& x, const cr_cstring::reverse_iterator& y) {
    return (x.get_itr()==y.get_itr());
}

static CR_NO_THROW inline bool operator!=(const cr_cstring::reverse_iterator& x, const cr_cstring::reverse_iterator& y) {
    return (x.get_itr()!=y.get_itr());
}

static CR_NO_THROW inline bool operator<(const cr_cstring::reverse_iterator& x, const cr_cstring::reverse_iterator& y) {
    return (x.get_itr()<y.get_itr());
}

static CR_NO_THROW inline bool operator<=(const cr_cstring::reverse_iterator& x, const cr_cstring::reverse_iterator& y) {
    return (x.get_itr()<=y.get_itr());
}

static CR_NO_THROW inline bool operator>(const cr_cstring::reverse_iterator& x, const cr_cstring::reverse_iterator& y) {
    return (x.get_itr()>y.get_itr());
}

static CR_NO_THROW inline bool operator>=(const cr_cstring::reverse_iterator& x, const cr_cstring::reverse_iterator& y) {
    return (x.get_itr()>=y.get_itr());
}


}				//end of namespace osal_stl

#endif				//__CR_CSTRING_H_

