/******************************************************************************
*
*                   INTEL CORPORATION PROPRIETARY INFORMATION
*       This software is supplied under the terms of a license agreement or
*       nondisclosure agreement with Intel Corporation and may not be copied
*       or disclosed except in accordance with the terms of that agreement.
*
*            Copyright (c) 2008-2009 Intel Corporation. All Rights Reserved.
*
*       All rights reserved.  No part of this program or publication may be
*       reproduced, transmitted, transcribed, stored in a retrieval system,
*       or translated into any language or computer language, in any form or
*       by any means, electronic, mechanical, magnetic, optical, chemical,
*       manual, or otherwise, without the prior written permission of Intel
*       Corporation.
*
*
*******************************************************************************/
#ifndef __CR_DEFAULT_FUNCTORS_H_
#define __CR_DEFAULT_FUNCTORS_H_

#include <stdlib.h>
#include <string.h>

namespace osal_stl {

    /* unary functor definition */
    template<typename arg, typename res>
    struct unary_function
    {
        typedef     arg     argument_type;
        typedef     res     result_type;
    };

    /* binary functor definition */
    template<typename arg1, typename arg2, typename res>
    struct binary_function
    {
        typedef arg1 first_argument_type;
        typedef arg2 second_argument_type;
        typedef res  result_type;
    };

    /* binary plus functor definition */
    template<typename T>
    struct cr_plus : public binary_function<T, T, T>
    {
        CR_NO_THROW T
        operator() (const T& x, const T& y) const
        { return (x + y); }
    };

    /* binary minus functor definition */
    template<typename T>
    struct cr_minus : public binary_function<T, T, T>
    {
        CR_NO_THROW T
        operator() (const T& x, const T& y) const
        { return (x - y); }
    };

    /* binary multiply functor definition */
    template<typename T>
    struct cr_multiplies : public binary_function<T, T, T>
    {
        CR_NO_THROW T
        operator() (const T& x, const T& y) const
        { return (x * y); }
    };

    /* binary divides functor definition */
    template<typename T>
    struct cr_divides : public binary_function<T, T, T>
    {
        CR_NO_THROW T
        operator() (const T& x, const T& y) const
        { return (x / y); }
    };

    /* binary modulus functor definition */
    template<typename T>
    struct cr_modulus : public binary_function<T, T, T>
    {
        CR_NO_THROW T
        operator() (const T& x, const T& y) const
        { return (x % y); }
    };

    /* unary negate functor definition */
    template<typename T>
    struct cr_negate : public unary_function<T, T>
    {
        CR_NO_THROW T
        operator() (const T& x) const
        { return -x; }
    };

    /* binary comparator == functor */
    template<typename E> struct cr_equal_to : public binary_function<E, E, bool> 
    {
        CR_NO_THROW bool
        operator()(const E& x, const E& y) const
        { return (x == y); }
    };

    /* binary comparator != functor */
    template<typename E> struct cr_not_equal_to : public binary_function<E, E, bool> 
    {
        CR_NO_THROW bool
        operator()(const E& x, const E& y) const
        { return (x != y); }
    };

    /* binary comparator > functor */
    template<typename E> struct cr_greater : public binary_function<E, E, bool> 
    {
        CR_NO_THROW bool
        operator()(const E& x, const E& y) const
        { return (x > y); }
    };

    /* binary comparator < functor */
    template<typename E> struct cr_less : public binary_function<E, E, bool> 
    {
        CR_NO_THROW bool
        operator()(const E& x, const E& y) const
        { return (x < y); }
    };

    /* binary comparator >= functor */
    template<typename E> struct cr_greater_equal : public binary_function<E, E, bool> 
    {
        CR_NO_THROW bool
        operator()(const E& x, const E& y) const
        { return (x >= y); }
    };

    /* binary comparator <= functor */
    template<typename E> struct cr_less_equal : public binary_function<E, E, bool> 
    {
        CR_NO_THROW bool
        operator()(const E& x, const E& y) const
        { return (x <= y); }
    };

    /* binary logical and (&&) functor */
    template<typename E> struct cr_logical_and : public binary_function<E, E, bool> 
    {
        CR_NO_THROW bool
        operator()(const E& x, const E& y) const
        { return (x && y); }
    };

    /* binary logical or (|| functor */
    template<typename E> struct cr_logical_or : public binary_function<E, E, bool> 
    {
        CR_NO_THROW bool
        operator()(const E& x, const E& y) const
        { return (x || y); }
    };

    /* unary logical not (!) functor */
    template<typename E> struct cr_logical_not : public unary_function<E, bool> 
    {
        CR_NO_THROW bool
        operator()(const E& x) const
        { return !x; }
    };

    /* binary bit and (&) functor */
    template<typename E> struct cr_bit_and : public binary_function<E, E, bool> 
    {
        CR_NO_THROW bool
        operator()(const E& x, const E& y) const
        { return (x & y); }
    };

    /* binary bit or (|) functor */
    template<typename E> struct cr_bit_or : public binary_function<E, E, bool> 
    {
        CR_NO_THROW bool
        operator()(const E& x, const E& y) const
        { return (x | y); }
    };

    /* binary bit xor (|) functor */
    template<typename E> struct cr_bit_xor : public binary_function<E, E, bool> 
    {
        CR_NO_THROW bool
        operator()(const E& x, const E& y) const
        { return (x ^ y); }
    };

    /* equal_to specialization for char* */
    template<>
    struct cr_equal_to<char*>
    {
        CR_NO_THROW bool
        operator()(const char* ch1, const char* ch2) const
        {
            return (strcmp(ch1, ch2) == 0);
        }
    };
 
    /* equal_to specialization for const char* */
    template<>
    struct cr_equal_to<const char*>
    {
        CR_NO_THROW bool
        operator()(const char* ch1, const char* ch2) const
        {
            return (strcmp(ch1, ch2) == 0);
        }
    };

	/* equal_to specialization for STRING */
    template<>
    struct cr_equal_to<STRING>
    {
        CR_NO_THROW bool
        operator()(const STRING ch1, const STRING ch2) const
        {
            return (strcmp(ch1.c_str(), ch2.c_str()) == 0);
        }
    };
 
    /* equal_to specialization for const STRING */
    template<>
    struct cr_equal_to<const STRING>
    {
        CR_NO_THROW bool
        operator()(const STRING ch1, const STRING ch2) const
        {
            return (strcmp(ch1.c_str(), ch2.c_str()) == 0);
        }
    };

    /* not_equal_to specialization for char* */
    template<>
    struct cr_not_equal_to<char*>
    {
        CR_NO_THROW bool
        operator()(const char* ch1, const char* ch2) const
        {
            return (strcmp(ch1, ch2) != 0);
        }
    };
 
    /* not_equal_to specialization for const char* */
    template<>
    struct cr_not_equal_to<const char*>
    {
        CR_NO_THROW bool
        operator()(const char* ch1, const char* ch2) const
        {
            return (strcmp(ch1, ch2) != 0);
        }
    };

    /* less specialization for char* */
    template<>
    struct cr_less<char*>
    {
        CR_NO_THROW bool
        operator()(const char* ch1, const char* ch2) const
        {
            return (strcmp(ch1, ch2) < 0);
        }
    };
 
    /* less specialization for const char* */
    template<>
    struct cr_less<const char*>
    {
        CR_NO_THROW bool
        operator()(const char* ch1, const char* ch2) const
        {
            return (strcmp(ch1, ch2) < 0);
        }
    };

    /* less specialization for char* */
    template<>
    struct cr_less<STRING>
    {
        CR_NO_THROW bool
        operator()(const STRING ch1, const STRING ch2) const
        {
            return (strcmp(ch1.c_str(), ch2.c_str()) < 0);
        }
    };
 
    /* less specialization for const char* */
    template<>
    struct cr_less<const STRING>
    {
        CR_NO_THROW bool
        operator()(const STRING ch1, const STRING ch2) const
        {
            return (strcmp(ch1.c_str(), ch2.c_str()) < 0);
        }
    };

    /* greater specialization for char* */
    template<>
    struct cr_greater<char*>
    {
        CR_NO_THROW bool
        operator()(const char* ch1, const char* ch2) const
        {
            return (strcmp(ch1, ch2) > 0);
        }
    };
 
    /* greater specialization for const char* */
    template<>
    struct cr_greater<const char*>
    {
        CR_NO_THROW bool
        operator()(const char* ch1, const char* ch2) const
        {
            return (strcmp(ch1, ch2) > 0);
        }
    };

    /* less_equal specialization for char* */
    template<>
    struct cr_less_equal<char*>
    {
        CR_NO_THROW bool
        operator()(const char* ch1, const char* ch2) const
        {
            return (strcmp(ch1, ch2) <= 0);
        }
    };
 
    /* less_equal specialization for const char* */
    template<>
    struct cr_less_equal<const char*>
    {
        CR_NO_THROW bool
        operator()(const char* ch1, const char* ch2) const
        {
            return (strcmp(ch1, ch2) <= 0);
        }
    };

    /* greater_equal specialization for char* */
    template<>
    struct cr_greater_equal<char*>
    {
        CR_NO_THROW bool
        operator()(const char* ch1, const char* ch2) const
        {
            return (strcmp(ch1, ch2) >= 0);
        }
    };
 
    /* greater_equal specialization for const char* */
    template<>
    struct cr_greater_equal<const char*>
    {
        CR_NO_THROW bool
        operator()(const char* ch1, const char* ch2) const
        {
            return (strcmp(ch1, ch2) >= 0);
        }
    };

	//-----------------------------------------------------------------------------
	// MurmurHash2A, by Austin Appleby

	// This is a variant of MurmurHash2 modified to use the Merkle-Damgard
	// construction. Bulk speed should be identical to Murmur2, small-key speed
	// will be 10%-20% slower due to the added overhead at the end of the hash.

	// This variant fixes a minor issue where null keys were more likely to
	// collide with each other than expected, and also makes the algorithm
	// more amenable to incremental implementations. All other caveats from
	// MurmurHash2 still apply.

	#define mmix(h,k) { k *= m; k ^= k >> r; k *= m; h *= m; h ^= k; }

	unsigned int inline MurmurHash2A ( const void * key, int len, unsigned int seed ) {

		const unsigned int m = 0x5bd1e995;
		const int r = 24;
		unsigned int l = len;

		const unsigned char * data = (const unsigned char *)key;

		unsigned int h = seed;

		while(len >= 4)
		{
			unsigned int k = *(unsigned int*)data;

			mmix(h,k);

			data += 4;
			len -= 4;
		}

		unsigned int t = 0;

		switch(len)
		{
		case 3: t ^= data[2] << 16;
		case 2: t ^= data[1] << 8;
		case 1: t ^= data[0];
		};

		mmix(h,t);
		mmix(h,l);

		h ^= h >> 13;
		h *= m;
		h ^= h >> 15;

		return h;
	}
	/*#define get16bits(d) (*((const uint16_t *) (d)))

	size_t inline SuperFastHash (const char * data, int len) {
		size_t hash = len, tmp;
		int rem;

		if (len <= 0 || data == NULL) return 0;

		rem = len & 3;
		len >>= 2;

		// Main loop 
		for (;len > 0; len--) {
			hash  += get16bits (data);
			tmp    = (get16bits (data+2) << 11) ^ hash;
			hash   = (hash << 16) ^ tmp;
			data  += 2*sizeof (uint16_t);
			hash  += hash >> 11;
		}

		// Handle end cases 
		switch (rem) {
		case 3: hash += get16bits (data);
			hash ^= hash << 16;
			hash ^= data[sizeof (uint16_t)] << 18;
			hash += hash >> 11;
			break;
		case 2: hash += get16bits (data);
			hash ^= hash << 11;
			hash += hash >> 17;
			break;
		case 1: hash += *data;
			hash ^= hash << 10;
			hash += hash >> 1;
		}

		// Force "avalanching" of final 127 bits 
		hash ^= hash << 3;
		hash += hash >> 5;
		hash ^= hash << 4;
		hash += hash >> 17;
		hash ^= hash << 25;
		hash += hash >> 6;

		return hash;
	}*/
    /* hash function object definition */
    template<class Key> struct cr_hash { };

    CR_NO_THROW inline size_t
    _cr_hash_string(const char* ch)
    {
		size_t hash  = 0;
		int length = strlen(ch);
		hash = MurmurHash2A(ch, length, 0);
        return hash;
    }
    
    /* cr_hash specialization for char */
    template<>
    struct cr_hash<char>
    {
        CR_NO_THROW size_t
        operator()(const char ch) const
        {
            return size_t(ch);
        }
    };
     
    /* cr_hash specialization for const char */
    template<>
    struct cr_hash<const char>
    {
        CR_NO_THROW size_t
        operator()(const char ch) const
        {
            return size_t(ch);
        }
    };
      
    /* cr_hash specialization for int */
    template<>
    struct cr_hash<int>
    {
        CR_NO_THROW size_t
        operator()(const int i) const
        {
            return size_t(i);
        }
    };
       
    /* cr_hash specialization for const int */
    template<>
    struct cr_hash<const int>
    {
        CR_NO_THROW size_t
        operator()(const int i) const
        {
            return size_t(i);
        }
    };
 
 
    /* cr_hash specialization for char* */
    template<>
    struct cr_hash<char*>
    {
        CR_NO_THROW size_t
        operator()(const char* ch) const
        {
            return _cr_hash_string(ch);
        }
    };
 
    /* cr_hash specialization for const char* */
    template<>
    struct cr_hash<const char*>
    {
        CR_NO_THROW size_t
        operator()(const char* ch) const
        {
            return _cr_hash_string(ch);
        }
    };

	/* cr_hash specialization for UINT8 */
    template<>
    struct cr_hash<UINT8>
    {
        CR_NO_THROW size_t
        operator()(UINT8 i) const
        {
            return size_t(i);
        }
    };

	/* cr_hash specialization for UINT16 */
    template<>
    struct cr_hash<UINT16>
    {
        CR_NO_THROW size_t
        operator()(UINT16 i) const
        {
            return size_t(i);
        }
    };

	/* cr_hash specialization for UINTN */
    template<>
    struct cr_hash<UINTN>
    {
        CR_NO_THROW size_t
        operator()(UINTN i) const
        {
            return size_t(i);
        }
    };

}//end of namespace osal_stl


#endif // __CR_DEFAULT_FUNCTORS_H_
