/******************************************************************************
*
*                   INTEL CORPORATION PROPRIETARY INFORMATION
*       This software is supplied under the terms of a license agreement or
*       nondisclosure agreement with Intel Corporation and may not be copied
*       or disclosed except in accordance with the terms of that agreement.
*
*            Copyright (c) 2008-2009 Intel Corporation. All Rights Reserved.
*
*       All rights reserved.  No part of this program or publication may be
*       reproduced, transmitted, transcribed, stored in a retrieval system,
*       or translated into any language or computer language, in any form or
*       by any means, electronic, mechanical, magnetic, optical, chemical,
*       manual, or otherwise, without the prior written permission of Intel
*       Corporation.
*
*
*******************************************************************************/
#ifndef __CR_VECTOR_H_
#define __CR_VECTOR_H_

#include <stdlib.h>
#include <string.h>

namespace osal_stl {

template<class T> class cr_vector {
    //public typedefs and member variables.
public:
    typedef T                   value_type;
    typedef const value_type    const_value_type;
    typedef value_type*         pointer;
    typedef value_type&         reference;
    typedef const value_type&   const_reference;
    typedef size_t              size_type;
    typedef size_t              difference_type;
    typedef value_type*         iterator;
    typedef const value_type*   const_iterator;

    const static int MAX_CR_VECTOR_SIZE = 1073741823;
    const static unsigned int NPOS_LENGTH=4294967295U;


    //public methods.
public:
    /* default constructor */
    CR_NO_THROW cr_vector() {
        cr_vector_ptr = (pointer) NULL;
        aloc_capacity = (size_type)0;
        aloc_size     = (size_type)0;
    }

    /* constructor */
    CR_NO_THROW cr_vector(size_type n) {
        cr_vector_ptr = (pointer) NULL;
        aloc_capacity = (size_type)0;
        aloc_size     = (size_type)0;

        resize(n);
    }

    /* constructor */
    CR_NO_THROW cr_vector(size_type n, const T& t) {
        cr_vector_ptr = (pointer) NULL;
        aloc_capacity = (size_type)0;
        aloc_size     = (size_type)0;

        resize(n, t);
    }

    /* copy constructor */
    CR_NO_THROW cr_vector(const cr_vector& aVec) {
        cr_vector_ptr = (pointer) NULL;
        aloc_capacity = (size_type)0;
        aloc_size     = (size_type)0;

        size_type s = aVec.size();
        reserve(s);
		for(size_type j=0; j < aVec.size(); j++)
        {
			new((void *)(cr_vector_ptr+j)) value_type(aVec[j]);
        }

		aloc_size = aVec.size();
    }

    CR_NO_THROW cr_vector(iterator f, iterator l) {
        cr_vector_ptr = (pointer) NULL;
        aloc_capacity = (size_type)0;
        aloc_size     = (size_type)0;

        size_type s = (l - f);
		reserve(s);

        for(iterator i = f; i != l; i++) {
           push_back(*i); 
        }
    }

    /* destructor*/
    CR_NO_THROW ~cr_vector() {
		clear();
    }

    CR_NO_THROW inline size_type size(void) const {
        return aloc_size;
    }

    CR_NO_THROW inline size_type max_size(void) const {
        return (size_type) cr_vector::MAX_CR_VECTOR_SIZE;
    }

    CR_NO_THROW inline size_type capacity(void) const {
        return aloc_capacity;
    }

    CR_NO_THROW inline bool empty(void) const {
        return ((const size_type)size() == 0);
    }

    CR_NO_THROW inline reference operator[](size_type n) {
		return at(n);
    }

    CR_NO_THROW inline const_reference operator[](size_type n) const {
        if (n < size()) {
			const_iterator cv = begin();
            return (const_reference) *(cv+n);
        }
		else {
			printf("Error: %s accessing out of bound index.\n", __FUNCTION__);
		}

		const_iterator itr = end();
		
        return (const_reference) *itr;
    }

	CR_NO_THROW inline reference at(size_type n) {
        
		if (n < size())
		{
			iterator cv = begin();
			return (reference) *(cv + n);
		}
		else
		{
			printf("Error: %s accessing out of bound index.\n", __FUNCTION__);
		}

		iterator itr = end();

        return *itr;
    }

    CR_NO_THROW inline const_reference at(size_type n) const {
        if (n < size()) {
			const_iterator cv = begin();
            return (const_reference) *(cv+n);
        }
		else {
			printf("Error: %s accessing out of bound index.\n", __FUNCTION__);
		}

		const_iterator itr = end();
		return (const_reference) *itr;
    }

    CR_NO_THROW inline iterator begin(void) {
        if (cr_vector_ptr)
        return &cr_vector_ptr[0];
        else
        return (iterator)0;
    }

    CR_NO_THROW inline iterator end(void) {
        if (cr_vector_ptr)
        return &cr_vector_ptr[size()];
        else
        return (iterator)0;
    }

    CR_NO_THROW inline const_iterator begin(void) const {
        if (cr_vector_ptr)
        return (const_iterator) &cr_vector_ptr[0];
        else
        return (const_iterator) NULL;
    }

    CR_NO_THROW inline const_iterator end(void) const {
        if (cr_vector_ptr)
        return (const_iterator) &cr_vector_ptr[size()];
        else
        return (const_iterator) NULL;
    }

    CR_NO_THROW inline void resize(size_type n, T t = T()) {
        size_type len = size();
		
        if ((n <= cr_vector::MAX_CR_VECTOR_SIZE) && (len!=n)) {

            if (len < n) {
                size_type aloc = n*sizeof(T);

				//Increasing the memory, only if current capacity is less.
				if(aloc_capacity < n)
				{
					cr_vector_ptr = (pointer) realloc(cr_vector_ptr, aloc);

					if(cr_vector_ptr == (pointer)0)
					{
						printf("ERROR:: Unable to Malloc memory in file %s at line %d\n",__FILE__,__LINE__);
					}

					aloc_capacity = n;
				}

                for (size_type i = len; i < n; i++) {     
					new((void *)(cr_vector_ptr+i)) value_type(t);
                }

                aloc_size     = n;
            }
            else {
                //Dropping extra elements.
				for(size_type i = n; i < len; i++)
					(cr_vector_ptr+i)->~T();

                aloc_size = n;
            }
        }
    }

    CR_NO_THROW void reserve(size_type n) {
        if (n > capacity())
        {
            cr_vector_ptr = (pointer) realloc(cr_vector_ptr, n*sizeof(T));
			if(cr_vector_ptr == (pointer)0)
			{
				printf("ERROR:: Unable to Malloc memory in file %s at line %d\n",__FILE__,__LINE__);
			}

            aloc_capacity = n;
        }
    }

    CR_NO_THROW reference front(void) {
        return cr_vector_ptr[0];
    }

    CR_NO_THROW const_reference front(void) const {
        return (const_reference)cr_vector_ptr[0];
    }

    CR_NO_THROW reference back(void) {
        return cr_vector_ptr[size()-1];
    }

    CR_NO_THROW const_reference back(void) const {
        return (const_reference) cr_vector_ptr[size()-1];
    }

    CR_NO_THROW void push_back(const T& t) {
        size_type s = this->size();
        size_type capcty = this->capacity();
		size_type aloc = capcty;

        while (aloc < (s+1)) {
          aloc  = (aloc!=0)?aloc*2:1;
		}

		if (aloc != capcty)
		{
			cr_vector_ptr = (pointer) realloc(cr_vector_ptr, (aloc)*sizeof(T));
			if(cr_vector_ptr == (pointer)0)
			{
				printf("ERROR:: Unable to Malloc memory in file %s at line %d\n",__FILE__,__LINE__);
			}

			aloc_capacity = aloc;
		}

		new((void *)(cr_vector_ptr+s)) value_type(t);
        
        aloc_size++;
    }

    CR_NO_THROW void pop_back(void) {
        size_type s = size();
        if (s > 0) {
            pointer pElem = (cr_vector_ptr+s-1);
            pElem->~T();
            aloc_size--;
        }
    }

    CR_NO_THROW void swap(cr_vector& aVec) {
        size_type s1 = size();
        size_type s2 = aVec.size();
        size_type c1 = capacity();
        size_type c2 = aVec.capacity();

        pointer ptr = cr_vector_ptr;
        cr_vector_ptr = aVec.get_pointer();
        aloc_size       = s2;
        aloc_capacity   = c2;

        aVec.set_pointer(ptr);
        aVec.set_size(s1);
        aVec.set_capacity(c1);
    }

    CR_NO_THROW inline iterator insert(iterator pos, const T& x) {
        iterator r = (iterator)npos;

        if ((pos >= begin()) && (pos <= end())) {
           insert(pos, 1, x);
           r = pos;
        }
       
        return r;
    }

    CR_NO_THROW inline void insert(iterator pos, size_type n, const T& x) {

        iterator b = begin();
        iterator e = end();

        if (pos>=b && pos <= e) {
           size_type c = capacity();
           size_type s = size();
		   size_type elSize = 1;//sizeof(T);
		   size_type diff = (size_type) (pos-cr_vector_ptr);
           size_type s_pos = diff/elSize;

           if(c < (s+n)) {

               size_type aloc = c;
               do{
                   aloc  = (aloc!=0)?aloc*2:1;
               }while(aloc < (s+n));

               cr_vector_ptr = (pointer) realloc(cr_vector_ptr, (aloc)*sizeof(T));
				if(cr_vector_ptr == (pointer)0)
				{
					printf("ERROR:: Unable to Malloc memory in file %s at line %d\n",__FILE__,__LINE__);
				}

               aloc_capacity = aloc;
           }

           size_type rem = (s-s_pos);
           if (rem != 0)
           {
               char* src = (char*) (cr_vector_ptr + s_pos);
               char* dst = (char*) src + n*sizeof(T);
               size_type bytes = rem*sizeof(T);
               memmove(dst, src, bytes);
           }

           pointer ptr = cr_vector_ptr + s_pos;
           for(size_type i = 0; i < n; i++)
           {
			   new((void *)(ptr)) value_type(x);
               ptr++;
           }

           aloc_size = s+n;
        }
    }

    CR_NO_THROW inline void insert(iterator pos, iterator f, iterator l) {
        iterator b = begin();
        iterator e = end();

		size_t diff = pos - begin();
		
        if (pos>=b && pos<= e) {
            size_type c = capacity();
            size_type s = size();
            size_type n = (l - f);
            
            if (c < (s+n)) {

               size_type aloc = c;
               do{
                   aloc  = (aloc!=0)?aloc*2:1;
               }while(aloc < (s+n));

               cr_vector_ptr = (pointer) realloc(cr_vector_ptr, (aloc)*sizeof(T));
				if(cr_vector_ptr == (pointer)0)
				{
					printf("ERROR:: Unable to Malloc memory in file %s at line %d\n",__FILE__,__LINE__);
				}

               aloc_capacity = aloc;

			   pos = begin() + diff;
            }

            for(iterator i = f; i!=l; i++)
            {
                insert (pos++, 1, *i);
            }
        }
    }

    CR_NO_THROW inline iterator erase(iterator pos) {
        return erase(pos, pos);
    }

    CR_NO_THROW inline iterator erase(iterator first, iterator last) {
        iterator b = begin();
        iterator e = end();

        if ((first<=last) && (first>=b) && (first <= e) && (last>=b) && (last<=e)) {
            size_type s = size();
			size_type elSize = 1;//sizeof(T)
            size_type s_pos = (size_type)(first - cr_vector_ptr)/elSize; 
            size_type e_pos = (size_type)(last - cr_vector_ptr)/elSize;
            size_type n = (s_pos==e_pos)?1:(e_pos - s_pos);
            size_type rem   = s - e_pos;

            if (rem) {
                char * dst = (char*) (cr_vector_ptr + s_pos);
			
				/* Call destructor & free the pointer */
				for(size_t i=0; i < n; i++)
					(cr_vector_ptr+s_pos+i)->~T();

                size_type bytes = (rem-1)*sizeof(T);
                char * src  = dst + n*sizeof(T);
                memmove(dst, src, bytes);
            }

            aloc_size-=n;
        }

        return first;
    }

    CR_NO_THROW inline void clear(void) {
		if(cr_vector_ptr != (pointer)0)
		{
            /* Call destructor & free the pointer */
			for(size_t i=0; i < aloc_size; i++)
				(cr_vector_ptr+i)->~T();

			free(cr_vector_ptr);
		}

		cr_vector_ptr = (pointer) NULL;
		aloc_capacity = (size_type)0;
		aloc_size     = (size_type)0;
    }

    CR_NO_THROW inline cr_vector& operator=(const cr_vector& aVec) {
		//check if not self
		if( this != &aVec)
		{
			//First clear
			clear();

			size_type s = aVec.size();
			reserve(s);

			for(size_type j=0; j < s; j++)
			{
				new((void *)(cr_vector_ptr+j)) value_type(aVec[j]);
			}

			aloc_size = s;
		}
        return *this;
    }

protected:
    CR_NO_THROW inline void set_capacity(size_type c) {
        aloc_capacity = c;
    }

    CR_NO_THROW inline void set_size(size_type s) {
        aloc_size = s; 
    }

    CR_NO_THROW inline void set_pointer(pointer ptr) {
        cr_vector_ptr = ptr;
    }

    CR_NO_THROW inline pointer get_pointer(void) {
        return cr_vector_ptr;
    }

	static const size_type npos = (size_type) NPOS_LENGTH;

    //private methods.
private: 

    //private member variables.
private:
    pointer     cr_vector_ptr;
    size_type   aloc_capacity;
    size_type   aloc_size;
};

template <class T>
CR_NO_THROW bool operator==(const cr_vector<T>& a, const cr_vector<T>& b) {
    typedef cr_vector<T> _Self;
    typedef typename _Self::size_type size_type; 
    typedef typename _Self::const_iterator  const_iterator;

    size_type s1 = a.size();
    size_type s2 = b.size();

    bool result = false;
    int r = 1;

    if (s1 == s2)
    {
        const_iterator j = b.begin();
        result = true;
        for (const_iterator i = a.begin(); i != a.end(); i++, j++)
        {
            r = memcmp(i, j, sizeof(T));  

            if (r != 0)
            {
                result = false;
                break;
            }
        }
    }

    return result;
}

template <class T>
CR_NO_THROW bool operator!=(const cr_vector<T>& a, const cr_vector<T>& b) {
    return !(a==b);
}

template <class T>
CR_NO_THROW bool operator<(const cr_vector<T>& a, const cr_vector<T>& b) {
    typedef cr_vector<T> _Self;
    typedef typename _Self::size_type size_type; 
    typedef typename _Self::const_iterator  const_iterator;

    size_type s1 = a.size();
    size_type s2 = b.size();

    bool result = false;
    int r = -1;

    const_iterator i;
    if (s1 < s2)
    {
        const_iterator j = b.begin();
        for (i = a.begin(); i != a.end(); i++, j++)
        {
            if (*i >= *j)
            {
                break;
            }
        }

        if (i == a.end())
        {
            result = true;
        }
    }
    else
    {
        const_iterator j = a.begin();
        for (i = b.begin(); i != b.end(); i++, j++)
        {
            size_type s = sizeof(T);

            if (*j >= *i)
            {
                result = false;
                break;
            }
        }

        if (s1 != 0)
        {
            if (i == b.end())
            {
                result = true;
            }
        }
    }


    return result;
}

template <class T>
CR_NO_THROW bool operator>(const cr_vector<T>& a, const cr_vector<T>& b) {
    typedef cr_vector<T> _Self;
    typedef typename _Self::size_type size_type; 
    typedef typename _Self::const_iterator  const_iterator;

    size_type s1 = a.size();
    size_type s2 = b.size();

    bool result = false;
    int r = 1;

    const_iterator i;
    if (s1 < s2)
    {
        const_iterator j = b.begin();
        for (i = a.begin(); i != a.end(); i++, j++)
        {
            if (*i <= *j)
            {
                break;
            }
        }

        if (i == a.end())
        {
            result = true;
        }
    }
    else
    {
        const_iterator j = a.begin();
        for (i = b.begin(); i != b.end(); i++, j++)
        {
            if (*j <= *i)
            {
                result = false;
                break;
            }
        }

        if (s1 != 0)
        {
            if (i == b.end())
            {
                result = true;
            }
        }

    }

    return result;
}

} //end of namespace osal_stl

#endif //__CR_VECTOR_H_
