/* Copyright (C) 2002 Jean-Marc Valin */
/*
   @file ltp_altivec.c
   @brief Long-Term Prediction functions (altivec version)
*/
/*
   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   - Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.

   - Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.

   - Neither the name of the Xiph.org Foundation nor the names of its
   contributors may be used to endorse or promote products derived from
   this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR
   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
   EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
   PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "bench.h"
#include "memalign.h"
// #define IACA_MARKS_OFF // undef for analysis by iaca-lin32/bin/iaca.sh
//#include "iacaMarks.h"
#include "altivec2avx.h"


static float cinner_prod(float *a, float *b, int len)
{
  int i;
  float res = 0;

  for (i = 0; i < len; i++)
    res += a[i] * b[i];

  return res;
}


static float SSEinner_prod(float *x, float *y, int k)
{
  typedef float v4sf __attribute__ ((vector_size(16)));   // floating point vector type
  float inner_product = 0.0, temp[4];
  v4sf acc, X, Y;                  // 4x32-bit float registers
  int i;

  acc = __builtin_ia32_xorps(acc, acc); // zero the accumulator
  for (i = 0; i < (k - 3); i += 4) {
    X = __builtin_ia32_loadups(&x[i]); // load groups of four floats
    Y = __builtin_ia32_loadups(&y[i]);
    acc = __builtin_ia32_addps(acc, __builtin_ia32_mulps(X, Y));
  }

  __builtin_ia32_storeups(temp, acc); // add the accumulated values
  inner_product = temp[0] + temp[1] + temp[2] + temp[3];
  for (; i < k; i++)               // add up the remaining floats
    inner_product += x[i] * y[i];

  return inner_product;
}


static float inner_prod(float *a, float *b, int len)
{
  int i;
  float sum;

  int a_aligned = (((unsigned long)a) & 15) ? 0 : 1;
  int b_aligned = (((unsigned long)b) & 15) ? 0 : 1;

  __vector float MSQa, LSQa, MSQb, LSQb;
  __vector unsigned char maska, maskb;
  __vector float vec_a, vec_b;
  __vector float vec_result;

  vec_result = (__vector float)vec_splat_u8(0);

  if ((!a_aligned) && (!b_aligned)) {
    // This (unfortunately) is the common case.
    maska = vec_lvsl(0, a);
    maskb = vec_lvsl(0, b);

    MSQa = vec_ld(0, a);
    MSQb = vec_ld(0, b);

    for (i = 0; i < len; i+=8) {

      a += 4;
      LSQa = vec_ld(0, a);
      vec_a = vec_perm(MSQa, LSQa, maska);

      b += 4;
      LSQb = vec_ld(0, b);
      vec_b = vec_perm(MSQb, LSQb, maskb);

      vec_result = vec_madd(vec_a, vec_b, vec_result);

      a += 4;
      MSQa = vec_ld(0, a);
      vec_a = vec_perm(LSQa, MSQa, maska);

      b += 4;
      MSQb = vec_ld(0, b);
      vec_b = vec_perm(LSQb, MSQb, maskb);

      vec_result = vec_madd(vec_a, vec_b, vec_result);

    }
  } else if (a_aligned && b_aligned) {

    for (i = 0; i < len; i+=8) {
      vec_a = vec_ld(0, a);
      vec_b = vec_ld(0, b);
      vec_result = vec_madd(vec_a, vec_b, vec_result);
      a += 4; b += 4;
      vec_a = vec_ld(0, a);
      vec_b = vec_ld(0, b);
      vec_result = vec_madd(vec_a, vec_b, 
			    vec_result);
      a += 4; b += 4;
    }

  } else if (a_aligned) {
    maskb = vec_lvsl(0, b);
    MSQb = vec_ld(0, b);

    for (i = 0; i < len; i+=8) {

      vec_a = vec_ld(0, a);
      a += 4;

      b += 4;
      LSQb = vec_ld(0, b);
      vec_b = vec_perm(MSQb, LSQb, maskb);

      vec_result = vec_madd(vec_a, vec_b, vec_result);

      vec_a = vec_ld(0, a);
      a += 4;

      b += 4;
      MSQb = vec_ld(0, b);
      vec_b = vec_perm(LSQb, MSQb, maskb);

      vec_result = vec_madd(vec_a, vec_b, vec_result);
    }
  } else if (b_aligned) {
    maska = vec_lvsl(0, a);
    MSQa = vec_ld(0, a);

    for (i = 0; i < len; i+=8) {

      a += 4;
      LSQa = vec_ld(0, a);
      vec_a = vec_perm(MSQa, LSQa, maska);

      vec_b = vec_ld(0, b);
      b += 4;

      vec_result = vec_madd(vec_a, vec_b, vec_result);

      a += 4;
      MSQa = vec_ld(0, a);
      vec_a = vec_perm(LSQa, MSQa, maska);

      vec_b = vec_ld(0, b);
      b += 4;

      vec_result = vec_madd(vec_a, vec_b, vec_result);
    }
  }

  vec_result = vec_add(vec_result, vec_sld(vec_result, vec_result, 8));
  vec_result = vec_add(vec_result, vec_sld(vec_result, vec_result, 4));
  vec_ste(vec_result, 0, &sum);

  return sum; 
}


void make_data(float *a, float *b, int n)
{
  int i;
  float scale = 1 / sqrt(n);
  for (i = 0; i < n; i++) {
    a[i] = (float)(i+1) * scale;
    b[i] = scale / (i+1);
  }
}


int main(int nargs, char *argv[])
{
  int i, n = 32;
  long besttime = 0;
  float *a, *b, r;

  if (nargs > 1)
    n = atoi(argv[1]);

  a = (float*)MALLOC(n * sizeof(float));
  b = (float*)MALLOC(n * sizeof(float));
  make_data(a, b, n);

  for (i = 0; i < BENCH_ITS; i++) {
    time_val t0, t1;
    long thistime;
    t0 = get_cycles();
    r = inner_prod(a, b, n);
    t1 = get_cycles();
    thistime = t1 - t0;
    if (i == 0 || thistime < besttime)
      besttime = thistime;
  }
  printf("Altivec result=%f cycles=%ld\n", r, besttime);
  
  for (i = 0; i < BENCH_ITS; i++) {
    time_val t0, t1;
    long thistime;
    t0 = get_cycles();
    r = cinner_prod(a, b, n);
    t1 = get_cycles();
    thistime = t1 - t0;
    if (i == 0 || thistime < besttime)
      besttime = thistime;
  }
  printf("      C result=%f cycles=%ld\n", r, besttime);

  for (i = 0; i < BENCH_ITS; i++) {
    time_val t0, t1;
    long thistime;
    t0 = get_cycles();
    r = SSEinner_prod(a, b, n);
    t1 = get_cycles();
    thistime = t1 - t0;
    if (i == 0 || thistime < besttime)
      besttime = thistime;
  }
  printf("    SSE result=%f cycles=%ld\n", r, besttime);

  free(b);
  free(a);
  return 0;
}
