//
// H264 Hardware Encoder
// Copyright (c) 2010 Intel Corporation. All Rights Reserved.
//

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <getopt.h>
#include <X11/Xlib.h>

#include <unistd.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <va/va.h>
#include <va/va_x11.h>

#include "h264enc.h"

#define CHECK_VASTATUS(va_status,func)                                         \
	if (va_status != VA_STATUS_SUCCESS) {                                      \
		fprintf(stderr,"%s:%s (%d) failed,exit\n", __func__, #func, __LINE__); \
		exit(1);                                                               \
	}//else fprintf(stderr,"%s:%s (%d) OK\n", __func__, #func, __LINE__);

static unsigned int frame_width = 0;
static unsigned int frame_height = 0;
static unsigned int frame_rate = 30;
static unsigned int intra_count = 30;
static unsigned int frame_bitrate = 8000000;
static unsigned int initial_qp = 24;
static unsigned int minimal_qp = 1;

enum {
	SH_LEVEL_1=10,
	SH_LEVEL_1B=11,
	SH_LEVEL_2=20,
	SH_LEVEL_3=30,
	SH_LEVEL_31=31,
	SH_LEVEL_32=32,
	SH_LEVEL_4=40,
	SH_LEVEL_5=50
};
static unsigned char level_idc = SH_LEVEL_3;

static Display *x11_display;
static VAConfigID config_id;
static VADisplay va_display;
static VAContextID context_id;

#define NUM_SURFACES 3//1 source, 1 reconstructed, 1 reference
static VASurfaceID surfaces[NUM_SURFACES];
static VASurfaceID src_surface_id, rec_surface_id, ref_surface_id;
static VABufferID coded_buf_id;


static void load_yuyv_frame(unsigned char *inbuf, VASurfaceID surface_id)
{
	VAImage image;
	VAStatus va_status;
	void *pbuffer=NULL;
	unsigned char *psrc = inbuf;
	unsigned char *pdst = NULL;
	unsigned char *dst_y, *dst_uv;
	unsigned char *src_u, *src_v;
	unsigned char *dst_uv_line = NULL;
	int i,j;

	va_status = vaDeriveImage(va_display, surface_id, &image);
	va_status = vaMapBuffer(va_display, image.buf, &pbuffer);
	pdst = (unsigned char *)pbuffer;

	dst_uv_line = pdst + image.offsets[1];
	dst_uv = dst_uv_line;
	for (i=0; i<frame_height; i+=2){
		dst_y = (pdst + image.offsets[0]) + i*image.pitches[0];
		for (j=0; j<(frame_width/2); ++j){
			*(dst_y++) = psrc[0];//y1;
			*(dst_uv++) = psrc[1];//u;
			*(dst_y++) = psrc[2];//y1;
			*(dst_uv++) = psrc[3];//v;
			psrc+=4;
		}

		dst_y = (pdst + image.offsets[0]) + (i+1)*image.pitches[0];
		for (j=0; j<frame_width/2; ++j){
			*(dst_y++) = psrc[0];//y1;
			*(dst_y++) = psrc[2];//y2;
			psrc+=4;
		}

		dst_uv_line += image.pitches[1];
		dst_uv = dst_uv_line;
	}

	va_status = vaUnmapBuffer(va_display, image.buf);
	va_status = vaDestroyImage(va_display, image.image_id);
}


static int save_h264_frame(int fd, VABufferID buffer_id)
{
	void *pdata = NULL;
	int data_size;
	int data_offset;
	int w_size;
	VAStatus va_status;

	va_status = vaMapBuffer(va_display, buffer_id, &pdata);
	CHECK_VASTATUS(va_status,vaMapBuffer)
	if (pdata == NULL){
		printf("%d:%s: Critical problem: pdata==0!\n", __LINE__, __func__);
		va_status = vaUnmapBuffer(va_display, buffer_id);
		CHECK_VASTATUS(va_status,vaUnmapBuffer)
		return 0;
	}

	data_size = *((unsigned long *) pdata); //first DWord is the coded video size
	if (data_size == 0){
		printf("%d:%s: Critical problem: data_size==0!\n", __LINE__, __func__);
		va_status = vaUnmapBuffer(va_display, buffer_id);
		CHECK_VASTATUS(va_status,vaUnmapBuffer)
		return 0;
	}
	data_offset = *((unsigned long *) (pdata + 4)); //second DWord is byte offset

	w_size = write(fd, pdata+data_offset, data_size);
	if (w_size != data_size)
		fprintf(stderr, "Trying to write %d bytes, but actual %d bytes\n", data_size, w_size);

	va_status = vaUnmapBuffer(va_display, buffer_id);
	CHECK_VASTATUS(va_status,vaUnmapBuffer)

	return 1;
}

static int fd_h264 = -1;
void open_output_h264_file(const char *fname)
{
	fd_h264 = open(fname, O_CREAT|O_TRUNC|O_RDWR, S_IRUSR|S_IWUSR | S_IRGRP|S_IWGRP | S_IROTH);
	if (fd_h264 == -1){
		printf("Open file %s failed, exit\n", fname);
		exit(1);
	}
}

void close_output_h264_file()
{
	if (fd_h264 != -1)
		close(fd_h264);
}


int encode_frame(unsigned char *inbuf)
{
	VAEncPictureParameterBufferH264 pic_param;
	VAEncSliceParameterBuffer slice_param;
	VAStatus va_status;
	VABufferID pic_param_buf_id, slice_param_buf_id;

	VASurfaceStatus surface_status;
	static int i=0;

	load_yuyv_frame(inbuf, src_surface_id);

	va_status = vaBeginPicture(va_display, context_id, src_surface_id);
	CHECK_VASTATUS(va_status,vaBeginPicture)

	if (i==0) {
		VABufferID seq_param_buf_id;
		VAEncSequenceParameterBufferH264 seq_param = {0};

		seq_param.seq_parameter_set_id = 0;
		seq_param.level_idc = level_idc;
		seq_param.picture_width_in_mbs = frame_width / 16;
		seq_param.picture_height_in_mbs = frame_height / 16;
		seq_param.bits_per_second = frame_bitrate;
		seq_param.frame_rate = frame_rate;
		seq_param.initial_qp = initial_qp;
		seq_param.min_qp = minimal_qp;
		seq_param.basic_unit_size = 0;
		seq_param.intra_period = intra_count;
		seq_param.vui_flag = 0;

		va_status = vaCreateBuffer(va_display, context_id, VAEncSequenceParameterBufferType,
			sizeof(seq_param), 1, &seq_param, &seq_param_buf_id);
		CHECK_VASTATUS(va_status,vaCreateBuffer)
		va_status = vaRenderPicture(va_display, context_id, &seq_param_buf_id, 1);
		CHECK_VASTATUS(va_status,vaRenderPicture)
	}

	pic_param.reference_picture = ref_surface_id;
	pic_param.reconstructed_picture= rec_surface_id;
	pic_param.coded_buf = coded_buf_id;
	pic_param.picture_width = frame_width;
	pic_param.picture_height = frame_height;
	pic_param.last_picture = 0;
	va_status = vaCreateBuffer(va_display, context_id, VAEncPictureParameterBufferType,
		sizeof(pic_param), 1, &pic_param, &pic_param_buf_id);
	CHECK_VASTATUS(va_status,vaCreateBuffer)
	va_status = vaRenderPicture(va_display, context_id, &pic_param_buf_id, 1);
	CHECK_VASTATUS(va_status,vaRenderPicture)

	//one frame, one slice
	slice_param.start_row_number = 0;
	slice_param.slice_height = frame_height/16; //Measured by MB
	slice_param.slice_flags.bits.is_intra = ((i % intra_count) == 0);
	slice_param.slice_flags.bits.disable_deblocking_filter_idc = 0;
	va_status = vaCreateBuffer(va_display, context_id, VAEncSliceParameterBufferType,
		sizeof(slice_param), 1, &slice_param, &slice_param_buf_id);
	CHECK_VASTATUS(va_status,vaCreateBuffer)
	va_status = vaRenderPicture(va_display, context_id, &slice_param_buf_id, 1);
	CHECK_VASTATUS(va_status,vaRenderPicture)

	va_status = vaEndPicture(va_display, context_id);
	CHECK_VASTATUS(va_status,vaEndPicture)
	va_status = vaSyncSurface(va_display, src_surface_id);
	CHECK_VASTATUS(va_status,vaSyncSurface)

	surface_status = 0;
	va_status = vaQuerySurfaceStatus(va_display, src_surface_id, &surface_status);
	CHECK_VASTATUS(va_status,vaQuerySurfaceStatus)
	if (!(surface_status & VASurfaceReady))
		printf("WARNING!! vaSyncSurface() is not VASurfaceReady!\n");

	printf("encode frame:%d\n", i);
	if (!save_h264_frame(fd_h264, coded_buf_id))
		return 0;

	//if a frame is skipped, current frame still use last reference frame
	if ((surface_status & VASurfaceSkipped) == 0) {
		// swap ref/rec
		VASurfaceID tmp = rec_surface_id;
		rec_surface_id = ref_surface_id;
		ref_surface_id = tmp;
	} else
		printf("==skipped frame %d\n", i);
	++i;
	return 1;
}

void encoder_init(unsigned int width, unsigned int height)
{
	VAStatus va_status;
	int major_ver, minor_ver;

	VAEntrypoint entrypoints[5];
	int num_entrypoints,slice_entrypoint;
	VAConfigAttrib attrib[2];

	unsigned int codedbuf_size;

	frame_width = width;
	frame_height = height;

	//Init VAAPI
	x11_display = XOpenDisplay(":0.0");
	va_display = vaGetDisplay(x11_display);
	va_status = vaInitialize(va_display, &major_ver, &minor_ver);
	CHECK_VASTATUS(va_status,vaInitialize)

	attrib[0].type = VAConfigAttribRTFormat;
	attrib[0].value = VA_RT_FORMAT_YUV420;
	attrib[1].type = VAConfigAttribRateControl;
	attrib[1].value = VA_RC_VBR;

	va_status = vaCreateConfig(va_display, VAProfileH264Baseline, VAEntrypointEncSlice, attrib, 2, &config_id);//needed to create context
	CHECK_VASTATUS(va_status,vaCreateConfig)
	va_status = vaCreateSurfaces(va_display, frame_width, frame_height, VA_RT_FORMAT_YUV420, NUM_SURFACES, surfaces);//needed to read source data
	CHECK_VASTATUS(va_status,vaCreateSurfaces)
	va_status = vaCreateContext(va_display, config_id, frame_width, frame_height, VA_PROGRESSIVE, surfaces, NUM_SURFACES, &context_id);
	CHECK_VASTATUS(va_status,vaCreateContext)

	src_surface_id = surfaces[0];
	//the last two frames are reference/reconstructed frame
	rec_surface_id = surfaces[NUM_SURFACES - 1];
	ref_surface_id = surfaces[NUM_SURFACES - 2];

	codedbuf_size = (frame_width * frame_height * 400) / (16*16);
	va_status = vaCreateBuffer(va_display, context_id, VAEncCodedBufferType, codedbuf_size, 1, NULL, &coded_buf_id);
	CHECK_VASTATUS(va_status,vaCreateBuffer)
}

void encoder_close()
{
	VAStatus va_status;
	va_status = vaDestroyBuffer(va_display, coded_buf_id);
	CHECK_VASTATUS(va_status,vaDestroyBuffer)
	va_status = vaDestroyContext(va_display, context_id);
	CHECK_VASTATUS(va_status,vaDestroyContext)
	va_status = vaDestroySurfaces(va_display, surfaces, NUM_SURFACES);
	CHECK_VASTATUS(va_status,vaDestroySurfaces)
	va_status = vaDestroyConfig(va_display, config_id);
	CHECK_VASTATUS(va_status,vaDestroyConfig)
	va_status = vaTerminate(va_display);
	CHECK_VASTATUS(va_status,vaTerminate)
	XCloseDisplay(x11_display);
}

