
/** @file bindings.js
 *  @brief Bindings infrastructure
 *  @details Bindings is common method to attach C++ based functionality
 *    to JScript infrastructure.
 *    Pure JScript program has no ways to communicate with
 *    operating system, but using bindings such functionality may
 *    be added to program
 *  @usage
 *    var c = required("component.js");
 *    var comp = c.Component(component_id);
 *  @see Wizard, Storage, Registry, RegistryKey, System, Origin, Cache
 */

/** @class Storage
 *  @brief Storage - storage representation object
 *  @details Storage is tree based data structure. Every node has set of
 *    attributes: name, value, child elements. Also every node may be stored
 *    or loaded to/from database to save information between sessions.
 *  @attr string name      - readonly, name of current node, string
 *  @attr data value       - read/write, value of current node, string, number
 *  @attr array childs     - readonly, array of names of child nodes
 *  @attr node parent      - readonly, parent node (if exists)
 *  @attr string path      - readonly, path to current node, see NOTES
 *  @attr bool initialized - readonly, boolean value, true if current node or one of subnodes
 *                           was initialized by any value, else - false
 *  @note Storage is simple way to save structural data. Every node of storage may
 *    save simple value (integer, string), save this data into database and later
 *    restore it.
 *    Every node has name. By name node may be obtained from parent node, for example:
 *    <pre>
 *      var my_storage = parent("my node name");
 *    </pre>
 *    In this example node 'my_storage' is accessed by 'my node name' id of parent node.
 *    But 'my_storage' node may have child nodes too, for example:
 *    <pre>
 *      var one_more_storage = my_storage("one more storage");
 *    </pre>
 *    These two examples may be bit changed: for example you don't need middle storage
 *    element, and to avoid creation of it you may write code above like:
 *    <pre>
 *      var one_more_storage = parent("my node name")("one more storage");
 *    </pre>
 *    or
 *    <pre>
 *      var one_more_storage = parent("my node name::one more storage");
 *    </pre>
 *    Node name 'my node name::one more storage' is concatenation of 'my node name' and
 *    'one more storage' names using symbol '::' as delimiter. This way allows access
 *    to any child element directly to any level without creating middle objects.
 *    There is no special API to create storage object. Such objects are created
 *    automatically on first access to it.
 *  @usage
 *    var s = Storage("my storage"); // get 'my storage' element
 *    var c = s("child element");    // get 'my storage::child element'
 *    var t = Storage("my storage::child element"); // get 'my storage::child element' directly
 *    c.value = 123;                 // set value 123
 *    var t = Storage();             // get root element
 */

/** @fn Storage(string id)
 *  @brief function to create storage object
 *  @details Creates root storage element
 *  @return root element of storage or element for specified id
 *  @param string id optional id or path to node to create or access
 *  @usage
 *    var s = Storage("my storage"); // get 'my storage' element
 *    var t = Storage();             // get root element
 */

/** @method Storage Read(string key)
 *  @brief read storage from database
 *  @details Read storage value and all child elements from database.
 *  @note Read function doesn't change name of current node, it updates
 *    only value and child elements
 *  @param string key database key to read, see Write method for details
 *  @usage
 *    var s = Storage("my storage"); // get 'my storage' element
 *    s.Read("key in database");
 *  @see Write
 */

/** @method Storage Write
 *  @brief write storage to database
 *  @details Write storage value and all child elements into database.
 *    key argument is string used as unique identification in database
 *    to restore saved value by Read function. In case if any key already
 *    exists in database - such data is overwritten
 *  @param string key database key to write
 *  @usage
 *    var s = Storage("my storage"); // get 'my storage' element
 *    s.value = "123 string";
 *    s.Write("key in database");
 *  @see Read
 */

/** @method Storage Clear
 *  @brief clear current element and detach it from parent node
 *  @details clear current element and detach it from parent node
 *  @usage
 *    var s = Storage("my storage"); // get 'my storage' element
 *    s.Clear();
 *  @see Write Read
 */


/** @class Filesystem
 *  @brief file system operations representation object
 *  @details Filesystem object includes set of methods to operate with file system
 *  @attr string exe_dir readonly, directory where executable is located
 *  @attr string exe_name readonly, module name of executable (without path)
 */

/** @method Filesystem SpecialFolder(integer id)
 *  @brief get system special folder value
 *  @details Get special folder path value. Windows shell stores set of well known directories
 *    which may be used in installation scenario.
 *  @param integer id identificator of directory, see NOTES for details
 *  @return string special directory value
 *
 *  @attr string system              windows system32 directory
 *  @attr string windows             windows directory
 *  @attr string program_files       program files directory (program files (x86) on 64-bit systems)
 *  @attr string desktop             common desktop directory
 *  @attr string start_menu_programs common menu programs directory
 *  @attr string start_menu_startup  common startup directory
 *  @attr string start_menu          common start menu directory
 *  @attr string app_data            common application data directory
 *  @attr string program_files_x86   program files x86 directory
 *  @attr string common              program files/common files directory (x86 on 64-bit systems)
 *  @attr string common_x86          program files (x86)/common files directory
 *
 *  @note To get directory value used SHGetSpecialFolderPath system API. Argument 'id'
 *    of SpecialFolder method is same as csidl argument, for details lee
 *    MSDN documentation for SHGetSpecialFolderPath function at:
 *    http://msdn.microsoft.com/en-us/library/windows/desktop/bb762204(v=vs.85).aspx
 *  @usage
 *    var s = FileSystem.SpecialFolder(37); // get system directory
 *    var s2 = FileSystem.SpecialFolder.system; // get system directory too
 */

/** @method Filesystem GetTemp
 *  @brief get system temp directory
 *  @details Returns path to windows temp directory
 *  @return string path to windows temp directory
 *  @usage
 *    var t = FileSystem.GetTemp();
 */

/** @method Filesystem CreateDirectory(string path)
 *  @brief create directory
 *  @param string path path to directory to create
 *  @details Creates directory. In case if parent directory doesn't exists - it
 *    will be created too
 *  @return boolean true  on success
 *  @return boolean false on failure
 *  @note If target directory already exists - function returns true
 *  @usage
 *    var t = FileSystem.CreateDirectory("c:\\my_dir\\my_subdir");
 */

/** @method Filesystem IsDirectory(string path)
 *  @brief check if path is directory
 *  @param string path path to directory or file to evaluate
 *  @details Function checks for target path is directory
 *  @return boolean true target path is directory
 *  @return boolean false target path is not directory (file or link)
 *  @usage
 *    if(FileSystem.IsDirectory("c:\\my_dir\\my_subdir"))
 *    {
 *        // do something here...
 *    }
 */

/** @method Filesystem FindFiles(string path, string mask)
 *  @brief find files in directory by mask
 *  @param string path - path to directory to scan
 *  @param string mask - pattern to search
 *  @return array Function returns array of file names matched mask
 *  @details Function search files in specified directory (but not in subdirectories) and
 *    returns array of file names matched mask
 *  @usage
 *    var files = FileSystem.FindFiles("c:\\windows", "*.dll");
 *  @see FindFilesRecursive MakePath
 */

/** @method Filesystem FindFilesRecursive
 *  @brief find files in directory and all subdirectories by mask
 *  @param string path - path to directory to scan
 *  @param string mask - pattern to search
 *  @details Function search files in specified directory and in subdirectories and
 *    returns array of file names matched mask.
 *    Returned file names include relative to input directory path
 *  @return array Function returns array of file names matched mask.
 *  @usage
 *    var files = FileSystem.FindFilesRecursive("c:\\windows", "*.dll");
 *  @see FindFiles MakePath
 */

/** @method Filesystem MakePath(string rpath, string base)
 *  @brief normalize path
 *  @param string rpath - relative path
 *  @param string base  - base path
 *  @details Function normalizes path. To describe how this function work here is simple example:
 *    <pre>
 *      var path = FileSystem.MakePath("system32", "c:\\windows");
 *    </pre>
 *    result of this call is string 'c:\windows\system32'.
 *    In most of cases function just concatenates base & relative paths, additionaly there is
 *    evaluation for slashes & etc.
 *  @return string Normalized path, string
 *  @note Returned file names includes relative to input directory path
 *  @usage
 *    var path = FileSystem.MakePath("system32", "c:\\windows"); // result is c:\windows\system32
 *  @see FindFiles FindFilesRecursive
 */

/** @method Filesystem Directory(string path)
 *  @brief get nearest directory for path
 *  @param string path - path to analyze
 *  @details Function returns nearest directory for path. In case if input path is directory
 *    then path itself is returned. If input path is file - parent directory will be
 *    returned.
 *  @return string Path to directory, string
 *  @usage
 *    var p1 = FileSystem.Directory("c:\\windows\\system32"); // result is c:\windows\system32
 *    var p2 = FileSystem.Directory("c:\\windows\\system32\\cmd.exe"); // result is c:\windows\system32
 *  @see Parent
 */

/** @method Filesystem Same(string path1, string path2)
 *  @brief check if two different paths point to same object (directory or file)
 *  @param string path1 - path to compare
 *  @param string path2 - path to compare
 *  @details Two different paths may point to same file or directory, but these paths may
 *    have different notations, for example absolute path & relative path. This
 *    function check if target object (file or directory) is same for both paths.
 *  @return boolean true  - both paths specify same object
 *  @return boolean false - paths specify different objects
 *  @note In case if one (or both) paths doesn't exist - function always returns 'false'
 *  @usage
 *    if(FileSystem.Same(p1, p2))
 *    {
 *        // it is same object
 *    }
 */

/** @method Filesystem Parent(string path)
 *  @brief get parent directory
 *  @param string path - path to get parent directory
 *  @details Function returns parent directory for specified path
 *  @return string String, parent directory, or null when parent directory is not available (for example for c: drive)
 *  @usage
 *    var parent = FileSystem.Parent("c:\\windows"); // result is c:\
 *  @see Directory
 */

/** @method Filesystem Size(string path)
 *  @brief calculate size of file or directory
 *  @param string path - path to file or directory to analyze
 *  @details Function returns size of file (in case if path specifies file) or summary size of all
 *    files in directory & all subdirectories
 *  @return number Number, size of files in bytes
 *  @usage
 *    var size = FileSystem.Size("c:\\windows");
 *  @see FindFiles FindFilesRecursive
 */

/** @method Filesystem ReadFileUTF8(string file)
 *  @brief read UTF8 text file and it's content
 *  @param string file - path to file
 *  @details This function is useful for GUI processing: some RTF documents may be
 *    loaded by this function and passed to appropriate control using
 *    Wizard/Notify method.
 *  @return string String, content of file
 *  @usage
 *    var text = FileSystem.ReadFileUTF8("c:\\program\\eula.rtf");
 *  @see Notify
 */

/** @method Filesystem Exists(string file)
 *  @brief check for target path exists
 *  @param string file - path to file or directory
 *  @details This function check if target file or directory exists
 *  @return boolean true  - file or directory exists
 *  @return boolean false - file or directory doesn't exist
 *  @usage
 *    if(FileSystem.Exists("c:\\program\\eula.rtf"))
 *    {
 *        // file exists
 *    }
 */

/** @class Getopt
 *  @brief command line parameters representation object
 *  @details Every program may accept set of command line switches during startup.
 *    This object provides easy way to access to such switches.
 *
 *  @note Command line parameters may have different notations. In current implementation
 *    used the most general scheme, may be processed all notations below:
 *    <pre>
 *      --switch=value
 *      --switch="space containing value"
 *      --switch value
 *      --switch "space containing value"
 *      --switch
 *
 *      -switch=value
 *      -switch="space containing value"
 *      -switch value
 *      -switch "space containing value"
 *      -switch
 *    </pre>
 */

/** @method Getopt Exists(string id)
 *  @brief check if any command line switch was specified by user
 *  @details Check if any command line switch was specified by user
 *  @param string id - command line switch name to evaluate
 *  @return boolean true - command line switch was specified
 *  @return boolean false - command line switch was not specified
 *  @usage
 *    if(GetOpt.Exists("param"))
 *    {
 *        // param argument was specified, for example:
 *        //     setup.exe --param
 *        // or
 *        //     setup.exe --param=value
 *    }
 *  @see Get
 */

/** @method Getopt Get(string id)
 *  @brief get command line value
 *  @details Every command line argument has name and may have value, for example:
 *    <pre>
 *      --my_argument=my_value
 *    </pre>
 *    Value of command line switch may be obtained by argument name.
 *  @param string id - command line switch name to get
 *  @return string value of command line switch or empty string
 *  @usage
 *    if(GetOpt.Exists("param"))
 *    {
 *        var arg = GetOpt.Get("my_argument");
 *    }
 *  @see Exists GetDefault
 */

/** @method Getopt GetDefault(string id, string default)
 *  @brief get command line value, in case if switch absent - use default value
 *  @details Every command line argument has name and may have value, for example:
 *    <pre>
 *      --my_argument=my_value
 *    </pre>
 *    Value of command line switch may be obtained by argument name.
 *  @param string id      - command line switch name to evaluate
 *  @param string default - default value, returned in case if switch is absent
 *  @return string value of command line switch or default value if switch is absent
 *  @usage
 *    var arg = GetOpt.GetDefault("my_argument", "my_default_value");
 *  @see Exists Get
 */


/** @class RegistryKey
 *  @brief windows registry manipulation object
 *  @details RegistryKey object provides ability to manipulate windows registry
 *  @see Registry
 */

/** @fn Registry(string root, string key)
 *  @brief constructor for RegistryKey object
 *  @details Creates RegistryKey object
 *  @param string root - string, top level registry key, should be one of:
 *    <pre>
 *      "HKLM" - local machine hive
 *      "HKCU" - current user hive
 *      "HKCR" - classes root hive
 *      "HKU"  - users hive
 *    </pre>
 *  @param string key - registry key name
 *  @return RegistryKey RegistryKey object
 *  @note Registry function doesn't create registry key, it just create virtual
 *    object witch may manipulate by attached registry key (create, remove)
 *  @usage
 *    var reg_key = Registry("HKLM", "Software\\MyProgram");
 */

/** @method RegistryKey Exists
 *  @brief check if attached registry key exists
 *  @details Function check if attached registry key exists
 *  @return boolean true  - if registry key exists
 *  @return boolean false - if registry key doesn't exist
 *  @usage
 *    var reg_key = Registry("HKLM", "Software\\MyProgram");
 *    if(reg_key.Exists())
 *    {
 *        // key exists
 *    }
 *  @see Registry
 */

/** @method RegistryKey Create
 *  @brief create attached registry key
 *  @details Function creates attached registry key
 *  @return boolean true  - if registry key created
 *  @return boolean false - if registry key could not be created
 *  @note See Registry function NOTES for details
 *  @usage
 *    var reg_key = Registry("HKLM", "Software\\MyProgram");
 *    if(!reg_key.Exists())
 *    {
 *        // create key
 *        reg_key.Create();
 *    }
 *  @see Registry Exists Remove
 */

/** @method RegistryKey Remove
 *  @brief remove attached registry key
 *  @details Function removes attached registry key and all subkeys and values
 *  @return tboolean rue  - if registry key removed
 *  @return boolean false - if registry key could not be removed
 *  @usage
 *    var reg_key = Registry("HKLM", "Software\\MyProgram");
 *    if(reg_key.Exists())
 *    {
 *        // remove key
 *        reg_key.Remove();
 *    }
 *  @see Registry Exists Create
 */

/** @method RegistryKey Subkeys
 *  @brief get array of subkey names
 *  @details Function returns array of subkey names
 *  @return array Array of strings with subkeys names
 *  @usage
 *    var reg_key = Registry("HKLM", "Software");
 *    if(reg_key.Exists())
 *    {
 *        var sub = reg_key.Subkeys();
 *    }
 *  @see Registry Values
 */

/** @method RegistryKey Values
 *  @brief Values - get array of values names
 *  @details Function returns array of values names
 *  @return array Array of strings with values names
 *  @usage
 *    var reg_key = Registry("HKLM", "Software");
 *    if(reg_key.Exists())
 *    {
 *        var sub = reg_key.Values();
 *    }
 *  @see Registry Subkeys
 */

/** @method RegistryKey Value(string name, string value)
 *  @brief Value - set value to attached registry key or get value
 *  @details Set/get value to attached registry key.
 *  @param string name  - string, value name to set/get
 *  @param string value - optional, new value to write to registry value
 *  @return data If 'value' argument is omitted - function returns value of registry value :)
 *    else
 *  @return boolean true  - if value set successful
 *  @return boolean false - value could not be set
 *  @usage
 *    var reg_key = Registry("HKLM", "Software\\MyProgram");
 *    if(!reg_key.Exists())
 *    {
 *        // set value
 *        reg_key.Value("my_string_value", "new_value");
 *        reg_key.Value("my_number_value", 123);
 *    }
 *  @see Registry RemoveValue
 */

/** @method RegistryKey RemoveValue(string name)
 *  @brief RemoveValue - remove attached registry value
 *  @details Function removes value from registry key
 *  @param string name - value name to remove
 *  @return boolean true  - if registry value removed
 *  @return boolean false - if registry value could not be removed
 *  @usage
 *    var reg_key = Registry("HKLM", "Software\\MyProgram");
 *    if(reg_key.Exists())
 *    {
 *        // remove value
 *        reg_key.RemoveValue("my_value");
 *    }
 *  @see Registry Value
 */

/** @method RegistryKey Key(string key)
 *  @brief Key - create RegistryKey object attached to child windows registry key
 *  @details Function creates RegistryKey object attached to child windows registry key
 *  @param string name - registry subkey name
 *  @return RegistryKey RegistryKey object attached to child registry key, or null on fail
 *  @usage
 *    var reg_key = Registry("HKLM", "Software");
 *    if(reg_key.Exists())
 *    {
 *        var sub_key = reg_key.Key("MyProgram");
 *    }
 *  @note This method doesn't create windows registry key - it just create virtual object
 *    to manipulate with windows registry (create, remove)
 *  @see Registry
 */

/** @method RegistryKey WowRedirect(boolean redirect)
 *  @brief Enable/disable redirection of 32-bit registry keys on 64-bit system
 *  @details Enable/disable redirection of 32-bit registry keys on 64-bit system, see notes for details
 *  @param boolean redirect
 *     - true to enable redirection (default)
 *     - false to disable redirection
 *  @usage
 *    var reg_key = Registry("HKLM", "Software");
 *    reg_key.WowRedirect(true);
 *  @note When redirection is enabled on 64-bit system registry keys (for example)
 *    <pre>HKEY_LOCAL_MACHINE\\SOFTWARE\\<b>Wow6432Node</b>\\Microsoft\\VisualStudio\\10.0\\Setup</pre>
 *    and
 *    <pre>HKEY_LOCAL_MACHINE\\SOFTWARE\\Microsoft\\VisualStudio\\10.0\\Setup</pre>
 *    are same and you may use same registry keys names for all platforms, this
 *    is default behavior. But to enable access to native 64-bit registry key call method
 *    <code>WowRedirect(true);</code>
 *  @see Registry
 */


/** @class Origin
 *  @brief Information about current executing script file
 *  @details Origin object provides information about current executing script file
 */

/** @method Origin Directory
 *  @brief Directory - get directory where current executing script file located
 *  @details Function returns directory where current executing script file located
 *  @return string String, directory where current executing script file located
 *  @usage
 *    var dir = Origin.Directory();
 *  @see Origin File
 */

/** @method Origin File
 *  @brief File - get current executing script file name
 *  @details Function returns current executing script file name
 *  @return string String, current script file name
 *  @usage
 *    var file = Origin.File();
 *  @see Origin Directory
 */


/** @fn Guid
 *  @brief Guid - create guid
 *  @details Creates new GUID value
 *  @return string Function returns new GUID value string, format: {1CF29EE3-030B-40C9-84FA-7298D5477F9E}
 *  @usage
 *    var guid = Guid();
 */


/** @class System
 *  @brief System - information about system
 *  @details System object provides information about system
 */

/** @method System IsAdmin
 *  @brief IsAdmin - check if current process has administrator permissions
 *  @details Function checks if current process has administrator permissions
 *  @return boolean true  - if process has administrative permissions
 *  @return boolean false - if process has no administrative permissions
 *  @usage
 *    if(System.IsAdmin())
 *    {
 *        // launched process has administrator permissions
 *    }
 */

/** @method System ProcessorArch()
 *  @brief Get native processor architecture
 *  @return number processor architecture, equal to one of attribute
 *  @attr number pa_x86     x86 processor
 *  @attr number pa_intel64 intel64 processor
 *  @attr number pa_ia64    Itanium based processor
 *  @usage
 *    if(System.ProcessorArch() == System.ProcessorArch.pa_x86)
 *    {
 *        // this is native x86 processor
 *    }
 */

/** @method System SecondInstance()
 *  @brief Check if current instance of MICL was launched as second instance
 *  @return boolean true  - current instance is <b>NOT</b> first instance
 *  @return boolean false - current instance is first instance
 *  @usage
 *    if(System.SecondInstance())
 *    {
 *        // this instance is not unique
 *    }
 */

/** @method System ProcessList()
 *  @brief Get process list running on system
 *  @return Object process_map, where key is process id (PID), value - executable path
 *  @usage
 *    var proc_list = System.ProcessList();
 *    for(var i in proc_list)
 *    {
 *        Log("process: pid: " + i + " image: " + proc_list[i]);
 *    }
 *  @note In case if installation process has no administrative permissions then full
 *    image path could not be obtained. For such processes only executable name returned.
 */



/** @class Cache
 *  @brief Cache - information about cache
 *  @details Cache object provides information about cache
 *  @note During installation process installer may copy itself into cache directory
 *    to provide ability manipulate software when original media is not available.
 *    Cache object provides way to operate such caches.
 */

/** @method Cache ExeDir
 *  @brief ExeDir - get executable module location directory
 *  @details Get executable module location directory
 *  @return string Directory where executable module located, string
 *  @usage
 *    var exedir = Cache.ExeDir();
 *  @see Filesystem::exe_dir
 */

/** @method Cache PluginsDir
 *  @brief PluginsDir - get startup plugins directory
 *  @details Get startup plugins directory.
 *  @return string Directory where from original plugins are loaded, string
 *  @usage
 *    var plug_dir = Cache.PluginsDir();
 *  @see CachePluginsDir
 */

/** @method Cache CacheDir
 *  @brief CacheDir - get cache directory
 *  @details Get cache directory
 *  @return string Directory where cache located, string
 *  @usage
 *    var cache_dir = Cache.CacheDir();
 *  @see ExeDir
 ******
 */

/** @method Cache CachePluginsDir
 *  @brief CachePluginsDir - get cached plugins directory
 *  @details Get cached plugins directory.
 *  @return string Additional directory where installer search plugins, string
 *  @usage
 *    var plug_dir = Cache.CachePluginsDir();
 *  @see PluginsDir
 */

/** @method Cache Cached
 *  @brief Cached - check if program was started from cache directory
 *  @details Check if program was started from cache directory
 *  @return boolean true  - program was started from cache directory
 *  @return boolean false - program was started not from cache directory
 *  @usage
 *    if(Cache.Cached())
 *    {
 *        // started from cache
 *    }
 *  @see ExeDir
 */

/** @method Cache Config
 *  @brief Config - get path to configuration file
 *  @details Function returns path to configuration file
 *  @return string Path to configuration file, string
 *  @usage
 *    if(Cache.Cached())
 *    {
 *        // started from cache
 *    }
 *  @note Configuration file includes basic global configuration information. It is XML based
 *    document which may include any information common for any product line. Example of
 *    such file:
 *    <pre>
 *      <?xml version="1.0" encoding="utf-8"?>
 *      <config>
 *        <cache dir_id="35" subdir="Intel/installer/intelhaxm/cache/" noload="1"/>
 *        <registry root="HKCU" key="SOFTWARE\Intel\Installer\intelhaxm\Cache" value="{0D79780C-64E5-4C15-ADD5-D2AACEC6CA45}"/>
 *      </config>
 *    </pre>
 */


/** @class Execute
 *  @brief Execute - launcher of scripts or external commands
 *  @details Execute object provides ability to execute java scripts or external programs
 *  @note Every installation scenario may include several script files. To involve such scripts into
 *    common infrastructure Execute::JScript method should be used.
 */

/** @method Execute JScript(string file)
 *  @brief JScript - execute JScript file and return result
 *  @details Execute JavaScript scenario file and return result. Script is executed in global context
 *    of caller, all global created functions or variables will be accessible in
 *    executing script, and all global objects created in launched script will be
 *    accessible in other scripts. Function returns result of script execution.
 *  @return data Result of script execution
 *  @note There is no way to pass any arguments into launched script.
 *    Some example: script file 'my_script.js':
 *      <pre>
 *      new function()
 *      {
 *        this.property_a = 123;
 *        this.function_b = function(a)
 *        {
 *          this.property_a = a;
 *        }
 *      }
 *      </pre>
 *      this script file creates new object and set there property 'property_a' and
 *      function 'function_b'.
 *    To execute this script example code may be used:
 *    <pre>
 *      var obj = required("my_script.js");
 *    </pre>
 *    after execution this code variable 'obj' will handle new created object
 *  @see URL
 */

/** @method Execute URL(string url)
 *  @brief URL - start external program
 *  @details This function starts external application and immediately returns. External application
 *    may be executable file, internet address, etc... See ShellExecuteEx API info at
 *    http://msdn.microsoft.com/en-us/library/windows/desktop/bb762154(v=vs.85).aspx
 *  @param string url - url to execute. It may be WWW address (default browser will be
 *    launched) or application.
 *  @return boolean true  - program is started successfully
 *  @return boolean false - failed to start program
 *  @note There is no way to pass any arguments into launched program
 *  @usage
 *    if(Execute.URL("http://intel.com"))
 *    {
 *      // program started
 *    }
 *  @see JScript
 */


/** @class StringList
 *  @brief StringList - localization strings manipulation
 *  @details StringList object provides ability to load and format localized strings.
 *    Every localized string has unique ID and string content. ID is ASCII string
 *    (may include symbols, numbers and special symbols, except '$', '[', ']', ':').
 *    which used as key for searching in localization table, this key is
 *    case sensitive, for example IDs 'string' and 'String' are different.
 */

/** @method StringList Load(string path)
 *  @brief Load - load localized strings from directory and all subdirectories
 *  @param string path - path to root directory where localization files located
 *  @details Localized string information may be located in different directories. By
 *    default system loads strings from directory where executable located. But
 *    during scenario execution additional directories may be loaded to system.
 *  @return boolean true  - on success
 *  @return boolean false - on failure
 *  @note Function loads all .xmc files from specified directory and all subdirectories
 *  @usage
 *    StringList.Load("my_localization_path");
 */

/** @method StringList Get(string id, boolean no_format)
 *  @brief Get - get localized string
 *  @param string id - string id to load
 *  @param boolean no_format - optional, non-empty (non-false) value means that output string
 *     will not be formatted. For details see StringList::Format method
 *  @details Function loads localized string by string id
 *  @return string Localized string
 *  @note String id is case sensitive
 *  @usage
 *    StringList.Get("localID", false);
 *    or same:
 *    StringList.Get("localID");
 *  @see StringList Format FormatNamed
 */

/** @method StringList Format(string template, ...)
 *  @brief Format - format localized string
 *  @param string template - template string to format
 *  @param data ...        - additional values to format string, supported up to 4 additional arguments
 *  @brief Template is string which has format:
 *      <pre>
 *      'I like [stringID] to play'
 *      </pre>
 *    During processing subsequence [stringID] will be replaced by
 *    data from localization table. Additionally if expanded value
 *    of 'stringID' includes text like '[something]' - this text will be
 *    expanded too.
 *
 *    Supported several templates to modify formatted string:
 *      <pre>
 *      [:string] - expanded by value from Storage(":string")
 *      </pre>
 *      <pre>
 *      [$s] - expanded to string passed in additional arguments, similar
 *             to printf function of POSIX standard
 *      </pre>
 *      <pre>
 *      [$size] - expanded as size representation, for example 1000 number is
 *                expanded as 1Kb, used 1000 basis
 *      </pre>
 *      <pre>
 *      [$xsize] - expanded as size representation, for example 1024 number is
 *                 expanded as 1Kb, used 1024 basis
 *      </pre>
 *  @return string Formatted string
 *  @usage
 *    StringList.Format("[stringID]"); - just get localized string 'stringID'
 *    StringList.Format("My expanded string: [stringID], done"); - more complex case, stringID is
 *                                                                 expanded into another string
 *    StringList.Format("[::storage::node]"); - replaced by storage value
 *    StringList.Format("hey [$s]", "you"); - result is: 'hey you', [$s] here replaced by second
 *                                            argument value
 *    StringList.Format("hey [$s] and [$s]", "you", "me"); - result is: 'hey you and me', one
 *                                                           more argument
 *    StringList.Format("size is: [$size]", 1024 * 10); - result: 'size is 10Kb'
 *  @see StringList FormatNamed Storage
 */

/** @method StringList FormatNamed(string template, object string_map)
 *  @brief FormatNamed - format localized string using additional object for localization
 *  @param string template   - template string to format
 *  @param object string_map - additional string map
 *  @details This function is similar to StringList::Format, but it process additional
 *    ability for format string.
 *    During template processing in case if [keyword] found it try to search this
 *    keyword in string_map object and in case if it could not be found there - continue
 *    search in localization table. The simplest way to explain how it works is
 *    provide example:
 *    <pre>
 *      // creating string map object
 *      var string_map = {id1: "first string", id2: "[localizedID]"};
 *      // string map object is regular JScript object, where every attribute is string
 *      // let's assume that in localization table 'localizedID' record
 *      //   has value EXPANDED_STRING_VALUE.
 *      // now let's use this object:
 *      var output = StringList.FormatNamed("my id1 is: [id1], but id two is [id2]", string_map);
 *      // result of this function is:
 *      // 'my id1 is: first string, but id two is EXPANDED_STRING_VALUE'
 *    </pre>
 *    As you can see you may create any combinations of localization strings very easy
 *  @return string Formatted string
 *  @see StringList Format
 */


/** @class DB
 *  @brief DB - database manipulation object
 *  @details In some cases installer needs to save information to any persistent
 *    storage for later using (for example save list of installed products).
 *    Database object helps to manipulate such information.
 *    Data stored in database in tree-based container, see Storage for details.
 *    Every record in database has unique key. The main way to read/write data
 *    is using Storage object. In fact database is just Storage which is saved
 *    to file and later may be restored.
 *  @see Storage Read Write
 */

/** @method DB Connect(string directory)
 *  @brief Connect - connect to database
 *  @param string directory - path to directory where database located
 *  @details Database may be located in any directory. By default system is trying to
 *    locate database in cache directory, but this behavior may be changed by
 *    Connect function.
 *  @return boolean true  - on success
 *  @return boolean false - on failure
 *  @note Function should be called before any database related calls. Only first
 *    call of DB::Connect function takes affect, other calls are ignored.
 *  @usage
 *    DB.Connect("c:\\program Files\\Installer");
 */

/** @method DB Exists(string key)
 *  @brief Exists - check if key exists in database
 *  @param string key - key string in database
 *  @details Database saves data using 'key' identification. Key is string in format
 *    <code>'string1::string2::string3'</code>, in fact database uses same key representation
 *    as for Storage object manipulation.
 *  @return boolean true  - key exists in database
 *  @return boolean false - key doesn't exist in database
 *  @usage
 *    if(DB.Exist("string1::string2"))
 *    {
 *        // key exist
 *    }
 *  @see Storage Read Write
 */

/** @method DB Head(string key)
 *  @brief Head - get nearest existing key for specified string
 *  @param string key - key string in database to evaluate
 *  @details In general this function for internal use only, but it may be useful for others.
 *    To explain how it works let's look at example: let's assume that any script
 *    add record to database under key 'sub1::sub2::sub3', later another script
 *    is trying to get nearest existing key for string:
 *    <pre>
 *    'sub1::sub2::sub3::sub4::sub5::sub6'.
 *    </pre>
 *    Result of function will be: 'sub1::sub2::sub3::sub4'.
 *    Result will include existing key ''sub1::sub2::sub3' and one non-existing
 *    level subkey 'sub4'. This function is used to implement DB::Dumper object.
 *  @return string Function returns nearest subkey in database.
 *  @usage
 *    var storage = Storage("my_storage_key"); // create storage
 *    storage.value = "value";
 *    storage.Save("sub1::sub2::sub3"); // save data to database
 *    var head = DB::Head("sub1::sub2::sub3::sub4::sub5::sub6");
 *    // result is string 'sub1::sub2::sub3::sub4'
 *  @see Storage Storage::Read Storage::Write Exists
 */

/** @method DB Remove(string key)
 *  @brief Remove - remove specified key from database and all subkeys
 *  @param string key - key in database to remove
 *  @details To clean database DB::Remove function may be used.
 *  @return boolean true  - on success
 *  @return boolean false - on failure
 *  @usage
 *    DB::Remove("sub1::sub2::sub3");
 *  @see Storage Storage::Read Storage::Write Exists
 */

/** @method DB Value(string key)
 *  @brief Value - get value by specified key
 *  @param string key - key in database to get value
 *  @details This is helper function. It returns value of Storage saved by specified key.
 *    Let's assume that any script saved data using code:
 *    <pre>
 *      var storage = Storage("my_storage");
 *      storage.value = "my value";
 *      storage.Save("my_key");
 *    </pre>
 *    Two versions of code below will give same result:
 *    <pre>
 *      var storage = Storage("my_storage");
 *      storage.Read("my_key");
 *      var value = storage.value;
 *    </pre>
 *    and
 *    <pre>
 *      var value = DB.Value("my_key");
 *    </pre>
 *    In both cases variable 'value' will have value 'my value'
 *  @return data Value saved under specified key or null if key is not exists
 *  @usage
 *    DB::Value("sub1::sub2::sub3");
 *  @see Storage Read Write Exists
 */

/** @class XML
 *  @brief XML - xml files processing class
 *  @details In some cases XML files are useful container to save any product specific
 *    data. XML class is simple way to read such data.
 *  @attr string name - tag name for element
 *  @attr string text - text for for element
 *  @attr hash attributes - hash of attributes, for example:
 *    <pre>
 *      var a = node.attributes["attrib"];
 *    </pre>
 *    or same
 *    <pre>
 *      var a = node.attributes.attrib;
 *    </pre>
 *  @see Storage Read Write
 */

/** @fn XML(string path, string xpath)
 *  @brief Constructor for XML object. Open XML file and return XML object
 *  @param string path  - path to file to open
 *  @param string xpath - optional, XPath to apply to opened XML file
 *  @details Constructor for XML object. Open XML file and return XML object
 *  @return XML - array of XML nodes matched to xpath parameter
 *  @return XML - root node of file if xpath parameter is not specified
 *  @return XML - empty object if failed to open or process file
 *  @usage
 *    var root = XML("c:\\my_file.xml"); // return root node element
 *    or
 *    var nodes = XML("c:\\my_file.xml", "//tag_name"); // return array of 'tag_name' elements
 */


/** @method XML subnodes(string xpath)
 *  @brief Get array of subnodes matched xpath
 *  @param string xpath - XPath pattern to select subnodes
 *  @details Get array of subnodes matched xpath
 *  @return array - array of XML objects matched xapth
 *  @see node
 */

/** @method XML node(string xpath)
 *  @brief Get only one node matched xpath
 *  @param string xpath - XPath pattern to select node
 *  @details Get only one node matched xpath
 *  @return XML - XML object matched xapth
 *  @see subnodes
 */


/** @class Splash
 *  @brief Splash - binding to splash plugin
 *  @details Splash plugin provides ability to display any picture during system
 *    initialization and show any text messages
 */

/** @method Splash Show(string path)
 *  @brief Show splash screen
 *  @param string path optional, path to image to display (bmp, jpg, png)
 *  @usage
 *    Splash.Show("c:\\program files\\image.png");
 *  or
 *    Splash.Show();
 *  @see Hide
 */

/** @method Splash Hide
 *  @brief Hide splash screen
 *  @usage
 *    Splash.Hide();
 *  @see Show
 */

/** @method Splash Status(string message)
 *  @brief Show status message
 *  @param string message text of message to display, plain text
 *  @usage
 *    Splash.Status("i'm doing something");
 *  @see Title
 */

/** @method Splash Title(string message)
 *  @brief Show title text
 *  @param string message text of title to display, RTF format
 *  @note By default splash screen has no title. To enable title function
 *    MoveTitle should be called to set title position and Title method should be
 *    called before splash screen showed
 *  @usage
 *    Splash.MoveTitle(10, 40, 90, 60);
 *    Splash.Title(FileSystem.ReadFileUTF8("c:\\program files\\title.rtf"));
 *      // set content of title.rtf file to title
 *  @see Status MoveTitle
 */

/** @method Splash MoveTitle(number x1, number y1, number x2, number y2)
 *  @brief Set title control position on splash screen
 *  @param number x1 left border of text area
 *  @param number y1 top border of text area
 *  @param number x2 right border of text area
 *  @param number y2 bottom border of text area
 *  @note x* and y* coordinates are number in range 0..99.
 *    0 - left/top border of splash screen, 99 - right/bottom border
 *    of splash screen
 *  @usage
 *    Splash.MoveTitle(10, 40, 90, 60); // set title position
 *  @see Status Title MoveStatus
 */

/** @method Splash MoveStatus(number x1, number y1, number x2, number y2)
 *  @brief Set status control position on splash screen
 *  @param number x1 left border of text area
 *  @param number y1 top border of text area
 *  @param number x2 right border of text area
 *  @param number y2 bottom border of text area
 *  @note x* and y* coordinates are number in range 0..99.
 *    0 - left/top border of splash screen, 99 - right/bottom border
 *    of splash screen
 *  @usage
 *    Splash.MoveStatus(10, 90, 90, 99); // set status position
 *  @see Status Title MoveTitle
 */

/** @fn CreateProcess(string module, string arguments, boolean wait)
 *  @brief Function creates new process
 *  @details Creates new process and read output
 *  @return Object element with created process results. this object has attributes:<br>
 *    <code>string output</code> - output of created process<br>
 *    <code>int exitcode</code> - process exit code<br>
 *    <code>boolean failed</code> - in case if process failed to start value of this attribute is <code>true</code>
 *  @param string module module name for new created process, set empty string to not specify module
 *  @param string arguments command line arguments passed to new process
 *  @param boolean wait optional, default - false, true means wait for process exit and save output
 *  @usage
 *    var proc = CreateProcess("", "notepad.exe"); // launch Notepad and continue
 *    var pro2 = CreateProcess("", "cmd.exe /c dir", true); // launch 'dir' command
 *    if(!pro2.failed) // if success - log output of command
 *      Log(pro2.output);
 *  @note In case if wait parameter is not specified or equal false - <code>output</code> and
 *    <code>exitcode</code> attributes of return object are not contained any useful values
 */






