
/** @file component.js
 *  @brief component.js - basic implementation of Component object
 *  @details this module includes basic implementation of Component object.
 *    here defined action_t & state_t enumerations, Component object
 *    implementation
 *  @usage
 *    var c = required("component.js");
 *    var comp = c.Component(component_id);
 *  @see product.js
 */
new function()
{
    var ns_installer = Namespace("Root.installer");
    var base_script_dir = Origin.Directory();
    var load = function(name) {return required(FileSystem.MakePath(name, base_script_dir));};

    var ns_dump      = load("dumper.js");
    var ns_event     = load("event.js");
    var ns_enums     = load("enums.js");
    var ns_upgrade   = load("upgrade.js");
    var ns_version   = load("version.js");

    var ns = this;
    //###############################################################
    // Component constructor
    //###############################################################
    this.Create = this.Create || function (inf, src, prc, ex_init)
    {
        if(!inf)
            return null;

        var r_info = inf.GetInfo();
        if(!r_info)
            return null;

        var cmp = ns_installer.Installer.Components[r_info.Id()];

        if (!cmp) {
            cmp = new ns.Component();
            if (!cmp.Init(r_info, src, prc))
                return null;
        }

        var cln = cmp.Clone();

        if(ex_init)
            ex_init.call(cln);

        return cln;
    }
    //###############################################################
    //class Component
    //###############################################################
    /** @class Component
     *  @brief Basic Component functionality
     *  @details This is base implementation of Component functionality & should not be used
     *    in development. This implementation is basis for specific
     *    components implementation (ComponentMSI, ComponentARP, etc.)
     *  @attr string action_t - Component actions
     *  @attr string state_t  - Component states
     *  @see Product
     */

    /** @fn Component(string id)
     *  @brief Constructor for Component object
     *  @details This function creates basic Component object.
     *  @param string id - id of component to create. In case if Component
     *    with same id is created - link to this component is returned
     *  @usage
     *    var c = required("component.js");
     *    var comp = c.Component(component_id);
     *  @see Product
     */
    this.Component = this.Component || function ()
    {
        ns_enums.BindTo(this);

        this.m_component = this;

        this.obj_type = "component";
        this.Dependencies = {};
        this.Groups = [];
        this.CnfgOpts = {};
        this.OnCommit = new ns_event.Event(this);
        this.OnActionChange = new ns_event.Event(this);
    }
    //###############################################################
    // Init method
    //###############################################################
    this.Component.prototype.Init = function (inf, src, prc)
    {
        this.Log = this.Log || log_helper("Component id = " + (inf.Name() ? inf.Name() : inf.Id()) + ": ");

        this.InitInf(inf);
        this.InitSrc(src);
        this.InitPrc(prc);

        this.m_error    = ""; // error description
        this.pure       = true; // virtual component, like general product
        this.m_disabled = this.disabled_t.no;
        this.m_install_dir_base = "";
        this.m_install_dir_own = "";
        this.m_install_dir_locked = false;

        this.m_offline = false;

        this.m_apply_upgrade_done = false;
        this.m_apply_remove_done  = false;
        this.m_apply_install_done = false;

        this.m_signed = false;

        this.m_order = 100;

        //this.m_dependency_processor = null;
        this.p_action(this.action_t.none);

        var ns_cmp = this;
        ns_installer.Installer.AddResetAction(
            function ()
            {
                ns_cmp.p_action(ns_cmp.action_t.none);
                ns_cmp.m_install_dir_base = "";
                ns_cmp.m_install_dir_own = "";
                ns_cmp.m_install_dir_locked = false;
            });

        this.m_parent   = null;
        this.m_orig_obj = null; // for clone it is defined with original object

        this.dumper     = ns_dump.Dumper("dmpr_for_" + this.m_name);
        this.src_dumper = ns_dump.Dumper("src_for_" + this.m_name);
        //this.Configure();

        this.m_clones = [];

        ns_installer.Installer.AddComponent(this);

        return true;
    }
    //###############################################################
    this.Component.prototype.InitInf = function(inf)
    {
        this.m_inf      = inf;
        this.m_id       = this.m_inf.Id();
        this.m_name     = this.m_inf.Name();
        this.m_dscrpt   = this.m_inf.Description();
        this.m_error_dscrpt = this.m_inf.ErrorDescription ? this.m_inf.ErrorDescription() : "";
        this.m_version  = ns_version.Version(this.m_inf.Version());
        this.m_size     = this.m_inf.Size();

        return true;
    }
    //###############################################################
    this.Component.prototype.InitSrc = function(src)
    {
        this.m_component.m_src = src;

        return true;
    }
    //###############################################################
    this.Component.prototype.InitPrc = function(prc)
    {
        this.m_component.m_prc = prc;

        if(this.m_prc)
        {
            if(this.m_prc.Owner)
                this.m_prc.Owner(this);
            else
                Log(Log.l_warning, "No Owner method exists in processor. Legacy version?");
        }

        this.p_state( (this.m_prc && this.m_prc.State) ? this.m_prc.State() : this.state_t.absent);

        // original component can be created without processor
        // processor can be added later for clones
        // therefore state for original should be reinitialized also
        if(!this.IsOriginal())
            this.Original().p_state( (this.m_prc && this.m_prc.State) ? this.m_prc.State() : this.state_t.absent);

        return true;
    }
    //###############################################################
    this.Component.prototype.Original = function(){return this.m_orig_obj;}
    //###############################################################
    this.Component.prototype.IsOriginal = function(){return !this.m_orig_obj;}
    //###############################################################
    this.Component.prototype.Info = function(){return this.m_inf;}
    //###############################################################
    this.Component.prototype.Source = function(src)
    {
        if(src)
        {
            this.m_component.m_src = src;
            this.Log("source set: \"" + ((this.m_src && this.m_src.File) ? this.m_src.File() : "") + "\"")
            this.DoSourceRenew();
        }

        return this.m_src;
    }
    //###############################################################
    this.Component.prototype.DoSourceRenew = function(){}
    //###############################################################
    this.Component.prototype.Processor = function(prc)
    {
        if(prc)
        {
            this.Log("set new processor");
            this.InitPrc(prc);
            this.DoProcessorRenew();
        }

        return this.m_prc;
    }
    //###############################################################
    this.Component.prototype.DoProcessorRenew = function(){}
    //###############################################################
    this.Component.prototype.Offline = function (val)
    {
        if(arguments.length > 0 )
            if(val)
            {
                this.m_component.m_offline = true;
                this.DoOfflineSet();
            }

        return this.m_offline;
    }
    //###############################################################
    this.Component.prototype.DoOfflineSet = function()
    {
        if(this.State() == this.state_t.absent && (!this.Source() || !this.Source().Resolved()))
            this.Disabled(true);
    }
    //###############################################################
    this.Component.prototype.Id = function(){return this.m_id;}
    //###############################################################
    this.Component.prototype.Name = function (nm)
    {
        if(nm) this.m_component.m_name = nm;

        return this.m_name;
    }
    //###############################################################
    this.Component.prototype.Description = function (dscr)
    {
        if(dscr) this.m_component.m_dscrpt = dscr;

        return this.m_dscrpt;
    }
    //###############################################################
    this.Component.prototype.ErrorDescription = function (dscr)
    {
        if(dscr) this.m_component.m_error_dscrpt = dscr;

        return this.m_error_dscrpt;
    }
    //###############################################################
    this.Component.prototype.Order = function (ord)
    {
        if(typeof(ord) == "number")
            this.m_order = ord;
        
        return this.m_order;
    }
    //###############################################################
    this.Component.prototype.Parent = function (prn)
    {
        if(prn) this.m_parent = prn;

        return this.m_parent;
    }
    //###############################################################
    this.Component.prototype.Root = function ()
    {
        var root = this;

        for (var parent = this.m_parent; parent; root = parent, parent = parent.Parent());

        return root;
    }
    //###############################################################
    this.Component.prototype.Clones = function (){return this.m_clones;}
    //###############################################################
    // method Configure
    //###############################################################
    /*
    this.Component.prototype.Configure = function ()
    {
        this.m_size = this.m_inf.Size();
        this.m_error = ""; // error description
        this.pure = true; // virtual component, like general product
        this.m_disabled = this.disabled_t.no;
        this.p_state(this.state_t.absent);
        this.p_action(this.action_t.none);
        this.m_install_dir_base = "";
        this.m_install_dir_own = "";
        this.m_install_dir_locked = false;
        this.m_version = ns_version.Version();
        //this.m_dependency_processor = null;

        var ns_cmp = this;
        ns_installer.Installer.AddResetAction(
            function ()
            {
                ns_cmp.p_action(ns_cmp.action_t.none);
                ns_cmp.m_install_dir_base = "";
                ns_cmp.m_install_dir_own = "";
                ns_cmp.m_install_dir_locked = false;
            });
    }
    */
    //###############################################################
    // Add configuration option
    //###############################################################
    this.Component.prototype.AddCnfgOpt = function (opt, val)
    {
        if(!opt)
            return false;

        if(this.Disabled() == this.disabled_t.yes)
        {
            this.Log("AddCnfgOpt: component is disabled -> cnfg option can't be added");
            return true;
        }

        this.m_component.CnfgOpts[opt] = val;
        return true;
    }
    //###############################################################
    // configuration options
    //###############################################################
    this.Component.prototype.ConfigurationOptions = function ()
    {
        return this.CnfgOpts;
    }
    //###############################################################
    //FormCmdOptions method definition
    //###############################################################
    this.Component.prototype.FormCmdOptions = function (opts)
    {
        if(!opts)
            return "";

        var params = "";

        for(var opt in opts)
        {
            var o = opts[opt];
            if(typeof(o) == "function")
                var val = safecall(function(){return o();});
            else
                var val = o;

            if(typeof(val) != "undefined")
            {
                val = val.toString();
                if(val.match(/\s/)) // if it contains spaces
                    val = '"' + val + '"';

                params += opt + "=" + val + " ";
            }
        }

        return params;
    }
    //###############################################################
    // Add component to specified Group
    //###############################################################
    this.Component.prototype.AddToGroup = function (grp_id)
    {
        if(!grp_id || this.p_state() != this.state_t.installed)
            return false;
        this.Log("@@@ add to group " + grp_id);
        if(!ns_installer.Installer.Groups[grp_id])
            ns_installer.Installer.Groups[grp_id] = {};

        var grp = ns_installer.Installer.Groups[grp_id];
        grp[this.m_id] = this;

        this.Groups.push(grp_id);

        return true;
    }
    //###############################################################
    // Detach
    //###############################################################
    this.Component.prototype.Detach = function ()
    {
        this.Log("Detaching begin");
        if(this.Parent())
        {
            this.Parent().RemoveComponent(this.Id());
            this.m_parent = null;
        }

        if(this.IsOriginal())
            for(var i in this.m_clones)
                this.m_clones[i].Detach();
        this.Log("Detaching end");
        return true;
    }
    //###############################################################
    // Checking for already installed previous versions which can be upgraded
    //###############################################################
    this.Component.prototype.CheckForUpgrade = function ()
    {
        if(this.Disabled() == this.disabled_t.yes)
        {
            this.Log("CheckForUpgrade: component is disabled -> check isn't required");
            return true;
        }

        if(this.p_state() == this.state_t.installed)
        {
            this.Log("CheckForUpgrade: component is installed -> check isn't required");
            return false;
        }

        this.Log("CheckForUpgrade: begin");

        this.Upgrade.Check();

        this.Log("CheckForUpgrade: completed");

        return true;
    }
    //###############################################################
    // Checking for already installed previous versions which can be upgraded
    //###############################################################
    this.Component.prototype.UpgradeState = function ()
    {
        if(this.Disabled() == this.disabled_t.yes)
        {
            this.Log("UpgradeState: component is disabled -> none to upgrade");
            return this.upgrade_state_t.none;
        }

        if(this.Action() == this.action_t.none || this.Action() == this.action_t.remove)
        {
            this.Log("UpgradeState: component action is none/remove -> upgrade_state_t.none");
            return this.upgrade_state_t.none;
        }

        return this.Upgrade.State();
    }
    //###############################################################
    // Setting of dependencies processor
    //###############################################################
    this.Component.prototype.ProcessDependency = function (obj)
    {
        //this.Log("it is default dependency processor");
        return true;
    }
    //###############################################################
    // Dependencies setting
    //###############################################################
    this.Component.prototype.Depend = function (alias, obj)
    {
        var cur_cmp = this;
        obj.OnActionChange.Connect(function(changed_obj){ return cur_cmp.ProcessDependency(changed_obj);});
        this.Dependencies[alias] = obj;
    }
    //###############################################################
    // it is responsible for setting/getting component's base install dir
    //###############################################################
    /** @method Component InstallDirBase(string directory)
     *  @brief InstrallDirBase - set/get base installation directory
     *  @details set/get base installation directory. In case if input parameter
     *    omitted - returns current installation directory
     *  @param string directory - optional, new installation directory
     *  @return string in case if input parameter is ommitted function returns
     *    value of base installation directory, string
     *  @usage
     *    var c = Component.InstrallDirBase();
     *    // or
     *    Component.InstrallDirBase(directory);
     *  @see InstallDirOwn InstallDir LockInstallDir
     */
     this.Component.prototype.InstallDirBase = function ()
    {
        if(this.Disabled() == this.disabled_t.yes)
        {
            this.Log("InstallDirBase: component is disabled -> no installdir");
            return "";
        }

        if(this.m_orig_obj)
            return this.m_orig_obj.InstallDirBase(arguments[0]);

        if(arguments[0] && this.m_install_dir_locked != true)
        {
            this.Log("Setting installdirBase = " + arguments[0]);
            this.m_install_dir_base = arguments[0];
            this.DoInstallDirBaseChange(arguments[0]);
        }

        return this.m_install_dir_base;
    }
    //###############################################################
    // following method is for definition by inheritors
    //###############################################################
     this.Component.prototype.DoInstallDirBaseChange = function (val){return true;}
    //###############################################################
    // it is reasponsible for setting/getting component's own install dir
    //###############################################################
    /** @method Component InstallDirOwn(string directory)
     *  @brief InstrallDirOwn - set/get own installation subdirectory
     *  @details set/get own installation subdirectory. In case if input parameter
     *    omitted - returns current installation subdirectory
     *  @param string directory - optional, new installation subdirectory
     *  @return string in case if input parameter is ommitted function returns
     *    value of own installation subdirectory, string
     *  @usage
     *    var c = Component.InstrallDirOwn();
     *    // or
     *    Component.InstrallDirOwn(directory);
     *  @see InstallDirBase InstallDir LockInstallDir
     */
    this.Component.prototype.InstallDirOwn = function ()
    {
        if(this.Disabled() == this.disabled_t.yes)
        {
            this.Log("InstallDirOwn: component is disabled -> no installdir");
            return "";
        }

        if(this.m_orig_obj)
            return this.m_orig_obj.InstallDirOwn(arguments[0]);

        if(arguments[0] && this.m_install_dir_locked != true)
        {
            this.m_install_dir_own = arguments[0];
            this.DoInstallDirOwnChange(arguments[0]);
        }

        return this.m_install_dir_own;
    }
    //###############################################################
    // following method is for definition by inheritors
    //###############################################################
     this.Component.prototype.DoInstallDirOwnChange = function (val){return true;}
    //###############################################################
    // it is reasponsible for getting component's install dir
    //###############################################################
    /** @method Component InstallDir
     *  @brief InstrallDir - get complete installation subdirectory
     *  @details get complete installation subdirectory.
     *  @return string Function returns value of complete installation directory, string
     *  @usage
     *    var c = Component.InstrallDir();
     *  @see InstallDirBase InstallDirOwn LockInstallDir
     */
    this.Component.prototype.InstallDir = function ()
    {
        if(this.Disabled() == this.disabled_t.yes)
        {
            this.Log("InstallDir: component is disabled -> no installdir");
            return "";
        }

        if(this.m_orig_obj)
            return this.m_orig_obj.InstallDir();

        if(this.m_install_dir_own != "")
            return FileSystem.MakePath(this.m_install_dir_own, this.m_install_dir_base);

        return this.m_install_dir_base;
    }
    //###############################################################
    // it is responsible for locking component's base and own install dirs
    //###############################################################
    /** @method Component LockInstallDir(boolean lock)
     *  @brief LockInstallDir - lock/unlock installation directory (base & own)
     *  @details lock/unlock installation directory (base & own)
     *    If installation directory is locked - all requests to update
     *      base or own installation directory are ignored
     *  @param boolean lock - lock installation directory.
     *           non-empty value - lock directory
     *           empty value - unlock installation directory
     *  @usage
     *    Component.LockInstallDir(true);
     *  @see InstallDirBase InstallDirOwn InstallDir
     */
    this.Component.prototype.LockInstallDir = function ( val )
    {
        if(this.m_orig_obj)
            return this.m_orig_obj.LockInstallDir(val);

        if(Boolean(val))
            this.m_install_dir_locked = true;
        else
            this.m_install_dir_locked = false;
    }
    //###############################################################
    // p_state to store/receive state property
    //###############################################################
    this.Component.prototype.p_state = function (v)
    {
        if(v) this.m_state = v;

        return this.m_state;
    }
    //###############################################################
    // p_action to store/receive action property
    //###############################################################
    this.Component.prototype.p_action = function (v)
    {
        if(v) this.m_action = v;

        return this.m_action;
    }
    //###############################################################
    // Disabled to store/receive disabled property
    //###############################################################
    this.Component.prototype.Disabled = function (val)
    {
        //clone can't disable original component
        //if(this.m_orig_obj)
        //    return this.m_orig_obj.Disabled(val);

        if(val != undefined)
        {
            this.Log("Setting disabled " + val);

            if(val == this.disabled_t.mix || val)
            {
                this.Action(this.action_t.none);
                this.m_disabled = this.disabled_t.yes;
            }
            else
                this.m_disabled = this.disabled_t.no;
        }
        return this.m_disabled;
    }
    //###############################################################
    this.Component.prototype.Signed = function(sign)
    {
        if(arguments.length)
            this.m_component.m_signed = sign;

        return this.m_signed;
    }
    //###############################################################
    // Disabled to store/receive disabled property
    //###############################################################
    this.Component.prototype.Version = function (ver)
    {
        if(this.m_orig_obj)
            return this.m_orig_obj.Version(arguments[0]);

        //if(!arguments.length)
        //    return this.m_version;

        if(ver)
            this.m_component.m_version = ns_version.Version(ver);

        return this.m_version;
    }
    //###############################################################
    // TestRemove to process shared components
    //###############################################################
    this.Component.prototype.TestRemove = function ()
    {
        this.Log("TestRemove: check if this component can be removed");
        if(this.Disabled() == this.disabled_t.yes)
        {
            this.Log("TestRemove: component is disabled -> can't remove");
            return false;
        }

        if(this.IsOriginal())
            return (this.State() == this.state_t.installed) ? true : false;

        if(this.m_orig_obj.State() == this.state_t.installed)
        {
            // we should check if there are other clients for this component before removing it
            // we considering only clones with none empty Parent
            this.Log("TestRemove: check all clones ...");
            for(var i = 0, cln; cln = this.m_clones[i]; i++)
                if( cln.Parent() &&
                    (cln.GetState() == this.state_t.installed && cln.GetAction() != this.action_t.remove) ||
                    (cln.GetState() == this.state_t.absent    && cln.GetAction() == this.action_t.install)
                    )
                {
                    this.Log("TestRemove: clone " + i + " state = " + cln.GetState() + " action = " + cln.GetAction() + " - > can't be removed due to there are other clients!");
                    return false; // component can't be removed
                }
                else
                {
                    if(!cln.Parent())
                        this.Log("TestRemove: clone " + i + " is orphaned (parent is undefined)!");
                    else
                        this.Log("TestRemove: clone " + i + " state = " + cln.GetState() + " action = " + cln.GetAction() + "!");
                }
        }
        else
        {
            this.Log("TestRemove: component isn't installed -> component can't be removed");
            return false;
        }
        this.Log("TestRemove: there isn't any clients which install or has this component installed -> component can be removed");
        return true;//component can be removed
    }
    //###############################################################
    // TestInstall to process shared components
    //###############################################################
    this.Component.prototype.TestInstall = function ()
    {
        if(this.Disabled() == this.disabled_t.yes)
        {
            this.Log("TestInstall: component is disabled -> can't install");
            return false;
        }

        if( (this.m_orig_obj && this.m_orig_obj.State() == this.state_t.absent) ||
            (this.IsOriginal() && this.State() == this.state_t.absent)
          )
          return true;

        return false;
    }
    //###############################################################
    this.Component.prototype.ApplyUpgrade = function ()
    {
        if(this.m_apply_upgrade_done)
        {
            this.Log("ApplyUpgrade: was already done -> skip");
            return true;
        }

        this.m_apply_upgrade_done = true;

        if(this.Disabled() == this.disabled_t.yes)
        {
            this.Log("ApplyUpgrade: component is disabled -> can't upgrade");
            return true;
        }

        return this.DoApplyUpgrade();
    }
    //###############################################################
    this.Component.prototype.DoApplyUpgrade = function ()
    {
        this.Log("DoApplyUpgrade begin");
        if(this.p_action() == this.action_t.install && this.TestInstall())
            this.Upgrade.Apply();
        this.Log("DoApplyUpgrade end");
        return true;
    }
    //###############################################################
    this.Component.prototype.ApplyRemove = function (dmp)
    {
        this.Log("ApplyRemove begin");
        if(this.m_apply_remove_done)
        {
            this.Log("ApplyRemove: was already done -> skip");
            return true;
        }

        this.m_apply_remove_done = true;

        /*
        if(this.Disabled() == this.disabled_t.yes)
        {
            this.Log("ApplyRemove: component is disabled -> can't remove");
            return true;
        }
        */
        return this.DoApplyRemove(dmp);
    }
    //###############################################################
    this.Component.prototype.DoApplyRemove = function (dmp)
    {
        this.Log("DoApplyRemove begin");
        if (this.Action() == this.action_t.remove && this.TestRemove()) // check that there isn't any other clients for this component and it can be removed
        {
            this.Log("DoApplyRemove: " + this.Name()  + " will be removed");

            var act = null;

            if(this.m_prc && this.m_prc.RemoveAct)
            {
                act = this.m_prc.RemoveAct(this);
                if(!act)
                {
                    this.Log("DoApplyRemove: component can't be removed due to component processor didn't return any action");
                    return false;
                }
            }
            else
            {
                this.Log("DoApplyRemove: component can't be removed due to component processor isn't defined or it doesn't has method RemoveAct");
                return false;
            }


            if(dmp && dmp.IsDumper)
            {
                var d_act = this.dumper.AddAction(act, "remove " + this.Name());
                d_act.Attribute("countable", true);
                d_act.Attribute("name", this.Name());
                dmp.AddAction(this.dumper,"dmpr_" + this.Name());
            }
            else
            {
                this.Log("DoApplyRemove: Can't schedule actions - input dumper is undefined or not a dumper (!dmp.IsDumper)");
                return false;
            }
        }
        this.Log("DoApplyRemove end");
        return true;
    }
    //###############################################################
    this.Component.prototype.ApplyInstall = function (dmp)
    {
        if(this.m_apply_install_done)
        {
            this.Log("ApplyInstall: was already done -> skip");
            return true;
        }

        this.m_apply_install_done = true;

        if(this.Disabled() == this.disabled_t.yes)
        {
            this.Log("ApplyInstall: component is disabled -> can't install");
            return true;
        }

        this.DoApplyResolveSrc(this.dumper);
        return this.DoApplyInstall(dmp);
    }
    //###############################################################
    this.Component.prototype.DoApplyInstall = function (dmp)
    {
        if(this.Action() == this.action_t.install)
        {
            this.Log("DoApplyInstall: " + this.Name()  + " will be installed");

            var act = null;

            if(this.m_prc && this.m_prc.InstallAct)
            {
                act = this.m_prc.InstallAct(this);
                if(!act)
                {
                    this.Log("DoApplyInstall: component can't be installed due to component processor didn't return any action");
                    return false;
                }
            }
            else
            {
                this.Log("DoApplyInstall: component can't be installed due to component processor isn't defined or it doesn't has method InstallAct");
                return false;
            }

            if(dmp && dmp.IsDumper)
            {
                var d_act = this.dumper.AddAction(act, "install " + this.Name());
                d_act.Attribute("countable", true);
                d_act.Attribute("name", this.Name());
                dmp.AddAction(this.dumper,"dmpr_" + this.Name());
            }
            else
            {
                this.Log("DoApplyInstall: Can't schedule actions - input dumper is undefined or not a dumper (!dmp.IsDumper)");
                return false;
            }
        }
        return true;
    }
    //###############################################################
    this.Component.prototype.ApplyResolveSrc = function (dmp)
    {
        if(this.Disabled() == this.disabled_t.yes)
        {
            this.Log("ApplyResolveSrc: component is disabled -> resolve src isn't required");
            return true;
        }

        return this.DoApplyResolveSrc(dmp);
    }
    //###############################################################
    this.Component.prototype.DoApplyResolveSrc = function (dmp)
    {
        if (this.Action() == this.action_t.install)
        {
            Log("Component.DoApplyResolveSrc started: " + this.Name());
            if(!this.m_src)
            {
                this.Log(Log.l_error, "DoApplyResolveSrc: Source is undefined - nothing to resolve!");
                return false;
            }

            var pre_build_op = this.m_src.Resolve();

            if(pre_build_op)
            {
                if(dmp && dmp.IsDumper)
                {
                    pre_build_op.Group("Download");
                    pre_build_op.Attribute("name", this.Name());
                    dmp.AddAction(pre_build_op, "resolving_sources_for" + this.Name());
                }
                else
                {
                    this.Log("DoApplyResolveSrc: Can't schedule actions - input dumper is undefined or not a dumper (!dmp.IsDumper)");
                    return false;
                }
            }
            else
            {
                this.Log("DoApplyResolveSrc: Nothing actions required for sources resolution");
            }
        }
        return true;
    }
    //###############################################################
    this.Component.prototype.PreAction = function () { return this.dumper.PreAction(); }
    //###############################################################
    this.Component.prototype.PostAction = function () { return this.dumper.PostAction(); }
    //###############################################################
    //this.Component.prototype.OnCommit = function () { }
    this.Component.prototype.Commit = function ()
    {
        if(this.Disabled() == this.disabled_t.yes)
        {
            this.Log("Commit: component is disabled -> can't commit");
            return true;
        }

        if(this.DoCommit())
        {
            this.OnCommit.Call();
            return true;
        }

        return false;
    }
    //###############################################################
    this.Component.prototype.DoCommit = function ()
    {

        this.Log("DoCommit: " + this.Name()  + " will be committed");

        if(this.m_prc)
        {
            return this.m_prc.Commit(this);
        }
        else
        {
            this.Log("DoCommit: component can't be committed due to component processor isn't defined");
            return false;
        }

        return true;
    }
    //###############################################################
    this.Component.prototype.Action = function (act)
    {
        if(act)
            return this.SetAction(act);

        return this.GetAction();
    }
    //###############################################################
    /** @method Component SetAction(action_t action)
     *  @brief Component.SetAction - set desired action to Component
     *  @details set desired installation action to Component
     *  @param action_t action - property of action_t type
     *  @return boolean true -  if succeed
     *  @return boolean false - if failed
     *  @usage
     *    Component.SetAction(Component.action_t.install);
     *  @see GetAction, action_t
     */
    this.Component.prototype.SetAction = function (act)
    {
      if(!act)
        return false;

      if(act && act != this.action_t.remove && this.Disabled() == this.disabled_t.yes)
      {
        this.Log("SetAction: only 'remove' action can be set for disabled component -> action '" + act + "' can't be set");
        return false;
      }

      if(act == this.p_action())
      {
            this.Log("SetAction: current action \"" + act + "\" will not be changed due to it is already the same as input act = \""+ act +"\"");
            return true;
      }

      if(this.IsOriginal()) // orig_obj is a link to original obj
      {
        this.Log("Performing SetAction (act = " + act + ") for original component -> this action will be passed to all clones");
        for(var i = 0, cln; cln = this.m_clones[i]; i++)
            cln.SetAction(act);
      }

      this.Log("SetAction: act = \"" + act + "\"");

      if( (this.p_state() == this.state_t.installed && act == this.action_t.install) ||
          (this.p_state() == this.state_t.absent && act == this.action_t.remove)
        )

      {
        if(this.p_action() == this.action_t.none)
        {
            this.Log("SetAction: component state = " + this.p_state() + ", current action = \"none\" -> it will not be changed");
            return true;
        }

        this.Log("SetAction: component state = " + this.p_state() + " -> action will be set as \"none\"");

        this.p_action(this.action_t.none);
        this.OnActionChange.Call();

        return true;
      }

      this.p_action(act);
      this.OnActionChange.Call();

      return true;
    }
    //###############################################################
    this.Component.prototype.State = function (st)
    {
        if(st)
            return this.SetState(st);

        return this.GetState();
    }
    //###############################################################
    /** @method Component GetAction
     *  @brief Component.SetAction - get Component action
     *  @details get Component action
     *  @return action_t returns one of action_t attributes
     *  @usage
     *    var act = Component.GetAction();
     *  @see SetAction action_t
     */
    this.Component.prototype.GetAction = function ()
    {
    /*
      if(this.Disabled() == this.disabled_t.yes)
      {
        this.Log("GetAction: component is disabled therefore action = action_t.none.");
        return this.action_t.none;
      }
    */
      return this.p_action();
    }
    //###############################################################
    this.Component.prototype.SetState = function (st)
    {
      if(!st)
        return false;

      if(this.IsOriginal()) // orig_obj is a link to original obj
      {
        this.Log("Performing SetState (st = " + st + ") for original component -> this state will be passed to all clones");
        for(var i = 0, cln; cln = this.m_clones[i]; i++)
            cln.SetState(act);
      }

      this.p_state(st);
      return true;
    }
    //###############################################################
    /** @method Component GetState
     *  @brief Component.GetState - get Component state
     *  @details get Component state
     *  @return action_t returns one of state_t attributes
     *  @usage
     *    var stat = Component.GetState();
     *  @see GetAction state_t
     */
    this.Component.prototype.GetState = function ()
    {
      return this.p_state();
    }
    //###############################################################
    /** @method Component Size
     *  @brief Component.Size - get Component size
     *  @details request disk space required to process desired action
     *  @return number disk space required to process action, bytes
     *  @usage
     *    var size = Component.Size();
     *  @see GetAction GetState
     */
    this.Component.prototype.Size = function (val)
    {
      if(this.Disabled() == this.disabled_t.yes)
      {
        this.Log("Size: component is disabled therefore size = 0");
        return 0;
      }

      if(val)
        this.m_component.m_size = val;
      //var state = this.GetState();
      //var act   = this.GetAction();
      //if( state == this.state_t.absent && act == this.action_t.install &&
      //    ( this.IsOriginal() || (this.m_orig_obj && this.m_orig_obj.GetState() == this.state_t.absent)) )
      return this.m_size;
    }
    //###############################################################
    this.Component.prototype.Clone = function ()
    {
        //this.Log("Generate Clone");
        function mid(obj)
        {
            this.m_orig_obj   = obj;
            this.cln_state  = obj.State();
            this.cln_action = obj.Action();
            this.m_parent   = null; // clone has own parent
            this.m_disabled = obj.Disabled(); // clone has own disabled property
            this.m_order    = obj.Order();

            this.Log = log_helper("Clone " + obj.Log.Prefix());
        }

        mid.prototype = this;

        var clone = function()
        {
            this.Upgrade  = new ns_upgrade.Upgrade(this);
            this.OnCommit = new ns_event.Event(this);
            this.OnActionChange = new ns_event.Event(this);
            this.Dependencies = {};
        }
        clone.prototype = new mid(this);
        //###############################################################
        // p_state to store/receive state property
        //###############################################################
        clone.prototype.p_state = function (v)
        {
            if(v) this.cln_state = v;

            return this.cln_state;
        }
        //###############################################################
        // p_action to store/receive action property
        //###############################################################
        clone.prototype.p_action = function (v)
        {
            if(v) this.cln_action = v;

            return this.cln_action;
        }
        //###############################################################
        clone.prototype.Clone = function(){ return this.m_orig_obj.Clone();}
        //###############################################################
        var new_clone = new clone();
        this.m_clones.push(new_clone);
        return new_clone;
    }
    //###############################################################
    // RestorePoint method definition
    //###############################################################
    this.Component.prototype.RestorePoint = function (st)
    {
        var rp = st ? st : Storage("*");

        rp("id").value = this.m_id;
        rp("name").value = this.m_name;
        rp("description").value = this.m_dscrpt;
        rp("obj_type").value = this.obj_type;
        rp("install_dir_base").value = this.m_install_dir_base;
        rp("install_dir_own").value = this.m_install_dir_own;
        rp("version").value = this.m_version.Str();

        var groups = rp("groups");
        for(var i in this.Groups)
            groups(i).value = this.Groups[i];

        return rp;
    }
}// namespace Root.component

//##########################################################
// Backup
//##########################################################
/*
    //###############################################################
    // method InitFromRP
    //###############################################################
    this.Component.prototype.InitFromRP = function (rp)
    {
        if(!rp || rp("id").value == undefined)
            return false;

        this.m_id = rp("id").value;
        this.Log = this.Log || log_helper("Component id = " + id + ": ");
        this.m_name = rp("name").value;
        this.m_dscrpt = rp("description").value;

        var groups = rp("groups");
        for(var i in groups.childs)
                this.AddToGroup(groups(groups.childs[i]).value);

        this.Configure();

        this.m_install_dir_base = rp("install_dir_base").value;
        this.m_install_dir_own = rp("install_dir_own").value;
        this.m_version = ns_version.Version(rp("version").value);

        this.m_clones = [];

        ns_installer.Installer.AddComponent(this);
        return true;
    }
*/
