/** @file component.js
 *  @brief component.js - basic implementation of Component object
 *  @details this module includes basic implementation of Component object.
 *    here defined action_t & state_t enumerations, Component object
 *    implementation
 *  @usage
 *    var c = required("component.js");
 *    var comp = c.Component(component_id);
 *  @see product.js
 */
new function()
{
    var base_script_dir = Origin.Directory();
    var load = function(name) {return required(FileSystem.MakePath(name, base_script_dir));};

    var ns_installer = load("installer.js");
    var ns_dump      = load("dumper.js");
    var ns_event     = load("event.js");
    var ns_enums     = load("enums.js");
    var ns_upgrade   = load("upgrade3.js");
    var ns_version   = load("version.js");
    var ns_dir       = load("cascade_dir.js");
    var ns_prop_set  = load("property_set.js");
    var ns_prop      = load("property.js");
    var ns_cnfg      = load("configurator.js");
    var ns_cont      = load("container.js");

    var blank_f = function(){return ""};
    
    var P = function(val){return ns_prop.Property(val);}
    var ConstP = function(val){return ns_prop.Constant(val);}
    var PBool = function(val)
    {
      var p = ns_prop.Property(val);
      p.Transform = function(val){ return val ? true : false; }  
      return p;
    }

    var FilterNotEmpty = function(val)
    {
        if(typeof(val) == undefined || val == null) 
            return false;
        
        return true;
    }

    var PNotEmpty = function(val)
    {
      var p = ns_prop.Property(val);
      p.Filter = FilterNotEmpty;
      return p;
    }

    var PNumber = function(val)
    {
      var p = ns_prop.Property(val);
      p.Filter = function(val){ return (typeof(val) == "number" ? true : false); }
      return p;
    }

    var ns = this;
    //###############################################################
    // Component constructor
    // input hash object has following fields:
    //  Mandatory:
    //      Info
    //  Optional:
    //      Source
    //      Processor
    //      StateManager
    //      ExInit - callback which is called for created component for additional initialization
    //               as ExInit.call(component); 
    //###############################################################
    this.Create = function (_in) 
    {
        if(!_in)
            return null;
        
        if(!_in.Info)
            return null;

        var r_info = _in.Info.GetInfo();
        if(!r_info || !r_info.Id || !r_info.Id())
        {
            Log(Log.l_error, "Attempt to create component with undefined Id - input info isn't defined or doesn't have Id or Id() is empty");
            return null;
        }

        var cmp = ns_installer.Installer.Components[r_info.Id()];

        var args = {};
        
        for(var i in _in)
            args[i] = _in[i];

        args.Info = r_info;

        if (!cmp)
        {
            cmp = ns.Component(args);

            if(!cmp)
                return null;
        }

        var cln = cmp.Clone();
        
        //if(_in.ExInit)
        //    _in.ExInit.call(cln);

        return cln;
    }
    //###############################################################
    //class Component
    //###############################################################
    /** @class Component
     *  @brief Basic Component functionality
     *  @details This is base implementation of Component functionality & should not be used
     *    in development. This implementation is basis for specific
     *    components implementation (ComponentMSI, ComponentARP, etc.)
     *  @attr string action_t - Component actions
     *  @attr string state_t  - Component states
     *  @see Product
     */

    /** @fn Component(string id)
     *  @brief Constructor for Component object
     *  @details This function creates basic Component object.
     *  @param string id - id of component to create. In case if Component
     *    with same id is created - link to this component is returned
     *  @usage
     *    var c = required("component.js");
     *    var comp = c.Component(component_id);
     *  @see Product
     */

    function BindPrivate(obj)
    {
        obj.CloneId = P();
        obj.Configurator = ConstP(ns_cnfg.ComponentConfigurator(obj));
        //###############################################################
        obj.IsOriginal = function(){return obj.Original() == obj;}
        //###############################################################
        obj.Parent = P();
        //###############################################################
        obj.Order = PNumber(100);
        //###############################################################
        // State to store/receive state property
        //###############################################################
        obj.State = P();
        obj.State.Get = function()
        {
            if(typeof(obj.State.DefaultGet()) == "undefined")
                return (obj.StateManager() && obj.StateManager().State) ? obj.StateManager().State() : obj.state_t.absent;

            return obj.State.DefaultGet();
        }

        obj.State.Filter = function(val)
        {   
            if(typeof(val) == undefined || val == null 
               || !(val == obj.state_t.absent || val == obj.state_t.installed))
            {
                obj.Log("try to set state with incorrect value =\"" + val + "\"");
                return false;
            }

            return true;
        }
        //###############################################################
        // p_action to store/receive action property
        //###############################################################
        obj.p_action = P(obj.action_t.none);
        //###############################################################
        // p_size to store/receive size property
        //###############################################################
        obj.p_size = P();
        obj.p_size.Filter = FilterNotEmpty;
        obj.p_size.Get = function ()
        {
            if(typeof(obj.p_size.DefaultGet()) == "undefined")
                return obj.Info().Size();
            
            return obj.p_size.DefaultGet();
        }
        //###############################################################
        // Disabled to store/receive disabled property
        //###############################################################
        obj.Disabled = P(obj.disabled_t.no);
        obj.Disabled.Transform = function(val){ return (val == obj.disabled_t.mix || val) ? obj.disabled_t.yes : obj.disabled_t.no; }
        obj.Disabled.Subscribe(function (val)
        {
            obj.Log("Setting disabled " + val);
            
            if(val == obj.disabled_t.yes)
                obj.Action(obj.action_t.none);
        });
        //###############################################################
        /** @method Component Size
         *  @brief Component.Size - get Component size
         *  @details request disk space required to process desired action
         *  @return number disk space required to process action, bytes
         *  @usage
         *    var size = Component.Size();
         *  @see GetAction GetState
         */
        obj.Size = function (val)
        {
          if(obj.Disabled() == obj.disabled_t.yes)
          {
            obj.Log("Size: component is disabled therefore size = 0");
            return 0;
          }

          return obj.p_size(val);
        }
        //###############################################################
        obj.Root = function ()
        {
            var root = obj;

            for (var parent = obj.Parent(); parent; root = parent, parent = parent.Parent());

            return root;
        }
        //###############################################################
        // Detach component from parent obj
        //###############################################################
        obj.Detach = function ()
        {
            obj.Log("Detaching begin");
            
            if(obj.Parent())
            {
                obj.Parent().Components().Remove(obj.Id());
                obj.Parent(null);
            }

            if(obj.IsOriginal())
                obj.Clones().Apply(function(el){el.Detach(); return true;});
                    
            obj.Log("Detaching end");
            return true;
        }
        //###############################################################
        // Checking for already installed previous versions which can be upgraded
        //###############################################################
        obj.CheckForUpgrade = function ()
        {
            if(obj.Disabled() == obj.disabled_t.yes)
            {
                obj.Log("CheckForUpgrade: component is disabled -> check isn't required");
                return true;
            }

            if(obj.State() == obj.state_t.installed)
            {
                obj.Log("CheckForUpgrade: component is installed -> check isn't required");
                return false;
            }

            obj.Log("CheckForUpgrade: begin");

            obj.Upgrade().Check();

            obj.Log("CheckForUpgrade: completed");

            return true;
        }
        //###############################################################
        // Upgrade state for this component
        //###############################################################
        obj.UpgradeState = function ()
        {
            if(obj.Disabled() == obj.disabled_t.yes)
            {
                obj.Log("UpgradeState: component is disabled -> none to upgrade");
                return obj.upgrade_state_t.none;
            }

            if(obj.Action() == obj.action_t.none || obj.Action() == obj.action_t.remove)
            {
                obj.Log("UpgradeState: component action is none/remove -> upgrade_state_t.none");
                return obj.upgrade_state_t.none;
            }

            return obj.Upgrade().State();
        }
        //###############################################################
        // Setting of dependencies processor
        //###############################################################
        obj.ProcessDependency = function (_obj)
        {
            //obj.Log("it is default dependency processor");
            return true;
        }
        //###############################################################
        // Dependencies setting
        //###############################################################
        obj.Depend = function (alias, iobj)
        {
            iobj.Action.Subscribe(function(changed_obj){ return obj.ProcessDependency(changed_obj);});
            obj.Dependencies().Add(alias, iobj);
        }
        //###############################################################
        obj.Action = function (act)
        {
            if(act)
                return obj.SetAction(act);

            return obj.p_action();
        }
        obj.Action.Subscribe = obj.p_action.Subscribe;
        //###############################################################
        /** @method Component SetAction(action_t action)
         *  @brief Component.SetAction - set desired action to Component
         *  @details set desired installation action to Component
         *  @param action_t action - property of action_t type
         *  @return boolean true -  if succeed
         *  @return boolean false - if failed
         *  @usage
         *    Component.SetAction(Component.action_t.install);
         *  @see GetAction, action_t
         */
        obj.SetAction = function (act)
        {
          if(!act)
            return false;

          if(act && act != obj.action_t.remove && obj.Disabled() == obj.disabled_t.yes)
          {
            obj.Log("SetAction: only 'remove' action can be set for disabled component -> action '" + act + "' can't be set");
            return false;
          }
      
          if(act == obj.p_action())
          {
                obj.Log("SetAction: current action \"" + act + "\" will not be changed due to it is already the same as input act = \""+ act +"\"");
                return true;
          }

          if(obj.IsOriginal()) // orig_obj is a link to original obj
          {
            obj.Log("Performing SetAction (act = " + act + ") for original component -> this action will be passed to all clones");
            obj.Clones().Apply(function(el){obj.Log("setting action for clone " + el.CloneId()); el.Action(act); return true;});
          }

          obj.Log("SetAction: act = \"" + act + "\"");

          if( (obj.State() == obj.state_t.installed && act == obj.action_t.install) ||
              (obj.State() == obj.state_t.absent && (act == obj.action_t.remove || act == obj.action_t.repair))
            )
          {
            if(obj.Action() == obj.action_t.none)
            {
                obj.Log("SetAction: component state = " + obj.State() + ", current action = \"none\" -> it will not be changed");
                return true;
            }
        
            obj.Log("SetAction: component state = " + obj.State() + " -> action will be set as \"none\"");

            obj.p_action(obj.action_t.none);

            return true;
          }

          obj.p_action(act);

          return true;
        }
        //###############################################################
        // Add component to specified Group
        //###############################################################
        obj.AddToGroup = function (grp_id)
        {
            if(!grp_id || obj.State() != obj.state_t.installed)
                return false;
            
            obj.Log("adding to group " + grp_id);
            
            if(!ns_installer.Installer.Groups[grp_id])
                ns_installer.Installer.Groups[grp_id] = {};

            var grp = ns_installer.Installer.Groups[grp_id];
            grp[obj.Id()] = obj;

            obj.Groups().Add(grp_id, grp_id);

            return true;
        }

        //###############################################################
        // RestorePoint method definition
        //###############################################################
        obj.RestorePoint = function (st)
        {
            var rp = st ? st : Storage("*");

            rp("id").value = obj.Id();
            rp("name").value = obj.Name();
            rp("description").value = obj.Description();
            rp("obj_type").value = obj.Type();
            rp("install_dir_base").value = obj.InstallDir.Base();
            rp("install_dir_own").value = obj.InstallDir.Own();

            if(obj.Version().ObjType && obj.Version().ObjType() == ns_version.TypeName())
                rp("version").value = obj.Version().Str();

            var groups = rp("groups");
            for(var i in obj.Groups().Items())
                groups(i).value = obj.Groups().Items()[i];

            return rp;
        }
    }
    //###############################################################
//###############################################################
// Props adjustment
//###############################################################
    function BindBase(_in)
    {
        var obj = this;
        //var base = {};

        ns_enums.BindTo(obj);

        obj.Info       = ConstP(_in.Info);

        obj.Log         = log_helper("Component name/id = " + ((obj.Info().Name) ? obj.Info().Name() : obj.Info().Id()) + ": ");

        obj.Source     = PNotEmpty();
        obj.Processor  = PNotEmpty();
        obj.StateManager  = PNotEmpty();
        //###############################################################
        // Props adjustment
        //###############################################################
        obj.Source.Subscribe(function(src)
        {
            obj.Log("source set: \"" + ((src.File) ? src.File() : "") + "\"");
        });
        //###############################################################
        obj.Processor.Subscribe(function(prc)
        {
            obj.Log("set new processor");
            if(prc.Owner)
                prc.Owner(obj);
        });
        //###############################################################
        obj.StateManager.Subscribe(function(mng)
        {
            obj.Log("set new state manager");
            if(mng.Owner)
                mng.Owner(obj);
        });


        obj.Id          = ConstP(obj.Info().Id());
        obj.Name        = ConstP(obj.Info().Name());
        obj.Version     = ConstP(ns_version.Version(obj.Info().Version()));
        obj.Description = P(obj.Info().Description());
        obj.ErrorDescription = function(){return obj.Info().ErrorDescription ? obj.Info().ErrorDescription : blank_f;}
        
        obj.Type       = P(obj.Info().Type ? obj.Info().Type() : "component");
        
        obj.InstallDir = ns_dir.Directory();

        obj.ConfigurationOptions = ConstP(ns_prop_set.PropertySet());
        obj.InstallConfigurationOptions = obj.ConfigurationOptions;
        obj.RemoveConfigurationOptions = ConstP(ns_prop_set.PropertySet());

        obj.CustomProperties = ConstP(ns_prop_set.PropertySet());
        obj.Clones     = ConstP(ns_cont.Container());
        obj.Groups     = ConstP(ns_cont.Container());
        obj.Upgrade    = ConstP(new ns_upgrade.Upgrade(obj));
        obj.Dumper     = ConstP(ns_dump.Dumper("dmpr_for_" + (obj.Info().Name ? obj.Info().Name() : obj.Info().Id())) );
        //obj.PreAction = function () { return obj.Dumper.PreAction(); }
        //obj.PostAction = function () { return obj.Dumper.PostAction(); }

        obj.Original   = ConstP(obj);

        obj.Dependencies = ConstP(ns_cont.Container());

        obj.Offline = PBool(false);
        obj.Offline.Filter = function(val){if(val) return true;}
        obj.Offline.Subscribe(function(val)
        {
            if(val)
                if(obj.State() == obj.state_t.absent && (!obj.Source() || !obj.Source().Resolved()))
                    obj.Disabled(true);
        });

        obj.Source(_in.Source);
        obj.Processor(_in.Processor);
        obj.StateManager((_in.StateManager ? _in.StateManager : _in.Processor));

        //###############################################################
        obj.ExInit = _in.ExInit ? _in.ExInit : function() {};
        //###############################################################
        obj.Clone = function()
        {
            var cln = {};

            for(var key in obj)
                cln[key] = obj[key];
            /*
            Log("logging obj version ...");
            var o = obj.Version();
            for(var p in o)
                Log(p + " = " + o[p]);

            Log("logging clone version ...");
            var o = cln.Version();
            for(var p in o)
                Log(p + " = " + o[p]);
            Log("completed ...");
            */
            cln.Log = log_helper("Clone " + obj.Log.Prefix());
            BindPrivate(cln);

            cln.Action(obj.Action());
            cln.Disabled(obj.Disabled()); // clone has own disabled property

            cln.CloneId(Guid());

            obj.Clones().Add(cln.CloneId(), cln);

            obj.ExInit.call(obj);

            //cln.Clones().Apply(function(el){Log("there is clone id = " + el.CloneId()); return true;});
            return cln;
        }

     //###############################################################
     // Create base component part
     //###############################################################
     /*
        for(var key in obj)
            base[key] = obj[key];

        return base;
     */
    }
    //###############################################################
    // Component class
    //###############################################################
    this.Component = function(_in)
    {
        var cmp = {};

        BindBase.call(cmp, _in);
        BindPrivate(cmp);

        ns_installer.Installer.AddComponent(cmp);

        return cmp;
    }//Component function
}// namespace Component