/** @file source.js
 *  @brief Implementation file aliases
 */

 /*
    Source - is object to handle files
    every Source object has methods:
        Id - get id of element, string
        Resolved - is current Source resolved (file exists), boolean
        Resolve - create DumperAction/Dumper element to resolve
 */
new function()
{
    var load = function(name) {return required(FileSystem.MakePath(name, Origin.Directory()));};

    var ns = this;
    var ns_dmp = load("dumper.js");

    // ***************** helper functions ************
    var enum_array = function(arr, func)
    {
        if(arr && func)
        {
            for(var i in arr)
            {
                var s = arr[i];
                if(func(s))
                    return true;
            }
        }
        return false;
    }

    var CreateDumper = function(dumper)
    {
        if(!dumper)
            return new ns_dmp.Dumper();

        return dumper;
    }

    var Source = function()
    {
        var id = Guid();
        var source = {};
        var dependencies = [];

        source.Id = function(_id)
        {
            if(_id)
                id = _id;
            else
                return id;
        }

        source.Resolved = function()
        {
            for(var i in dependencies)
            {
                var d = dependencies[i];
                if(d && d.Resolved)
                { // if object has dependencies && all of them are resolved - return true
                    if(!d.Resolved())
                        return false;
                }
            }

            return true;
        }

        source.Resolve = function(dumper)
        {
            var dmp = CreateDumper(dumper);

            for(var i in dependencies)
            {
                var d = dependencies[i];
                if(d && d.Resolve)
                    d.Resolve(dmp);
            }

            if(dmp.IsEmpty())
                return null;
            return dmp;
        }

        source.AddDependency = function(dep)
        {
            if(dep)
                dependencies.push(dep);
        }

        source.Filter = function(cb)
        {
            if(cb(source))
                return true;

            for(var i in dependencies)
            {
                var d = dependencies[i];
                if(d && d.Filter)
                    if(d.Filter(cb))
                        return true;
            }

            return false;
        }

        return source;
    }

    // ***************** public API ***************

    // ***************** FileSource ****************
    /*  FileSource - is simple source object which only checks for existing file
        _file - path to file to be evaluated
    */
    this.FileSource = function(_file, _size)
    {
        var source = Source();

        var file = _file;
        var size = _size ? _size : 0;
        var chksum = undefined;

        Log("File source object: " + file + " : " + size);

        var orig_resolved = source.Resolved;

        source.File = function() {return file;}
        source.Size = function() {return FileSystem.Exists(file) ? FileSystem.Size(file) : size;}
        
        source.Resolved = function()
        {
            if(FileSystem.Exists(file) && orig_resolved)
                return orig_resolved();
            return false;
        }

        source.Chksum = function(sum)
        {
            if(arguments.length)
                chksum = parseInt(sum);
            return chksum;
        }

        source.IsChksumValid = function()
        {
            if(source.Resolved())
            {
                if(typeof(chksum) == "undefined")
                    return FileSystem.ChkSum(source.File()) == chksum;
            }
            return false;
        }

        return source;
    }

    // ***************** UrlSource ****************
    /*  UrlSource - source object to download file
        _file - path to file to save
        _url - url to download file from
    */
    this.UrlSource = function(_file, _url, _size)
    {
        var ns_download = load("dumper_download.js");

        var source = ns.FileSource(_file, _size);

        var url = _url;

        Log("Url source object: " + url + " : " + source.File() + " : " + source.Size());

        source.Url = function() {return url;}
        var orig_resolve = source.Resolve;

        source.Resolve = function(dumper)
        {
            if(source.Resolved())
                return null;

            var d = CreateDumper(dumper);
            
            if(orig_resolve)
                orig_resolve(d);

            if(url && source.File())
            {
                var download = ns_download.Download();
                if(download)
                {
                    download.Url(url);
                    download.File(source.File());
                    download.Skip = function() {return source.Resolved();}
                    d.AddAction(download).Attribute("countable", true);
                }
            }

            if(d.IsEmpty())
                return null;
            else
                return d;
        }
        
        return source;
    }

    // ***************** ScopeSource ****************
    /*  ScopeSource - source object to collect set of files. it is collection of
                     source objects to process
    */
    this.ScopeSource = function()
    {
        var source = Source();

        var sources = [];

        source.AddSource = function(src)
        {
            if(src)
                sources.push(src);
        }

        source.Resolved = function()
        {
            return !enum_array(sources, function(s)
                {
                    if(s && s.Resolved && !s.Resolved())
                        return true;
                    return false;
                });
        }

        source.Resolve = function()
        {
            var dmp = ns_dmp.Dumper();

            enum_array(sources, function(s)
                {
                    if(s && s.Resolve)
                    {
                        var d = s.Resolve();
                        if(d)
                            dmp.AddAction(d);
                    }
                    return false;
                });

            return dmp;
        }

        source.Priority = function()
        {
            var prio = 0;
            enum_array(sources, function(s)
                {
                    if(s && s.Priority)
                        prio += s.Priority();
                    return false;
                });
            return prio;
        }

        return source;
    }

    // ***************** KeyFileSource ****************
    /*  KeyFileSource - mix of FileSource & ScopeSource. it is collection
                       of source object, but it has attribute File which
                       returns path to key file (msi for example)
        _file - path to file
    */
    this.KeyFileSource = function(file)
    {
        var source = ns.ScopeSource();
        var file_alias = ns.FileSource(file);
        source.AddSource(file_alias);
        source.File = file_alias.File;

        return source;
    }

    // ***************** Transform ****************
    /*
        Transform - transformation object.
        _from - source object for transform FROM
        _to - source object for transform TO
    */
    this.Transform = function(_from, _to)
    {
        var transform = {};

        var from = _from;
        var to = _to;

        transform.From = function() {return from;}
        transform.To = function() {return to;}

        transform.CanTransformTo = function(source)
        {
            if(source && source.Id)
            {
                if(to.Id() == source.Id())
                    return true;
            }

            return false;
        }

        return transform;
    }

    // ***************** Manager API ****************
    var manager = {};
    var transforms = [];

    var get_transforms_for = function(source)
    {
        if(source)
        {
            var trances = [];
            enum_array(transforms, function(t)
                {
                    if(t && t.CanTransformTo && t.CanTransformTo(source))
                        trances.push(t);
                    return false;
                });
            if(trances.length)
                return trances;
        }

        return null;
    }
}

