//###############################################################
// This file contains definition for:
//  enum ftr_state_t
//  class Feature
//###############################################################
//Namespace("Root.feature", function () 
new function ()
{
    var base_script_dir = Origin.Directory();
    var load = function(name) {return required(FileSystem.MakePath(name, base_script_dir));};

    //var ns_component = Namespace("Root.component");
    var ns_installer = Namespace("Root.installer");
    var ns_component = load("component.js");
    var ns_event     = load("event.js");
    var ns_enums     = load("enums.js");
    var ns_upgrade   = load("upgrade.js");
    var ns_version   = load("version.js");
    var ns_dumper    = load("dumper.js");

    var icons_dir = FileSystem.MakePath("Icons", FileSystem.Parent(Origin.Directory()));
    Node.icon_broken_menu = "icon_broken_menu";
    Action.FeatureTreeNodeIcon({image:icons_dir + "\\broken.png", id: "icon_broken_menu"});

    function SortByOrder(a,b){ return (a.Order ? a.Order() : 100) - (b.Order ? b.Order() : 100);}

    var ns = this;
    //###############################################################
    // Feature constructor
    //###############################################################
    this.Create = this.Create || function(id, ex_init)
    {
        var ftr = new ns.Feature();
        if(!ftr.Init(id))
            return null;
        
        if(ex_init)
            ex_init.call(ftr);

        return ftr;
    }
    //###############################################################
    // class Feature
    //###############################################################
    this.Feature = this.Feature || function () 
    {
        ns_enums.BindTo(this);

        this.obj_type = "feature";
        this.OnCommit = new ns_event.Event(this);
        this.OnActionChange = new ns_event.Event(this);
        this.Upgrade  =  new ns_upgrade.Upgrade(this);
        this.Groups = [];
        this.CnfgOpts = {};
    }
    //###############################################################
    this.Feature.prototype.Init = function(id)
    {
        this.Log = log_helper("Feature id = " + id + ": ");
        
        if(!id)
        {
            this.Log("Can't create feature with undefined id");
            return false;
        }
        this.m_id = id;
        this.m_name = id;
        this.m_dscrpt = "";
        this.m_error_dscrpt = "";
        this.m_visible = true;
        this.m_mandatory = false;
        this.m_expanded = false;
        this.m_size = 0;
        this.m_error = ""; // error description
        this.m_components = {};
        this.m_components_order = []; // keeping cmps order
        this.m_features = {};
        this.m_features_order = []; // keeping ftrs order
        this.m_node = null;
        this.m_parent = null;
        this.m_install_dir_base = "";
        this.m_install_dir_own = "";
        this.m_install_dir_locked = false;
        this.m_version = ns_version.Version();
        this.m_priority = 0;
        this.m_order = 100;
        this.m_prev_act = this.action_t.none;
        this.m_curr_act = this.action_t.none; // just for identifying implicit action change (i.e. when one of sub features' action was changed)
        this.m_offline = false;

        this.m_action_update_in_prgrs = false; // for preventing cicle call of SetAction.
        this.m_action_to_set = null;           // store info regarding new action to set

        this.m_apply_upgrade_done = false;
        this.m_apply_remove_done  = false;
        this.m_apply_install_done = false;
        
        this.m_error_handler = function(){return false;}

        this.Dependencies = {};

        return true;
    }
    //###############################################################
    this.Feature.prototype.Id = function() {return this.m_id;}
    //###############################################################
    this.Feature.prototype.Name = function (nm)
    {
        if(nm) this.m_name = nm;

        return this.m_name;
    }
    //###############################################################
    this.Feature.prototype.Description = function (dscr)
    {
        if(dscr) this.m_dscrpt = dscr;
        
        return this.m_dscrpt;
    }
    //###############################################################
    this.Feature.prototype.ErrorDescription = function (dscr)
    {
        if(dscr) this.m_error_dscrpt = dscr;
        
        return this.m_error_dscrpt;
    }
    //###############################################################
    this.Feature.prototype.Expanded = function (val)
    {
        if(val != undefined)
            this.m_expanded = val;
        
        return this.m_expanded;
    }
    //###############################################################
    this.Feature.prototype.Visible = function (val)
    {
        if(val != undefined)
        {
            this.m_visible = val;
            if(!val)// only false value should be passed down
                for (var i in this.m_features)
                    this.m_features[i].Visible(false);
        }
        
        return this.m_visible;
    }
    //###############################################################
    this.Feature.prototype.Mandatory = function (val)
    {
        if(val != undefined)
            this.m_mandatory = val;
        
        return this.m_mandatory;
    }
    //###############################################################
    // Version to store/receive version property
    //###############################################################
    this.Feature.prototype.Version = function (val)
    {
        if(val)
            this.m_version = ns_version.Version(val);
        
        return this.m_version;
    }
    //###############################################################
    this.Feature.prototype.Priority = function (prio)
    {
        if(typeof(prio) == "number")
            this.m_priority = prio;
        
        return this.m_priority;
    }
    //###############################################################
    this.Feature.prototype.Order = function (ord)
    {
        if(typeof(ord) == "number")
            this.m_order = ord;
        
        return this.m_order;
    }
    //###############################################################
    this.Feature.prototype.Offline = function (val)
    {
        if(arguments.length > 0 )
            if(val)
            {
                this.m_offline = true;
                for(var key in this.m_components)        
                    this.m_components[key].Offline(true);
        
                for (var key in this.m_features)
                    this.m_features[key].Offline(true);
            }

        return this.m_offline;
    }
    //###############################################################
    this.Feature.prototype.Features = function() { return this.m_features;}
    this.Feature.prototype.FeaturesOrder = function() { return this.m_features_order;}
    //###############################################################
    this.Feature.prototype.FeaturesFullSet = function() 
    {
        var set = {};
        
        var sub_set = {};
        for (var i in this.m_features)
        {
            set[i] = this.m_features[i];

            sub_set = this.m_features[i].FeaturesFullSet();
            for (var key in sub_set)
                set[key] = sub_set[key];
        }

        return set;
    }
    //###############################################################
    this.Feature.prototype.Components = function() { return this.m_components;}
    this.Feature.prototype.ComponentsOrder = function() { return this.m_components_order;}
    //###############################################################
    this.Feature.prototype.ComponentsFullSet = function() 
    {
        var set = {};
        
        for (var key in this.m_components)
            set[key] = this.m_components[key];
        
        var sub_set = {};
        for (var i in this.m_features)
        {
            sub_set = this.m_features[i].ComponentsFullSet();
            for (var key in sub_set)
                set[key] = sub_set[key];
        }

        return set;
    }
    //###############################################################
    // Detach
    //###############################################################
    this.Feature.prototype.Detach = function ()
    {
        this.Log("Detaching begin");
        if(this.Parent())
        {
            this.Parent().RemoveFeature(this.Id());
            this.m_parent = null;
        }

        for (var key in this.m_components)        
            this.m_components[key].Detach();
        
        for (var key in this.m_features)
            this.m_features[key].Detach();

        this.DetachNode();
        this.Log("Detaching end");
    }
    //###############################################################
    // Add configuration option
    //###############################################################
    this.Feature.prototype.AddCnfgOpt = function (opt, val)
    {
        if(!opt)
            return false;
        
        if(this.Disabled() == this.disabled_t.yes)
        {
            this.Log("AddCnfgOpt: feature is fully disabled -> cnfg option can't be added");
            return true;
        }

        this.CnfgOpts[opt] = val;

        for (var key in this.m_components)
            this.m_components[key].AddCnfgOpt(opt, val);
        
        for (var key in this.m_features)
            this.m_features[key].AddCnfgOpt(opt, val);
 
        return true;
    }
    //###############################################################
    // configuration options
    //###############################################################
    this.Feature.prototype.ConfigurationOptions = function ()
    {
        return this.CnfgOpts;
    }
    //###############################################################
    // Add feature to specified Group
    //###############################################################
    this.Feature.prototype.AddToGroup = function (grp_id)
    {
        this.Log(" @@@ try add to group " + grp_id);
        if(!grp_id)
            return false;
        
        this.Log("@@@ add to group " + grp_id);

        this.Groups.push(grp_id);

        //Log(" this.getstate = " + this.GetState());
        //if(this.GetState() != this.state_t.installed)
        //    return true;
        //this.Log("@@@ add to group- product is installed adding to installer " + grp_id); 

        if(!ns_installer.Installer.Groups[grp_id])
            ns_installer.Installer.Groups[grp_id] = {};

        var grp = ns_installer.Installer.Groups[grp_id];
        grp[this.m_id] = this;
        
        return true;
    }
    //###############################################################
    // Check that feature belongs to specified Group
    //###############################################################
    this.Feature.prototype.InGroup = function (grp_id)
    {
        if(!grp_id) 
            return false;

        if(this.State() != this.state_t.installed)
            return false;
        
        if(!ns_installer.Installer.Groups[grp_id])
            ns_installer.Installer.Groups[grp_id] = {};

        if(ns_installer.Installer.Groups[grp_id][this.m_id] == this)
            return true;
        
        return false;
    }
    //###############################################################
    // it is reasponsible for setting/getting component's base install dir
    //###############################################################
    this.Feature.prototype.InstallDirBase = function (val)
    {
        if(this.Disabled() == this.disabled_t.yes)
        {
            this.Log("InstallDirBase: feature is fully disabled -> no installdir");
            return "";
        }

        if(val && this.m_install_dir_locked != true)
        {
            this.Log("Setting installdirbase = " + val);
            this.m_install_dir_base = val;
            
            for (var key in this.m_components)        
                this.m_components[key].InstallDirBase(this.InstallDir());
        
            for (var key in this.m_features)
                this.m_features[key].InstallDirBase(this.InstallDir());

            this.DoInstallDirBaseChange(val);
        }

        return this.m_install_dir_base;
    }
    //###############################################################
    // following method is for definition by inheritors
    //###############################################################
    this.Feature.prototype.DoInstallDirBaseChange = function (val){ return true;}
    //###############################################################
    // it is reasponsible for setting/getting component's own install dir
    //###############################################################
    this.Feature.prototype.InstallDirOwn = function (val)
    {
        if(this.Disabled() == this.disabled_t.yes)
        {
            this.Log("InstallDirOwn: feature is fully disabled -> no installdir");
            return "";
        }

        if(val && this.m_install_dir_locked != true)
        {
            this.Log("Setting installdirown = " + val);
            this.m_install_dir_own = val;

            for (var key in this.m_components)        
                this.m_components[key].InstallDirBase(this.InstallDir());
        
            for (var key in this.m_features)
                this.m_features[key].InstallDirBase(this.InstallDir());

            this.DoInstallDirOwnChange(val);
        }
        return this.m_install_dir_own;
    }
    //###############################################################
    // following method is for definition by inheritors
    //###############################################################
    this.Feature.prototype.DoInstallDirOwnChange = function (val){ return true;}
    //###############################################################
    // it is reasponsible for getting component's install dir
    //###############################################################
    this.Feature.prototype.InstallDir = function ()
    {
        if(this.Disabled() == this.disabled_t.yes)
        {
            this.Log("InstallDir: feature is fully disabled -> no installdir");
            return "";
        }
        
        return FileSystem.MakePath(this.m_install_dir_own, this.m_install_dir_base);
    }
    //###############################################################
    // it is reasponsible for locking component's base and own install dirs
    //###############################################################
    this.Feature.prototype.LockInstallDir = function ( val )
    {
        if(Boolean(val))
            this.m_install_dir_locked = true;
        else 
            this.m_install_dir_locked = false;     
    }
    //###############################################################
    this.Feature.prototype.Hit = function () 
    {
        var ftr = this;

        return function () {
            // this presents corresponding Node here

            var menu = ftr.Menu();
            var menu_arr = [];

            for (var i in menu)
                menu_arr.push(menu[i]);

            var user_choice = this.Menu(menu_arr);

            ftr.Log("User choice = " + user_choice);

            ftr.Action(menu[user_choice].act);

            ftr.Root().Refresh();
        }
    }
    //###############################################################
    this.Feature.prototype.SetNode = function (n) 
    {
        if(!n)
            return false;

        this.DetachNode();

        this.m_node = n;

        return true;
    }
    //###############################################################
    this.Feature.prototype.GetNode = function () 
    {
        return this.m_node;
    }
    //###############################################################
    this.Feature.prototype.Parent = function (ftr) 
    {
        if(ftr)
            this.m_parent = ftr;

        return this.m_parent;
    }
    //###############################################################
    this.Feature.prototype.SetParent = function (ftr) 
    {
        if(!ftr)
            return false;

        this.m_parent = ftr;

        return true;
    }
    //###############################################################
    this.Feature.prototype.GetParent = function () 
    {
        return this.m_parent;
    }
    //###############################################################
    this.Feature.prototype.Root = function () 
    {
        var root = this;

        for (var parent = this.m_parent; parent; root = parent, parent = parent.Parent());

        return root;
    }
    //###############################################################
    this.Feature.prototype.Refresh = function () 
    {
        for (var ftr in this.m_features) 
            this.m_features[ftr].Refresh();        

        this.RefreshNode();
    }
    //###############################################################
    this.Feature.prototype.RefreshNode = function () 
    {
        if (this.m_node) {
            this.m_node.id          = this.Id();
            this.m_node.name        = this.Name();
            this.m_node.description = this.Description();
            this.m_node.disabled    = (this.Disabled() == this.disabled_t.yes) ? true : false;
            this.m_node.error       = this.ErrorDescription();
            this.m_node.icon        = this.Icon();
            this.m_node.size        = this.Size();
            this.m_node.hit         = this.Hit();
            this.m_node.expanded    = this.Expanded();
            this.m_node.priority    = this.Priority();

            this.m_node.Refresh();

            return true;
        }

        return false;
    }
    //###############################################################
    this.Feature.prototype.ClearNode = function () 
    {
        if (this.m_node) {
            this.m_node.id          = "";
            this.m_node.name        = "";
            this.m_node.description = "";
            this.m_node.disabled    = false;
            this.m_node.error       = "";
            this.m_node.icon        = "";
            this.m_node.size        = "";
            this.m_node.hit         = function(){};
            this.m_node.expanded    = false;
            this.m_node.priority    = 0;

            this.m_node.Refresh();

            return true;
        }

        return false;
    }
    //###############################################################
    this.Feature.prototype.DetachNode = function () 
    {
        if (this.m_node) 
        {
            this.ClearNode();
            this.m_node.Detach();
            this.m_node = null;
            return true;
        }

        return false;
    }
    //###############################################################
    this.Feature.prototype.AddComponent = function (cmp) 
    {
        if(!cmp)
        {
            this.Log("attempt to add undefined component");
            return false;
        }
        
        if(cmp.Id())
        {
            if(this.m_components[cmp.Id()])
            {
                this.Log("component with id = " + cmp.Id() + " already exists");
                return true;
            }

            this.Log("add component " + cmp.Id());
            this.m_components[cmp.Id()] = cmp;
            this.m_components_order.push(cmp);
            this.m_components_order.sort(SortByOrder);
            cmp.InstallDirBase(this.InstallDir());
            cmp.Parent(this);
            cmp.OnActionChange.Connect(this.ProcessChildActionChange, this);
        }
        else
        {
            this.Log("can't add component with undefined id");
            return false;
        }

        return true;
    };
    //###############################################################
    this.Feature.prototype.RemoveComponent = function (id)
    {
        if(!id)
        {
            this.Log("attempt to remove component with undefined id");
            return false;
        }

        if(this.m_components[id])
        {
            delete this.m_components[id];
            
            var i = -1;
            
            for(i in this.m_components_order)
            {
                if(this.m_components_order[i].Id() == id)
                    break;
            }
            
            if(i > -1)
                this.m_components_order.splice(i,1); // removing component from array

            this.Log("componen with id = " + id + " was removed");
        }
        else
        {
            this.Log("component with id = " + id + " doesn't present");
        }
        
        return true;
    }
    //###############################################################
    this.Feature.prototype.AddFeature = function (ftr) 
    {
        if(!ftr)
        {
            this.Log("attempt to add undefined feature");
            return false;
        }
        
        if(ftr.Id())
        {
            if(this.m_features[ftr.Id()])
            {
                this.Log("feature with id = " + ftr.Id() + " already exists");
                return true;
            }

            this.Log("add feature " + ftr.Id());
            this.m_features[ftr.Id()] = ftr;
            this.m_features_order.push(ftr);
            this.m_features_order.sort(SortByOrder);
            ftr.InstallDirBase(this.InstallDir());
            ftr.Parent(this);
            if(ftr.Visible() && !this.Visible())
                ftr.Visible(false);
            ftr.OnActionChange.Connect(this.ProcessChildActionChange, this);
        }
        else
        {
            this.Log("can't add feature with undefined id");
            return false;
        }

        return true;
    };
    //###############################################################
    this.Feature.prototype.RemoveFeature = function (id)
    {
        if(!id)
        {
            this.Log("attempt to remove feature with undefined id");
            return false;
        }

        if(this.m_features[id])
        {
            delete this.m_features[id];
            
            var i = -1;
            
            for(i in this.m_features_order)
            {
                if(this.m_features_order[i].Id() == id)
                    break;
            }
            
            if(i > -1)
                this.m_features_order.splice(i,1); // removing feature from array

            this.Log("feature with id = " + id + " was removed");
        }
        else
        {
            this.Log("feature with id = " + id + " doesn't present");
        }
        
        return true;
    }
    //###############################################################
    // Disabled to store/receive disabled property
    //##############################################################
    this.Feature.prototype.Disabled = function (val)
    {
        if(val != undefined)
            this.set_disabled(val ? true : false);

        return this.get_disabled();
    }
    //###############################################################
    this.Feature.prototype.set_disabled = function(val)
    {    
        this.Log("Setting disabled");
        for (var cmp in this.m_components)
            this.m_components[cmp].Disabled(val);
        
        for (var ftr in this.m_features)
            this.m_features[ftr].Disabled(val);
    }
    //###############################################################
    this.Feature.prototype.get_disabled = function()
    {
        var dis_val = undefined;
        var tmp_dis;

        for (var cmp in this.m_components)
        {
            tmp_dis = this.m_components[cmp].Disabled();

            if(dis_val == undefined)
                dis_val = tmp_dis;
            
            if(tmp_dis != dis_val)
                return this.disabled_t.mix;
        }
                
        for (var ftr in this.m_features)
        {
            tmp_dis = this.m_features[ftr].Disabled();
            if(dis_val == undefined)
                dis_val = tmp_dis;
            
            if(tmp_dis != dis_val)
                return this.disabled_t.mix;
        }
   
        if(dis_val == undefined) // feature is empty
            return this.disabled_t.no;

        return dis_val;
    }
    //###############################################################
    this.Feature.prototype.OwnAction = function (act)
    {
        if(act)
            return this.SetOwnAction(act);

        return this.GetOwnAction();
    }
    //###############################################################
    this.Feature.prototype.SetOwnAction = function (act) 
    {
        // in case of setting own action to remove it should be done by common way for all childs
        if(act == this.action_t.remove || (act == this.action_t.none && this.OwnState() == this.state_t.absent))
        {
            this.Log("SetOwnAction: request for setting 'remove' action (or 'none' with current own state absent) -> it will be done for all childs");
            return this.Action(act);
        }

        if(act && act != this.action_t.remove && this.Disabled() == this.disabled_t.yes)
        {
            this.Log("SetOwnAction: only 'remove' action can be set for disabled component -> action '" + act + "' can't be set");
            return this.action_t.none;
        }
        
        // act is action from this.action_t enum
        if (!act)
        {
            this.Log("SetOwnAction: input is undefined -> return GetAction()");
            return this.GetOwnAction();
        }

        if(this.m_action_update_in_prgrs)
        {
            if(act == this.m_action_to_set)
            {
                this.Log("SetOwnAction: setting action to " + act + " is already in progress");
                return act;
            }
            else
            {
                this.Log(Log.l_error, "SetOwnAction: setting action to " + this.m_action_to_set + " is progress, but received the concurent request to set act into " + act);
                return this.m_action_to_set;
            }
        }
                        
        var curr_act = this.GetAction();

        if( curr_act == act)
        {
            this.Log("SetOwnAction: current feature full action \"" + curr_act + "\" will not be changed due to it is already the same as input act = \""+ act +"\"");
            return act;
        }
            
        this.Log("SetOwnAction act = \"" + act + "\"");

        this.m_action_update_in_prgrs = true;
        this.m_action_to_set = act;

        for (var cmp in this.m_components)
            this.m_components[cmp].Action(act);
        
        this.DoSetOwnAction(act);

        var new_act = this.GetAction();
        if(curr_act == new_act)
        {
            this.Log("SetOwnAction: action \"" + act + "\" was applied for own components only, full feature action remain \""+ curr_act +"\"");
        }
        else
        {
            this.ProcessActionChange(new_act);
        }
        
        this.m_action_update_in_prgrs = false;

        return act;
    }
    //###############################################################
    this.Feature.prototype.DoSetOwnAction = function(act) { return true; }
    //###############################################################
    this.Feature.prototype.GetOwnAction = function ()
    {
        var none_installed = false;
        var none_absent    = false;
        var install  = false;
        var remove   = false;

        for (var key in this.m_components)
        {
            if(this.m_components[key].Disabled() == this.disabled_t.yes)
                continue;

            var act = this.m_components[key].Action();
            var state = this.m_components[key].State();
            
            if(act == this.action_t.install)
                install = true;
            else if(act == this.action_t.remove)
                remove = true;
            else if(act == this.action_t.none && state == this.state_t.installed)
                none_installed = true;
            else if(act == this.action_t.none && state == this.state_t.absent)
                none_absent = true;
        }

        var curr_act = this.action_t.none;

        if((install && remove) || (none_installed && remove) || (none_absent && install))
            curr_act = this.action_t.mix; 
        else if(install || (install && none_installed))
            curr_act = this.action_t.install;
        else if(remove || (remove && none_absent))
            curr_act = this.action_t.remove;
        else //if(none_installed || none_absent)
            curr_act = this.action_t.none;

        return curr_act;
    }
    //###############################################################
    this.Feature.prototype.PrevAction = function (act)
    {
        if(act)
            return this.m_prev_act = act;

        return this.m_prev_act ? this.m_prev_act : this.GetAction();
    }
    //###############################################################
    this.Feature.prototype.Action = function (act)
    {
        if(act)
            return this.SetAction(act);

        return this.GetAction();
    }
    //###############################################################
    // 
    //##############################################################
    this.Feature.prototype.ProcessChildActionChange = function (chld)
    {
        if(this.m_action_update_in_prgrs)
        {
            this.Log("ProcessChildActionChange: child changed action, action update is in progress - > process it after action update completed");
            return true;
        }

        this.Log("ProcessChildActionChange: begin");
        
        var curr_act = this.Action();
        
        if(this.m_curr_act != curr_act)
        {
            this.Log("ProcessChildActionChange: previous ftr action was " + this.m_curr_act + ", new " + curr_act);

            this.m_action_update_in_prgrs = true;

            this.m_action_to_set = curr_act;

            //this.m_curr_action is responsible for storing action before current change
            //sub components should be marked for install only in cases
            // 1. prev_act == remove && state == installed incoming action is install | mix | none
            // 2. prev_act == none && state == absent incoming action is install | mix | remove -> remove should be skip
            
            var st = this.State();

            if((this.m_curr_act == this.action_t.remove && st == this.state_t.installed)|| 
                (this.m_curr_act == this.action_t.none && st == this.state_t.absent && curr_act != this.action_t.remove))
            {
                
                this.m_action_to_set = this.action_t.install;
                
                for(var cmp in this.m_components)
                    this.m_components[cmp].Action(this.action_t.install);

                for(var ftr in this.m_features)
                {
                    if(this.m_features[ftr].Mandatory())
                        this.m_features[ftr].Action(this.action_t.install);
                }
            }

            this.ProcessActionChange(this.Action());
            
            this.m_action_update_in_prgrs = false;
        }
        this.Log("ProcessChildActionChange: end");
    }
    //###############################################################
    this.Feature.prototype.ProcessActionChange = function (act)
    {
        this.Log("ProcessActionChange: begin, prev action = " + this.m_curr_act + "; new action = " + act);
        
        this.PrevAction(this.m_curr_act);
        this.m_curr_act = act;
        
        this.DoProcessActionChange(this.m_curr_act);

        this.OnActionChange.Call();

        this.Log("ProcessActionChange: end");
    }
    //###############################################################
    this.Feature.prototype.DoProcessActionChange = function (act){ return true; }
    //###############################################################
    this.Feature.prototype.SetAction = function (act) 
    {
        if(act && act != this.action_t.remove && this.Disabled() == this.disabled_t.yes)
        {
            this.Log("SetAction: only 'remove' action can be set for disabled component -> action '" + act + "' can't be set");
            return this.action_t.none;
        }
        
        // act is action from this.action_t enum
        if (!act)
        {
            this.Log("SetAction: input is undefined -> return GetAction()");
            return this.GetAction();
        }

        if(this.m_action_update_in_prgrs)
        {
            if(act == this.m_action_to_set)
            {
                this.Log("SetAction: setting action to " + act + " is already in progress");
                return act;
            }
            else
            {
                this.Log(Log.l_error, "SetAction: setting action to " + this.m_action_to_set + " is progress, but received the concurent request to set act into " + act);
                return this.m_action_to_set;
            }
        }
                        
        var curr_act = this.GetAction();

        if( curr_act == act)
        {
            this.Log("SetAction: current action \"" + curr_act + "\" will not be changed due to it is already the same as input act = \""+ act +"\"");
            return act;
        }
            
        this.Log("SetAction act = \"" + act + "\"");

        this.m_action_update_in_prgrs = true;
        this.m_action_to_set = act;

        for (var cmp in this.m_components)
            this.m_components[cmp].Action(act);
        
        for (var ftr in this.m_features)
            this.m_features[ftr].Action(act);
        
        this.DoSetAction(act);

        var new_act = this.GetAction();
        if(curr_act == new_act)
        {
            this.Log("SetAction: action \"" + act + "\" was applied for all child elements but real action remain \""+ curr_act +"\"");
        }
        else
        {
            this.ProcessActionChange(new_act);
        }
        
        this.m_action_update_in_prgrs = false;

        return act;
    }
    //###############################################################
    this.Feature.prototype.DoSetAction = function(act) { return true; }
    //###############################################################
    this.Feature.prototype.GetAction = function ()
    {
        /*
        if(this.Disabled() == this.disabled_t.yes)
        {
            this.Log("GetAction: feature is fully disabled -> action_t.none");
            return this.action_t.none;
        }
        */
        var none_installed = false;
        var none_absent    = false;
        var install  = false;
        var remove   = false;

        for (var key in this.m_features)
        {
            if(this.m_features[key].Disabled() == this.disabled_t.yes)
                continue;

            var act = this.m_features[key].Action();
            var state = this.m_features[key].State();

            if(act == this.action_t.mix)
                return this.action_t.mix;
            else if(act == this.action_t.install)
                install = true;
            else if(act == this.action_t.remove)
                remove = true;
            else if(act == this.action_t.none && state == this.state_t.installed)
                none_installed = true;
            else if(act == this.action_t.none && state == this.state_t.absent)
                none_absent = true;
        }

        for (var key in this.m_components)
        {
            if(this.m_components[key].Disabled() == this.disabled_t.yes)
                continue;

            var act = this.m_components[key].Action();
            var state = this.m_components[key].State();
            
            if(act == this.action_t.install)
                install = true;
            else if(act == this.action_t.remove)
                remove = true;
            else if(act == this.action_t.none && state == this.state_t.installed)
                none_installed = true;
            else if(act == this.action_t.none && state == this.state_t.absent)
                none_absent = true;
        }

        var curr_act = this.action_t.none;

        //if((install && remove) || (none_installed && none_absent) || (none_installed && remove) || (none_absent && install))
        if((install && remove) || (none_installed && remove) || (none_absent && install))
            curr_act = this.action_t.mix; 
        else if(install || (install && none_installed))
            curr_act = this.action_t.install;
        else if(remove || (remove && none_absent))
            curr_act = this.action_t.remove;
        else //if(none_installed || none_absent)
            curr_act = this.action_t.none;

        return curr_act;
    }
    //###############################################################
    this.Feature.prototype.StateConsistent = function ()
    {
        var state = null;
        for (var key in this.m_components)        
        {
            if(this.m_components[key].Disabled() == this.disabled_t.yes)
                continue;

            if(!state)
                state = this.m_components[key].State();
            else if(this.m_components[key].State() != state)
                return false;
        }

        for (var key in this.m_features)
        {
            if(this.m_features[key].Disabled() == this.disabled_t.yes)
                continue;

            if(!this.m_features[key].StateConsistent())
                return false;
            else if(!state)
                state = this.m_features[key].State();
            else if(this.m_features[key].State() != state)
                return false;
        }

        return true;
    }
    //###############################################################
    this.Feature.prototype.OwnState = function (st)
    {
        if(st)
            this.Log("state can't be assigned to the feature, it can be done for components only");

        for (var key in this.m_components)
        {
            if(this.m_components[key].Disabled() == this.disabled_t.yes)
                continue;

            if(this.m_components[key].State() == this.state_t.installed)
                return this.state_t.installed;
        }
        return this.state_t.absent;
    }
    //###############################################################
    this.Feature.prototype.State = function (st)
    {
        if(st)
            this.Log("state can't be assigned to the feature, it can be done for components only");

        return this.GetState();
    }
    //###############################################################
    this.Feature.prototype.GetState = function ()
    {
        for (var key in this.m_components)
        {
            if(this.m_components[key].Disabled() == this.disabled_t.yes)
                continue;

            if(this.m_components[key].State() == this.state_t.installed)
                return this.state_t.installed;
        }

        for (var key in this.m_features)
        {
            if(this.m_features[key].Disabled() == this.disabled_t.yes)
                continue;

            if(this.m_features[key].State() == this.state_t.installed)
                return this.state_t.installed;
        }
        return this.state_t.absent;
    }
    //###############################################################
    this.Feature.prototype.CheckForUpgrade = function ()
    {
        if(this.Disabled() == this.disabled_t.yes)
        {
            this.Log("CheckForUpgrade: feature is fully disabled -> check isn't required");
            return;
        }
        
        this.Log("CheckForUpgrade: begin");

        if(this.State() == this.state_t.installed)
        {
            this.Log("CheckForUpgrade: feature is installed -> check for own upgrade isn't required but check for childs will perform");
        }
        else
        {
            this.Upgrade.Check();
        }

        for (var key in this.m_features)
            this.m_features[key].CheckForUpgrade();

        for (var key in this.m_components)
            this.m_components[key].CheckForUpgrade();
        
        this.Log("CheckForUpgrade: completed");
    }
    //###############################################################
    this.Feature.prototype.UpgradeState = function ()
    {
        if(this.Disabled() == this.disabled_t.yes)
        {
            this.Log("UpgradeState: feature is fully disabled -> upgrade_state_t.none");
            return this.upgrade_state_t.none;
        }

        var act = this.Action();

        if(act == this.action_t.none || act == this.action_t.remove)
        {
            this.Log("UpgradeState: feature action is none/remove -> upgrade_state_t.none");
            return this.upgrade_state_t.none;
        }

        var curr_state = this.Upgrade.State();

        var t_state = curr_state;

        for (var key in this.m_features)
        {
            if(this.m_features[key].Disabled() == this.disabled_t.yes)
                continue;

            t_state = this.m_features[key].UpgradeState();
            
            if(t_state == this.upgrade_state_t.none)
                continue;

            if(curr_state != this.upgrade_state_t.none && t_state != curr_state)
            {
                curr_state = this.upgrade_state_t.mix;
                return curr_state;
            }
            else
                curr_state = t_state;
        }

        for (var key in this.m_components)
        {
            if(this.m_components[key].Disabled() == this.disabled_t.yes)
                continue;

            t_state = this.m_components[key].UpgradeState();
            
            if(t_state == this.upgrade_state_t.none)
                continue;

            if(curr_state != this.upgrade_state_t.none && t_state != curr_state)
            {
                curr_state = this.upgrade_state_t.mix;
                return curr_state;
            }
            else
                curr_state = t_state;
        }

        return curr_state;
    }
    //###############################################################
    this.Feature.prototype.Size = function ()
    {
        if(this.Disabled() == this.disabled_t.yes)
        {
            this.Log("Size: feature is fully disabled -> size = 0");
            return 0;
        }

        return this.get_size();
    }
    //###############################################################
    this.Feature.prototype.get_size = function ()
    {
        var size = 0;

        for (var key in this.m_components)
            if( this.m_components[key].State() == this.state_t.absent && 
                this.m_components[key].Action() == this.action_t.install)
                size += this.m_components[key].Size();

        for (var ftr in this.m_features) 
            size += this.m_features[ftr].Size();
        
        return size;
    }
    //###############################################################
    this.Feature.prototype.Icon = function () 
    {
        if(this.Disabled() == this.disabled_t.yes)
        {
            this.Log("Icon: feature is fully disabled -> disabled icon");
            return Node.icon_broken_menu;
            //return Node.icon_uninstalled_menu;
        }

        var state = this.State();
        var consistent = this.StateConsistent();
        var act   = this.Action();
        
        this.Log("Icon define: state = " + state + " act = " + act);
        if(act == this.action_t.none)
        {
            if (state == this.state_t.absent)
                return Node.icon_uninstalled_menu;
            else if (state == this.state_t.installed && consistent)
                return Node.icon_installed_menu;
        }
        else if(act == this.action_t.install)
            return Node.icon_install_menu;
        else if (act == this.action_t.remove)
            return Node.icon_uninstall_menu;        
        
        // for cases when act == this.action_t.mix or state == this.state_t.installed && !consistent
        return Node.icon_childs_differ_menu;        
    }
    //###############################################################
    this.Feature.prototype.Menu = function () 
    {
        if(this.Disabled() == this.disabled_t.yes)
        {
            this.Log("Menu: feature is fully disabled -> empty menu");
            return {};
        }

        var menu = {};

        var item_install =
            {
                id: "install",
                name: "Install",
                icon: Node.icon_install,
                act: this.action_t.install
            };

        var item_do_not_remove =
            {
                id: "install",
                name: "Do not remove",
                icon: Node.icon_installed,
                act: this.action_t.install
            };

        var item_remove =
            {
                id: "remove",
                name: "Remove",
                icon: Node.icon_uninstall,
                act: this.action_t.remove
            };

        var item_do_not_install =
            {
                id: "remove",
                name: "Do not install",
                icon: Node.icon_uninstalled,
                act: this.action_t.remove
            };

        var state = this.State();
        var consistent = this.StateConsistent();
        var act   = this.Action();
      
        if(act == this.action_t.none)
        {
            
            if(!consistent)
            {
                menu["install"] = item_install;
                menu["remove"] = item_remove;
            }
            else if(state == this.state_t.installed)
            {
                menu["install"] = item_do_not_remove;
                menu["remove"] = item_remove;
            }
            else
            {
                menu["install"] = item_install;
                menu["remove"] = item_do_not_install;
            }
        }
        else if(act == this.action_t.mix)
        {
            if(!consistent)
            {
                menu["install"] = item_install;
                menu["remove"] = item_remove;
            }
            else if(state == this.state_t.installed)
            {
                menu["install"] = item_do_not_remove;
                menu["remove"] = item_remove;
            }
            else
            {
                menu["install"] = item_install;
                menu["remove"] = item_do_not_install;
            }
        }
        else if(act == this.action_t.remove)
        {
            if(!consistent)
            {
                menu["install"] = item_install;
                menu["remove"] = item_remove;
            }
            else if(state == this.state_t.installed)
            {
                menu["install"] = item_do_not_remove;
                menu["remove"] = item_remove;
            }
        }
        else if(act == this.action_t.install)
        {
            if(!consistent)
            {
                menu["install"] = item_install;
                menu["remove"] = item_remove;
            }
            else if(state == this.state_t.absent)
            {
                menu["install"] = item_install;
                menu["remove"] = item_do_not_install;
            }
        }

        if(this.Mandatory())
            delete menu["remove"];

        return menu;
    }
    //###############################################################
    // ApplyUpgrade method definition
    //###############################################################
    this.Feature.prototype.ApplyUpgrade = function () 
    {
        if(this.Disabled() == this.disabled_t.yes)
        {
            this.Log("Apply Upgrade: Feature is fully disabled -> apply upgrade isn't required");
            return true;
        }

        return this.DoApplyUpgrade();
    }
    //###############################################################
    // DoApplyUpgrade method targeted to be redefinded by descendants
    //###############################################################
    this.Feature.prototype.DoApplyUpgrade = function () 
    {
        if(this.m_apply_upgrade_done)
        {
            this.Log("ApplyUpgrade: was already done -> skip");
            return true;
        }

        this.m_apply_upgrade_done = true;
        
        // upgrade should be applied only in case if action not like remove or none

        var act = this.Action();
        if(act == this.action_t.remove || act == this.action_t.none)
            return true;
        
        this.Log("ApplyUpgrade begin");

        // features own upgrade should be taken only in case if it is absent
        if(this.State() == this.state_t.absent)
            this.Upgrade.Apply();

        for (var cmp in this.m_components_order)
            if(!this.m_components_order[cmp].ApplyUpgrade())
            {
                this.Log("component id = " + this.m_components_order[cmp].Id() + " name = " + this.m_components_order[cmp].Name() + " caused failure during ApplyUpgrade!");
                return false;
            }

        for (var ftr in this.m_features_order)
            if(!this.m_features_order[ftr].ApplyUpgrade())
            {
                this.Log("feature id = " + this.m_features_order[ftr].Id() + " name = " + this.m_features_order[ftr].Name() + " caused failure during ApplyUpgrade!");
                return false;
            }

        this.Log("ApplyUpgrade end");
        return true;
    }
    //###############################################################
    // ApplyResolveSrc method definition
    //###############################################################
    this.Feature.prototype.ApplyResolveSrc = function (dmp)
    {
        if(this.Disabled() == this.disabled_t.yes)
        {
            this.Log("ApplyResolveSrc: Feature is fully disabled -> src resolve isn't required");
            return true;
        }

        return this.DoApplyResolveSrc(dmp);
    }
    //###############################################################
    // DoApplyResolveSrc method targeted to be redefinded by descendants
    //###############################################################
    this.Feature.prototype.DoApplyResolveSrc = function (dmp)
    {
        for (var cmp in this.m_components_order)
            if(!this.m_components_order[cmp].ApplyResolveSrc(dmp))
            {
                this.Log("component id = " + this.m_components_order[cmp].Id() + " name = " + this.m_components_order[cmp].Name() + " caused failure during ApplyResolveSrc!");
                return false;
            }

        for (var ftr in this.m_features_order)
            if(!this.m_features_order[ftr].ApplyResolveSrc(dmp))
            {
                this.Log("feature id = " + this.m_features_order[ftr].Id() + " name = " + this.m_features_order[ftr].Name() + " caused failure during ApplyResolveSrc!");
                return false;
            }

        return true;
    }
    //###############################################################
    // ApplyRemove method definition
    //###############################################################
    this.Feature.prototype.ApplyRemove = function (dmp)
    {
        if(this.m_apply_remove_done)
        {
            this.Log("ApplyRemove: was already done -> skip");
            return true;
        }
        
        this.m_apply_remove_done = true;

    /*
        if(this.Disabled() == this.disabled_t.yes)
        {
            this.Log("ApplyRemove: Feature is fully disabled -> remove isn't required");
            return true;
        }
    */
        return this.DoApplyRemove(dmp);
    }
    //###############################################################
    // DoApplyRemove method targeted to be redefinded by descendants
    //###############################################################
    this.Feature.prototype.DoApplyRemove = function (dmp)
    {
        this.Log("DoApplyRemove started");
        var cmps_rvrs = this.m_components_order.concat([]);
        cmps_rvrs.reverse();

        var ftrs_rvrs = this.m_features_order.concat([]);
        ftrs_rvrs.reverse();

        for (var ftr in ftrs_rvrs)
            if(!ftrs_rvrs[ftr].ApplyRemove(dmp))
            {
                this.Log("feature id = " + this.m_features_order[ftr].Id() + " name = " + this.m_features_order[ftr].Name() + " caused failure during ApplyRemove!");
                return false;
            }
        
        for (var cmp in cmps_rvrs)
            if(!cmps_rvrs[cmp].ApplyRemove(dmp))
            {
                this.Log("component id = " + this.m_components_order[cmp].Id() + " name = " + this.m_components_order[cmp].Name() + " caused failure during ApplyRemove!");
                return false;
            }

        this.Log("DoApplyRemove done");
        return true;
    }
    //###############################################################
    // ApplyInstall method definition
    //###############################################################
    this.Feature.prototype.ApplyInstall = function (dmp)
    {
        if(this.m_apply_install_done)
        {
            this.Log("ApplyInstall: was already done -> skip");
            return true;
        }

        this.m_apply_install_done = true;
        
        if(this.Disabled() == this.disabled_t.yes)
        {
            this.Log("ApplyInstall: Feature is fully disabled -> install isn't required");
            return true;
        }

        return this.DoApplyInstall(dmp);
    };
    //###############################################################
    // DoApplyInstall method targeted to be redefinded by descendants
    //###############################################################
    this.Feature.prototype.DoApplyInstall = function (dmp)
    {
        var apply_dumper = ns_dumper.Dumper("apply_dumper_for_" + this.Id());
        dmp.AddAction(apply_dumper);
        apply_dumper.IgnoreError = this.m_error_handler;

        for (var cmp in this.m_components_order)
            if(!this.m_components_order[cmp].ApplyInstall(apply_dumper))
            {
                this.Log("component id = " + this.m_components_order[cmp].Id() + " name = " + this.m_components_order[cmp].Name() + " caused failure during ApplyInstall!");
                return false;
            }

        for (var ftr in this.m_features_order)
            if(!this.m_features_order[ftr].ApplyInstall(apply_dumper))
            {
                this.Log("feature id = " + this.m_features_order[ftr].Id() + " name = " + this.m_features_order[ftr].Name() + " caused failure during ApplyInstall!");
                return false;
            }

        return true;
    }
    //###############################################################
    // Commit method definition
    //###############################################################
    this.Feature.prototype.Commit = function () 
    { 
        if(this.Disabled() == this.disabled_t.yes)
        {
            this.Log("Commit: Feature is fully disabled -> commit isn't required");
            return true;
        }

        if(this.DoCommit())
        {
            this.OnCommit.Call();
            return true;
        }

        return false;
    };
    //###############################################################
    // DoCommit method targeted to be redefinded by descendants
    //###############################################################
    this.Feature.prototype.DoCommit = function () 
    {
        for (var cmp in this.m_components_order)
            if(!this.m_components_order[cmp].Commit())
            {
                this.Log("component id = " + this.m_components_order[cmp].Id() + " name = " + this.m_components_order[cmp].Name() + " caused failure during commit!");
            }

        for (var ftr in this.m_features_order)
            if(!this.m_features_order[ftr].Commit())
            {
                this.Log("feature id = " + this.m_features_order[ftr].Id() + " name = " + this.m_features_order[ftr].Name() + " caused failure during commit!");
            }
        
        return true;
    }
    //###############################################################
    // RestorePoint method definition
    //###############################################################
    this.Feature.prototype.RestorePoint = function (st)
    { 
        var rp = st ? st : Storage("*");

        rp("id").value = this.m_id;
        rp("name").value = this.m_name;
        rp("description").value = this.m_dscrpt;
        rp("visible").value = this.m_visible ? 1 : 0;
        rp("obj_type").value = this.obj_type;
        rp("install_dir_base").value = this.m_install_dir_base;
        rp("install_dir_own").value = this.m_install_dir_own;
        rp("version").value = this.m_version.Str();

        var ftr_rp = rp("Features");
        var cmp_rp = rp("Components");

        for (var cmp in this.m_components)
            this.m_components[cmp].RestorePoint(cmp_rp(cmp));

        for (var ftr in this.m_features)
            this.m_features[ftr].RestorePoint(ftr_rp(ftr));

        var groups = rp("groups");
        for(var i in this.Groups)
            groups(i).value = this.Groups[i];

        if(this.CnfgOpts)
        {
            var cnfg_opts = rp("ConfigurationOptions");
            for(var i in this.CnfgOpts)
                cnfg_opts(i).value = this.CnfgOpts[i];
        }

        return rp;
    }

    //###############################################################
    this.Feature.prototype.ErrorHandler = function (handler)
    {
        if(handler) this.m_error_handler = handler;

        return this.m_error_handler;
    }

    //###############################################################
    // Components function
    // returns components object
    //###############################################################
    this.Feature.prototype.Components = function (){ return this.m_components; }
    //###############################################################
    // Setting of dependencies processor
    //###############################################################
    this.Feature.prototype.ProcessDependency = function (obj){return true;}
    //###############################################################
    // Dependencies setting
    //###############################################################
    this.Feature.prototype.Depend = function (alias, obj)
    {
        var cur_cmp = this;
        obj.OnActionChange.Connect(function(changed_obj){ return cur_cmp.ProcessDependency(changed_obj);});
        this.Dependencies[alias] = obj;
    }
} // namespace Root.feature

/** @class Node
 *  @brief Binding to Feature infrastructure
 *  @details Feature is C++ handled object with has set of predefined attributes,
 *    most of them are read/write, additionaly it includes methods to operate with
 *    content of feature selection dialog.
 *  @attr string id          feature id, read/write
 *  @attr string name        feature name, this value will be displayed on feature tree, read/write
 *  @attr string description deature description, this value will be displayed on feature
 *                           description pane, read/write
 *  @attr string error       error message, in case if feature is disabled - this value
 *                           will be displayed on feature description pane, read/write
 *  @attr integer size       feature size, this value will be displayed on feature
 *                           size column, read/write
 *  @attr integer priority   feature priority, features on same level are sorted
 *                           according this value (this value doesn't affect to
 *                           installation sequence), read/write
 *  @attr string icon        icon id to display feature on tree, read/write
 *  @attr integer disabled   non-zero value means that feature is disabled (default: false), read/write
 *  @attr integer expanded   non-zero value means that feature is expanded (default: true), read/write
 *  @attr string guid        guild of feature element, this value is automatically generated
 *                           on object creation and should be used as argument for AddChild
 *                           method, readonly
 *  @attr function hit       callback function to be called when user click on feature, see
 *                           notes for details, read/write
 *  @note
 *    Feature is bi-directional element: on one side it represents data available from script
 *    engine, on second side - feature selection dialog uses feature objects to display
 *    tree-based information on tree control and provide feedback to script on any user's
 *    actions: for example call callback function when user click on feature icon. To catch
 *    click on feature element developer should create callback function, like
 *    <pre>
 *      var feature_hit = function()
 *      {
 *          return 0;
 *      }
 *    </pre>
 *    and initialize <code>hit</code> attribute by this function:
 *    <pre>
 *      var n = Node();
 *      n.hit = feature_hit;
 *    </pre>
 *    During feature selection dialog processing when user click on element <code>n</code>
 *    function <code>feature_hit</code> will be called
 *  @see AddChild Menu
 */

/** @fn Node
 *  @brief Constructor for Node object
 *  @return Node - new created Node object
 */

/** @method Node AddChild(string guid)
 *  @brief Append child node to current node
 *  @details Features may be organized on tree-based manner on feature selection dialog.
 *    To do this method AddChild should be used - this method adds link to child node
 *    for current object.
 *  @param string guid - guid of child node to add, see guid attribute
 *  @usage
 *    // example below demonstrates how one node may be linked as child to another node
 *      var n_parent = Node();
 *      n_parent.name = "parent";
 *      var n_child = Node();
 *      n_child = "child";
 *      n_parent.AddChild(n_child.guid);
 */

/** @method Node Menu(array menu_items)
 *  @brief Show pop-up menu
 *  @details When user clicks on feature element on feature selection dialog ayn action may be
 *    processed, for example pop-up menu may be diaplayed to provide user choice to
 *    install/remove any component.
 *  @param array menu_items - array of menu items to display on pop-up menu, see notes
 *    for details
 *  @return data <code>id</code> of element selected by user or empty value
 *  @usage
 *    // example below demonstrates displaing menu on clicking to feature
 *      var node = Node(); // create and initialize node
 *      node.name = "my example node";
 *
 *      var node_hit = function()
 *      {
 *          var menu_items = [];
 *          menu_items.push({name:"menu item 1", icon:"install_icon", id:"install"});
 *          menu_items.push({name:"menu item 2", icon:"uninstall_icon", id:"uninstall"});
 *          var result = node.Menu(menu_items);
 *          if(!result)
 *          { // user canceled menu
 *          }
 *          else if(result == "install")
 *          { // user selected item "menu item 1"
 *          }
 *          else if(result == "uninstall")
 *          { // user selected item "menu item 2"
 *          }
 *      }
 *
 *      // here should be code to display feature selection dialog
 *  @note
 *    Method Menu accepts array of menu elements to display. Menu element is object
 *    which should have attributes:<br>
 *    name - string, text of element to display on pop-up menu<br>
 *    icon - string, icon name to display<br>
 *    id - value to return from Menu method if user select this specific element<br>
 *    To demonstrate how it can be used let's look at example. Let's assume that we created
 *    and initialized Node element:
 *    <pre>
 *      var node = Node();
 *      node.name = "example node";
 *      node.description = "example node description";
 *    </pre>
 *    Now create and initialize callback function:
 *    <pre>
 *      node.hit = function()
 *      {
 *          var menu_items = []; // create array
 *          menu_items.push({name:"menu item 1", icon:"install_icon", id:"install"}); // append menu item
 *          menu_items.push({name:"menu item 2", icon:"uninstall_icon", id:"uninstall"}); // append menu item
 *          var result = node.Menu(menu_items); // call Menu method
 *          if(!result) // process return value
 *          { // user canceled menu
 *          }
 *          else if(result == "install")
 *          { // user selected item "menu item 1"
 *          }
 *          else if(result == "uninstall")
 *          { // user selected item "menu item 2"
 *          }
 *      }
 *    </pre>
 *    As you can see usage of this function is simple
 *  @see Refresh
 */

/** @method Node Refresh
 *  @brief Refresh feature element on feature selection dialog
 *  @details After Node element is updated (for example updated text or icon) corresponding
 *    element on feature selection dialog should be refreshed. Refresh method initiate
 *    re-drawing tree control.
 *  @usage
 *      var node = Node();
 *      node.icon = "install";
 *      node.hit = function()
 *      {
 *          node.icon = "uninstall"; // this action will not affect to GUI
 *          node.Refresh(); // redraw element
 *      }
 *  @see Menu
 */


//##########################################################
// Backup
//##########################################################
    /*
    //###############################################################
    // Feature constructor from RestorePoint
    //###############################################################
    this.CreateFromRP = this.CreateFromRP || function(rp, restore_mngr)
    {
        if(!rp || rp("id").value == "")
            return null;

        var ftr = new ns.Feature();
        if(!ftr.InitFromRP(rp, restore_mngr))
            return null;
        
        return ftr;
    }

    //###############################################################
    this.Feature.prototype.InitFromRP = function(rp, restore_mngr)
    {
        if(!rp || rp("id").value == undefined)
            return false;
        
        this.m_id = rp("id").value;
        this.Log = log_helper("Feature id = " + this.m_id + ": ");

        this.m_name = rp("name").value;
        this.m_dscrpt = rp("description").value;
        this.m_visible = rp("visible").value == "1" ? true : false;
        this.m_size = 0;
        this.m_error = ""; // error description
        this.m_components = {};
        this.m_components_order = [];
        this.m_features = {};
        this.m_features_order = [];
        this.m_node = null;
        this.m_parent = null;
        this.m_install_dir_base = rp("install_dir_base").value;
        this.m_install_dir_own = rp("install_dir_own").value;
        this.m_version = ns_version.Version(rp("version").value);
        this.m_install_dir_locked = false;
        this.Dependencies = {};

        var groups = rp("groups");
        for(var i in groups.childs)
                this.AddToGroup(groups(groups.childs[i]).value);

        if(!restore_mngr)
        {
            this.Log("restore manager isn't defined -> subfeatures and components will not be restored. Restoring complete.");
            return true;
        }

        var ftr_rp = rp("Features");
        var cmp_rp = rp("Components");
        
        this.Log(" restoring components");
        for (var i in cmp_rp.childs)
        {
            this.Log(" restoring component " + cmp_rp.childs[i]);
            var obj = restore_mngr(cmp_rp(cmp_rp.childs[i]));
            if(obj)
                this.m_components[cmp_rp.childs[i]] = obj;
        }
        this.Log(" restoring features");
        for (var i in ftr_rp.childs)
        {
            this.Log(" restoring feature" + ftr_rp.childs[i]);
            var obj = restore_mngr(ftr_rp(ftr_rp.childs[i]))
            if(obj)
            {
                obj.SetParent(this);
                this.m_features[ftr_rp.childs[i]] = obj;
            }
        }

        return true;
    }
    */
