#!/usr/bin/perl -w
#
# (C) Copyright IBM Corporation 2006.
#       Released under GPL v2.
#       Author : Ram Pai (linuxram@us.ibm.com)
#
# contributions from Malahal Naineni, Adrian Rodriguez, Darrick Wong,
#		Joel Diaz.
#
# Changelog:
#
use Getopt::Long;
use strict;
use Cwd;

my $debug = 0; #set this to one for some extra info
my $toolspath=qx(dirname $0);chomp $toolspath;

my $version="0.129";
my $libdir="/lib/modules";
my $moduleext="ko";
my $updatedirectory="updates";
my $cdir = cwd();
my (%opt, $trpm, $rpm, $rpms, @modules, $kernel, $flavor, $fixid);

my $status="SUCCESS";
my %ret = (
	"SUCCESS" 		  => 0,
	"INTERNAL_INCONSISTENCY1" => 1,
	"INTERNAL_INCONSISTENCY2" => 2,
	"DRIVER_CONFLICT"  	  => 3,
	"ABI_FAILURE" 		  => 4,
	"IBM_VALIDATION_PENDING"  => 5,
	"RPM_CORRUPTION"	  => 6,
	"FILESYSTEM_PROBLEM"	  => 7,
	"INTERNAL_INCONSISTENCY3" => 8,
	"UNSUPPORTED_DISTRO" 	  => 9,
	"UNSUPPORTED_ARCHITECTURE"  => 10,
	"RPM_NOEXIST"  		  => 11,
	"RPM_BUILD_PROBLEM"  	  => 12,
	"DRIVER_OVERWRITE_CONFLICT"  => 13,
	"KERNEL_RPM_NOT_INSTALLED"  => 14,
	"NON_WHITELIST_SYMBOL"    => 15,
);

#When running from Director, we inherit a not so useful PATH...deal with that.
$ENV{'PATH'} = "$ENV{'PATH'}:/bin:/usr/bin:/usr/local/bin";
my $redhat=0; my $el;
my $suse=0;
if (-e "/etc/redhat-release"){
        $redhat=(split /\s+/, qx(grep "Enterprise Linux" /etc/redhat-release))[6];
        $redhat=~ s/\..*//;
        $moduleext="o" if ($redhat == 3);
        $updatedirectory = "extra" if ($redhat == 5);
        $el = ($redhat == 5) ? "el5" : "EL";
}else{
        qx(grep 'Red Hat' /etc/issue);
        if ($? == 0){
                $redhat=(split /\s+/, qx(grep "Enterprise Linux" /etc/issue))[6];
                $redhat=~ s/\..*//;
                $moduleext="o" if ($redhat == 3);
                $updatedirectory = "extra" if ($redhat == 5);
                $el = ($redhat == 5) ? "el5" : "EL";
        }
}

if (-e "/etc/SuSE-release"){
        $suse=(split /\s+/, qx(grep -i "SUSE LINUX Enterprise" /etc/SuSE-release))[4];
}else{
        qx(grep -i 'SUSE' /etc/issue);
        if ($? == 0){
                $suse=(split /\s+/, qx(grep -i "SUSE LINUX Enterprise" /etc/issue))[6];
        }
}

unless ($redhat =~ m/[345]/ or $suse =~ /9|10|11/) {
	printf "UNSUPPORTED DISTRO\n";
	exit $ret{"UNSUPPORTED_DISTRO"};
}

my $r_ihv = '(\w+)';
my $r_drivername = '(\w+)';
my $r_flavor = ($redhat >=5) ? '(-(\w+)){0,1}' : '-(\w+)';
my $r_kmp = ($suse >= 10) ? "-kmp" : '';
my $r_kmod = ($redhat >= 5) ? "kmod-" : '';
my $r_name = "${r_ihv}-${r_drivername}${r_kmp}${r_flavor}";
my $r_dist= ($redhat) ? "\.${el}" : '';
my $r_driverver = '(\d+\w*([\._]\d+\w*)+)';
my $r_kernelver = '(\d+(\.\d+)+_\d+(\.\d+)*)'."($r_dist)";
my $r_libkernelver = '(\d+(\.\d+)+-\d+(\.\d+)*)'."($r_dist)";
my $r_ver = ($redhat >=5) ? "${r_driverver}" : "${r_driverver}_${r_kernelver}";
my $r_rev = '(\d+)';
my $r_arch = '(\w+(_\w+)*)';
my $r_rpm = "${r_kmod}${r_name}-${r_ver}-${r_rev}\.${r_arch}\.rpm";
my $r_driver_epoch = '((\d+):)?';
my $r_driver_version = '(\d+\w*([\._]\d+\w*)+)';
my $r_driver_extended_version =  '(-(\w+))?$';
my $r_driver_version_format = "^${r_driver_epoch}${r_driver_version}${r_driver_extended_version}\$";

if (not GetOptions(\%opt, "l|flavor=s", 
		  "g|force-if-overridden",
		  "a|simulate-abi-breakage",
		  "d|always-dup",
		  "r|rpm=s",
		  "f|force",
		  "P|fixid=s",
		  "o|override",
		  "h|help",
		  "v|help-help",
		  "k|kernel=s",
		  "q|query",
		  "u|update",
		  "e|requires",
		  "I|add-initrd",
		  "c|arch=s",
		  "Y|yes") or defined $opt{'h'}) {
        usage();
}
usage_verbose() if (defined $opt{'v'});

#Get the location of the fixid
if (defined $opt{'P'}) {
	use Cwd 'abs_path';
	$fixid = $opt{'P'};
	$fixid = abs_path($fixid);
} else {
	$fixid=$cdir;
}
	$fixid =~ s/\ /\\ /g;

if (defined $opt{'k'}) {
	$kernel=$opt{'k'};
	(my $r_tmp = $r_kernelver) =~ s/_/-/g;
	usage() if ($kernel !~ m/^${r_tmp}$/);
	$kernel =~ s/\.${el}.*// if ($redhat);
} else {
	$kernel=get_current_kernel();
}

if (defined $opt{'l'}) {
	$flavor=$opt{'l'};
} else {
	$flavor=get_current_flavor();
}

my $kernel_rpm;
#check if the corresponding kernel rpm is already installed.
if (not kernel_rpm_installed($kernel, $flavor, \$kernel_rpm)) {
	print "ERROR: kernel $kernel_rpm is not installed\n";
	exit $ret{"KERNEL_RPM_NOT_INSTALLED"};
}

my $arch= defined $opt{'c'} ? $opt{'c'} : (qx(rpm -q $kernel_rpm --qf '%{ARCH}\n'))[0]; 
chomp $arch;
if ($? or not $arch) {
	printf "UNDEFINED ARCHITECTURE\n";
	exit $ret{"UNSUPPORTED_ARCHITECTURE"};
}


# TODO: REMOVEME
# SLES10 puts ppc instead of ppc64 in the rpm ARCH tag.
if ($arch eq "ppc") {
	$arch = "ppc64";
}

my $tmp=qx(mktemp -dq /var/tmp/ibm.XXXXXX);chomp $tmp;
my $abiextract = "$toolspath/abiextract";
my $kexport_extract = "$toolspath/kexport_extract";
my $rpmdir=get_rpm_dir();
my ${unpacked_rpmdir}="$tmp/tmp";
my @duprpmlist=();
my @installrpmlist=();
my $install = '';
my $overwrite = undef;
my $erase = undef;

#Get the list of rpms
if (defined $opt{'r'}) {
	$trpm = $opt{'r'};
	unless ( -f $trpm and system("rpm -qp $trpm")) {
		print "ERROR:$trpm does not exist or is corrupted\n";
		exit $ret{"RPM_NOEXIST"};
	}
	$trpm = "$cdir/$trpm"  if ($trpm !~ m(^/));
} else {
	$trpm = get_rpm($fixid, $kernel, $flavor);
}

if (not defined $trpm) {
	print "NOTE: No installable rpm found\n";
	exit $ret{"RPM_NOEXIST"};
}

if (defined $opt{'e'}) {
	system("rpm -q --requires -p $trpm");
	exit $?;
}
#if queried, process the  query option and return
if (defined $opt{'q'}) {
	my $rpm = $trpm;
	my ($rpmkernel, $rpmflavor)=get_rpm_kernel_flavor($rpm);
	if (unpackrpm($rpm, ${unpacked_rpmdir}, \@modules)) {
		print "ERROR: $rpm cannot be unpacked\n",
		exit $ret{"INTERNAL_INCONSISTENCY1"};
	}
	my $retval;
	foreach (@modules) {
		my $this_module_version = qx(modinfo -F version $_);
		next unless(query($_, $kernel, $rpmflavor, $this_module_version,
				 \$retval));
		qx(rm -rf $tmp);
		exit $retval;
	}
	print "NOTE: the up-to-date and out-of-date information is based on ",
	      "the proper use of versioning scheme in the distro-driver ",
	      "A manual inspection is recommended\n";
	qx(rm -rf $tmp);
	exit 0;
}
usage() if (not defined $opt{'u'});

print "\n\tDrivers will be installed/migrated to $kernel version\n\n";
if (not defined $opt{'Y'}) {
	#Check with the user if he/she has verfied if migrating the modules
	#to new kernel is acceptable.
 	print "WARNING: PLEASE CHECK WITH IBM IF THE MIGRATION OF THE ",
		"FOLLOWING rpm TO KERNEL $kernel IS BLESSED\n\t";
	print "$trpm\n\t";
	print "\nOK TO CONTINUE [Yy/Nn]:";
	$_ = <STDIN>;
	if ( not /^[yY]$/ ) {
		$status = "IBM_VALIDATION_PENDING";
		exit $ret{$status};
	}
}
# yes this loop will execute only once. However it has been structured
# this way to avoid goto in case of failure. 'next' will take us 
# to the right place.
foreach $rpm ($trpm) {
	my $rpm_short_name = qx(basename "$rpm");
	print '-'x70,"\n","Checking $rpm_short_name......\n";

	my @installedrpms;
	my ($rpmkernel, $rpmflavor)=get_rpm_kernel_flavor($rpm);
	unless ($rpmkernel or $rpmflavor) {
		print "WARNING:$rpm  kernel-version or kernel-flavor ",
			"is undefined.\nCheck if $rpm corrupted?\n";
		next;
	}

	my($rpm_name, $rpmdriver_version, $rpmkernel_version) = 
			get_rpm_details($rpm);
	#check if a driver rpm for that kernel/flavor is already
	#installed.
	if (driver_rpm_already_installed($kernel, 
				$flavor, $rpm_name, $rpmkernel_version,
				$rpmdriver_version, 
				\$install, \$erase)) {
		if (not defined $opt{'f'} and not defined $opt{'g'}) {
			$erase=undef;
			print "Hey! rpm is already installed. Nothing to do\n";
			next;
		}
	}

        if (not defined $opt{'a'} and ($redhat == 5 or $suse == 10 or $suse == 11)) {
                if (grep {/ksym\(\w+\)\s*=\s*\w+/} qx(rpm -q --requires -p $trpm)) {
                        if ($redhat) {
                                print "ERROR: Non Whitelist symbol detected\n";
                                $status = "NON_WHITELIST_SYMBOL";
                                next;
                        } elsif ($suse) {
                                print "ERROR: RPM is corrupt\n";
                                $status = "RPM_CORRUPTION";
                                next;
                        }
                }
        }

	if (unpackrpm($rpm, ${unpacked_rpmdir}, \@modules)) {
		print "ERROR: $rpm cannot be unpacked\n",
		exit $ret{"INTERNAL_INCONSISTENCY1"};
	}
	unless (@modules) {
		print "ERROR: no driver modules found in $rpm\n";
		exit $ret{"RPM_CORRUPTION"};
	}

	if (check_modules_paths(\@modules)) {
		print "ERROR: modules in $rpm do not install driver in ",
			"'${updatedirectory}' directory\n";
		exit $ret{"RPM_BUILD_PROBLEM"};
	}

	if (not defined $opt{'f'} and 
		   installed_driver_conflict(\@modules, $kernel,
				$rpmflavor, $opt{'o'}, $opt{'g'},
				\$overwrite)) {
		$status = "DRIVER_CONFLICT";
		next;
	}

	#check if the kernels' ABI matches that required by the modules.
	if (defined $opt{'a'} or
			check_abi_mismatch(\@modules, $kernel, $rpmflavor)) {
		print "ERROR: ABI breakage detected\n";
		$status = "ABI_FAILURE";
		next;
	}

        if (not defined $opt{'d'} and $rpmkernel eq $kernel) {
                push(@installrpmlist, $rpm);
                next;
        }

	if (kmp_enabled()) {
		push(@installrpmlist, $rpm);
		next;
	} 
	print "Attempting to generate & install a duped rpm for this kernel\n";
	my $duprpm;
	next if (buildduprpm($rpm, ${unpacked_rpmdir}, $kernel, $rpmkernel, 
			$rpmflavor, \$duprpm, \$status));
	push(@duprpmlist, $duprpm);
}
if (not defined $opt{'Y'} and defined $overwrite) {
	#Check with the user if overwriting some of the already
	#installed files is ok.
	print "WARNING: SOME OF THE INSTALLED FILES WILL BE OVERWRITTEN\n";
	print "\nOK TO CONTINUE [Yy/Nn]:";
	$_ = <STDIN>;
	if ( not /^[yY]$/ ) {
		$status = "DRIVER_OVERWRITE_CONFLICT";
		exit $ret{$status};
	}
}

qx(rm -rf $tmp);

print '-'x70,"\n";
if ($ret{$status}) {
	#ABORT PHASE
	#UNDO ALL THAT WE DID TILL NOW.
	printf("\t\tERROR: $status\n");
	printf("MIGRATION OF THE RPMS CANNOT BE COMPLETED SUCESSFULLY\n");
	printf("BOOTING UP ON THE $kernel VERSION WILL MISS THESE DRIVERS\n");
	printf("CONTACT IBM FOR SUPPORT\n");
	printf("TILL THEN DO NOT BOOT ON THE $kernel KERNEL\n");
	foreach (@duprpmlist) {
		qx(rm -f $_);
	}
	exit $ret{$status};
}

#NOTE WE ARE WORKING UNDER THE ASSUMPTION THAT NO TWO DRIVERS FROM ANY OF
#IBM SUPPLIED RPMS WILL HAVE THE SAME NAME AND INSTALL IN THE SAME DIRECTORY!

#TIME TO COMMIT ALL THE HARD WORK!
my $oldflag='';
if (defined $opt{'f'} or defined $opt{'g'} or defined $overwrite) {
	$oldflag="--force";
}

if (defined $erase) {
	foreach ($erase) {
		qx(rpm -e $_ 2> /dev/null);
		if($? != 0) {
			warn __LINE__, "SCRIPT INTERNAL INCONISISTENCY! REPORT TO IBM IMMEDIATELY\n";
			exit $ret{"INTERNAL_INCONSISTENCY1"};
		}
	}
}

foreach (@installrpmlist) {
	#push all the modules into initrd if specified.
	add_initrd($_) if (defined $opt{'I'});

	#Need to add an entry into /etc/depmod.d/depmod.conf
	#before installing the RPM, otherwise the soft-links
	#in weak-updates will not override the kernel driver
	#when depmod is run as part of the RPM installation.
	add_override_to_depmod_rhel5($_) if ($redhat == 5);

	# rpm command does not like space in the path , So escape space and fool rpm.
	$_ =~ s/\ /\\ /g;
	qx(rpm $oldflag $install -ivh "$_" 2>/dev/null);
	if($? != 0) {
		warn __LINE__, " SCRIPT INTERNAL INCONISISTENCY! REPORT TO IBM IMMEDIATELY\n";
		exit $ret{"INTERNAL_INCONSISTENCY1"};
	} else {
		my $base = qx(basename $_); chomp $base;
		printf "$base installed successfully\n";
	}
}
foreach (@duprpmlist) {
	#push all the modules into initrd if specified.
	add_initrd($_) if (defined $opt{'I'});
	qx(rpm $oldflag -U $_ 2>/dev/null);
	if ($? != 0) {
		warn __LINE__, "SCRIPT INTERNAL INCONISISTENCY! REPORT TO IBM IMMEDIATELY\n";
		exit $ret{"INTERNAL_INCONSISTENCY2"};
	} else {
		my $base = qx(basename $_); chomp $base;
		printf "$base installed successfully\n";
	}
}
printf "SUCCESS\n";
exit $ret{"SUCCESS"};


##############################################################
# 	Beginning of helper functions			     #
##############################################################
sub usage {
        print "Usage:\n\n$0", ' --update [--flavor flavor] ',
		'[--kernel kernelversion]',
		'[--arch architecture]',
		"\n\t", '[ --add-initrd ] [--force | --force-if-overridden | --override]',"\n\n",
	    "$0",' --help ',"\n\n",
	    "$0",' --query [ --kernel kernelversion ] [ --flavor flavor ]',"\n",
	    "$0",' --requires [--kernel kernelversion] [--flavor flavor]',"\n\n",
	    "-yes : Total-Yes mode, WARNING:Be careful while using this mode\n",
	    "-kernel version:install/migrate to this version of the kernel\n",
	    "-flavor flavor: flavor of the kernel to install/migrate\n",
	    "-override : override the distro driver if its at lower version \n",
	    "\tor distro driver is already overriden and this driver is \n",
	    "\tlatest \n",
	    "-force : force install this driver\n",
	    "\twithout -override or -force this driver is installed only\n",
	    	"\tif distro ships this driver \n",
	    "-force-if-overridden : force install this driver only if its\n",
	    	"\talready overriden.\n",
	    "--add-initrd : add the drivers to the initrd  \n\n",
	    "-query : query the fixid driver and compare it with the one ",
		"installed \n",
	    "-requires : query what the rpm requires \n",
            "-help : help \n\n",
	    "If no other mode is specified then by default install/migrate\n",
	    "\tmode gets triggered \n",
	    "\n\nThis is version $version of the script\n";
	die "bye\n";
}

sub usage_verbose {
        print "Usage:\n\n$0", ' --update [--fixid dir | --rpm rpm] ',
		'[--flavor flavor]',
		'[--arch architecture]',
		"\n\t", '[--yes] [--force] [--kernel kernelversion] ',
		'[ --force-if-overridden | --override | --force ] ', 
		'[--add-initrd] ',
		'[--always-dup] [--simulate-abi-breakage]',"\n\n",
	    "$0",' --help ',"\n\n",
	    "$0",' --help-help ',"\n\n",
	    "$0",' --query [ --kernel kernelversion ] [ --flavor flavor ]',"\n",
		,"\t",'[ --fixid dir ] [ --rpm rpm ]', "\n\n",
	    "$0", '--requires [--kernel kernelversion] [--flavor flavor]',"\n",
		,"\t",'[ --fixid dir ] [ --rpm rpm ]', "\n\n",
	    "-yes : Total-Yes mode, WARNING:Be careful while using this mode\n",
	    "-rpm : the rpm is assumed to reside in the same directory. If\n",
	    "\t not, use this option to specify the location of the rpm \n",
	    "-kernel version:install/migrate to this version of the kernel\n",
	    "--fixid dir: location of the untared FIXID\n",
	    "--flavor flavor: flavor of the kernel to install/migrate\n",
	    "--override : override the distro driver if its at lower version \n",
	    "\tor distro driver is already overriden and this driver is \n",
	    "\tlatest \n",
	    "-force : force install this driver\n",
	    "\twithout -override or -force this driver is installed only\n",
	    	"\tif distro ships this driver \n",
	    "-force-if-overridden : force install this driver only if its\n",
	    	"\talready overriden.\n",
	    "--add-initrd : add the drivers to the initrd  \n",
	    "-simulate-abi-breakage : assume ABI is broken.\n",
	    	"\tUse for ABI failure simulation only\n\n",
	    "-query : query the fixid driver and compare it with the one ",
		"installed \n",
	    "-requires : query what the rpm requires \n",
            "-help : help \n",
            "-help-help : verbose help describing some hidden options\n\n",
	    "If no other mode is specified then by default install/migrate\n",
	    "\tmode gets triggered \n",
	    "\n\nThis is version $version of the script\n";
	die "bye\n";
}

sub kmp_enabled {
	return ($suse == 10 || $suse == 11 || $redhat == 5);
}

sub get_mkinitrd_kernel {
	my $kernel = shift;
	my $flavor = shift;

	if ($redhat) {
		$flavor =~ s/^default$//;
		return "${kernel}.${el}${flavor}";
	} elsif ($suse == 9 || $suse ==10) {
		return "${kernel}-${flavor}";
	} elsif ($suse == 11) {
                my $kerdir = qx(echo $kernel | sed -e 's/\\.[0-9]*\$//');
                chomp $kerdir;
                return "${kerdir}-${flavor}";
	}
	return;
}

sub get_moduledir_name {
	my $kernel = shift;
	my $flavor = shift;
	if ($redhat) {
		$flavor =~ s/^default$//;
		return "${kernel}.${el}${flavor}";
	} elsif ($suse == 9 || $suse == 10) {
		return "${kernel}-${flavor}";
	} elsif ($suse == 11) {
		my $kerdir = qx(echo $kernel | sed -e 's/\\.[0-9]*\$//');
		chomp $kerdir;
		return "${kerdir}-${flavor}";
	}
	return;
}

sub get_module_dir {
	my $kernel = shift;
	my $flavor = shift;

	my $dirname=get_moduledir_name($kernel, $flavor);
	return "${libdir}/$dirname";
}

sub get_pcimap_file {
	my $kernel = shift;
	my $flavor = shift;

	my $moddir =  get_module_dir($kernel, $flavor);
	return "$moddir/modules.pcimap";
}

sub get_depmod_file {
	my $kernel = shift;
	my $flavor = shift;

	my $moddir =  get_module_dir($kernel, $flavor);
	return "$moddir/modules.dep";
}

sub get_rpm_dir {
	my $rpmdir;
	if ($redhat) {
		$rpmdir="/usr/src/redhat/RPMS/$arch";
	} elsif ($suse) {
		$rpmdir="/usr/src/packages/RPMS/$arch";
	}
	return $rpmdir;
}

sub get_current_kernel {
	my $kernel=qx(uname -r);
	chomp $kernel;
	if ($redhat) {
		$kernel =~ s/\.${el}.*//;
	} elsif ($suse) {
		$kernel =~ s/-\w+$//;
	} else {
		$kernel=undef;
	}
	return $kernel;
}

sub get_current_flavor {
	my $flavor=qx(uname -r); chomp $flavor;
	if ($redhat) {
		$flavor =~ s/^.*.${el}$/default/;
		$flavor =~ s/^.*.${el}(\w+)$/$1/;
	} elsif ($suse) {
		$flavor =~ s/^.*-(\w+)$/$1/;
	} else {
		$flavor=undef;
	}
	return $flavor;
}

sub get_rhel5_rpm_kernelversion {
	my $rpm = shift;
	my $flag = ($rpm =~ /\.rpm$/) ? "-p" : '';
	my $i= (grep {m($libdir/([\w\d\.-]+)$)} qx(rpm -q --list $flag "$rpm"))[0]; 
	chomp $i;
	$i =~ s($libdir/)();
	$i =~ s(${r_dist}.*)();
	$i =~ s/-/_/;
	return $i;
}

sub get_rpm_details {
	my $rpm = shift;
	my $flag = ($rpm =~ /\.rpm$/) ? "-p" : '';
	#get the version of the kernel this rpm is for
	# "rpm" command does not like space in the path so escape space.
	$rpm =~ s/\ /\\ /g;

	my($rpm_name, $rpm_version) = 
		split /\s+/, qx(rpm -q --qf '%{NAME} %{VERSION}' $flag "$rpm");
	$rpm_version =~ /${r_ver}/;
	my $rpmdriver_version = $1;
	my $rpmkernel_version;

	if ($redhat == 5) {
		$rpmkernel_version = get_rhel5_rpm_kernelversion($rpm);
	} else {
		$rpmkernel_version = $3;
	}

	return ($rpm_name, $rpmdriver_version, $rpmkernel_version);
}

sub correct_rpm_name_format {
	my $rpm = shift;
	return ($rpm =~ m(.*/${r_rpm}));
}

sub get_rpm_kernel_flavor {
	my $rpm = shift;
	my $kernel; my $fl;

	$rpm =~ s/\ /\\ /g;
	$rpm =~ /.*\/${r_rpm}/;

	if ($redhat == 5) {
		$kernel = get_rhel5_rpm_kernelversion($rpm);
		$fl = defined $4 ? $4 : "default";
		$kernel =~ s/_/-/g;
	} else {
		$fl = $3;
		($kernel = $6) =~ s/_/-/g;
	}
	return ($kernel, $fl);
}

# compare the first array and the second array from start
# index to end index and tell if they both are identical.
sub same_version {
	my $first = shift;
	my $second = shift;
	my $start = shift;
	my $end = shift;

	foreach ($start..$end) {

		next if (not defined ${$first}[$_] and not defined ${$second}[$_]);
		return 0 if (not defined ${$first}[$_] and defined ${$second}[$_]);
		return 0 if (defined ${$first}[$_] and not defined ${$second}[$_]);
		return 0 if (${$first}[$_] != ${$second}[$_]);
	}
	return 1;
}

sub get_rpm {
	my $fixid = shift;
	my $kernel = shift;
	my $flavor = shift;
	my ($retrpm, $retbuild, $rpm_kver, $fl, @rpms, $rpm);
	my @kver = split "[\.-]", $kernel;
	my $dir;

	if ($suse) {
		(my $tarch = $arch) =~ s/i\d86/i386/;
		@rpms=<$fixid/sles${suse}/${tarch}/update/SUSE-SLES/${suse}/rpm/*.${arch}.rpm>;
	} else {
		$dir="rhel${redhat}";
		@rpms = qx(find $fixid/$dir/RPMS -name "*.${arch}.rpm");
	}

	#############################################################
	# get the exact rpm
	# if not get the rpm belonging to the same build number
	# if not get the rpm belonging to the highest older build
	# if not get the rpm belonging to the lowest newer build
	#############################################################

	#for sles9, the first 4 fields in the version string
	#are constant. And 3 for other releases rhel3/rhel4/rhel5/sles10. 
	#Odd ha?
	my $no_kernel_fields = ($suse == 9)? 4 : 3; 

	my $kbuild = $kver[$no_kernel_fields];
	chomp @rpms;
	foreach $rpm (@rpms) {
		#check if the rpm name is in the expected format 
		unless (correct_rpm_name_format($rpm)) {
			print "WARN: IGNORING: unexpected rpm name format",
			      "\n$rpm\n";
			next;
		}

		($rpm_kver, $fl) = get_rpm_kernel_flavor($rpm);
		next unless ($fl eq $flavor);

		#get the rpm's version
		my @rpmver = split '[\.-]', $rpm_kver;
		my $rpmbuild = $rpmver[$no_kernel_fields];

		#ensure that the major, minor and maint version of the
		#kernel and the kernel version of the rpm match.
		next if (not same_version(\@kver, \@rpmver, 
				0, $no_kernel_fields-1));
		
		return $rpm if (same_version(\@kver, \@rpmver, 
				$no_kernel_fields, (@kver-1)));


		#return the rpm, if it matches the targeted kernel's 
		#build number. if the build numbers match, means that
		#the the rpm is in the errata range. implying a high
		#probability that the driver is ABI compatible.
		if ($rpmbuild == $kbuild) {
			$retrpm = $rpm; $retbuild = $rpmbuild;
		} elsif ((not defined $retbuild) or 
			(($rpmbuild < $kbuild and 
				($retbuild > $kbuild or $retbuild < $rpmbuild)) 
					or
			($rpmbuild > $kbuild and $rpmbuild < $retbuild))) {
				$retrpm = $rpm; $retbuild = $rpmbuild;
		}
	}
	return $retrpm;
}

sub kernel_rpm_installed {
	my $kernel = shift;
	my $flavor = shift;
	my $rpm = shift;
	if ($redhat) {
		$flavor =~ s/(.*)/$1-/;
		$flavor =~ s/default-//;
		${$rpm} = "kernel-${flavor}$kernel.${el}";
		qx(rpm -q ${$rpm});
	} elsif ($suse) {
		${$rpm} = "kernel-${flavor}-$kernel";
                ${$rpm} = qx(rpm -qa | grep ${$rpm});
                chomp ${$rpm};
	}
	return ($? eq 0);
}


#for x version of the kernel check if y version of driver is 
#already installed?
sub driver_rpm_already_installed {
	my $kernel_ver = shift;
	my $flavor = shift;
	my $rpm_name = shift;
	my $rpm_kversion = shift;
	my $rpm_dversion = shift;
	my $install = shift;
	my $erase = shift;

	if ($suse == 10 || $suse ==11) {
		return 1 if (kmp10_already_installed($rpm_name, 
				$rpm_kversion, $rpm_dversion,
				$install, $erase));
		return 0;
	}

	if ($redhat == 5) {
		if (rhel5_rpm_already_installed($kernel_ver,
				$flavor, $rpm_name, $rpm_dversion, $install,
				$erase)) {
			return 1;
		} else {
			return 0;
		}
	}

	$kernel_ver =~ s/-/_/;
	#check if the lab generated rpm is already installed
	return 1 if (rpm_already_installed($kernel_ver, 
			$flavor, $rpm_name, $rpm_dversion, $install,
			$erase));

	#check if a dup rpm is installed
	return 1 if (dup_already_installed($kernel_ver, 
			$flavor, $rpm_name, $rpm_dversion, $install,
			$erase));

	#check if the kernel-update-tool has already migrated the driver
	return 1 if ($suse == 9 and 
			kmp9_already_installed($kernel_ver,
				$flavor, $rpm_name, $rpm_dversion));

	return 0;
}


# return 1 if rpm is already installed. However if the rpm is not installed,
# then return the flags and the rpms to be erased in order to install this rpm.
#
# All our lab generated rpms generated for different kernels unfortunately
# contain the same name tag. This mean rpm thinks that the same rpm is being
# installed even though the rpm is for a diffrent kernel. To work around that
# this routine checks the versions for the rpms and makes the following
# decision
#
# If the kernel version for which this rpm is built is the same as the the one
# which is already installed and if the driver version is is newer, erase the
# installed rpm and install this rpm. However if the driver version is equal or
# lower just say 'driver is already installed'.
#
# If the kernel version of which this rpm is built is different than the one
# already installed, than 'force' install this rpm. NOTE: the 'force' may feel
# scary. Dont worry; we are installing the rpm for a different kernel.
#
sub rpm_already_installed {
	my $kern_version = shift;
	my $flavor = shift;
	my $rpm_name = shift;
	my $rpm_version = shift;
	my $install = shift;
	my $installed_rpm = shift;

	#get the versions of all the kernels this driver is installed for
	my @rpms = qx(rpm -q --whatprovides ${rpm_name}); chomp @rpms;
	return 0 unless ($? eq 0);

	foreach $rpm (@rpms) {
		chomp $rpm;
		my($rpm_name, $tdriver_ver, $tkernel_ver) = 
			get_rpm_details($rpm);
		if ($tkernel_ver eq $kern_version) {
			${$install} = '';
			${$installed_rpm} = $rpm;
			return 1 unless (newer_version($rpm_version, 
					$tdriver_ver));
			return 0;
		}
		${$install} = "--force" unless(newer_version($kern_version,
						$tkernel_ver));
	}
	return 0;
}

# The name of this function is misleading.  We aren't really checking only
# if an RPM has been installed.  A soft link in the right place also 
# satisfies the driver being installed for the targeted kernel.
# In RHEL5 a driver is installed into whatever kernel it was compiled against.
# Then a script runs and makes soft links to this driver for any ABI compatible
# kernels.
# So we need to see if our targeted kernel either has a driver already in its
# directory, or if there is a soft link to a real driver.
sub rhel5_rpm_already_installed {
	my $target_kern_version = shift;
	my $target_flavor = shift;
	my $new_rpm_name = shift;
	my $new_rpm_version = shift;
	my $install = shift;
	my $installed_rpm = shift;

	#get the versions of all the kernels this driver is installed for
	my @rpms = qx(rpm -q --whatprovides ${new_rpm_name}); chomp @rpms;
	return 0 unless ($? eq 0);
	chomp @rpms;

	foreach $rpm (@rpms) {
		my($trpm_name, $tdriver_ver, $tkernel_ver) =
			get_rpm_details($rpm);
		$tkernel_ver =~ s/_/-/g;
		if ($tkernel_ver eq $target_kern_version) {
			${$install} = '';
			${$installed_rpm} = $rpm;
			if (newer_version($new_rpm_version, $tdriver_ver)) {
				#Return 0 b/c the new RPM has a 
				#newer version
				return 0;
			} else {
				return 1;
			}
		} else {
			# While this RPM may not be for the targeted kernel
			# there may be softlinks in the targeted kernel
			# that point to the RPM in question's drivers.
			# Check if the targeted kernel has soft links
			# in the 'weak-updates' directory.
			my @rpms_driver_path = 
			    qx(rpm -ql $rpm | grep ${moduleext}\$ );
			my @target_kernel_driver_path = @rpms_driver_path;
                        # Replace each entry's path extra with weak-updates
                        # b/c that's where the soft-links live in RHEL5
                        for (@target_kernel_driver_path)
                            { s/extra/weak-updates/g }

			# Get the kernel version the RPM installs into
			my $rpm_kernel_version = (get_rpm_details($rpm))[2];
			$rpm_kernel_version =~ s/_/-/g;

			# Add flavor information so it looks like 'uname -r'
			# output
			(my $ttarget_flavor = $target_flavor) =~ s/default//;
			$rpm_kernel_version = 
			    "${rpm_kernel_version}.el5${ttarget_flavor}";

			# Make target_kernel_version look like 'uname -r'
			# output
			my $ttarget_kern_version = 
			    "${target_kern_version}.el5${ttarget_flavor}";

			# Replace the original kernel version with
			# the target one.
			# Now we have the path where we could potentially have
			# a soft link to the real driver.
		        for (@target_kernel_driver_path) 
			    { s|$rpm_kernel_version|$ttarget_kern_version|g }

			chomp @target_kernel_driver_path;
                        chop  @rpms_driver_path;

			# just move on to the next module.
                        foreach my $one_module (@target_kernel_driver_path){   ##
	                             if ( ! -e $one_module && 
			                ! -l $one_module) {
			               	next;
		                 	}
                              
	      
		        	my $link_points_to = 
			        qx(readlink $one_module);
		        	chomp $link_points_to;

			        # Finally can test whether the soft link file
			        # points to one of the  driver installed by
                                # the RPM.
		                foreach my $one_rpm (@rpms_driver_path) {
                         	        if ($link_points_to eq $one_rpm) {
				                my $rpms_driver_version = 
				                    (get_rpm_details($rpm))[1];
				                if (newer_version($new_rpm_version,
				                   $rpms_driver_version)) {
					               ${$install} = '';
					               ${$installed_rpm} = $rpm;

					               # Return 0 b/c the new RPM has a
					               # newer version
					               return 0;
				                } else {
					               ${$install} = '';
					               ${$installed_rpm} = $rpm;
					               return 1;                                 ##
				                }
			                }
		               } #end foreach $one_rpm
		
	                } #end foreach $one_module
                } 
        } #end foreach $rpm
	return 0;
}

sub dup_already_installed {
	my $kern_version = shift;
	my $flavor = shift;
	my $rpm_name = shift;
	my $rpm_version = shift;
	my $install = shift;
	my $installed_rpm = shift;

	$kern_version = "${kern_version}.${el}" if ($redhat);
	$rpm_name =~ s/-$flavor//;
	my @rpms = qx(rpm -q --whatprovides ${rpm_name}-${kern_version}-${flavor});
	chomp @rpms;
	return 0 if ($?);
	
	foreach $rpm (@rpms) {
		my $ver = (grep {s(${rpm_name}_for_${kern_version}-${flavor} = ${r_ver}-${r_rev})($1)} qx(rpm -q --provides $rpm))[0];
		${$installed_rpm} = $rpm;
		return 1 unless(newer_version($rpm_version, $ver));
	}
	return 0;
}

sub kmp9_already_installed {
	my $kern_version = shift;
	my $flavor = shift;
	my $rpm_name = shift;
	my $rpm_version = shift;

	$rpm_name =~ s/-$flavor//;
	my @rpms = qx(rpm -q --whatrequires kernel-${flavor}); chomp @rpms;
	return 0 if ($?);

	foreach $rpm (@rpms) {
		my ($name, $version) = split /\s+/, 
				qx(rpm -q --qf '%{NAME} %{VERSION}' $rpm);

		next unless ($name =~ m/^$rpm_name/ and defined $version);

		(my $kver = $version) =~ s/.*_for_${r_kernelver}_$flavor/$1/;
		next unless (defined $kver and ($kver eq $kern_version));

		(my $dver = $version) =~ s/${r_ver}.*_for_.*/$1/;
		return 1 unless(newer_version($rpm_version, $dver));
	}
	return 0;
}

#
# return 1 if a kmp with the same name is built for the same kernel
# as that of the kmp is already installed and has a driver which is
# same or better version.
#
sub kmp10_already_installed {
	my $my_name = shift;
	my $my_kver = shift;
	my $my_dver = shift;
	my $install = shift;
	my $installed_rpm = shift;

	my @rpms = qx(rpm -q --whatprovides ${my_name});
	return 0 if ($?);
	chomp @rpms;

	foreach $rpm (@rpms) {

		#get kernel version and driver version of the rpm.
		my $ver = qx(rpm -q --qf '%{VERSION}' $rpm);
		next unless (defined $ver);

		(my $kver = $ver) =~ s/${r_ver}/$3/;
		(my $dver = $ver) =~ s/${r_ver}/$1/;
		next unless (defined $kver and ($kver eq $my_kver));

		${$installed_rpm} = $rpm;
		return 1 unless(newer_version($my_dver, $dver));
	}
	return 0;
}

sub get_kernel_symvers {
	my $kernel = shift;
	my $flavor  = shift;
	my $modules  = shift;
	my $modsymvers = shift;
	my $rpm;

	${$modsymvers}="${tmp}Module";
	if ($redhat) {
		$flavor =~ s/(.*)/$1-/;
		$flavor =~ s/default-//;
		$rpm="kernel-${flavor}$kernel.${el}";

		if ($redhat == 5) {
			# We already have an equivalent Module.symvers in RHEL5
			# ...it's found in /boot/symvers-kernver.gz
			my $msymvers = qx(rpm -q --list $rpm | grep symvers);
			chomp $msymvers;
			if(system("gunzip < $msymvers > ${$modsymvers}")) {
				die "gunzip command failed";
                        }
                        # Now add the module symbol information from the 
                        # current drivers included in the fixid.
                        # This allows for a situation where module A included
                        # in the fixid depends on module B which is also 
                        # included in the fixid. Without this information
                        # in our generated module symbol file, then 
                        # we'd get an ABI break detection which isn't true
                        # b/c the fixid provides to praticular symbols to 
                        # itself.
                        system("$kexport_extract @{$modules} >> ${$modsymvers}");       
		} else {
			my $vmlinuu = ($redhat == 3) ?  "vmlinux" : "vmlinuz";
			my @files= grep{m(${libdir}.*\.${moduleext}|${vmlinuu})}
					qx(rpm -q --list $rpm); chomp @files;

			# Collect all symbol information in ${$modsymvers}
			qx($kexport_extract @files @{$modules} > ${$modsymvers} 2>/dev/null);
			print "Check if $kexport_extract is installed\n" if($?);
		}
		return $?;
	} elsif ($suse == 9 or $suse == 10 or $suse == 11) {
		$rpm=  "kernel-${flavor}-$kernel";
		print "rpm=$rpm \n kernel=$kernel \n flavor=$flavor \n suse $suse \n";
	#	my $msymvers = qx(rpm -q --list $rpm | grep Module.symvers); ##removed axr
	
##new way of finding the symvers file -- rpm way wasn't working correctly
my $msymvers = qx( find /usr/src/linux*  | grep Module.symvers | grep $flavor | grep $kernel);

	return $? if ($?);
		
		chomp $msymvers;
		system("cp -f $msymvers ${$modsymvers}") or
			system("$kexport_extract @{$modules} >> ${$modsymvers}");
		return $?;
	}
}

#
# check_abi_mismatch Module.symvers kernelversion
# 	if the ABI of the module matches that of the kernel return 0
#	else return 1
#
sub check_abi_mismatch {
	my $modules = shift;
	my $kernel  = shift;
	my $flavor  = shift;
	my (%SYMBOL, $kernel_symvers, $ret);

	if (get_kernel_symvers($kernel, $flavor, $modules, \$kernel_symvers)) {
		print "$kernel is not installed. Cannot find Module.symvers ",
			"file corresponding to this kernel. Sorry\n";
		return 1;
	}

	my $module=@{$modules}[0];

	if (open(MODULE_SYMVERS, $kernel_symvers)) {
		while ( <MODULE_SYMVERS> ) {
			chomp;
			my ($crc, $sym) = (split)[0,1];
			$SYMBOL { $sym } =  $crc;
		}
	} else {
		print "Sorry, cannot open the kernel's symbol ";
		print "version file $kernel_symvers: $!\n";
		return 1;
	}
	close MODULE_SYMVERS;

	foreach $module (@{$modules}) {
		unless (open(MODULE_MODULE, "$abiextract $module |")) {
			print "Sorry, cannot open the module's object file $!\n";
			print "Check if $abiextract is installed\n";
			qx(rm -f "$kernel_symvers");
			return 1;
		}
		#compare the symbols to $SYMBOL
		$ret=0;
		while ( <MODULE_MODULE> ) {
			chomp;
			my ($m_crc, $symbol) = (split)[0,1];
			if (not defined $m_crc) {
				next;
			}
			if ( $m_crc !~ /^(0x|R\w+_|R)[0-9a-f]{7,8}$/ ) {
				next;
			}

			next if ($redhat == 3 and $symbol eq "__this_module");

			if (not defined $SYMBOL{$symbol}) {
				print "ERROR: $symbol: missing\n";
				$ret=1;
				last;
			}
			my $crc = $SYMBOL{$symbol};
			if ($crc ne $m_crc) {
				print "ERROR: $symbol: ABI do not match\n";
				$ret=1;
				last;
			}
		}
		close(MODULE_MODULE);
		last if ($ret);
	}
	qx(rm -f "$kernel_symvers");
	return $ret;
}

#return success if the driver was installed by the distro.
sub distrodriver {
	my $module = shift;chomp $module;

	return 0 unless (-f $module);

	my $vendor = qx(rpm -q --file --qf '%{VENDOR}' $module); chomp $vendor;
	if ($vendor =~ m(Red Hat|RedHat|SuSE|SUSE)) {
		#verify that the file is as installed by the distro and 
		#has not been modified
		my $mod = (grep {s(.* (.*/$module)$)($1)} qx(rpm -V --file $module))[0];
		return (not defined $mod);
	}
	return 0;
}

# compare a version string with another version string
# and return 1 if the first version string is recent.
sub newer_version {
	my $ihv_ver = shift; #IHV module path
	my $dist_ver = shift; #distro module path

	$ihv_ver =~ m(${r_driver_version_format});
	my $ihv_epoch = (defined $1) ? $2 : 0;
	$ihv_ver = $3;

	$dist_ver =~ m(${r_driver_version_format});
	my $dist_epoch = (defined $1) ? $2 : 0;
	$dist_ver = $3;

	my @i = split '[\._]', $ihv_ver;
	my @n = split '[\._]', $dist_ver;
	unshift(@i, $ihv_epoch); unshift(@n, $dist_epoch);

	foreach (0..(@n-1)) {
		return 0 if (not defined $i[$_]);

		(my $i_num = $i[$_]) =~ s/(\d+)(\w*)/$1/;
		(my $i_str = $i[$_]) =~ s/(\d+)(\w*)/$2/;
		(my $d_num = $n[$_]) =~ s/(\d+)(\w*)/$1/;
		(my $d_str = $n[$_]) =~ s/(\d+)(\w*)/$2/;

		return 0 if (($i_num < $d_num) or 
			(($i_num == $d_num) and ($i_str lt $d_str)));
		return 1 if (($i_num > $d_num) or 
				(($i_num == $d_num) and ($i_str gt $d_str)));
	}
	return 0;
}

sub check_driver_version_format {
	my $ver = shift;
	return ($ver =~ m(${r_driver_version_format}));
}

sub module_file_already_exists {
	my $mod = shift;
	my $kernel = shift;
	my $flavor = shift;
	my $ret = shift;
	my $r;

	(${$ret} = $mod ) =~ s($tmp/tmp)();
	my $moddir = get_moduledir_name($kernel, $flavor);
	if ($redhat) {
		$flavor =~ s/^default$//;
		$r = "${libdir}/${r_libkernelver}${flavor}/";
	} elsif ($suse) {
		$r = "${libdir}/${r_libkernelver}-${flavor}/";
	}
	${$ret} =~ s(${r})(${libdir}/$moddir/);
	return (-f ${$ret});
}

sub installed_driver_conflict {
	my $modules = shift;
	my $kernel = shift;
	my $flavor = shift;
	my $override = shift;
	my $force_only_if_overriden = shift;
	my $exists = shift;
	my $conflict = 0;
	my $mod;
	foreach my $module (@{$modules}) {
		#set the 'exists' flag if the module overwrites a 
		#installed module.
		if (module_file_already_exists($module, $kernel, $flavor, 
					\$mod)) {
			${$exists}=1;
			warn __LINE__ , "WARNING: module $mod overwrites a ",
				"already installed module\n";
		}
		$mod = qx(basename $module); chomp $mod;
		my $active_module = get_active_module($mod, $kernel, $flavor);
		if (not defined $override and defined $active_module and 
				distrodriver($active_module)) {
			(my $m = $module) =~ s($tmp/tmp)();
			$m = qx(basename $m); chomp $m;
			print "Distro driver present for module $m.",
		 		" Use --override option ",
				"to override and install this driver\n";
			return 1;
		}

		next if ($force_only_if_overriden);

		# there is no good mechanism to test if distro driver is newer,
		# in rhel3.
		next if ($redhat == 3);

		my $ibmver = (split /\s+/, qx(modinfo -F version $module))[0];
		unless (check_driver_version_format($ibmver)) {
			$ibmver="";
			print "The driver version $ibmver is not in ",
				"${r_driver_version_format} format\n",
				"Use the force option '-f' ",
				"if you want to install this driver.\n";
			return 1;
		}
		chomp $ibmver;

		next unless (defined $active_module);
		next if (distrodriver($active_module));

		my $activever=(split /\s+/,
			qx(modinfo -F version $active_module 2>/dev/null))[0];
		next unless (defined $activever);
		chomp $activever;

		unless (check_driver_version_format($activever)) {
			print "WARNING:Distribution supplied module",
				" $active_module has nonstandard version ",
				"format $activever.\n";
			next;
		}
		$conflict++ unless (newer_version($ibmver, $activever));
	}

	return ($conflict == @{$modules});
}

sub scriptlet {
	my $rpm = shift;
	my @script=qx(rpm -q --qf '%%pre\n%{PREIN}\n%%post\n%{POSTIN}\n%%preun\n%{PREUN}\n%%postun\n%{POSTUN}\n%%verifyscript\n%{VERIFYSCRIPT}\n' -p "$rpm");
	return @script;
}


#add the new modules in the list of modules to be used to generate in the
#initrd.
sub add_initrd {
	return 0 unless ($suse == 10 || $suse == 11 || $redhat == 5);

	my $rpm = shift;
	my $initrdfile;
	$initrdfile="/etc/sysconfig/kernel" if ($suse == 10 || $suse == 11);
	$initrdfile="/etc/modprobe.conf" if ($redhat == 5);
	#get all the modules in the rpm.
	my @modules = (grep {s(.*/(\w([-_\w]+\w)*)\.ko$)($1)} qx(rpm -q --list -p $rpm));
	my $driver_name = qx(rpm -qp $rpm --queryformat "%{NAME}" | awk -F- '{print \$3}');
	chomp @modules;
	chomp $driver_name;

	if ($suse == 10 || $suse == 11) {
		#get the initrd modules.
		my @existingmod = grep {s{^\s*INITRD_MODULES\s*=\s*"(.*)\"}{$1}} qx(cat $initrdfile);
		chomp @existingmod;
		@existingmod = split /\s+/, "@existingmod";

		#generate a merged list. TODO, there must be a better way to do this.
		my %unique;
		foreach my $mod (@modules,@existingmod) {
			$unique{$mod} = 1;
		}
		my $mod = join " ", sort keys %unique;

		qx(perl -p -i -e 's{^\\s*(INITRD_MODULES\\s*=\\s*)"\.+"}{\$1"$mod"};' $initrdfile);
		return $?;
	} else {
		#For each module, check if there is already an entry in /etc/depmod.conf
		#if there isn't then add an entry
		my @line_contents;
		my $not_commented;
		foreach my $mod (@modules) {
			$not_commented = 0;
			if (@line_contents = qx(grep "${driver_name}\\s${mod}" $initrdfile)) {
				foreach my $line (@line_contents) {
					#Found a line...but is it commented out?
					if ( $line =~ m|^\s*#.*${mod}| ) {
						print "modprobe.conf entry found, but it's commented out, so we'll add our own\n" if ($debug);
					}else{
						print "found uncommented entry for the module\n" if ($debug);
						$not_commented = 1;
					}
				}
			}else{
				print "no entries for $mod found in modprobe.conf\n" if ($debug);
			}

			#if there's already an entry for this module, just skip to the next one
			if ($not_commented == 1) {
				next;
			}
			
			#Add an entry for this module.
			#Since there's no way to know whether we're dealing with a network or
			#boot device, we'll just make every entry appear to be a boot device
			if ( system("echo \"alias scsi_hostadapter_$driver_name $mod\" >> /etc/modprobe.conf") < 0) {
				die "couldn't add entry into /etc/modprobe.conf";
			}

		} #end foreach @modules
	}#end else $redhat == 5
}

#
# build a duplicate rpm using the given rpm for the given kernel
# and return the name of the duplicate rpm and the status of the
# operation
#
sub buildduprpm {
	my $rpm = shift;
	my $unpacked_rpmdir = shift;
	my $kernel = shift;
	my $rpmkernel = shift;
	my $rpmflavor = shift;
	my $duprpm = shift;
	my $status = shift;

	$rpm =~ s/\ /\\ /g;
	my ($rpmarch, $name, $version, $release, $copyright, $group, $source, 
	 $url, $distribution, $vendor, $packager, $summary) = split /\n/, qx(rpm -q --queryformat '%{ARCH}\n%{NAME}\n%{VERSION}\n%{RELEASE}\n%{COPYRIGHT}\n%{GROUP}\n%{SOURCE}\n%{URL}\n%{DISTRIBUTION}\n%{VENDOR}\n%{PACKAGER}\n%{SUMMARY}' -p "$rpm");

	unless (defined $name) {
		${$status} = "RPM_CORRUPTION";
		print "Warning! rpm corruption -p $rpm ? : $!\n";
		return 1;
	}

	$name =~ s/(\w+-\w+)-.*/$1/;

	my $description=`rpm -q --queryformat '%{DESCRIPTION}' -p "$rpm"`;
	my $provides=`rpm -q --provides -p "$rpm"`;
	my $mkinitrd_kernel=get_mkinitrd_kernel($kernel, $rpmflavor);
	qx(mkdir -p $tmp);

	(my $tmpkernel = $kernel) =~ s/-/_/;
	$tmpkernel = "${tmpkernel}.${el}" if ($redhat);

	if ($redhat) {
		(my $flavor = $rpmflavor) =~ s/default//;
		$rpmkernel = "${rpmkernel}.${el}${flavor}";
	}

	if ($suse == 9 || $suse ==10) {
		$rpmkernel = "${rpmkernel}-${rpmflavor}";
	}  elsif ($suse == 11) {
		$rpmkernel = qx(echo ${rpmkernel} | sed -e 's/\\.[0-9]*\$//');
		chomp $rpmkernel;
		$rpmkernel = "${rpmkernel}-${rpmflavor}";
	}
	#generate a specfile
	unless (open(SPEC_HANDLE, ">$tmp/spec")) {
		qx(rm -f $tmp);
		${$status} = "FILESYSTEM_PROBLEM";
		print "Sorry, cannot open the specfile $!\n";
		return 1;
	}
	select SPEC_HANDLE;

	#rpm forces us not to use hypen in the version string.
	print "Summary: $summary\n";
	print "Name: ${name}_for_${tmpkernel}${r_kmp}-${rpmflavor}\n";
	print "Version: ${version}\n";
	print "Release: ${release}\n";
	print "Copyright: $copyright\n";
	print "Group: $group\n";
	print "Source: ibm-${name}_ibmdup-$kernel\n";
	print "URL: $url\n";
	print "Distribution: $distribution\n";
	print "Vendor: $vendor\n";
	print "Packager: $packager\n";
	print "Provides: ${name}-${tmpkernel}-${rpmflavor}\n";
	print "Requires: kernel-${rpmflavor} = ${kernel}\n" if ($suse == 9);
	print "BuildRoot: ${unpacked_rpmdir}\n";
	print "%description\n $description\n";

	print "%prep\n";
	print "touch \$RPM_SOURCE_DIR/ibm-${name}_ibmdup-$kernel\n";
	print "mv \$RPM_BUILD_ROOT/${libdir}/${rpmkernel}/ ${tmp}/tmp1\n";

	print "%build\n";

	print "%install\n";
	print "mkdir -p \$RPM_BUILD_ROOT/${libdir}/\n";
	print "mv $tmp/tmp1 \$RPM_BUILD_ROOT/${libdir}/${mkinitrd_kernel}\n";
	print "%files\n";
	print "%defattr(-, root, root, 0755)\n";
	print "${libdir}/$mkinitrd_kernel/${updatedirectory}\n";
	print "\n";

	print "%clean\n";
	print "rm -rf \$RPM_BUILD_ROOT\n";

	#migrate the scriptlets too.
	foreach (scriptlet($rpm)) {
		$_ =~ s/^\(none\)//;
		#change all the references to older kernels 
		#to this kernel.
		$_ =~ s/(KERNELRELEASES=).*/${1}"${mkinitrd_kernel}"/;
		print "$_";
	}

	close SPEC_HANDLE;
	select STDOUT;
	
	qx(rpmbuild --target $arch -ba $tmp/spec 2> /dev/null);
	if ($? != 0) {
		${$status} = "INTERNAL_INCONSISTENCY3";
		print "Warning! rpm internal inconsistency building ",
			"$rpm ? : $!\n";
		return 1;
	}
	${$duprpm}="${rpmdir}/${name}_for_${tmpkernel}${r_kmp}-${rpmflavor}-${version}-${release}.${arch}.rpm";
	return 0;
}

sub unpackrpm {
	my $rpm = shift;
	my $rpmdir = shift;
	my $modules = shift;
	# rpm2cpio, does not like the escape character, and rpm does not work without the escape.
	qx(mkdir -p $rpmdir; cd $rpmdir; rpm2cpio "$rpm" | cpio -ivd --no-absolute-filenames --quiet \* 2>&1);

	return 1 if ($?);
	@{$modules}=qx(find $rpmdir -name \*.${moduleext} -print);
	chomp @{$modules};
	return 0;
}

#return true if the modules are installed in the ${updatesdirectory}.
sub check_modules_paths {
	my $modules = shift;
	
	foreach (@{$modules}) {
		if ($redhat == 5) {
			return 1 if ($_ !~ m(${updatedirectory}/(${r_ihv}-)?${r_drivername}/\w+\.${moduleext}$));
		} else {
			return 1 if ($_ !~ m(${updatedirectory}/\w+\.${moduleext}$));
		}
	}
	return 0;
}

sub get_active_module {
	my $modname = shift; #name of the distro module
	my $kernel = shift;
	my $flavor = shift;
	my $depmod = get_depmod_file($kernel, $flavor);
        my $dmodule_path = (qx(grep "${libdir}/.*/$modname:" $depmod 2>/dev/null) )[0];
	$dmodule_path =~ s((^${libdir}/.*):.*)($1) if (defined $dmodule_path);

	chomp $dmodule_path if (defined $dmodule_path);

	return $dmodule_path;
}

sub query {
	my $module = shift;
	my $kernel = shift;
	my $flavor = shift;
	my $driverver = shift;
	my $ret = shift;
	my ($up2date, $installedver);

	#get the active driver if one is present.
	my $mod = qx(basename $module); chomp $mod;
	(my $modname = $mod )=~ s/\.${moduleext}//;
	my $kmodule = get_active_module($mod, $kernel, $flavor);

	#check if we have anything currently installed 
	#and whether the active driver is installed by the distro
	my $no_installed_version = (not defined $kmodule) ? 1 : 0;
	my $out_of_tree = (defined $kmodule && 
				not distrodriver($kmodule)) ? 1 : 0; 

	print "drivername\tkernel\t\t", 'installed',"\t\t", 
				'provided',"\n",'-'x70,"\n";
	print "$modname\t$kernel-$flavor", "\t\t";

	if ($redhat == 3) {
		$up2date = 0;
		$installedver = (defined $kmodule) ?  'unknown' :  'null';
	} else {
		my $myver = (split /\s+/, qx(modinfo -F version $module))[0];
		my $kver;
		if (defined $kmodule) {
			$kver = (split /\s+/, qx(modinfo -F version $kmodule))[0];
			$kver = 0.0 if (not defined $kver);
			$up2date = not newer_version($myver, $kver);
		} else {
			$up2date = 0;
		}
		$installedver = (defined $kmodule) ?  "$kver" :  'null';
	} 
	print "$installedver\t\t\t$driverver\n",'-'x70,"\n";

	print "Active module for $mod";
	if ($no_installed_version) {
		print ": nothing-installed";
	} elsif ($out_of_tree) {
		print ": out-of-tree";
	} else {
		print ": in-tree";
	}

	if ($up2date) {
		print " up-to-date\n";
	} else {
		print " out-of-date\n";
	}

	#check if the version of the active driver is later than ours.
	${$ret}=0 if ($out_of_tree and $up2date);
	${$ret}=1 if ($out_of_tree and not $up2date);
	${$ret}=2 if (not $out_of_tree and $up2date);
	${$ret}=3 if (not $out_of_tree and not $up2date);
	return 0;
}

#For rhel5 we have to add an entry in /etc/depmod.d/depmod.conf
#to override the Redhat supplied driver whenever the target kernel
#This is b/c our driver (in the non-compiled against case) lives
#in /lib/modules/<kernel>/weak-updates/, and by default, any
#driver provided by Redhat masks any driver in 'weak-updates'.
sub add_override_to_depmod_rhel5 {
	my $rpm = shift;
	$rpm =~ s/\ /\\ /g;
	my $depmod_conf = "/etc/depmod.d/depmod.conf";
	my @module_list = grep {m($libdir/.*.${moduleext}$)} qx(rpm -q --list -p "$rpm"); 
	chomp @module_list;

	#Add an 'override' entry for each module
	foreach my $ko_entry (@module_list) {
               my $module_name = qx(basename $ko_entry);
		#Get the directory name under 'extra' that this installs to
		my $rpm_dir_name = qx(dirname $ko_entry | sed -e 's|.*/||' );

               $module_name =~ s|\.${moduleext}||;
               chomp($module_name);

               #Check if this module already has an entry in depmod.conf
               if ( -e $depmod_conf && qx(grep $module_name $depmod_conf) ){
                       print "Found an existing entry for $module_name in $depmod_conf\n";
                       next;
               }
               qx(echo "override $module_name * weak-updates/$rpm_dir_name" >> $depmod_conf);
       }
       return 0;
}
