/****************************************************************************
* @file    adsTel.h
*
* Contents: This file contains the ADS Telephony API definition.
*
* @par
* INTEL CONFIDENTIAL
* Copyright 2009 Intel Corporation All Rights Reserved.
* 
* The source code contained or described herein and all documents related to the
* source code ("Material") are owned by Intel Corporation or its suppliers or
* licensors. Title to the Material remains with Intel Corporation or its
* suppliers and licensors. The Material may contain trade secrets and proprietary
* and confidential information of Intel Corporation and its suppliers and
* licensors, and is protected by worldwide copyright and trade secret laws and
* treaty provisions. No part of the Material may be used, copied, reproduced,
* modified, published, uploaded, posted, transmitted, distributed, or disclosed
* in any way without Intels prior express written permission.
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery
* of the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be
* express and approved by Intel in writing.
* 
* Include any supplier copyright notices as supplier requires Intel to use.
* Include supplier trademarks or logos as supplier requires Intel to use,
* preceded by an asterisk.
* An asterisked footnote can be added as follows: 
*   *Third Party trademarks are the property of their respective owners.
* 
* Unless otherwise agreed by Intel in writing, you may not remove or alter this
* notice or any other notice embedded in Materials by Intel or Intels suppliers
* or licensors in any way.
* 
*  version: ADS.L.1.1.0-160
****************************************************************************/

/*
 ****************************************************************************
 * Doxygen group definitions
 ***************************************************************************/

/**
 ****************************************************************************
 * @defgroup adsTel ADS Telephony Framework API
 *        Public API for the ADS Telephony Framework.
 *
 * @purpose
 ***************************************************************************/

/**
 ****************************************************************************
 * @defgroup adsTelInitShutdown ADS APIs for Init/Shutdown
 * @ingroup adsTel
 *        Initialization and shutdown functions for the ADS Telephony
 *        Framework.
 *
 * @purpose
 *        The functions in this group are responsible for all initialization
 *        and shutdown for the ADS Telephony Framework.
 ***************************************************************************/

 /**
 ****************************************************************************
 * @defgroup adsTelEndpoint ADS APIs for Endpoint Allocation and Configuration
 * @ingroup adsTel
 *        Pipeline instance allocation and deallocation functions for the ADS
 *        Telephony Framework.
 *
 * @purpose
 *        The functions in this group are responsible for all aspects of
 *        pipeline allocation for the ADS Telephony Framework.
 ***************************************************************************/

 /**
 ****************************************************************************
 * @defgroup adsTelRouter ADS APIs for Router Allocation
 * @ingroup adsTel
 *        Router instance allocation and deallocation functions for the ADS
 *        Telephony Framework.
 *
 * @purpose
 *        The functions in this group are responsible for all aspects of
 *        router allocation for the ADS Telephony Framework.
 ***************************************************************************/

 /**
 ****************************************************************************
 * @defgroup adsTelCallback ADS API and Event Callbacks
 * @ingroup adsTel
 *        Functions and type definitions of the ADS Telephony Framework for
 *        API and events callbacks.
 *
 * @purpose
 *        The functions and type definitions in this group are responsible for
 *        all aspects of API status and events reporting for the ADS Telephony
 *        Framework.
 ***************************************************************************/
#ifndef ADS_TEL_H_
#define ADS_TEL_H_

#include "adsTelTypes.h"


/****************************************************************************/
/******************* INTERFACE DEFINITION ***********************************/
/****************************************************************************/

/**
 * @ingroup adsTelInitShutdown
 *
 * @description
 *        This function initializes the ADS Telephony Framework and
          allocates any memory and threads that are needed.
 * @context
 *        Calling function thread.
 * @assumptions
 *        None.
 * @sideEffects
 *        None.
 * @param
 *        adsTelInitParams [IN] - Initialization parameters.
 *
 * @blocking
 *        Yes.
 * @reentrant
 *        No.
 * @threadSafe
 *        Yes.
 * @retval
 *        None.
 * @pre
 *        None.
 * @post
 *        None.
 * @note
 *        None.
 * @see
 *        None.
 */
AdsTelStatus_t
adsTelInit(AdsTelInit_t adsTelInitParams);


/**
 * @ingroup adsTelInitShutdown
 *
 * @description
 *        This function uninitializes the ADS Telephony Framework and
 *        any associated memory and threads.
 * @context
 *        Calling function thread.
 * @assumptions
 *        None.
 * @sideEffects
 *        None.
 * @blocking
 *        Yes.
 * @reentrant
 *        No.
 * @threadSafe
 *        Yes.
 * @retval
 *        None.
 * @pre
 *        None.
 * @post
 *        None.
 * @note
 *        None.
 * @see
 *        None.
 */
AdsTelStatus_t
adsTelUnInit(void);


/**
 * @ingroup adsTelInitShutdown
 *
 * @description
 *        This function triggers the framework data processing threads.
 * @context
 *        Calling function thread.
 * @assumptions
 *        None.
 * @sideEffects
 *        None.
 * @blocking
 *        Yes.
 * @reentrant
 *        Yes.
 * @threadSafe
 *        Yes.
 * @retval
 *        None.
 * @pre
 *        None.
 * @post
 *        None.
 * @note
 *        None.
 * @see
 *        None.
 */
void adsTelFwTrigger(void);


/**
 * @ingroup adsTelEndpoint
 *        Add a tone to the Tone Generator.
 * @description
 *        This function adds a new tone which can be played by the tone
 *        generator components.
 *
 * @context
 *        Calling function thread.
 * @assumptions
 *        None.
 * @sideEffects
 *        None.
 * @blocking
 *        Yes.
 * @reentrant
 *        No.
 * @threadSafe
 *        Yes.
 *
 * @param
 *        toneId [IN] - The tone ID to use. The allowable range is 0-255;
 * @param
 *        numSegs [IN] - The number of segments in the tone.
 * @param
 *        pToneSegs [IN] - A pointer to an array of the tone segments.
 *
 * @retval
 *        ADS_TEL_STATUS_SUCCESS The function executed successfully.
 * @retval
 *        ADS_TEL_STATUS_FAIL The function did not execute successfully.
 * @pre
 *        The framework must have been previously initialized.
 * @post
 *        None.
 * @note
 *        If a call progress tone ID is used then the call progress tone
 *        will be overwritten with the current tone information.
 * @see
 *        None.
 */
AdsTelStatus_t adsTelTgToneBuild(unsigned int     toneId,
                                 unsigned int     numSegs,
                                 AdsTelToneSeg_t* pToneSegs);


/**
 * @ingroup adsTelEndpoint
 *        Build a Tone Detector Tone.
 * @description
 *        This function adds a new tone which can be detected by the tone
 *        detector components. This API can only be called before the
 *        framework has been initialized.
 *
 * @context
 *        Calling function thread.
 * @assumptions
 *        None.
 * @sideEffects
 *        None.
 * @blocking
 *        Yes.
 * @reentrant
 *        No.
 * @threadSafe
 *        Yes.
 *
 * @param
 *        toneId [IN] - The tone ID to use. The allowable range is 16-255;
 * @param
 *        pToneInfo [IN] - A pointer to the tone detection criterion
 *                         information.
 *
 * @retval
 *        ADS_TEL_STATUS_SUCCESS The function executed successfully.
 * @retval
 *        ADS_TEL_STATUS_FAIL The function did not execute successfully.
 * @pre
 *        The framework must be uninitialized.
 * @post
 *        None.
 * @note
 *        None.
 * @see
 *        None.
 */
AdsTelStatus_t adsTelTdToneBuild(unsigned int      toneId,
                                 AdsTelToneInfo_t* pToneInfo);


/**
 * @ingroup adsTelEndpoint
 *        Set the amplitutude detection range for the Tone Detector.
 * @description
 *        The Tone Detector is able to detect the pre-defined DTMF tones in
 *        the full amplitude level range of +3 ~ -45 dBm. The application
 *        can use this function to set a specific amplitude range. Only the
 *        signals within this amplitude range can be detected as the DTMF
 *        tones. This API can only be called before the framework has
 *        been initialized.
 *
 * @context
 *        Calling function thread.
 * @assumptions
 *        None.
 * @sideEffects
 *        None.
 * @blocking
 *        Yes.
 * @reentrant
 *        No.
 * @threadSafe
 *        Yes.
 *
 * @param
 *        category [IN] - The tone ID category. Only DTMF is currently
 *                        supported.
 * @param
 *        ampMinF0 [IN] - Minimum amplitude of the low frequency,
 *                        +3 ~ -45 in dBm.
 * @param
 *        ampMaxF0 [IN] - Maximum amplitude of the low frequency,
 *                        +3 ~ -45 in dBm.
 * @param
 *        ampMinF1 [IN] - Minimum amplitude of the high frequency,
 *                        +3 ~ -45 in dBm.
 * @param
 *        ampMaxF1 [IN] - Maximum amplitude of the high frequency,
 *                        +3 ~ -45 in dBm.
 *
 * @retval
 *        ADS_TEL_STATUS_SUCCESS The function executed successfully.
 * @retval
 *        ADS_TEL_STATUS_FAIL The function did not execute successfully.
 * @pre
 *        The framework must be uninitialized.
 * @post
 *        None.
 * @note
 *        None.
 * @see
 *        None.
 */
AdsTelStatus_t adsTelTdAmplitudeRangeSet(AdsTelTdCategory_t category,
                                         int                ampMinF0,
                                         int                ampMaxF0,
                                         int                ampMinF1,
                                         int                ampMaxF1);


/**
 * @ingroup adsTelEndpoint
 *        Set the tone paramters for the Tone Tenerator.
 * @description
 *        The DTMF tone generation has the default parameters of 100 ms
 *        tone-on and tone-off duration and -3dBm level. This function
 *        allows the users to change the default parameters. This API can
 *        only be called after the framework has been initialized.
 *
 * @context
 *        Calling function thread.
 * @assumptions
 *        None.
 * @sideEffects
 *        None.
 * @blocking
 *        Yes.
 * @reentrant
 *        No.
 * @threadSafe
 *        Yes.
 *
 * @param
 *        toneOn [IN] -  tone on duration in ms. Range 1 ~ FFFFFFF.
 * @param
 *        toneOff [IN] - tone off duration in ms. Range 1 ~ FFFFFFF.
 * @param
 *        ampdBm [IN] - total tone level in dBm, must be in 0 ~ -45 range.
 *
 * @retval
 *        ADS_TEL_STATUS_SUCCESS The function executed successfully.
 * @retval
 *        ADS_TEL_STATUS_FAIL The function did not execute successfully.
 * @pre
 *        The framework must be initialized.
 * @post
 *        None.
 * @note
 *        None.
 * @see
 *        None.
 */
AdsTelStatus_t adsTelTgDtmfParamSet(int toneOn, int toneOff, int ampdBm);


/**
 * @ingroup adsTelEndpoint
 *        Create an endpoint.
 * @description
 *        This function is responsible for creating an endpoint and
 *        connecting it to a router. The endpoint can be either PCM or IP.
 *        The components in both the inbound and outbound pipelines will be
 *        initialised with default values. The reply from the framework will
 *        be posted via @ref adsTelFwCb_t. The possible @ref adsTelFwCb_t
 *        return statuses are:\n
 *            @ref ADS_TEL_STATUS_SUCCESS \n
 *            @ref ADS_TEL_STATUS_FAIL.\n
 *        See @ref adsTelFwCb_t for the return buffer format for each return
 *        status.
 *
 *
 * @context
 *        Calling function thread.
 * @assumptions
 *        None.
 * @sideEffects
 *        None.
 * @blocking
 *        No.
 * @reentrant
 *        Yes.
 * @threadSafe
 *        Yes.
 *
 * @param
 *        pTransId [OUT] - Tracking number for the API return call.
 * @param
 *        inPipe [IN] - Inbound pipeline type.
 * @param
 *        outPipe [IN] - Outbound pipeline type.
 * @param
 *        pcmDataType [IN] - The PCM data input format. This parameter is only
 *                           valid for PCM endpoints and can be set to NULL for
 *                           IP endpoints.
 * @param
 *        inCb [IN] - The function that the framework will call when
 *                    requesting input data.
 * @param
 *        outCb [IN] - The function that the framework will call when
 *                     providing output data.
 * @param
 *        priority [IN] - Priority of setup. A priority of 911 means that the
 *                        if the max number of endpoints is already setup
 *                        then one will be destroyed to create this one.
 *                        This functionality is not currently supported in
 *                        the framework.
 * @param
 *        rtrHandle [IN] - The handle of the router that the pipeline
 *                         connect to.
 * @param
 *        pEpHandle [OUT] - The handle for the pipeline.
 *
 * @retval
 *        ADS_TEL_STATUS_SUCCESS The function executed successfully and the
 *                               operation was sent to the framework for
 *                               processing.
 * @retval
 *        ADS_TEL_STATUS_FAIL The function did not execute successfully.
 * @retval
 *        ADS_TEL_STATUS_INVALID_PARAM An invalid input parameter was supplied.
 * @pre
 *        The associated router must have already been created.
 * @post
 *        None.
 * @note
 *        Both trhe inbound and outbound callbacks will be disconnected by
 *        default on endpoint creation. See @ref adsTelEpConStatusSet().
 * @see
 *        @ref adsTelRouterCreate().
 */
AdsTelStatus_t
adsTelEpCreate(unsigned int*      pTransId,
               AdsTelPipeType_t   inPipe,
               AdsTelPipeType_t   outPipe,
               AdsTelPcmData_t    pcmDataType,
               adsTelEpDataCb_t   inCb,
               adsTelEpDataCb_t   outCb,
               AdsTelCallPrior_t  priority,
               AdsTelRtrHandle_t  rtrHandle,
               AdsTelEpHandle_t*  pEpHandle);


/**
 * @ingroup adsTelEndpoint
 *        Set the connection status of an endpoint.
 * @description
 *        This function enables or disables the outbound and inbound
 *        callbacks for an endpoint. The reply from the framework will be
 *        posted via @ref adsTelFwCb_t. The possible @ref adsTelFwCb_t return
 *        statuses are:\n
 *            @ref ADS_TEL_STATUS_SUCCESS \n
 *            @ref ADS_TEL_STATUS_FAIL\n
 *        See @ref adsTelFwCb_t for the return buffer format for each return
 *        status.
 * @context
 *        Calling function thread.
 * @assumptions
 *        None.
 * @sideEffects
 *        None.
 * @param
 *        pTransId [OUT] -   Tracking number for the API return call.
 * @param
 *        epHandle [IN] - The endpoint to set the connection status for.
 * @param
 *        conStatus [IN] -   The status to set the connection to.
 *
 * @blocking
 *        No.
 * @reentrant
 *        Yes.
 * @threadSafe
 *        Yes.
 * @retval
 *        ADS_TEL_STATUS_SUCCESS on success or else the appropriate error code.
 * @pre
 *        The adsTel framework must be initialized.
 * @post
 *        None.
 * @note
 *        None.
 * @see
 *        @ref adsTelInit().
 */

AdsTelStatus_t
adsTelEpConStatusSet(unsigned int*       pTransId,
                     AdsTelEpHandle_t    epHandle,
                     AdsTelEpConStatus_t conStatus);


/**
 * @ingroup adsTelEndpoint
 *        Set multiple parameters for an endpoint component.
 * @description
 *        This function sets a number of component parameters up to
 *        @ref ADS_TEL_MAX_PARAMS. When an endpoint is created all component
 *        parameters are initialised with default values. This function can be
 *        used to change the configuration of any component. The reply from the
 *        framework will be posted via @ref adsTelFwCb_t. The possible
 *        @ref adsTelFwCb_t return statuses are:\n
 *            @ref ADS_TEL_STATUS_SUCCESS \n
 *            @ref ADS_TEL_STATUS_FAIL\n
 *        See @ref adsTelFwCb_t for the return buffer format for each return
 *        status.
 *
 * @context
 *        Calling function thread.
 * @assumptions
 *        None.
 * @sideEffects
 *        None.
 * @blocking
 *        No.
 * @reentrant
 *        Yes.
 * @threadSafe
 *        Yes.
 *
 * @param
 *        pTransId [IN] - Tracking number for the API return call.
 * @param
 *        epHandle [IN] - Endpoint handle.
 * @param
 *        pipeDir [IN] - Pipeline direction (in/out).
 * @param
 *        compId [IN] - The component location within the the pipeline. The
 *                      index is 0 based.
 * @param
 *        numParams [IN] - The number of parameters being set.
 * @param
 *        pParamIDs [IN] - Pointer to an array of parameter IDs of max
 *                         size @ref ADS_TEL_MAX_PARAMS.
 * @param
 *        pParamVals [IN] - Pointer to an array of parameter values of max
 *                          size @ref ADS_TEL_MAX_PARAMS.
 * @retval
 *        ADS_TEL_STATUS_SUCCESS The function executed successfully and the
 *                               operation was sent to the framework for
 *                               processing.
 * @retval
 *        ADS_TEL_STATUS_FAIL The function did not execute successfully.
 * @retval
 *        ADS_TEL_STATUS_INVALID_PARAM An invalid input parameter was supplied.
 * @pre
 *        The endpoint must have already been created.
 * @post
 *        None.
 * @note
 *        None.
 * @see
 *        @ref adsTelEpCreate().
 */

 
 
AdsTelStatus_t
adsTelEpCompParamsSet(unsigned int*      pTransId,
                      AdsTelEpHandle_t   epHandle,
                      AdsTelPipeDir_t    pipeDir,
                      unsigned int       compId,
                      unsigned int       numParams,
                      unsigned int*      pParamIDs,
                      int*      pParamVals);


/**
 * @ingroup adsTelEndpoint
 *        Get all parameters values for an endpoint component.
 * @description
 *        This function return all the parameter values for a component
 *        wia the framework callback. The reply from the framework will
 *        be posted via @ref adsTelFwCb_t. The possible @ref adsTelFwCb_t
 *        return statuses are:\n
 *            @ref ADS_TEL_STATUS_GET_PARAMS\n
 *        See @ref adsTelFwCb_t for the return buffer format for each return
 *        status.
 *
 * @context
 *        Calling function thread.
 * @assumptions
 *        None.
 * @sideEffects
 *        None.
 * @blocking
 *        No.
 * @reentrant
 *        Yes.
 * @threadSafe
 *        Yes.
 *
 * @param
 *        pTransId [IN] - Tracking number for the API return call.
 * @param
 *        epHandle [IN] - Endpoint handle.
 * @param
 *        pipeDir [IN] - Pipeline direction (in/out).
 * @param
 *        compId [IN] - The component location within the the pipeline. The
 *                      index is 0 based.
 *
 * @retval
 *        ADS_TEL_STATUS_SUCCESS The function executed successfully and the
 *                               operation was sent to the framework for
 *                               processing.
 * @retval
 *        ADS_TEL_STATUS_FAIL The function did not execute successfully.
 * @retval
 *        ADS_TEL_STATUS_INVALID_PARAM An invalid input parameter was supplied.
 * @pre
 *        The endpoint must have already been created.
 * @post
 *        None.
 * @note
 *        None.
 * @see
 *        @ref adsTelEpCreate() \n
 *        @ref adsTelEpCompParamsSet().
 */
AdsTelStatus_t
adsTelEpCompParamsGet(unsigned int*     pTransId,
                      AdsTelEpHandle_t  epHandle,
                      AdsTelPipeDir_t   pipeDir,
                      unsigned int      compId);


/**
 * @ingroup adsTelEndpoint
 *        Play a tone or multiple tones on a tone generator component.
 * @description
 *        This function will play out all the supplied tones on a tone
 *        generator. Note that if a non DTMF tone is played it will play
 *        continously until the tone generator is explicitly stopped with
 *        @ref adsTelEpCompStop(). The reply from the framework will be
 *        posted via @ref adsTelFwCb_t. The possible @ref adsTelFwCb_t
 *        return statuses are:\n
 *            @ref ADS_TEL_STATUS_TG_CMPLT \n
 *            @ref ADS_TEL_STATUS_FAIL\n
 *        See @ref adsTelFwCb_t for the return buffer format for each return
 *        status.
 *
 * @context
 *        Calling function thread.
 * @assumptions
 *        None.
 * @sideEffects
 *        None.
 * @blocking
 *        No.
 * @reentrant
 *        Yes.
 * @threadSafe
 *        Yes.
 *
 * @param
 *        pTransId [OUT] - Tracking number for the API return call.
 * @param
 *        epHandle [IN] - Endpoint handle.
 * @param
 *        pipeDir [IN] - Pipeline direction (in/out).
 * @param
 *        compId [IN] - The component location within the the  pipeline. The
 *                      index is 0 based.
 * @param
 *        numTones [IN] - The number of tones to be played up to a maximum of
 *                        @ref ADS_TEL_MAX_TONES.
 * @param
 *        pToneIds [IN] - A pointer to an array (of size numTones) of the
 *                        tone IDs to be played. See  for a list of the
 *                        valid tone IDs.
 *
 * @retval
 *        ADS_TEL_STATUS_SUCCESS The function executed successfully and the
 *                               operation was sent to the framework for
 *                               processing.
 * @retval
 *        ADS_TEL_STATUS_FAIL The function did not execute successfully.
 * @retval
 *        ADS_TEL_STATUS_INVALID_PARAM An invalid input parameter was supplied.
 * @pre
 *        The endpoint must have already been created.
 * @post
 *        None.
 * @note
 *        None.
 * @see
 *        @ref adsTelEpCreate().
 */
AdsTelStatus_t
adsTelEpTonePlay(unsigned int*      pTransId,
                 AdsTelEpHandle_t   epHandle,
                 AdsTelPipeDir_t    pipeDir,
                 unsigned int       compId,
                 unsigned int       numTones,
                 unsigned int*      pToneIds);


/**
 * @ingroup adsTelEndpoint
 *        Play FSK Data on a tone generator component.
 * @description
 *        This function will play out all the supplied FSK data bytes on a
 *        tone generator. The reply from the framework will be posted via
 *        @ref adsTelFwCb_t. The possible @ref adsTelFwCb_t return
 *        statuses are:\n
 *            @ref ADS_TEL_STATUS_TG_CMPLT \n
 *            @ref ADS_TEL_STATUS_FAIL\n
 *        See @ref adsTelFwCb_t for the return buffer format for each return
 *        status.
 *
 * @context
 *        Calling function thread.
 * @assumptions
 *        None.
 * @sideEffects
 *        None.
 * @blocking
 *        No.
 * @reentrant
 *        Yes.
 * @threadSafe
 *        Yes.
 *
 * @param
 *        pTransId [OUT] - Tracking number for the API return call.
 * @param
 *        epHandle [IN] - Endpoint handle.
 * @param
 *        pipeDir [IN] - Pipeline direction (in/out).
 * @param
 *        compId [IN] - The component location within the the  pipeline. The
 *                      index is 0 based.
 * @param
 *        numBytes [IN] - The number of FSK Bytes to be played up to a maximum
 *                        of @ref ADS_TEL_MAX_FSK_BYTES.
 * @param
 *        pFskData [IN] - A pointer to an array (of size numBytes) of the
 *                        FSK Bytes to be sent.
 *
 * @retval
 *        ADS_TEL_STATUS_SUCCESS The function executed successfully and the operation
 *                               was sent to the framework for processing.
 * @retval
 *        ADS_TEL_STATUS_FAIL The function did not execute successfully.
 * @retval
 *        ADS_TEL_STATUS_INVALID_PARAM An invalid input parameter was supplied.
 * @pre
 *        The endpoint must have already been created.
 * @post
 *        None.
 * @note
 *        None.
 * @see
 *        @ref adsTelEpCreate().
 */
AdsTelStatus_t
adsTelEpFskPlay(unsigned int*     pTransId,
                AdsTelEpHandle_t  epHandle,
                AdsTelPipeDir_t   pipeDir,
                unsigned int      compId,
                unsigned int      numBytes,
                char*             pFskData);


/**
 * @ingroup adsTelEndpoint
 *        Receive a tone or multiple tones on a tone detector component.
 * @description
 *        This function will attempt to detect the number of expected tones
 *        on a tone detector unless one of the timeout conditions is met.
 *        The reply from the framework will be posted via @ref adsTelFwCb_t
 *        The possible @ref adsTelFwCb_t return statuses are:\n
 *            @ref ADS_TEL_STATUS_TD_RCV_CMPLT \n
 *            @ref ADS_TEL_STATUS_FAIL.
 *        See @ref adsTelFwCb_t for the return buffer format for each return
 *        status.
 *
 *
 * @context
 *        Calling function thread.
 * @assumptions
 *        None.
 * @sideEffects
 *        None.
 * @blocking
 *        No.
 * @reentrant
 *        Yes.
 * @threadSafe
 *        Yes.
 *
 * @param
 *        pTransId [OUT] - Tracking number for the API return call.
 * @param
 *        epHandle [IN] - Endpoint handle.
 * @param
 *        pipeDir [IN] - Pipeline direction (in/out).
 * @param
 *        compId [IN] - The component location within the the  pipeline. The
 *                      index is 0 based.
 * @param
 *        totalTimeout [IN] - Total time out (in 10ms units).
 * @param
 *        firstDigitTimeout [IN] - First digit time out (in 10ms units).
 * @param
 *        interDigitTimeout [IN] - Inter digit time out (in 10ms units).
 * @param
 *        termDigit [IN] - OR'd terminate digit bits.
 * @param
 *        numDigits [IN] - number of digits to receive.
 *
 * @retval
 *        ADS_TEL_STATUS_SUCCESS The function executed successfully and the
 *                               operation was sent to the framework for
 *                               processing.
 * @retval
 *        ADS_TEL_STATUS_FAIL The function did not execute successfully.
 * @retval
 *        ADS_TEL_STATUS_INVALID_PARAM An invalid input parameter was supplied.
 * @pre
 *        The endpoint must have already been created.
 * @post
 *        None.
 * @note
 *        None.
 * @see
 *        @ref adsTelFwCb_t \n
 *        @ref adsTelEpCreate().
 */
AdsTelStatus_t
adsTelEpToneRecv(unsigned int*     pTransId,
                 AdsTelEpHandle_t  epHandle,
                 AdsTelPipeDir_t   pipeDir,
                 unsigned int      compId,
                 unsigned int      totalTimeout,
                 unsigned int      firstDigitTimeout,
                 unsigned int      interDigitTimeout,
                 unsigned int      termDigit,
                 unsigned int      numDigits);


/**
 * @ingroup adsTelEndpoint
 *        Receive FSK bytes on a tone detector component.
 * @description
 *        This function will attempt to detect FSK data on a tone detector
 *        component up to @ref ADS_TEL_MAX_FSK_BYTES unless the operation
 *        times out first. The reply from the framework will be posted
 *        via @ref adsTelFwCb_t. The possible @ref adsTelFwCb_t
 *        return statuses are:\n
 *            @ref ADS_TEL_STATUS_TD_RCV_FSK_CMPLT \n
 *            @ref ADS_TEL_STATUS_FAIL.
 *        See @ref adsTelFwCb_t for the return buffer format for each return
 *        status.
 *
 *
 * @context
 *        Calling function thread.
 * @assumptions
 *        None.
 * @sideEffects
 *        None.
 * @blocking
 *        No.
 * @reentrant
 *        Yes.
 * @threadSafe
 *        Yes.
 *
 * @param
 *        pTransId [OUT] - Tracking number for the API return call.
 * @param
 *        epHandle [IN] - Endpoint handle.
 * @param
 *        pipeDir [IN] - Pipeline direction (in/out).
 * @param
 *        compId [IN] - The component location within the the  pipeline. The
 *                      index is 0 based.
 * @param
 *        timeout [IN] - Total time out (in 10ms units).
 *
 * @retval
 *        ADS_TEL_STATUS_SUCCESS The function executed successfully and the
 *                               operation was sent to the framework for
 *                               processing.
 * @retval
 *        ADS_TEL_STATUS_FAIL The function did not execute successfully.
 * @retval
 *        ADS_TEL_STATUS_INVALID_PARAM An invalid input parameter was supplied.
 * @pre
 *        The endpoint must have already been created.
 * @post
 *        None.
 * @note
 *        None.
 * @see
 *        @ref adsTelFwCb_t \n
 *        @ref adsTelEpCreate().
 */
AdsTelStatus_t
adsTelEpFskRecv(unsigned int*     pTransId,
                AdsTelEpHandle_t  epHandle,
                AdsTelPipeDir_t   pipeDir,
                unsigned int      compId,
                unsigned int      timeout);


/**
 * @ingroup adsTelEndpoint
 *        Play an audio file.
 * @description
 *        This function will play an audio file on a player component.
 *        The audio file to be played must have been registered with
 *        @ref adsTelRegisterWavFile(). To stop the audio playing use
 *        @ref adsTelEpCompStop(). The reply from the framework will be posted
 *        via @ref adsTelFwCb_t. The possible return statuses are:\n
 *            @ref ADS_TEL_STATUS_SUCCESS \n
 *            @ref ADS_TEL_STATUS_INVALID_PARAM \n
 *            @ref ADS_TEL_STATUS_FAIL
 *
 * @context
 *        Calling function thread.
 * @assumptions
 *        None.
 * @sideEffects
 *        None.
 * @blocking
 *        No.
 * @reentrant
 *        Yes.
 * @threadSafe
 *        Yes.
 *
 * @param
 *        pTransId [OUT] - Tracking number for the API return call.
 * @param
 *        epHandle [IN] - Endpoint handle.
 * @param
 *        pipeDir [IN] - Pipeline direction (in/out).
 * @param
 *        compId [IN] - The component location within the the  pipeline. The
 *                      index is 0 based.
 * @param
 *        pSegments [IN] - Pointer to an array of audio segments to play of
 *                         max size @ref ADS_TEL_PLY_MAX_SEGS.
 *
 * @param
 *        nSegs [IN] - The number of segments to play.
 * @retval
 *        ADS_TEL_STATUS_SUCCESS The function executed successfully and the
 *                               operation was sent to the framework for
 *                               processing.
 * @retval
 *        ADS_TEL_STATUS_FAIL The function did not execute successfully.
 * @retval
 *        ADS_TEL_STATUS_INVALID_PARAM An invalid input parameter was supplied.
 * @pre
 *        The endpoint must have already been created.
 * @post
 *        None.
 * @note
 *        None.
 * @see
 *        @ref adsTelEpCreate() \n
 *        @ref adsTelRegisterWavFile() \n
 *        @ref adsTelEpCompStop().
 */
AdsTelStatus_t
adsTelEpPlayAudio(unsigned int*      pTransId,
                  AdsTelEpHandle_t   epHandle,
                  AdsTelPipeDir_t    pipeDir,
                  unsigned int       compId,
                  AdsTelMediaDesc_t* pSegments,
                  short              nSegs);


/**
 * @ingroup adsTelEndpoint
 *        Register a wav file.
 * @description
 *        This function will register a wav file into the ADS Telephony
 *        Framework. A maximum of 32 files can be registered.
 *        The possible return statuses are:\n
 *            @ref ADS_TEL_STATUS_SUCCESS \n
 *            @ref ADS_TEL_STATUS_INVALID_FORMAT \n
 *            @ref ADS_TEL_STATUS_FAIL
 *
 *
 * @context
 *        Calling function thread.
 * @assumptions
 *        None.
 * @sideEffects
 *        None.
 * @blocking
 *        Yes.
 * @reentrant
 *        Yes.
 * @threadSafe
 *        Yes.
 *
 * @param
 *        filename [IN] - The name of the wav file to register.
 *
 * @param
 *        pMediaHandle [OUT] - The media handle returned by the framework. This
 *                             handle is valid if the function returns
 *                             ADS_TEL_STATUS_SUCCESS.
 *
 * @retval
 *        ADS_TEL_STATUS_SUCCESS The function executed successfully and the
 *                               operation was sent to the framework for
 *                               processing.
 * @retval
 *        ADS_TEL_STATUS_FAIL The function did not execute successfully.
 * @pre
 *        The adsTel framework must be initialized.
 * @post
 *        None.
 * @note
 *        None.
 * @see
 *        @ref adsTelEpPlayAudio() \n
 *        @ref adsTelEpCompStop().
 */
AdsTelStatus_t
adsTelRegisterWavFile(char                 *filename,
                      AdsTelMediaHandle_t  *pMediaHandle);



/**
 * @ingroup adsTelEndpoint
 *        Start a component.
 * @description
 *        This function will start a component.
 *
 *        The reply from the framework will be posted via @ref adsTelFwCb_t
 *        The possible @ref adsTelFwCb_t return statuses are:\n
 *            @ref ADS_TEL_STATUS_SUCCESS \n
 *            @ref ADS_TEL_STATUS_FAIL\n
 *        See @ref adsTelFwCb_t for the return buffer format for each return
 *        status.
 *
 *
 * @context
 *        Calling function thread.
 * @assumptions
 *        None.
 * @sideEffects
 *        None.
 * @blocking
 *        No.
 * @reentrant
 *        Yes.
 * @threadSafe
 *        Yes.
 *
 * @param
 *        pTransId [OUT] - Tracking number for the API return call.
 * @param
 *        epHandle [IN] - Endpoint handle.
 * @param
 *        pipeDir [IN] - Pipeline direction (in/out).
 * @param
 *        compId [IN] - The component location within the the  pipeline. The
 *                      index is 0 based.
 *
 * @retval
 *        ADS_TEL_STATUS_SUCCESS The function executed successfully and the
 *                               operation was sent to the framework for
 *                               processing.
 * @retval
 *        ADS_TEL_STATUS_FAIL The function did not execute successfully.
 * @retval
 *        ADS_TEL_STATUS_INVALID_PARAM An invalid input parameter was supplied.
 * @pre
 *        The endpoint must have already been created.
 * @post
 *        None.
 * @note
 *        This API is not currently supported by any component.
 * @see
 *        @ref adsTelFwCb_t \n
 *        @ref adsTelEpCreate() \n
 *        @ref adsTelEpCompStop().
 */
AdsTelStatus_t
adsTelEpCompStart(unsigned int*     pTransId,
                  AdsTelEpHandle_t  epHandle,
                  AdsTelPipeDir_t   pipeDir,
                  unsigned int      compId);


/**
 * @ingroup adsTelEndpoint
 *        Stop a component.
 * @description
 *        This function will stop a component. It is only applicable to
 *        the following components:\n
 *           Tone Detector \n
 *           Tone generator \n
 *           Player \n
 *        The reply from the framework will be posted via @ref adsTelFwCb_t
 *        The possible @ref adsTelFwCb_t return statuses are:\n
 *            @ref ADS_TEL_STATUS_SUCCESS \n
 *            @ref ADS_TEL_STATUS_FAIL.\n
 *        See @ref adsTelFwCb_t for the return buffer format for each return
 *        status.
 *
 *
 * @context
 *        Calling function thread.
 * @assumptions
 *        None.
 * @sideEffects
 *        None.
 * @blocking
 *        No.
 * @reentrant
 *        Yes.
 * @threadSafe
 *        Yes.
 *
 * @param
 *        pTransId [OUT] - Tracking number for the API return call.
 * @param
 *        epHandle [IN] - Endpoint handle.
 * @param
 *        pipeDir [IN] - Pipeline direction (in/out).
 * @param
 *        compId [IN] - The component location within the the pipeline.
 *
 * @retval
 *        ADS_TEL_STATUS_SUCCESS The function executed successfully and the
 *                               operation was sent to the framework for
 *                               processing.
 * @retval
 *        ADS_TEL_STATUS_FAIL The function did not execute successfully.
 * @retval
 *        ADS_TEL_STATUS_INVALID_PARAM An invalid input parameter was supplied.
 * @pre
 *        The endpoint must have already been created.
 * @post
 *        None.
 * @note
 *        None.
 * @see
 *        @ref adsTelFwCb_t \n
 *        @ref adsTelEpCreate() \n
 *        @ref adsTelEpCompStart().
 */
AdsTelStatus_t
adsTelEpCompStop(unsigned int*    pTransId,
                 AdsTelEpHandle_t epHandle,
                 AdsTelPipeDir_t  pipeDir,
                 unsigned int     compId);


/**
 * @ingroup adsTelEndpoint
 *        Reset a component.
 * @description
 *        This function will reset a component.
 *        The reply from the framework will be posted via @ref adsTelFwCb_t
 *        The possible @ref adsTelFwCb_t return statuses are:\n
 *            @ref ADS_TEL_STATUS_SUCCESS \n
 *            @ref ADS_TEL_STATUS_FAIL\n
 *        See @ref adsTelFwCb_t for the return buffer format for each return
 *        status.
 *
 *
 * @context
 *        Calling function thread.
 * @assumptions
 *        None.
 * @sideEffects
 *        None.
 * @blocking
 *        No.
 * @reentrant
 *        Yes.
 * @threadSafe
 *        Yes.
 *
 * @param
 *        pTransId [OUT] - Tracking number for the API return call.
 * @param
 *        epHandle [IN] - Endpoint handle.
 * @param
 *        pipeDir [IN] - Pipeline direction (in/out).
 * @param
 *        compId [IN] - The component location within the the  pipeline. The
 *                      index is 0 based.
 *
 * @retval
 *        ADS_TEL_STATUS_SUCCESS The function executed successfully and the
 *                               operation was sent to the framework for
 *                               processing.
 * @retval
 *        ADS_TEL_STATUS_FAIL The function did not execute successfully.
 * @retval
 *        ADS_TEL_STATUS_INVALID_PARAM An invalid input parameter was supplied.
 * @pre
 *        The endpoint must have already been created.
 * @post
 *        None.
 * @note
 *        This API is not currently supported by any components.
 * @see
 *        @ref adsTelFwCb_t \n
 *        @ref adsTelEpCreate() \n
 */
AdsTelStatus_t
adsTelEpCompReset(unsigned int*    pTransId,
                  AdsTelEpHandle_t epHandle,
                  AdsTelPipeDir_t  pipeDir,
                  unsigned int     compId);


/**
 * @ingroup adsTelEndpoint
 *        Destroy an endpoint.
 * @description
 *        This function will remove an endpoint from its associated router
 *        and then destroy the endpoint.
 *        The reply from the framework will be posted via @ref adsTelFwCb_t
 *        The possible @ref adsTelFwCb_t return statuses are:\n
 *            @ref ADS_TEL_STATUS_SUCCESS \n
 *            @ref ADS_TEL_STATUS_FAIL\n
 *        See @ref adsTelFwCb_t for the return buffer format for each return
 *        status.
 *
 * @context
 *        Calling function thread.
 * @assumptions
 *        None.
 * @sideEffects
 *        None.
 * @blocking
 *        No.
 * @reentrant
 *        Yes.
 * @threadSafe
 *        Yes.
 *
 * @param
 *        pTransId [OUT] - Tracking number for the API return call.
 * @param
 *        epHandle [IN] - The endpoint to destroy.
 *
 * @retval
 *        ADS_TEL_STATUS_SUCCESS The function executed successfully and the
 *                               operation was sent to the framework for
 *                               processing.
 * @retval
 *        ADS_TEL_STATUS_FAIL The function did not execute successfully.
 * @retval
 *        ADS_TEL_STATUS_INVALID_PARAM An invalid input parameter was supplied.
 * @pre
 *        The endpoint must have already been created.
 * @post
 *        None.
 * @note
 *        None.
 * @see
 *        @ref adsTelFwCb_t \n
 *        @ref adsTelEpCreate() \n
 */
AdsTelStatus_t
adsTelEpDestroy(unsigned int*      pTransId,
                AdsTelEpHandle_t   epHandle);


/**
 * @ingroup adsTelRouter
 *        Create a router instance.
 * @description
 *        Create a router instance. By setting the priority to 911 the router
 *        will be set up even if the max number of routers has already been
 *        set up by deleting an existing router instance. The router type
 *        should be set to @ref ADS_TEL_RTR_NORMAL for calls of up to three
 *        endpoints. For conference calls of greater than three endpoints
 *        @ref ADS_TEL_RTR_CONF type should be used.
 *        The reply from the framework will be posted via @ref adsTelFwCb_t
 *        The possible @ref adsTelFwCb_t return statuses are:\n
 *            @ref ADS_TEL_STATUS_SUCCESS \n
 *            @ref ADS_TEL_STATUS_FAIL\n
 *        See @ref adsTelFwCb_t for the return buffer format for each return
 *        status.
 *
 * @context
 *        Calling function thread.
 * @assumptions
 *        None.
 * @sideEffects
 *        None.
 * @blocking
 *        No.
 * @reentrant
 *        Yes.
 * @threadSafe
 *        Yes.
 *
 * @param
 *        pTransId [OUT] - Tracks the API return call in non blocking mode.
 *                         This can be set to NULL in blocking mode.
 * @param
 *        priority [IN] - The priority of the call.
 *
 * @param
 *        rtrType [IN] - The router type.
 *
 * @param
 *        pRtrHandle [OUT] - The returned router handle.
 *
 * @retval
 *        ADS_TEL_STATUS_SUCCESS The function executed successfully and the
 *                               operation was sent to the framework for
 *                               processing.
 * @retval
 *        ADS_TEL_STATUS_FAIL The function did not execute successfully.
 * @retval
 *        ADS_TEL_STATUS_INVALID_PARAM An invalid input parameter was supplied.
 * @pre
 *        The adsTel framework must be initialized.
 * @post
 *        None.
 * @note
 *        Call priority is not currently supported and all calls should be
 *        created with @ref ADS_TEL_CALL_PRIOR_NORMAL.
 *        Router type is not currently supported and all routers should be
 *        set up with @ref ADS_TEL_RTR_NORMAL router type setting.
 * @see
 *        @ref adsTelFwCb_t \n
 *        @ref adsTelInit() \n
 */
AdsTelStatus_t
adsTelRouterCreate(unsigned int*      pTransId,
                   AdsTelCallPrior_t  priority,
                   AdsTelRtrType_t    rtrType,
                   AdsTelRtrHandle_t* pRtrHandle);


/**
 * @ingroup adsTelRouter
 *        Destroy a router instance.
 * @description
 *        Destroy a router instance. A router instance cannot be destroyed if
 *        there are still endpoints connected to the router.
 *        The reply from the framework will be posted via @ref adsTelFwCb_t
 *        The possible @ref adsTelFwCb_t return statuses are:
 *            @ref ADS_TEL_STATUS_SUCCESS \n
 *            @ref ADS_TEL_STATUS_FAIL\n
 *        See @ref adsTelFwCb_t for the return buffer format for each return
 *        status.
 *
 * @context
 *        Calling function thread.
 * @assumptions
 *        None.
 * @sideEffects
 *        None.
 * @blocking
 *        No.
 * @reentrant
 *        Yes.
 * @threadSafe
 *        Yes.
 *
 * @param
 *        pTransId [OUT] - Tracking number for the API return call.
 * @param
 *        rtrHandle [IN] - The router to destroy.
 *
 * @retval
 *        ADS_TEL_STATUS_SUCCESS The function executed successfully and the
 *                               operation was sent to the framework for
 *                               processing.
 * @retval
 *        ADS_TEL_STATUS_FAIL The function did not execute successfully.
 * @retval
 *        ADS_TEL_STATUS_INVALID_PARAM An invalid input parameter was supplied.
 * @pre
 *        The router must have already been created.
 * @post
 *        None.
 * @note
 *        None.
 * @see
 *        @ref adsTelFwCb_t \n
 *        @ref adsTelRouterCreate() \n
 */
AdsTelStatus_t
adsTelRouterDestroy(unsigned int*     pTransId,
                    AdsTelRtrHandle_t rtrHandle);


#endif /*ADS_TEL_H_*/

