/* Reference code of ARIRANG 224/256/284/512 family (64 bit). */

#define BEECRYPT_DLL_EXPORT

#include "arirang.h"

// Left and right rotation
#define ROTL_DWORD(x, n) ( (uint32_t)((x) << (n)) | (uint32_t)((x) >> (32-(n))) )
#define ROTL_QWORD(x, n) ( (uint64_t)((x) << (n)) | (uint64_t)((x) >> (64-(n))) )

// Reverse the byte order of uint32_t and uint16_t.
#define ENDIAN_REVERSE_DWORD(dwS)	( (ROTL_DWORD((dwS),  8) & 0x00ff00ff) | (ROTL_DWORD((dwS), 24) & 0xff00ff00) )
#define ENDIAN_REVERSE_QWORD(w, x) \
 {   uint64_t tmp = (w); \
     tmp = (tmp >> 32) | (tmp << 32); \
     tmp = ((tmp & 0xff00ff00ff00ff00ULL) >> 8) | ((tmp & 0x00ff00ff00ff00ffULL) << 8); \
     (x) = ((tmp & 0xffff0000ffff0000ULL) >> 16) | ((tmp & 0x0000ffff0000ffffULL) << 16); \
 }

// Move uint32_t type to uint8_t type and uint8_t type to uint32_t type
#if WORDS_BIGENDIAN

#define BIG_B2D(B, D)		D = *(uint32_t *)(B)
#define BIG_D2B(D, B)		*(uint32_t *)(B) = (uint32_t)(D)
#define LITTLE_B2D(B, D)	D = ENDIAN_REVERSE_DWORD(*(uint32_t *)(B))
#define LITTLE_D2B(D, B)	*(uint32_t *)(B) = ENDIAN_REVERSE_DWORD(D)

// Compression function
#define GetData(x) x

#else

#define BIG_B2D(B, D)		D = ENDIAN_REVERSE_DWORD(*(uint32_t *)(B))
#define BIG_Q2B(D, B)		ENDIAN_REVERSE_QWORD(D, *(uint64_t *)(B))
#define BIG_D2B(D, B)		*(uint32_t *)(B) = ENDIAN_REVERSE_DWORD(D)
#define LITTLE_B2D(B, D)	D = *(uint32_t *)(B)
#define LITTLE_D2B(D, B)	*(uint32_t *)(B) = (uint32_t)(D)

// Compression function
#define GetData(x) ENDIAN_REVERSE_DWORD(x)

#endif

const hashFunction arirang256 = {
    .name = "ARIRANG-256",
    .paramsize = sizeof(arirangParam),
    .blocksize = 64,
    .digestsize = 256/8,	/* XXX default to ARIRANG-256 */
    .reset = (hashFunctionReset) arirangReset,
    .update = (hashFunctionUpdate) arirangUpdate,
    .digest = (hashFunctionDigest) arirangDigest
};

static const uint32_t K256[16] = {
    0x517cc1b7, 0x76517cc1, 0xbd76517c, 0x2dbd7651,
    0x272dbd76, 0xcb272dbd, 0x90cb272d, 0x0a90cb27,
    0xec0a90cb, 0x5bec0a90, 0x9a5bec0a, 0xe69a5bec,
    0xb7e69a5b, 0xc1b7e69a, 0x7cc1b7e6, 0x517cc1b7
};

static const uint64_t K512[16] = {
    0x517cc1b727220a94ULL, 0x2db6517cc1b72722ULL,
    0xe6952db6517cc1b7ULL, 0x90cbe6952db6517cULL,
    0x7cca90cbe6952db6ULL, 0xcb237cca90cbe695ULL,
    0x765ecb237cca90cbULL, 0xec01765ecb237ccaULL,
    0xb7e9ec01765ecb23ULL, 0xbd7db7e9ec01765eULL,
    0x9a5fbd7db7e9ec01ULL, 0x5be89a5fbd7db7e9ULL,
    0x0a945be89a5fbd7dULL, 0x27220a945be89a5fULL,
    0xc1b727220a945be8ULL, 0x517cc1b727220a94ULL
};

static const uint8_t sbx[] = {	/* Prepare S-box */
0x63,0x7c,0x77,0x7b,0xf2,0x6b,0x6f,0xc5,0x30,0x01,0x67,0x2b,0xfe,0xd7,0xab,0x76,
0xca,0x82,0xc9,0x7d,0xfa,0x59,0x47,0xf0,0xad,0xd4,0xa2,0xaf,0x9c,0xa4,0x72,0xc0,
0xb7,0xfd,0x93,0x26,0x36,0x3f,0xf7,0xcc,0x34,0xa5,0xe5,0xf1,0x71,0xd8,0x31,0x15,
0x04,0xc7,0x23,0xc3,0x18,0x96,0x05,0x9a,0x07,0x12,0x80,0xe2,0xeb,0x27,0xb2,0x75,
0x09,0x83,0x2c,0x1a,0x1b,0x6e,0x5a,0xa0,0x52,0x3b,0xd6,0xb3,0x29,0xe3,0x2f,0x84,
0x53,0xd1,0x00,0xed,0x20,0xfc,0xb1,0x5b,0x6a,0xcb,0xbe,0x39,0x4a,0x4c,0x58,0xcf,
0xd0,0xef,0xaa,0xfb,0x43,0x4d,0x33,0x85,0x45,0xf9,0x02,0x7f,0x50,0x3c,0x9f,0xa8,
0x51,0xa3,0x40,0x8f,0x92,0x9d,0x38,0xf5,0xbc,0xb6,0xda,0x21,0x10,0xff,0xf3,0xd2,
0xcd,0x0c,0x13,0xec,0x5f,0x97,0x44,0x17,0xc4,0xa7,0x7e,0x3d,0x64,0x5d,0x19,0x73,
0x60,0x81,0x4f,0xdc,0x22,0x2a,0x90,0x88,0x46,0xee,0xb8,0x14,0xde,0x5e,0x0b,0xdb,
0xe0,0x32,0x3a,0x0a,0x49,0x06,0x24,0x5c,0xc2,0xd3,0xac,0x62,0x91,0x95,0xe4,0x79,
0xe7,0xc8,0x37,0x6d,0x8d,0xd5,0x4e,0xa9,0x6c,0x56,0xf4,0xea,0x65,0x7a,0xae,0x08,
0xba,0x78,0x25,0x2e,0x1c,0xa6,0xb4,0xc6,0xe8,0xdd,0x74,0x1f,0x4b,0xbd,0x8b,0x8a,
0x70,0x3e,0xb5,0x66,0x48,0x03,0xf6,0x0e,0x61,0x35,0x57,0xb9,0x86,0xc1,0x1d,0x9e,
0xe1,0xf8,0x98,0x11,0x69,0xd9,0x8e,0x94,0x9b,0x1e,0x87,0xe9,0xce,0x55,0x28,0xdf,
0x8c,0xa1,0x89,0x0d,0xbf,0xe6,0x42,0x68,0x41,0x99,0x2d,0x0f,0xb0,0x54,0xbb,0x16,
};

static const uint8_t F2[] = {	/* i*2 in GF(256) */
0x00,0x02,0x04,0x06,0x08,0x0a,0x0c,0x0e,0x10,0x12,0x14,0x16,0x18,0x1a,0x1c,0x1e,
0x20,0x22,0x24,0x26,0x28,0x2a,0x2c,0x2e,0x30,0x32,0x34,0x36,0x38,0x3a,0x3c,0x3e,
0x40,0x42,0x44,0x46,0x48,0x4a,0x4c,0x4e,0x50,0x52,0x54,0x56,0x58,0x5a,0x5c,0x5e,
0x60,0x62,0x64,0x66,0x68,0x6a,0x6c,0x6e,0x70,0x72,0x74,0x76,0x78,0x7a,0x7c,0x7e,
0x80,0x82,0x84,0x86,0x88,0x8a,0x8c,0x8e,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,
0xa0,0xa2,0xa4,0xa6,0xa8,0xaa,0xac,0xae,0xb0,0xb2,0xb4,0xb6,0xb8,0xba,0xbc,0xbe,
0xc0,0xc2,0xc4,0xc6,0xc8,0xca,0xcc,0xce,0xd0,0xd2,0xd4,0xd6,0xd8,0xda,0xdc,0xde,
0xe0,0xe2,0xe4,0xe6,0xe8,0xea,0xec,0xee,0xf0,0xf2,0xf4,0xf6,0xf8,0xfa,0xfc,0xfe,
0x1b,0x19,0x1f,0x1d,0x13,0x11,0x17,0x15,0x0b,0x09,0x0f,0x0d,0x03,0x01,0x07,0x05,
0x3b,0x39,0x3f,0x3d,0x33,0x31,0x37,0x35,0x2b,0x29,0x2f,0x2d,0x23,0x21,0x27,0x25,
0x5b,0x59,0x5f,0x5d,0x53,0x51,0x57,0x55,0x4b,0x49,0x4f,0x4d,0x43,0x41,0x47,0x45,
0x7b,0x79,0x7f,0x7d,0x73,0x71,0x77,0x75,0x6b,0x69,0x6f,0x6d,0x63,0x61,0x67,0x65,
0x9b,0x99,0x9f,0x9d,0x93,0x91,0x97,0x95,0x8b,0x89,0x8f,0x8d,0x83,0x81,0x87,0x85,
0xbb,0xb9,0xbf,0xbd,0xb3,0xb1,0xb7,0xb5,0xab,0xa9,0xaf,0xad,0xa3,0xa1,0xa7,0xa5,
0xdb,0xd9,0xdf,0xdd,0xd3,0xd1,0xd7,0xd5,0xcb,0xc9,0xcf,0xcd,0xc3,0xc1,0xc7,0xc5,
0xfb,0xf9,0xff,0xfd,0xf3,0xf1,0xf7,0xf5,0xeb,0xe9,0xef,0xed,0xe3,0xe1,0xe7,0xe5,
};

static const uint8_t F3[] = {	/* i*3 in GF(256) */
0x00,0x03,0x06,0x05,0x0c,0x0f,0x0a,0x09,0x18,0x1b,0x1e,0x1d,0x14,0x17,0x12,0x11,
0x30,0x33,0x36,0x35,0x3c,0x3f,0x3a,0x39,0x28,0x2b,0x2e,0x2d,0x24,0x27,0x22,0x21,
0x60,0x63,0x66,0x65,0x6c,0x6f,0x6a,0x69,0x78,0x7b,0x7e,0x7d,0x74,0x77,0x72,0x71,
0x50,0x53,0x56,0x55,0x5c,0x5f,0x5a,0x59,0x48,0x4b,0x4e,0x4d,0x44,0x47,0x42,0x41,
0xc0,0xc3,0xc6,0xc5,0xcc,0xcf,0xca,0xc9,0xd8,0xdb,0xde,0xdd,0xd4,0xd7,0xd2,0xd1,
0xf0,0xf3,0xf6,0xf5,0xfc,0xff,0xfa,0xf9,0xe8,0xeb,0xee,0xed,0xe4,0xe7,0xe2,0xe1,
0xa0,0xa3,0xa6,0xa5,0xac,0xaf,0xaa,0xa9,0xb8,0xbb,0xbe,0xbd,0xb4,0xb7,0xb2,0xb1,
0x90,0x93,0x96,0x95,0x9c,0x9f,0x9a,0x99,0x88,0x8b,0x8e,0x8d,0x84,0x87,0x82,0x81,
0x9b,0x98,0x9d,0x9e,0x97,0x94,0x91,0x92,0x83,0x80,0x85,0x86,0x8f,0x8c,0x89,0x8a,
0xab,0xa8,0xad,0xae,0xa7,0xa4,0xa1,0xa2,0xb3,0xb0,0xb5,0xb6,0xbf,0xbc,0xb9,0xba,
0xfb,0xf8,0xfd,0xfe,0xf7,0xf4,0xf1,0xf2,0xe3,0xe0,0xe5,0xe6,0xef,0xec,0xe9,0xea,
0xcb,0xc8,0xcd,0xce,0xc7,0xc4,0xc1,0xc2,0xd3,0xd0,0xd5,0xd6,0xdf,0xdc,0xd9,0xda,
0x5b,0x58,0x5d,0x5e,0x57,0x54,0x51,0x52,0x43,0x40,0x45,0x46,0x4f,0x4c,0x49,0x4a,
0x6b,0x68,0x6d,0x6e,0x67,0x64,0x61,0x62,0x73,0x70,0x75,0x76,0x7f,0x7c,0x79,0x7a,
0x3b,0x38,0x3d,0x3e,0x37,0x34,0x31,0x32,0x23,0x20,0x25,0x26,0x2f,0x2c,0x29,0x2a,
0x0b,0x08,0x0d,0x0e,0x07,0x04,0x01,0x02,0x13,0x10,0x15,0x16,0x1f,0x1c,0x19,0x1a,
};

static const uint8_t F4[] = {	/* i*4 in GF(256) */
0x00,0x04,0x08,0x0c,0x10,0x14,0x18,0x1c,0x20,0x24,0x28,0x2c,0x30,0x34,0x38,0x3c,
0x40,0x44,0x48,0x4c,0x50,0x54,0x58,0x5c,0x60,0x64,0x68,0x6c,0x70,0x74,0x78,0x7c,
0x80,0x84,0x88,0x8c,0x90,0x94,0x98,0x9c,0xa0,0xa4,0xa8,0xac,0xb0,0xb4,0xb8,0xbc,
0xc0,0xc4,0xc8,0xcc,0xd0,0xd4,0xd8,0xdc,0xe0,0xe4,0xe8,0xec,0xf0,0xf4,0xf8,0xfc,
0x1b,0x1f,0x13,0x17,0x0b,0x0f,0x03,0x07,0x3b,0x3f,0x33,0x37,0x2b,0x2f,0x23,0x27,
0x5b,0x5f,0x53,0x57,0x4b,0x4f,0x43,0x47,0x7b,0x7f,0x73,0x77,0x6b,0x6f,0x63,0x67,
0x9b,0x9f,0x93,0x97,0x8b,0x8f,0x83,0x87,0xbb,0xbf,0xb3,0xb7,0xab,0xaf,0xa3,0xa7,
0xdb,0xdf,0xd3,0xd7,0xcb,0xcf,0xc3,0xc7,0xfb,0xff,0xf3,0xf7,0xeb,0xef,0xe3,0xe7,
0x36,0x32,0x3e,0x3a,0x26,0x22,0x2e,0x2a,0x16,0x12,0x1e,0x1a,0x06,0x02,0x0e,0x0a,
0x76,0x72,0x7e,0x7a,0x66,0x62,0x6e,0x6a,0x56,0x52,0x5e,0x5a,0x46,0x42,0x4e,0x4a,
0xb6,0xb2,0xbe,0xba,0xa6,0xa2,0xae,0xaa,0x96,0x92,0x9e,0x9a,0x86,0x82,0x8e,0x8a,
0xf6,0xf2,0xfe,0xfa,0xe6,0xe2,0xee,0xea,0xd6,0xd2,0xde,0xda,0xc6,0xc2,0xce,0xca,
0x2d,0x29,0x25,0x21,0x3d,0x39,0x35,0x31,0x0d,0x09,0x05,0x01,0x1d,0x19,0x15,0x11,
0x6d,0x69,0x65,0x61,0x7d,0x79,0x75,0x71,0x4d,0x49,0x45,0x41,0x5d,0x59,0x55,0x51,
0xad,0xa9,0xa5,0xa1,0xbd,0xb9,0xb5,0xb1,0x8d,0x89,0x85,0x81,0x9d,0x99,0x95,0x91,
0xed,0xe9,0xe5,0xe1,0xfd,0xf9,0xf5,0xf1,0xcd,0xc9,0xc5,0xc1,0xdd,0xd9,0xd5,0xd1,
};

static const uint8_t F8[] = {	/* i*8 in GF(256) */
0x00,0x08,0x10,0x18,0x20,0x28,0x30,0x38,0x40,0x48,0x50,0x58,0x60,0x68,0x70,0x78,
0x80,0x88,0x90,0x98,0xa0,0xa8,0xb0,0xb8,0xc0,0xc8,0xd0,0xd8,0xe0,0xe8,0xf0,0xf8,
0x1b,0x13,0x0b,0x03,0x3b,0x33,0x2b,0x23,0x5b,0x53,0x4b,0x43,0x7b,0x73,0x6b,0x63,
0x9b,0x93,0x8b,0x83,0xbb,0xb3,0xab,0xa3,0xdb,0xd3,0xcb,0xc3,0xfb,0xf3,0xeb,0xe3,
0x36,0x3e,0x26,0x2e,0x16,0x1e,0x06,0x0e,0x76,0x7e,0x66,0x6e,0x56,0x5e,0x46,0x4e,
0xb6,0xbe,0xa6,0xae,0x96,0x9e,0x86,0x8e,0xf6,0xfe,0xe6,0xee,0xd6,0xde,0xc6,0xce,
0x2d,0x25,0x3d,0x35,0x0d,0x05,0x1d,0x15,0x6d,0x65,0x7d,0x75,0x4d,0x45,0x5d,0x55,
0xad,0xa5,0xbd,0xb5,0x8d,0x85,0x9d,0x95,0xed,0xe5,0xfd,0xf5,0xcd,0xc5,0xdd,0xd5,
0x6c,0x64,0x7c,0x74,0x4c,0x44,0x5c,0x54,0x2c,0x24,0x3c,0x34,0x0c,0x04,0x1c,0x14,
0xec,0xe4,0xfc,0xf4,0xcc,0xc4,0xdc,0xd4,0xac,0xa4,0xbc,0xb4,0x8c,0x84,0x9c,0x94,
0x77,0x7f,0x67,0x6f,0x57,0x5f,0x47,0x4f,0x37,0x3f,0x27,0x2f,0x17,0x1f,0x07,0x0f,
0xf7,0xff,0xe7,0xef,0xd7,0xdf,0xc7,0xcf,0xb7,0xbf,0xa7,0xaf,0x97,0x9f,0x87,0x8f,
0x5a,0x52,0x4a,0x42,0x7a,0x72,0x6a,0x62,0x1a,0x12,0x0a,0x02,0x3a,0x32,0x2a,0x22,
0xda,0xd2,0xca,0xc2,0xfa,0xf2,0xea,0xe2,0x9a,0x92,0x8a,0x82,0xba,0xb2,0xaa,0xa2,
0x41,0x49,0x51,0x59,0x61,0x69,0x71,0x79,0x01,0x09,0x11,0x19,0x21,0x29,0x31,0x39,
0xc1,0xc9,0xd1,0xd9,0xe1,0xe9,0xf1,0xf9,0x81,0x89,0x91,0x99,0xa1,0xa9,0xb1,0xb9,
};

static const uint8_t F9[] = {	/* i*9 in GF(256) */
0x00,0x09,0x12,0x1b,0x24,0x2d,0x36,0x3f,0x48,0x41,0x5a,0x53,0x6c,0x65,0x7e,0x77,
0x90,0x99,0x82,0x8b,0xb4,0xbd,0xa6,0xaf,0xd8,0xd1,0xca,0xc3,0xfc,0xf5,0xee,0xe7,
0x3b,0x32,0x29,0x20,0x1f,0x16,0x0d,0x04,0x73,0x7a,0x61,0x68,0x57,0x5e,0x45,0x4c,
0xab,0xa2,0xb9,0xb0,0x8f,0x86,0x9d,0x94,0xe3,0xea,0xf1,0xf8,0xc7,0xce,0xd5,0xdc,
0x76,0x7f,0x64,0x6d,0x52,0x5b,0x40,0x49,0x3e,0x37,0x2c,0x25,0x1a,0x13,0x08,0x01,
0xe6,0xef,0xf4,0xfd,0xc2,0xcb,0xd0,0xd9,0xae,0xa7,0xbc,0xb5,0x8a,0x83,0x98,0x91,
0x4d,0x44,0x5f,0x56,0x69,0x60,0x7b,0x72,0x05,0x0c,0x17,0x1e,0x21,0x28,0x33,0x3a,
0xdd,0xd4,0xcf,0xc6,0xf9,0xf0,0xeb,0xe2,0x95,0x9c,0x87,0x8e,0xb1,0xb8,0xa3,0xaa,
0xec,0xe5,0xfe,0xf7,0xc8,0xc1,0xda,0xd3,0xa4,0xad,0xb6,0xbf,0x80,0x89,0x92,0x9b,
0x7c,0x75,0x6e,0x67,0x58,0x51,0x4a,0x43,0x34,0x3d,0x26,0x2f,0x10,0x19,0x02,0x0b,
0xd7,0xde,0xc5,0xcc,0xf3,0xfa,0xe1,0xe8,0x9f,0x96,0x8d,0x84,0xbb,0xb2,0xa9,0xa0,
0x47,0x4e,0x55,0x5c,0x63,0x6a,0x71,0x78,0x0f,0x06,0x1d,0x14,0x2b,0x22,0x39,0x30,
0x9a,0x93,0x88,0x81,0xbe,0xb7,0xac,0xa5,0xd2,0xdb,0xc0,0xc9,0xf6,0xff,0xe4,0xed,
0x0a,0x03,0x18,0x11,0x2e,0x27,0x3c,0x35,0x42,0x4b,0x50,0x59,0x66,0x6f,0x74,0x7d,
0xa1,0xa8,0xb3,0xba,0x85,0x8c,0x97,0x9e,0xe9,0xe0,0xfb,0xf2,0xcd,0xc4,0xdf,0xd6,
0x31,0x38,0x23,0x2a,0x15,0x1c,0x07,0x0e,0x79,0x70,0x6b,0x62,0x5d,0x54,0x4f,0x46,
};

static const uint8_t FA[] = {	/* i*10 in GF(256) */
0x00,0x0a,0x14,0x1e,0x28,0x22,0x3c,0x36,0x50,0x5a,0x44,0x4e,0x78,0x72,0x6c,0x66,
0xa0,0xaa,0xb4,0xbe,0x88,0x82,0x9c,0x96,0xf0,0xfa,0xe4,0xee,0xd8,0xd2,0xcc,0xc6,
0x5b,0x51,0x4f,0x45,0x73,0x79,0x67,0x6d,0x0b,0x01,0x1f,0x15,0x23,0x29,0x37,0x3d,
0xfb,0xf1,0xef,0xe5,0xd3,0xd9,0xc7,0xcd,0xab,0xa1,0xbf,0xb5,0x83,0x89,0x97,0x9d,
0xb6,0xbc,0xa2,0xa8,0x9e,0x94,0x8a,0x80,0xe6,0xec,0xf2,0xf8,0xce,0xc4,0xda,0xd0,
0x16,0x1c,0x02,0x08,0x3e,0x34,0x2a,0x20,0x46,0x4c,0x52,0x58,0x6e,0x64,0x7a,0x70,
0xed,0xe7,0xf9,0xf3,0xc5,0xcf,0xd1,0xdb,0xbd,0xb7,0xa9,0xa3,0x95,0x9f,0x81,0x8b,
0x4d,0x47,0x59,0x53,0x65,0x6f,0x71,0x7b,0x1d,0x17,0x09,0x03,0x35,0x3f,0x21,0x2b,
0x77,0x7d,0x63,0x69,0x5f,0x55,0x4b,0x41,0x27,0x2d,0x33,0x39,0x0f,0x05,0x1b,0x11,
0xd7,0xdd,0xc3,0xc9,0xff,0xf5,0xeb,0xe1,0x87,0x8d,0x93,0x99,0xaf,0xa5,0xbb,0xb1,
0x2c,0x26,0x38,0x32,0x04,0x0e,0x10,0x1a,0x7c,0x76,0x68,0x62,0x54,0x5e,0x40,0x4a,
0x8c,0x86,0x98,0x92,0xa4,0xae,0xb0,0xba,0xdc,0xd6,0xc8,0xc2,0xf4,0xfe,0xe0,0xea,
0xc1,0xcb,0xd5,0xdf,0xe9,0xe3,0xfd,0xf7,0x91,0x9b,0x85,0x8f,0xb9,0xb3,0xad,0xa7,
0x61,0x6b,0x75,0x7f,0x49,0x43,0x5d,0x57,0x31,0x3b,0x25,0x2f,0x19,0x13,0x0d,0x07,
0x9a,0x90,0x8e,0x84,0xb2,0xb8,0xa6,0xac,0xca,0xc0,0xde,0xd4,0xe2,0xe8,0xf6,0xfc,
0x3a,0x30,0x2e,0x24,0x12,0x18,0x06,0x0c,0x6a,0x60,0x7e,0x74,0x42,0x48,0x56,0x5c,
};

////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION NAME  : step256
//
// DESCRIPTION    : Step function of ARIRANG-224 and ARIRANG-256
//
// PARAMETERS     : R  - working variables
//                  M1,M2 - message block
//
// RETURNS        : void
//
////////////////////////////////////////////////////////////////////////////////
static
void step256(uint32_t R[8], uint32_t M1, uint32_t M2)
{
    uint32_t temp1,temp2;

    // Message XOR
    R[0] ^= M1;
    R[4] ^= M2;

#define bx(x, n)	((uint8_t)((x) >> (8 * n)))

    // Sub-byte
    temp1 =    (uint32_t)(sbx[bx(R[0], 0)]) ^ ((uint32_t)(sbx[bx(R[0], 1)]) <<  8) ^ ((uint32_t)(sbx[bx(R[0], 2)]) << 16) ^ ((uint32_t)(sbx[bx(R[0], 3)]) << 24);
    temp2 =    (uint32_t)(sbx[bx(R[4], 0)]) ^ ((uint32_t)(sbx[bx(R[4], 1)]) <<  8) ^ ((uint32_t)(sbx[bx(R[4], 2)]) << 16) ^ ((uint32_t)(sbx[bx(R[4], 3)]) << 24);

    // MDS transformation
    temp1 =  ( (uint32_t)(F2[bx(temp1,0)]) ^ (uint32_t)(F3[bx(temp1,1)]) ^ (uint32_t)(   bx(temp1,2) ) ^ (uint32_t)(   bx(temp1,3) )       ) ^
	     (((uint32_t)(   bx(temp1,0) ) ^ (uint32_t)(F2[bx(temp1,1)]) ^ (uint32_t)(F3[bx(temp1,2)]) ^ (uint32_t)(   bx(temp1,3) )) <<  8) ^
	     (((uint32_t)(   bx(temp1,0) ) ^ (uint32_t)(   bx(temp1,1) ) ^ (uint32_t)(F2[bx(temp1,2)]) ^ (uint32_t)(F3[bx(temp1,3)])) << 16) ^
	     (((uint32_t)(F3[bx(temp1,0)]) ^ (uint32_t)(   bx(temp1,1) ) ^ (uint32_t)(   bx(temp1,2) ) ^ (uint32_t)(F2[bx(temp1,3)])) << 24);

    temp2 =  ( (uint32_t)(F2[bx(temp2,0)]) ^ (uint32_t)(F3[bx(temp2,1)]) ^ (uint32_t)(   bx(temp2,2) ) ^ (uint32_t)(   bx(temp2,3) )       ) ^
	     (((uint32_t)(   bx(temp2,0) ) ^ (uint32_t)(F2[bx(temp2,1)]) ^ (uint32_t)(F3[bx(temp2,2)]) ^ (uint32_t)(   bx(temp2,3) )) <<  8) ^
	     (((uint32_t)(   bx(temp2,0) ) ^ (uint32_t)(   bx(temp2,1) ) ^ (uint32_t)(F2[bx(temp2,2)]) ^ (uint32_t)(F3[bx(temp2,3)])) << 16) ^
	     (((uint32_t)(F3[bx(temp2,0)]) ^ (uint32_t)(   bx(temp2,1) ) ^ (uint32_t)(   bx(temp2,2) ) ^ (uint32_t)(F2[bx(temp2,3)])) << 24);

#undef	bx

    R[1] ^= temp1;
    R[2] ^= ROTL_DWORD(temp1, 13);
    R[3] ^= ROTL_DWORD(temp1, 23);
    R[5] ^= temp2;
    R[6] ^= ROTL_DWORD(temp2, 29);
    R[7] ^= ROTL_DWORD(temp2, 7);

    // Register swap
    temp1 = R[7];
    R[7] = R[6];
    R[6] = R[5];
    R[5] = R[4];
    R[4] = R[3];
    R[3] = R[2];
    R[2] = R[1];
    R[1] = R[0];
    R[0] = temp1;
}

////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION NAME  : step512
//
// DESCRIPTION    : Step function of ARIRANG-384 and Arirang-512
//
// PARAMETERS     : R  - working variables
//                  M1,M2 - message block
//
// RETURNS        : void
//
////////////////////////////////////////////////////////////////////////////////
static
void step512(uint64_t R[8], uint64_t M1, uint64_t M2)
{
    uint64_t temp1, temp2;

    // Message XOR
    R[0] ^= M1;
    R[4] ^= M2;

#define bx(x, n)	((uint8_t)((x) >> (8 * n)))

    // Sub-byte
    temp1 =      (uint64_t)(sbx[bx(R[0], 0)])         ^ ((uint64_t)(sbx[bx(R[0], 1)]) <<  8) ^ ((uint64_t)(sbx[bx(R[0], 2)]) << 16) ^ ((uint64_t)(sbx[bx(R[0], 3)]) << 24) ^
		((uint64_t)(sbx[bx(R[0], 4)]) <<  32) ^ ((uint64_t)(sbx[bx(R[0], 5)]) << 40) ^ ((uint64_t)(sbx[bx(R[0], 6)]) << 48) ^ ((uint64_t)(sbx[bx(R[0], 7)]) << 56);
    temp2 =      (uint64_t)(sbx[bx(R[4], 0)])         ^ ((uint64_t)(sbx[bx(R[4], 1)]) <<  8) ^ ((uint64_t)(sbx[bx(R[4], 2)]) << 16) ^ ((uint64_t)(sbx[bx(R[4], 3)]) << 24) ^
		((uint64_t)(sbx[bx(R[4], 4)]) <<  32) ^ ((uint64_t)(sbx[bx(R[4], 5)]) << 40) ^ ((uint64_t)(sbx[bx(R[4], 6)]) << 48) ^ ((uint64_t)(sbx[bx(R[4], 7)]) << 56);

    // MDS transformation
    temp1 = ( (uint64_t)(   bx(temp1,0) ) ^ (uint64_t)(F2[bx(temp1,1)]) ^ (uint64_t)(FA[bx(temp1,2)]) ^ (uint64_t)(F9[bx(temp1,3)]) ^ (uint64_t)(F8[bx(temp1,4)]) ^ (uint64_t)(   bx(temp1,5) ) ^ (uint64_t)(F4[bx(temp1,6)]) ^ (uint64_t)(   bx(temp1,7) )        ) ^
	    (((uint64_t)(   bx(temp1,0) ) ^ (uint64_t)(   bx(temp1,1) ) ^ (uint64_t)(F2[bx(temp1,2)]) ^ (uint64_t)(FA[bx(temp1,3)]) ^ (uint64_t)(F9[bx(temp1,4)]) ^ (uint64_t)(F8[bx(temp1,5)]) ^ (uint64_t)(   bx(temp1,6) ) ^ (uint64_t)(F4[bx(temp1,7)])) <<  8 ) ^
	    (((uint64_t)(F4[bx(temp1,0)]) ^ (uint64_t)(   bx(temp1,1) ) ^ (uint64_t)(   bx(temp1,2) ) ^ (uint64_t)(F2[bx(temp1,3)]) ^ (uint64_t)(FA[bx(temp1,4)]) ^ (uint64_t)(F9[bx(temp1,5)]) ^ (uint64_t)(F8[bx(temp1,6)]) ^ (uint64_t)(   bx(temp1,7) )) << 16 ) ^
	    (((uint64_t)(   bx(temp1,0) ) ^ (uint64_t)(F4[bx(temp1,1)]) ^ (uint64_t)(   bx(temp1,2) ) ^ (uint64_t)(   bx(temp1,3) ) ^ (uint64_t)(F2[bx(temp1,4)]) ^ (uint64_t)(FA[bx(temp1,5)]) ^ (uint64_t)(F9[bx(temp1,6)]) ^ (uint64_t)(F8[bx(temp1,7)])) << 24 ) ^
	    (((uint64_t)(F8[bx(temp1,0)]) ^ (uint64_t)(   bx(temp1,1) ) ^ (uint64_t)(F4[bx(temp1,2)]) ^ (uint64_t)(   bx(temp1,3) ) ^ (uint64_t)(   bx(temp1,4) ) ^ (uint64_t)(F2[bx(temp1,5)]) ^ (uint64_t)(FA[bx(temp1,6)]) ^ (uint64_t)(F9[bx(temp1,7)])) << 32 ) ^
	    (((uint64_t)(F9[bx(temp1,0)]) ^ (uint64_t)(F8[bx(temp1,1)]) ^ (uint64_t)(   bx(temp1,2) ) ^ (uint64_t)(F4[bx(temp1,3)]) ^ (uint64_t)(   bx(temp1,4) ) ^ (uint64_t)(   bx(temp1,5) ) ^ (uint64_t)(F2[bx(temp1,6)]) ^ (uint64_t)(FA[bx(temp1,7)])) << 40 ) ^
	    (((uint64_t)(FA[bx(temp1,0)]) ^ (uint64_t)(F9[bx(temp1,1)]) ^ (uint64_t)(F8[bx(temp1,2)]) ^ (uint64_t)(   bx(temp1,3) ) ^ (uint64_t)(F4[bx(temp1,4)]) ^ (uint64_t)(   bx(temp1,5) ) ^ (uint64_t)(   bx(temp1,6) ) ^ (uint64_t)(F2[bx(temp1,7)])) << 48 ) ^
	    (((uint64_t)(F2[bx(temp1,0)]) ^ (uint64_t)(FA[bx(temp1,1)]) ^ (uint64_t)(F9[bx(temp1,2)]) ^ (uint64_t)(F8[bx(temp1,3)]) ^ (uint64_t)(   bx(temp1,4) ) ^ (uint64_t)(F4[bx(temp1,5)]) ^ (uint64_t)(   bx(temp1,6) ) ^ (uint64_t)(   bx(temp1,7) )) << 56 );

    temp2 = ( (uint64_t)(   bx(temp2,0) ) ^ (uint64_t)(F2[bx(temp2,1)]) ^ (uint64_t)(FA[bx(temp2,2)]) ^ (uint64_t)(F9[bx(temp2,3)]) ^ (uint64_t)(F8[bx(temp2,4)]) ^ (uint64_t)(   bx(temp2,5) ) ^ (uint64_t)(F4[bx(temp2,6)]) ^ (uint64_t)(   bx(temp2,7) )        ) ^
	    (((uint64_t)(   bx(temp2,0) ) ^ (uint64_t)(   bx(temp2,1) ) ^ (uint64_t)(F2[bx(temp2,2)]) ^ (uint64_t)(FA[bx(temp2,3)]) ^ (uint64_t)(F9[bx(temp2,4)]) ^ (uint64_t)(F8[bx(temp2,5)]) ^ (uint64_t)(   bx(temp2,6) ) ^ (uint64_t)(F4[bx(temp2,7)])) <<  8 ) ^
	    (((uint64_t)(F4[bx(temp2,0)]) ^ (uint64_t)(   bx(temp2,1) ) ^ (uint64_t)(   bx(temp2,2) ) ^ (uint64_t)(F2[bx(temp2,3)]) ^ (uint64_t)(FA[bx(temp2,4)]) ^ (uint64_t)(F9[bx(temp2,5)]) ^ (uint64_t)(F8[bx(temp2,6)]) ^ (uint64_t)(   bx(temp2,7) )) << 16 ) ^
	    (((uint64_t)(   bx(temp2,0) ) ^ (uint64_t)(F4[bx(temp2,1)]) ^ (uint64_t)(   bx(temp2,2) ) ^ (uint64_t)(   bx(temp2,3) ) ^ (uint64_t)(F2[bx(temp2,4)]) ^ (uint64_t)(FA[bx(temp2,5)]) ^ (uint64_t)(F9[bx(temp2,6)]) ^ (uint64_t)(F8[bx(temp2,7)])) << 24 ) ^
	    (((uint64_t)(F8[bx(temp2,0)]) ^ (uint64_t)(   bx(temp2,1) ) ^ (uint64_t)(F4[bx(temp2,2)]) ^ (uint64_t)(   bx(temp2,3) ) ^ (uint64_t)(   bx(temp2,4) ) ^ (uint64_t)(F2[bx(temp2,5)]) ^ (uint64_t)(FA[bx(temp2,6)]) ^ (uint64_t)(F9[bx(temp2,7)])) << 32 ) ^
	    (((uint64_t)(F9[bx(temp2,0)]) ^ (uint64_t)(F8[bx(temp2,1)]) ^ (uint64_t)(   bx(temp2,2) ) ^ (uint64_t)(F4[bx(temp2,3)]) ^ (uint64_t)(   bx(temp2,4) ) ^ (uint64_t)(   bx(temp2,5) ) ^ (uint64_t)(F2[bx(temp2,6)]) ^ (uint64_t)(FA[bx(temp2,7)])) << 40 ) ^
	    (((uint64_t)(FA[bx(temp2,0)]) ^ (uint64_t)(F9[bx(temp2,1)]) ^ (uint64_t)(F8[bx(temp2,2)]) ^ (uint64_t)(   bx(temp2,3) ) ^ (uint64_t)(F4[bx(temp2,4)]) ^ (uint64_t)(   bx(temp2,5) ) ^ (uint64_t)(   bx(temp2,6) ) ^ (uint64_t)(F2[bx(temp2,7)])) << 48 ) ^
	    (((uint64_t)(F2[bx(temp2,0)]) ^ (uint64_t)(FA[bx(temp2,1)]) ^ (uint64_t)(F9[bx(temp2,2)]) ^ (uint64_t)(F8[bx(temp2,3)]) ^ (uint64_t)(   bx(temp2,4) ) ^ (uint64_t)(F4[bx(temp2,5)]) ^ (uint64_t)(   bx(temp2,6) ) ^ (uint64_t)(   bx(temp2,7) )) << 56 );

#undef	bx

    R[1] ^= temp1;
    R[2] ^= ROTL_QWORD(temp1, 29);
    R[3] ^= ROTL_QWORD(temp1, 41);
    R[5] ^= temp2;
    R[6] ^= ROTL_QWORD(temp2, 53);
    R[7] ^= ROTL_QWORD(temp2, 13);

    // Register swap
    temp1 = R[7];
    R[7] = R[6];
    R[6] = R[5];
    R[5] = R[4];
    R[4] = R[3];
    R[3] = R[2];
    R[2] = R[1];
    R[1] = R[0];
    R[0] = temp1;
}

////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION NAME  : Arirang_Compression256
//
// DESCRIPTION    : Counter addition and compression function of Arirang-224 and ARIRANG-256
//
// PARAMETERS     : sp - a structure that holds the hashState information
//
// RETURNS        : void
//
////////////////////////////////////////////////////////////////////////////////
static
void Arirang_Compression256(arirangParam *sp)
{
    uint32_t	R[8], W[32];
    int i;

    // Counter Addition
    sp->workingvar[0] ^= ((uint32_t*)sp->counter)[1];
    sp->workingvar[4] ^= ((uint32_t*)sp->counter)[0];

    // Message Schedue
    for (i = 0; i < 16; i++)
	W[i] = GetData(((uint32_t*)sp->block)[i]);

    W[16] = ROTL_DWORD((W[ 9] ^ W[11] ^ W[13] ^ W[15] ^ K256[ 0]),  5);
    W[17] = ROTL_DWORD((W[ 8] ^ W[10] ^ W[12] ^ W[14] ^ K256[ 1]), 11);
    W[18] = ROTL_DWORD((W[ 1] ^ W[ 3] ^ W[ 5] ^ W[ 7] ^ K256[ 2]), 19);
    W[19] = ROTL_DWORD((W[ 0] ^ W[ 2] ^ W[ 4] ^ W[ 6] ^ K256[ 3]), 31);

    W[20] = ROTL_DWORD((W[14] ^ W[ 4] ^ W[10] ^ W[ 0] ^ K256[ 4]),  5);
    W[21] = ROTL_DWORD((W[11] ^ W[ 1] ^ W[ 7] ^ W[13] ^ K256[ 5]), 11);
    W[22] = ROTL_DWORD((W[ 6] ^ W[12] ^ W[ 2] ^ W[ 8] ^ K256[ 6]), 19);
    W[23] = ROTL_DWORD((W[ 3] ^ W[ 9] ^ W[15] ^ W[ 5] ^ K256[ 7]), 31);

    W[24] = ROTL_DWORD((W[13] ^ W[15] ^ W[ 1] ^ W[ 3] ^ K256[ 8]),  5);
    W[25] = ROTL_DWORD((W[ 4] ^ W[ 6] ^ W[ 8] ^ W[10] ^ K256[ 9]), 11);
    W[26] = ROTL_DWORD((W[ 5] ^ W[ 7] ^ W[ 9] ^ W[11] ^ K256[10]), 19);
    W[27] = ROTL_DWORD((W[12] ^ W[14] ^ W[ 0] ^ W[ 2] ^ K256[11]), 31);

    W[28] = ROTL_DWORD((W[10] ^ W[ 0] ^ W[ 6] ^ W[12] ^ K256[12]),  5);
    W[29] = ROTL_DWORD((W[15] ^ W[ 5] ^ W[11] ^ W[ 1] ^ K256[13]), 11);
    W[30] = ROTL_DWORD((W[ 2] ^ W[ 8] ^ W[14] ^ W[ 4] ^ K256[14]), 19);
    W[31] = ROTL_DWORD((W[ 7] ^ W[13] ^ W[ 3] ^ W[ 9] ^ K256[15]), 31);

    // Register Initialize
    for (i = 0; i < 8; i++)
	R[i] = (uint32_t)sp->workingvar[i];

    // 1 Round
    step256(R, W[16], W[17]);
    step256(R, W[ 0], W[ 1]);
    step256(R, W[ 2], W[ 3]);
    step256(R, W[ 4], W[ 5]);
    step256(R, W[ 6], W[ 7]);

    step256(R, W[18], W[19]);
    step256(R, W[ 8], W[ 9]);
    step256(R, W[10], W[11]);
    step256(R, W[12], W[13]);
    step256(R, W[14], W[15]);

    // 2 Round
    step256(R, W[20], W[21]);
    step256(R, W[ 3], W[ 6]);
    step256(R, W[ 9], W[12]);
    step256(R, W[15], W[ 2]);
    step256(R, W[ 5], W[ 8]);

    step256(R, W[22], W[23]);
    step256(R, W[11], W[14]);
    step256(R, W[ 1], W[ 4]);
    step256(R, W[ 7], W[10]);
    step256(R, W[13], W[ 0]);

    // Feedforward_1
    for (i = 0; i < 8; i++)
	R[i] ^= sp->workingvar[i];

    // 3 Round
    step256(R, W[24], W[25]);
    step256(R, W[12], W[ 5]);
    step256(R, W[14], W[ 7]);
    step256(R, W[ 0], W[ 9]);
    step256(R, W[ 2], W[11]);

    step256(R, W[26], W[27]);
    step256(R, W[ 4], W[13]);
    step256(R, W[ 6], W[15]);
    step256(R, W[ 8], W[ 1]);
    step256(R, W[10], W[ 3]);

    // 4 Round
    step256(R, W[28], W[29]);
    step256(R, W[ 7], W[ 2]);
    step256(R, W[13], W[ 8]);
    step256(R, W[ 3], W[14]);
    step256(R, W[ 9], W[ 4]);

    step256(R, W[30], W[31]);
    step256(R, W[15], W[10]);
    step256(R, W[ 5], W[ 0]);
    step256(R, W[11], W[ 6]);
    step256(R, W[ 1], W[12]);

    // Feedforward_2
    for (i = 0; i < 8; i++)
	sp->workingvar[i] ^= R[i];

    sp->counter[0]++;
}

////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION NAME  : ARIRANG_Compression512
//
// DESCRIPTION    : Counter Addition and compression function of ARIRANG-384 and ARIRANG-512
//
// PARAMETERS     : sp - a structure that holds the hashState information
//
// RETURNS        : void
//
////////////////////////////////////////////////////////////////////////////////
static
void Arirang_Compression512(arirangParam *sp)
{
    uint64_t	R[8], W[32];
    int i;

    // Counter Addition
    sp->workingvar[0] ^= sp->counter[1];
    sp->workingvar[4] ^= sp->counter[0];

    // Message Scheduling
    for (i = 0; i < 16; i++)
	W[i] = (uint64_t)(GetData(((uint32_t*)sp->block)[2*i+1])) | ((uint64_t)(GetData(((uint32_t*)sp->block)[2*i])) << 32);

    W[16] = ROTL_QWORD((W[ 9] ^ W[11] ^ W[13] ^ W[15] ^ K512[ 0]), 11);
    W[17] = ROTL_QWORD((W[ 8] ^ W[10] ^ W[12] ^ W[14] ^ K512[ 1]), 23);
    W[18] = ROTL_QWORD((W[ 1] ^ W[ 3] ^ W[ 5] ^ W[ 7] ^ K512[ 2]), 37);
    W[19] = ROTL_QWORD((W[ 0] ^ W[ 2] ^ W[ 4] ^ W[ 6] ^ K512[ 3]), 59);

    W[20] = ROTL_QWORD((W[14] ^ W[ 4] ^ W[10] ^ W[ 0] ^ K512[ 4]), 11);
    W[21] = ROTL_QWORD((W[11] ^ W[ 1] ^ W[ 7] ^ W[13] ^ K512[ 5]), 23);
    W[22] = ROTL_QWORD((W[ 6] ^ W[12] ^ W[ 2] ^ W[ 8] ^ K512[ 6]), 37);
    W[23] = ROTL_QWORD((W[ 3] ^ W[ 9] ^ W[15] ^ W[ 5] ^ K512[ 7]), 59);

    W[24] = ROTL_QWORD((W[13] ^ W[15] ^ W[ 1] ^ W[ 3] ^ K512[ 8]), 11);
    W[25] = ROTL_QWORD((W[ 4] ^ W[ 6] ^ W[ 8] ^ W[10] ^ K512[ 9]), 23);
    W[26] = ROTL_QWORD((W[ 5] ^ W[ 7] ^ W[ 9] ^ W[11] ^ K512[10]), 37);
    W[27] = ROTL_QWORD((W[12] ^ W[14] ^ W[ 0] ^ W[ 2] ^ K512[11]), 59);

    W[28] = ROTL_QWORD((W[10] ^ W[ 0] ^ W[ 6] ^ W[12] ^ K512[12]), 11);
    W[29] = ROTL_QWORD((W[15] ^ W[ 5] ^ W[11] ^ W[ 1] ^ K512[13]), 23);
    W[30] = ROTL_QWORD((W[ 2] ^ W[ 8] ^ W[14] ^ W[ 4] ^ K512[14]), 37);
    W[31] = ROTL_QWORD((W[ 7] ^ W[13] ^ W[ 3] ^ W[ 9] ^ K512[15]), 59);

    // Register Initialize
    for (i = 0; i < 8; i++)
	R[i] = sp->workingvar[i];

    // 1 Round
    step512(R, W[16], W[17]);
    step512(R, W[ 0], W[ 1]);
    step512(R, W[ 2], W[ 3]);
    step512(R, W[ 4], W[ 5]);
    step512(R, W[ 6], W[ 7]);

    step512(R, W[18], W[19]);
    step512(R, W[ 8], W[ 9]);
    step512(R, W[10], W[11]);
    step512(R, W[12], W[13]);
    step512(R, W[14], W[15]);

    // 2 Round
    step512(R, W[20], W[21]);
    step512(R, W[ 3], W[ 6]);
    step512(R, W[ 9], W[12]);
    step512(R, W[15], W[ 2]);
    step512(R, W[ 5], W[ 8]);

    step512(R, W[22], W[23]);
    step512(R, W[11], W[14]);
    step512(R, W[ 1], W[ 4]);
    step512(R, W[ 7], W[10]);
    step512(R, W[13], W[ 0]);

    // Feedforward_1
    for (i = 0; i < 8; i++)
	R[i] ^= sp->workingvar[i];

    // 3 Round
    step512(R, W[24], W[25]);
    step512(R, W[12], W[ 5]);
    step512(R, W[14], W[ 7]);
    step512(R, W[ 0], W[ 9]);
    step512(R, W[ 2], W[11]);

    step512(R, W[26], W[27]);
    step512(R, W[ 4], W[13]);
    step512(R, W[ 6], W[15]);
    step512(R, W[ 8], W[ 1]);
    step512(R, W[10], W[ 3]);

    // 4 Round
    step512(R, W[28], W[29]);
    step512(R, W[ 7], W[ 2]);
    step512(R, W[13], W[ 8]);
    step512(R, W[ 3], W[14]);
    step512(R, W[ 9], W[ 4]);

    step512(R, W[30], W[31]);
    step512(R, W[15], W[10]);
    step512(R, W[ 5], W[ 0]);
    step512(R, W[11], W[ 6]);
    step512(R, W[ 1], W[12]);

    // Feedforward_2
    for (i = 0; i < 8; i++)
	sp->workingvar[i] ^= R[i];

    // Increment Counter
    sp->counter[0]++; if (sp->counter[0] == 0) sp->counter[1]++;
}

static const uint64_t _init_224[] = {
    0xcbbb9d5d, 0x629a292a, 0x9159015a, 0x152fecd8,
    0x67332667, 0x8eb44a87, 0xdb0c2e0d, 0x47b5481d
};

static const uint64_t _init_256[] = {
    0x6a09e667, 0xbb67ae85, 0x3c6ef372, 0xa54ff53a,
    0x510e527f, 0x9b05688c, 0x1f83d9ab, 0x5be0cd19
};

static const uint64_t _init_384[] = {
    0xcbbb9d5dc1059ed8ULL, 0x629a292a367cd507ULL,
    0x9159015a3070dd17ULL, 0x152fecd8f70e5939ULL,
    0x67332667ffc00b31ULL, 0x8eb44a8768581511ULL,
    0xdb0c2e0d64f98fa7ULL, 0x47b5481dbefa4fa4ULL
};

static const uint64_t _init_512[] = {
    0x6a09e667f3bcc908ULL, 0xbb67ae8584caa73bULL,
    0x3c6ef372fe94f82bULL, 0xa54ff53a5f1d36f1ULL,
    0x510e527fade682d1ULL, 0x9b05688c2b3e6c1fULL,
    0x1f83d9abfb41bd6bULL, 0x5be0cd19137e2179ULL
};

int arirangInit(arirangParam *sp, int hashbitlen)
{
    const uint64_t *iv = NULL;
    uint32_t blocklen = 0;

    switch (hashbitlen) {
    case 224:
	iv = _init_224;
	blocklen = 64;		/* ARIRANG224_BLOCK_LEN */
	break;
    case 256:
	iv = _init_256;
	blocklen = 64;		/* ARIRANG256_BLOCK_LEN */
	break;
    case 384:
	iv = _init_384;
	blocklen = 128;		/* ARIRANG384_BLOCK_LEN */
	break;
    case 512:
	iv = _init_512;
	blocklen = 128;		/* ARIRANG512_BLOCK_LEN */
	break;
    default:
	return 2;	/* BAD_HASHLEN */
	break;
    }

    memset(sp, 0, sizeof(*sp));
    sp->hashbitlen = hashbitlen;
    sp->blocklen = blocklen;

    if (iv)
	memcpy(sp->workingvar, iv, sizeof(sp->workingvar));
    return 0;
}

int
arirangReset(arirangParam *sp)
{
    return arirangInit(sp, sp->hashbitlen);
}

int arirangUpdate(arirangParam *sp, const byte *data, size_t size)
{
    uint64_t databitlen = 8 * size;
    uint32_t RemainedLen;
    uint32_t PartLen;
    uint64_t databytelen;
    uint64_t temp;

    sp->remainderbit = databitlen & 7;

    // If length of data is not multiple of 8, databytelen = databitlen / 8 + 1;
    databytelen = ( databitlen >> 3) + (sp->remainderbit != 0);

    // Compute the number of hashed bytes mod ARIRANG_BLOCK_LEN
    RemainedLen = (sp->count[0] >> 3) % sp->blocklen;

    // Compute the number of bytes that can be filled up
    PartLen = sp->blocklen - RemainedLen;

    // Update count (number of toatl data bits)
    temp = sp->count[0] + (databytelen << 3);
    if (temp < sp->count[0])	sp->count[1]++;
    sp->count[0]=temp;

    if (databytelen > PartLen
     || (databytelen == PartLen && sp->remainderbit == 0))
    {
	memcpy(sp->block + RemainedLen, data, (int)PartLen);
	if (sp->hashbitlen < 257) Arirang_Compression256(sp);
	else Arirang_Compression512(sp);

	data += PartLen;
	databytelen -= PartLen;
	RemainedLen = 0;

	while (databytelen > sp->blocklen
	   || (databytelen == sp->blocklen && sp->remainderbit == 0))
	{
	    memcpy((uint8_t *)sp->block, data, (int)sp->blocklen);
	    if (sp->hashbitlen < 257) Arirang_Compression256(sp);
	    else Arirang_Compression512(sp);

	    data += sp->blocklen;
	    databytelen -= sp->blocklen;
	}
    }

    //	Buffer remaining input
    memcpy((uint8_t *)sp->block + RemainedLen, data, (int)databytelen);

    return 0;
}

int arirangDigest(arirangParam *sp, byte *digest)
{
    uint32_t i, dwIndex;
    uint32_t temp=(sp->blocklen >> 3);
    uint64_t count[2];

    // Padding the message
    if (sp->remainderbit) {
	// Length of data isn't multiple of 8
	count[0] = sp->count[0] + sp->remainderbit - 8;
	count[1] = sp->count[1];
	dwIndex = ((count[0] + (sp->blocklen<<3) ) >> 3) % sp->blocklen;
	sp->block[dwIndex] &= 0xff-(1<<(8-sp->remainderbit))+1;
	sp->block[dwIndex++] ^= 0x80>>(sp->remainderbit);
    } else {
	// Length of data is multiple of 8
	count[0] = sp->count[0];
	count[1] = sp->count[1];
	dwIndex = (count[0] >> 3) % sp->blocklen;
	sp->block[dwIndex++] = 0x80;
    }

    if (dwIndex > (sp->blocklen - temp)) {
	memset((uint8_t *)sp->block + dwIndex, 0, (int)(sp->blocklen - dwIndex));
	if (sp->hashbitlen < 257) Arirang_Compression256(sp);
	else Arirang_Compression512(sp);

	memset((uint8_t *)sp->block, 0, (int)sp->blocklen - temp);
    } else
	memset((uint8_t *)sp->block + dwIndex, 0, (int)(sp->blocklen - dwIndex - temp));

#if !WORDS_BIGENDIAN
    count[0] = ENDIAN_REVERSE_DWORD(((uint32_t*)count)[1])
	| ((uint64_t)(ENDIAN_REVERSE_DWORD(((uint32_t*)count)[0])) << 32);
    count[1] = ENDIAN_REVERSE_DWORD(((uint32_t*)count)[3])
	| ((uint64_t)(ENDIAN_REVERSE_DWORD(((uint32_t*)count)[2])) << 32);
#endif

    // Fixed counter value for the last message block
    if (sp->hashbitlen > 257) {
	((uint64_t *)sp->block)[sp->blocklen/8-2] = count[1];
	((uint64_t *)sp->block)[sp->blocklen/8-1] = count[0];
	sp->counter[1] = 0xb7e151628aed2a6aULL;
	sp->counter[0] = 0xbf7158809cf4f3c7ULL;
    } else {
	((uint64_t *)sp->block)[sp->blocklen/8-1] = count[0];
	sp->counter[0] = 0xb7e151628aed2a6aULL;
    }

    if (sp->hashbitlen <257) Arirang_Compression256(sp);
    else Arirang_Compression512(sp);

    if (sp->hashbitlen <257)
	for (i = 0; i < (uint32_t)(sp->hashbitlen >> 3); i += 4)
	    BIG_D2B((sp->workingvar)[(i*2) / 8], &digest[i]);
    else
	for (i = 0; i < (uint32_t)(sp->hashbitlen >> 3); i += 8)
	    BIG_Q2B((sp->workingvar)[i / 8], &digest[i]);

    return 0;
}
