/******************************************************************************
*
*                   INTEL CORPORATION PROPRIETARY INFORMATION
*       This software is supplied under the terms of a license agreement or
*       nondisclosure agreement with Intel Corporation and may not be copied
*       or disclosed except in accordance with the terms of that agreement.
*
*            Copyright (c) 2008-2009 Intel Corporation. All Rights Reserved.
*
*       All rights reserved.  No part of this program or publication may be
*       reproduced, transmitted, transcribed, stored in a retrieval system,
*       or translated into any language or computer language, in any form or
*       by any means, electronic, mechanical, magnetic, optical, chemical,
*       manual, or otherwise, without the prior written permission of Intel
*       Corporation.
*
*
*******************************************************************************/
#ifndef __CR_HASH_MAP_H_
#define __CR_HASH_MAP_H_

#include <os/include/cr_malloc.h>
#include <string.h>
#include <stdio.h>

#include <os/include/efi/cr_default_functors.h>
#include <os/include/efi/cr_vector.h>
#include <os/include/efi/cr_pair.h>

namespace osal_stl {

#define MAX_CR_PRIMES 30

static const unsigned long _cr_prime_list[MAX_CR_PRIMES] = 
{
3ul, 53ul, 127ul, 283ul, 541ul, 1039ul,
1543ul, 3079ul, 6151ul, 9283ul, 12289ul, 24593ul, 49157ul, 98317ul,
196613ul, 393241ul, 786433ul, 1572869ul, 3145739ul, 6291469ul,
2582917ul, 25165843ul, 50331653ul,   100663319ul, 201326611ul,
402653189ul, 805306457ul, 1610612741ul, 3221225473ul, 4294967291ul
};

template<class val>
struct HashNode
{
    HashNode*   m_next;
    val     m_val;
};

template<class Key, class Data, class HashFcn = cr_hash<Key>, class EqualKey = cr_equal_to<Key> >
class cr_hash_map {

    //public typedefs and member variables.
public:
    typedef Key                 key_type;
    typedef Data                data_type;
    typedef HashFcn             hasher;
    typedef EqualKey            key_equal;
    typedef cr_pair<const key_type, data_type> value_type;
    typedef value_type*         pointer;
    typedef value_type&         reference;
    typedef const value_type&     const_reference;
    typedef size_t              size_type;
    typedef size_t              difference_type;
    typedef HashNode<value_type>    node_type;

    typedef class iterator {
    public:
        typedef cr_hash_map<Key, Data, HashFcn, EqualKey>    HashTable_type;
        typedef HashTable_type*                              HashTablePtr_type;
        typedef typename    HashTable_type::node_type    node_type;
        typedef typename    HashTable_type::value_type   value_type;
        typedef typename    HashTable_type::reference    reference;
        typedef typename    HashTable_type::pointer      pointer;
        typedef typename    HashTable_type::size_type    size_type;
        typedef node_type*  nodeptr_type;

        /* default constructor */
        CR_NO_THROW inline iterator() {
            m_nptr      = (nodeptr_type) 0;
            m_hashPtr   = (HashTablePtr_type)0;
        }

        /* constructor */
        CR_NO_THROW inline iterator(nodeptr_type nptr, HashTablePtr_type hptr) 
            : m_nptr(nptr), m_hashPtr(hptr) { }

        /* operator* */
        CR_NO_THROW reference operator*() const {
            return m_nptr->m_val;
        }

        /* operator-> */
        CR_NO_THROW pointer operator->() const {
            return &(operator*());
        }

        /* ++operator */
        CR_NO_THROW inline iterator& operator++(void) {
            const nodeptr_type start = m_nptr;
            m_nptr = m_nptr->m_next;

            /* check if there there is anything in the next node */
            if (m_nptr == (nodeptr_type) 0)
            {
                /* If not traverse the hash table, get the first */ 
                for (size_type bucket = m_hashPtr->get_bucket_num(m_hashPtr->get_key(start->m_val), m_hashPtr->hashtable.size());
                    ++bucket < m_hashPtr->hashtable.size(); )
                {
                    m_nptr = m_hashPtr->hashtable[bucket];

                    if (m_nptr)
                        break;
                }
            }

            return *this;
        }

        /* operator++ */
        CR_NO_THROW inline iterator operator++(int) {
            iterator tmp = *this;
            ++*this;
            return tmp;
        }

        /* operator== */
        CR_NO_THROW inline bool operator==(const iterator& itr) const {
            return (m_nptr == itr.m_nptr);
        }

        /* operator!= */
        CR_NO_THROW inline bool operator!=(const iterator& itr) const {
            return (m_nptr != itr.m_nptr);
        }

        CR_NO_THROW inline nodeptr_type        get_nptr(void) { return m_nptr; }

    protected:
        nodeptr_type        m_nptr;
        HashTablePtr_type   m_hashPtr;
    } iterator;

    typedef const iterator  const_iterator;

    //public methods.
public:
    /* default constructor */
    CR_NO_THROW cr_hash_map() {
        buckets = get_max_bucket_count(MIN_CR_HASH_MAP);
        hashtable.reserve(buckets);
        hashtable.insert(hashtable.end(), (HashTableSize)buckets, (nodeptr_type)0);
        m_num_elements = 0;
    }

    /* destructor */
    CR_NO_THROW ~cr_hash_map() {
        clear();
    }

    /* constructor */
    CR_NO_THROW cr_hash_map(size_type n) {
        buckets = get_max_bucket_count(n);
        hashtable.reserve(buckets);
        hashtable.insert(hashtable.end(), (HashTableSize)buckets, (nodeptr_type)0);
        m_num_elements = 0;
    }

    /* constructor */
    CR_NO_THROW cr_hash_map(size_type n, const hasher& h) {
        buckets = get_max_bucket_count(n);
        hashtable.reserve(buckets);
        hashtable.insert(hashtable.end(), (HashTableSize)buckets, (nodeptr_type)0);
        m_num_elements = 0;
        m_hash = h; 
    }

    /* constructor */
    CR_NO_THROW cr_hash_map(size_type n, const hasher& h, const key_equal& k) {
        buckets = get_max_bucket_count(n);
        hashtable.reserve(buckets);
        hashtable.insert(hashtable.end(), (HashTableSize)buckets, (nodeptr_type)0);
        m_num_elements  = 0;
        m_hash          = h; 
        m_equals        = k;
    }

    /* copy contructor */
    CR_NO_THROW cr_hash_map(const cr_hash_map& ht) {
		m_hash  = ht.m_hash;
        m_equals = ht.m_equals;
        copy_hashtable(ht);
    }

    /* assignment operator */
    CR_NO_THROW inline cr_hash_map& operator=(const cr_hash_map& ht) {
		//printf("%s function called.\n",__FUNCTION__);
        /* assign only if not ourselves */
        if (&ht != this)
        {
            clear();
            m_hash  = ht.m_hash;
            m_equals = ht.m_equals;
            copy_hashtable(ht);
        }
       
        return *this; 
    }

    CR_NO_THROW inline void clear(void) {
        for (size_type i=0; i<hashtable.size(); ++i)
        {
            nodeptr_type nPtr = hashtable[i];
            while(nPtr != (nodeptr_type) 0)
            {
                nodeptr_type next = nPtr->m_next;
                delete_node(nPtr);
                nPtr = next;

				hashtable[i] = (nodeptr_type) 0;
            }
            
        }

        m_num_elements = 0;
    }//end of clear

    CR_NO_THROW inline size_type erase(const key_type& k) {
        const size_type n = get_bucket_num(k, hashtable.size());
        nodeptr_type start = hashtable[n];
        size_type   noErased = 0;

        if (start)
        {
            nodeptr_type cur = start;
            nodeptr_type next = cur->m_next;
            /* first remove all but the starting node */
            while(next)
            {
                if (m_equals(get_key(next->m_val), k))
                {
                    cur->m_next = next->m_next;
                    delete_node(next);
                    next = cur->m_next;
                    ++noErased;
                    --m_num_elements;
                }
                else
                {
                    cur     = next;
                    next    = cur->m_next;
                }
            }

            /* Remove the starting node now */
            if (m_equals(get_key(start->m_val), k))
            {
                hashtable[n] = start->m_next;
                delete_node(start);
                ++noErased;
                --m_num_elements;
            }
        }

        return noErased;
    }//end of erase

    CR_NO_THROW inline void erase(iterator pos) {
        nodeptr_type nptr = pos.get_nptr();

        if (nptr)
        {
            const size_type n = get_bucket_num(get_key(nptr->m_val), hashtable.size());
            nodeptr_type start = hashtable[n];

            if (start == nptr)
            {
                hashtable[n] = start->m_next;
                delete_node(start);
                --m_num_elements;
            }
            else
            {
                nodeptr_type next = start->m_next;
                while(next)
                {
                    if (next == nptr)
                    {
                        start->m_next = next->m_next;
                        delete_node(next);
                        --m_num_elements;
                        break;
                    }
                    else
                    {
                        start   = next;
                        next    = start->m_next; 
                    }
                } 
            }
        }
    }//end of erase

    CR_NO_THROW inline void erase(iterator first, iterator last) {
        for (iterator i=first; i!=last; )
        {
            iterator tmp = i;
            ++tmp; erase(i); i = tmp;
        }
    }//end of erase

    CR_NO_THROW inline data_type& operator[](const key_type& k) {
		//printf("calling find_or_insert.\n");
		reference v = find_or_insert( value_type( k, data_type() ));
		//printf("returning data_type.\n");
        return v.second; 
    }

    CR_NO_THROW inline cr_pair<iterator, bool> insert(const value_type& x) {
        iterator i;
        bool inserted = false;

        if ((i = find(x.first)) != end())
        {
            inserted = false;
        }
        else 
        {
            find_or_insert(value_type(x.first, x.second));
            i = find(x.first);
            inserted = true;
        }

        return cr_pair<iterator, bool>(i, inserted);
    }

    CR_NO_THROW inline void insert(iterator first, iterator last) {
        for (iterator i = first; i != last; i++)
        {
            insert(*i);
        }
    }

    CR_NO_THROW inline void swap(cr_hash_map& ht) {

        /* First swap hasher function */
        hasher tmpHash    = m_hash;
        m_hash            = ht.m_hash;   
        ht.m_hash         = tmpHash;

        /* Now swap equals function */
        key_equal tmpEq = m_equals;
        m_equals        = ht.m_equals;
        ht.m_equals     = tmpEq;

        /* Now swap the hashtable */
        hashtable.swap(ht.hashtable);

        /* Now swap the num of elements */
        size_type s = m_num_elements;
        m_num_elements = ht.m_num_elements;
        ht.m_num_elements = s;
    }

    CR_NO_THROW inline size_type max_size(void) const {
        return size_type(-1);
    }

    CR_NO_THROW inline hasher hash_funct() const {
        return m_hash;
    }

    CR_NO_THROW inline key_equal key_eq() const {
        return m_equals;
    }

    CR_NO_THROW inline void resize(size_type n) {
        const size_type bucketSize = hashtable.size();

        /* Resize the hashtable only if its smaller */
        if (n > bucketSize)
        {
            const size_type tmpN = get_max_bucket_count(n);

            if (tmpN > n)
            {
                HashTable tmpHashTable(tmpN, (nodeptr_type)(0));
                for (size_type bucket = 0; bucket < bucketSize; ++bucket)
                {
                    /* Get the node */
                    nodeptr_type i = hashtable[bucket];
                    while (i)
                    {
                        size_type nbucketNum = get_bucket_num( get_key(i->m_val), tmpN );
                        hashtable[bucket]           = i->m_next;
                        i->m_next                   = tmpHashTable[nbucketNum];
                        tmpHashTable[nbucketNum]    = i;
                        i                           = hashtable[bucket];
                    }
                }

                /* Now swap the temp hashtable */
                hashtable.swap(tmpHashTable);
            }
        }
    }

    CR_NO_THROW inline size_type bucket_count() const {
        return buckets;
    }

    CR_NO_THROW inline size_type size(void) const {
        return m_num_elements;
    }

    CR_NO_THROW inline bool empty(void) const {
        return (size() == 0);
    }

    CR_NO_THROW inline iterator begin(void) {
        for (size_type n=0; n < hashtable.size(); ++n)
        {
            if (hashtable[n])
            {
                return iterator(hashtable[n], this);
            }
        }

        return end();
    }

    CR_NO_THROW inline iterator end(void) {
        return iterator((nodeptr_type)0, this);
    }

    CR_NO_THROW inline const_iterator begin(void) const {
        return (const_iterator)begin();
    }

    CR_NO_THROW inline const_iterator end(void) const {
        return (const_iterator)end();
    }

    CR_NO_THROW inline iterator find (const key_type& k) {
        size_type n = get_bucket_num(k, hashtable.size());
        nodeptr_type start;
        for(start = hashtable[n]; (start && !m_equals(get_key(start->m_val), k));
            start = start->m_next) { }

        return iterator(start, this);
    }
    
    CR_NO_THROW inline size_type count(const key_type& k) {
        iterator i = find(k);
        size_type count = 0;
        
        if (i != end())
        {
            count = 1;
        }

        return count;
    }

    //protected member variables.
protected:
    typedef node_type*                      nodeptr_type;
    typedef cr_vector<nodeptr_type>         HashTable;
    typedef typename HashTable::iterator    HashTableItr;
    typedef typename HashTable::size_type   HashTableSize;

    hasher          m_hash;
    key_equal       m_equals;
    size_type       buckets;
    HashTable       hashtable;
    size_type       m_num_elements; 

    //protected methods.
protected: 
    CR_NO_THROW inline size_type   get_max_bucket_count(const size_type& n) {
        size_type bc;
        size_type i;
        for (i=0; i < (sizeof(_cr_prime_list)/sizeof(_cr_prime_list[0])); i++)
        {
            bc = _cr_prime_list[i];
            if (bc >= n) break; 
        }

        return bc;
    }

    CR_NO_THROW inline key_type get_key(const value_type& val) const {
        return (key_type)val.first;
    }

    CR_NO_THROW inline size_type get_bucket_num(const key_type key, const size_type size) const {
        return (m_hash(key)%size);
    }

    CR_NO_THROW inline void copy_hashtable(const cr_hash_map& ht) {
        hashtable.clear();
        //hashtable.reserve(ht.hashtable.size(), (nodeptr_type)0);
        hashtable.reserve(ht.hashtable.size());
        hashtable.insert(hashtable.end(), ht.hashtable.size(), (nodeptr_type)0);
        
        /*
         * For all the nodes of to be copied hastable
         * get the collision linked nodes.
         * make a local copy and keep tracing.
         */
        
        for (size_type i = 0; i < ht.hashtable.size(); i++)
        {
            nodeptr_type hNodePtr = ht.hashtable[i];

            if (hNodePtr)
            {
                nodeptr_type tmpPtr = get_new_node(hNodePtr->m_val);
                hashtable[i]        = tmpPtr;

                /* copy collision list */
                for (nodeptr_type next = hNodePtr->m_next;
                    next!=(nodeptr_type)0;
                    hNodePtr = next, next = hNodePtr->m_next)
                {
                    tmpPtr->m_next  = get_new_node(next->m_val);
                    tmpPtr          = tmpPtr->m_next; 
                }//end of for

            }// end of if
        }//end of for
        
        m_num_elements  = ht.m_num_elements;
    }//end of copy_hashtable

    CR_NO_THROW inline nodeptr_type get_new_node(const value_type& val) {
        nodeptr_type nPtr = (nodeptr_type)MALLOC(sizeof(node_type));
        

        if (nPtr != (nodeptr_type)0)
        {
            /* Copy the contents */

            /* Now call constructors */
            pointer pVal = &(nPtr->m_val);
			//printf("get_new_node: calling firsts constructor.\n");
            new((void *)&(pVal->first)) key_type(val.first);
			//printf("get_new_node: calling seconds constructor.\n");
            new((void *)&(pVal->second)) data_type(val.second);

			nPtr->m_next = (nodeptr_type) 0;
        } 
		else
		{
			printf("ERROR:: Unable to Malloc memory in file %s at line %d\n",__FILE__,__LINE__);
		}

		//printf("returning from get_new_node hash_map with nPtr = %xll.\n", (UINT64)nPtr);
        return nPtr;
    }

    CR_NO_THROW inline void delete_node(nodeptr_type nPtr) {
        if (nPtr) {
            /* call destructors */
            const key_type* kp = &(nPtr->m_val.first);
			//printf("delete_node: calling firsts destructor.\n");
            kp->~Key();

            data_type* dp = &(nPtr->m_val.second);
			//printf("delete_node: calling seconds destructor.\n");
            
			//temp fix -- added to avoid hang issue in EFI - BTP Platform
			dp->~Data();

            nPtr->m_next = (nodeptr_type) 0;

            FREE(nPtr);
        }
    }

    CR_NO_THROW inline reference find_or_insert(const value_type& val) {
		//printf("inside find_or_insert.\n");
		size_type n = get_bucket_num(get_key(val), hashtable.size());
        nodeptr_type start = hashtable[n];
		size_type i=0;
		/* check if the key already exists */
		if (m_num_elements > 0) {
			for (nodeptr_type cur = start; cur; cur = cur->m_next)
			{
				if (m_equals(get_key(cur->m_val), get_key(val)))
				{
					//printf("found the key returning.\n");
					return cur->m_val;
				}

				i++;
				if(i > 0xFFFF)
				{
					printf("%s: Looks like a big collision list. CheckThis out.\n",__FUNCTION__);
					//exit(0);
				}
			}
		}

		//printf("key not found.\n");

        /* we are here if the key is not in the hashtable */
        /* Add it */
		//printf("called resize in hash with value = %d\n", m_num_elements+1);
        resize(m_num_elements + 1);
		//printf("after resize.\n");

        nodeptr_type tmp = get_new_node(val);
        tmp->m_next      = start; 
        hashtable[n]     = tmp;
        ++m_num_elements;
		//printf("returning element.\n");
        return tmp->m_val;
    }//end of find_or_insert

	private:
		const static size_t MIN_CR_HASH_MAP=9283ul;
};

} //end of namespace osal_stl

#endif //__CR_HASH_MAP_H_
