<#
  .Notes
	Copyright (c) 2013-2014, Intel Corporation. All rights reserved.
  .Synopsis 
	Configures the solution with the settings in the specified configuration profile
  .Description
	Configures the solution with the settings in the specified configuration profile.
	You can supply the profile using one of these parameters:
	ProfileName - Use this parameter if you created the profile using the Console component of Intel SCS.
				  When using this parameter you must also supply the address of the RCS (in the RCS parameter).
	ProfileFile - Use this parameter if you created the profile using the Profile Editor component of Intel SCS.
	ProfileXML  - Use this parameter if you want to supply the XML of the profile directly in the command line.

  .Example
	.\SCS-Configure -ComputerName MyHost.Domain.com -Component 0b93a9cb-64b1-43ed-bc33-e5b0b8692694 -RCS rcs.domain.com -ProfileName "ProfileinRCS"
  .Example
	.\SCS-Configure -ComputerName MyHost.Domain.com -Component 0b93a9cb-64b1-43ed-bc33-e5b0b8692694 -ProfileFile "MyLocalXMLFile.xml"
  .Example
	.\SCS-Configure -ComputerName MyHost.Domain.com -Component 0b93a9cb-64b1-43ed-bc33-e5b0b8692694 -ProfileXML "<Management><ManagedElement><Profile>Profile Data</Profile></ManagedElement></Management>"
  .Outputs
	None
#>
[CmdletBinding(DefaultParameterSetName='None')]
Param(
	[Parameter(Position=0, Mandatory=$True, HelpMessage='Fully qualified component identifer of the solution component on which to perform the operation')][string]$Component,
	[Parameter(Mandatory=$False, HelpMessage='Address of the computer on which to perform the operation (default: localhost)')][string]$ComputerName = $env:ComputerName,
	[Parameter(Mandatory=$False, HelpMessage='Credentials to use for authenticating to the Solutions Framework (default: current user)')][Object]$Credential = $null,
	[Parameter(Position=1, Mandatory=$False, HelpMessage='The XML content of the profile (without whitespaces)', ValueFromPipeline=$True, ParameterSetName='LocalProfile')][string]$ProfileXML,
	[Parameter(Mandatory=$False, HelpMessage='The profile XML file', ValueFromPipeline=$True, ParameterSetName='LocalProfile')][string]$ProfileFile,	
	[Parameter(Mandatory=$True,  HelpMessage='The address of the RCS where the profile is located (if using the ProfileName parameter', ParameterSetName='RemoteProfile')][string]$RCS,
	[Parameter(Mandatory=$True,  HelpMessage='The name of the profile to retrieve from the RCS', ParameterSetName='RemoteProfile')][string]$ProfileName,
	[Parameter(Mandatory=$False, HelpMessage='Access credentials to use to access the RCS (default: current user')][object]$RCSCredential
) 
	
if ($ProfileXML -And $ProfileFile)
{
	Write-Error "ProfileXML cannot be used with ProfileFile"
	exit 2
}		

	
$profileText = $(
	if ($RCS) {
		$(
			if ($RCSCredential) {
				Get-WMIObject -Credential $RCSCredential -Authentication 6 -Errorvariable err -Namespace Root\Intel_RCS_Editor -ComputerName $RCS -Class RCS_Profile -Filter "ElementName ='$($ProfileName)'" | select -First 1
			} else {
				Get-WMIObject -Authentication 6 -Errorvariable err -Namespace Root\Intel_RCS_Editor -ComputerName $RCS -Class RCS_Profile -Filter "ElementName ='$($ProfileName)'" | select -First 1
			}
		).Text
	} else {
		if ($ProfileFile)
		{
			try 
			{
				[xml]$xml = Get-Content $ProfileFile
				$xml.PreserveWhitespace = $true
				$xml.OuterXml
			}
			catch [Exception] 
			{
				Write-Error "Failed to load the XML profile file $ProfileFile "
				Write-Error "$_.Exception.gettype().fullName"				
				exit $_.Exception.ErrorCode
			}
		}
		else 
		{
			$ProfileXML
		}
	}
)

if ($profileText -eq $null) {
	if($err.Count -gt 0) {$mg=$err[0];} else { $mg="";}
	Write-Error "Failed to load the XML profile. $mg"
	exit 4
}

$filter = $(
	if ($Component -Match '%') {
		"ComponentID LIKE ""$Component"""
	} else {
		"ComponentID = ""$Component"""
	}
)

$SCS_ManagedElement = $( # Retrieve the specified ManagedElement
	if ($Credential) {
		Get-WMIObject -Computer $ComputerName -Credential $cred -Authentication 6 -Errorvariable err -Namespace Root\Intel_SCS_Framework -Class SCS_ManagedElement -Filter $filter
	} else {
		Get-WMIObject -Computer $ComputerName -Authentication 6 -Errorvariable err -Namespace Root\Intel_SCS_Framework -Class SCS_ManagedElement -Filter $filter
	}
) | Select -First 1 # If the filter returns multiple components, the cmdlet will run on only the first component

if ($SCS_ManagedElement -eq $null) {
	if($err.Count -gt 0) {$mg=$err[0];} else { $mg="";}
	Write-Error "Failed to load component $Component with error: $mg"
	Write-Host "Please check the event log for more details."
	exit 5
}

$method = "Configure"
$params = $SCS_ManagedElement.GetMethodParameters($method)
$params.Configuration = $profileText
try {
	$ret = $SCS_ManagedElement.InvokeMethod($method, $params, $null)	
} catch [Exception] {
	Write-Error "Failed to execute the $method command with message: $_.Exception.Message"
	Write-Host "Please check the event log for more details."
	exit $_.Exception.ErrorCode
}
$exitcode = $ret.ReturnValue

exit $exitcode
