<#
  .Notes
	Copyright (c) 2013-2014, Intel Corporation. All rights reserved.
  .Synopsis 
	Gets the events that currently exist for the specified solution component, or all solution components, and optionally clears the events list
  .Description
	Gets the events that currently exist for the specified solution component, or all solution components, and optionally clears the events list
  .Example
	.\SCS-GetEvents -ComputerName MyHost.Domain.com -Component 0b93a9cb-64b1-43ed-bc33-e5b0b8692694
  .Outputs
	The number of events
	
#>
[CmdletBinding(DefaultParameterSetName="Normal")]
Param(
	[Parameter(Position=0,Mandatory=$False, HelpMessage='Fully qualified component identifer of the solution component on which to perform the operation')][string]$Component = "",
	[Parameter(Mandatory=$False, HelpMessage='Address of the computer on which to perform the operation (default: localhost)')][string]$ComputerName = $env:ComputerName,
	[Parameter(Mandatory=$False, HelpMessage='Credentials to use for authenticating to the Solutions Framework (default: current user)')][Object]$Credential = $null,
	[Parameter(Mandatory=$False, HelpMessage='Clear the events retrieved', ParameterSetName='ClearEvents')][switch]$Clear,
	[Parameter(Mandatory=$False, HelpMessage='Count the number of events for each component', ParameterSetName='Count')][switch]$Count
)

$SCS_SolutionEvent_Class = $(
	if ($Credential) {
		Get-WMIObject -Computer $ComputerName -Credential $cred -Authentication 6 -Errorvariable err -Namespace Root\Intel_SCS_Framework -List SCS_SolutionEvent
	} else {
		Get-WMIObject -Computer $ComputerName -Authentication 6 -Errorvariable err -Namespace Root\Intel_SCS_Framework -List SCS_SolutionEvent
	}
) | Select -First 1

if ($SCS_SolutionEvent_Class -eq $null) {
	if($err.Count -gt 0) {$mg=$err[0];} else { $mg="";}
	Write-Error "Failed to access the Solutions Framework. $mg"	
	exit 1
}

if ($Count) { # Count events from each component
	$method = "CountEvents"
	$params = $SCS_SolutionEvent_Class.PSBase.GetMethodParameters($method)
	if ($Component) {
		$params.ComponentID = $Component
	}
	try {
		$ret = $SCS_SolutionEvent_Class.PSBase.InvokeMethod($method, $params, $null)
	} catch [Exception] {
		Write-Error "Failed to execute the $method command with message: $_.Exception.Message"
		Write-Host "Please check the event log for more details."
		exit $_.Exception.ErrorCode
	}
	$exitcode = $ret.ReturnValue
	if ($ret.ReturnValue -eq 0) {
		@{ Component = $_.ComponentID; Count = $params.Count }
	} else {
		@{ Component = $_.ComponentID; Error = $ret.ReturnValue }
	}
	exit $exitcode
}

$filter = $(
	if ($Component -ne "") {
		if ($ComponentID -Match '%') {
			"ComponentID LIKE ""$Component"""
		} else {
			"ComponentID = ""$Component"""
		}
	} else {
		""
	}
)

$(
    if ($Credential) {
        Get-WMIObject -Computer $ComputerName -Credential $cred -Authentication 6 -Errorvariable err -Namespace Root\Intel_SCS_Framework -Class SCS_SolutionEvent -Filter $filter
    } else {
        Get-WMIObject -Computer $ComputerName -Authentication 6 -Errorvariable err -Namespace Root\Intel_SCS_Framework -Class SCS_SolutionEvent -Filter $filter
    }
) | %{ 
	@{ 
		Component = $_.ComponentID
		ID = $_.EventID
		Timestamp = $_.Timestamp
		Severity = $_.Severity
		ErrorCode = $_.ErrorCode
		Operation = $_.Operation
		Details = $_.Details
	}
}
