<#
  .Notes
	Copyright (c) 2013-2014, Intel Corporation. All rights reserved.
  .Synopsis 
	Gets the Key Performance Indicators (KPI) data of the specified solution component, or all solution components, and optionally publishes the data to the RCS
  .Description
	Gets the Key Performance Indicators (KPI) data of the specified solution component, or all solution components, and optionally publishes the data to the RCS.
	Data can only be published to an RCS that is installed in database mode.
	Running this cmdlet on an RCS in non-database mode will return an error.
  .Example
	.\SCS-ReadKPI -ComputerName MyHost.Domain.com -Component 0b93a9cb-64b1-43ed-bc33-e5b0b8692694 -Publish rcs.domain.com -Reset
  .Outputs
	An XML document containing the solution KPI data as reported by the solution host plugin. For information about the format and content of this XML document, refer to the documentation supplied with the solution. If the data is published to the RCS, nothing is returned in the output of the cmdlet. 
	
#>
[CmdletBinding()]
Param(
	[Parameter(Position=0,Mandatory=$False, HelpMessage='Fully qualified component identifer of the solution component on which to perform the operation')][string]$Component = "",
	[Parameter(Mandatory=$False, HelpMessage='Address of the computer on which to perform the operation (default: localhost)')][string]$ComputerName = $env:ComputerName,
	[Parameter(Mandatory=$False, HelpMessage='Credentials to use for authenticating to the Solutions Framework (default: current user)')][Object]$Credential = $null,
    [Parameter(Mandatory=$False, HelpMessage='Address of the RCS to which to publish the KPI data (default: do not publish)')][string]$Publish = $null,
	[Parameter(Mandatory=$False, HelpMessage='Whether to reset the KPI counters after retrieving the KPI data')][switch]$Reset
)

$filter = $(
	if ($Component -ne "") {
		if ($ComponentID -Match '%') {
			"ComponentID LIKE ""$Component"""
		} else {
			"ComponentID = ""$Component"""
		}
	} else {
		""
	}
)

$SCS_ManagedElement = $( #Retrieve the specified ManagedElement
    if ($Credential) {
        Get-WMIObject -Computer $ComputerName -Credential $cred -Authentication 6 -Errorvariable err -Namespace Root\Intel_SCS_Framework -Class SCS_ManagedElement -Filter $filter
    } else {
        Get-WMIObject -Computer $ComputerName -Authentication 6 -Errorvariable err -Namespace Root\Intel_SCS_Framework -Class SCS_ManagedElement -Filter $filter
    }
) | Select -First 1 # If the filter returns multiple components, the cmdlet will run on only the first component

if ($SCS_ManagedElement -eq $null) {
	if($err.Count -gt 0) {$mg=$err[0];} else { $mg="";}
	Write-Error "Failed to load component $Component with error: $mg" 
	Write-Host "Please check the event log for more details."	
	exit 1
}

$method = $(
	if ($Reset) {
		"GetAndResetKPIs"
	} else {
		"GetKPIs"
	}
)

$params = $SCS_ManagedElement.GetMethodParameters($method)
if ($Publish) {
    $params.RCSAddress = $Publish
}

try {
	$ret = $SCS_ManagedElement.InvokeMethod($method, $params, $null)
	$exitcode = $ret.ReturnValue

	if (!$Publish) {
		$ret.KPIData
	}
} catch [Exception] {
	Write-Error "Failed to execute the $method command with message: $_.Exception.Message" 
	Write-Host "Please check the event log for more details."
	exit $_.Exception.ErrorCode
}
exit $exitcode
