<#
  .Notes
	Copyright (c) 2013-2014, Intel Corporation. All rights reserved.
  .Synopsis 
	Runs solution discovery of the specified solution component, or all solution components, and optionally publishes the data to the RCS
  .Description
	Runs solution discovery of the specified solution component, or all solution components, and optionally publishes the data to the RCS.
	Data can only be published to an RCS that is installed in database mode.
	Running this cmdlet on an RCS in non-database mode will return an error.
  .Example
	.\SCS-SolutionDiscovery -ComputerName MyHost.Domain.com -Component 0b93a9cb-64b1-43ed-bc33-e5b0b8692694
  .Outputs
	An XML document containing the solution discovery data. For information about the format and content of this XML document, refer to the documentation supplied with the solution. If the data is published to the RCS, nothing is returned in the output of the cmdlet. 
#>
[CmdletBinding()]
Param(
	[Parameter(Position=1)][string]$bad_param,
	[Parameter(Position=0,Mandatory=$False, HelpMessage='Fully qualified component identifer of the solution component on which to perform the operation (default: all components)')][string]$Component = "",
	[Parameter(Mandatory=$False, HelpMessage='Address of the computer on which to perform the operation (default: localhost)')][string]$ComputerName = $env:ComputerName,
	[Parameter(Mandatory=$False, HelpMessage='Credentials to use for authenticating to the Solutions Framework (default: current user)')][Object]$Credential = $null,
    [Parameter(Mandatory=$False, HelpMessage='Address of the RCS to which to publish the solution discovery data (default: do not publish)')][string]$Publish = $null
)

if ($bad_param) {
	Write-Host "Invalid positional parameter"
	exit 5
}

$filter = $(
	if ($Component -ne "") {
		if ($Component -Match '%') {
			"ComponentID LIKE ""$Component"""
		} else {
			"ComponentID = ""$Component"""
		}
	} else {
		""
	}
)

$SCS_ManagedElements = $( # Retrieve the specified ManagedElement
    if ($Credential) {
        Get-WMIObject -Computer $ComputerName -Credential $cred -Authentication 6 -Errorvariable err -Namespace Root\Intel_SCS_Framework -Class SCS_ManagedElement -Filter $filter
    } else {
        Get-WMIObject -Computer $ComputerName -Authentication 6 -Errorvariable err -Namespace Root\Intel_SCS_Framework -Class SCS_ManagedElement -Filter $filter
    }
) 

if ($SCS_ManagedElements -eq $null) {
	if($err.Count -gt 0) {$mg=$err[0];} else { $mg="";}
	Write-Error "Failed to load component $Component with error: $mg"
	Write-Host "Please check the event log for more details."
	exit 1
}

$method = "SolutionDiscovery"

foreach ($SCS_ManagedElement in $SCS_ManagedElements)
{
	$params = $SCS_ManagedElement.GetMethodParameters($method)
	if ($Publish) {
		$params.RCSAddress = $Publish
	}
	try {
		$ret = $SCS_ManagedElement.InvokeMethod($method, $params, $null)	
		$exitcode = $ret.ReturnValue
		
		if (!$Publish) {
			$ret.DiscoveryData
		}
	} catch [Exception] {
		Write-Error "Failed to execute the SolutionDiscovery command with message: $_.Exception.Message" 
		Write-Host "Please check the event log for more details."
  	     	exit $_.Exception.ErrorCode
	}
}

exit $exitcode 
