#!/bin/bash

# Copyright (c) 2012 Wind River Systems, Inc.
# The right to copy, distribute, modify, or otherwise
# make use of this software may be licensed only pursuant
# to the terms of an applicable Wind River license agreement.

# The following are return values
# 0: success
# 1: failure
# 2: smart error
# 3: xml_parse error
# 4: argument error
# 5: file not available
declare _SUCCESS=0
declare _FAIL=1
declare _SMART_ERR=2
declare _REPO_PARSER_ERR=3
declare _ARGUMENT_ERR=4
declare _FILE_NOT_EXIST=5

# The following are the package status
declare _INIT_STATUS=0
declare _NOT_INSTALLED=1
declare _REINSTALLABLE=2
declare _UPGRADABLE=3

declare outputStr
declare outputFile="/var/lib/wr-updt/.cmd_output"
declare outputFile_lock="/var/lib/wr-updt/.cmd_output.lock"

declare pid

declare fore_green="32;1m"
declare fore_red="31;1m"
declare fore_blue="34;1m"
declare color_off="\033[0m"
declare color_info="\033[${fore_blue}"
declare color_ok="\033[${fore_green}"
declare color_fail="\033[${fore_red}"

unlockFile()
{
    [[ -z "$1" ]] && exec_cmd "rm -rf $outputFile_lock" || exec_cmd "rm -rf $1"
}

lockFile()
{
    [[ -z "$1" ]] && exec_cmd "touch $outputFile_lock" || exec_cmd "touch $1"
}

waitForUnlock()
{
    [[ -z "$1" ]] && {
        while [[ -e ${outputFile_lock} ]]; do
            sleep 1
        done
    } || {
        while [[ -e "$1" ]]; do
            sleep 1
        done
    }
}

exec_cmd()
{
    args=("$@")
    eval "${args[@]} > /dev/null 2>&1"
}

exec_cmd_parser()
{
    waitForUnlock $default_updt_xml_lock
    lockFile $default_updt_xml_lock

    args=("$@")
    eval "${args[@]} > /dev/null 2>&1"
    ret=$?

    unlockFile $default_updt_xml_lock
    return $ret
}

exec_cmd_toStd()
{
    args=("$@")
    eval "${args[@]}"
}

exec_cmd_toStr()
{
    args=("$@")
    outputStr=`eval "${args[@]}"`
    ret=$?
    return $ret
}

exec_cmd_toStr_parser()
{
    waitForUnlock $default_updt_xml_lock
    lockFile $default_updt_xml_lock

    args=("$@")
    outputStr=`eval "${args[@]}"`
    ret=$?

    unlockFile $default_updt_xml_lock
    return $ret
}

exec_cmd_toFile()
{
    local ret=0
    unset pid

    while [[ -e ${outputFile_lock} ]]; do
        sleep 1
    done

    lockFile
    
    args=("$@")
    eval "${args[@]}  > ${outputFile} 2>&1 &"
    pid="$!"
    while [[ -d "/proc/$pid" ]]; do
        echo -n "."
        sleep 1
    done

    wait $!
    ret="$?"

    cp ${outputFile} ${outputFile}${pid}
    unlockFile

    return $ret
}

#Find the keywords in the file
#$1: filename
#$2: keyword
#$X: keyword
declare keyword_lineNum=0
matchKeywordInFile()
{
    unset keyword_lineNum

    local ret=0
    local fileName="$1"
    shift
    local keyWord=("$@")
    [[ -e $fileName ]] && {
        [[ -z "${keyWord[@]}" ]] && ret=$_ARGUMENT_ERR || {
            local item
            local grepStr
            local matchStr_lineText

            for item in "${keyWord[@]}"; do
                grepStr=${grepStr}" | grep \"${item}\""
            done
            exec_cmd_toStr "cat -n ${fileName}${grepStr}"
            [[ $? -eq 0 ]] && {
                matchStr_lineText=$outputStr
                keyword_lineNum=`echo $matchStr_lineText | awk '{print $1}'`
                ret=$_SUCCESS
            } || ret=$_FAIL
        }
    } || ret=$_FILE_NOT_EXIST
    return $ret
}

#Find the keywords in the string
#$1: string
#$2: keyword
#$X: keyword
matchKeywordInString()
{
    local str="$1"
    shift
    local keyword=("$@")

    local ret=0

    [[ -z "$str" ]] && ret=$_ARGUMENT_ERR || {
        local item
        local grepStr
        for item in "${keyword[@]}"; do
            grepStr=${grepStr}" |grep $item"
        done
        exec_cmd "echo $str ${grepStr}"
        [[ $? -eq 0 ]] && ret=$_SUCCESS || ret=$_FAIL
    }
    return $ret
}

usage()
{
    WR_UPDT=wr-updt
    echo
    echo "Usage: $WR_UPDT [commands]"
    echo "Commands List:
    sync
    sync --verbose
    sync --flag=[1-6]

    showrepo
    addrepo
    delrepo
    setrepo
    updtrepo

    download
    query
    install
    reinstall
    upgrade
    remove
"
    echo "Examples:
    $WR_UPDT sync                                         : do online update with remote repository server once
    $WR_UPDT sync --verbose                               : do online update with remote repository server once in verbose mode
    $WR_UPDT sync repo1                                   : do online update with specific repository repo1
    $WR_UPDT sync repo1 repo2 ...                         : do online update with specific repository repo1, repo2, ...
    $WR_UPDT sync repo1 repo2 ... --verbose               : do online update with specific repository repo1, repo2, ... in verbose mode
    $WR_UPDT sync --flag=3                                : do online update to install the packages specified by --flag
                                                          : by default, the flag is set to 1 (upgradable packages only)
                                                          : 1 means upgradable packages only; 2 means reinstallable packages only
                                                          : 3 means not_installed packages only; 4 means packages for flag 1 and flag 2
                                                          : 5 means packages for flag 1 and flag 3; 6 means packages for flag 1, flag 2 and flag 3.

    $WR_UPDT showrepo                                     : show the configured repository info
    $WR_UPDT showrepo repo1                               : show the repository info for repo1
    $WR_UPDT showrepo repo1 repo2 ...                     : show the repository info for repo1, repo2, ...
    $WR_UPDT addrepo repo_name baseurl=http://xxxx/xxx    : add a new repository
    $WR_UPDT delrepo repo1                                : delete existing repository repo1
    $WR_UPDT delrepo repo1 repo2 ...                      : delete existing repository repo1, repo2 ...
    $WR_UPDT setrepo repo_name baseurl=http://xxx/xxx     : configure the existing repository with new baseurl
    $WR_UPDT updtrepo                                     : update repodata for all the existing repositories
    $WR_UPDT updtrepo repo1                               : update repodata for repository repo1
    $WR_UPDT updtrepo repo1 repo2 ...                     : update repodata for repository repo1, repo2 ...

    $WR_UPDT download pkg1
    $WR_UPDT download pkg1 pkg2 ...                       : download packages from the configured repo
    $WR_UPDT query pkg1
    $WR_UPDT query pkg1 pkg2 ...                          : query packages to see if they are already installed
    $WR_UPDT install pkg1
    $WR_UPDT install pkg1 pkg2 ...                        : install packages
    $WR_UPDT reinstall pkg1
    $WR_UPDT reinstall pkg1 pkg2 ...                      : reinstall packages
    $WR_UPDT upgrade pkg1
    $WR_UPDT upgrade pkg1 pkg2 ...                        : upgrade packages if newer version of pkgs are available in the remote repo
    $WR_UPDT remove pkg1
    $WR_UPDT remove pkg1 pkg2 ...                         : remove installed packages
"
}
