#!/bin/bash

# Copyright (c) 2012 Wind River Systems, Inc.
# The right to copy, distribute, modify, or otherwise
# make use of this software may be licensed only pursuant
# to the terms of an applicable Wind River license agreement.

declare default_lib_path="/var/lib/wr-updt"
declare default_repo_xml_path="/var/lib/wr-updt/repo_xml"
declare default_updt_xml="/var/lib/wr-updt/repo_xml/updt.xml"
declare default_updt_xml_lock="/var/lib/wr-updt/repo_xml/.updt.xml_lock"
declare default_primary_xml="/var/lib/wr-updt/repo_xml/primary.xml"
declare default_smart_repo_path="/var/lib/smart/channels"
declare default_smart_lib_path="/var/lib/smart"

declare is_remote_repo_found=0

#show the normal repo info
get_repo_info()
{
    local repo=("$@")
    exec_cmd_toStd "$SMART channel --show ${repo[@]}"
}

#add rpm-md type repo server
add_repo_md()
{
    local ret=0
    [[ $# -lt 2 ]] && ret=$_ARGUMENT_ERR || {
        local repo_name="$1"
        local repo_url="$2"
        exec_cmd_toFile "$SMART channel --show $repo_name" > /dev/null 2>&1
        local tmpPID=$pid
        matchKeywordInFile ${outputFile}${tmpPID} "Channel" "not found"
        local line=$keyword_lineNum
        exec_cmd "rm -rf ${outputFile}${tmpPID}"
        [[ $line -eq 0 ]] && {
            echo -e "${color_info}warning: $repo_name already exists, please choose a different name!${color_off}"
            return $ret 
        }

        exec_cmd 'echo $repo_url | grep "baseurl=" | grep "http:\|ftp:\|file:"'
        [[ $? -eq 0 ]] && {
            exec_cmd_toFile "$SMART channel --add $repo_name $repo_url type=rpm-md -y" > /dev/null 2>&1
            tmpPID=$pid
            [[ $? -eq 0 ]] && ret=$_SUCCESS || ret=$_FAIL
        } || ret=$_ARGUMENT_ERR
    }

    if [ $ret -eq $_ARGUMENT_ERR ]; then
        echo -e "${color_fail}argument error${color_off}"
    elif [ $ret -eq $_FAIL ]; then
        echo -e "${color_fail}Failed to add repo!${color_off}"
        cat ${outputFile}${tmpPID}
    else
        echo -e "${color_info}rpm repository $repo_name is added successfully!${color_off}"
    fi
    exec_cmd "rm -rf ${outputFile}${tmpPID}"
    
    return $ret
}

#update the repo property
set_repo_property()
{
    local ret=0

    [[ $# -gt 1 ]] && {
        local repo_name=$1
        shift
        local newValues=("$@")
        exec_cmd "$SMART channel --set $repo_name $newValues"
        [[ $? -eq 0 ]] && {
            echo $newValues | grep "baseurl=" > /dev/null 2>&1
            [[ $? -eq 0 ]] && baseurl=`echo $newValues | awk -F"=" '{print $2}'`
            exec_cmd_parser "$REPO_PARSER -r$repo_name -b$baseurl -B"
            [[ $? -eq 0 ]] && ret=$_SUCCESS || ret=$_FAIL
        }|| ret=$_FAIL
    } || ret=$_ARGUMENT_ERR

    return $ret
}

#delete a repo
del_repo()
{
    unset passList
    unset failList

    local ret=0
    local repo_list=("$@")

    [[ -z "${repo_list[@]}" ]] && ret=$_FAIL || {
        local item
        for item in ${repo_list[@]}; do
            exec_cmd "$SMART channel --remove $item -y"
            [[ $? -eq 0 ]] && passList="${passList}${item} " || failList="${failList}${item} "
            exec_cmd_parser "$REPO_PARSER -r$item -D"
        done
        [[ -z "$failList" ]] && ret=$_SUCCESS || ret=$_FAIL
    }

    return $ret
}

declare -a RepoList
declare -a RepoBaseUrlList
getRemoteRepos()
{
    unset RepoList
    unset RepoBaseUrlList
    unset is_remote_repo_found

    local ret=0
    local i=0
    local repo_info

    local repo_list=("$@")
    exec_cmd_toFile "$SMART channel --show ${repo_list[@]}"
    local tmpPID=$pid
    [[ -z "${repo_list[@]}" ]] && {
        while read line; do
            repo_info="${repo_info}${line} ###"
            [[ -z "$line" ]] && {
                echo $repo_info | grep "baseurl =" | grep "rpm-md"  > /dev/null 2>&1
                [[ $? -eq 0 ]] && {
                    tmpRepoName=`echo $repo_info | sed -e 's/\[//g' -e 's/\]//g' | awk -F"###" '{print $1}'`
                    tmpRepoBaseUrl=`echo $repo_info | sed -e 's/###/=/g' -e 's/^.*baseurl/^baseurl/g' | awk -F"=" '{print $2}'`
                    tmpIP=`echo ${tmpRepoBaseUrl} | awk -F"/" '{print $3}'`
                    ping $tmpIP -c 5 | grep "100% packet loss" > /dev/null 2>&1
                    [[ $? -ne 0 ]] && {
                        RepoList[$i]=$tmpRepoName
                        RepoBaseUrlList[$i]=$tmpRepoBaseUrl
                        i=`expr $i + 1`
                    } || exec_cmd_parser "$REPO_PARSER -r$tmpRepoName -D" # else, remove the repo
                }
                unset repo_info
            } 
        done < ${outputFile}${tmpPID}
    } || {
        while read line; do
            repo_info="${repo_info}${line} ###"
            [[ -z "$line" ]] && {
                for repo in ${repo_list[@]}; do
                    echo $repo_info | grep "\[${repo}\]" | grep "rpm-md" | grep "baseurl =" > /dev/null 2>&1
                    [[ $? -eq 0 ]] && {
                        tmpRepoBaseUrl=`echo $repo_info | sed -e 's/###/=/g' -e 's/^.*baseurl/^baseurl/g' | awk -F"=" '{print $2}'`
                        tmpIP=`echo ${tmpRepoBaseUrl} | awk -F"/" '{print $3}'`
                        ping $tmpIP -c 5 | grep "100% packet loss" > /dev/null 2>&1
                        [[ $? -ne 0 ]] && {
                            RepoList[$i]=$repo
                            RepoBaseUrlList[$i]=$tmpRepoBaseUrl
                            i=`expr $i + 1`
                        } || exec_cmd_parser "$REPO_PARSER -r$repo -D" # else, remove the repo
                        break
                    }
                done
                unset repo_info
            }
        done < ${outputFile}${tmpPID}
    }

    exec_cmd "rm ${outputFile}${tmpPID}"
    [[ ! -z "${RepoList[@]}" ]] && {
        [[ ${#RepoList[@]} -eq ${#RepoBaseUrlList[@]} ]] && {
            ret=0
            i=0 
            j=1
            is_remote_repo_found=1
            echo -e "\n${color_info}found remote repo link ${color_off}...${color_ok} done${color_off}"
            for item in ${RepoList[@]}; do
                [[ $verbose -ne 0 ]] && echo -e "$j ${item}:${RepoBaseUrlList[$i]} "
                i=`expr $i + 1`
                j=`expr $j + 1`
            done
        }
    } || {
        is_remote_repo_found=0
        ret=$_FAIL
        echo -e "\n${color_fail}no remote repos are found ${color_off}...${color_fail} failed${color_off}"
    }

    return $ret
}

#package status
#0: init status, just the available package in the repo
#1: not installed, packages detected by smart update
#2: after installation, pkgs will be marked as reinstallable
#3: upgradable packages, detected by smart newer, then those not installed packages will be new packages

#update the repodata with a specific repo
updt_repo()
{
    local repo_list=("$@")
    local repo
    local not_installed_list

    #clean smart cache first
    exec_cmd "rm -rf $default_smart_lib_path/cache"
    exec_cmd "rm -rf $default_smart_repo_path/*"

    echo -n -e "${color_info}start syncing repodata ${color_off}"
    exec_cmd_toFile "$SMART update ${repo_list[@]}" # detects not installed packages(new packages and upgradable packages)
    local tmpPID=$pid
    local ret=$?
    [[ $ret -eq 0 ]] && {
        local g_repo_list
        local g_repo_baseurl_list
        getRemoteRepos "${repo_list[@]}"
        [[ $is_remote_repo_found -eq 1 ]] && {
            g_repo_list=(${RepoList[@]})
            g_repo_baseurl_list=(${RepoBaseUrlList[@]})
            getUpdtList ${outputFile}${tmpPID}
            [[ ! -z "${pkg_entries[@]}" ]] && not_installed_list=${pkg_entries[@]} # will update the not installed list in updt.xml later

            [[ ! -d $default_repo_xml_path ]] && mkdir $default_repo_xml_path > /dev/null 2>&1
            
            i=0
            for repo in ${g_repo_list[@]}; do
                exec_cmd_parser "$REPO_PARSER -r$repo -Q"
                [[ $? -eq 0 ]] && exec_cmd_parser "$REPO_PARSER -r$repo -D"
                primary_xml=`ls $default_smart_repo_path | grep $repo%% | grep _repodata_primary.xml$`
                cp ${default_smart_repo_path}/$primary_xml $default_primary_xml > /dev/null 2>&1
                exec_cmd_parser "$REPO_PARSER -r${repo} -X"
                exec_cmd_parser "$REPO_PARSER -r${repo} -b${g_repo_baseurl_list[$i]} -B" # set repo baseurl in updt.xml
                i=`expr $i + 1`                
            done

            [[ ! -z "${not_installed_list[@]}" ]] && {
                local pkg # if the package can be found in multiple repos, what to do?
                for pkg in ${not_installed_list[@]}; do
                    tmp_pkg=`echo $pkg | sed 's/@/./'`
                    echo $tmp_pkg | grep "^${SYS_RPM_PKG}-[0-9]*" > /dev/null 2>&1
                    [[ $? -eq 0 ]] && sys_rpm_not_installed=1
                    echo $tmp_pkg | grep ":" > /dev/null 2>&1
                    [[ $? -eq 0 ]] && tmp_pkg=`echo $tmp_pkg | sed 's/-[0-9]:/-/g'`
                    set_to_not_installed ${tmp_pkg}
                done
            }

            set_to_upgradable

            [[ $sys_rpm_upgradable -ne 1 ]] && {
                exec_cmd_toFile "$SMART query --installed ${SYS_RPM_PKG}" > /dev/null 2>&1
                tmp_pid=$pid
                cat ${outputFile}${tmp_pid} | grep "^${SYS_RPM_PKG}-[0-9]*" > /dev/null 2>&1
                [[ $? -ne 0 ]] && {
                    exec_cmd_toStr_parser "$REPO_PARSER -f$_NOT_INSTALLED -L"
                    pkgs=$outputStr
                    pkgs=(`echo $pkgs | sed -e 's/\\n/ /g'`)
                    for pkg in ${pkgs[@]}; do
                        echo $pkg | awk -F: '{print $3}' | grep "^${SYS_RPM_PKG}-[0-9]*" > /dev/null 2>&1
                        [[ $? -eq 0 ]] && {
                            sys_rpm_not_installed=1
                            break
                        }
                    done
                } || sys_rpm_not_installed=0
                exec_cmd "rm -rf ${outputFile}${tmp_pid}"
            }

        } || ret=$_FAIL

    } || ret=$_FAIL

    [[ $verbose -ne 0 ]] && cat ${outputFile}${tmpPID} | sed -e '/^$/d'
    exec_cmd "rm -rf ${outputFile}${tmpPID}"
    echo -e "${color_info}finish syncing repodata${color_off}"
    return $ret
}

declare upgradable_list # newer versions of the already installed packages are available
declare reinstallable_list # already installed packages with the same version but minor update are available
declare not_installed_list # new and not installed packages are available

# by setting flag_OnlineUpdt to 1-6, you can do the sync with different kind of package combs
# by default, set it to 1, to install the upgrades only

#repo_sync with specific repo server
repo_sync()
{
    local repo_list=("$@")
    local ret=0

    unset not_installed_list
    unset reinstallable_list
    unset upgradable_list

    pushd $default_repo_xml_path > /dev/null 2>&1

    echo -e "${color_info}start online updates${color_off}";
    updt_repo ${repo_list[@]} #repos are updated and available updates are saved into updt_table
    [[ $is_remote_repo_found -eq 0 ]] && {
        popd $default_repo_xml_path > /dev/null 2>&1
        echo -e "${color_info}finished online updates!${color_off}"
        return $_FAIL
    }
    
    local g_repo_list=(${RepoList[@]})

    [[ -e $default_updt_xml ]] && {
        [[ $flag_OnlineUpdt -eq 3 || $flag_OnlineUpdt -eq 5 || $flag_OnlineUpdt -eq 6 ]] && {
            for repo in ${g_repo_list[@]}; do
                exec_cmd_toStr_parser "$REPO_PARSER -r$repo -f$_NOT_INSTALLED -L"
                not_installed_list=${not_installed_list}" "$outputStr
            done
            lines=(`echo $not_installed_list | sed 's/\\n/ /g'`)
            not_installed_list=(${lines[@]})
            [[ ! -z "${not_installed_list[@]}" ]] && {
                echo -e "${color_info}get the new package list ${color_off}...${color_ok} done${color_off}"
                i=1
                for line in ${lines[@]}; do
                    echo -e "$i $line"| awk -F":" '{print $1":"$3}'
                    i=`expr $i + 1`
                done
            } || echo -e "${color_info}no interesting new packages available ${color_off}...${color_ok} done${color_off}"
        }

        [[ $flag_OnlineUpdt -eq 2 || $flag_OnlineUpdt -eq 4 || $flag_OnlineUpdt -eq 6 ]] && {
            for repo in ${g_repo_list[@]}; do
                exec_cmd_toStr_parser "$REPO_PARSER -r$repo -f$_REINSTALLABLE -L"
                reinstallable_list=${reinstallable_list}" "$outputStr
            done
            lines=(`echo $reinstallable_list | sed 's/\\n/ /g'`)
            reinstallable_list=(${lines[@]})
            [[ ! -z "${reinstallable_list[@]}" ]] && {
                echo -e "${color_info}get the reinstallable package list ${color_off}...${color_ok} done${color_off}"
                i=1
                for line in ${lines[@]}; do
                    echo -e "$i $line" | awk -F":" '{print $1":"$3}'
                    i=`expr $i + 1`
                done
            } || echo -e "${color_info}no interesting reinstallable packages available ${color_off}...${color_ok} done${color_off}"
        }

        [[ $flag_OnlineUpdt -eq 1 || $flag_OnlineUpdt -eq 4 || $flag_OnlineUpdt -eq 5 || $flag_OnlineUpdt -eq 6 ]] && {
            for repo in ${g_repo_list[@]}; do
                exec_cmd_toStr_parser "$REPO_PARSER -r$repo -f$_UPGRADABLE -L"
                upgradable_list=${upgradable_list}" "$outputStr
            done
            lines=(`echo $upgradable_list | sed 's/\\n/ /g'`)
            upgradable_list=(${lines[@]})
            [[ ! -z "${upgradable_list[@]}" ]] && {
                echo -e "${color_info}get the upgradable package list ${color_off}...${color_ok} done${color_off}"
                i=1
                for line in ${lines[@]}; do
                    echo -e "$i $line" | awk -F":" '{print $1":"$3}'
                    i=`expr $i + 1`
                done
            } || echo -e "${color_info}no interesting upgradable packages available ${color_off}...${color_ok} done${color_off}"
        }
    }
    # show the available pkg list in the repo
    # updatable_list=`repo_parser -N `
    #[ ! -z "$updatable_list" ]] && echo "updatable packages: $updatable_list"

    # force to install system rpm package if not installed
    install_system_rpm

    #0: sync repo by default 
    #1: install the upgradable packages
    #2: install the reinstallable packages
    #3: install the not installed packages
    #4: install 1 + 2 (upgradable and reinstallable)
    #5: install 1 + 3 (upgradable and not installed)
    #6: install 1 + 2 + 3 (all pkgs in the repo)
    case $flag_OnlineUpdt in
        1)
            [[ ! -z "${upgradable_list[@]}" ]] && {
                pkgs_upgrade ${upgradable_list[@]}
            }
            ;;
        2)
            [[ ! -z "${reinstallable_list[@]}" ]] && {
                pkgs_reinstall ${reinstallable_list[@]}
            }
            ;;
        3)
            [[ ! -z "${not_installed_list[@]}" ]] && {
                pkgs_install ${not_installed_list[@]}
            }
            ;;
        4)
            [[ ! -z "${reinstallable_list[@]}" ]] && {
                pkgs_reinstall ${reinstallable_list[@]}
            }
            [[ ! -z "${upgradable_list[@]}" ]] && {
                pkgs_upgrade ${upgradable_list[@]}
            }
            ;;
        5)
            [[ ! -z "${not_installed_list[@]}" ]] && {
                pkgs_install ${not_installed_list[@]}
            }
            [[ ! -z "${upgradable_list[@]}" ]] && {
                pkgs_upgrade ${upgradable_list[@]}
            }
            ;;
        6)
            [[ ! -z "${not_installed_list[@]}" ]] && {
                pkgs_install ${not_installed_list[@]}
            }
            [[ ! -z "${reinstallable_list[@]}" ]] && {
                pkgs_reinstall ${reinstallable_list[@]}
            }
            [[ ! -z "${upgradable_list[@]}" ]] && {
                pkgs_upgrade ${upgradable_list[@]}
            }
            ;;
        *);;
    esac

    popd $default_repo_xml_path > /dev/null 2>&1
    echo -e "${color_info}finished online updates!${color_off}"
}
