#!/bin/bash
#===============================================================================
# Copyright (c) 2014-2015 Wind River Systems, Inc.
# The right to copy, distribute, modify, or otherwise
# make use of this software may be licensed only pursuant
# to the terms of an applicable Wind River license agreement.
#
#===============================================================================
WRA_UPTIME=180
WRA_SYS_VERSION="/etc/sys-version.txt"
WRA_TRIGGER_TIMER_FLAG="/var/wra/otaflag/timer_trigger"
WRA_BOOTONCE_FLAG="/var/wra/otaflag/bootonce"
WRA_ROLLBACK_FLAG="/var/wra/otaflag/rollback_enable"
WRA_ALARM="/var/wra/otaflag/alarm"

# find FAT dev
function find_FAT_dev(){
	# find the mount point of /
	ROOTDEV=`df / | tail -1 | awk '{ print $1 }'`
	LV_INFO=`sudo lvdisplay ${ROOTDEV} -c 2> /dev/null`
	if [ ! -z $LV_INFO ]; then
		# find the volume group name
		IFS=':' read -a INFO <<< ${LV_INFO}
		ROOTDEV_VG_NAME=${INFO[1]}
		# use pvdisplay to print all available volume and save the names
		DEVS=`pvdisplay -m | grep "PV Name" | awk '{print $NF}'`
		# DEVS="/dev/sdb2 /dev/sda2"
		for i in $DEVS ; do
			VG_NAME=`pvdisplay $i -c | tail -1 | awk -F ':' '{print $2}'`
			if [ ${VG_NAME} == ${ROOTDEV_VG_NAME} ] ; then
				echo "${i%?}1"
			fi
		done
	else
		# non LVM kernel upgrade
		# replace the last digit with 1, ie /dev/mmcblk0p1, /dev/sda1
		echo "${ROOTDEV%?}1"
	fi
}

# clear bootonce in FAT
function clear_bootonce_in_FAT(){
	FAT_DEV=$(find_FAT_dev)
	if [ -n "${FAT_DEV}" ] ; then
		temp_dir="$(sudo mktemp -d -p ./)"
		sudo mount ${FAT_DEV} $temp_dir
		rm -rf ${temp_dir}/bootonce
		sudo umount $temp_dir
		rm -rf $temp_dir
	fi
}

# copy bootonce kernel, initramfs to FAT partition
function copy_new_kernel(){
	# mount the boot directory
	echo "Found boot partition in $1"
	echo "Mount $1"
	temp_dir="$(sudo mktemp -d -p ./)"
	sudo mount $1 $temp_dir

	# for u-boot, remove the bootonce flag in the FAT
	if [ -e ${temp_dir}/bootonce ] ; then
		sudo rm -rf ${temp_dir}/bootonce
	fi
	# rename uImage
	if [ -e ${temp_dir}/uImage-bootonce ] ; then
		sudo mv ${temp_dir}/uImage-bootonce ${temp_dir}/uImage
	fi
	# rename zImage
	if [ -e ${temp_dir}/zImage-bootonce ] ; then
		sudo mv ${temp_dir}/zImage-bootonce ${temp_dir}/zImage
	fi
	# rename bzImage
	if [ -e ${temp_dir}/bzImage-bootonce ] ; then
		sudo mv ${temp_dir}/bzImage-bootonce ${temp_dir}/bzImage
	fi
	# rename bzImage-bootonce.auth to bzImage.auth
	if [ -e ${temp_dir}/bzImage-bootonce.auth ] ; then
		sudo mv ${temp_dir}/bzImage-bootonce.auth ${temp_dir}/bzImage.auth
	fi
	# rename idp-initramfs.img
	if [ -e ${temp_dir}/idp-initramfs-bootonce.img ] ; then
		sudo mv ${temp_dir}/idp-initramfs-bootonce.img ${temp_dir}/idp-initramfs.img
	fi
	# rename idp-initramfs.img.auth
	if [ -e ${temp_dir}/idp-initramfs-bootonce.img.auth ] ; then
		sudo mv ${temp_dir}/idp-initramfs-bootonce.img.auth ${temp_dir}/idp-initramfs.img.auth
	fi
	# umount both directory
	sudo umount $temp_dir
	rm -rf $temp_dir

	# bootonce kernel successfully bootup, send message to info cloud
	echo "Bootonce kernel successfully bootup, replaced the old kernel with the new one." |  su -c "wra-util --sendalarm \"Updated with New Kernel\"" wra
}

# do not run more than one instance of this wr-iot-watchdog-test.sh because copying kernel image
# may keep longer time and watchdog will expired again and run this script
[ "$(pgrep -fn $0)" -ne "$(pgrep -fo $0)" ] && exit 0

# if WRA_TRIGGER_TIMER_FLAG exists then check for WRA_UPTIME
if [ -e ${WRA_TRIGGER_TIMER_FLAG} ] ; then
	UPTIME=`cat /proc/uptime | awk '{print $1}'`
	if (( ${UPTIME%.*} > ${WRA_UPTIME} )) ; then 
		# WRA_TRIGGER_TIMER_FLAG exists and it is longer then WRA_UPTIME seconds
		# create an alarm file to be send to cloud during next bootup
		echo -n "Agent connection failed after software update, rootfs will " >> ${WRA_ALARM}
		echo -n "rollback to previous snapshot. " >> ${WRA_ALARM}
		echo -n "Failed software version is " >> ${WRA_ALARM}
		cat ${WRA_SYS_VERSION} >> ${WRA_ALARM}
		echo -n ". " >> ${WRA_ALARM}

		# if the rootfs does not have sanpshot, do not reboot since it cannot rollback
		ROOTDEV=`df / | tail -1 | awk '{ print $1 }'`
		LV_INFO=`sudo lvdisplay ${ROOTDEV}_snapshot`
		if [ -z ${LV_INFO} ] ; then
			echo -n "Snapshot not available, cannot rollback." >> ${WRA_ALARM}
			# stop watchdog
			clear_bootonce_in_FAT
			systemctl stop wr-iot-watchdog
			exit 0
		fi

		# set the WRA_ROLLBACK_FLAG so that in initramfs bootup it will trigger a rollback
		clear_bootonce_in_FAT
		touch ${WRA_ROLLBACK_FLAG}
		exit -1
	fi
else
	# WRA_TRIGGER_TIMER_FLAG does not exist, everything should be ok
	# if the alarm file exists then send it to the cloud
	if [ -e ${WRA_ALARM} ] ; then
		# append the current WRA_SYS_VERSION to the alarm
		echo -n "Current software version is " >> ${WRA_ALARM}
		cat ${WRA_SYS_VERSION} >> ${WRA_ALARM}
		echo -n ". " >> ${WRA_ALARM}
		# send to cloud
		cat ${WRA_ALARM} | su -c "wra-util --sendalarm \"Rollback Occurred\"" wra
		# delete the alarm in the device
		mv ${WRA_ALARM} ${WRA_ALARM}.sent
	fi

	# stop the watchdog
	echo "Stopping wr-iot-watchdog service because target is up longer than"
	echo "${WRA_UPTIME} seconds and ${WRA_TRIGGER_TIMER_FLAG} does not exist,"
	echo "remove all rollback flags."

	# clean up rollback flag
	if [ -e ${WRA_ROLLBACK_FLAG} ] ; then
		rm ${WRA_ROLLBACK_FLAG}
	fi
	
	# if cmdline have bootonce and we reach here. It is a bootonced kernel and the bootonce flag is enabled
	# then copy bzImage-bootonce to bzImage
	CHECK_BOOTONCE=`cat /proc/cmdline | grep -o bootonce`
	if [ "${CHECK_BOOTONCE}" = "bootonce" -a -e ${WRA_BOOTONCE_FLAG} ] ; then
		# find the mount point of /mnt and the size of the LV
		FAT_DEV=$(find_FAT_dev)
		if [ -n "${FAT_DEV}" ] ; then
				echo "Found FAT partition in ${FAT_DEV}"
				echo "Copying new kernel to ${FAT_DEV}"
				copy_new_kernel ${FAT_DEV}
		else
				echo "Failed to find FAT partition"
				echo "Failed to find FAT parition,  cannot copy new kernel image." |  su -c "wra-util --sendalarm \"Kernel upgrade failed\"" wra
		fi
	fi

	# clean up bootonce flag
	if [ -e ${WRA_BOOTONCE_FLAG} ] ; then
		rm ${WRA_BOOTONCE_FLAG}
	fi

	# clear bootonce flag in FAT
	clear_bootonce_in_FAT

	# stop watchdog
	systemctl stop wr-iot-watchdog
fi
exit 0
