#!/usr/bin/python

# Created by: Dave Bush 
# Creation date: June 1, 2015
# Copyright (c) 2015 Wind River Systems, Inc.

"""
Wind River Agent OTA Update Script
================================================

This script updates the system from an OTA update package 

PRE-REQUISITES:
    --------------
     - Python 2.6 or 2.7

"""

import sys
import json
import os
import subprocess
import re
import time
import os.path
import tempfile
from pprint import pprint
from subprocess import call

# Preview mode
PREVIEW = 0

mnt = 0 
boot_mnt = 0
json_fn = 'idp-update.json'
log_fn = '/var/wra/files/default/wr_install_updates.log'
reboot_delay_fn = '/var/wra/files/default/reboot_delay'
found_kernel_image = 0
found_agent_image = 0
reboot = 0
reboot_delay = 0
reboot_agent = 0
rm_agent_name = ''

# Script return codes
RPM_REMOVE_PKG_ERR = 10
RPM_ADD_PKG_ERR = 20
RPM_NOT_SIGNED_ERR = 25
MNT_UP_PKG_ERR = 30
UMNT_UP_PKG_ERR = 40
MNT_BOOT_PART_ERR = 50 
UMNT_BOOT_PART_ERR = 60
RM_MNT_PNT_ERR = 70
INVALID_JSON_DATA = 80
UNSPECIFIED_ERR = 1
SUCCESS = 0
SUCCESS_AND_REBOOT = 100

# write to logfile and post data to the cloud
def log_file_write(msg):
  # open/write/close the file so that "rpm -v"  can append to the same file
  log_file=open(log_fn,'a+')
  log_file.write(msg)
  log_file.close()
  # send to cloud without LF character
  msg = msg.replace('\n', ' ');
  cmd = 'echo ' + '\"' + msg + '\"' + ' | wra-util --senddata wr_install_updates.log '
  exec_shell_cmd(cmd)
  return

def exec_shell_cmd1(cmd):
  try:
    p = subprocess.Popen(cmd.split(), stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    stdout, stderr = p.communicate()
    exit_code = p.wait()

    if stdout:
      log_file_write("INFO: {0}\n".format(stdout))
      out = stdout

    if stderr:
      log_file_write("ERROR: {0}\n".format(stderr))

    if exit_code:
      return (1,stdout,stderr)

    return (0,stdout,stderr)

  except OSError as e:
    log_file_write("ERROR: {0}\n".format(e))
    return (1,stdout,stderr)

def exec_shell_cmd(cmd):
  try:
    if os.system(cmd):
      log_file_write("INFO: Shell command returned: {0}\n".format(cmd))
      return 1
  except OSError as e:
    log_file_write("ERROR: OS exception after executing shell command: {0}\n".format(cmd))
    return 1  

 
  return 0 

def list_directory(directory):
  dirs = os.listdir(directory) 
  for file in dirs:
    print file

def mount_update_package(pkg):
  mount_point = tempfile.mkdtemp(prefix='mnt')
  log_file_write("INFO: Mounting {0} at {1}\n".format(pkg,mount_point))
  cmd = 'sudo mount ' + pkg + ' ' + mount_point 
  
  if exec_shell_cmd(cmd):
    cleanup_and_exit(MNT_UP_PKG_ERR)

  return mount_point

def unmount_update_package():

  # Unmount the update package
  cmd = 'sudo umount ' + mnt

  if exec_shell_cmd(cmd):
    log_file_write("INFO: Exiting with return code {0}\n".format(UMNT_UP_PKG_ERR))
    sys.exit(UMNT_UP_PKG_ERR)

  # Remove the mount point
  try:
    rc = os.rmdir(mnt)
  except OSError, msg:
    log_file_write("INFO: Exiting with return code {0}\n".format(RM_MNT_PNT_ERR))
    sys.exit(RM_MNT_PNT_ERR)
 
def mount_boot_partition(partition):
  mount_point = tempfile.mkdtemp(prefix='boot')
  log_file_write("INFO: Mounting {0} at {1}\n".format(partition,mount_point))
  cmd = 'sudo mount ' + partition + ' ' + mount_point 
  
  if exec_shell_cmd(cmd):
    cleanup_and_exit(1)

  return mount_point

def unmount_boot_partition(mount):
  cmd = 'sudo umount ' + mount

  if exec_shell_cmd(cmd):
    log_file_write("INFO: Exiting with return code {0}\n".format(1))
    sys.exit(1)

  # Remove the mount point
  try:
    rc = os.rmdir(mount)
  except OSError, msg:
    log_file_write("INFO: Exiting with return code {0}\n".format(1))
    sys.exit(1)

def install_kernel():
  cmd = 'sudo pvdisplay -c'
  rc,stdout,stderr = exec_shell_cmd1(cmd)

  # The command failed
  if rc !=0:
    log_file_write("ERROR: Failed to discover boot partition\n")
    return 1

  # Compute the boot partition 
  if stdout:
    tokens = stdout.split(':')
    match = re.match(r'(.*).$', tokens[0].strip())

    # We found the rootfs partition
    if match:
      boot_partition = match.groups()[0] + '1' 
      log_file_write("INFO: Boot partition is {0}\n".format(boot_partition))

      # Mount the boot partition
      mount_point = mount_boot_partition(boot_partition)

      # Copy uImage to boot partition as zImage-bootonce
      if os.path.exists(os.path.join(mount_point,'zImage')):
        zImage_name = os.path.realpath("/boot/zImage");
        log_file_write("INFO: copy zImage to {}\n".format(boot_partition))
        cmd = 'sudo cp ' + zImage_name + ' ' + os.path.join(mount_point,'zImage-bootonce')

        if exec_shell_cmd(cmd):
          log_file_write("ERROR: failed to copy {0} to {1}\n".format(zImage_name, mount_point))
          rc = 1

      # Copy uImage to boot partition as uImage-bootonce
      if os.path.exists(os.path.join(mount_point,'uImage')):
        uImage_name = os.path.realpath("/boot/uImage");
        log_file_write("INFO: copy uImage to {}\n".format(boot_partition))
        cmd = 'sudo cp ' + uImage_name + ' ' + os.path.join(mount_point,'uImage-bootonce')

        if exec_shell_cmd(cmd):
          log_file_write("ERROR: failed to copy {0} to {1}\n".format(uImage_name, mount_point))
          rc = 1

        # create a directory-flag in FAT partition for u-boot to fatls
        cmd = 'sudo mkdir ' +  os.path.join(mount_point,'bootonce')
        exec_shell_cmd(cmd)

      # Copy bzImage to boot partition as bzImage-bootonce
      if os.path.exists(os.path.join(mount_point,'bzImage')):
        bzImage_name = os.path.realpath("/boot/bzImage");
        log_file_write("INFO: copy bzImage to {}\n".format(boot_partition))
        cmd = 'sudo cp ' + bzImage_name + ' ' + os.path.join(mount_point,'bzImage-bootonce')

        if exec_shell_cmd(cmd):
          log_file_write("ERROR: failed to copy {0} to {1}\n".format(bzImage_name, mount_point))
          rc = 1

        # Copy bzImage.auth to boot partition as bzImage-bootonce.auth
        bzImage_auth_name = bzImage_name + '.auth'
        if os.path.exists(bzImage_auth_name):
          log_file_write("INFO: copy {0} to {1}\n".format(bzImage_auth_name, boot_partition))
          cmd = 'sudo cp ' +  bzImage_auth_name + ' ' + os.path.join(mount_point,'bzImage-bootonce.auth')

          if exec_shell_cmd(cmd):
            log_file_write("ERROR: failed to copy {0} to {1}\n".format(bzImage_auth_name, mount_point))
            rc = 1

      # Copy idp-initramfs.img to the boot parition as idp-initramfs-bootonce.img
      #if os.path.exists(os.path.join(mount_point,'idp-initramfs.img')):
      if os.path.exists('/boot/idp-initramfs.img'):
        log_file_write("INFO: copy idp-initramfs to {}\n".format(boot_partition))
        cmd = 'sudo cp /boot/idp-initramfs.img ' + os.path.join(mount_point,'idp-initramfs-bootonce.img')

        if exec_shell_cmd(cmd):
          log_file_write("ERROR: failed to copy idp-initramfs.img to {0}\n".format(mount_point))
          rc = 1

      # Copy idp-initramfs.img.auth to the boot parition as idp-initramfs-bootonce.img.auth
      #if os.path.exists(os.path.join(mount_point,'idp-initramfs.img.auth')):
      if os.path.exists('/boot/idp-initramfs.img.auth'):
        log_file_write("INFO: copy idp-initramfs.img.auth to {}\n".format(boot_partition))
        cmd = 'sudo cp /boot/idp-initramfs.img.auth ' + os.path.join(mount_point,'idp-initramfs-bootonce.img.auth')

        if exec_shell_cmd(cmd):
          log_file_write("ERROR: failed to copy idp-initramfs.img to {0}\n".format(mount_point))
          rc = 1

      # check grub's savedefault to entry 1 which is the bootonce entry
      # find the name of /EFI/BOOT/*.default which is difference for each arch
      if os.path.isdir(os.path.join(mount_point,'EFI/BOOT/')):
        for filename in os.listdir(os.path.join(mount_point,'EFI/BOOT/')):
          if filename.endswith(".default"):
            # sudo sh -c "echo -n 1 > EFI/BOOT/BOOTX64.default"
            cmd = 'sudo sh -c \"echo -n 1 > ' + os.path.join(mount_point,'EFI/BOOT/') + filename +'\"'
            log_file_write("INFO: Setting grub's entry file  " + os.path.join(mount_point,'EFI/BOOT/') + filename + "\n")
            if exec_shell_cmd(cmd):
              log_file_write("ERROR: Failed to set grub's entry file  " + os.path.join(mount_point,'EFI/BOOT/') + filename + "\n")
              rc = 1

      # set the bootonce flag
      exec_shell_cmd("touch /var/wra/otaflag/bootonce")

      # Unmount the boot partition
      unmount_boot_partition(mount_point)

    else:
      log_file_write("ERROR: Could not find boot partition.")
      return 1

  return rc

def cleanup_and_exit(rc):
  
  if mnt:
    json_data.close()
    unmount_update_package()
  log_file_write("INFO: Exiting with return code {0}\n".format(rc))

  # special case to remove agent
  if rm_agent_name :
    print "### Removing " + rm_agent_name + " ###"
    # rpm remove agent with --no-scripts, "--no-scripts" is needed otherwise the preuninstall
    # scriptlet will try use systemctl stop wr-iot-agent and cause rpm failed with signal 15
    cmd = 'sudo rpm -e --nodeps --noscripts ' + rm_agent_name
    os.system(cmd)
    # kill the agent, which will also kill the ota child process
    os.system("killall wra")
    sys.exit(rc)

  if reboot_agent:
    print "### reboot agent via shell cmd ##"
    exec_shell_cmd("touch /var/wra/otaflag/timer_trigger")
    cmd = "sudo /sbin/reboot"
    exec_shell_cmd(cmd)

  sys.exit(rc)


def remove_packages(package_list):
  global rm_agent_name

  # Check if we have an actional list
  if type(package_list) is not list:
      cleanup_and_exit(INVALID_JSON_DATA)

  # Remove packages one-by-one
  for pkg in package_list:

    # check if it is a wr-iot-agent package and handle it at the end
    cmd = 'sudo rpm -q ' + pkg
    rc,stdout,stderr = exec_shell_cmd1(cmd)

    ptrn = re.compile('wr-iot-agent-[0-9]+')
    if ptrn.match(stdout):
      log_file_write("INFO: {0} will be removed at the end\n".format(pkg))
      rm_agent_name=pkg
      continue

    # Check if the package is installed
    cmd = 'sudo rpm -q ' + pkg 

    if PREVIEW == 1:
      cmd = 'echo ' + cmd

    log_file_write("INFO: Checking if {0} is installed ...\n".format(pkg))

    # If the package was installed then remove it
    if exec_shell_cmd(cmd) == 0:
      cmd = 'sudo rpm -e --nodeps ' + pkg + ' | tee -a {} | wra-util --senddata wr_install_updates.log --prefix \"RPM REMOVE: \"'.format(log_fn)

      if PREVIEW == 1:
        cmd = 'echo ' + cmd

      log_file_write("INFO: {0} is installed, removing it now ...\n".format(pkg))

      exec_shell_cmd(cmd)

    else:
      log_file_write("INFO: {0} was not installed.\n".format(pkg))


def update_packages(package_list): 
  global found_kernel_image
  global found_agent_image
  global boot_mnt
  global agent_tmpdir
  agent_rpm_list = []
 
  # Check if we have an actional list
  if type(package_list) is not list:
      cleanup_and_exit(INVALID_JSON_DATA)

  # check if the agent package exists then move it to a temporray dir and we will treat it seperately 
  agent_tmpdir = tempfile.mkdtemp(prefix='wra')
  ptrn = re.compile('wr-iot-agent+')
  for s in package_list:
      m = ptrn.match(s)
      if m:
          agent_rpm_list.append(s);
          log_file_write("INFO: found agent " + s + "\n")
          found_agent_image = 1
          mv_agent = 'sudo mv ' + os.path.join(mnt,s) + ' ' + agent_tmpdir
          status = exec_shell_cmd(mv_agent)
          if status:
              log_file_write("ERROR: Failed to separate Agent installation\n");
              found_agent_image = 0
              exec_shell_cmd('rm -rf '+ agent_tmpdir)
 
  # Perform an upgrade
  cmd = 'sudo rpm -Uv --force  --nodeps ' + os.path.join(mnt,'*.rpm | tee -a {} | wra-util --senddata wr_install_updates.log --prefix \"RPM UPDATE: \"'.format(log_fn))
  if PREVIEW == 1:
    cmd = 'echo ' + cmd
  exec_shell_cmd(cmd)
  # agent upgrade 
  if found_agent_image:
    # during wr-iot-agent rpm installation, data cannot be sent to cloud, send the info now
    for s in agent_rpm_list:
        cmd = 'echo ' + '\"'+ s + '\"' + ' | wra-util --senddata wr_install_updates.log'
        exec_shell_cmd(cmd);
    # start rpm install
    log_file_write("INFO: Upgrading Agent Separately\n")
    cmd = 'sudo rpm -U --force --nodeps --noscripts ' + os.path.join(agent_tmpdir,'*.rpm')
    exec_shell_cmd(cmd)

  rm_agent_tmpdir = 'rm -rf ' + agent_tmpdir
  exec_shell_cmd(rm_agent_tmpdir)

  if any("kernel-image" in s for s in package_list):
    found_kernel_image = 1

#  if any("wr-iot-agent-" in s for s in package_list):
#   found_agent_image = 1

def reboot_request(request):
  global reboot, reboot_delay
  if "rebootOnCompletion" in request:
    if request["rebootOnCompletion"] == "YES":
      log_file_write("INFO: A system reboot has been requested by the update package.\n")
      reboot = 1

  if "rebootDelay" in request:
    reboot_delay = request["rebootDelay"]
    log_file_write("INFO: A reboot delay of {0} seconds was requested by the update package.\n".format(reboot_delay))


# Main
def main(argv):
  """Main routine"""
  global mnt, log_fn, log_file, json_data, reboot_agent
  rc = 0

  try:
    # Just log to the working directory if in preview mode
    if PREVIEW:
      log_fn = "wr_install_updates.log"

    log_file=open(log_fn,'a+')
  except IOError as e:
    print "Could not open log file: %s" % log_fn
    sys.exit(1)

  # close the file so that other process can append into the same log file
  log_file.close()

  if len(argv) < 1:
    log_file_write("INFO: Did not specify an update package file.\n")
    sys.exit(1)

  # Update mount point
  mnt = mount_update_package(argv[0])

  # Slurp up the json file
  json_file=os.path.join(mnt,json_fn)

  try:
    json_data=open(json_file)
  except IOError as e:
    log_file_write("ERROR: {0}\n".format(e.strerror))
    cleanup_and_exit(1)

  try:
    data = json.load(json_data)
  except ValueError:
    log_file_write("ERROR: Could not load json data.\n")
    cleanup_and_exit(1)

  # Process packages to be removed 
  if "packages" in data.get("remove", {}):
    log_file_write("INFO: Removing rpm packages\n")
    remove_packages(data["remove"]["packages"])

  # Check for a reboot request
  if "remove" in data:
    reboot_request(data["remove"])

  # Process packages to be added 
  if "packages" in data.get("local-install", {}):
    log_file_write("INFO: Installing rpm packages\n")
    update_packages(data["local-install"]["packages"])

  # Copy the kernel image to the boot partition
  if found_kernel_image:
    log_file_write("INFO: kernel-image rpm is installed, copying kernel to boot partition\n")
    install_kernel()

  # Check for a reboot request
  if "local-install" in data:
    reboot_request(data["local-install"])

  # If found the agent image set the reboot_agent flag
  if found_agent_image:
    log_file_write("INFO: wr-iot-agent rpm is installed, request a reboot\n")
    reboot_agent = 1

  json_data.close()

  # Clean up
  log_file_write("INFO: Install completed successfully.\n")

  # Let the agent know the installation was successful but a reboot was requested
  os.path.exists(reboot_delay_fn) and os.remove(reboot_delay_fn)
  if reboot:
    # return success and reboot, delay is put into the reboot_delay file 
    rc = SUCCESS_AND_REBOOT
    text_file = open(reboot_delay_fn, "w")
    text_file.write("{0}".format(reboot_delay))
    text_file.close()
    log_file_write("Reboot with delay of {0} seconds.\n".format(reboot_delay))
  cleanup_and_exit(rc)

if __name__ == "__main__":
    main(sys.argv[1:])
