/*
 * The Initial Developer of the Original Code is International
 * Business Machines Corporation. Portions created by IBM
 * Corporation are Copyright (C) 2005 International Business
 * Machines Corporation. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the Common Public License as published by
 * IBM Corporation; either version 1 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * Common Public License for more details.
 *
 * You should have received a copy of the Common Public License
 * along with this program; if not, a copy can be viewed at
 * http://www.opensource.org/licenses/cpl1.0.php.
 */

/* 
 *  TCG Applications                                => Combinations of TPM acc conventions to form application
 *   |      |     |
 *   |      |     v
 *   |      |  libTpm_Version() @ lib_tpmacc.c      => Conventions encapsulation, combine wrapped APIs
 *   |      |               |
 *   |      v               v
 *   |  contextCreate() @ tpm_tspi.c                => Wrapper layer of APIs to alter TSS library, if needed
 *   |         |
 *   v         v
 *  Tspi_Context_Create() @ TSS Library             => Basice APIs specified by TCG standard
 */


#ifndef __TPM_ACTLIBS_H__
#define __TPM_ACTLIBS_H__

#include <stdio.h>
#include <stdlib.h>
#include <trousers/tss.h>
#include <trousers/trousers.h>
#include <openssl/evp.h>
#include <sys/stat.h>
#include <errno.h>

#include "tpm_tspi.h"
#include "tpm_utils.h"
#ifdef COMMON_UTILS
#include "log.h"
#else
#define dbg_err(fmt,args...)            printf(fmt, ##args)
#define dbg_warn(fmt,args...)           
#define dbg_info(fmt,args...)           
#endif

#define TSS_WELL_KNOWN_SECRET \
        {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}

typedef struct {
    TSS_FLAG        mode;
    UINT32          len;
    BYTE *          secret;
} libTpm_Secret;

typedef struct {
    char        hdr_string[32];
    char        ftr_string[32];
    char        tss_string[32];
    char        evp_string[32];
    char        enc_string[32];
    char        keytype_sym[32];
    char        cipher_aes256cbc[32];
    char        secret[32];
    char        iv[32];
} libTpm_SealFmt;

#define TSSKEY_DEFAULT_SIZE 768
#define EVPKEY_DEFAULT_SIZE 512

typedef enum {
    ENOTSSHDR = 0,
    ENOTSSFTR,
    EWRONGTSSTAG,
    EWRONGEVPTAG,
    EWRONGDATTAG,
    EWRONGKEYTYPE,
    EBADSEEK,
    ETSPICTXCREAT,
    ETSPICTXCNCT,
    ETSPICTXCO,
    ETSPICTXLKBU,
    ETSPICTXLKBB,
    ETSPISETAD,
    ETSPIGETPO,
    ETSPIPOLSS,
    ETSPIDATU,
    ETSPIPOLATO,
} PARSE_RESULT;

#define TSS_ERROR_CODE(x)	(x & 0xFFF)
#define TSS_ERROR_LAYER(x)	(x & 0x3000)

#ifdef SHOW_SUCCESS_TSS_CALL
#define TSS_CALL(func, retval) do { \
    char    func_name[63+1] = {0}; \
\
    sscanf(#func, "%60s", func_name); \
    sscanf(func_name, "%[_a-zA-Z]*s", func_name); \
    strcat(func_name, "()"); \
\
    retval = func; \
    if (TSS_SUCCESS != retval) { \
        printf("\n"); \
        printf("%s(%d): %-30s return %-30s(0x%x), exit!\n", __FUNCTION__, __LINE__, func_name, libTpm_ErrString(retval), retval); \
        printf("\n"); \
        return retval;  \
    } \
    else { \
       printf("%s(%d): %-30s return %-30s(0x%x), OK!\n", __FUNCTION__, __LINE__, func_name, libTpm_ErrString(retval), retval); \
    } \
\
} while(0)

#else

#define TSS_CALL(func, retval) do { \
    char    func_name[31+1] = {0}; \
\
    sscanf(#func, "%31s", func_name); \
    sscanf(func_name, "%[_a-zA-Z]*s", func_name); \
\
    retval = func; \
    if (TSS_SUCCESS != retval) { \
        printf("\n"); \
        printf("%s(%d): %-30s return %-30s(0x%x), exit!\n", __FUNCTION__, __LINE__, func_name, libTpm_ErrString(retval), retval); \
        printf("\n"); \
        return retval;  \
    } \
\
} while(0)

#endif

TSS_RESULT libTpm_CloseSession (TSS_HCONTEXT hContext);
TSS_RESULT libTpm_OpenSession (TSS_HCONTEXT * phContext, TSS_HTPM * phTpm);

TSS_RESULT libTpm_CreateRsaBindKey (TSS_HCONTEXT hContext, libTpm_Secret srkSecret, TSS_HKEY * phBindingKey, UINT32 ulKeyLength);
TSS_RESULT libTpm_CreateRsaSignKey (TSS_HCONTEXT hContext, libTpm_Secret srkSecret, TSS_HKEY * phMSigningKey);
TSS_RESULT libTpm_CreateRsaSignKey2 (TSS_HCONTEXT hContext, libTpm_Secret srkSecret, TSS_HKEY * phMSigningKey, UINT32 ulKeyLength);
TSS_RESULT libTpm_LoadRsaSignKey (TSS_HCONTEXT hContext, libTpm_Secret srkSecret, TSS_UUID uuidKey, TSS_HKEY * phLoadedKey);
TSS_RESULT libTpm_SaveRsaSignKey (TSS_UUID uuidKey, TSS_HKEY hSavedKey);

TSS_RESULT libTpm_LoadRsaBindKey (TSS_HCONTEXT hContext, libTpm_Secret srkSecret, TSS_UUID uuidKey, TSS_HKEY * phLoadedKey);
TSS_RESULT libTpm_SaveRsaBindKey (TSS_UUID uuidKey, TSS_HKEY hSavedKey);

TSS_RESULT libTpm_ResetDaLock(TSS_HCONTEXT hContext, TSS_HTPM hTpm, libTpm_Secret srkSecret);
TSS_RESULT libTpm_RsaSignHashValue (TSS_HCONTEXT hContext, TSS_HKEY hMSigningKey, BYTE* prgbHashValue,           /* IN  */
                                    TSS_HHASH * phHash, UINT32 * pulSignatureLength, BYTE ** pprgbSignature);    /* OUT */

TSS_RESULT libTpm_RsaVerifySignature(libTpm_Secret srkSecret, TSS_UUID uuidKey, BYTE *rgbHashValue, UINT32 ulHashValueLength, BYTE *rgbSignature, UINT32 ulSignatureLength);
TSS_RESULT libTpm_RsaVerifySigWithK(TSS_HCONTEXT hContext, TSS_HKEY hMSigningKey, BYTE *rgbHashValue, UINT32 ulHashValueLength, BYTE *rgbSignature, UINT32 ulSignatureLength);

char * libTpm_ErrString (TSS_RESULT r);
TSS_RESULT libTpm_CreateRsaSealKey (TSS_HCONTEXT hContext, libTpm_Secret srkSecret, libTpm_Secret sealSecret, TSS_HKEY * phMSigningKey);
TSS_RESULT libTpm_LoadRsaSealKey (TSS_HCONTEXT hContext, libTpm_Secret srkSecret, libTpm_Secret sealSecret, TSS_UUID uuidKey, TSS_HKEY * phSealKey);
TSS_RESULT libTpm_SaveRsaSealKey (TSS_UUID uuidKey, TSS_HKEY hSavedKey);
TSS_RESULT libTpm_SealFileWithK (TSS_HCONTEXT hContext, TSS_HOBJECT hTpm, TSS_HKEY hSealKey, libTpm_Secret sealSecret, libTpm_SealFmt sealFmt, char in_filename[], 
                                 char out_filename[]);
TSS_RESULT libTpm_UnsealFileWithK(TSS_HCONTEXT hContext, TSS_HKEY hSealKey, libTpm_Secret sealSecret, libTpm_SealFmt sealFmt,
                                  char in_filename[], char out_filename[]);

TSS_RESULT libTpm_GetPubEndorsementKey (void);
TSS_RESULT libTpm_Version (void);
int libTpm_ParseTssEvpKeys(char in_filename[], libTpm_SealFmt sealFmt,         /* IN */
                                          BYTE ** prgbTssKey, int * pulTssKeyLength, int * pulTssKeyDataSize,   /* OUT */
                                          BYTE ** prgbEvpKey, int * pulEvpKeyLength, int * pulEvpKeyDataSize,   /* OUT */
                                          BIO ** pBmem,                                 /* OUT */
                                          char ** prgbData, BIO ** pBioData);           /* OUT */


#endif /* __TPM_ACTLIBS_H__ */
