 /* wra.h - Wind River IoT Agent API Interface Header */

/*
 * Copyright (c) 2011 2014 Wind River Systems, Inc.
 *
 * The right to copy, distribute, modify or otherwise make use
 * of this software may be licensed only pursuant to the terms
 * of an applicable Wind River license agreement.
 */

/*
modification history
--------------------
01a,17oct14,dwb written.
*/

/*
DESCRIPTION
Wind River HDC Agent API

INCLUDE FILES: wra_types.h
*/


#ifndef WRA_H
#define WRA_H

#include <wra_types.h>

#define WRA_TM_MAJOR_VER 1
#define WRA_TM_MINOR_VER 2

/*! Application Execution Handler */
typedef int (*wra_app_ex_handler)(void *dev_h, const char *appname, const char *args);

/*! Agent handle */
typedef struct wra_handle_t *wra_handle;

/*! Service identification handle */
typedef struct wra_service_handle_t *wra_service_handle;

/*! Agent notification handle */
typedef struct wra_notification_handle_t *wra_notification_handle;

/*! Telemetry handle */
typedef struct wra_tm_handle_t *wra_tm_handle;

/*! Telemetry faux storage class */
typedef char wra_tm[32];

/*! Return status */
typedef int wra_status;

/*! Time stamp */
struct wra_timestamp_s
{
  /*! seconds */
  long tv_sec;
  /*! microseconds */
  long tv_usec;
};

/*! Time stamp */
typedef struct wra_timestamp_s wra_timestamp;

/*! wra info structure */
typedef struct wra_info_s
{
  /*! ping rate */
  wra_timestamp ping_rate;
  /*! last contact time */
  wra_timestamp last_contact;
} wra_info_t;

/*

 ****************************************************************************
 *                      FUNCTIONS
 ****************************************************************************
 */

/****************************************************************************
*
* wra_gethandle - Get a handle to the WR Agent.
*
* Call this function to get a calling context to access the WR Agent.
*
* RETURNS: A handle to the WR Agent API
*
* ERRON : WRA_NULL if the Agent is un-initialized.
*/

wra_handle wra_gethandle
(
        void
);

/****************************************************************************
*
* wra_getinfo - Get info from the WR Agent.
*
* Call this function to get a populated wra_info_t structure.  See wra_info_t
* definition for details.
*
* RETURNS: WRA_SUCCESS if the operation was successful.
*
* ERRNO: WRA_ERR_FAILED otherwise
*/

wra_status wra_getinfo
(
  wra_handle wra_h,
  wra_info_t *wra_info
);

/****************************************************************************
*
* wra_tm_create - Allocate a telemetry object and initialize it.
*
* Allocate a telemetry object that can then be used in subsequent calls
* to the wra_tm_* interface functions to set and get its attributes. Calling
* this function allocates memory from the heap to store the object.
* It is not necessary to call wra_tm_init after calling this function.
*
* Parameters:
* \is
* \i <const char *tmtype>
* [in] The type of telemetry.
* \i <const char *tmname>
* [in] The name of the telemetry object.
* \ie
*
* RETURNS: WRA_SUCCESS if the operation was successful.
*
* ERRNO: WRA_ERR_FAILED otherwise
*
*/

wra_tm_handle wra_tm_create
(
        const char *tmtype,
        const char *tmname
);

/****************************************************************************
*
* wra_tm_destroy - Free the contents of a telemetry object.
*
* Free the contents of a telemetry object when it is no longer needed.
*
* Parameters:
* \is
* \i <wra_tm_handle tm>
* [in] A telemetry object whose contents to free.
* \ie
*
* RETURNS: WRA_SUCCESS if the operation was successful.
*
* ERRNO: WRA_ERR_FAILED otherwise
*
*/

wra_status wra_tm_destroy
(
        wra_tm_handle tm
);

/***************************************************************************
*
* wra_tm_reset - Reset a telemetry object with default attributes.
*
* Reset a telemetry object that can then be used in subsequent calls
* to the wra_tm_* interface functions to set and get its attributes.
*
* Parameters:
* \is
* \i <tm>
* [out] A telemetry object to be initialized.
* \i <tmname>
* [in] The name of the telemetry object.
* \ie
*
* RETURNS: A valid handle to the reset telemetry object if successful.
*
* ERRNO: WRA_NULL otherwise
*
*/

wra_tm_handle wra_tm_reset
(
        wra_tm_handle tm,
        const char *tmname
);

/****************************************************************************
*
* wra_tm_subscribe - Subscribe to data telemetry.
*
* Register to receive incoming telemetry data from the server.
*
* Parameters:
* \is
* \i <wra_handle wra_h>
* [in] A valid handle to the WR Agent.
* \i <wra_tm_handle tm>
* [in] A valid, initialized handle to a telemetry object.
*     (Only data telemetry is support)
* \i <const char *tmname>
* [in] The name of the telemetry object.
* \i <wra_timestamp *tmo>
* [in] A time out value to specify how long to wait without
*      receiving a telemetry object. A NULL value specifies
*      that the application will wait indefinitely.
* \ie
*
* RETURNS: WRA_SUCCESS if the operation was successful.
*
* ERRNO: WRA_ERR_FAILED otherwise
*
*/

wra_status wra_tm_subscribe
(
 wra_handle wra_h,
 wra_tm_handle tm,
 const char *tmname,
 wra_timestamp *tmo
 );

/***************************************************************************
*
* wra_tm_setvalue_string - Set the textual value of a telemetry object
*                          attribute.
*
* Set the textual value of the specified attribute in a telemetry object.
*
* Parameters:
* \is
* \i <tm>
* [in] A valid, initialized telemetry object.
* \i <attr>
* [in] The attribute to set.
* \i <value>
* [in] The attribute value.
* \ie
*
* RETURNS: WRA_SUCCESS if the operation was successful.
*
* ERRNO: WRA_ERR_FAILED otherwise.
*/

wra_status wra_tm_setvalue_string
(
        wra_tm_handle tm,
        const char *attr,
        const char *value
);

/***************************************************************************
*
* wra_tm_getvalue_string - Get the textual value of a telemetry object
*                          attribute.
*
* Get the textual value of the specified attribute from a telemetry object.
*
* Parameters:
* \is
* \i <tm>
* [in] A valid, initialized telemetry object.
* \i <attr>
* [in] The attribute to get.
* \i <value>
* [out] A reference to the attribute value.
* \ie
*
* RETURNS: WRA_SUCCESS if the operation was successful.
*
* ERRNO: WRA_ERR_FAILED otherwise.
*/

wra_status wra_tm_getvalue_string
(
        wra_tm_handle tm,
        const char *attr,
        char **value
);

/***************************************************************************
*
* wra_tm_setvalue_int - Set the integer value of a telemetry object
*                       attribute.
*
* Set the 32-bit unsigned value of the specified attribute in a telemetry
* object.
*
*
* Parameters:
* \is
* \i <tm>
* [in] A valid, initialized telemetry object.
* \i <attr>
* [in] The attribute to set.
* \i <value>
* [in] The attribute value.
* \ie
*
* RETURNS: WRA_SUCCESS if the operation was successful.
*
* ERRNO: WRA_ERR_FAILED otherwise.
*/

wra_status wra_tm_setvalue_int
(
        wra_tm_handle tm,
        const char *attr,
        int value
);

/***************************************************************************
*
* wra_tm_getvalue_int - Get the integer value of a telemetry object attribute.
*
* Get the 32-bit unsigned value of the specified attribute from a telemetry
* object.
*
* Parameters:
* \is
* \i <tm>
* [in] A valid, initialized telemetry object.
* \i <attr>
* [in] The attribute to get.
* \i <value>
* [out] A reference to the attribute value.
* \ie
*
* RETURNS: WRA_SUCCESS if the operation was successful.
*
* ERRNO: WRA_ERR_FAILED otherwise.
*/

wra_status wra_tm_getvalue_int
(
        wra_tm_handle tm,
        const char *attr,
        int *value
);

/***************************************************************************
*
* wra_tm_setvalue_bool - Set the boolean value of a telemetry object
*                        attribute.
*
* Set the boolean value of the specified attribute in a telemetry object.
*
* Parameters:
* \is
* \i <tm>
* [in] A valid, initialized telemetry object.
* \i <attr>
* [in] The attribute to set.
* \i <value>
* [in] The attribute value.
* \ie
*
* RETURNS: WRA_SUCCESS if the operation was successful.
*
* ERRNO: WRA_ERR_FAILED otherwise.
*/

wra_status wra_tm_setvalue_bool
(
        wra_tm_handle tm,
        const char *attr,
        bool value
);

/***************************************************************************
*
* wra_tm_getvalue_bool - Get the boolean value of a telemetry object
*                        attribute.
*
* Get the boolean value of the specified attribute from a telemetry object.
*
* Parameters:
* \is
* \i <tm>
* [in] A valid, initialized telemetry object.
* \i <attr>
* [in] The attribute to get.
* \i <value>
* [in] A reference to the attribute value.
* \ie
*
* RETURNS: WRA_SUCCESS if the operation was successful.
*
* ERRNO: WRA_ERR_FAILED otherwise.
*/

wra_status wra_tm_getvalue_bool
(
        wra_tm_handle tm,
        const char *attr,
        bool *value
);

/***************************************************************************
*
* wra_tm_setvalue_double - Set the double value of a telemetry object
*                          attribute.
*
* Set the double value of the specified attribute in a telemetry object.
*
* Parameters:
* \is
* \i <tm>
* [in] A valid, initialized telemetry object.
* \i <attr>
* [in] The attribute to set.
* \i <value>
* [in] The attribute value.
* \ie
*
* RETURNS: WRA_SUCCESS if the operation was successful.
*
* ERRNO: WRA_ERR_FAILED otherwise.
*/

wra_status wra_tm_setvalue_double
(
        wra_tm_handle tm,
        const char *attr,
        double value
);

/***************************************************************************
*
* wra_tm_getvalue_double - Get the double value of a telemetry object
*                          attribute.
*
* Get the double value of the specified attribute in a telemetry object.
*
* Parameters:
* \is
* \i <tm>
* [in] A valid, initialized telemetry object.
* \i <attr>
* [in] The attribute to set.
* \i <value >
* [in] A reference to the attribute value.
* \ie
*
* RETURNS: WRA_SUCCESS if the operation was successful.
*
* ERRNO: WRA_ERR_FAILED otherwise.
*/

wra_status wra_tm_getvalue_double
(
        wra_tm_handle tm,
        const char *attr,
        double *value
);

/***************************************************************************
*
* wra_tm_setaux - Set the auxiliary telemetry information of a telemetry
*                 object attribute.
*
* The value must resolve to a valid type expected by the
* attribute, or the operation will not complete successfully.
*
* Parameters:
* \is
* \i <tm>
* [in] A valid, initialized telemetry object.
* \i <attr>
* [in] The attribute to set.
* \i <value>
* [in] A reference to the attribute value.
* \ie
*
* RETURNS: WRA_SUCCESS if the operation was successful.
*
* ERRNO: WRA_ERR_FAILED otherwise.
*/

wra_status wra_tm_setaux
(
        wra_tm_handle tm,
        const char *attr,
        wra_tm_handle aux
);

/***************************************************************************
*
* wra_tm_getaux - Get the auxiliary telemetry information of a telemetry
*                 object attribute.
*
* The calling function must provide an initialized value that is expected
* by the attribute, or the operation will not complete successfully.
*
* Parameters:
* \is
* \i <tm>
* [in] A valid, initialized telemetry object.
* \i <attr>
* [in] The attribute to set.
* \i <value>
* [in] The attribute value
* \ie
*
* RETURNS: WRA_SUCCESS if the operation was successful.
*
* ERRNO: WRA_ERR_FAILED otherwise.
*/

wra_status wra_tm_getaux
(
        wra_tm_handle tm,
        const char *attr,
        wra_tm_handle aux
);

/***************************************************************************
*
* wra_tm_settimestamp - Set the telemetry object time stamp.
*
* Set the time stamp of the specified telemetry object.
*
* Parameters:
* \is
* \i <tm>
* [in] A valid, initialized telemetry object.
* \i <tstamp>
* [in] A valid, initialized time stamp.
* \ie
*
* RETURNS: WRA_SUCCESS if the operation was successful.
*
* ERRNO: WRA_ERR_FAILED otherwise.
*
*/

wra_status wra_tm_settimestamp
(
        wra_tm_handle tm,
        wra_timestamp *timestamp
);

/***************************************************************************
*
* wra_tm_gettimestamp - Get the telemetry object time stamp.
*
* Get the time stamp of the specified telemetry object.
*
* Parameters:
* \is
* \i <tm>
* [in] A valid, initialized telemetry object.
* \i <tstamp>
* [out] A reference to the time stamp value.
* \ie
*
* RETURNS: WRA_SUCCESS if the operation was successful.
*
* ERRNO: WRA_ERR_FAILED otherwise.
*
*/

wra_status wra_tm_gettimestamp
(
        wra_tm_handle tm,
        wra_timestamp *timestamp
);

/***************************************************************************
*
* wra_tm_post - Send a telemetry object to the server.
*
* Transmit the telemetry data by way of the WR Agent.
*
* Parameters:
* \is
* \i <wra_h>
* [in] A valid handle to the WR Agent.
* \i <tm>
* [in] The address of the telemetry object.
* \i <wra_service_h>
* [in] A valid service identification handle.
* \i <wra_notification_h>
* [in] A valid notification handle.
* \ie
*
* RETURNS: WRA_SUCCESS if the operation was successful.
*
* ERRNO: WRA_ERR_NO_MEMORY if internal resources are exhausted
*        WRA_ERR_EAGAIN if the agent is not ready
*        WRA_ERR_FULL if the transmission queue is full
*        WRA_ERR_FAILED otherwise
*
*/

wra_status wra_tm_post
(
        wra_handle wra_h,
        wra_tm_handle tm,
        wra_service_handle wra_service_h,
        wra_notification_handle wra_notification_h
);

/***************************************************************************
*
* wra_tm_post_default - Send a telemetry object to the server.
* SYNOPSIS
* void wra_tm_post_default
*       (
*       wra_h,
*       tm
*       )
*
* DESCRIPTION
* Transmit the telemetry by way of the WR Agent.
*
* Parameters:
* \is
* \i <wra_h>
* [in] A valid handle to the WR Agent.
* \i <tm>
* [in] The address of the telemetry object.
* \ie
*
* RETURNS: WRA_SUCCESS if the operation was successful.
*
* ERRNO: WRA_ERR_NO_MEMORY if internal resources are exhausted
*        WRA_ERR_EAGAIN if the agent is not ready
*        WRA_ERR_FULL if the transmission queue is full
*        WRA_ERR_FAILED otherwise
*
*/


#define wra_tm_post_default(wra_h,tm) wra_tm_post((wra_h),(tm),WRA_NULL,WRA_NULL)

/***************************************************************************
*
* wra_reg_app_ex_handler(DEPRECATED) - Register an application
* execution handler.
*
*
* Call this routine to register a handler for
* an application execution action that is sent from the server.
* When the server sends an application execution action
* to the device, it contains an "appname" and an "args" string.
* The agent spawns a task that invokes the handler
* registered using this API against the appname.
* One handler can be registered against more than one appname.
* You can register handlers against a maximum of 128 unique appnames.
*
* The maximum length of appname is 32 characters.
*
* It is up to the handler to parse the arguments.
* The "appname" and "args" parameters are not guaranteed to be
* valid after the handler returns.
*
*
* Parameters:
* \is
* \i <wra_h>
* [in] A valid handle to the WR Agent.
* \i <app_ex_handler>
* [in] Application execution handler to register.
* \i <appname>
* [in] Application name to register the handler against.
* \ie
*
* RETURNS: WRA_SUCCESS if the operation was successful.
*
* ERRNO: WRA_ERR_FAILED otherwise
*
*/

int wra_reg_app_ex_handler
(
    wra_handle wra_h,
    wra_app_ex_handler app_ex_handler,
    const char *appname
);

/***************************************************************************
*
* wra_action_subscribe - Subscribe to an action.
*
* Call this routine to register a handler for an action sent from the server.
* When the server sends an action
* to the device, it contains an "appname" and an "args" string.
* The agent spawns a task that invokes the handler registered the appname.
* One handler can be registered against more than one appname.
* You can register handlers against a maximum of 128 unique appnames.
*
* The maximum length of the appname is 32 characters.
*
* It is up to the handler to parse the arguments.
* The "appname" and "args" parameters are not guaranteed to be valid
* after the handler returns.
*
*
* Parameters:
* \is
* \i <wra_h>
* [in] A valid handle to the WR Agent.
* \i <app_ex_handler>
* [in] Application action handler to register.
* \i <appname>
* [in] Application name to register the handler against.
* \ie
*
* RETURNS: WRA_SUCCESS if the operation was successful.
*
* ERRNO: WRA_ERR_FAILED otherwise
*
*/
int wra_action_subscribe
(
    wra_handle wra_h,
    wra_app_ex_handler app_ex_handler,
    const char *appname
);

/***************************************************************************
*
* wra_action_unsubscribe - Unsubscribe an application action handler.
*
* The application handler registered against the specified appname is
* removed, and the application will no longer receive action notifications
* from the server for the specified appname.
*
* Parameters:
* \is
* \i <wra_h>
* [in] A valid handle to the WR Agent.
* \i <appname>
* [in] Application name to unsubscribe from.
* \ie
*
* RETURNS: WRA_SUCCESS if the operation was successful.
*
* ERRNO: WRA_ERR_FAILED otherwise
*
*/
wra_status wra_action_unsubscribe
(
    wra_handle wra_h,
    const char*  appname
);

/***************************************************************************
*
* wra_action_wait - Wait for action requests from the server.
*
* Call this function after registering a handler for
* an action using the wra_action_subscribe function.
* The application blocks until it receives an action request from server.
* It executes the action handler in the application context user space or
* in a dedicated agent thread in case of flat memory.
* Alternately, the application can specify a timeout, and the function
* returns if the application does not receive an action before the timeout
* expires.
*
* Parameters:
* \is
* \i <wra_h>
* [in] A valid handle to the WR Agent.
* \i <time_out>
* [in] A NULL value specifies that application will block until it receives
*      action request from the server. Alternatively, the application can
*      specify how long to wait for an action request.
* \ie
*
* RETURNS: WRA_SUCCESS - if the registered action was successfully executed.
*          WRA_ERR_EAGAIN - if a timeout was specified and no action
*                         was received within the specified time period.
* ERRNO: WRA_ERR_FAILED otherwise.
*
*/

wra_status wra_action_wait
(
    wra_handle wra_h,
    wra_timestamp *tm
);

/***************************************************************************
*
* wra_file_download - Subscribe to a file download from the server.
*
* Wait for the server to download the specified file.
* Optionally, specify how long to wait for the file.
*
* Parameters:
* \is
* \i <wra_h>
* [in] A valid handle to WR Agent.
* \i <filename>
* [in] name of the file to download.
* \i <dir>
* [in] The absolute path of the directory where the file is copied
*      when it is received from the server. If a NULL value is
*      specified, the file is copied to a preconfigured directory.
*      The agent configuration file specifies the preconfigured
*      directory.
* \i <time_out>
* [in] A NULL value specifies that the application will block
       until it receives the file.
*      Alternatively, the application can specify a timeout, and the
*      function returns if the application does not receive the file
*      before the timeout expires.
* \ie
*
* RETURNS: WRA_SUCCESS if the operation was successful.
*
* ERRNO:
*             WRA_ERR_BAD_PARAM - file name or path is  too long
*                               - destination directory does not exist
*             WRA_ERR_ETIMEDOUT - operation timed out
*             WRA_ERR_FAILED otherwise
*
*/

wra_status  wra_file_download
(
    wra_handle wra_h,
    const char *filename,
    const char* dir,
    wra_timestamp *time_out
);

/***************************************************************************
*
* wra_file_upload - Upload the specified file to server.
*
* Upload a file to the server. Specify the absolute path of the file name.
*
* Parameters:
* \is
* \i <wra_h>
* [in] A valid handle to WR Agent.
* \i <filepath>
* [in] Absolute path of the file from where agent will upload it to server.
* \i <time_out>
* [in] A NULL value specifies that the application will block until the agent
*      finishes uploading the file.
*      Alternatively, the application can specify a timeout, and if the file
*      is not uploaded before the timeout expires, the function return code
*      indicates that the file failed to upload due to timing out.
* \ie
*
* RETURNS: WRA_SUCCESS if the operation was successful.
*
* ERRNO:
*             WRA_ERR_BAD_PARAM - file does not exist
*             WRA_ERR_ETIMEDOUT - operation timed out
*             WRA_ERR_FAILED otherwise
*/

wra_status  wra_file_upload
(
    wra_handle wra_h,
    const char *filepath,
    wra_timestamp *time_out
);

/***************************************************************************
*
* wra_file_upload_ex - Upload the specified file to server with extra parameters
*
* Upload a file to the server. Specify the absolute path of the file name and
* file hint
*
* Parameters:
* \is
* \i <wra_h>
* [in] A valid handle to WR Agent.
* \i <filepath>
* [in] Absolute path of the file from where agent will upload it to server.
* \i <file_param>
* [in] A string to identify file param which is used by the platform
* \i <time_out>
* [in] A NULL value specifies that the application will block until the agent
*      finishes uploading the file.
*      Alternatively, the application can specify a timeout, and if the file
*      is not uploaded before the timeout expires, the function return code
*      indicates that the file failed to upload due to timing out.
* \ie
*
* RETURNS: WRA_SUCCESS if the operation was successful.
*
* ERRNO:
*             WRA_ERR_BAD_PARAM - file does not exist
*             WRA_ERR_ETIMEDOUT - operation timed out
*             WRA_ERR_FAILED otherwise
*/

wra_status  wra_file_upload_ex
(
    wra_handle wra_h,
    const char *filepath,
    const char *file_param,
    wra_timestamp *time_out
);

/***************************************************************************
*
* wra_delete_handle - wra context from user space.
*
* Call this function to cleanup the agent resources (such as queues and
* subscriptions) during the application life cycle.
*
* Parameters:
* \is
* \i <wra_h>
* [in] A valid handle to the WR Agent.
* \ie
*
* RETURNS: WRA_SUCCESS if the operation was successful.
*
* ERRNO: WRA_ERR_FAILED otherwise.
*
*/

int wra_delete_handle
(
    wra_handle wra_h
);

/***************************************************************************
*
* wra_restore_factory_software_image - reboot with the factory image.
*
* Cause the target to wipe out the root filesystem and replace it with the
* factory software image stored in the VFAT partition.  This function only
* returns if there was an error.  Normally it does not return and the target
* reboots.
*
* Parameters:
* \is
* \i <wra_h>
* [in] A valid handle to the WR Agent.
* \ie
*
* RETURNS: WRA_ERR_FAILED.
*
*/

wra_status wra_restore_factory_software_image
(
    wra_handle wra_h
);

#endif /* WRA_H */
