/**
 * @section Copyright (c) 2014 Wind River Systems, Inc.
 *
 * The right to copy, distribute or otherwise make use of this software
 * may be licensed only pursuant to the terms of an applicable Wind River
 * license agreement.
 *
 * @file wra_h.h
 * @date Oct 18, 2014 
 * @authors hfc and dwb
 * 
 * @section WindRiver Agent for Edge Management System
 * 
 * @brief private Header file. Should not be included by user application.
 * 
 *
 * modification history
   --------------------
   0a,29Dec14,hfc add support for message base communication
 */

#ifndef WRA_PRIVATE_H
#define WRA_PRIVATE_H

/*
 ****************************************************************************
 *                     DESCRIPTION
 ****************************************************************************
 * Internal header file.
 */

/*
 ****************************************************************************
 *                      INCLUDE FILES
 ****************************************************************************
 */

#include "wra_utils.h"
#include <wra_types.h>
#include <wra.h>
#include "AeOSLocal.h"
#include "AeTypes.h"
#include "AeError.h"
#include "AeOS.h"
#include "AeInterface.h"
#include <mqueue.h>
#ifdef __cplusplus
extern "C" {
#endif

/*
 ****************************************************************************
 *                     DEFINES
 ****************************************************************************
 */
/* default ping rate in case it's less than this at compile time or from cloud
 * see also AeDRM.c, where a magic number instead of this macro is used */
#define WRA_DEFAULT_PING_RATE           5000 /*Unit: ms (millisecond)*/

#define WRA_RTP_QUEUE_NAME              "/wra_rtp_q"
#define WRA_RTP_QUEUE_NAME_MAXLEN       32
#define WRA_CHANNEL_LOCATION                 CHANNEL_LOCATION
#define WRA_CHANNLE_NAME_MAXLEN          128
#define WRA_RTP_MAX_NO_MSG              10
#define MSG_PRI_NORMAL 0
#define WRA_TM_MAGIC 0x43218765
#define WRA_AGENT_MAGIC 0x87654321

#define WRA_TRUE 1
#define WRA_FALSE 0
#define WRA_TM_MAX_NAME_LEN 128
#define WRA_TM_MAX_STR_VALUE_LEN 256 
#define WRA_TM_MAX_DESC_LEN 256 
#define WRA_AP_EX_APPNAME_LEN   32
#define WRA_SUBSCRIBE_NAME_LEN   WRA_TM_MAX_NAME_LEN
#define WRA_AP_EX_ARGS_LEN   256
#define WRA_AP_EX_REG_INTERNAL_MAX  8
#define WRA_AP_EX_REG_EXTERNAL_MAX  128
#define WRA_AP_EX_REG_MAX   (WRA_AP_EX_REG_INTERNAL_MAX+WRA_AP_EX_REG_EXTERNAL_MAX)    
#define WRA_DATA_ENT_REG_MAX WRA_AP_EX_REG_MAX
#define WRA_FILE_ENT_REG_MAX WRA_AP_EX_REG_MAX

#define WRA_SUBSCRIBE_FILENAME_LEN  128
#define WRA_SUBSCRIBE_FILEPATH_LEN  256
#define WRA_SUBSCRIBE_FILEPARAM_LEN 128

#define WRA_OPAQUE_IPC_BUFF_SIZE 2048
/* Telemetry subscrition table size */
#define WRA_TM_SUBSCRIPTION_ENTRIES_MAX 128
/*#define WRA_TM_DEBUG*/
#define WRA_TM_DEBUG
#ifdef WRA_TM_DEBUG
#define WRA_TM_LOG(msg) printf(msg"\n");
#else
#define WRA_TM_LOG(msg)
#endif

#define WRA_ALIGN(sz, align)  (((sz) + (align) - 1) & ~((align) - 1))
#define WRA_WEB_HTTP_PROXY "http"
#define WRA_WEB_SOCKS_PROXY "socks"
/*
 *===========================================================================
 *                     WRA_IF_INVALID_HANDLE                
 *===========================================================================
 * Check the handle context and magic
 */
#define WRA_IF_INVALID_HANDLE(ctx) \
  if((ctx) == WRA_NULL || (ctx)->magic != WRA_AGENT_MAGIC)

/*
 *===========================================================================
 *                     WRA_TM_VALIDATE                
 *===========================================================================
 * Check the telemetry object and magic 
 */
#define WRA_TM_IF_INVALID_TM(tm) \
    if((tm) == WRA_NULL || (tm)->magic != WRA_TM_MAGIC)

/*
 *===========================================================================
 *                     WRA_TM_IF_INVALID_ARGS                
 *===========================================================================
 * Check the telemetry object, its magic, and name 
 */
#define WRA_TM_IF_INVALID_ARGS(tm,name) \
    if((tm) == WRA_NULL || (tm)->magic != WRA_TM_MAGIC || \
        (name) == WRA_NULL) 

/*
 *===========================================================================
 *                     WRA_TM_IF_INVALID_INDEX              
 *===========================================================================
 * 
 */
#define WRA_TM_IF_INVALID_INDEX(attr) \
    if(attr < 0 || attr >= WRA_AE_NUMKEYS)

/*
 *===========================================================================
 *                     WRA_TM_IF_TYPE_NOT_SUPPORTED
 *===========================================================================
 */
#define WRA_TM_IF_TYPE_NOT_SUPPORTED(type,attr) \
    if(!((type) & wra_tm_keydb[(attr)].supported)) \

/*
 *===========================================================================
 *                     WRA_TM_LKUP_NAME
 *===========================================================================
 */
#define WRA_TM_LKUP_NAME(attrid,attrname) \
  do { \
    int i,found; \
    attrid = WRA_AE_NUMKEYS; \
    for(i = 0, found = WRA_FALSE; found == WRA_FALSE && i < WRA_AE_NUMKEYS; i++) \
    if(strcmp(wra_tm_keydb[i].name,attrname) == 0) { \
      attrid = wra_tm_keydb[i].id; \
      found = WRA_TRUE; \
    } \
  } while((0))

/*
 *===========================================================================
 *                     WRA_TM_ATTR_VALIDATE                
 *===========================================================================
 * Check the telemetry attribut and magic 
 */
#define WRA_TM_ATTR_VALIDATE(obj) \
    if((obj) == WRA_NULL || (obj)->tm.magic != WRA_TM_MAGIC)

/*
 *===========================================================================
 *                     WRA_TM_ATTR_INITIALIZE                
 *===========================================================================
 * Initialize the telemtry attribute 
 */
#define WRA_TM_ATTR_INITIALIZE(obj,tm_type) \
  do { \
    (obj)->tm.magic = WRA_TM_MAGIC; \
    (obj)->tm.type = tm_type; \
    (obj)->tm.prio = AeDRMQueuePriorityNormal; \
  } while((0))

/*
 *===========================================================================
 *                     WRA_TM_SETCURRENTTIME                
 *===========================================================================
 * Set the current time as the timestamp in a telemetry object
 */
#define WRA_TM_SETCURRENTTIME(obj) \
  do { \
    AeTimeValue tstamp; \
    AeGetCurrentTime(&tstamp); \
    (obj)->tm.tstamp.tv_sec = tstamp.iSec; \
    (obj)->tm.tstamp.tv_usec = tstamp.iMicroSec; \
  } while((0))

/*
 *===========================================================================
 *                     WRA_UTIL_SAFE_STRNCPY                
 *===========================================================================
 * Safe strncpy
 */
#define WRA_UTIL_SAFE_STRNCPY(dptr,sptr,maxlen) \
  do { \
    memset((dptr),'\0',maxlen); \
    strncpy((dptr),(sptr),maxlen-1); \
  } while((0))

/*
 ****************************************************************************
 *                     TYPES
 ****************************************************************************
 */


  /* maximum size for a configuration parameter such as 
   * serial number , model number , host etc. */
#define WRA_MAX_PARAMETER_SIZE            256

  typedef enum {
    SERIAL_NO = 0,
    MODEL_NUMBER,
    HOST,
    HOST_PORT,
    PING_RATE,
    ON_BOARD,
#ifdef ENABLE_REMOTE_SESSION
    REMOTE_SESSION,
#endif
    LOG_FILE,                            /**< just identify a file handle to log, do not need to read the file*/
    LAST_PARAM                           /**< has to be last */
  } wra_conf_file_type;

  typedef enum {
    NOT_ON_BOARDED=0,
    ON_BOARDED,
  }wra_ob_stats;

  typedef enum {
    NOT_REGISTERED=0,
    REGISTERED,
  }wra_reg_stats;

/**
* wra daemon status
* This flag is exclusively set by wra daemon task, hence synchronization is not needed. 
*/

  typedef enum {
    WRA_INIT = 0, /* Always keep value of this enum as 0 */
    WRA_RUNNING,
    WRA_RESTART,
    WRA_RESTART_HARD,
    WRA_FACTORY_RESET,
    WRA_REBOOT,
    WRA_INIT_SSL,
    WRA_STOPPED
  }wra_daemon_stat;

  /**
  * file read/write/exists status for 
  * file upload/download
  */
#define WRA_EXISTS           (1 << 0)
#define WRA_IS_READABLE (1 << 1)
#define WRA_IS_WRITABLE  (1 << 2)
#define WRA_IS_DIR            (1 << 3)
  /**
   * @struct wra_config_f
   *
   * @brief WindRiver Agent configuration files structure
   *
   */

  typedef struct wra_config_f {
    char *files[LAST_PARAM];
  }wra_config_f_t;

  /**
   * @struct wra_media_st
   *
   * @brief WindRiver Agent media handle structure
   *
   */

  typedef struct wra_media_st {
    char *path;                             /**< storage path with read/write access. */
    char *ob_config;                        /**< onboard configuration directory */
    FILE *log_fd;                           /**< valid log file handle */
    wra_config_f_t wra_config;                  /**< configuration files */
  } wra_media;


  typedef struct clould_st {
    char              *host;              /**< Fully Qualified host name/IP address */
    char              *port;              /**< layer 4 port number */
    char              *proto;             /**< protocol such as https */
    char              *service;           /**< Service name i.e. eMessage */
}cloud_t;

/**
* @struct wra_stat_st
*
* @brief WindRiver Agent status counters
* Do not change this directly outside of wra daemon task.
*
*/
typedef struct wra_stat_st {
    wra_daemon_stat     wra_d_stat;         /**< wra daemon status */
    wra_ob_stats        wra_is_onboard;     /**< if = ON_BOARDED, device is on-boarded else not */
    wra_reg_stats       wra_is_registered;  /**< indicate the device registration status */
    long                wra_last_contact_t;
}wra_stat_t;

enum{
ACTION_SUBSCRIPTION_TABLE = 0,
TM_SUBSCRIPTION_TABLE,
FILE_SUBSCRIPTION_TABLE,
TOTAL_SUBSCRIPTION_TABLE             /**< has to be last */
};

typedef struct wra_usr_space_cookie_st {
    /*user space handler and msgQ info*/
    wra_app_ex_handler user_space_handler;
    uint64_t q_remote_add;                    /**< apps queue address in agent space   */
    long devid;                                      /**< future place holder for slave device */
} wra_usr_space_cookie_t;

typedef struct wra_dev_handle_st{
    /* future filed to support more user space parameter*/
    wra_usr_space_cookie_t usr_cookie;
}wra_dev_handle_t;


  /* Telemetry API */
typedef struct
{
    uint32_t magic;
    int type;
    char name[WRA_TM_MAX_NAME_LEN];
    char desc[WRA_TM_MAX_DESC_LEN];
    AeTimeValue timestamp;
    int prio;
    int32_t severity;
} wra_tm_t;

typedef struct
{
    wra_tm_t tm;
    char string_value[WRA_TM_MAX_STR_VALUE_LEN];
    AeDRMDataItem aeDataItem;
} wra_data_t;

typedef struct
{
    wra_tm_t tm;                             /* Common Telemetry Attributes */
    char data_name[WRA_TM_MAX_NAME_LEN]; /* data name */
    char string_value[WRA_TM_MAX_STR_VALUE_LEN]; /* data string value */
    char condition[WRA_TM_MAX_STR_VALUE_LEN];    /* data string value */
    AeDRMAlarm aeAlarm;                          /* Axeda specific alarm */
    AeDRMDataItem aeDataItem;                    /* Axeda specific dataitem */
} wra_alarm_t;

typedef struct wra_event_st
{
    wra_tm_t tm;    /* Common Telemetry Attributes */
    AeDRMEvent aeEvent; /* Axedia specific event */
} wra_event_t;

/* tm post ipc message */
typedef struct wra_ae_post_msg_st
{
    AeInt32 srvid;
    AeInt32 devid;
    AeDRMQueuePriority prio;
    int status;
    union
    {
        wra_data_t wr_di;
        wra_alarm_t wr_al;
        wra_event_t wr_e;
    }u;
} wra_ae_post_msg_t;

/**
* @struct wra_subscribe_entry_st
*
* @brief WindRiver Agent common subscription Table
*
*/
typedef struct wra_subscribe_tbl_st *wra_subscribe_tbl_t;

/**
* @struct wra_subscribe_common_entry_st
*
* @brief WindRiver Agent common subscription element
*
*/
typedef struct wra_subscribe_common_entry_st{
    bool used; /* Set for used entries. Cleared for available entries */
    bool kernel_space;  /*is this entry belong to a flat memory application*/
    bool internal; /* This entry is reserved for internal actions/files/data-item*/
    AeMutex app_sem;   /*lock for flat memory based app */
    long devid;         /* master device */
    int next_item_ix; /* next similar item index */
    int this_item_ix;  /* my index in global array */
    wra_usr_space_cookie_t usr_cookie;
}wra_sub_com_en_t;

/**
* @struct wra_app_ex_entry_st
*
* @brief WindRiver Agent Application Action Table Entry
*
*/
typedef struct wra_app_ex_entry_st {
    wra_sub_com_en_t com;
    char appname[WRA_AP_EX_APPNAME_LEN];
    wra_app_ex_handler handler;
} wra_app_ex_entry_t;


/**
* @struct wra_tm_subscribe_entry_st
*
* @brief WindRiver Agent tm subscription  Table Entry
*
*/
typedef struct wra_tm_subscribe_entry_st {
    wra_sub_com_en_t com;
    char name[WRA_SUBSCRIBE_NAME_LEN];
    wra_data_t data;
} wra_tm_subscribe_entry_t;

/**
* @struct wra_file_item_entry_st
*
* @brief WindRiver Agent file subscription  Table Entry
*
*/
typedef struct wra_file_subscribe_entry_st {
    wra_sub_com_en_t com;
    bool is_file_upload;
    char filename[WRA_SUBSCRIBE_FILENAME_LEN];
    char path[WRA_SUBSCRIBE_FILEPATH_LEN];
    char param[WRA_SUBSCRIBE_FILEPARAM_LEN];
} wra_file_susbcribe_entry_t;

/**
* @struct wra_<tm|action|file>_subscribe_tbl_st
*
* @brief WindRiver Agent subscription  Tables
*
*/

typedef struct wra_action_subscribe_tbl_st {
    AeMutex    lock;         /* lock for action subscription table */
    wra_app_ex_entry_t app_ex_en[WRA_DATA_ENT_REG_MAX];
} wra_action_subscribe_tbl_t;

typedef struct wra_file_subscribe_tbl_st {
    AeMutex             lock;                                              /**< file subscribe table lock*/
    wra_file_susbcribe_entry_t file_subs_ent[WRA_FILE_ENT_REG_MAX];
} wra_file_subscribe_tbl_t;

typedef struct wra_tm_subscribe_tbl_st {
    AeMutex             lock;                                              /**< tm subscribe table lock*/
    wra_tm_subscribe_entry_t tm_subs_ent[WRA_DATA_ENT_REG_MAX];
} wra_tm_subscribe_tbl_t;

/**
* @struct wra_app_ex_htbl_st
*
* @brief WindRiver Agent subscription Master Table
* for action, tm and file.
* Note: Due to the current relatively small size of the table,
*       there is no need to use any hashing, so this is a 
*       simple array
*
*/
typedef struct wra_app_ex_htbl_st {
    wra_action_subscribe_tbl_t act_tbl;
    wra_tm_subscribe_tbl_t tm_tbl;
    wra_file_subscribe_tbl_t file_tbl;
    bool init;
} wra_app_ex_htbl_t;

typedef struct wra_app_ex_msg_st {
    wra_app_ex_handler handler;
    char appname[WRA_AP_EX_APPNAME_LEN];
    char args[WRA_AP_EX_ARGS_LEN];
    wra_dev_handle_t dev_h;
    bool kernel_space; 
    AeMutex app_sem;        /*semaphore to send signal for app*/ 
} wra_app_ex_msg_t;

typedef struct wra_app_q_st
{
    uint64_t q_addr;                                                                 /**< app's msg Q address */
    uint64_t q_remote_add;                                                      /**< app's msg Q address in agents space */
    char      q_name[WRA_RTP_QUEUE_NAME_MAXLEN];          /**<  app's msq name , needs to be unique per app */
}wra_app_q_t;

enum {
    WRA_STATUS_QUEUE = 0,                            	                     /**< queue for status msg */
    WRA_ACTION_QUEUE ,                                                       /**< queue for action subscribe */
    WRA_DATA_QUEUE,                                                            /**< queue for data-item subscribe */
    WRA_FILE_QUEUE,                                                             /**< queue for file subscribe */
    WRA_TOTAL_QUEUE                                                           /**< need to be last */
};

typedef struct wra_client_ctx_st
{
    uint32_t            magic;
    uint64_t            agent_msgQ;                 /**< wra agent's msq q  in app's address space. Apps will use this q to send msg to Agent*/
    wra_app_q_t         queue_t[WRA_TOTAL_QUEUE];   
    AeInt32             iDeviceId;
    AeInt32             iServerId;
    AeTimeValue         pingRate;
    AeTimeValue         lastContact;
} wra_client_ctx_t;

/**
* @struct wra_ctx_st
*
* @brief WindRiver Agent Context structure
*
*/
typedef struct wra_ctx_st
{
    char                **factory_cfg;      /**<Stores the initial factory configuration */
    uint32_t            magic; 
    wra_media           wra_storage;        /**< contains path to config,log and other files */
    wra_stat_t          wra_stat;           /**< wra various status counters */
    cloud_t             cloud;              /**< cloud identifier */
    AeChar              *url;               /**< cloud location to connect */
    AeTimeValue         pingRate;           /**< use to set the default ping rate */
    AeMutex             wra_lock;           /**< WRAgent status lock */ 
    AeInt32             iDeviceId;
    AeInt32             iServerId;
    AeChar              *pOwner;
    AeBool              bSecure;            /**< if true use https */
    AeBool              bDebug;
    AeChar              *pSerialNumber;
    AeChar              *pModelNumber;
    AeBool              bPersist;
    AeBool              bServerAuth;        /**< server certificate verify enable/disable */
    AeChar              *pCACertFile;       /**< server certificate CA file name */
    AeInt32             wra_q_size;         /**< WRAgent internal maximum data queue size in byte */
    AeTimeValue         timeLimit;          /**< Agent Execute time limit */
    wra_app_ex_htbl_t   app_ex_htbl;        /**< WRAgent internal subscription  Tables */
    WRA_MSG_Q_ID        app_ex_msgQ;        /**< WRAgent internal Application Execution Handler Queue */
    WRA_TASK_ID         app_ex_tid;         /**< WRAgent internal Application Execution Handler Task ID */
    WRA_MSG_Q_ID        svr_ex_msgQ;        /**< WRAgent public message Queue */
    WRA_TASK_ID         svr_ex_tid;         /**< WRAgent public message queue handler Task ID */
    struct wra_tm_handle_t *wra_event;      /**< Pointer to Event container that is reusable for internal events */
    struct wra_rs_config *wra_rs;           /**< Pointer to remote session config list */
    char                *net_status;        /**< Network status */
    /* proxy related configuration */
    int proxy_proto;
    AeChar *proxy_server;                                                      /**< proxy server address */
    signed short proxy_port;
    AeChar *proxy_user;
    AeChar *proxy_passwd;
  } wra_ctx;

enum
{
    APP_HANDLE_REQUEST=0,
    APP_HANDLE_RESPONSE,
    APP_HANDLE_DELETE_REQUEST,
    FILE_HANDLE_DELETE_REQUEST,
    TM_HANDLE_DELETE_REQUEST,
    DATA_HANDLE_DELETE_REQUEST,
    APP_DATA_POST_REQUEST,
    APP_ALARM_POST_REQUEST,
    APP_EVENT_POST_REQUEST,
    APP_POST_RESPONSE,
    APP_DATA_REGISTRATION_REQUEST,
    APP_DATA_REGISTRATION_RESPONSE,
    APP_INFO_REQUEST,
    APP_INFO_RESPONSE,
    APP_ACTION_REGISTRATION_REQUEST,
    APP_ACTION_REGISTRATION_RESPONSE,
    APP_ACTION_UNREGISTRATION_REQUEST,
    APP_ACTION_UNGISTRATION_RESPONSE,
    APP_ACTION_HANDLE_REQUEST,
    APP_ACTION_HANDLE_RESPONSE,
    APP_TM_SUBSCRIBE_REQUEST,
    APP_TM_SUBSCRIBE_RESPONSE,
    APP_TM_UNSUBSCRIBE_REQUEST,
    APP_FILE_SUBSCRIBE_REQUEST,
    APP_FILE_SUBSCRIBE_RESPONSE,
    APP_FILE_UNSUBSCRIBE_REQUEST,
    APP_FILE_UNSUBSCRIBE_RESPONSE,
    APP_SFR_REQUEST,
    APP_SFR_RESPONSE,
    WRA_UTIL_SHOW,
    WRA_UTIL_SHOW_RESPONSE
};


/* application action subscribe ipc message */
typedef struct app_reg_msg_st
{
      char name[WRA_AP_EX_APPNAME_LEN];
      char args[WRA_AP_EX_ARGS_LEN];
      uint64_t  q_remote_add;                                                  /*app's recv msgQ address  in agent space*/
      int (*user_handler)(void*, const char*, const char*);   /*valid at user space address */
      int (*kernel_handler)(void*, const char*, const char*); /*valid at kernel space */
      int internal_app;
}app_reg_msg_t;

/* application file subscribe ipc message */
typedef struct file_subscribe_msg_st
{
      char filename[WRA_SUBSCRIBE_FILENAME_LEN];
      char filepath[WRA_SUBSCRIBE_FILEPATH_LEN];
      char channel_name[WRA_CHANNLE_NAME_MAXLEN];
      char fileparam[WRA_SUBSCRIBE_FILEPARAM_LEN]; 
      bool is_file_upload;
      long devid;
      int server_channel; /*server side sock info */
      int errorcode; 
}file_subscribe_msg_t;

typedef struct tm_subscribe_msg_st
{
  char name[WRA_SUBSCRIBE_NAME_LEN];
  char channel_name[WRA_CHANNLE_NAME_MAXLEN];
  long devid;
  int errorcode;
  int server_channel;
}tm_subscribe_msg_t;

/* application opaque data container for ipc message */
typedef struct app_util_msg_st
{
    char buff[WRA_OPAQUE_IPC_BUFF_SIZE];
} app_util_msg_t;

/* common ipc message container */
typedef struct wra_ae_msg_st 
{
    int msg_type;
    int status;                                                                          /* to convey the status */
    char recv_msgQ_name[WRA_RTP_QUEUE_NAME_MAXLEN]; /* app's  global msgQ name. Only use 
                                                                                                 by wra agent for the very first msg from 
                                                                                                 app and use this to retrieve the msgQ id.
                                                                                                 Use msqQ id later on to send back msg
                                                                                                 to app.*/
    uint64_t recv_msgQ;                                                            /* app's msgQ in agent's address space.*/
    union
    {
        wra_client_ctx_t  hdl;
        wra_ae_post_msg_t pmsg;
        app_reg_msg_t app_msg;
        file_subscribe_msg_t file_msg;
        tm_subscribe_msg_t tm_msg;
        app_util_msg_t op_data;
        wra_info_t wra_info;
    }data;
} wra_ae_msg_t __attribute__((aligned(8)));

/* common ipc message container */
typedef struct wra_channel_msg_st 
{
    int msg_type;
    int status;                                                                          /* to convey the status */
    char channel_name[WRA_CHANNLE_NAME_MAXLEN];         /* app's  global msgQ name. Only use 
                                                                                                 by wra agent for the very first msg from 
                                                                                                 app and use this to retrieve the msgQ id.
                                                                                                 Use msqQ id later on to send back msg
                                                                                                 to app.*/
    int channel_id;                                                                     /* app's msgQ in agent's address space.*/
    union
    {
        app_reg_msg_t app_msg;
        file_subscribe_msg_t file_msg;
    }data;
} wra_channel_msg_t __attribute__((aligned(8)));

/*#define WRA_RTP_APP_MSG_SIZE WRA_ALIGN(sizeof(wra_ae_msg_t),_CACHE_ALIGN_SIZE) */
#define WRA_RTP_APP_MSG_SIZE sizeof(wra_ae_msg_t)
#define WRA_RTP_APP_MAX_MSG  10

/* Message queue default attributes */
#define WRA_MQ_DEFAULT_MAX_MSG 10
#define WRA_MQ_DEFAULT_MAX_SIZE sizeof(wra_ae_msg_t) 
#define WRA_MQ_DEFAULT_FLAGS O_CREAT|O_RDONLY|O_EXCL
#define WRA_MQ_DEFAULT_PERMS S_IRUSR|S_IWUSR

enum {
    WRA_AE_NAME = 0,
    WRA_AE_DESC,
    WRA_AE_SEVERITY,
    WRA_AE_PRIORITY,
    WRA_AE_DATA,
    WRA_AE_DATATYPE,
    WRA_AE_ACTIVE,
    WRA_AE_ACK,
    WRA_AE_CONDITION,
    WRA_AE_DATAAUX,
    WRA_AE_NUMKEYS
  };

  enum {
    WRA_SET_TAG = 0,
    WRA_GET_TAG,
    WRA_SET_U32,
    WRA_GET_U32,
    WRA_SET_U64,
    WRA_GET_U64,
    WRA_SET_S32,
    WRA_GET_S32,
    WRA_SET_S64,
    WRA_GET_S64,
    WRA_SET_FLOAT,
    WRA_GET_FLOAT,
    WRA_SET_DOUBLE,
    WRA_GET_DOUBLE,
    WRA_SET_BOOL,
    WRA_GET_BOOL
  };

enum {
  WRA_TM_DATA = 1,
  WRA_TM_ALARM = 2,
  WRA_TM_EVENT = 4
};

typedef struct wra_tm_hash_s
{
  char name[64]; 
  int id;
  int supported; 
} wra_tm_hash;

/* Debug Log APIs */
int wra_set_log_level(int level);
int wra_get_log_level(void);
const char *wra_get_log_level_str(int level);

  /* file storage and parsing utilities */

  /**
   *
   * @fn  void wra_set_media(wra_ctx *ctx, char *p)
   *
   * @brief this function sets the storage media on wra context. 
   *
   * @param [in] wra context    - pointer to wra_context
   * @param [in] path name      - pointer to a string
   * @return
   *  - None
   */
  void wra_set_media(wra_ctx *ctx, char *p);
  /**
   *
   * @fn  void wra_get_media(char *p)
   *
   * @brief this function get the storage path from wra context. 
   *
   * @param [in] void
   * @return 
   *    - return the a pointer to storage path or NULL
   */
  char *wra_get_media(wra_ctx *ctx);

  /**
   *
   * @fn  void wra_open_log(char *p, char *mode)
   *
   * @brief Open a file stream for logging. 
   *
   * @param [in] path name      - pointer to a string
   * @param [in] mode           - file open mode string
   * @return 
   *    - return 1 on success else 0
   */
int wra_open_log(wra_ctx *ctx, char *mode);
void wra_close_log(wra_ctx *ctx);
int wra_get_config(wra_ctx *ctx);
int wra_rm_config_dir(wra_ctx *ctx,char *dir_name, int recursive);
int wra_mk_config_dir(wra_ctx *ctx,char *dir_name);
char *wra_get_abspath(const char *path, char *f_name, char *ext);
char * wra_create_url(wra_ctx *ctx);
int wra_create_sn(char *buf, int buf_l);
void wra_set_network_status(char *status);
void wra_get_cur_time(AeTimeValue *pTime);

/* message queue related helper function*/
wra_status wra_client_mq_create(wra_client_ctx_t *ctx, int mq_idx, wra_mq_attr *mq_attr);

/**
    internal event handlers related to file up/download 
    */
void wra_set_registration_hooks();
void wra_set_dataitem_recv_hooks();
void wra_set_pingrate_update_hooks();
void wra_set_file_download_hooks();
void wra_set_file_upload_hooks();
AeError wra_file_upload_start(wra_ctx *ctx, char *f_name);
AeError wra_file_upload_start_ex(wra_ctx *ctx, char *f_name, char *param);

/* internal event helper functions */
wra_status wra_send_agent_start_event(wra_ctx *ctx, int isrestart, int ishard);
wra_status wra_send_agent_factory_reset_event(wra_ctx *ctx);
wra_status wra_send_reboot_event(wra_ctx *ctx);
wra_status wra_create_internal_event(wra_ctx *ctx);
wra_status wra_destroy_internal_event(wra_ctx *ctx);

/* subscription related function */
void wra_entry_reset(wra_sub_com_en_t *com, int table);
wra_status wra_entry_find_n_reset(wra_ctx *ctx,wra_sub_com_en_t *e,uint64_t cookie,long devid,int table);
int wra_send_file_req_dataitem(wra_ctx *wractx, const char *filename);
wra_status wra_delete_common_subscription(wra_ctx *ctx, uint64_t cookie,long devid,int table);


wra_static_inline void wra_srv_channel_close(wra_client_ctx_t *ctx,wra_ae_msg_t *snd_msg,int req_type,int srv_fd,wra_timestamp *tm)
{
    snd_msg->msg_type = req_type;
    if (req_type == APP_FILE_UNSUBSCRIBE_REQUEST)
    {
         snd_msg->data.file_msg.devid = ctx->iDeviceId;
         snd_msg->data.file_msg.server_channel = srv_fd;
    }
    if (req_type == APP_TM_UNSUBSCRIBE_REQUEST)
    {
        snd_msg->data.tm_msg.devid = ctx->iDeviceId;
        snd_msg->data.tm_msg.server_channel = srv_fd;
    }
    wra_msg_send((WRA_MSG_Q_ID)ctx->agent_msgQ,(char *)snd_msg,sizeof(*snd_msg),MSG_PRI_NORMAL, tm);
}

wra_static_inline int wra_post_data(wra_ctx *ctx,AeDRMQueuePriority prio,AeDRMDataItem *di)
{
    int rc = WRA_ERR_FAILED;
    if (ctx->wra_stat.wra_d_stat != WRA_RUNNING)
        return WRA_ERR_EAGAIN;

    AeMutexLock(&ctx->wra_lock);
    rc = AeDRMPostDataItem(ctx->iDeviceId, ctx->iServerId, prio,di);
    AeMutexUnlock(&ctx->wra_lock);
    return rc;
}

wra_static_inline int wra_post_alarm(wra_ctx *ctx,AeDRMQueuePriority prio,AeDRMAlarm *alarm)
{
    int rc = WRA_ERR_FAILED;
    if (ctx->wra_stat.wra_d_stat != WRA_RUNNING)
        return WRA_ERR_EAGAIN;

    AeMutexLock(&ctx->wra_lock);
    rc = AeDRMPostAlarm(ctx->iDeviceId, ctx->iServerId, prio,alarm);
    AeMutexUnlock(&ctx->wra_lock);
    return rc;
}

wra_static_inline int wra_post_event(wra_ctx *ctx,AeDRMQueuePriority prio,AeDRMEvent *e)
{
    int rc = WRA_ERR_FAILED;
    if (ctx->wra_stat.wra_d_stat != WRA_RUNNING)
        return WRA_ERR_EAGAIN;

    AeMutexLock(&ctx->wra_lock);
    rc = AeDRMPostEvent(ctx->iDeviceId, ctx->iServerId, prio,e);
    AeMutexUnlock(&ctx->wra_lock);
    return rc;
}
#if 0
#define WRA_LOG_OPEN(idt, opt) wra_log_open(idt, (opt))
#define WRA_LOG_CLOSE()        wra_log_close()
#define WRA_LOG(p, fmt, ...)   wra_log((p), fmt, ##__VA_ARGS__)
#define WRA_LOG_SET(mask)      wra_log_mask_set((mask))
#endif

#ifndef WRA_NO_IPC

/* #define DEBUG_WRA_RTP */
/* user space routine */
#define WRA_LOG(fmt, ...) printf(fmt, ##__VA_ARGS__)
#if defined(DEBUG_WRA_RTP)
#define WRA_DEBUG(fmt, ...) printf(fmt, ##__VA_ARGS__)
#else
#define WRA_DEBUG(fmt, ...) 
#endif


/*
 *===========================================================================
 *                     WRA_CLIENT_MSG_INIT 
 *===========================================================================
 * Initialize the msgQ message 
 */
#define WRA_CLIENT_MSG_INIT(ctx,msg,priority) \
  do { \
    msg.recv_msgQ = ctx->queue_t[WRA_STATUS_QUEUE].q_remote_add; \
    msg.data.pmsg.devid = ctx->iDeviceId; \
    msg.data.pmsg.srvid = ctx->iServerId; \
    msg.data.pmsg.prio = priority; \
  } while((0))

/*
 *===========================================================================
 *                     IF_WRA_SVR_REQ 
 *===========================================================================
 * Make a request to the agent server 
 */
#define IF_WRA_SVR_REQ(rc,ctx,msg,mq) \
  if((rc = wra_msg_send((WRA_MSG_Q_ID)(ctx)->agent_msgQ,(char *)&msg,sizeof(msg),MSG_PRI_NORMAL,NULL)) == WRA_SUCCESS) \
    if((rc = wra_msg_receive((WRA_MSG_Q_ID)(ctx)->queue_t[mq].q_addr,(char *)&msg, sizeof(msg),NULL,NULL)) == WRA_SUCCESS) \

/*
 *===========================================================================
 *                     WRA_SVR_MSG_SEND
 *===========================================================================
 * Send a response to the client 
 */
#define WRA_SVR_MSG_SEND(status,mqid,msg,tmo) \
  status = WRA_ERR_FAILED; \
  if(mqid != WRA_MSG_Q_ID_NULL) status = wra_msg_send(mqid,(char *)&msg,sizeof(msg),MSG_PRI_NORMAL,tmo) \

/*
 *===========================================================================
 *                     WRA_SVR_REG_RESP
 *===========================================================================
 * Send a response to the client 
 */
#define WRA_SVR_REG_RESP(op_status,send_status,mqid,msg,msgid,tmo) \
  msg.msg_type = msgid; \
  msg.status = op_status; \
  msg.recv_msgQ = (uint64_t)mqid; \
  WRA_SVR_MSG_SEND(send_status,mqid,msg,tmo) \

/*
 *===========================================================================
 *                     WRA_CLIENT_POST 
 *===========================================================================
 * Post to the agent server 
 */
#define WRA_CLIENT_POST(rc,resp,ctx,msg) \
  IF_WRA_SVR_REQ(rc,ctx,msg,WRA_STATUS_QUEUE) \
    if (msg.msg_type == APP_POST_RESPONSE ) \
        resp = msg.data.pmsg.status

/*
 *===========================================================================
 *                     WRA_ACTION_REGISTRATION 
 *===========================================================================
 * Register an action with the agent server 
 */
#define WRA_ACTION_REGISTRATION(rc,resp,ctx,msg) \
  IF_WRA_SVR_REQ(rc,ctx,msg,WRA_ACTION_QUEUE) \
    if (msg.msg_type == APP_ACTION_REGISTRATION_RESPONSE) \
        resp = msg.status 

/*
 *===========================================================================
 *                     WRA_DATA_REGISTRATION 
 *===========================================================================
 * Register for telemetry data with the agent server 
 */
#define WRA_DATA_REGISTRATION(rc,resp,ctx,msg) \
  IF_WRA_SVR_REQ(rc,ctx,msg,WRA_DATA_QUEUE) \
    if (msg.msg_type == APP_DATA_REGISTRATION_RESPONSE) \
        resp = msg.status 

#define WRA_GET_SUBSCRIPTION_ENTRY(ent_current,htbl,tbl_type, next_ix) \
	do{ \
        if ((tbl_type) == ACTION_SUBSCRIPTION_TABLE) \
            (ent_current) = (wra_sub_com_en_t *)&((wra_action_subscribe_tbl_t *)(htbl))->app_ex_en[next_ix]; \
        if ((tbl_type) == TM_SUBSCRIPTION_TABLE) \
            (ent_current) = (wra_sub_com_en_t *)&((wra_tm_subscribe_tbl_t *)(htbl))->tm_subs_ent[next_ix]; \
        if ((tbl_type) == FILE_SUBSCRIPTION_TABLE) \
            (ent_current) = (wra_sub_com_en_t *)&((wra_file_subscribe_tbl_t *)(htbl))->file_subs_ent[next_ix]; \
	}while((0))

#define WRA_GET_SUBSCRIPTION_LOCK(lock,htbl,tbl_type)\
    do{\
        if ((tbl_type) == ACTION_SUBSCRIPTION_TABLE)\
            (lock) = ((wra_action_subscribe_tbl_t *)(htbl))->lock;\
        if ((tbl_type) == TM_SUBSCRIPTION_TABLE)\
            (lock) = ((wra_tm_subscribe_tbl_t *)(htbl))->lock;\
        if ((tbl_type) == FILE_SUBSCRIPTION_TABLE)\
            (lock) = ((wra_file_subscribe_tbl_t *)(htbl))->lock;\
    }while((0))

wra_static_inline char* wra_msgq_prefix(int msgq_type)
{
	switch(msgq_type)
	{
	case WRA_STATUS_QUEUE:
		return "wrac_s";
	case WRA_ACTION_QUEUE:
		return "wrac_a";
	case WRA_DATA_QUEUE:
		return "wrac_d";
	case WRA_FILE_QUEUE:
		return "wrac_f";
	default:
		return NULL;
	}
}

wra_static_inline int wra_queue_name_to_delete_handle(int ix)
{
	switch(ix)
	{
	case WRA_STATUS_QUEUE:
		return DATA_HANDLE_DELETE_REQUEST;
	case WRA_ACTION_QUEUE:
		return APP_HANDLE_DELETE_REQUEST;
	case WRA_DATA_QUEUE:
		return TM_HANDLE_DELETE_REQUEST;
	case WRA_FILE_QUEUE:
		return FILE_HANDLE_DELETE_REQUEST;
	default:
		return -1;
	}
}

/*!
 *===========================================================================
 *                    wra_entry_size_get
 *===========================================================================
 */

wra_static_inline size_t wra_entry_size_get(int table)
{
	switch(table)
	{
	case ACTION_SUBSCRIPTION_TABLE:
		return sizeof(wra_app_ex_entry_t);
	case FILE_SUBSCRIPTION_TABLE:
		return sizeof(wra_file_susbcribe_entry_t);
	case TM_SUBSCRIPTION_TABLE:
		return sizeof(wra_tm_subscribe_entry_t);
	}
	return 0;
}
/*!
 *===========================================================================
 *                    wra_entry_get_keyname: not reentrant 
 *===========================================================================
 */
wra_static_inline char * wra_entry_get_keyname(wra_sub_com_en_t *com, int table)
{
	wra_file_susbcribe_entry_t * file_ent = NULL;
	
    if (table == ACTION_SUBSCRIPTION_TABLE)
        return ((wra_app_ex_entry_t *)com)->appname;
    if (table == FILE_SUBSCRIPTION_TABLE)
    {
       file_ent = (wra_file_susbcribe_entry_t *)com;
       if (file_ent->is_file_upload)
           return file_ent->path;
       return file_ent->filename; 
    }
    if (table == TM_SUBSCRIPTION_TABLE)
         return ((wra_tm_subscribe_entry_t *)com)->name;
    return NULL;
}

/*!
 *===========================================================================
 *                    wra_get_subscription_htbl
 *===========================================================================
 */
wra_static_inline wra_subscribe_tbl_t wra_get_subscription_htbl(wra_ctx *ctx, int type) {

    if (ctx->app_ex_htbl.init)
    {
        if (type == ACTION_SUBSCRIPTION_TABLE)
            return (wra_subscribe_tbl_t)&ctx->app_ex_htbl.act_tbl;
        if (type == TM_SUBSCRIPTION_TABLE)
            return (wra_subscribe_tbl_t)&ctx->app_ex_htbl.tm_tbl;
        if (type == FILE_SUBSCRIPTION_TABLE)
            return (wra_subscribe_tbl_t)&ctx->app_ex_htbl.file_tbl;

    }
    return NULL;
}

/*!
 *===========================================================================
 *                    wra_entry_key_compare : not thread safe , consider taking lock before calling this function
 *===========================================================================
 */
wra_static_inline wra_status wra_entry_key_compare(const wra_sub_com_en_t *e,int table,const char *key1,const char *key2,const char *key3)
{
     if (table == ACTION_SUBSCRIPTION_TABLE)
     {
          (void)key2;
          (void)key3;
          if(strcmp( ((wra_app_ex_entry_t *)e)->appname, key1) == 0)
              return WRA_SUCCESS;
     }
     if (table == FILE_SUBSCRIPTION_TABLE)
     {
         (void)key2;
         (void)key3;
         if( (((wra_file_susbcribe_entry_t *)e)->is_file_upload==0) && 
                 strcmp( ((wra_file_susbcribe_entry_t *)e)->filename, key1) == 0)
             return WRA_SUCCESS;
         if( (((wra_file_susbcribe_entry_t *)e)->is_file_upload==1) && 
                  strcmp( ((wra_file_susbcribe_entry_t *)e)->path, key1) == 0)
            return WRA_SUCCESS;
     }
     if (table == TM_SUBSCRIPTION_TABLE)
     {
       if(strcmp( ((wra_tm_subscribe_entry_t *)e)->name, key1) == 0)
         return WRA_SUCCESS;
     }
     return WRA_ERR_FAILED;
}
/*!
 *===========================================================================
 *                    wra_entry_search: not reentrant
 *===========================================================================
 */
wra_static_inline wra_sub_com_en_t *wra_entry_search(wra_ctx *ctx, 
                                                 const char *key1, const char *key2, const char *key3, int table, int build_index) {
    unsigned int i;
    wra_sub_com_en_t *ent = NULL;
    	   for (i=0; i < WRA_AP_EX_REG_MAX; i++) {
    		   WRA_GET_SUBSCRIPTION_ENTRY(ent,wra_get_subscription_htbl(ctx,table),table, i);
    		   if (ent->used && (wra_entry_key_compare(ent,table,key1,key2,key3)== WRA_SUCCESS) ) {
    			   if (build_index == TRUE)
    			       ent->this_item_ix = i;
            	   return ent;
    		   }
    	   }
    return NULL;
}

#endif /* WRA_NO_IPC */

#ifdef __cplusplus
}
#endif

#endif /* WRA_PRIVATE_H */
