--[[
LuCI - Lua Configuration Interface

Copyright 2008 Steven Barth <steven@midlink.org>
Copyright 2008-2011 Jo-Philipp Wich <xm@subsignal.org>

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

	http://www.apache.org/licenses/LICENSE-2.0

$Id$
]]--

module("luci.controller.admin.system", package.seeall)

function index()
	local admin_config={["hidesavebtn"]=true}

	entry({"admin", "system"}, alias("admin", "system", "system"), _("System"), 30).index = true
	entry({"admin", "system", "system"}, cbi("admin_system/system"), _("System"), 1)
	entry({"admin", "system", "clock_status"}, call("action_clock_status"))

	entry({"admin", "system", "admin"}, cbi("admin_system/admin", admin_config), _("Administration"), 2)

	if nixio.fs.access("/usr/bin/smart") then
		entry({"admin", "system", "packages"}, call("action_packages"), _("Software"), 10)
		entry({"admin", "system", "packages", "ready"}, call("action_packages_ready"))
		entry({"admin", "system", "packages", "smart"}, call("action_packages_settings"))
	end

	entry({"admin", "system", "startup"}, form("admin_system/startup"), _("Systemd Units"), 45)
	entry({"admin", "system", "crontab"}, form("admin_system/crontab"), _("Scheduled Tasks"), 46)

	entry({"admin", "system", "timesyncd"}, cbi("admin_system/timesyncd"), _("Time Synchronization"), 47)

	if nixio.fs.access("/etc/config/fstab") then
		entry({"admin", "system", "fstab"}, cbi("admin_system/fstab"), _("Mount Points"), 50)
		entry({"admin", "system", "fstab", "mount"}, cbi("admin_system/fstab/mount"), nil).leaf = true
		entry({"admin", "system", "fstab", "swap"},  cbi("admin_system/fstab/swap"),  nil).leaf = true
	end

	entry({"admin", "system", "tpm"}, cbi("admin_system/tpm"), _("TPM"), 60)
	entry({"admin", "system", "tpm_apply"}, call("action_tpm_apply"))

	-- if nixio.fs.access("/sys/class/leds") then
	-- 	entry({"admin", "system", "leds"}, cbi("admin_system/leds"), _("<abbr title=\"Light Emitting Diode\">LED</abbr> Configuration"), 60)
	-- end

	--entry({"admin", "system", "flashops"}, call("action_flashops"), _("Backup / Flash Firmware"), 70)
	--entry({"admin", "system", "flashops", "backupfiles"}, form("admin_system/backupfiles"))
	entry({"admin", "system", "backup_restore"}, call("action_backup_restore"), _("Backup / Restore"), 80)

	entry({"admin", "system", "reboot"}, call("action_reboot"), _("Reboot"), 90)
end

function action_tpm_apply()
	local tcsd_status = luci.http.formvalue("status")
	if tcsd_status == luci.i18n.translate("Running") then
		luci.sys.systemd.stop("tcsd")
	else
		luci.sys.systemd.start("tcsd")
		luci.util.exec("sleep 0.5")
	end

	luci.http.redirect(luci.dispatcher.build_url(
	"admin", "system", "tpm"
	))
end

function action_clock_status()
	local set = luci.http.formvalue("set")
	if set ~= nil then
		luci.sys.call("date -s '%s'" %{ set })
	end

	luci.http.prepare_content("application/json")
	luci.http.write_json({ timestring = os.date("%c") })
end

function action_packages()
	local smart   = require("luci.model.smart")
	local sys     = require("luci.sys")
	local submit  = luci.http.formvalue("submit")
	local changes = false
	local install = { }
	local remove  = { }
	local stdout  = { "" }
	local stderr  = { "" }
	local out, err

	-- Display
	local display = luci.http.formvalue("display") or "installed"

	-- Letter
	local letter = string.byte(luci.http.formvalue("letter") or "A", 1)
	letter = (letter == 35 or (letter >= 65 and letter <= 90)) and letter or 65

	-- Search query
	local query = luci.http.formvalue("query")
	query = (query ~= '') and query or nil


	-- Packets to be installed
	local ninst = submit and luci.http.formvalue("install")
	local nver = submit and luci.http.formvalue("version")
	local uinst = nil

	-- Install from URL
	local url = luci.http.formvalue("url")
	if url and url ~= '' and submit then
		uinst = url
	end

	-- Do install
	if ninst then
		if nver then
			install[ninst], out, err = smart.install(ninst.."/"..nver)
		else
			install[ninst], out, err = smart.install(ninst)
		end
		stdout[#stdout+1] = out
		stderr[#stderr+1] = err
		changes = true
	end

	if uinst then
		local pkg
		for pkg in luci.util.imatch(uinst) do
			install[uinst], out, err = smart.install(pkg)
			stdout[#stdout+1] = out
			stderr[#stderr+1] = err
			changes = true
		end
	end

	-- Remove packets
	local rem = submit and luci.http.formvalue("remove")
	if rem then
		remove[rem], out, err = smart.remove(rem)
		stdout[#stdout+1] = out
		stderr[#stderr+1] = err
		changes = true
	end


	-- Update all packets
	local update = luci.http.formvalue("update")
	if update then
		update, out, err = smart.update()
		stdout[#stdout+1] = out
		stderr[#stderr+1] = err
	end


	-- Upgrade all packets
	local upgrade = luci.http.formvalue("upgrade")
	if upgrade then
		upgrade, out, err = smart.upgrade()
		stdout[#stdout+1] = out
		stderr[#stderr+1] = err
	end


	-- List state
	local no_lists = true
	local old_lists = false
	local tmp = nixio.fs.dir(sys.package.cache_path)
	if tmp then
		for tmp in tmp do
			no_lists = false
			tmp = nixio.fs.stat(sys.package.cache_path..tmp)
			if tmp and tmp.mtime < (os.time() - (24 * 60 * 60)) then
				old_lists = true
				break
			end
		end
	end


	luci.template.render("admin_system/packages", {
		display   = display,
		letter    = letter,
		query     = query,
		install   = install,
		remove    = remove,
		update    = update,
		upgrade   = upgrade,
		no_lists  = no_lists,
		old_lists = old_lists,
		stdout    = table.concat(stdout, ""),
		stderr    = table.concat(stderr, "")
	})

	-- Remove index cache
	if changes then
		nixio.fs.unlink("/tmp/luci-indexcache")
	end
end

function action_packages_ready()
	local smart = require("luci.model.smart")
	ready = smart.ready()

	luci.http.prepare_content("application/json")
	luci.http.write_json({ ready = ready })
end

function action_packages_settings()
	local smart = require("luci.model.smart")
	local action = luci.http.formvalue("action")
	local channel = luci.http.formvalue("channel")
	local chntype = luci.http.formvalue("chntype")
	local baseurl = luci.http.formvalue("baseurl")

	luci.template.render("admin_system/smart", {
		action  = action,
		channel = channel,
		chntype = chntype,
		baseurl = baseurl
	})
end

function action_flashops()
	local sys = require "luci.sys"
	local fs  = require "luci.fs"

	local upgrade_avail = nixio.fs.access("/lib/upgrade/platform.sh")
	local reset_avail   = os.execute([[grep '"rootfs_data"' /proc/mtd >/dev/null 2>&1]]) == 0

	local restore_cmd = "tar -xzC/ >/dev/null 2>&1"
	local backup_cmd  = "sysupgrade --create-backup - 2>/dev/null"
	local image_tmp   = "/tmp/firmware.img"

	local function image_supported()
		-- XXX: yay...
		return ( 0 == os.execute(
			". /lib/functions.sh; " ..
			"include /lib/upgrade; " ..
			"platform_check_image %q >/dev/null"
				% image_tmp
		) )
	end

	local function image_checksum()
		return (luci.sys.exec("md5sum %q" % image_tmp):match("^([^%s]+)"))
	end

	local function storage_size()
		local size = 0
		if nixio.fs.access("/proc/mtd") then
			for l in io.lines("/proc/mtd") do
				local d, s, e, n = l:match('^([^%s]+)%s+([^%s]+)%s+([^%s]+)%s+"([^%s]+)"')
				if n == "linux" or n == "firmware" then
					size = tonumber(s, 16)
					break
				end
			end
		elseif nixio.fs.access("/proc/partitions") then
			for l in io.lines("/proc/partitions") do
				local x, y, b, n = l:match('^%s*(%d+)%s+(%d+)%s+([^%s]+)%s+([^%s]+)')
				if b and n and not n:match('[0-9]') then
					size = tonumber(b) * 1024
					break
				end
			end
		end
		return size
	end


	local fp
	luci.http.setfilehandler(
		function(meta, chunk, eof)
			if not fp then
				if meta and meta.name == "image" then
					fp = io.open(image_tmp, "w")
				else
					fp = io.popen(restore_cmd, "w")
				end
			end
			if chunk then
				fp:write(chunk)
			end
			if eof then
				fp:close()
			end
		end
	)

	if luci.http.formvalue("backup") then
		--
		-- Assemble file list, generate backup
		--
		local reader = ltn12_popen(backup_cmd)
		luci.http.header('Content-Disposition', 'attachment; filename="backup-%s-%s.tar.gz"' % {
			luci.sys.hostname(), os.date("%Y-%m-%d")})
		luci.http.prepare_content("application/x-targz")
		luci.ltn12.pump.all(reader, luci.http.write)
	elseif luci.http.formvalue("restore") then
		--
		-- Unpack received .tar.gz
		--
		local upload = luci.http.formvalue("archive")
		if upload and #upload > 0 then
			luci.template.render("admin_system/applyreboot")
			luci.sys.reboot()
		end
	elseif luci.http.formvalue("image") or luci.http.formvalue("step") then
		--
		-- Initiate firmware flash
		--
		local step = tonumber(luci.http.formvalue("step") or 1)
		if step == 1 then
			if image_supported() then
				luci.template.render("admin_system/upgrade", {
					checksum = image_checksum(),
					storage  = storage_size(),
					size     = nixio.fs.stat(image_tmp).size,
					keep     = (not not luci.http.formvalue("keep"))
				})
			else
				nixio.fs.unlink(image_tmp)
				luci.template.render("admin_system/flashops", {
					reset_avail   = reset_avail,
					upgrade_avail = upgrade_avail,
					image_invalid = true
				})
			end
		--
		-- Start sysupgrade flash
		--
		elseif step == 2 then
			local keep = (luci.http.formvalue("keep") == "1") and "" or "-n"
			luci.template.render("admin_system/applyreboot", {
				title = luci.i18n.translate("Flashing..."),
				msg   = luci.i18n.translate("The system is flashing now.<br /> DO NOT POWER OFF THE DEVICE!<br /> Wait a few minutes before you try to reconnect. It might be necessary to renew the address of your computer to reach the device again, depending on your settings."),
				addr  = (#keep > 0) and "192.168.1.1" or nil
			})
			fork_exec("killall dropbear uhttpd; sleep 1; /sbin/sysupgrade %s %q" %{ keep, image_tmp })
		end
	elseif reset_avail and luci.http.formvalue("reset") then
		--
		-- Reset system
		--
		luci.template.render("admin_system/applyreboot", {
			title = luci.i18n.translate("Erasing..."),
			msg   = luci.i18n.translate("The system is erasing the configuration partition now and will reboot itself when finished."),
			addr  = "192.168.1.1"
		})
		fork_exec("killall dropbear uhttpd; sleep 1; mtd -r erase rootfs_data")
	else
		--
		-- Overview
		--
		luci.template.render("admin_system/flashops", {
			reset_avail   = reset_avail,
			upgrade_avail = upgrade_avail
		})
	end
end

function action_backup_restore()
	local sys = require "luci.sys"
	local fs  = require "luci.fs"

	local upgrade_avail = nixio.fs.access("/lib/upgrade/platform.sh")
	local reset_avail   = os.execute([[grep '"rootfs_data"' /proc/mtd >/dev/null 2>&1]]) == 0

	---local restore_cmd = "tar -xzC/ >/dev/null 2>&1"
	local restore_cmd = "env -i /www/cgi-bin/wridp-restore.sh - 2>&1"
	local restore_file = "/tmp/wridp-restore.tgz"

	--local backup_cmd  = "wridp-backup.sh - 2>&1"
	local backup_cmd = "env -i /www/cgi-bin/wridp-backup.sh 2>&1"
	local backup_file = "/tmp/wridp-backup.tgz"


	if luci.http.formvalue("backup") then
		--
		-- Assemble file list, generate backup
		--

		--luci.sys.call("env -i /sbin/ifdown %q >/dev/null 2>/dev/null" % iface)
		local backup_output = luci.sys.exec(backup_cmd)

		local tmp = nixio.fs.stat(backup_file)
		if tmp and tmp.mtime > (os.time() - ( 60 * 60)) then
			-- backup succeed
			local reader = ltn12_popen("cat " .. backup_file)
			luci.http.header('Content-Disposition', 'attachment; filename="backup-%s-%s.tar.gz"' % {
				luci.sys.hostname(), os.date("%Y-%m-%d")})
			luci.http.header('Cache-Control', 'no-store, no-cache, max-age=0, must-revalidate')
			luci.http.prepare_content("application/x-targz")
			luci.ltn12.pump.all(reader, luci.http.write)
			--remove the temporary archive
			luci.sys.call("rm /tmp/wridp-backup.tgz >/dev/null 2>/dev/null")

		else
			-- backup failed

			luci.template.render("admin_system/backup_restore", {
						msg   = "Backup fails! \n" .. backup_output .. "\n"
					})

			return;

			--luci.http.prepare_content("text/plain")
			--luci.http.write("Failure message:\n")
			--luci.http.write(backup_output)

			--luci.http.prepare_content("application/json")
			--luci.http.write_json(backup_output)

			--luci.http.header('Content-Disposition', 'attachment; filename="backup-failed-%s-%s.log"' % {
			--      luci.sys.hostname(), os.date("%Y-%m-%d")})
			--luci.http.prepare_content("text/plain")
			--luci.ltn12.pump.all(backup_output, luci.http.write)
		end


	elseif luci.http.formvalue("restore") then
		--
		-- Unpack received .tar.gz
		--
		local upload = luci.http.formvalue("archive")
		local reboot_immediately = luci.http.formvalue("reboot_checked")
		if upload and #upload > 0 then
			nixio.fs.writefile(restore_file, upload)
			local tmp = nixio.fs.stat(restore_file)
			if tmp and tmp.mtime > (os.time() - (60 * 60)) then
				-- backup succeed

				local restore_output = luci.sys.exec(restore_cmd)


				--if reboot_immediately and reboot_immediately == "checked" then
				if reboot_immediately then

					luci.template.render("admin_system/backup_restore", {
						msg   = "Restore is done! \nWill reboot the system!\n"
					})
					luci.sys.reboot()
					return
				else

					luci.template.render("admin_system/backup_restore", {
						msg   = "Restore is done!"
					})

					return
				end
			end
		else
			luci.template.render("admin_system/backup_restore", {
						msg   = "Failed to restore!\n Please make sure you select file for upload, and system storage has enough space for temporary usage.\n"
					})

			return
		end

		luci.template.render("admin_system/backup_restore")
	else
		--
		-- Overview
		--
		luci.template.render("admin_system/backup_restore", {
			reset_avail   = reset_avail,
			upgrade_avail = upgrade_avail
		})
	end
end

function action_passwd()
	local p1 = luci.http.formvalue("pwd1")
	local p2 = luci.http.formvalue("pwd2")
	local stat = nil

	if p1 or p2 then
		if p1 == p2 then
			stat = luci.sys.user.setpasswd("root", p1)
		else
			stat = 10
		end
	end

	luci.template.render("admin_system/passwd", {stat=stat})
end

function action_reboot()
	local reboot = luci.http.formvalue("reboot")
	luci.template.render("admin_system/reboot", {reboot=reboot})
	if reboot then
		fork_exec("sleep 1; reboot");
	end
end

function fork_exec(command)
	local pid = nixio.fork()
	if pid > 0 then
		return
	elseif pid == 0 then
		-- change to root dir
		nixio.chdir("/")

		-- patch stdin, out, err to /dev/null
		local null = nixio.open("/dev/null", "w+")
		if null then
			nixio.dup(null, nixio.stderr)
			nixio.dup(null, nixio.stdout)
			nixio.dup(null, nixio.stdin)
			if null:fileno() > 2 then
				null:close()
			end
		end

		-- replace with target command
		nixio.exec("/bin/sh", "-c", command)
	end
end

function ltn12_popen(command)

	local fdi, fdo = nixio.pipe()
	local pid = nixio.fork()

	if pid > 0 then
		fdo:close()
		local close
		return function()
			local buffer = fdi:read(2048)
			local wpid, stat = nixio.waitpid(pid, "nohang")
			if not close and wpid and stat == "exited" then
				close = true
			end

			if buffer and #buffer > 0 then
				return buffer
			elseif close then
				fdi:close()
				return nil
			end
		end
	elseif pid == 0 then
		nixio.dup(fdo, nixio.stdout)
		fdi:close()
		fdo:close()
		nixio.exec("/bin/sh", "-c", command)
	end
end
