--[[
LuCI - Lua Configuration Interface

Copyright 2008 Steven Barth <steven@midlink.org>
Copyright 2010-2012 Jo-Philipp Wich <xm@subsignal.org>
Copyright 2010 Manuel Munz <freifunk at somakoma dot de>

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

	http://www.apache.org/licenses/LICENSE-2.0

$Id$
]]--

require "luci.fs"
require "luci.sys"
require "luci.util"

local units = { }

for _, unit in pairs(luci.sys.systemd.units()) do
	local enabled = luci.sys.systemd.enabled(unit)

	units[unit] = {
		unit    = unit,
		enabled = enabled,
	}
end


m = SimpleForm("systemdmgr", translate("Systemd Units"), translate("You can enable or disable installed systemd units here. Changes will applied after a device reboot.<br /><strong>Warning: If you disable essential systemd units like \"netifd\", your device might become inaccessible!</strong>"))
m.reset = false
m.submit = false


s0 = m:section(Table, units, translate("Systemd Units"))

n0 = s0:option(DummyValue, "unit", translate("Unit"))


e0 = s0:option(Button, "endisable", translate("Enable/Disable"))

e0.render = function(self, section, scope)
	if units[section].enabled then
		self.title = translate("Enabled")
		self.inputstyle = "save"
	else
		self.title = translate("Disabled")
		self.inputstyle = "reset"
	end

	Button.render(self, section, scope)
end

e0.write = function(self, section)
	if units[section].enabled then
		units[section].enabled = false
		luci.sys.systemd.disable(section)
	else
		units[section].enabled = true
		luci.sys.systemd.enable(section)
	end
end


start0 = s0:option(Button, "start", translate("Start"))
start0.inputstyle = "apply"
start0.write = function(self, section)
	luci.sys.systemd.start(section)
end

restart0 = s0:option(Button, "restart", translate("Restart"))
restart0.inputstyle = "reload"
restart0.write = function(self, section)
	luci.sys.systemd.restart(section)
end

stop0 = s0:option(Button, "stop", translate("Stop"))
stop0.inputstyle = "remove"
stop0.write = function(self, section)
	luci.sys.systemd.stop(section)
end

if not nixio.fs.access("/sys/kernel/security/ima") then
	f = SimpleForm("rc", translate("Local Startup"),
		translate("This is the content of /etc/rc.local. Insert your own commands here (in front of 'exit 0') to execute them at the end of the boot process."))

	t = f:field(TextValue, "rcs")
	t.rmempty = true
	t.rows = 20

	function t.cfgvalue()
		return luci.fs.readfile("/etc/rc.local") or ""
	end

	function f.handle(self, state, data)
		if state == FORM_VALID then
			if data.rcs then
				luci.fs.writefile("/etc/rc.local", data.rcs:gsub("\r\n", "\n"))
			end
		end
		return true
	end
end

return m, f
