--[[
LuCI - Lua Configuration Interface

Copyright 2008 Steven Barth <steven@midlink.org>
Copyright 2008 Jo-Philipp Wich <xm@leipzig.freifunk.net>

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

	http://www.apache.org/licenses/LICENSE-2.0

$Id$
]]--
local util = require('luci.util')
local fs = require('luci.fs')
local sys = require('luci.sys')

local servers = {}
local servers_hide = {}
local fallbacks = {}
local fallbacks_hide = {}
local lines = util.execi("cat /etc/systemd/timesyncd.conf")
for line in lines do
  local s, k, v = line:match("^ *(#*)(%S+)=(.*)")
  if k == "NTP" then
    if s != '' then
      v:gsub("%s*(%S+)", function(c) servers_hide[#servers_hide+1] = c end)
    else
      v:gsub("%s*(%S+)", function(c) servers[#servers+1] = c end)
    end
  elseif k == "FallbackNTP" then
    if s != '' then
      v:gsub("%s*(%S+)", function(c) fallbacks_hide[#fallbacks_hide+1] = c end)
    else
      v:gsub("%s*(%S+)", function(c) fallbacks[#fallbacks+1] = c end)
    end
  end
end


f = SimpleForm("timesyncd", translate("Time Synchronization"), translate("Synchronizes the system time"))
f.reset  = false
f.readinput = false

o = f:field(Flag, "enable", translate("Enable Time Synchronization"))
o.rmempty = false
o.enabled = true
o.disabled = false

function o.cfgvalue(self)
  return sys.init.enabled("systemd-timesyncd")
end

function o.write(self, section, value)
  if value == self.enabled then
    sys.init.enable("systemd-timesyncd")
    sys.init.start("systemd-timesyncd")
  else
    sys.init.stop("systemd-timesyncd")
    sys.init.disable("systemd-timesyncd")
  end
end

for i=1,4 do
  local label = " "
  if i == 1 then label = translate("NTP Servers") end
  n = f:field(Value, "server", label)
  n.datatype = "host"
  n.optional = false
  n.rmempty = false
  if servers[i] == nil or servers[i] == '' then
    n.placeholder = servers_hide[i]
  end

  function n.cfgvalue(self, section)
    return servers[i]
  end

  function n.write(self, section, value)
    servers[i] = value[i]
  end
end

for i=1,4 do
  local label = " "
  if i == 1 then label = translate("Fallback NTP Servers") end
  n = f:field(Value, "fallback_server", label)
  n.datatype = "host"
  n.optional = false
  n.rmempty = false
  if fallbacks[i] == nil or fallbacks[i] == '' then
    n.placeholder = fallbacks_hide[i]
  end

  function n.cfgvalue(self, section)
    return fallbacks[i]
  end

  function n.write(self, section, value)
    fallbacks[i] = value[i]
  end
end

function f.handle(self, state, data)
  if state == FORM_VALID then
    local output = ''
    local lines = util.execi("cat /etc/systemd/timesyncd.conf")
    for line in lines do
      local k, v = line:match("^ *(%S+)= *(.*)")
      if k != "NTP" and k != "FallbackNTP" then
        output = output..line.."\n"
      end
    end
    -- NTP Servers
    output = output.."NTP="
    for i,v in ipairs(servers) do
      if i != 1 then output = output.." " end
      output = output..v
    end
    output = output.."\n"
    -- Fallback NTP Servers
    output = output.."FallbackNTP="
    for i,v in ipairs(fallbacks) do
      if i != 1 then output = output.." " end
      output = output..v
    end
    output = output.."\n"
    fs.writefile("/etc/systemd/timesyncd.conf", output)
    return true
  end
end

return f
