--[[
LuCI - Lua Configuration Interface

(c) 2008-2011 Jo-Philipp Wich <xm@subsignal.org>
(c) 2008 Steven Barth <steven@midlink.org>

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

http://www.apache.org/licenses/LICENSE-2.0

]]--

local os   = require "os"
local io   = require "io"
local fs   = require "nixio.fs"
local util = require "luci.util"
local sys  = require "luci.sys"

local type  = type
local pairs = pairs
local ipairs = ipairs
local error = error
local table = table

local use_cache = true
local cache_files = {}

local smart = "smart --quiet"
local icfg = "/etc/smart.conf"

local all_cache = "all.cache"
local installed_cache = "installed.cache"

cache_files["query --show-format='$name - $version - $summary\\n'"] = all_cache
cache_files["query --installed --show-format='$name - $version - $summary\\n'"] = installed_cache

--- LuCI OPKG call abstraction library
module "luci.model.smart"


-- Internal action function
local function _action(cmd, ...)
	if fs.access("/tmp/luci-runcache") then
		return 255, "", "smart is busy, please retry after a while."
	end

	local pkg = ""
	for k, v in pairs({...}) do
		local pn, pv = v:gsub("'", ""):match("([^/]+)/([^/]+)")
		if pv then
			pkg = pkg .. " '" .. pn .. "-" .. pv .. "'"
		else
			pkg = pkg .. " '" .. v:gsub("'", "") .. "'"
		end
	end

	local c = "%s %s %s >/tmp/smart.stdout 2>/tmp/smart.stderr" %{ smart, cmd, pkg }
	local r = os.execute(c)
	local e = fs.readfile("/tmp/smart.stderr")
	local o = fs.readfile("/tmp/smart.stdout")

	fs.unlink("/tmp/smart.stderr")
	fs.unlink("/tmp/smart.stdout")

	if r == 0 then
		if cmd == "remove" then
			for k, v in pairs({...}) do
				os.execute("sed -i '/^"..v:gsub("'", "").." - /d' "..sys.package.cache_path..installed_cache);
			end
		elseif cmd == "install" then
			for k, v in pairs({...}) do
				local pn, pv = v:gsub("'", ""):match("([^/]+)/([^/]+)")
				if pv then
					os.execute("sed -n '/^"..pn.." - "..pv.." - /p' "..sys.package.cache_path..all_cache.." >> "..sys.package.cache_path..installed_cache);
				else
					os.execute("sed -n '/^"..v:gsub("'", "").." - /p' "..sys.package.cache_path..all_cache.." >> "..sys.package.cache_path..installed_cache);
				end
			end
		end
	end

	return r, o or "", e or ""
end

-- Internal parser function
local function _parselist(rawdata)
	if type(rawdata) ~= "function" then
		error("OPKG: Invalid rawdata given")
	end

	local data = {}
	local c = {}
	local l = nil

	for line in rawdata do
		if line:sub(1, 1) ~= " " then
			local key, val = line:match("(.-): ?(.*)%s*")

			if key and val then
				if key == "Package" then
					c = {Package = val}
					data[val] = c
				elseif key == "Status" then
					c.Status = {}
					for j in val:gmatch("([^ ]+)") do
						c.Status[j] = true
					end
				else
					c[key] = val
				end
				l = key
			end
		else
			-- Multi-line field
			c[l] = c[l] .. "\n" .. line
		end
	end

	return data
end

-- Internal lookup function
local function _lookup(act, pkg)
	if fs.access("/tmp/luci-runcache") then
		return 255, "", "smart is busy, please retry after a while."
	end

	local cmd = smart .. " " .. act
	if pkg then
		cmd = cmd .. " '" .. pkg:gsub("'", "") .. "'"
	end

	-- OPKG sometimes kills the whole machine because it sucks
	-- Therefore we have to use a sucky approach too and use
	-- tmpfiles instead of directly reading the output
	local tmpfile = os.tmpname()
	os.execute(cmd .. (" >%s 2>/dev/null" % tmpfile))

	local data = _parselist(io.lines(tmpfile))
	os.remove(tmpfile)
	return data
end

function ready()
	if fs.access("/tmp/luci-runcache") then
		return false
	end

	return true
end

--- Return information about installed and available packages.
-- @param pkg Limit output to a (set of) packages
-- @return Table containing package information
function info(pkg)
	return _lookup("info", pkg)
end

--- Return the package status of one or more packages.
-- @param pkg Limit output to a (set of) packages
-- @return Table containing package status information
function status(pkg)
	return _lookup("info", pkg)
end

--- Install one or more packages.
-- @param ... List of packages to install
-- @return Boolean indicating the status of the action
-- @return OPKG return code, STDOUT and STDERR
function install(...)
	return _action("install", ...)
end

--- Determine whether a given package is installed.
-- @param pkg Package
-- @return Boolean
function installed(pkg)
	local p = status(pkg)[pkg]
	return (p and p.Status and p.Status.installed)
end

--- Remove one or more packages.
-- @param ... List of packages to install
-- @return Boolean indicating the status of the action
-- @return OPKG return code, STDOUT and STDERR
function remove(...)
	return _action("remove", ...)
end

--- Update package lists.
-- @return Boolean indicating the status of the action
-- @return OPKG return code, STDOUT and STDERR
function update()
	--local r, o, e = _action("update")
	if fs.access("/tmp/luci-runcache") then
		return 255, "", "smart is busy, please retry after a while."
	end

	local cmd = smart .. " " .. "update\n"
	local action
	local cache
	local pat

	os.execute("mkdir -p "..sys.package.cache_path)
	for action, cache in pairs(cache_files) do
		cmd = cmd..smart.." "..action ..
			(pat and (" '%s'" % pat:gsub("'", "")) or "") ..
			" >"..sys.package.cache_path..cache.."\n"
		--os.execute(cmd.." >"..sys.package.cache_path..cache)
		if cache == all_cache then
			cmd = "smart channel --disable rpmsys\n"..cmd.."smart channel --enable rpmsys\n"
		end
	end
	cmd = cmd.."rm -f /tmp/luci-runcache\n"

	fs.writefile("/tmp/luci-runcache", cmd)
	os.execute("sh /tmp/luci-runcache &")

	return 0, "Package lists are updating, please wait for a while.", ""
end

--- Upgrades all installed packages.
-- @return Boolean indicating the status of the action
-- @return OPKG return code, STDOUT and STDERR
function upgrade()
	return _action("upgrade")
end

-- List helper
function _list(action, pat, cb)
	local cmd = smart .. " " .. action ..
		(pat and (" '%s'" % pat:gsub("'", "")) or "")
	local regex = ".*"
	if pat then
		regex = pat:gsub('\*', '.*'):lower()
	end
	if use_cache then
		if cache_files[action] then
			cmd = "cat " .. sys.package.cache_path..cache_files[action].." 2>/dev/null| awk -F\" - \" '$1 ~ /^"..regex.."/'"
		else
			cmd = ""
		end
	end

	local fd = io.popen(cmd)

	if fd then
		local name, version, desc
		while true do
			local line = fd:read("*l")
			if not line then break end

			name, version, desc = line:match("^(.-) %- (.-) %- (.+)")

			if not name then
				name, version = line:match("^(.-) %- (.+)")
				desc = ""
			end

			if name then
				cb(name, version, desc)
			end

			name    = nil
			version = nil
			desc    = nil
		end

		fd:close()
	end
end

--- List all packages known to smart.
-- @param pat	Only find packages matching this pattern, nil lists all packages
-- @param cb	Callback function invoked for each package, receives name, version and description as arguments
-- @return	nothing
function list_all(pat, cb)
	_list("query --show-format='$name - $version - $summary\\n'", pat, cb)
end

--- List installed packages.
-- @param pat	Only find packages matching this pattern, nil lists all packages
-- @param cb	Callback function invoked for each package, receives name, version and description as arguments
-- @return	nothing
function list_installed(pat, cb)
	_list("query --installed --show-format='$name - $version - $summary\\n'", pat, cb)
end

--- Find packages that match the given pattern.
-- @param pat	Find packages whose names or descriptions match this pattern, nil results in zero results
-- @param cb	Callback function invoked for each patckage, receives name, version and description as arguments
-- @return	nothing
function find(pat, cb)
	_list("query --show-format='$name - $version - $summary\\n'", pat, cb)
end


--- Determines the overlay root used by smart.
-- @return		String containing the directory path of the overlay root.
function overlay_root()
	return "/"
end
