#!/bin/bash
#
#  Copyright (C) 2009 McAfee, Inc.  All rights reserved.
#
# This scripts gathers some information about the installed 
# product - can be run only by user with root priv.
# Information extracted is put in the same directory as an archive
#
# Please read before editing.
# * In case you are adding some capability to the the script please 
#   follow the format of output of report and output to terminal. 
#   Also while outputting to the terminal please use `$ECHO2TERM'
#   instead of `echo'. This is because in case of `-q' option we 
#   comment out all the output to terminal and this requires special
#   identification. 
#
# DO NOT `Quiet' THE _ERROR_ MESSAGES - THEY _SHOULD_ COME TO SCREEN
# 

# helper function for querying if an element is in an array
# @@ first argument : element 
# @@ second argument: array 
# @@ returns 1 if true (element is in array) and 0 if false (otherwise)
THIS_PROJ=""
THIS_PLATFORM="`uname`";
THIS_ARCH="`uname -m`";
if [ "$THIS_PLATFORM" = "Linux" ]; then
    if [ -f "/etc/redhat-release" ]; then
	RELEASE_PATH="/etc/redhat-release"
    elif [ -f "/etc/SuSE-release" ]; then
	RELEASE_PATH="/etc/SuSE-release"
    elif [ -f "/VERSION.os" ]; then
	RELEASE_PATH="/VERSION.os"
    elif [ -f "/etc/issue" ]; then
	RELEASE_PATH="/etc/issue"
    fi

    if [ ! -z $RELEASE_PATH ]; then
	RETVAL1=`eval head -n 1 $RELEASE_PATH | awk '{print $1}'`
    	if [ "$RETVAL1" = "CentOS" ]; then
	    THIS_DISTRIBUTION="CentOS"
    	elif [ "$RETVAL1" = "Enterprise" ]; then
	    THIS_DISTRIBUTION="OracleEL"
    	elif [ "$RETVAL1" = "SUSE" ]; then
	    THIS_DISTRIBUTION="SUSE"
    	elif [ "$RETVAL1" = "Wind" ]; then
	    THIS_DISTRIBUTION="WindRiver"
    	fi
    fi
    if [ "$RELEASE_PATH" = "/VERSION.os" ]; then
    	THIS_DISTRIBUTION="WindRiver"
	THIS_PROJ="WindXer"
    fi
fi

_contains () 
{
    ELEMENT=$1;   
    shift;
    ARRAY=$*;
    for j in $ARRAY; do 
	[ "$j" = "$ELEMENT" ] && return 1;
    done 
    return 0;
}


# All variable definitions should be put in this function
define_variables () 
{
    # update PATH just in case it's not what we assume. 
    PATH=$PATH:/bin:/sbin:/usr/bin:/usr/sbin:/usr/local/bin:/usr/local/sbin;
    [ `uname` = "SunOS" ] && PATH=$PATH:/usr/ccs/bin;
    [ `uname` = "HP-UX" ] && PATH=$PATH:/usr/contrib/bin;

    # current Gatherinfo version
    GATHERINFO_VERSION="6.6.2";

    # put hostname in the filename to be generated
    SC_HOST_NAME=`hostname | awk -F. {'print $1'}`;

    # platforms for which security product is also supported. `uname` string.
    SECURITY="Linux SunOS"

    SC_COPY_TO_TMP=0
    
    # following is the name of the target directory and the corresponding
    # tar file name that will be created by the program. 
    _contains "-n" "$@"; 
    if [ $? -eq 0 ]; then 
	    d=`date +%m-%d-%y_%H-%M-%S`; 
	    TARGET_DIR=`pwd`/gatherinfo_data; 
	    TAR_FILE_NAME=gatherinfo-$SC_HOST_NAME-$d.tar
	    TAR_FILE_PATH=`pwd`/gatherinfo-$SC_HOST_NAME-$d.tar
    else
	    TARGET_DIR=`pwd`/gatherinfo_data; 
    	    if [ "$THIS_DISTRIBUTION" == "WindRiver" ] ; then
	        TAR_FILE_NAME=gatherinfo.tar;
	        TAR_FILE_PATH=`pwd`/gatherinfo.tar;
	    else
	        TAR_FILE_NAME=gatherinfo-$SC_HOST_NAME.tar;
	        TAR_FILE_PATH=`pwd`/gatherinfo-$SC_HOST_NAME.tar;
	    fi
    	    SC_COPY_TO_TMP=1
    fi 

    # definitions of output file names 
    LOG=$TARGET_DIR/gatherinfo.log; 
    LDDOUT=$TARGET_DIR/ldd.out; 
    CKSUMOUT=$TARGET_DIR/chksum.out; 
    STATUSOUT=$TARGET_DIR/status.out; 
    SECUREDUPTAR=$TARGET_DIR/var.log.secure.tar; 
    MESSAGESDUPTAR=$TARGET_DIR/var.log.messages.tar; 
    SYSLOGUPDATETAR=$TARGET_DIR/var.log.syslog.tar;
    KERLOGUPDATETAR=$TARGET_DIR/var.log.kerlog.tar; 
    LOGDIR=$TARGET_DIR; 
    ETC_FILES_TAR=$TARGET_DIR/etc.files.tar
    ETC_SYSTEMD_FILES_TAR=$TARGET_DIR/etc.systemd.files.tar
    VERSIONOUT=$TARGET_DIR/version.out; 
    RELEASEOUT=$TARGET_DIR/release.out; 
    TRACEOUT=$TARGET_DIR/trace.out;
    OBJDUMPOUT=$TARGET_DIR/dump.out;
    BACKTRACEOUT=$TARGET_DIR/core.bt.out;
    SS_CONFDUP=$TARGET_DIR/solidcore.conf.dup;
    INSTALLINFO=$TARGET_DIR/install.misc.info.out;
    DRVINFO=$TARGET_DIR/drvinfo.out;
    PROCDEV=$TARGET_DIR/proc.devices.dup;
    PROCFS=$TARGET_DIR/proc.filesystems.dup;
    PROCSLAB=$TARGET_DIR/proc.slabinfo.dup;
    PROCSOLIDCORE=$TARGET_DIR/proc.solidcore.dup;
    ETCFS=$TARGET_DIR/etc.filesystems.dup;
    DMSGOUT=$TARGET_DIR/dmesg.out;
    JRNLOUT=$TARGET_DIR/jrnlboot.out
    JRNLSCOUT=$TARGET_DIR/jrnlscsrvc.out
    OLDSYSLOG=$TARGET_DIR/oldsyslog.dup;
    TUNEABLEINFO=$TARGET_DIR/tunables.out;
    CORETAR=$TARGET_DIR/coretar.out;
    GIVERSION=$TARGET_DIR/gatherinfo.version;
    LSOUT=$TARGET_DIR/ls.out;
    FSOUT=$TARGET_DIR/features.out;
    EVENTOUT=$TARGET_DIR/event.out;
    MONOUT=$TARGET_DIR/monls.out;
    BYPASSOUT=$TARGET_DIR/bypassls.out;
    UPDOUT=$TARGET_DIR/updls.out;
    XRAYOUT=$TARGET_DIR/xray.out;

    # sadmin commands which are common for all platforms 
    SADMIN="sadmin"
    STATUS="status"
    VERSION="version" 
    LIST_SOLIDIFIED="ls"
    FEATURES="features" 
    EVENT="event" 
    LIST="list" 
    MON="mon" 
    BYPASS="bypass" 
    UPD="updaters"
    XRAY="xray"
    SINK="sink"
    RECOVER="recover"
    # common variable names for all platforms
    SOLIDIFIER_TAR_BALL=$TARGET_DIR/solidcore.tar;

    if [ "$THIS_DISTRIBUTION" == "WindRiver" ] ; then
        SS_SW="solidcores3";
        SCSRVCLOG_PATH="/usr/local/mcafee/solidcore/log"; 
	SCVERSION_PATH="/usr/local/mcafee/solidcore/solidcore.version"
	if [ "$THIS_PROJ" != "WindXer" ] ; then
            S3INSTALL="$SCSRVCLOG_PATH/install*.log";
            S3UNINSTALL="/tmp/solidcores3_uninstall.log"
	fi
    else
        SCSRVCLOG_PATH="/var/log/mcafee/solidcore"; 
        SS_SW="solidcoreS3"; 
        SCSRVCLOG_PATH="/var/log/mcafee/solidcore";
        S3INSTALL="$SCSRVCLOG_PATH/install*.log $SCSRVCLOG_PATH/solidcoreS3_install*.log /tmp/bitrock_installer*.log /tmp/solidcoreS3_install*.log";
        S3UNINSTALL="/tmp/solidcoreS3_uninstall.log"
    fi

    # define commands for different platforms to be used here.
    if [ `uname` = Linux ]; then
	#Determine Linux distro
	if [ -f "/etc/redhat-release" ]; then
        RELEASE_PATH="/etc/redhat-release"
    	elif [ -f "/etc/SuSE-release" ]; then
        RELEASE_PATH="/etc/SuSE-release"
   	 elif [ -f "/etc/lsb-release" ]; then
        RELEASE_PATH="/etc/lsb-release"
    fi

    if [ ! -z $RELEASE_PATH ]; then
        if [ $RELEASE_PATH = "/etc/lsb-release" ]; then
            RETVAL1=`cat $RELEASE_PATH | grep -w DISTRIB_DESCRIPTION | head -n 1 | awk -F"=" '{print $2}' | awk -F"\"" '{print $2}'|awk '{print $1}'`
            if [ "$RETVAL1" = "Ubuntu" ]; then
                THIS_DISTRIBUTION="Ubuntu"
            fi
        else
            RETVAL1=`eval head -n 1 $RELEASE_PATH | awk '{print $1}'`
            if [ "$RETVAL1" = "CentOS" ]; then
                THIS_DISTRIBUTION="CentOS"
            elif [ "$RETVAL1" = "Enterprise" ]; then
                THIS_DISTRIBUTION="OracleEL"
            elif [ "$RETVAL1" = "SUSE" ]; then
                THIS_DISTRIBUTION="SUSE"
            fi
        fi
    fi

        # Assign to Linux specific commands
	SW_VERIFY_CMD="";
	# use *scsrvc* to collect the file.
       SCSRVCLNK3="";
       SCSRVCLNK5="";
       SCSRVCLNK2="";
       SCSRVCLNK4="";
       SECURELOG="";  #only for linux
       SYSLOGUB="";
	KERLOG="";
	if [ "$THIS_DISTRIBUTION" = "Ubuntu" ]; then
		SW_VERIFY_CMD="dpkg -s ";
		# use *scsrvc* to collect the file.
        	SCSRVCLNK3="/etc/rc3.d/*scsrvc";
        	SCSRVCLNK5="/etc/rc5.d/*scsrvc"; 
		SCSRVCLNK2="/etc/rc2.d/*scsrvc";
		SCSRVCLNK4="/etc/rc4.d/*scsrvc"
		SECURELOG="/var/log/auth.log";  #only for Ubuntu
		SYSLOGUB="/var/log/syslog";
		KERLOG="/var/log/kern.log"; 
    	else
		if [ "$THIS_PROJ" != "WindXer" ] ; then
		    SW_VERIFY_CMD="rpm -q ";
		    SECURELOG="/var/log/secure";  #only for linux
		fi
		# use *scsrvc* to collect the file.
        	SCSRVCLNK3="/etc/rc.d/rc3.d/*scsrvc";
        	SCSRVCLNK5="/etc/rc.d/rc5.d/*scsrvc";
	fi 
	TRACE="strace -f";
	DUMP_COMMAND="objdump"
	DUMP_CMD="objdump -dxr"; 
	if [ "$THIS_PROJ" = "WindXer" ] ; then
	    CKSUM_CMD="md5sum";
	else
	    CKSUM_CMD="sha1sum"; 
	fi
	
	VARMSGLOG="/var/log/messages"; 
	LOGMSGNAME="messages"
	LDD="ldd"; 
	DRIVER="/dev/scdrv"; 
	COMPRESS="gzip"; 
	DMESG="dmesg"; 
	SADMINLINK="/usr/sbin/sadmin";
	# use *scsrvc* to collect the file.
        if [ "$THIS_DISTRIBUTION" == "WindRiver" -a "$THIS_PROJ" != "WindXer" ] ; then
	    SCSRVCLNK3="/etc/rc3.d/*scsrvc";
            SCSRVCLNK5="/etc/rc5.d/*scsrvc";
	else
	    SCSRVCLNK3="/etc/rc.d/rc3.d/*scsrvc"; 
	    SCSRVCLNK5="/etc/rc.d/rc5.d/*scsrvc"; 
	fi
	SCSRVCSERVICE="/etc/init.d/scsrvc";
	LINUX_GET_PROC_DEVICES="cat /proc/devices"; 
	LINUX_GET_PROC_FILESYSTEMS="cat /proc/filesystems";
	LINUX_GET_PROC_SLABINFO="cat /proc/slabinfo";
	
	ps -p 1 | grep -i "systemd" >/dev/null
	if [ $? -eq 0 ]; then
	    JRNLBOOT="journalctl -b";
	    JRNLSCSRVC="journalctl -u scsrvc";
	    SYSTEMDFILES="/etc/systemd/system/multi-user.target.wants"
	fi
    elif [ `uname` = SunOS ]; then
	# Assign to Solaris specific commands
    	SW_VERIFY_CMD="pkginfo -l"; 
	TRACE="truss -f"; 
	# Testing for the existence of elfdump is PATH else using /usr/ccs/bin
	DUMP_CMD="/usr/ccs/bin/elfdump";
	if [ ! -f $DUMP_CMD ]; then
	    DUMP_CMD="elfdump";
	fi
	CKSUM_CMD="cksum"; 
	VARMSGLOG="/var/adm/messages"; 
	LOGMSGNAME="messages"
	LDD="ldd"; 
	DRIVER="/devices/pseudo/scdrv@1:0"; 
	COMPRESS="gzip"; 
	DMESG="dmesg"; 
	SADMINLINK="/usr/sbin/sadmin";
	SCSRVCLNK3="/etc/rc2.d/*scsrvc"; 
	SCSRVCLNK5="/etc/rc0.d/*scsrvc"; 
	SCSRVCSERVICE="/etc/init.d/scsrvc";
	SOLARIS_GET_ETC_SYSTEM="cat /etc/system";
    elif [ `uname` = HP-UX ]; then
	# Assign to HP-UX specific commands
    	SW_VERIFY_CMD="swlist"; 
	TRACE="tusc -f"; 
	DUMP_CMD=""; 
	CKSUM_CMD="cksum"; 
	VARMSGLOG="/var/adm/syslog/syslog.log"; 
	LOGMSGNAME="syslog.log"
	COMPRESS="gzip"; 
	DMESG="/usr/sbin/dmesg"; 
	LDD="/usr/ccs/bin/ldd"; 
	DRIVER="/dev/scdrv"; 
	if [ `uname -r` = B.11.00 -o `uname -r` = B.11.11 ]; then
		TUNABLE="kmtune"
	else	    
		TUNABLE="kctune";
	fi
	SADMINLINK="/usr/sbin/sadmin";
	SCSRVCLNK3="/sbin/rc2.d/*scsrvc"; 
	SCSRVCLNK5="/sbin/rc1.d/*scsrvc"
	SCSRVCSERVICE="/sbin/init.d/scsrvc";
	HPUX_GET_OLD_SYSLOG_LOG="cp -p /var/adm/syslog/OLDsyslog.log $OLDSYSLOG";
	HPUX_RESMON_LOG_PATH="/var/opt/resmon";
	HPUX_RESMON_LOG_TAR="$TARGET_DIR/var.opt.resmon.log.tar";
	HPUX_TOMBSTONES_PATH="/var/tombstones";
	HPUX_TOMBSTONES_TARG_DIR="$TARGET_DIR/var_tombstones"; 
    elif [ `uname` = AIX ]; then
	# Assign to AIX specific commands
  	SW_VERIFY_CMD="lslpp -l"; 
	TRACE="truss -f"; 
	DUMP_CMD="dump -X32_64 -r -oh -Tv"; 
	CKSUM_CMD="cksum"; 
	COMPRESS="compress"; 
	# DMESG missing from AIX
	SADMINLINK="/usr/sbin/sadmin";
    	SCSRVCLNK3="/etc/rc.d/init.d/*scsrvc"
	SCSRVCLNK5="/etc/rc.d/rc0.d/*scsrvc"
	SCSRVCSERVICE="/etc/scsrvc";
	LDD="ldd"; 
    fi

    # error codes definition 
    ERR_SOLIDIFIER_NOT_INSTALLED=100;
    ERR_ANOTHER_SADMIN_COMMAND_RUNNING=101;
    ERR_USER_NOT_ROOT=103;
    ERR_CONFIG_FILE_NOT_FOUND=104; 
    ERR_CONFIG_FILE_CORRUPT=105;
    ERR_INSTALL_DIR_NOT_FOUND=106;
    ERR_SOLIDIFIER_SERVICE_NOT_FOUND=107;
    ERR_SOLIDIFIER_CMDLINE_NOT_FOUND=108;

    # This is for some shell handling. 
    SH=/bin/sh; 
    BASH=/bin/bash;
    # following are the options given to echo for different shells
    # 	-n : continue in same line (Bash)
    #	-e : translate special characters like \r, \t (Bash)
    #	\c : continue in same line (sh)
    #
    [ "$SHELL" = "$BASH" ] && { 
	NOPT="";
	EOPT=""; }

   ECHO2TERM=echo
 
   ROOT_UID="0";
}

# cleanup function
gicleanup () 
{
    if [ "$1" != "1" ] && [ "$1" != "0" ]; then
    	echo ""; # for a new line on 'Ctrl + C' 
    fi 
    
    [ -d "$TARGET_DIR" ] && rm -rf $TARGET_DIR; 
    [ -f "$TAR_FILE_PATH" ] && rm -f $TAR_FILE_PATH; 
    
    # with "1" argument this will act as normal cleanup
    # function without exiting. see use in precondition tests.
    [ "$1" != "1" ] && exit 0;
}
trap "gicleanup" 2

# Print the usage of the tool. Call it with first parameter 
# as the return value. If called without any first parameter
# then default behaviour is return 0. 
usage()
{
    PROG_NAME=`basename $0`

    echo "Usage :"
    echo "$PROG_NAME -h|--help"
    echo "$PROG_NAME -v|--version"
    echo "$PROG_NAME -c | --core <core-file>"
    echo "$PROG_NAME -q";

    if [ ! -z "$1" ] && [ "$1" != "0" ]
    then
    	exit $1
    else
	echo "-h --help		        Displays help for Gatherinfo."; 
    	echo "-v --version		        Displays version of Gatherinfo."; 
	echo "-c --core <core-file>		use this core file for getting backtrace"; 
	echo "-q				quiet mode."
	echo "-n				No time stamp in output file name"
	echo ""; 
 	exit 0;
    fi
}

# function to check if the available space meets min. requirement
check_for_available_disk_space ()
{
    THIS_PLATFORM="`uname`";
    FOUND_LOW_DISK_SPACE=0;
    SC_INSTALL_DIR="/usr/local"
    ANALYSIS_REQD_DISK_SPACE_KB="72000"; # 70Mbytes
    if [ "$THIS_PLATFORM" = "Linux" ]; then
        CMDS_DISK_SPACE_INFO="df -kT";
     elif [ "$THIS_PLATFORM" = "SunOS" ]; then
       if [ `uname -r` = "5.10" ]; then
                CONTRACT="ctrun -l child -o pgrponly "
        else
                CONTRACT=""
        fi
       CMDS_DISK_SPACE_INFO="$CONTRACT df -k";
     elif [ "$THIS_PLATFORM" = "HP-UX" ]; then
        CMDS_DISK_SPACE_INFO="bdf";
     elif [ "$THIS_PLATFORM" = "AIX" ]; then  
        CMDS_DISK_SPACE_INFO="df -k";
    else
      echo "ERROR: Unsupported/Unknown platform [ "$THIS_PLATFORM" ]."
      exit 1;
    fi
    STR_DISKSPACE=`$CMDS_DISK_SPACE_INFO $SC_INSTALL_DIR`;
    # df output is slightly different on some platforms. keep updating.
    if [ "$THIS_PLATFORM" = "Linux" ]; then
        diff_factor=`echo $STR_DISKSPACE | tr '\n' ' ' | awk '{print $8}'`
        if [ "$diff_factor" = "on" ]; then
            DISKSPACE_KB=`echo $STR_DISKSPACE | awk {'print $13'}`;
        else
            DISKSPACE_KB=`echo $STR_DISKSPACE | awk {'print $12'}`;
        fi
    elif [ "$THIS_PLATFORM" = "SunOS" ]; then
        DISKSPACE_KB=`echo $STR_DISKSPACE | awk {'print $11'}`;
    elif [ "$THIS_PLATFORM" = "HP-UX" ]; then
        DISKSPACE_KB=`echo $STR_DISKSPACE | awk {'print $11'}`;
    elif [ "$THIS_PLATFORM" = "AIX" ]; then
        DISKSPACE_KB=`echo $STR_DISKSPACE | awk {'print $11'}`;
    fi;
    if [ $DISKSPACE_KB -le $ANALYSIS_REQD_DISK_SPACE_KB ]; then
        DISKSPACE_MB=`expr $DISKSPACE_KB / 1024`;
        echo "Total disk space available is less than the required space. Required disk space: 70 MB."\
               "Available disk space: $DISKSPACE_MB MB $MSG_LABEL_DISK_SPACE";
        echo  "Please free required disk space and rerun the script"
        exit 1;
    fi;
}

# Print out the version of the tool and return 0.
version()
{
    echo "GatherInfo version $GATHERINFO_VERSION"
}

# function to print the result of some operation. 
print_result ()
{
   # Do not do the terminal play of displaying [OK], [FAILED] messages
   # not portable across consoles and slow as well.

   echo ;
   return;

    # move to the begining of line
    echo $NOPT $EOPT "\r"
	
    # move 65 spaces to the right
    # tput cuf 65
    # using cufl we can do it at once but that is not supported by
    # console (non-xterm) and some platforms (like 7.2)
    i=65
    while [ "$i" != "0" ]
    do
	tput cuf1 
	i=`expr $i - 1`
    done
    
    echo $NOPT " ... "
    
    [ -z "$1" ] && { 
	tput setf 1 
	echo $EOPT "[SKIPPED]";
	tput setf 0
	return;
    }
    
    if [ $1 -eq 0 ]
    then
	tput setf 2
	echo $EOPT "[  OK  ]"
	tput setf 0
    else
	tput setf 4
    	echo $EOPT "[FAILED]"
	tput setf 0
    fi	    
}

archive_results()
{
    $ECHO2TERM $NOPT $EOPT "Final Phase. Archiving the target directory.."
    currentdir=`pwd`; 
    nextdir=`dirname $TARGET_DIR`; 
    TARGET_BASE=`basename $TARGET_DIR`; 
    cd $nextdir; 
    tar -cvf $TAR_FILE_PATH $TARGET_BASE > /dev/null 2>> $LOG; 
    
    # this definition of FOPT is just to ensure that if COMPRESS is changed
    # tomorrow then too it works ... 
    if [ "$COMPRESS" = "gzip" ]; then
	FOPT="-f";
    else
	FOPT="";
    fi;
    
    $COMPRESS $FOPT $TAR_FILE_PATH 2>> $LOG; 
    RES=$?; 
    [ "$QOPT" != "1" ] && print_result $RES; 
    
    $ECHO2TERM ""; 
    cd $currentdir/../;

    # copy to /tmp if the command is from SC.
    if [ "$SC_COPY_TO_TMP" = 1 ]; then
	cp -p $TAR_FILE_PATH* /tmp/s3gatherinfo.tar.gz
    	echo "Collect the result file : /tmp/s3gatherinfo.tar.gz"
    else
	PDIR=`pwd`
        if [ "$PDIR" = "/" ] || [ "$PDIR" = "//" ]; then
                PDIR=
        fi
    	echo "Collect the result file : $PDIR/`ls $TAR_FILE_NAME*`"
    fi
}

# Check for root user being installing ss
check_privileges () 
{ 
    if [ "$UID" -ne "$ROOT_UID" ]; then
	echo "Only root user is allowed to run the tool."; 
	echo "Note: If you have root privilege then try exporting UID."; 
	exit $ERR_USER_NOT_ROOT;
    fi;
}

create_target_directory () 
{
    rm -rf $TARGET_DIR; 
    mkdir -p $TARGET_DIR; 
}

# Helper functions 

# 
# contains arg1 arg2
# returns 1 (success) if arg1 (array) contains element arg2 (member)
# returns 0 (failure) otherwise
#
contains () 
{
    for i in $1; do 
	if [ $i = $2 ]; then
	    return 1;
	fi 
    done 
    return 0;
}

#
# 	[ S C R I P T   S T A R T S   H E R E ] 
# 

define_variables "$@";
check_privileges 
check_for_available_disk_space
create_target_directory

# Parse the command line 
while [ $# -gt 0 ]
do
    case "$1" in

    -h|--help)  
    		gicleanup 1; 
    		usage; 
		;; 
		
    -v|--version) 
    		gicleanup 1; 
		version; 
		exit 0; 
		;; 

    -c|--core) 
    		if [ ! -z "$2" ]; then
			COREFILENAME=$2; 
			shift; 
		else
			echo "$1 option requires the core file name"; 
			usage 1; 
		fi
    		;;
    
    -n) 
    		# keep this so that this option remains valid
    		;; 

    -q)
    	 # this will be used for not calling print_result function 
         # which due to some play on terminal causes delay that is 
    	 # not required in case of `quiet' run.
	 QOPT=1;

	 ECHO2TERM=":"
	 ;;

    *) 
    	echo "option \"$1\" is not supported"; 
	gicleanup 1; 
	usage 1;  
	;;
    esac
    shift 
done 

$ECHO2TERM $EOPT "----------------------------------------------------------------------------"
$ECHO2TERM $EOPT "                     [ Pre condition tests ]"
$ECHO2TERM $EOPT "----------------------------------------------------------------------------"

if [ `uname` != AIX ]; then
	# Check if the McAfee Solidifier is installed.
	$ECHO2TERM $NOPT "Checking if McAfee Solidifier is installed.. "
	#$SW_VERIFY_CMD $SS_SW > /dev/null 2>> $LOG
	if [ $? -ne 0 ]
	then
		[ "$QOPT" != "1" ] && print_result 1
		echo $NOPT "McAfee Solidifier is not installed. "
		echo "Please install McAfee Solidifier before executing this script."
		gicleanup 1
		exit $ERR_SOLIDIFIER_NOT_INSTALLED;
	fi
	[ "$QOPT" != "1" ] && print_result 0
fi

# Check for another instance of sadmin-admin already running.
$ECHO2TERM $NOPT "Checking that no other sadmin command running.. "
COMM_STATUS=`sadmin version | grep "command is already"`
if [ ! -z "$COMM_STATUS" ]; then
    [ "$QOPT" != "1" ] && print_result 1
    echo $NOPT "Another Instance of sadmin is running. "
    echo "Please close it and try again."
    gicleanup 1
    exit $ERR_ANOTHER_SADMIN_COMMAND_RUNNING;
fi	
#[ "$QOPT" != "1" ] && print_result 0

if [ -z "$SS_CONF" ]
then
	SS_CONF=/etc/mcafee/solidcore/solidcore.conf
fi

#check if local cli is recovered or not. 	 

IS_CLI_LOCKED=0
LOCK_STAT=`sadmin features | grep "has been locked down"`
if [ ! -z "$LOCK_STAT" ]; then
    IS_CLI_LOCKED=1
    echo "ERROR: Local cli is locked."
    echo "Please enter password to recover cli."
    echo
    $SADMIN $RECOVER
fi

# check if some other command is running
if [ "$IS_CLI_LOCKED" -eq "0" ]; then
    COMM_STAT=`sadmin features | grep "command is already"`
#    if [ ! -z "$COMM_STAT" ]; then
#        echo "Failed to get cli lockdown status. Continuing execution...."
#       echo
#    fi
fi

# Continue execution in case of locked cli.
sadmin features | grep "has been locked down" > /dev/null
if [ "$?" -eq 0 ]; then
       echo "WARNING: Continuing execution with locked cli."
       echo
fi

echo
$ECHO2TERM $NOPT "Checking for the presence of config file.. "
if [ ! -r "$SS_CONF" ]
then
	[ "$QOPT" != "1" ] && print_result 1	# it is here so failure
	echo "Configuration file $SS_CONF is not found or is not readable."
	echo "Please export SS_CONF env variable with readable solidcore.conf config file"
	gicleanup 1
	exit $ERR_CONFIG_FILE_NOT_FOUND;
fi
[ "$QOPT" != "1" ] && print_result 0

if [ -z "$SS_HOME" ]
then
	SS_HOME=`cat $SS_CONF | grep "^InstallDir" | awk '{printf ($3)}'`
fi
if [ -z "$SS_HOME" ]
then
	[ "$QOPT" != "1" ] && print_result 1	# it is here so failure
	echo "SS_HOME couldnt be set from $SS_CONF file"
	echo "Please check if you have correct $SS_CONF file"
	gicleanup 1
	exit $ERR_CONFIG_FILE_CORRUPT;
fi

$ECHO2TERM $NOPT "Checking for the presence of install directory.. "
if [ ! -d "$SS_HOME" ]
then
	[ "$QOPT" != "1" ] && print_result 1
	echo "Install directory $SS_HOME doesnt exist or is not a directory."
	echo "Please export SS_HOME env variable with ss install path directory"
	gicleanup 1
	exit $ERR_INSTALL_DIR_NOT_FOUND;
fi
[ "$QOPT" != "1" ] && print_result 0

if [ -z "$SCSRVC_BIN" ]
then
	SCSRVC_BIN=$SS_HOME/bin/scsrvc

fi

$ECHO2TERM $NOPT "Checking for the presence of scsrvc binary.. "
if [ ! -x $SCSRVC_BIN ]
then
	[ "$QOPT" != "1" ] && print_result 1
	echo "Binary $SCSRVC_BIN is not found, or is not executable."
	gicleanup 1
	exit $ERR_SOLIDIFIER_SERVICE_NOT_FOUND;
fi
[ "$QOPT" != "1" ] && print_result 0


$ECHO2TERM $NOPT "Checking for the presence of sadmin binary.. "
if [ ! -x "$SS_HOME/bin/$SADMIN" ]
then
	[ "$QOPT" != "1" ] && print_result 1	# it is here so failure
	echo "Binary $SADMIN is not found, or is not executable."
	gicleanup 1
	exit $ERR_SOLIDIFIER_CMDLINE_NOT_FOUND;
fi
[ "$QOPT" != "1" ] && print_result 0		# it is here so success


$ECHO2TERM ""
$ECHO2TERM $EOPT "----------------------------------------------------------------------------"
$ECHO2TERM $EOPT "                     [ GatherInfo Collection ] "
$ECHO2TERM $EOPT "----------------------------------------------------------------------------"

SAVEDIR="`pwd`";
cd `dirname $0`; 

#*****************************************************************
$ECHO2TERM $NOPT "Collecting mcafee/solidcore tarball "
#*****************************************************************
ODIR=`pwd`;
cd `dirname "$SS_HOME"`

tar -cvf "$SOLIDIFIER_TAR_BALL" solidcore 1>/dev/null 2>&1;

if [ "$COMPRESS" = "gzip" ]; then
    FOPT="-f";
else
    FOPT="";
fi;

$COMPRESS $FOPT $SOLIDIFIER_TAR_BALL; 
RES=0;
cd $ODIR;
[ "$QOPT" != "1" ] && print_result $RES


#*****************************************************************
$ECHO2TERM $NOPT "Collecting scanalyzer output "
#*****************************************************************
SCANALYZER_PATH=$SS_HOME/tools/scanalyzer/scanalyzer.sh
if [ ! -f $SCANALYZER_PATH ]; then
	echo "scanalyzer.sh tool missing skipping.";
	[ "$QOPT" != "1" ] && print_result
else
        TEMPFILE_NAME="/tmp/scanalyzer.XXXX"
	sh $SCANALYZER_PATH > $TEMPFILE_NAME
	TFILENAME=`cat $TEMPFILE_NAME | grep "Collect the report" | awk {'print $4'}`;
	[ -f "$TFILENAME" ] && mv -- "$TFILENAME" "$TARGET_DIR";
	RES=$?
	if [ $RES -ne 0 ]; then
	    exit $RES;
	fi
	[ "$QOPT" != "1" ] && print_result $RES
	rm -rf $TEMPFILE_NAME
	rm -rf data
fi

cd $SAVEDIR;

#*****************************************************************
$ECHO2TERM $NOPT "Collecting logfiles "
#*****************************************************************
CUR_SCSRVCLOG_PATH=`cat $SS_CONF | grep "^LogFilePath" | awk '{ printf $3 }'`

mkdir $LOGDIR/default_location_logs

# Collect scsrvc-lock file 
if [ "$THIS_DISTRIBUTION" == "WindRiver" ] ; then
    cp -p $SS_HOME/run/.scsrvc-lock $LOGDIR/ 2>> $LOG
else
    cp -p /tmp/.scsrvc-lock $LOGDIR/ 2>> $LOG
fi

# collect solidcore.log from /tmp
TMP_LOGFILE_PATH=/tmp/solidcore.log

if [ "$THIS_PROJ" != "WindXer" ] ; then
# Collect install/uninstall logs
cp -p $S3INSTALL $S3UNINSTALL $LOGDIR/ 2>> $LOG
fi

cp -p $TMP_LOGFILE_PATH $LOGDIR/tmp.solidcore.log 2>> /dev/null

cp -p $SCSRVCLOG_PATH/solidcore.log* $LOGDIR/default_location_logs 2>> $LOG
RES=$?
[ "$QOPT" != "1" ] && print_result $RES

if [ "$CUR_SCSRVCLOG_PATH" != "$SCSRVCLOG_PATH" ]; then

    mkdir "$LOGDIR"/moved_logs
    
    cp -p $CUR_SCSRVCLOG_PATH/solidcore.log* $LOGDIR/moved_logs 2>> $LOG
    RES=$?
    [ "$QOPT" != "1" ] && print_result $RES
fi

SCSRVCSTAT=`$SCSRVC_BIN $STATUS 1>/dev/null 2>&1`;
echo "$SCSRVCSTAT" | grep -i "stopped" 2>/dev/null; 
[ "$?" = "0" ] || RES=0;

if [ "$RES" = "0" ]; then 
	#*****************************************************************
	$ECHO2TERM $NOPT "Collecting sadmin commands output "
	#*****************************************************************
	$ECHO2TERM "";
	
	#*****************************************************************
	$ECHO2TERM $NOPT "          + status "
	#*****************************************************************
	echo "sadmin status output is as below: " >  $STATUSOUT 
	echo "======================================" >> $STATUSOUT
	$SADMIN $STATUS 1>> $STATUSOUT 2>> $LOG
	RES=$?
	[ "$QOPT" != "1" ] && print_result $RES

	#*****************************************************************
	$ECHO2TERM $NOPT "          + version "
	#*****************************************************************
	echo "sadmin version output is as below: " >  $VERSIONOUT 
	echo "======================================" >> $VERSIONOUT
	$SADMIN $VERSION 1>> $VERSIONOUT 2>> $LOG
	RES=$?
	[ "$QOPT" != "1" ] && print_result $RES

	# contains function is defined above 
	contains "$SECURITY" `uname`; 
	if [ $? -eq 1 ] 
	then
	    #*****************************************************************
	    $ECHO2TERM $NOPT "          + list-solidified "
	    #*****************************************************************
	    echo "sadmin ls output is as below: " >  $LSOUT 
	    echo "======================================" >> $LSOUT
	    $SADMIN $LIST_SOLIDIFIED 1>> $LSOUT 2>> $LOG
	    RES=$?
	    [ "$QOPT" != "1" ] && print_result $RES
	fi

	#*****************************************************************
	$ECHO2TERM $NOPT "          + features status "
	#*****************************************************************
	echo "sadmin features status output is as below: " >  $FSOUT 
	echo "======================================" >> $FSOUT
	$SADMIN $FEATURES $LIST -d 1>> $FSOUT 2>> $LOG
	RES=$?
	[ "$QOPT" != "1" ] && print_result $RES
	
	#*****************************************************************
	$ECHO2TERM $NOPT "          + evt "
	#*****************************************************************
	echo "sadmin evt output is as below: " >  $EVENTOUT 
	echo "======================================" >> $EVENTOUT
	$SADMIN $EVENT $SINK 1>> $EVENTOUT 2>> $LOG
	RES=$?
	[ "$QOPT" != "1" ] && print_result $RES

	#*****************************************************************
	$ECHO2TERM $NOPT "          + mon list "
	#*****************************************************************
	echo "sadmin mon list output is as below: " >  $MONOUT 
	echo "======================================" >> $MONOUT
	$SADMIN $MON $LIST 1>> $MONOUT 2>> $LOG
	RES=$?
	[ "$QOPT" != "1" ] && print_result $RES

	#*****************************************************************
	$ECHO2TERM $NOPT "          + xray "
	#*****************************************************************
	echo "sadmin xray output is as below: " >  $XRAYOUT 
	echo "======================================" >> $XRAYOUT
	$SADMIN $XRAY 1>> $XRAYOUT 2>> $LOG
	RES=$?
	[ "$QOPT" != "1" ] && print_result $RES

	# contains function is defined above 
	contains "$SECURITY" `uname`; 
	if [ $? -eq 1 ] 
	then
		#$ECHO2TERM $NOPT "          + bypass list "
		#echo "sadmin bypass list output is as below: " >  $BYPASSOUT 
		#echo "======================================" >> $BYPASSOUT
		#$SADMIN $BYPASS $LIST 1>> $BYPASSOUT 2>> $LOG
		#RES=$?
		#[ "$QOPT" != "1" ] && print_result $RES

		$ECHO2TERM $NOPT "          + updaters list "
		echo "sadmin updaters list output is as below: " >  $UPDOUT 
		echo "======================================" >> $UPDOUT
		$SADMIN $UPD $LIST 1>> $UPDOUT 2>> $LOG
		RES=$?
		[ "$QOPT" != "1" ] && print_result $RES
	fi
fi

#*****************************************************************
$ECHO2TERM $NOPT "Collecting EVENTS " 
#*****************************************************************
if [ ! -z "$SECURELOG" ]
then
        tar -cvf $SECUREDUPTAR $SECURELOG* >> $LOG 2>&1 


fi

if [ ! -z "$SYSLOGUB" ]
then
        tar -cvf $SYSLOGUPDATETAR $SYSLOGUB* >> $LOG 2>&1
fi

if [ ! -z "$KERLOG" ]
then
        tar -cvf $KERLOGUPDATETAR $KERLOG* >> $LOG 2>&1
fi

if [ ! -z "$VARMSGLOG" ]
then
        SC_CUR_DIR=`pwd`
	SC_VAR_LOG_DIR=`dirname $VARMSGLOG`

	cd $SC_VAR_LOG_DIR
        tar -cvf $MESSAGESDUPTAR $LOGMSGNAME* >> $LOG 2>&1
	res=$?
	cd $SC_CUR_DIR
fi






[ "$QOPT" != "1" ] && print_result $res

#*****************************************************************
$ECHO2TERM $NOPT "Collecting solidcore.conf "
#*****************************************************************
cp -p $SS_CONF $SS_CONFDUP 2>> $LOG
res=`echo $?`
if [ $? -eq 0 ]
then
	chmod 644 $SS_CONFDUP 2>> $LOG
	res=`echo $?`
fi
[ "$QOPT" != "1" ] && print_result $res


#*****************************************************************
$ECHO2TERM $NOPT "Collecting release information "
#*****************************************************************
if [ "$THIS_PROJ" != "WindXer" ] ; then
    $SW_VERIFY_CMD $SS_SW > $RELEASEOUT 2>> $LOG
else
    cat $SCVERSION_PATH  > $RELEASEOUT 2>> $LOG
fi
res=$?
[ "$QOPT" != "1" ] && print_result $res
if [ "$THIS_PROJ" != "WindXer" ] ; then
#*****************************************************************
$ECHO2TERM $NOPT "Collecting trace output "
#*****************************************************************
echo "$TRACE output on $SCSRVC_BIN $STATUS is :" >  $TRACEOUT
echo "======================================" >> $TRACEOUT
$TRACE $SCSRVC_BIN $STATUS >> $TRACEOUT 2>&1
RES=$?
[ "$QOPT" != "1" ] && print_result $RES
fi
if [ `uname` != HP-UX ]
then
#*****************************************************************
$ECHO2TERM $NOPT "Collecting dump output "
#*****************************************************************
echo "$DUMP_CMD output on $SCSRVC_BIN is :" >  $OBJDUMPOUT
echo "=================================" >> $OBJDUMPOUT
which $DUMP_COMMAND > /dev/null 2>&1
if [ $? -eq 0 ] ; then
    $DUMP_CMD $SCSRVC_BIN >> $OBJDUMPOUT 2>> $LOG
    RES=$?
else
    echo "Command  $DUMP_CMD not found on this machine."
fi
[ "$QOPT" != "1" ] && print_result $RES
fi

#echo -ne "\tEnter the core file: "
#read core

#*****************************************************************
$ECHO2TERM $NOPT "Collecting backtrace (on core) output"
#*****************************************************************
core=$COREFILENAME
if [ -z "$core" ]
then
	echo "No core file given for backtrace" >> $LOG
	[ "$QOPT" != "1" ] && print_result  # without any parmater this prints SKIPPED
elif [ -r "$core" ]
then
	if [ `uname` = Linux ]	# for linux use gdb
	then
	GDB_BIN=`which gdb`
	if [ -x "$GDB_BIN" ]
	then
		$GDB_BIN -c $core > $BACKTRACEOUT << EOF 2>> $LOG
		file $SCSRVC_BIN
		bt
		quit
EOF
		[ "$QOPT" != "1" ] && print_result 0 
	else
		echo "$GDB_BIN is not an executable binary" >> $LOG
		[ "$QOPT" != "1" ] && print_result 1
	fi
	elif [ `uname` = SunOS ] # for solaris use pstack -F
	then
		pstack -F $core > $BACKTRACEOUT 2>> $LOG
		RES=$?
		[ "$QOPT" != "1" ] && print_result $RES
	elif [ `uname` = HP-UX ]
	then
		# tar-ing the core file and the SCSRVC binary together
		tar -cvf $CORETAR $core $SCSRVC_BIN >> $LOG
		RES=$?
		[ "$QOPT" != "1" ] && print_result $RES

	elif [ `uname` = AIX ]
	then
		dname="tmp-sc-d.$$"
		#dname=`mktemp -d tmp-sc-d.XXXXXX`
		snapcore -d$dname $core > /dev/null
		[ "$?" = "0" ] && cp -p $dname/snapcore*.pax.Z $LOGDIR
		rm -rf $dname
		RES=$?
		[ "$QOPT" != "1" ] && print_result $RES
	fi
else
	echo "core file $core is not readable" >> $LOG
	[ "$QOPT" != "1" ] && print_result 1
fi

#*****************************************************************
$ECHO2TERM $NOPT "Collecting ldd output"
#*****************************************************************
if [ ! -z "$LDD" ]
then
echo "$LDD $SCSRVC_BIN:" > $LDDOUT
$LDD $SCSRVC_BIN >> $LDDOUT 2>> $LOG
RES=$?
[ "$QOPT" != "1" ] && print_result $RES
else
echo "did not find ldd" >> $LOG
[ "$QOPT" != "1" ] && print_result # print skipped message
fi

#*****************************************************************
$ECHO2TERM $NOPT "Collecting the file (and checksum) information"
#*****************************************************************
ls -l $SS_CONF > $CKSUMOUT 2>> $LOG
# find on hp-ux does not support -ls option
find $SS_HOME  >> $CKSUMOUT 2>> $LOG
$CKSUM_CMD $SS_CONF >> $CKSUMOUT 2>> $LOG
for i in `find $SS_HOME -type f 2>/dev/null`
do
	$CKSUM_CMD $i >> $CKSUMOUT 2>> $LOG
done
RES=$?
[ "$QOPT" != "1" ] && print_result $RES

#this function needs to be updated with all the links that are created
#*****************************************************************
$ECHO2TERM $NOPT  "Collecting the install information"
#*****************************************************************
if [ -f $SADMINLINK ]; then
    echo "sadmin link created at: " >> $INSTALLINFO
    ls -l $SADMINLINK >> $INSTALLINFO 2>> $LOG
    res1=$?
else
    echo "File not exist, So can't list file $SADMINLINK" >> $LOG
fi

echo "scsrvc service will start from:" >> $INSTALLINFO
ls -l $SCSRVCSERVICE >> $INSTALLINFO 2>> $LOG
res2=$?
if [ -f $SCSRVCLNK3 ]; then
    ls -l $SCSRVCLNK3 >> $INSTALLINFO 2>> $LOG
    res2=$?
else
    echo "File not exist, So can't list file $SCSRVCLNK3" >> $LOG
fi
if [ -f $SCSRVCLNK5 ]; then
    ls -l $SCSRVCLNK5 >> $INSTALLINFO 2>> $LOG
else
    echo "File not exist, So can't list file $SCSRVCLNK5" >> $LOG
fi
res4=$?

res5=0
if [ ! -z "$SCSRVCLNK2" ]
then
ls -l $SCSRVCLNK2 >> $INSTALLINFO 2>> $LOG
res5=$?
fi

res6=0
if [ ! -z "$SCSRVCLNK4" ]
then
ls -l $SCSRVCLNK4 >> $INSTALLINFO 2>> $LOG
res6=$?
fi
[ "$QOPT" != "1" ] && print_result `expr $res1 + $res2 `

if [ ! -z "$DRIVER" ]
then
#*****************************************************************
$ECHO2TERM $NOPT "Collecting the driver information"
#*****************************************************************
echo "character device: " >> $DRVINFO
ls -l $DRIVER >> $DRVINFO 2>> $LOG
RES=$?
[ "$QOPT" != "1" ] && print_result $RES
fi

if [ ! -z "$DMESG" ]
then
#*****************************************************************
$ECHO2TERM $NOPT "Collecting the dmesg output"
#*****************************************************************
$DMESG >> $DMSGOUT 2>> $LOG
RES=$?
[ "$QOPT" != "1" ] && print_result $RES
fi

if [ ! -z "$JRNLBOOT" ]
then
#*****************************************************************
$ECHO2TERM $NOPT "Collecting the journalctl -b output"
#*****************************************************************
$JRNLBOOT >> $JRNLOUT 2>> $LOG
RES=$?
[ "$QOPT" != "1" ] && print_result $RES 
fi

if [ ! -z "$JRNLSCSRVC" ]
then
#*****************************************************************
$ECHO2TERM $NOPT "Collecting the journalctl -u scsrvc output"
#*****************************************************************
$JRNLSCSRVC >> $JRNLSCOUT 2>> $LOG
RES=$?
[ "$QOPT" != "1" ] && print_result $RES 
fi

#*****************************************************************
$ECHO2TERM $NOPT "Collecting data from /etc"
#*****************************************************************
cd /etc

ETC_FILE_DIR=etc_files_info
ETC_TEMP_DIR=$TARGET_DIR/$ETC_FILE_DIR
ETC_RECURSIVE_LISTING_FILE=$ETC_TEMP_DIR/etc_recursive_listing
mkdir $ETC_TEMP_DIR

ls -R -al > $ETC_RECURSIVE_LISTING_FILE

FILES_TO_TAR=`ls`

for file in $FILES_TO_TAR; do
    if [ ! -h "$file" ]	&&
       [ ! -d "$file" ] &&
       [ ! -p "$file" ] &&
       [ ! -c "$file" ] &&
       [ ! -b "$file" ]; then
	cp -p $file $ETC_TEMP_DIR/
    fi
done

cd $TARGET_DIR
tar -cvf $ETC_FILES_TAR $ETC_FILE_DIR/ > /dev/null 2>> $LOG;

if [ "$COMPRESS" = "gzip" ]; then
    FOPT="-f";
else
    FOPT="";
fi;

$COMPRESS $FOPT $ETC_FILES_TAR 2>> $LOG;
RES=$?;
[ "$QOPT" != "1" ] && print_result $RES;

rm -rf $ETC_TEMP_DIR

if [ ! -z "$SYSTEMDFILES" ]
then
#*****************************************************************
$ECHO2TERM $NOPT "Collecting unit files from $SYSTEMDFILES"
#*****************************************************************
cd $SYSTEMDFILES

ETC_SYSTEMD_FILE_DIR=etc_systemd_files_info
ETC_TEMP_DIR=$TARGET_DIR/$ETC_SYSTEMD_FILE_DIR
mkdir $ETC_TEMP_DIR

FILES_TO_TAR=`ls`

for file in $FILES_TO_TAR; do
    if [ ! -d "$file" ] &&
       [ ! -p "$file" ] &&
       [ ! -c "$file" ] &&
       [ ! -b "$file" ]; then
	cp -L $file $ETC_TEMP_DIR/
    fi
done

cd $TARGET_DIR
tar -cvf $ETC_SYSTEMD_FILES_TAR $ETC_SYSTEMD_FILE_DIR/ > /dev/null 2>> $LOG;

if [ "$COMPRESS" = "gzip" ]; then
    FOPT="-f";
else
    FOPT="";
fi;

$COMPRESS $FOPT $ETC_SYSTEMD_FILES_TAR 2>> $LOG;
RES=$?;
[ "$QOPT" != "1" ] && print_result $RES;

rm -rf $ETC_TEMP_DIR
fi

# This is some miscellaneous information that needs to be captured 
# on different platforms .. 
#*****************************************************************
$ECHO2TERM $NOPT "Collecting miscellaneous information"
#*****************************************************************

if [ `uname` = Linux ]
then
	$LINUX_GET_PROC_DEVICES >> $PROCDEV 2>> $LOG
	$LINUX_GET_PROC_FILESYSTEMS >> $PROCFS 2>> $LOG
	$LINUX_GET_PROC_SLABINFO >> $PROCSLAB 2>> $LOG
	if [ -f /boot/System.map-`uname -r` ]; then
	    cp -p /boot/System.map-`uname -r` $TARGET_DIR/ 2>> $LOG
        else
	    echo "File /boot/System.map-`uname -r` not exist" >> $LOG
	fi

elif [ `uname` = SunOS ]
then
	$SOLARIS_GET_ETC_SYSTEM >> $ETCFS 2>> $LOG
elif [ `uname` = HP-UX ]
then
	$HPUX_GET_OLD_SYSLOG_LOG 2>> $LOG
	$TUNABLE >> $TUNEABLEINFO 2>> $LOG
# for HPUX 11.31, dfstab contains list of exported file systems
	cp -p /etc/dfs/dfstab $TARGET_DIR/dfstab_entry 2>> /dev/null

	SC_CUR_DIR=`pwd`
	cd $HPUX_RESMON_LOG_PATH 
	tar -chvf $HPUX_RESMON_LOG_TAR log 2>> /dev/null
	cd $SC_CUR_DIR

	mkdir $HPUX_TOMBSTONES_TARG_DIR
	cp -p $HPUX_TOMBSTONES_PATH/ts98 $HPUX_TOMBSTONES_TARG_DIR/ 2>> /dev/null
	cp -p $HPUX_TOMBSTONES_PATH/ts99 $HPUX_TOMBSTONES_TARG_DIR/ 2>> /dev/null
elif [ `uname` = AIX ]
then
	USER_DBG=`cat /etc/syslog.conf | grep "user.debug" | tail -1 | awk {'print $2'}`
	if [ -f "$USER_DBG" ]; then
		cp -p $USER_DBG $TARGET_DIR/user_syslog.log
	fi
	KERN_DBG=`cat /etc/syslog.conf | grep "kern.debug" | tail -1 | awk {'print $2'}`
	if [ -f "$KERN_DBG" -a "$KERN_DBG" != "$USER_DBG" ]; then
		cp -p $KERN_DBG $TARGET_DIR/kern_syslog.log
	fi
	MISC_DBG=`cat /etc/syslog.conf | grep "*.debug" | tail -1 | awk {'print $2'}`
	if [ -f "$MISC_DBG" -a "$MISC_DBG" != "$USER_DBG" ]; then
		cp -p $MISC_DBG $TARGET_DIR/syslog.log
	fi
	MISC_CRIT=`cat /etc/syslog.conf | grep "*.crit" | tail -1 | awk {'print $2'}`
	if [ -f "$MISC_CRIT" ]; then
		cp -p $MISC_CRIT $TARGET_DIR/syslog_crit.log
	fi
	cp -p /etc/scsrvc $TARGET_DIR/etc_scsvc 1>/dev/null 2>/dev/null
	cp -p /etc/rc.scsrvc $TARGET_DIR/etc_rc_scsrvc 1>/dev/null 2>/dev/null


# Collect output of errpt -a
	errpt -a > $TARGET_DIR/error_report.out 2>/dev/null
fi

version >> $GIVERSION; 

# Archive the results.
archive_results
rm -rf $TARGET_DIR

$ECHO2TERM "" 
$ECHO2TERM "For [SKIPPED] or [FAILED] messages see gatherinfo.log after extracting the archive" 

$ECHO2TERM ""
$ECHO2TERM $EOPT "----------------------------------------------------------------------------"
