#!/bin/bash
#  Copyright (C) 2009 McAfee, Inc.  All rights reserved.
# This script collects information about the current system state
# as well as McAfee Solidifier product related info.
# To add functionality for collecting more info :
# add the platform specific command in define_commands
# mention the out file name for that particular command
# then call the collect function with proper arguments.
# Also make sure that no output (except from the error) should be 
# printed when "-q" option is provided.

THIS_PLATFORM="`uname`";
THIS_ARCH="`uname -m`";
if [ "$THIS_PLATFORM" = "Linux" ]; then
    if [ -f "/etc/redhat-release" ]; then
        RELEASE_PATH="/etc/redhat-release"
    elif [ -f "/etc/SuSE-release" ]; then
        RELEASE_PATH="/etc/SuSE-release"
    elif [ -f "/etc/lsb-release" ]; then
        RELEASE_PATH="/etc/lsb-release"
    elif [ -f "/VERSION.os" ]; then
        RELEASE_PATH="/VERSION.os"
    elif [ -f "/etc/issue" ]; then
        RELEASE_PATH="/etc/issue"
    fi

    if [ ! -z $RELEASE_PATH ]; then
        if [ $RELEASE_PATH = "/etc/lsb-release" ]; then
	    RETVAL1=`cat $RELEASE_PATH | grep -w DISTRIB_DESCRIPTION | head -n 1 | awk -F"=" '{print $2}' | awk -F"\"" '{print $2}'|awk '{print $1}'`
	    if [ "$RETVAL1" = "Ubuntu" ]; then
            	THIS_DISTRIBUTION="Ubuntu"
	    fi
        elif [ "$RELEASE_PATH" = "/etc/issue" ]; then
                release=`cat $RELEASE_PATH`
                RETVAL1=`echo $release | awk -F " " '{print $1" "$2" "$3}'`
	        if [ "$RETVAL1" = "Wind River Linux" ]; then
            	   THIS_DISTRIBUTION="WindRiver"
	        fi
        elif [ "$RELEASE_PATH" = "/VERSION.os" ]; then
	        RETVAL1=`cat $RELEASE_PATH | head -n 1 |  awk -F "-" '{ print $1 }' | awk -F "_" '{ print $2 }'`
	        if [ "$RETVAL1" = "wrl60" ]; then
            	   THIS_DISTRIBUTION="WindRiver"
	        fi
        else	    
    	    RETVAL1=`eval head -n 1 $RELEASE_PATH | awk '{print $1}'`
    	    if [ "$RETVAL1" = "CentOS" ]; then
            	THIS_DISTRIBUTION="CentOS"
    	    elif [ "$RETVAL1" = "Enterprise" ]; then
            	THIS_DISTRIBUTION="OracleEL"
    	    elif [ "$RETVAL1" = "SUSE" ]; then
            	THIS_DISTRIBUTION="SUSE"
    	    fi
    	fi
    fi
fi

# function for querying if an element is in an array
# first argument : element 
# second argument: array 
# Returns 1 if element is present else returns 0

_contains () 
{
    ELEMENT=$1  
    shift
    ARRAY=$*
    for j in $ARRAY; do 
	[ "$j" = "$ELEMENT" ] && return 1
    done 
    return 0
}

# definition of variables.
define_variables ()
{
    PATH=$PATH:/bin:/sbin:/usr/bin:/usr/sbin:/usr/local/bin:/usr/local/sbin;
    [ `uname` = "SunOS" ] && PATH=$PATH:/usr/ccs/bin;
    [ `uname` = "HP-UX" ] && PATH=$PATH:/usr/contrib/bin;

    SC_SCANALYZER_VERSION="6.6.2"

    SC_ORIG_DIR="`pwd`" 
    SC_TIME_STAMP=`date +"%m-%d-%y_%H-%M-%S"`
    SC_ROOT_UID=0
    SC_IS_SOLIDIFIER_INSTALLED=0

    # put hostname in the filename to be generated
    SC_HOSTNAME=`hostname | awk -F. {'print $1'}`;

    # Variable to track if compression is required or not.
    SC_NEEDS_COMPRESSION=0

    # handle `blank hostname' and `blanks in hostname'
    if [ -z "$SC_HOSTNAME" ] || \
	[ "`echo $SC_HOSTNAME | awk {'print $2'}`" ];
    then
	SC_HOSTNAME="info";
    fi 

    SC_TARGET_DIR="data";
    if [ "$NO_TIME_STAMP" -eq 0 ]; then
	OUT_REPORT="$SC_TARGET_DIR/report-$SC_HOSTNAME-$SC_TIME_STAMP";
    else
	OUT_REPORT="$SC_TARGET_DIR/scan_info"
	SC_NEEDS_COMPRESSION=1
    fi

    SC_REPORT_COMPRESSED=0

    # Generate a random name.
    SC_BASENAME=`basename $0`;

    # To handle different default shells for different platforms.
    SH=/bin/sh
    BASH=/bin/bash

    # following are the options given to echo for different shells
    #  	-n : continue in same line (Bash)
    #  	-e : translate special characters like \r, \t (Bash)

    [ "$SHELL" = "$BASH" ] && {
    NOPT="-n";
    EOPT="-e"; }

    # error codes
    SC_ERR_USER_NOT_ROOT=100;
    SC_ERR_INVALID_RUNTIME=101;
    SC_ERR_SOLIDIFIER_NOT_INSTALLED=102;
    SC_ERR_UNSUPPORTED_PLATFORM=103;
    SC_ERR_TOO_MANY_ARGUMENTS=104;
    SC_ERR_TOO_LESS_ARGUMENTS=105;
    SC_ERR_FILE_NOT_FOUND=106;
    SC_ERR_COULD_NOT_ARCHIVE=107;

    # some common messages for analysis file.

    CONF_FILE=/etc/mcafee/solidcore/solidcore.conf;
    if [ -f $CONF_FILE ]; then
	DEFAULT_PORT=`cat $CONF_FILE | grep ServicePort | awk {'print $3'}`
    else
	DEFAULT_PORT="5123";
    fi

    if [ "$THIS_PLATFORM" = "Linux" ]; then
    	if [ -f "$RELEASE_PATH" ]; then
            if [ "$RELEASE_PATH" = "/etc/issue" ]; then
                release=`cat $RELEASE_PATH`
                DISTRO_STRING=`echo $release | awk -F " " '{print $1" "$2" "$3}'`
	    elif [ $RELEASE_PATH = "/etc/lsb-release" ]; then
	    	DISTRO_STRING=`cat $RELEASE_PATH | grep -w DISTRIB_DESCRIPTION | head -n 1 | awk -F"=" '{print $2}' | awk -F"\"" '{print $2}' | cut -d '.' -f 1-2`
	    elif [ $RELEASE_PATH = "/VERSION.os" ]; then
	        DISTRO_STRING=`cat $RELEASE_PATH | head -n 1 |  awk -F "-" '{ print $1 }' | awk -F "_" '{ print $2 }'`
            else
                DISTRO_STRING=`cat $RELEASE_PATH | head -n 1 | awk -F " [(]" '{ print $1 }' | awk -F "." '{ print $1 }' | awk -F "-" '{ print $NF }'`
            fi
            DISTRO_STRING=`eval echo "$DISTRO_STRING"`
        else
	    DISTRO_STRING="UNKNOWN"
        fi
    elif [ "$THIS_PLATFORM" = "SunOS" ]; then
	CMD_GET_DISTRIBUTION="`uname -r`";
    	DISTRO_STRING="$CMD_GET_DISTRIBUTION"
    elif [ "$THIS_PLATFORM" = "HP-UX" ]; then
	CMD_GET_DISTRIBUTION="`uname -r`";
    	DISTRO_STRING="$CMD_GET_DISTRIBUTION"
    elif [ "$THIS_PLATFORM" = "AIX" ]; then
	CMD_GET_DISTRIBUTION="`uname -v`.`uname -r`";
    	DISTRO_STRING="$CMD_GET_DISTRIBUTION"
    fi

    MSG_ERR_UNSUPPORTED_OS="Installed OS ($DISTRO_STRING) is not supported by McAfee Solidifier. ";
    MSG_ERR_UNSUPPORTED_KERNEL="Installed kernel is not supported by McAfee Solidifier. ";
    MSG_ERR_LOG_REQD_DISK_SPACE="Total disk space available is less than the space required to generate scanalyzer report. Required disk space: 4 MB. Available disk space: ";
    MSG_ERR_LOW_DISK_SPACE="Total disk space available is less than the required space. Required disk space: 70 MB. Available disk space: ";
    MSG_ERR_LOW_DISK_ROOT_SPACE="Disk space available in root partition is less than the required space. Required disk space: 2 MB. Available disk space: ";
    MSG_ERR_UNSUPPORTED_FILESYSTEM="No supported filesystem found.";
    MSG_ERR_LOW_PHYSICAL_MEMORY="Total memory available is less than required memory. Required memory: 512000 Kbytes Available memory: ";
    MSG_ERR_SOLIDIFIER_INSTALLED="McAfee Solidifier Product already installed.";
    MSG_ERR_DEFAULT_PORT_IN_USE="Port $DEFAULT_PORT is already in use.";
    MSG_ERR_REQUIRED_PKG_MISSING="Following required packages are missing:";
    MSG_ERR_PROBLEMATIC_PKG_FOUND="Following problematic packages found:";
    MSG_ERR_PROBLEMATIC_MODULE_FOUND="Following problematic modules found:";
    MSG_ERR_GENKEX_MISSING="genkex is not present on this system.";
    MSG_ERR_STACK_PROTECTION_ON="User stack protection is enabled. Process\
    protection will be enabled on McAfee Solidifier installation.";
    MSG_ERR_SOLIDIFIER_CONFLICTS="Following file conflicts with McAfee Solidifier file:";
    MSG_ERR_NM_VERSION_NON_HP="nm binary installed on the system does not belong to HP version.";
    MSG_LABEL_OS_DISTRIBUTION="OS Distribution";
    MSG_LABEL_KERNEL="Kernel Version";
    MSG_LABEL_PATCH_LEVEL="Patch Level";
    MSG_LABEL_DISK_SPACE="Disk Space";
    MSG_LABEL_FILE_SYSTEMS="File Systems";
    MSG_LABEL_PROBLEM_OR_REQD_PKGS="";
    MSG_LABEL_PHYSICAL_MEMORY="Physical Memory";
    MSG_LABEL_SOLIDIFIER_INSTALLATION="McAfee Solidifier Product Installation";
    MSG_LABEL_PORTS="Default Port";
    MSG_LABEL_PREREQUISITE_PKGS="Pre-requisite Packages";
    MSG_LABEL_PROBLEMATIC_PKGS="Problematic Packages";
    MSG_LABEL_PROBLEMATIC_MODULES="Problematic Modules";
    MSG_LABEL_STACK_PROTECTION_ON="User Stack Protection (Process Protection)";
    MSG_LABEL_SOLIDIFIER_CONFLICTS="McAfee Solidifier Conflicts";
    MSG_LABEL_NM_VERSION="Version of nm Binary";

    # file names.
    LOGFILE="$SC_TARGET_DIR/scanalyzer.log"
    BLACKLIST_LOGFILE="$SC_TARGET_DIR/blacklist.log"

    OUT_TOOL_VERSION="$SC_TARGET_DIR/scanalyzer.version.info";
    OUT_SYSTEM_DATE="$SC_TARGET_DIR/date.out.info"; 
    OUT_SYSTEM_CONFIG="$SC_TARGET_DIR/system.config.info";
    OUT_CPU_INFO="$SC_TARGET_DIR/cpu.info";
    OUT_UPTIME="$SC_TARGET_DIR/uptime.info";
    OUT_SYSTEM_STATE="$SC_TARGET_DIR/system.state.info";
    OUT_PACKAGES_LIST="$SC_TARGET_DIR/pkg.list.info";
    OUT_SOLIDIFIER_PKG_INFO="$SC_TARGET_DIR/solidcore.pkg.info";
    OUT_PROCESS_INFO="$SC_TARGET_DIR/process.info";
    OUT_SERVICES_INFO="$SC_TARGET_DIR/installed.services.list.info";
    OUT_LOADED_MODULES="$SC_TARGET_DIR/loaded.modules.info";
    OUT_DISK_SPACE_INFO="$SC_TARGET_DIR/disk.space.info";
    OUT_ENV_VARS="$SC_TARGET_DIR/environment.vars.list.info";
    OUT_ETC_SYSTEM="$SC_TARGET_DIR/etc.system.info";
    OUT_SYSLOG_CONF="$SC_TARGET_DIR/syslog.conf.info";
    OUT_ETC_DIR_LISTING="$SC_TARGET_DIR/etc.rc.listing.info";
    OUT_PORTS_INFO="$SC_TARGET_DIR/ports.info";
    OUT_MOUNT_TABLE="$SC_TARGET_DIR/mount.table.info";
    OUT_FS_TABLE="$SC_TARGET_DIR/fs.table.info";
    OUT_KERNEL_SYMBOLS="$SC_TARGET_DIR/exported.kernel.symbols.info";
    OUT_PHYSICAL_RAM="$SC_TARGET_DIR/physical.memory.info";
    OUT_ADDR_SPACE_INFO="$SC_TARGET_DIR/addr.space.info";
    OUT_NW_INTF_INFO="$SC_TARGET_DIR/nw.intf.info";
    OUT_HOSTNAME_INFO="$SC_TARGET_DIR/hostname.info";
    OUT_ROUTE_TABLE="$SC_TARGET_DIR/route.table.info";
    OUT_GLIBC_VERSION="$SC_TARGET_DIR/glibc.version.info";
    OUT_LS_OPEN_FILES="$SC_TARGET_DIR/open.files.list.info";
    OUT_NFS_EXPORTS="$SC_TARGET_DIR/nfs.exports.info";
    OUT_OS_PATCH_INFO="$SC_TARGET_DIR/os.patch.info";
    OUT_RESOURCES_INFO="$SC_TARGET_DIR/resources.info";
    OUT_ICONV_INFO="$SC_TARGET_DIR/iconv.info";
    OUT_RC_LOG="$SC_TARGET_DIR/rc.log.info";
    OUT_OLD_RC_LOG="$SC_TARGET_DIR/rc.log.old.info";
    OUT_MOD_REGISTER="$SC_TARGET_DIR/mod_register.info";
    OUT_STAND_LISTING="$SC_TARGET_DIR/stand.listing.info";
    OUT_LOAD_MODS="$SC_TARGET_DIR/loadmods.info";
    OUT_DEV_INFO="$SC_TARGET_DIR/lsdev.out.info";
    OUT_SBIN_RC_LISTING="$SC_TARGET_DIR/sbin.rc.listing.info";
    OUT_SBIN_INITD_LISTING="$SC_TARGET_DIR/sbin.initd.listing.info";
    OUT_SWAGENT_LOG="$SC_TARGET_DIR/swagent.log.info";
    OUT_CRASH_DUMP="$SC_TARGET_DIR/crash.dump.info";
    OUT_SOL10_SMF="$SC_TARGET_DIR/smf.info";
    OUT_INITTAB_ENTRY="$SC_TARGET_DIR/inittab.info";
    OUT_PKGCHK_INFO="$SC_TARGET_DIR/pkhchk.info";
    OUT_LOCALE_INFO="$SC_TARGET_DIR/locale.info";
    OUT_PKG_VER_INFO="$SC_TARGET_DIR/pkhver.info";
    OUT_SYSD_INFO="$SC_TARGET_DIR/systemd.info";

    # capabilities Name.
    CAP_SYSTEM_CONFIG="System Configuration"; 
    CAP_CPU_INFO="CPU Details";
    CAP_UPTIME="Uptime";
    CAP_SYSTEM_STATE="System State Information";
    CAP_PACKAGES_LIST="List of Packages Installed";
    CAP_SOLIDIFIER_PKG_INFO="McAfee Solidifier Package(s) Information";
    CAP_PROCESS_INFO="Process Information";
    CAP_SERVICES_INFO="Installed Services Information";
    CAP_LOADED_MODULES="Loaded Modules (Kernel Extensions)";
    CAP_DISK_SPACE_INFO="Disk Space Information"; 
    CAP_ENV_VARS="Environment Variables";
    CAP_ETC_DIR_LISTING="/etc/rc Directory Listing (Recursive)";
    CAP_PORTS_INFO="Ports Information";
    CAP_MOUNT_TABLE="Mount Table";
    CAP_FS_TABLE="File System Table";
    CAP_KERNEL_SYMBOLS="Exported Kernel Symbols Information"; 
    CAP_PHYSICAL_RAM="Physical RAM";
    CAP_ADDR_SPACE_INFO="Address Space Information";
    CAP_NW_INTF_INFO="Network Interfaces Information";
    CAP_HOSTNAME_INFO="Hostname of the System";
    CAP_ROUTE_TABLE="Route Information";
    CAP_GLIBC_VERSION="Glibc Version";
    CAP_LS_OPEN_FILES="List of Open Files"; 
    CAP_NFS_EXPORTS="NFS Exports/Shares"; 
    CAP_OS_PATCH_INFO="Operating System Patch Information"; 
    CAP_RESOURCES_INFO="Resources Info";
    CAP_ICONV_INFO="Iconv Information";
    CAP_INITTAB_ENTRY="Init tab entry"
    CAP_RC_LOG="/etc/rc.log Information";
    CAP_OLD_RC_LOG="/etc/rc.log.old Information";
    CAP_MOD_REGISTER="mod_register Information";
    CAP_STAND_LISTING="/stand listing (Recursive)";
    CAP_LOAD_MODS="loadmod Information";
    CAP_DEV_INFO="Device Information";
    CAP_SBIN_RC_LISTING="/sbin/rc listing (Recursive)";
    CAP_SBIN_INITD_LISTING="/sbin/init.d listing (Recursive)";
    CAP_SWAGENT_LOG="swagent.log Information";
    CAP_CRASH_DUMP_INFO="Crash dump settings";
    CAP_SOL10_SMF="service managment framework data"
    CAP_SYSLOG_CONF="/etc/syslog.conf contents"
    CAP_PKGCHK_INFO="Required packages consistancy info."
    CAP_LOCALE_INFO="Locale info"
    CAP_PKG_VER_INFO="Packages version information."
    CAP_SYSD_INFO="Systemd specific information."

    # Linux disto related definitions.
    AS3="Red Hat Enterprise Linux AS release 3"
    ES3="Red Hat Enterprise Linux ES release 3"
    WS3="Red Hat Enterprise Linux WS release 3"

    AS4="Red Hat Enterprise Linux AS release 4"
    ES4="Red Hat Enterprise Linux ES release 4"
    WS4="Red Hat Enterprise Linux WS release 4"
    CS4="CentOS release 4"

    EL5="Red Hat Enterprise Linux Server release 5"
    OEL5="Enterprise Linux Enterprise Linux Server release 5"
    CS5="CentOS release 5"

    EL6="Red Hat Enterprise Linux Server release 6"
    OEL6="Enterprise Linux Enterprise Linux Server release 6"
    CS6="CentOS release 6"
    CS6_0="CentOS Linux release 6"
    
    SES9="SUSE LINUX Enterprise Server 9"
    SES9x="SuSE Linux 9"

    SES10="SUSE Linux Enterprise Server 10"

    SED11="SUSE Linux Enterprise Desktop 11"
    SES11="SUSE Linux Enterprise Server 11"

    LRH72="Red Hat Linux release 7"

    Opensuse10="openSUSE 10"
    Opensuse11="openSUSE 11"

    Ubuntu1204="Ubuntu 12.04"

    LWR14="wr1"
    LWR60="wrl60"
    LWR40="Wind River Linux 4.0"
    LWR42="Wind River Linux 4.2"
    LWR43="Wind River Linux 4.3"
    LWR="Wind River Linux"
}


# Function Definitions 

print_result()
{
    # move to the begining of line
    echo $NOPT $EOPT "\r"
    # move 65 spaces to the right
    # tput cuf 65

    i=65
    while [ "$i" != "0" ]
    do
	tput cuf1 
	i=`expr $i - 1`
    done
    
    echo $NOPT " ... "
    
    [ -z "$1" ] && { 
	tput setf 1 
	echo "[SKIPPED]";
	tput setf 0
	return;
    }
    
    if [ $1 -eq 0 ]
    then
	tput setf 2
	echo "[  OK  ]"
	tput setf 0
    else
	tput setf 4
    	echo "[FAILED]"
	tput setf 0
    fi	    
}

# function for collecting the info. 
# First argument  : capability name
# second argumnet : temp out file name
# third argument  : command for collecting info

collect () 
{
    capability="$1";
    outfile="$2"; 
    commands="$3"; 

    if [ -z "$capability" ] || [ -z "$outfile" ] || [ -z "$commands" ]; 
    then
    	return $ERR_INVALID_INVOCATION; 
    fi; 

    FIRST="";
    SECOND="";
    THIRD="";
    FOURTH="";
    FIFTH="";

    # special handling required for getting /etc/rc* listing since we use:
    # 'find /etc/rc* -name '*' -type f -exec ls -li {} \;' where ';' 
    # not optional. this is the only case where ';' cannot be misunderstood
    # as a delimiter. hence doing it. to go if we can find sth better.
    if echo $commands | grep ';' | grep 'find' >/dev/null 2>&1; then
    	FIRST="$commands";
    else 
	# we take care of upto five commands per capability. 
	# for supporting more, either edit the code or use another 
	# capability for some of the commands :) 
	num=`echo $commands | awk -F';' {'print NF'}`; 
	FIRST=`echo $commands | awk -F';' {'print $1'}`;
	SECOND=`echo $commands | awk -F';' {'print $2'}`;
	THIRD=`echo $commands | awk -F';' {'print $3'}`;
	FOURTH=`echo $commands | awk -F';' {'print $4'}`;
	FIFTH=`echo $commands | awk -F';' {'print $5'}`;
    fi; 

    
    
    RETFIRST=0;
    RETSECOND=0;
    RETTHIRD=0;
    RETFOURTH=0;
    RETFIFTH=0;
    
    if [ "$QOPT" = 0 ]; then
       echo "Collecting $capability.."
    fi
    echo "-----------------------------------------------" >>$outfile; 
    echo "Commands used for collecting $capability: " >>$outfile;
    [ "$FIRST" ] && echo "	>> $FIRST" >>$outfile;
    [ "$SECOND" ] && echo "	>> $SECOND" >>$outfile;
    [ "$THIRD" ] && echo "	>> $THIRD" >>$outfile;
    [ "$FOURTH" ] && echo "	>> $FOURTH" >>$outfile;
    [ "$FIFTH" ] && echo "	>> $FIFTH" >>$outfile;
    echo "-----------------------------------------------" >>$outfile; 
    
    EXECUTED=0;
    if [ "$FIRST" ]; then
	if [ "$THIS_DISTRIBUTION" = "Ubuntu" ] && [ "$FIRST" = "service --status-all" ]; then
		
		$FIRST >>$outfile 2>&1
		RETFIRST=$?;
		
	else 
		$FIRST >>$outfile 2>$LOGFILE; 
		RETFIRST=$?;
    	fi;

	echo "-----------------------------------------------" >>$outfile; 
	EXECUTED=1;
    fi 

    if [ "$SECOND" ]; then 
	$SECOND >>$outfile 2>$LOGFILE; 
    	RETSECOND=$?;
	echo "-----------------------------------------------" >>$outfile; 
	EXECUTED=1;
    fi 

	
    if [ "$THIRD" ]; then 
	$THIRD >>$outfile 2>$LOGFILE; 
    	RETTHIRD=$?;
	echo "-----------------------------------------------" >>$outfile; 
	EXECUTED=1;
    fi 
    
    if [ "$FOURTH" ]; then 
	$FOURTH >>$outfile 2>$LOGFILE; 
    	RETFOURTH=$?;
	echo "-----------------------------------------------" >>$outfile; 
	EXECUTED=1;
    fi 
	
    if [ "$FIFTH" ]; then 
	$FIFTH >>$outfile 2>$LOGFILE; 
    	RETFIFTH=$?;
	echo "-----------------------------------------------" >>$outfile; 
	EXECUTED=1;
    fi; 

    RES=`expr $RETFIRST + $RETSECOND + $RETTHIRD + $RETFOURTH + $RETFIFTH`;
    
    if [ $RES -ne 0 ]; then
	echo "Information not available." >> "$outfile";
	echo "-----------------------------------------------" >>$outfile; 
    fi
}

# print the usage of the tool. 
usage()
{
    HELPSTR_HELP="-h --help		Displays help for ScAnalyzer";
    HELPSTR_VERSION="-v --version	Displays version of ScAnalyzer";
    HELPSTR_INSTALLDIR="-r <install path>  	path for disk space check"
    HELPSTR_CMP="-d <rep1  rep2>	Compare differences in two ScAnalyzer reports";
    HELPSTR_OUTPUT1="-o <output file>	Writes the output to the output file."
    HELPSTR_QUIET="-q			quiet mode";
    HELPSTR_NOTS="-n			No time stamp in output file name";

    
    PROG_NAME=`basename $0`

    echo "Usage:"
    echo "$PROG_NAME -h|--help"
    echo "$PROG_NAME -v|--version" 
    echo "$PROG_NAME -d <report1> <report2>"
    echo "$PROG_NAME -o <output file>"
    echo "$PROG_NAME -r <install path>"
    echo "$PROG_NAME -q"
    echo "$PROG_NAME -n"

    # if `help' has been asked for display and exit with 0 ecode else 
    # simply exit with whatever ecode has been sent to the function 
    if [ ! -z "$1" ] && [ "$1" != 0 ]
    then 
 	   exit $1
    else
	    echo
	    echo "    $HELPSTR_HELP";
	    echo "    $HELPSTR_VERSION";
	    echo "    $HELPSTR_CMP";
	    echo "    $HELPSTR_INSTALLDIR"
	    echo "    $HELPSTR_OUTPUT1"
	    echo "    $HELPSTR_QUIET";
	    echo "    $HELPSTR_NOTS";
	    exit 0
    fi
}

# function to validate for the root user 
sc_validate_user ()
{
    CUID=`id | cut -b 5`; 
    if [ "$CUID" != "$SC_ROOT_UID" ]
    then
        echo "Must be root to run this script.";
        exit $SC_ERR_USER_NOT_ROOT;
    fi  
}

# print out the version of the tool
sc_version()
{
    echo "ScAnalyzer version: $SC_SCANALYZER_VERSION"
}

# function to compare the scanalyzer reports in text format 
# First argument  : First report file path (txt format) 
# Second argumnet : Second report file path (txt format) 
# Third argument  : Output report (difference) file path 

sc_compare_reports_txt () 
{
	REPORT1="$1"; 
	REPORT2="$2";
	DIFFREPORT="$3"; 

	[ -z "$REPORT1" ] && return 1; 
	[ -z "$REPORT2" ] && return 1; 
	[ -z "$DIFFREPORT" ] && return 1; 

	if [ $REPORT1 -ef $REPORT2 ]; then 
		echo "Both files point to same physical file on the device." 
		return 1; 
	fi 

	touch $DIFFREPORT 2>/dev/null 1>&2; 
	if [ $? -ne 0 ]; then 
		echo "Unable to create file $DIFFREPORT."
		echo "Check if target is write protected.";
		return 1;
	fi

	diff $REPORT1 $REPORT2 >$DIFFREPORT; 
	echo "Successfully prepare the comparison report $DIFFREPORT.";
	return 0; 
}


# function to compare the scanalyzer reports 
sc_compare_reports () 
{
    REPORT_FILE="diff"; 
    rm -f  $REPORT_FILE; 

    if [ ! -r "$SCAN_REPORT1" ]; then
	echo "Report file $SCAN_REPORT1 is not readable"; 
        exit $SC_ERR_FILE_NOT_FOUND;
    fi; 
    
    if [ ! -r "$SCAN_REPORT2" ]; then
	echo "Report file $SCAN_REPORT2 is not readable"
	exit $SC_ERR_FILE_NOT_FOUND;
    fi; 
    
    echo "Generating \"$REPORT_FILE\" report after comparing the reports";
    echo "$SCAN_REPORT1 and $SCAN_REPORT2";

    diff $SCAN_REPORT1 $SCAN_REPORT2 > $REPORT_FILE
}



# function for appending data to a single report. 
# first argument : capability name 
# second argument: output file name
_append_to_report () 
{
	HEADER="$1"; 
	FILENAME="$2"; 

	if [ -z "$HEADER" ] || [ -z "$FILENAME" ]; 
	then
		:
		return;
	fi; 

	if [ -f "$FILENAME" ]; then
		echo "" >> $OUT_REPORT;
		echo "[ $HEADER ]" >> $OUT_REPORT;
		cat $FILENAME >> $OUT_REPORT;
		echo "" >> $OUT_REPORT;
	fi 
}

# Function to append analysis message in the analysis report 
# first argument : severity [ 0: info; 1: warning; 2: critical ] 
# second argument: message to append 
# third argument : information being analyzed
_append_analysis () 
{
	[ "$1" = "0" ] && SEVERITY="INFORMATION:";
	[ "$1" = "1" ] && SEVERITY="WARNING:";
	[ "$1" = "2" ] && SEVERITY="CRITICAL:";
	MSG=$2; 

	if [ "$1" = "3" ]; then
	    echo "$MSG" >>$OUT_REPORT;	
	    return
        fi

	if [ -z "$SEVERITY" ] || [ -z "$MSG" ]; then return; fi 

	echo "[ Errors/Warnings while analyzing [$3] ]">>$OUT_REPORT;
	echo "$SEVERITY $MSG" >>$OUT_REPORT;	
	echo "" >>$OUT_REPORT;
}

# function meant for searching for blacklisted modules, driver, packages
scan_for_black_list()
{
    echo "Scanning for blacklisted installations..";
    echo "";
    for i in `cat $BLIST | grep -v \# | grep -v ^$`; 
    do
	ename=`echo $i|cut -d: -f1`
	etype=`echo $i|cut -d: -f2`
	if [ "$etype" = "package" ]; then
	    $CMD_PACKAGE_QUERY $ename >/dev/null 2>&1;
            if [ $? -eq 0 ]; then 
		echo " - Black-listed package $i found in the system"\
		    >> $BLACKLIST_LOGFILE;
	    fi; 
	elif [ "$etype" = "module" ]; then
	    $CMD_LOADED_MODULES | grep $ename>/dev/null 2>&1;
	    if [ $? -eq 0 ]; then
		echo " - Black-listed module $i found in the system"\
		    >> $BLACKLIST_LOGFILE;
	    fi; 
	elif [ "$etype" = "driver" ]; then
	    $CMD_LOADED_DRIVERS | grep $ename>/dev/null 2>&1;
	    if [ $? -eq 0 ]; then
		echo " - Black-listed driver $i found in the system"\
		    >> $BLACKLIST_LOGFILE;
	    fi
	fi; 
    done

    if [ -s "$BLACKLIST_LOGFILE" ]; then
	echo "Blacklisted installations found on the system:";
	cat $BLACKLIST_LOGFILE; 
    else
	echo "No blacklisted installations found on the system.";
    fi; 
    echo "";
}

# function to collect nw intf info on hp-ux
collect_nw_intf_info_hpux () 
{
    # separate function required since no standard command available on hpux
    # for collecting the information of all interfaces on the system
    # ---------------------- IMPL -----------------------------------
    # o collect interfaces using `netstat -iw`.
    # o iterate over them and pass to `ifconfig` to get the information.
    # --------------------------------------------------------------

    RES=0;
    echo "Collecting $CAP_NW_INTF_INFO.."; 
    echo "-----------------------------------------------" >>$OUT_NW_INTF_INFO; 
    echo " commands used: " >>$OUT_NW_INTF_INFO; 
    echo " 'netstat -i' to get the interfaces" >>$OUT_NW_INTF_INFO; 
    echo " 'ifconfig <intf>' to get intf information" >>$OUT_NW_INTF_INFO;
    echo "-----------------------------------------------" >>$OUT_NW_INTF_INFO; 

    for iii in `netstat -i | grep -v Name | cut -d' ' -f1`;
    do 
	if [ "$iii" ]; then  
		ifconfig $iii>>$OUT_NW_INTF_INFO; 
 	   	RES=`expr $? + $RES`
	fi 
    done 

    echo "-----------------------------------------------" >>$OUT_NW_INTF_INFO; 
}

get_arch ()
{
    echo 32
}

# Platform specific commands for capabilities.
define_commands ()
{
    if [ "$THIS_DISTRIBUTION" == "WindRiver" ] ; then
        S3CONTROL="solidcores3";
    else
        S3CONTROL="solidcoreS3";
    fi
	ANALYSIS_REQD_DISK_SPACE_ROOT_KB="2048";
        LOG_REQD_DISK_SPACE_KB="4096"  #4Mbytes
    if [ "$THIS_PLATFORM" = "Linux" ]; then
	if [ "$THIS_DISTRIBUTION" = "Ubuntu" ]; then
	    CMD_PACKAGE_QUERY="dpkg -l";
	  #  CMDS_PACKAGES_LIST="eval dpkg -l | sort";
	    CMDS_PACKAGES_LIST="eval dpkg -l";
	    S3CONTROL="solidcores3"
	    CMDS_SOLIDIFIER_PKG_INFO="dpkg -l $S3CONTROL";
	    CMDS_GLIBC_VERSION="dpkg-query -s libc6";
	    CMDS_ICONV_INFO="eval dpkg -l | grep -i iconv";
	    CMDS_PKGCHK=""
	else
	    CMD_PACKAGE_QUERY="rpm -q";
	    CMDS_PACKAGES_LIST="eval rpm -qa | sort";
	    CMDS_SOLIDIFIER_PKG_INFO="rpm -ql $S3CONTROL";
	    CMDS_GLIBC_VERSION="rpm -q glibc";
	    CMDS_ICONV_INFO="eval rpm -qa | grep -i iconv";
	    CMDS_PKGCHK="rpm --checksig "
	fi    

	CMD_LOADED_MODULES="lsmod";
	CMD_LOADED_DRIVERS="cat /proc/devices";
	ALL_FILESYSTEMS=`df -Tk / | grep -vw "1K" | tr '\n' ' ' | awk '{print $2}'`;
	PHYSICAL_MEMORY_KB=`eval free | grep Mem: | awk {'print $2'}`;
	CMD_COMPRESS="gzip";
	CMD_ZIP="zcat";
	CMD_TAR="tar";
	CMDS_SYSTEM_CONFIG="uname -a;cat $RELEASE_PATH";
	CMDS_CPU_INFO="cat /proc/cpuinfo";
	CMDS_UPTIME="uptime";
	CMDS_SYSTEM_STATE="top -b -n 1";
	CMDS_PROCESS_INFO="ps -aux";
	CMDS_SERVICES_INFO="service --status-all";
	CMDS_LOADED_MODULES="$CMD_LOADED_MODULES";
	CMDS_DISK_SPACE_INFO="df -kT";
	CMDS_ENV_VARS="eval env | sort";
	CMDS_ETC_DIR_LISTING="eval find /etc/rc* -name '*' -type f -exec ls -li
{} \;";
	CMDS_PORTS_INFO="netstat -a";
	CMDS_MOUNT_TABLE="cat /proc/mounts;cat /etc/auto.master";
	CMDS_FS_TABLE="cat /etc/fstab";
	CMDS_KERNEL_SYMBOLS="cat /proc/kallsyms"
	CMDS_PHYSICAL_RAM="free";
        if [ "$THIS_ARCH" = "ppc" ]; then
            CMDS_ADDR_SPACE_INFO="get_arch"
        else
            CMDS_ADDR_SPACE_INFO="getconf LONG_BIT";
        fi
	CMDS_NW_INTF_INFO="ifconfig";
	CMDS_HOSTNAME_INFO="hostname";
	CMDS_ROUTE_TABLE="netstat -nr";
	CMDS_LS_OPEN_FILES="lsof";
	CMDS_NFS_EXPORTS="cat /etc/exports";
	CMDS_OS_PATCH_INFO="";
	CMDS_RESOURCES_INFO="cat /proc/ioports;cat /proc/interrupts; cat /proc/dma; lsdev";
	CMDS_LOCALE_INFO="locale;locale -a"
	if [ "$THIS_DISTRIBUTION" = "CentOS" ] ||
	   [ "$THIS_DISTRIBUTION" = "OracleEL" ] ||
	   [ "$THIS_DISTRIBUTION" = "Ubuntu" ] ||
	   [ "$THIS_DISTRIBUTION" = "SUSE" ]; then
	    CMDS_CRASH_DUMP="";
	else
	    CMDS_CRASH_DUMP="crash -s";
	fi

	ps -p 1 | grep -i "systemd" >/dev/null
	if [ $? -eq 0 ]; then
	    CMDS_SERVICES_INFO="systemctl list-units --type service --all;systemctl list-unit-files --type service";
	    CMDS_UPTIME="uptime;systemd-analyze;systemd-analyze blame";
	    CMDS_SYSD_INFO="systemctl get-default;systemctl show -p Wants multi-user.target;systemctl list-jobs;systemctl list-units --all;systemctl --failed";
	fi

	# Analysis related definitions 

	ANALYSIS_SUPPORTED_DISTRIBUTIONS="";
	ANALYSIS_REQD_DISK_SPACE_KB="72000"; # 70Mbytes
	ANALYSIS_SUPPORTED_FILESYSTEMS="ext2 ext3 ext4 reiserfs nfs nfs4";
	ANALYSIS_REQD_RAM="10000"; # 10M
	ANALYSIS_PREREQUISITE_PKGS="";
	ANALYSIS_PROBLEMATIC_PKGS="";
	ANALYSIS_PROBLEMARIC_MODULES="solid_server replace_elf_handler";

    elif [ "$THIS_PLATFORM" = "SunOS" ]; then
# On solaris 10, when using SMF interface,
# If any of the command crashes, the SMF daemon will invoke the stop routine,
# causing the command interruption. To avoid this we will launch all the
# commands with a different contract.

    	if [ `uname -r` = "5.10" ]; then
		CONTRACT="ctrun -l child -o pgrponly "
	else
		CONTRACT=""
	fi

	CMD_PACKAGE_QUERY="$CONTRACT pkginfo -l";
	CMD_LOADED_MODULES="$CONTRACT modinfo";
	CMD_LOADED_DRIVERS="$CONTRACT modinfo";
	CMD_GET_ETC_SYSTEM="$CONTRACT cat /etc/system";
	ALL_FILESYSTEMS=`eval cat /etc/vfstab|grep -v \# | awk {'print $4'}`;
	PHYSICAL_MEMORY_MB=`eval prtconf|grep Memory|awk {'print $3'}`;
	if [ "$PHYSICAL_MEMORY_MB" ]; then 
		PHYSICAL_MEMORY_KB=`echo "$PHYSICAL_MEMORY_MB * 1024" | bc`;
	fi 
	CMD_COMPRESS="gzip";
	CMD_ZIP="zcat";
	CMD_TAR="tar";
	
	CMDS_SYSTEM_CONFIG="uname -a";
	CMDS_CPU_INFO="$CONTRACT psrinfo -v";
	CMDS_UPTIME="$CONTRACT uptime";
	CMDS_SYSTEM_STATE="$CONTRACT prstat -a 0 1";
	CMDS_PACKAGES_LIST="eval $CONTRACT pkginfo | sort";
	CMDS_SOLIDIFIER_PKG_INFO="$CONTRACT pkginfo -l $S3CONTROL";
	CMDS_PROCESS_INFO="$CONTRACT ps -aef";
	CMDS_SERVICES_INFO=""; 
	CMDS_LOADED_MODULES="$CONTRACT $CMD_LOADED_MODULES";
	CMDS_DISK_SPACE_INFO="$CONTRACT df -k";
	CMDS_ENV_VARS="eval $CONTRACT env | sort";
	CMDS_ETC_DIR_LISTING="eval find /etc/rc* -name '*' -type f -exec \
				ls -li {} \;";
	CMDS_PORTS_INFO="$CONTRACT netstat -a";
	CMDS_MOUNT_TABLE="$CONTRACT mount -p;cat /etc/auto_master";
	CMDS_FS_TABLE="$CONTRACT cat /etc/vfstab";
	CMDS_KERNEL_SYMBOLS="$CONTRACT /usr/ccs/bin/nm /dev/ksyms; $CONTRACT nm /dev/ksyms";
	CMDS_PHYSICAL_RAM="eval $CONTRACT prtconf | grep Memory";
	CMDS_ADDR_SPACE_INFO="$CONTRACT isainfo -kv";
	CMDS_NW_INTF_INFO="$CONTRACT ifconfig -a";
	CMDS_HOSTNAME_INFO="$CONTRACT hostname";
	CMDS_ROUTE_TABLE="$CONTRACT netstat -nr";
	CMDS_GLIBC_VERSION="";
	CMDS_LS_OPEN_FILES="";
	CMDS_NFS_EXPORTS="$CONTRACT cat /etc/dfs/dfstab";
	CMDS_OS_PATCH_INFO="$CONTRACT showrev -p";
	CMDS_RESOURCES_INFO="$CONTRACT prtconf";
	CMDS_ICONV_INFO="eval $CONTRACT pkginfo | grep -i iconv";
#	CMDS_PKGCHK="$CONTRACT pkgchk "
	CMDS_LOCALE_INFO="$CONTRACT locale;$CONTRACT locale -a"
	CMDS_CRASH_DUMP="$CONTRACT dumpadm";
	CMDS_SOL10_SMF="$CONTRACT svcs -a";
	CMDS_PKG_VER_INFO="$CONTRACT pkginfo -x "

	# Analysis related definitions 
	ANALYSIS_SUPPORTED_DISTRIBUTIONS="5.8 5.9 5.10";
	ANALYSIS_REQD_DISK_SPACE_KB="72000";
	ANALYSIS_SUPPORTED_FILESYSTEMS="ufs tmpfs vxfs nfs";
	ANALYSIS_REQD_RAM="491520"; # 480M
	if [ "$CMD_GET_DISTRIBUTION" = "5.10" ]; then
		ANALYSIS_PREREQUISITE_PKGS="SUNWuiu8";
	else
		ANALYSIS_PREREQUISITE_PKGS="SUNWuiu8 SUNWuiu8x";
    	fi
	ANALYSIS_PROBLEMATIC_PKGS="";
	ANALYSIS_PROBLEMARIC_MODULES="scdrv";

    elif [ "$THIS_PLATFORM" = "HP-UX" ]; then
	CMD_PACKAGE_QUERY="swlist";
	if [ "`uname -r`" = "B.11.23" -o "`uname -r`" = "B.11.31" ]; then
		CMD_LOADED_MODULES="kcmodule -v";
	else
		CMD_LOADED_MODULES="kmadmin -S";
	fi
	CMD_LOADED_DRIVERS="lsdev";
	ALL_FILESYSTEMS=`eval cat /etc/fstab|grep -v \# | awk {'print $3'}`;
	PHYSICAL_MEMORY_KB=`eval grep Physical: /var/adm/syslog/syslog.log | awk -F: {'print $5'} | awk {'print $1'}`;
	CMD_COMPRESS="gzip";
	CMD_ZIP="zcat";
	CMD_TAR="tar";
	CMD_NM_VERSION="nm -V";
	
	CMDS_SYSTEM_CONFIG="uname -a";
	CMDS_CPU_INFO="/opt/ignite/bin/print_manifest";
	if [ "`uname -r`" = "B.11.23" -o "`uname -r`" = "B.11.31" ]; then
		CMDS_CPU_INFO="$CMDS_CPU_INFO;machinfo";
	fi
	CMDS_UPTIME="uptime";
	CMDS_SYSTEM_STATE="top -d 1";
	CMDS_PACKAGES_LIST="eval swlist | sort";
	CMDS_SOLIDIFIER_PKG_INFO="swlist $S3CONTROL";
	CMDS_PROCESS_INFO="ps -aef";
	CMDS_SERVICES_INFO="";
	CMDS_LOADED_MODULES="kmadmin -S";
	CMDS_LOADED_MODULES="$CMD_LOADED_MODULES";
	CMDS_DISK_SPACE_INFO="bdf";
	CMDS_ENV_VARS="eval env | sort";
	CMDS_ETC_DIR_LISTING="eval find /etc/rc* -name '*' -type f -exec \
				ls -li {} \;";
	CMDS_PORTS_INFO="netstat -a";
	CMDS_MOUNT_TABLE="mount -p;cat /etc/auto_master";
	CMDS_FS_TABLE="cat /etc/fstab";
	CMDS_KERNEL_SYMBOLS="nm -g /stand/vmunix";
	CMDS_PHYSICAL_RAM="eval cat /var/adm/syslog/syslog.log|grep Physical:";
	CMDS_ADDR_SPACE_INFO="getconf KERNEL_BITS";

	# different way to get nw intf for hpux. so leaving it blank.
	CMDS_NW_INTF_INFO="";

	CMDS_HOSTNAME_INFO="hostname";
	CMDS_ROUTE_TABLE="netstat -nr";
	CMDS_GLIBC_VERSION="";
	CMDS_LS_OPEN_FILES="lsof";
	CMDS_NFS_EXPORTS="cat /etc/exports";
	CMDS_OS_PATCH_INFO="swlist";
	CMDS_RESOURCES_INFO="/opt/ignite/bin/print_manifest";
	CMDS_ICONV_INFO="eval swlist | grep -i iconv";
	CMDS_PKGCHK="swlist "
	CMDS_LOCALE_INFO="locale;locale -a"
	CMDS_RC_LOG="cat /etc/rc.log";
	CMDS_OLD_RC_LOG="cat /etc/rc.log.old";
	CMDS_MOD_REGISTER="cat /stand/dlkm/mod_register";
	CMDS_STAND_LISTING="ls -ilR /stand";
	CMDS_LOAD_MODS="cat /etc/loadmods";
	CMDS_DEV_INFO="lsdev";
	CMDS_SBIN_RC_LISTING="ls -ilR /sbin/rc*";
	CMDS_SBIN_INITD_LISTING="ls -ilR /sbin/init.d";
	CMDS_SWAGENT_LOG="cat /var/adm/sw/swagent.log";
	CMDS_CRASH_DUMP="crashconf";

	# Analysis related definitions. 
	ANALYSIS_SUPPORTED_DISTRIBUTIONS="B.11.00 B.11.11 B.11.23 B.11.31";
	ANALYSIS_REQD_DISK_SPACE_KB="72000";
	ANALYSIS_SUPPORTED_FILESYSTEMS="hfs vxfs";
	ANALYSIS_REQD_RAM="491520"; # 480M
	ANALYSIS_PREREQUISITE_PKGS="";
	ANALYSIS_PROBLEMATIC_PKGS="S3_Change_Ctrl";
	ANALYSIS_PROBLEMARIC_MODULES="scdrv";
	
    elif [ "$THIS_PLATFORM" = "AIX" ]; then
	CMD_PACKAGE_QUERY="lslpp -l";
	CMD_LOADED_MODULES="genkex";
	CMD_LOADED_DRIVERS="lsdev";
	CMD_GET_SYSLOG_CONF="cat /etc/syslog.conf";
	ALL_FILESYSTEMS=`eval cat /etc/filesystems | grep -v \* | grep  vfs |\
				awk {'print $3'}`;
	PHYSICAL_MEMORY_KB=`bootinfo -r`;
	CMD_COMPRESS="gzip";
	CMD_ZIP="zcat";
	CMD_TAR="tar";

	CMDS_SYSTEM_CONFIG="uname -a;/usr/bin/oslevel -r;/usr/bin/oslevel -qs";
	CMDS_CPU_INFO="lsconf";
	CMDS_UPTIME="uptime";
	CMDS_SYSTEM_STATE="ps cvg";
	CMDS_PACKAGES_LIST="eval lslpp -L all | sort";
	CMDS_SOLIDIFIER_PKG_INFO="lslpp -l $S3CONTROL";
	CMDS_PROCESS_INFO="ps -aef";
	CMDS_SERVICES_INFO="lssrc -a";
	CMDS_LOADED_MODULES="$CMD_LOADED_MODULES";
	CMDS_DISK_SPACE_INFO="df -k";
	CMDS_ENV_VARS="eval env | sort";
	CMDS_ETC_DIR_LISTING="eval find /etc/rc* -name '*' -type f -exec \
				ls -li {} \;";
	CMDS_PORTS_INFO="netstat -a";
	CMDS_MOUNT_TABLE="mount;cat /etc/auto_master";
	CMDS_FS_TABLE="cat /etc/filesystems";
	CMDS_KERNEL_SYMBOLS="nm -X32_64 -g /unix;dump -Tv -X32_64 /unix";
	CMDS_PHYSICAL_RAM="bootinfo -r";
	CMDS_ADDR_SPACE_INFO="bootinfo -K";
	CMDS_NW_INTF_INFO="ifconfig -a";
	CMDS_HOSTNAME_INFO="hostname";
	CMDS_ROUTE_TABLE="netstat -nr";
	CMDS_GLIBC_VERSION="";
	CMDS_LS_OPEN_FILES="lsof";
	CMDS_NFS_EXPORTS="cat /etc/exports";
	CMDS_OS_PATCH_INFO="instfix -i";
	CMDS_RESOURCES_INFO="lscfg -vp";
	CMDS_ICONV_INFO="eval lslpp -l | grep -i iconv";
	CMDS_PKGCHK="lslpp -L "
	CMDS_LOCALE_INFO="locale;locale -a"
	CMDS_CRASH_DUMP="sysdumpdev -l";
	CMDS_INITTAB_ENTRY="lsitab -a"
	
	# Analysis related definitions 
	# Follwing distributions are supported: 5.2, 5.3, 6.1
	ANALYSIS_SUPPORTED_DISTRIBUTIONS="5.2 5.3 6.1";
	ANALYSIS_REQD_DISK_SPACE_KB="72000";
	ANALYSIS_SUPPORTED_FILESYSTEMS="jfs jfs2 jfs2/aix nfs nfs3";
	ANALYSIS_REQD_RAM="491520"; # 480M
	ANALYSIS_PREREQUISITE_PKGS="";
	ANALYSIS_PROBLEMATIC_PKGS="";
	ANALYSIS_PROBLEMARIC_MODULES="scdrv";

else
    echo "ERROR: Unsupported/Unknown platform [ "$THIS_PLATFORM" ]."
    exit $SC_ERR_UNSUPPORTED_PLATFORM;
fi 
}

_common_capabilities () 
{
	
    collect "$CAP_SYSTEM_CONFIG" "$OUT_SYSTEM_CONFIG" "$CMDS_SYSTEM_CONFIG";
    collect "$CAP_CPU_INFO" "$OUT_CPU_INFO" "$CMDS_CPU_INFO";
    collect "$CAP_UPTIME" "$OUT_UPTIME" "$CMDS_UPTIME";
    collect "$CAP_SYSTEM_STATE" "$OUT_SYSTEM_STATE" "$CMDS_SYSTEM_STATE";
    collect "$CAP_PACKAGES_LIST" "$OUT_PACKAGES_LIST" "$CMDS_PACKAGES_LIST";
    collect "$CAP_SOLIDIFIER_PKG_INFO" "$OUT_SOLIDIFIER_PKG_INFO" \
						"$CMDS_SOLIDIFIER_PKG_INFO";
    collect "$CAP_PROCESS_INFO" "$OUT_PROCESS_INFO" "$CMDS_PROCESS_INFO";
    collect "$CAP_SERVICES_INFO" "$OUT_SERVICES_INFO" "$CMDS_SERVICES_INFO";
    collect "$CAP_LOADED_MODULES" "$OUT_LOADED_MODULES" "$CMDS_LOADED_MODULES";
    collect "$CAP_DISK_SPACE_INFO" "$OUT_DISK_SPACE_INFO" \
    						"$CMDS_DISK_SPACE_INFO";
    collect "$CAP_ENV_VARS" "$OUT_ENV_VARS" "$CMDS_ENV_VARS";
    collect "$CAP_ETC_DIR_LISTING" "$OUT_ETC_DIR_LISTING" \
    						"$CMDS_ETC_DIR_LISTING";
    collect "$CAP_PORTS_INFO" "$OUT_PORTS_INFO" "$CMDS_PORTS_INFO";
    collect "$CAP_MOUNT_TABLE" "$OUT_MOUNT_TABLE" "$CMDS_MOUNT_TABLE";
    collect "$CAP_FS_TABLE" "$OUT_FS_TABLE" "$CMDS_FS_TABLE";
    collect "$CAP_KERNEL_SYMBOLS" "$OUT_KERNEL_SYMBOLS" "$CMDS_KERNEL_SYMBOLS";
    collect "$CAP_PHYSICAL_RAM" "$OUT_PHYSICAL_RAM" "$CMDS_PHYSICAL_RAM";
    collect "$CAP_ADDR_SPACE_INFO" "$OUT_ADDR_SPACE_INFO" \
    						"$CMDS_ADDR_SPACE_INFO";
    collect "$CAP_NW_INTF_INFO" "$OUT_NW_INTF_INFO" "$CMDS_NW_INTF_INFO";
    collect "$CAP_DEV_INFO" "$OUT_DEV_INFO" "$CMD_LOADED_DRIVERS";
    collect "$CAP_PKGCHK_INFO" "$OUT_PKGCHK_INFO" "$CMDS_PKGCHK";
    collect "$CAP_LOCALE_INFO" "$OUT_LOCALE_INFO" "$CMDS_LOCALE_INFO";
    collect "$CAP_PKG_VER_INFO" "$OUT_PKG_VER_INFO" "$CMDS_PKG_VER_INFO";
    
    if [ "$THIS_PLATFORM" = "HP-UX" ]; then
    	collect_nw_intf_info_hpux;
    	collect "$CAP_RC_LOG" "$OUT_RC_LOG" "$CMDS_RC_LOG";
 	collect "$CAP_OLD_RC_LOG" "$OUT_OLD_RC_LOG" "$CMDS_OLD_RC_LOG";
	collect "$CAP_MOD_REGISTER" "$OUT_MOD_REGISTER" "$CMDS_MOD_REGISTER";
	collect "$CAP_STAND_LISTING" "$OUT_STAND_LISTING" \
	    		"$CMDS_STAND_LISTING"; 
	collect "$CAP_LOAD_MODS" "$OUT_LOAD_MODS" "$CMDS_LOAD_MODS";
	collect "$CAP_SBIN_RC_LISTING" "$OUT_SBIN_RC_LISTING" \
	    		"$CMDS_SBIN_RC_LISTING";
	collect "$CAP_SBIN_INITD_LISTING" "$OUT_SBIN_INITD_LISTING" \
	    		"$CMDS_SBIN_INITD_LISTING";
	collect "$CAP_SWAGENT_LOG" "$OUT_SWAGENT_LOG" \
	    		"$CMDS_SWAGENT_LOG";
    fi 

    collect "$CAP_HOSTNAME_INFO" "$OUT_HOSTNAME_INFO" "$CMDS_HOSTNAME_INFO";
    collect "$CAP_ROUTE_TABLE" "$OUT_ROUTE_TABLE" "$CMDS_ROUTE_TABLE";
    collect "$CAP_GLIBC_VERSION" "$OUT_GLIBC_VERSION" "$CMDS_GLIBC_VERSION";
    collect "$CAP_LS_OPEN_FILES" "$OUT_LS_OPEN_FILES" "$CMDS_LS_OPEN_FILES";
    collect "$CAP_NFS_EXPORTS" "$OUT_NFS_EXPORTS" "$CMDS_NFS_EXPORTS";
    collect "$CAP_OS_PATCH_INFO" "$OUT_OS_PATCH_INFO" "$CMDS_OS_PATCH_INFO";
    collect "$CAP_RESOURCES_INFO" "$OUT_RESOURCES_INFO" "$CMDS_RESOURCES_INFO";
    collect "$CAP_ICONV_INFO" "$OUT_ICONV_INFO" "$CMDS_ICONV_INFO";
    collect "$CAP_INITTAB_ENTRY" "$OUT_INITTAB_ENTRY" "$CMDS_INITTAB_ENTRY";
    if [ "$THIS_DISTRIBUTION" != "WindRiver" ] ; then
        collect "$CAP_CRASH_DUMP_INFO" "$OUT_CRASH_DUMP" "$CMDS_CRASH_DUMP"
    fi
    if [ `uname -r` = "5.10" ]; then
         collect "$CAP_SOL10_SMF" "$OUT_SOL10_SMF" "$CMDS_SOL10_SMF";
    fi
    if [ "$THIS_PLATFORM" = "Linux" ]; then
	ps -p 1 | grep -i "systemd" >/dev/null
	if [ $? -eq 0 ]; then
	    collect "$CAP_SYSD_INFO" "$OUT_SYSD_INFO" "$CMDS_SYSD_INFO";
	    return 0;
	fi
    fi

    return 0;
}

# function to generate the report header 
sc_generate_report_header () 
{
    COPYRIGHT="Copyright 2008-2015 McAfee, Inc. All Rights Reserved.";

    sc_version >> $OUT_REPORT;
    echo $COPYRIGHT >> $OUT_REPORT;
    echo "----------------------------------------------------------"\
    >>$OUT_REPORT;
}

# function to generate a report for scanalyzer data collected 
sc_generate_report () 
{
    date >> $OUT_SYSTEM_DATE;
    sc_version >> $OUT_TOOL_VERSION;

    if [ "$QOPT" = 0 ]; then
	echo	
	echo "Preparing Summary Report. " 
    fi

    # Analysis time 
    _append_to_report "Analysis Time" "$OUT_SYSTEM_DATE";
    _append_to_report "$CAP_HOSTNAME_INFO" "$OUT_HOSTNAME_INFO";
    _append_to_report "$CAP_SYSTEM_CONFIG" "$OUT_SYSTEM_CONFIG";
    _append_to_report "$CAP_ADDR_SPACE_INFO" "$OUT_ADDR_SPACE_INFO";
    _append_to_report "$CAP_UPTIME" "$OUT_UPTIME";
    _append_to_report "$CAP_CPU_INFO" "$OUT_CPU_INFO";
    _append_to_report "$CAP_DISK_SPACE_INFO" "$OUT_DISK_SPACE_INFO";
    _append_to_report "$CAP_PHYSICAL_RAM" "$OUT_PHYSICAL_RAM";
    _append_to_report "$CAP_PACKAGES_LIST" "$OUT_PACKAGES_LIST"; 
    _append_to_report "$CAP_SERVICES_INFO" "$OUT_SERVICES_INFO";
    _append_to_report "$CAP_PROCESS_INFO" "$OUT_PROCESS_INFO";
    _append_to_report "$CAP_LOADED_MODULES" "$OUT_LOADED_MODULES";
    _append_to_report "$CAP_PORTS_INFO" "$OUT_PORTS_INFO";
    _append_to_report "$CAP_ROUTE_TABLE" "$OUT_ROUTE_TABLE";
    _append_to_report "$CAP_ENV_VARS" "$OUT_ENV_VARS";
    _append_to_report "$CAP_SOLIDIFIER_PKG_INFO" "$OUT_SOLIDIFIER_PKG_INFO";
    _append_to_report "$CAP_SYSTEM_STATE" "$OUT_SYSTEM_STATE";
    _append_to_report "$CAP_ETC_DIR_LISTING" "$OUT_ETC_DIR_LISTING";
    _append_to_report "$CAP_FS_TABLE" "$OUT_FS_TABLE";
    _append_to_report "$CAP_MOUNT_TABLE" "$OUT_MOUNT_TABLE";
    _append_to_report "$CAP_KERNEL_SYMBOLS" "$OUT_KERNEL_SYMBOLS";
    _append_to_report "$CAP_NW_INTF_INFO" "$OUT_NW_INTF_INFO";
    _append_to_report "$CAP_GLIBC_VERSION" "$OUT_GLIBC_VERSION";
    _append_to_report "$CAP_LS_OPEN_FILES" "$OUT_LS_OPEN_FILES";
    _append_to_report "$CAP_NFS_EXPORTS" "$OUT_NFS_EXPORTS";
    _append_to_report "$CAP_OS_PATCH_INFO" "$OUT_OS_PATCH_INFO";
    _append_to_report "$CAP_RESOURCES_INFO" "$OUT_RESOURCES_INFO";
    _append_to_report "$CAP_ICONV_INFO" "$OUT_ICONV_INFO";
    _append_to_report "$CAP_CRASH_DUMP_INFO" "$OUT_CRASH_DUMP"
    _append_to_report "$CAP_DEV_INFO" "$OUT_DEV_INFO"
    _append_to_report "$CAP_PKGCHK_INFO" "$OUT_PKGCHK_INFO"
    _append_to_report "$CAP_LOCALE_INFO" "$OUT_LOCALE_INFO"
    _append_to_report "$CAP_PKG_VER_INFO" "$OUT_PKG_VER_INFO"
     
    if [ "$THIS_PLATFORM" = "HP-UX" ]; then
    	_append_to_report "$CAP_RC_LOG" "$OUT_RC_LOG"
 	_append_to_report "$CAP_OLD_RC_LOG" "$OUT_OLD_RC_LOG"
	_append_to_report "$CAP_MOD_REGISTER" "$OUT_MOD_REGISTER"
	_append_to_report "$CAP_STAND_LISTING" "$OUT_STAND_LISTING"
	_append_to_report "$CAP_LOAD_MODS" "$OUT_LOAD_MODS"
	_append_to_report "$CAP_SBIN_RC_LISTING" "$OUT_SBIN_RC_LISTING"
	_append_to_report "$CAP_SBIN_INITD_LISTING" "$OUT_SBIN_INITD_LISTING"
	_append_to_report "$CAP_SWAGENT_LOG" "$OUT_SWAGENT_LOG"
    fi 

    if [ "$THIS_PLATFORM" = "AIX" ]; then
	_append_to_report "$CAP_SYSLOG_CONF" "$OUT_SYSLOG_CONF"
	_append_to_report "$CAP_INITTAB_ENTRY" "$OUT_INITTAB_ENTRY";
    fi

   
    if [ `uname -r` = "5.10" ]; then
         _append_to_report "$CAP_SOL10_SMF" "$OUT_SOL10_SMF";
     fi

    if [ "$THIS_PLATFORM" = "Linux" ]; then
	ps -p 1 | grep -i "systemd" >/dev/null
	if [ $? -eq 0 ]; then
	    _append_to_report "$CAP_SYSD_INFO" "$OUT_SYSD_INFO"
	    return 0;
	fi
    fi

    # @@ Finalise @@
    echo "" >> $OUT_REPORT
    echo "-------------------- End of Report -------------------" >>$OUT_REPORT;
    echo
}

# function to perform misc. operation on the generated report file
# if command is from SC, we need to compress the out put file
# and if -o flag is specified, move the report to the specified location.
sc_postgen_fops()
{
    if [ $SC_NEEDS_COMPRESSION = 1 ]; then
    	sc_archive_results
    else
	if [ "$SC_IS_ARGUMENT_FILE_NAME" = 1 ]; then
	    SC_OP_DIR=`dirname $USE_NAME`;
	    mkdir -p $SC_OP_DIR > /dev/null 2>&1;
	    mv $OUT_REPORT $USE_NAME > /dev/null 2>&1;
	    if [ ! $? = 0 ]; then
		echo "Unable to create the file $USE_NAME"
		echo "Placing the report at default location."
		mypwd=`pwd`
		if [ "$mypwd" != "/" ]; then
	        	OUT_REPORT=$mypwd/$OUT_REPORT;
	        else
	       		OUT_REPORT=/$OUT_REPORT;
		fi
	    else
		OUT_REPORT=$USE_NAME;
	    fi
	else
	    mypwd=`pwd`
	    if [ "$mypwd" != "/" ]; then
	    	OUT_REPORT=$mypwd/$OUT_REPORT;
	    else
	    	OUT_REPORT=/$OUT_REPORT;
	    fi
	fi
    fi
}

# function to archive results 
sc_archive_results()
{
    echo "Archiving the collected data.. $NONL"
    ARCH_FILE=scan_info.tar
    tar -cvf $ARCH_FILE $OUT_REPORT > /dev/null 2>&1
    if [ $? -ne 0 ]; then
	echo "tar command has failed to create the archive."
	exit -1;
    fi

    $CMD_COMPRESS -f $ARCH_FILE > /dev/null 2>&1
    mv $ARCH_FILE.gz /tmp
    echo "Report generated at : /tmp/$ARCH_FILE.gz"
    SC_REPORT_COMPRESSED=1
}



check_for_supported_kernels ()
{
# For some old linux'es, -i option is not supported with uname. We default
# to X86 in such a case until we actually start supporting these distros on
# AMD64.
sys_arch=`uname -m 2>/dev/null`    
case $sys_arch in
i386)  host_arch="X86";;
i486)  host_arch="X86";;
i586)  host_arch="X86";;
i686)  host_arch="X86";;
amd64) host_arch="AMD64";;
x86_64)host_arch="AMD64";;
ppc)   host_arch="PPC32";;
esac

if [ -r /etc/redhat-release ]
    then
        REL_FILE="/etc/redhat-release"
    elif [ -r /etc/SuSE-release ]
    then
        REL_FILE="/etc/SuSE-release"
    elif [ -r /etc/lsb-release ]
    then
	REL_FILE="/etc/lsb-release"
    elif [ -f "/VERSION.os" ]; then
        REL_FILE="/VERSION.os"
    elif [ -f "/etc/issue" ]; then
        REL_FILE="/etc/issue"
    else
        echo "Error: unable to identify linux release"
    fi

    if [ $REL_FILE = "/etc/lsb-release" ]; then
	LINUX_REL=`cat $REL_FILE | grep -w DISTRIB_DESCRIPTION | head -n 1 | awk -F"=" '{print $2}' | awk -F"\"" '{print $2}' | cut -d '.' -f 1-2`
    elif [ $REL_FILE = "/VERSION.os" ]; then
        LINUX_REL=`cat $REL_FILE | head -n 1 |  awk -F "-" '{ print $1 }' | awk -F "_" '{ print $2 }'`
    elif [ -f "/etc/issue" ]; then
       release=`cat $REL_FILE`
       LINUX_REL=`echo $release | awk -F " " '{print $1" "$2" "$3}'`
    else
       LINUX_REL=`cat $REL_FILE | head -n 1 | awk -F " [(]" '{ print $1 }' | awk -F "." '{ print $1 }'`
    fi
    LINUX_REL=`eval echo "$LINUX_REL"`

    case $LINUX_REL in
    $AS3)host_distro="LEL3";;           	# Redhat AS 3
    $AS4)host_distro="LEL4";;           	# Redhat AS 4
    $ES3)host_distro="LEL3";;           	# Redhat ES 3
    $ES4)host_distro="LEL4";;           	# Redhat ES 4
    $WS3)host_distro="LEL3";;           	# Redhat WS 3
    $WS4)host_distro="LEL4";;           	# Redhat WS 4
    $EL5)host_distro="LEL5";;           	# Redhat EL 5
    $OEL5)host_distro="LEL5";;          	# Oracle EL 5
    $CS4)host_distro="LEL4";;           	# CentOS 4
    $CS5)host_distro="LEL5";;           	# CentOS 5
    $EL6)host_distro="LEL6";;           	# Redhat EL 6
    $OEL6)host_distro="LEL6";;			# Oracle EL 6
    $CS6)host_distro="LEL6";;			# CentOS 6
    $CS6_0)host_distro="LEL6";;			# CentOS 6
    $SES9)host_distro="LSES9";;         	# SLES 9
    $SES9x)host_distro="LSES9";;        	# Suse 9.3 Pro
    $SES10)host_distro="LSES10";;       	# SLES 10
    $SED11)host_distro="LSES11";;       	# SLED 11
    $SES11)host_distro="LSES11";;       	# SLES 11
    $LRH72)host_distro="LRH72";;       		# Redhat 7.2
    $Opensuse10)host_distro="LSES10";;          # OpenSuse 10
    $Opensuse11)host_distro="LSES11";;          # OpenSuse 11
    $LWR14)host_distro="LWR14";;                # WindRiver 1.4
    $LWR40)host_distro="LWR40";;                # WindRiver 4.0
    $LWR42)host_distro="LWR42";;                # WindRiver 4.2
    $LWR43)host_distro="LWR43";;                # WindRiver 4.3
    $LWR60)host_distro="LWR60";;                # WindRiver 6.0
    $LWR)host_distro="LWR";;                    # WindRiver 5.0 and onwards
    $Ubuntu1204)host_distro="LUBT12";;		# Ubuntu 12
    *)echo "Error: Linux release ("$LINUX_REL") is not supported."
    esac

# RHEL 3 on X86
KVER_VALS_LEL3_X86="   "

# RHEL 4 / CentOS 4 on X86

KVER_VALS_LEL4_X86="   "

# RHEL 5 / CentOS 5 / OEL 5 on X86

KVER_VALS_LEL5_X86=" 2.6.18-8.el5 2.6.18-8.el5PAE 2.6.18-53.el5 2.6.18-53.el5PAE 2.6.18-92.el5 2.6.18-92.el5PAE 2.6.18-128.el5 2.6.18-128.el5PAE 2.6.18-164.el5 2.6.18-164.el5PAE 2.6.18-194.el5 2.6.18-194.el5PAE 2.6.18-238.el5 2.6.18-238.el5PAE 2.6.18-274.el5 2.6.18-274.el5PAE 2.6.18-308.el5 2.6.18-308.el5PAE 2.6.18-348.el5 2.6.18-348.el5PAE 2.6.18-371.el5 2.6.18-371.el5PAE 2.6.18-238.48.1.el5=2.6.18-238.el5 2.6.18-238.48.1.el5PAE=2.6.18-238.el5PAE 2.6.18-238.49.1.el5=2.6.18-238.el5 2.6.18-238.49.1.el5PAE=2.6.18-238.el5PAE 2.6.18-238.50.1.el5=2.6.18-238.el5 2.6.18-238.50.1.el5PAE=2.6.18-238.el5PAE 2.6.18-238.51.1.el5=2.6.18-238.el5 2.6.18-238.51.1.el5PAE=2.6.18-238.el5PAE 2.6.18-238.52.1.el5=2.6.18-238.el5 2.6.18-238.52.1.el5PAE=2.6.18-238.el5PAE 2.6.18-348.1.1.el5=2.6.18-348.el5 2.6.18-348.1.1.el5PAE=2.6.18-348.el5PAE 2.6.18-348.12.1.el5=2.6.18-348.el5 2.6.18-348.12.1.el5PAE=2.6.18-348.el5PAE 2.6.18-348.16.1.el5=2.6.18-348.el5 2.6.18-348.16.1.el5PAE=2.6.18-348.el5PAE 2.6.18-348.18.1.el5=2.6.18-348.el5 2.6.18-348.18.1.el5PAE=2.6.18-348.el5PAE 2.6.18-348.19.1.el5=2.6.18-348.el5 2.6.18-348.19.1.el5PAE=2.6.18-348.el5PAE 2.6.18-348.2.1.el5=2.6.18-348.el5 2.6.18-348.2.1.el5PAE=2.6.18-348.el5PAE 2.6.18-348.21.1.el5=2.6.18-348.el5 2.6.18-348.21.1.el5PAE=2.6.18-348.el5PAE 2.6.18-348.22.1.el5=2.6.18-348.el5 2.6.18-348.22.1.el5PAE=2.6.18-348.el5PAE 2.6.18-348.23.1.el5=2.6.18-348.el5 2.6.18-348.23.1.el5PAE=2.6.18-348.el5PAE 2.6.18-348.25.1.el5=2.6.18-348.el5 2.6.18-348.25.1.el5PAE=2.6.18-348.el5PAE 2.6.18-348.27.1.el5=2.6.18-348.el5 2.6.18-348.27.1.el5PAE=2.6.18-348.el5PAE 2.6.18-348.28.1.el5=2.6.18-348.el5 2.6.18-348.28.1.el5PAE=2.6.18-348.el5PAE 2.6.18-348.3.1.el5=2.6.18-348.el5 2.6.18-348.3.1.el5PAE=2.6.18-348.el5PAE 2.6.18-348.4.1.el5=2.6.18-348.el5 2.6.18-348.4.1.el5PAE=2.6.18-348.el5PAE 2.6.18-348.6.1.el5=2.6.18-348.el5 2.6.18-348.6.1.el5PAE=2.6.18-348.el5PAE 2.6.18-371.1.2.el5=2.6.18-371.el5 2.6.18-371.1.2.el5PAE=2.6.18-371.el5PAE 2.6.18-371.3.1.el5=2.6.18-371.el5 2.6.18-371.3.1.el5PAE=2.6.18-371.el5PAE 2.6.18-371.4.1.el5=2.6.18-371.el5 2.6.18-371.4.1.el5PAE=2.6.18-371.el5PAE 2.6.18-371.6.1.el5=2.6.18-371.el5 2.6.18-371.6.1.el5PAE=2.6.18-371.el5PAE 2.6.18-371.8.1.el5=2.6.18-371.el5 2.6.18-371.8.1.el5PAE=2.6.18-371.el5PAE 2.6.18-371.9.1.el5=2.6.18-371.el5 2.6.18-371.9.1.el5PAE=2.6.18-371.el5PAE 2.6.18-371.11.1.el5=2.6.18-371.el5 2.6.18-371.11.1.el5PAE=2.6.18-371.el5PAE  "

# RHEL 6  on X86

KVER_VALS_LEL6_X86=" 2.6.32-71.el6.i686 2.6.32-131.0.15.el6.i686 2.6.32-220.el6.i686 2.6.32-279.el6.i686 2.6.32-358.el6.i686 2.6.32-431.el6.i686 2.6.32-358.el6.i686 2.6.32-431.el6.i686 2.6.32-279.22.1.el6.i686 2.6.32-220.31.1.el6.i686 2.6.32-358.23.2.el6.i686 2.6.39-100.5.1.el6uek.i686 2.6.39-200.24.1.el6uek.i686 2.6.39-400.17.1.el6uek.i686 2.6.39-400.211.1.el6uek.i686 2.6.39-300.17.2.el6uek.i686 2.6.32-131.37.1.el6.i686=2.6.32-131.0.15.el6.i686 2.6.32-131.38.1.el6.i686=2.6.32-131.0.15.el6.i686 2.6.32-131.39.1.el6.i686=2.6.32-131.0.15.el6.i686 2.6.32-220.32.1.el6.i686=2.6.32-220.31.1.el6.i686 2.6.32-220.34.1.el6.i686=2.6.32-220.31.1.el6.i686 2.6.32-220.34.2.el6.i686=2.6.32-220.31.1.el6.i686 2.6.32-220.38.1.el6.i686=2.6.32-220.31.1.el6.i686 2.6.32-220.39.1.el6.i686=2.6.32-220.31.1.el6.i686 2.6.32-220.41.1.el6.i686=2.6.32-220.31.1.el6.i686 2.6.32-220.42.1.el6.i686=2.6.32-220.31.1.el6.i686 2.6.32-220.45.1.el6.i686=2.6.32-220.31.1.el6.i686 2.6.32-220.46.1.el6.i686=2.6.32-220.31.1.el6.i686 2.6.32-279.23.1.el6.i686=2.6.32-279.22.1.el6.i686 2.6.32-279.25.1.el6.i686=2.6.32-279.22.1.el6.i686 2.6.32-279.25.2.el6.i686=2.6.32-279.22.1.el6.i686 2.6.32-279.31.1.el6.i686=2.6.32-279.22.1.el6.i686 2.6.32-279.33.1.el6.i686=2.6.32-279.22.1.el6.i686 2.6.32-279.34.1.el6.i686=2.6.32-279.22.1.el6.i686 2.6.32-279.37.2.el6.i686=2.6.32-279.22.1.el6.i686 2.6.32-279.39.1.el6.i686=2.6.32-279.22.1.el6.i686 2.6.32-279.41.1.el6.i686=2.6.32-279.22.1.el6.i686 2.6.32-279.42.1.el6.i686=2.6.32-279.22.1.el6.i686 2.6.32-279.43.1.el6.i686=2.6.32-279.22.1.el6.i686 2.6.32-279.43.2.el6.i686=2.6.32-279.22.1.el6.i686 2.6.32-279.46.1.el6.i686=2.6.32-279.22.1.el6.i686 2.6.32-358.0.1.el6.i686=2.6.32-358.el6.i686 2.6.32-358.11.1.el6.i686=2.6.32-358.el6.i686 2.6.32-358.14.1.el6.i686=2.6.32-358.el6.i686 2.6.32-358.18.1.el6.i686=2.6.32-358.el6.i686 2.6.32-358.2.1.el6.i686=2.6.32-358.el6.i686 2.6.32-358.28.1.el6.i686=2.6.32-358.23.2.el6.i686 2.6.32-358.32.3.el6.i686=2.6.32-358.23.2.el6.i686 2.6.32-358.37.1.el6.i686=2.6.32-358.23.2.el6.i686 2.6.32-358.41.1.el6.i686=2.6.32-358.23.2.el6.i686 2.6.32-358.44.1.el6.i686=2.6.32-358.23.2.el6.i686 2.6.32-358.46.1.el6.i686=2.6.32-358.23.2.el6.i686 2.6.32-358.46.2.el6.i686=2.6.32-358.23.2.el6.i686 2.6.32-358.6.1.el6.i686=2.6.32-358.el6.i686 2.6.32-358.6.2.el6.i686=2.6.32-358.el6.i686 2.6.32-431.1.2.el6.i686=2.6.32-431.el6.i686 2.6.32-431.11.2.el6.i686=2.6.32-431.el6.i686 2.6.32-431.3.1.el6.i686=2.6.32-431.el6.i686 2.6.32-431.5.1.el6.i686=2.6.32-431.el6.i686 2.6.32-431.17.1.el6.i686=2.6.32-431.el6.i686 2.6.32-431.20.3.el6.i686=2.6.32-431.el6.i686 2.6.32-431.20.5.el6.i686=2.6.32-431.el6.i686 2.6.32-431.23.3.el6.i686=2.6.32-431.el6.i686 2.6.39-300.17.3.el6uek.i686=2.6.39-300.17.2.el6uek.i686 2.6.39-300.26.1.el6uek.i686=2.6.39-300.17.2.el6uek.i686 2.6.39-400.17.2.el6uek.i686=2.6.39-400.17.1.el6uek.i686 2.6.39-400.21.2.el6uek.i686=2.6.39-400.17.1.el6uek.i686 2.6.39-400.24.1.el6uek.i686=2.6.39-400.17.1.el6uek.i686 2.6.39-400.109.5.el6uek.i686=2.6.39-400.17.1.el6uek.i686 2.6.39-400.214.3.el6uek.i686=2.6.39-400.211.1.el6uek.i686 2.6.39-400.214.4.el6uek.i686=2.6.39-400.211.1.el6uek.i686  "

# SuSE 9 on X86

KVER_VALS_LSES9_X86="   "

# SuSE 10 on X86

KVER_VALS_LSES10_X86=" 2.6.16.21-0.8-default 2.6.16.21-0.8-smp 2.6.16.21-0.8-bigsmp 2.6.16.46-0.12-default 2.6.16.46-0.12-smp 2.6.16.46-0.12-bigsmp 2.6.16.60-0.21-default 2.6.16.60-0.21-bigsmp 2.6.16.60-0.21-smp 2.6.16.60-0.54.5-bigsmp 2.6.16.60-0.54.5-default 2.6.16.60-0.54.5-smp 2.6.16.60-0.85.1-bigsmp 2.6.16.60-0.85.1-default 2.6.16.60-0.85.1-smp 2.6.16.60-0.101.1-bigsmp=2.6.16.60-0.85.1-bigsmp 2.6.16.60-0.101.1-default=2.6.16.60-0.85.1-default 2.6.16.60-0.101.1-smp=2.6.16.60-0.85.1-smp 2.6.16.60-0.103.1-bigsmp=2.6.16.60-0.85.1-bigsmp 2.6.16.60-0.103.1-default=2.6.16.60-0.85.1-default 2.6.16.60-0.103.1-smp=2.6.16.60-0.85.1-smp 2.6.16.60-0.105.1-bigsmp=2.6.16.60-0.85.1-bigsmp 2.6.16.60-0.105.1-default=2.6.16.60-0.85.1-default 2.6.16.60-0.105.1-smp=2.6.16.60-0.85.1-smp 2.6.16.60-0.107.1-bigsmp=2.6.16.60-0.85.1-bigsmp 2.6.16.60-0.107.1-default=2.6.16.60-0.85.1-default 2.6.16.60-0.107.1-smp=2.6.16.60-0.85.1-smp 2.6.16.60-0.113.1-bigsmp=2.6.16.60-0.54.5-bigsmp 2.6.16.60-0.113.1-default=2.6.16.60-0.54.5-default 2.6.16.60-0.113.1-smp=2.6.16.60-0.54.5-smp 2.6.16.60-0.123.1-bigsmp=2.6.16.60-0.54.5-bigsmp 2.6.16.60-0.123.1-default=2.6.16.60-0.54.5-default 2.6.16.60-0.123.1-smp=2.6.16.60-0.54.5-smp  "

# SuSE 11 on X86

KVER_VALS_LSES11_X86=" 2.6.27.19-5-pae 2.6.27.19-5-default 2.6.32.12-0.7-pae 2.6.32.12-0.7-default 3.0.13-0.27-default 3.0.13-0.27-pae 3.0.76-0.11-default 3.0.76-0.11-pae 3.0.74-0.6.6-default 3.0.74-0.6.6-pae 2.6.32.59-0.9-default=2.6.32.12-0.7-default 2.6.32.59-0.9-pae=2.6.32.12-0.7-pae 2.6.32.59-0.13-default=2.6.32.12-0.7-default 2.6.32.59-0.13-pae=2.6.32.12-0.7-pae 3.0.101-0.15-default=3.0.76-0.11-default 3.0.101-0.15-pae=3.0.76-0.11-pae 3.0.101-0.21-default=3.0.76-0.11-default 3.0.101-0.21-pae=3.0.76-0.11-pae 3.0.101-0.29-default=3.0.76-0.11-default 3.0.101-0.29-pae=3.0.76-0.11-pae 3.0.101-0.31-default=3.0.76-0.11-default 3.0.101-0.31-pae=3.0.76-0.11-pae 3.0.101-0.35-default=3.0.76-0.11-default 3.0.101-0.35-pae=3.0.76-0.11-pae 3.0.101-0.5-default=3.0.74-0.6.6-default 3.0.101-0.5-pae=3.0.74-0.6.6-pae 3.0.101-0.7.15-default=3.0.74-0.6.6-default 3.0.101-0.7.15-pae=3.0.74-0.6.6-pae 3.0.101-0.7.17-default=3.0.74-0.6.6-default 3.0.101-0.7.17-pae=3.0.74-0.6.6-pae 3.0.101-0.7.19-default=3.0.74-0.6.6-default 3.0.101-0.7.19-pae=3.0.74-0.6.6-pae 3.0.101-0.7.21-default=3.0.74-0.6.6-default 3.0.101-0.7.21-pae=3.0.74-0.6.6-pae 3.0.101-0.8-default=3.0.76-0.11-default 3.0.101-0.8-pae=3.0.76-0.11-pae 3.0.58-0.6.2-default=3.0.13-0.27-default 3.0.58-0.6.2-pae=3.0.13-0.27-pae 3.0.58-0.6.6-default=3.0.13-0.27-default 3.0.58-0.6.6-pae=3.0.13-0.27-pae 3.0.74-0.6.10-default=3.0.74-0.6.6-default 3.0.74-0.6.10-pae=3.0.74-0.6.6-pae 3.0.74-0.6.8-default=3.0.74-0.6.6-default 3.0.74-0.6.8-pae=3.0.74-0.6.6-pae 3.0.80-0.5-default=3.0.74-0.6.6-default 3.0.80-0.5-pae=3.0.74-0.6.6-pae 3.0.80-0.7-default=3.0.74-0.6.6-default 3.0.80-0.7-pae=3.0.74-0.6.6-pae 3.0.82-0.7-default=3.0.76-0.11-default 3.0.82-0.7-pae=3.0.76-0.11-pae 3.0.93-0.5-default=3.0.74-0.6.6-default 3.0.93-0.5-pae=3.0.74-0.6.6-pae 3.0.93-0.8-default=3.0.76-0.11-default 3.0.93-0.8-pae=3.0.76-0.11-pae  "

# Ubuntu 1204 on X86

KVER_VALS_LUBT12_X86=" 3.2.0-23-generic 3.2.0-23-generic-pae 3.5.0-23-generic 3.8.0-29-generic 3.8.0-31-generic 3.11.0-15-generic 3.2.0-29-generic-pae 3.11.0-17-generic 3.2.0-29-generic 3.8.0-33-generic 3.8.0-35-generic 3.8.0-36-generic 3.8.0-37-generic 3.11.0-13-generic 3.13.0-24-generic 3.13.0-27-generic 3.2.0-36-generic 3.2.0-38-generic 3.2.0-41-generic 3.2.0-53-generic 3.2.0-55-generic 3.2.0-57-generic 3.2.0-60-generic 3.2.0-36-generic-pae 3.2.0-38-generic-pae 3.2.0-41-generic-pae 3.2.0-53-generic-pae 3.2.0-55-generic-pae 3.2.0-57-generic-pae 3.2.0-60-generic-pae 3.2.0-44-generic 3.2.0-44-generic-pae 3.5.0-24-generic 3.5.0-26-generic 3.5.0-28-generic 3.5.0-31-generic  3.5.0-37-generic 3.5.0-40-generic 3.5.0-41-generic 3.5.0-43-generic 3.5.0-47-generic 3.5.0-48-generic 3.8.0-19-generic 3.8.0-30-generic=3.8.0-29-generic 3.8.0-32-generic=3.8.0-31-generic 3.8.0-34-generic=3.8.0-33-generic 3.8.0-38-generic=3.8.0-37-generic 3.8.0-39-generic=3.8.0-37-generic 3.8.0-41-generic=3.8.0-37-generic 3.8.0-42-generic=3.8.0-37-generic 3.8.0-44-generic=3.8.0-37-generic 3.11.0-14-generic=3.11.0-13-generic 3.11.0-18-generic=3.11.0-17-generic 3.11.0-19-generic=3.11.0-17-generic 3.11.0-20-generic=3.11.0-17-generic 3.11.0-22-generic=3.11.0-17-generic 3.11.0-23-generic=3.11.0-17-generic 3.11.0-24-generic=3.11.0-17-generic 3.11.0-26-generic=3.11.0-17-generic 3.13.0-29-generic=3.13.0-27-generic 3.13.0-30-generic=3.13.0-27-generic 3.13.0-32-generic=3.13.0-27-generic 3.2.0-37-generic=3.2.0-36-generic 3.2.0-39-generic=3.2.0-38-generic 3.2.0-40-generic=3.2.0-38-generic 3.2.0-43-generic=3.2.0-41-generic 3.2.0-45-generic=3.2.0-44-generic 3.2.0-48-generic=3.2.0-44-generic 3.2.0-49-generic=3.2.0-44-generic 3.2.0-51-generic=3.2.0-44-generic 3.2.0-52-generic=3.2.0-44-generic 3.2.0-54-generic=3.2.0-53-generic 3.2.0-56-generic=3.2.0-55-generic 3.2.0-58-generic=3.2.0-57-generic 3.2.0-59-generic=3.2.0-57-generic 3.2.0-61-generic=3.2.0-60-generic 3.2.0-63-generic=3.2.0-60-generic 3.2.0-64-generic=3.2.0-60-generic 3.2.0-65-generic=3.2.0-60-generic 3.2.0-67-generic=3.2.0-60-generic 3.5.0-25-generic=3.5.0-24-generic 3.5.0-27-generic=3.5.0-26-generic 3.5.0-30-generic=3.5.0-28-generic 3.5.0-32-generic=3.5.0-31-generic 3.5.0-34-generic=3.5.0-31-generic 3.5.0-36-generic=3.5.0-31-generic 3.5.0-39-generic=3.5.0-37-generic 3.5.0-42-generic=3.5.0-41-generic 3.5.0-44-generic=3.5.0-43-generic 3.5.0-45-generic=3.5.0-43-generic 3.5.0-46-generic=3.5.0-43-generic 3.5.0-49-generic=3.5.0-48-generic 3.5.0-51-generic=3.5.0-48-generic 3.5.0-52-generic=3.5.0-48-generic 3.5.0-54-generic=3.5.0-48-generic 3.8.0-21-generic=3.8.0-19-generic 3.8.0-22-generic=3.8.0-19-generic 3.8.0-23-generic=3.8.0-19-generic 3.8.0-25-generic=3.8.0-19-generic 3.8.0-26-generic=3.8.0-19-generic 3.8.0-27-generic=3.8.0-19-generic 3.2.0-37-generic-pae=3.2.0-36-generic-pae 3.2.0-39-generic-pae=3.2.0-38-generic-pae 3.2.0-40-generic-pae=3.2.0-38-generic-pae 3.2.0-43-generic-pae=3.2.0-41-generic-pae 3.2.0-45-generic-pae=3.2.0-44-generic-pae 3.2.0-48-generic-pae=3.2.0-44-generic-pae 3.2.0-49-generic-pae=3.2.0-44-generic-pae 3.2.0-51-generic-pae=3.2.0-44-generic-pae 3.2.0-52-generic-pae=3.2.0-44-generic-pae 3.2.0-54-generic-pae=3.2.0-53-generic-pae 3.2.0-56-generic-pae=3.2.0-55-generic-pae 3.2.0-58-generic-pae=3.2.0-57-generic-pae 3.2.0-59-generic-pae=3.2.0-57-generic-pae 3.2.0-61-generic-pae=3.2.0-60-generic-pae 3.2.0-63-generic-pae=3.2.0-60-generic-pae 3.2.0-64-generic-pae=3.2.0-60-generic-pae 3.2.0-65-generic-pae=3.2.0-60-generic-pae 3.2.0-67-generic-pae=3.2.0-60-generic-pae  "

# RHEL 4 / CentOS 4 on AMD64

KVER_VALS_LEL4_AMD64="   "

# RHEL 5 / CentOS 5 / OEL 5 on AMD64

KVER_VALS_LEL5_AMD64=" 2.6.18-8.el5 2.6.18-53.el5 2.6.18-92.el5 2.6.18-128.el5 2.6.18-164.el5 2.6.18-194.el5 2.6.18-238.el5 2.6.18-274.el5 2.6.18-308.el5 2.6.18-348.el5 2.6.18-371.el5 2.6.39-100.5.1.el5uek 2.6.39-300.26.1.el5uek 2.6.18-238.48.1.el5=2.6.18-238.el5 2.6.18-238.49.1.el5=2.6.18-238.el5 2.6.18-238.50.1.el5=2.6.18-238.el5 2.6.18-238.51.1.el5=2.6.18-238.el5 2.6.18-238.52.1.el5=2.6.18-238.el5 2.6.18-348.1.1.el5=2.6.18-348.el5 2.6.18-348.12.1.el5=2.6.18-348.el5 2.6.18-348.16.1.el5=2.6.18-348.el5 2.6.18-348.18.1.el5=2.6.18-348.el5 2.6.18-348.19.1.el5=2.6.18-348.el5 2.6.18-348.2.1.el5=2.6.18-348.el5 2.6.18-348.21.1.el5=2.6.18-348.el5 2.6.18-348.22.1.el5=2.6.18-348.el5 2.6.18-348.23.1.el5=2.6.18-348.el5 2.6.18-348.25.1.el5=2.6.18-348.el5 2.6.18-348.27.1.el5=2.6.18-348.el5 2.6.18-348.28.1.el5=2.6.18-348.el5 2.6.18-348.3.1.el5=2.6.18-348.el5 2.6.18-348.4.1.el5=2.6.18-348.el5 2.6.18-348.6.1.el5=2.6.18-348.el5 2.6.18-371.1.2.el5=2.6.18-371.el5 2.6.18-371.3.1.el5=2.6.18-371.el5 2.6.18-371.4.1.el5=2.6.18-371.el5 2.6.18-371.6.1.el5=2.6.18-371.el5 2.6.18-371.8.1.el5=2.6.18-371.el5 2.6.18-371.9.1.el5=2.6.18-371.el5 2.6.18-371.11.1.el5=2.6.18-371.el5 2.6.39-100.7.1.el5uek=2.6.39-100.5.1.el5uek  "

# RHEL 6 on AMD64

KVER_VALS_LEL6_AMD64=" 2.6.32-71.el6.x86_64 2.6.32-131.0.15.el6.x86_64 2.6.32-220.el6.x86_64 2.6.32-279.el6.x86_64 2.6.32-358.el6.x86_64 2.6.32-431.el6.x86_64 2.6.32-279.22.1.el6.x86_64 2.6.32-220.31.1.el6.x86_64 2.6.32-358.23.2.el6.x86_64 2.6.39-100.5.1.el6uek.x86_64 2.6.39-200.24.1.el6uek.x86_64 2.6.39-400.17.1.el6uek.x86_64 2.6.39-300.17.2.el6uek.x86_64 2.6.32-131.37.1.el6.x86_64=2.6.32-131.0.15.el6.x86_64 2.6.32-131.38.1.el6.x86_64=2.6.32-131.0.15.el6.x86_64 2.6.32-131.39.1.el6.x86_64=2.6.32-131.0.15.el6.x86_64 2.6.32-220.32.1.el6.x86_64=2.6.32-220.31.1.el6.x86_64 2.6.32-220.34.1.el6.x86_64=2.6.32-220.31.1.el6.x86_64 2.6.32-220.34.2.el6.x86_64=2.6.32-220.31.1.el6.x86_64 2.6.32-220.38.1.el6.x86_64=2.6.32-220.31.1.el6.x86_64 2.6.32-220.39.1.el6.x86_64=2.6.32-220.31.1.el6.x86_64 2.6.32-220.41.1.el6.x86_64=2.6.32-220.31.1.el6.x86_64 2.6.32-220.42.1.el6.x86_64=2.6.32-220.31.1.el6.x86_64 2.6.32-220.45.1.el6.x86_64=2.6.32-220.31.1.el6.x86_64 2.6.32-220.46.1.el6.x86_64=2.6.32-220.31.1.el6.x86_64 2.6.32-279.23.1.el6.x86_64=2.6.32-279.22.1.el6.x86_64 2.6.32-279.25.1.el6.x86_64=2.6.32-279.22.1.el6.x86_64 2.6.32-279.25.2.el6.x86_64=2.6.32-279.22.1.el6.x86_64 2.6.32-279.31.1.el6.x86_64=2.6.32-279.22.1.el6.x86_64 2.6.32-279.33.1.el6.x86_64=2.6.32-279.22.1.el6.x86_64 2.6.32-279.34.1.el6.x86_64=2.6.32-279.22.1.el6.x86_64 2.6.32-279.37.2.el6.x86_64=2.6.32-279.22.1.el6.x86_64 2.6.32-279.39.1.el6.x86_64=2.6.32-279.22.1.el6.x86_64 2.6.32-279.41.1.el6.x86_64=2.6.32-279.22.1.el6.x86_64 2.6.32-279.42.1.el6.x86_64=2.6.32-279.22.1.el6.x86_64 2.6.32-279.43.1.el6.x86_64=2.6.32-279.22.1.el6.x86_64 2.6.32-279.43.2.el6.x86_64=2.6.32-279.22.1.el6.x86_64 2.6.32-279.46.1.el6.x86_64=2.6.32-279.22.1.el6.x86_64 2.6.32-358.0.1.el6.x86_64=2.6.32-358.el6.x86_64 2.6.32-358.11.1.el6.x86_64=2.6.32-358.el6.x86_64 2.6.32-358.14.1.el6.x86_64=2.6.32-358.el6.x86_64 2.6.32-358.18.1.el6.x86_64=2.6.32-358.el6.x86_64 2.6.32-358.2.1.el6.x86_64=2.6.32-358.el6.x86_64 2.6.32-358.28.1.el6.x86_64=2.6.32-358.23.2.el6.x86_64 2.6.32-358.32.3.el6.x86_64=2.6.32-358.23.2.el6.x86_64 2.6.32-358.37.1.el6.x86_64=2.6.32-358.23.2.el6.x86_64 2.6.32-358.41.1.el6.x86_64=2.6.32-358.23.2.el6.x86_64 2.6.32-358.44.1.el6.x86_64=2.6.32-358.23.2.el6.x86_64 2.6.32-358.46.1.el6.x86_64=2.6.32-358.23.2.el6.x86_64 2.6.32-358.46.2.el6.x86_64=2.6.32-358.23.2.el6.x86_64 2.6.32-358.6.1.el6.x86_64=2.6.32-358.el6.x86_64 2.6.32-358.6.2.el6.x86_64=2.6.32-358.el6.x86_64 2.6.32-431.1.2.el6.x86_64=2.6.32-431.el6.x86_64 2.6.32-431.11.2.el6.x86_64=2.6.32-431.el6.x86_64 2.6.32-431.3.1.el6.x86_64=2.6.32-431.el6.x86_64 2.6.32-431.5.1.el6.x86_64=2.6.32-431.el6.x86_64 2.6.32-431.17.1.el6.x86_64=2.6.32-431.el6.x86_64 2.6.32-431.20.3.el6.x86_64=2.6.32-431.el6.x86_64 2.6.32-431.20.5.el6.x86_64=2.6.32-431.el6.x86_64 2.6.32-431.23.3.el6.x86_64=2.6.32-431.el6.x86_64 2.6.39-300.17.3.el6uek.x86_64=2.6.39-300.17.2.el6uek.x86_64 2.6.39-300.26.1.el6uek.x86_64=2.6.39-300.17.2.el6uek.x86_64 2.6.39-400.17.2.el6uek.x86_64=2.6.39-400.17.1.el6uek.x86_64 2.6.39-400.21.2.el6uek.x86_64=2.6.39-400.17.1.el6uek.x86_64 2.6.39-400.24.1.el6uek.x86_64=2.6.39-400.17.1.el6uek.x86_64 2.6.39-400.109.5.el6uek.x86_64=2.6.39-400.17.1.el6uek.x86_64 2.6.39-400.214.3.el6uek.x86_64=2.6.39-400.17.1.el6uek.x86_64 2.6.39-400.214.4.el6uek.x86_64=2.6.39-400.17.1.el6uek.x86_64  "

# SuSE 9 on AMD64

KVER_VALS_LSES9_AMD64="   "

# SuSE 10 on AMD64

KVER_VALS_LSES10_AMD64=" 2.6.16.21-0.8-default 2.6.16.21-0.8-smp 2.6.16.46-0.12-default 2.6.16.46-0.12-smp 2.6.16.60-0.21-default 2.6.16.60-0.21-smp 2.6.16.60-0.54.5-default 2.6.16.60-0.54.5-smp 2.6.16.60-0.85.1-default 2.6.16.60-0.85.1-smp 2.6.16.60-0.101.1-default=2.6.16.60-0.85.1-default 2.6.16.60-0.101.1-smp=2.6.16.60-0.85.1-smp 2.6.16.60-0.103.1-default=2.6.16.60-0.85.1-default 2.6.16.60-0.103.1-smp=2.6.16.60-0.85.1-smp 2.6.16.60-0.105.1-default=2.6.16.60-0.85.1-default 2.6.16.60-0.105.1-smp=2.6.16.60-0.85.1-smp 2.6.16.60-0.107.1-default=2.6.16.60-0.85.1-default 2.6.16.60-0.107.1-smp=2.6.16.60-0.85.1-smp 2.6.16.60-0.113.1-default=2.6.16.60-0.54.5-default 2.6.16.60-0.113.1-smp=2.6.16.60-0.54.5-smp 2.6.16.60-0.123.1-default=2.6.16.60-0.54.5-default 2.6.16.60-0.123.1-smp=2.6.16.60-0.54.5-smp  "

# WindRiver 1.4 on PPC32
KVER_VALS_LWR14_PPC32="@@LWR14_PPC32@@"

# WindRiver 4.0 on X86
KVER_VALS_LWR40_X86="@@LWR40_X86@@"

# Ubuntu 1204 on AMD64
KVER_VALS_LUBT12_AMD64=" 3.2.0-23-generic 3.5.0-23-generic 3.8.0-29-generic 3.8.0-31-generic 3.11.0-15-generic 3.2.0-29-generic 3.11.0-17-generic 3.8.0-33-generic 3.8.0-35-generic 3.8.0-36-generic 3.8.0-37-generic 3.11.0-13-generic 3.13.0-24-generic 3.13.0-27-generic 3.2.0-36-generic 3.2.0-38-generic 3.2.0-41-generic 3.2.0-53-generic 3.2.0-55-generic 3.2.0-57-generic 3.2.0-60-generic 3.2.0-44-generic  3.5.0-24-generic 3.5.0-26-generic 3.5.0-28-generic 3.5.0-31-generic 3.5.0-37-generic 3.5.0-40-generic 3.5.0-41-generic 3.5.0-43-generic 3.5.0-47-generic 3.5.0-48-generic  3.8.0-19-generic 3.2.0-37-generic=3.2.0-36-generic 3.2.0-39-generic=3.2.0-38-generic 3.2.0-40-generic=3.2.0-38-generic 3.2.0-43-generic=3.2.0-41-generic 3.2.0-45-generic=3.2.0-44-generic 3.2.0-48-generic=3.2.0-44-generic 3.2.0-49-generic=3.2.0-44-generic 3.2.0-51-generic=3.2.0-44-generic 3.2.0-52-generic=3.2.0-44-generic 3.2.0-54-generic=3.2.0-53-generic 3.2.0-56-generic=3.2.0-55-generic 3.2.0-58-generic=3.2.0-57-generic 3.2.0-59-generic=3.2.0-57-generic 3.2.0-61-generic=3.2.0-60-generic 3.2.0-63-generic=3.2.0-60-generic 3.2.0-64-generic=3.2.0-60-generic 3.2.0-65-generic=3.2.0-60-generic 3.2.0-67-generic=3.2.0-60-generic 3.8.0-30-generic=3.8.0-29-generic 3.8.0-32-generic=3.8.0-31-generic 3.8.0-34-generic=3.8.0-33-generic 3.8.0-38-generic=3.8.0-37-generic 3.8.0-39-generic=3.8.0-37-generic 3.8.0-41-generic=3.8.0-37-generic 3.8.0-42-generic=3.8.0-37-generic 3.8.0-44-generic=3.8.0-37-generic 3.11.0-14-generic=3.11.0-13-generic 3.11.0-18-generic=3.11.0-17-generic 3.11.0-19-generic=3.11.0-17-generic 3.11.0-20-generic=3.11.0-17-generic 3.11.0-22-generic=3.11.0-17-generic 3.11.0-23-generic=3.11.0-17-generic 3.11.0-24-generic=3.11.0-17-generic 3.11.0-26-generic=3.11.0-17-generic 3.13.0-29-generic=3.13.0-27-generic 3.13.0-30-generic=3.13.0-27-generic 3.13.0-32-generic=3.13.0-27-generic 3.5.0-25-generic=3.5.0-24-generic 3.5.0-27-generic=3.5.0-26-generic 3.5.0-30-generic=3.5.0-28-generic 3.5.0-32-generic=3.5.0-31-generic 3.5.0-34-generic=3.5.0-31-generic 3.5.0-36-generic=3.5.0-31-generic 3.5.0-39-generic=3.5.0-37-generic 3.5.0-42-generic=3.5.0-41-generic 3.5.0-44-generic=3.5.0-43-generic 3.5.0-45-generic=3.5.0-43-generic 3.5.0-46-generic=3.5.0-43-generic 3.5.0-49-generic=3.5.0-48-generic 3.5.0-51-generic=3.5.0-48-generic 3.5.0-52-generic=3.5.0-48-generic 3.5.0-54-generic=3.5.0-48-generic 3.8.0-21-generic=3.8.0-19-generic 3.8.0-22-generic=3.8.0-19-generic 3.8.0-23-generic=3.8.0-19-generic 3.8.0-25-generic=3.8.0-19-generic 3.8.0-26-generic=3.8.0-19-generic 3.8.0-27-generic=3.8.0-19-generic  "

KVER_VALS=$(eval echo \${KVER_VALS_${host_distro}_${host_arch}})

# WindRiver 4.2 on X86
KVER_VALS_LWR42_X86="@@LWR42_X86@@"

# WindRiver 4.3 on X86
KVER_VALS_LWR43_X86="@@LWR43_X86@@"

# WindRiver 5.0 and above on X86
KVER_VALS_LWR="3.14.58_IDP-XT_3.1-WR7.0.0.13_idp"

KVER_VALS=`eval echo \\${KVER_VALS_${host_distro}_${host_arch}}`

HOST_KERNEL_VERSION=`uname -r`

if [ "$RELEASE_PATH" != "/etc/issue" -a "$RELEASE_PATH" != "/VERSION.os" ] ; then
    SUPPORTED_KERNEL=0

    for vv in $KVER_VALS
    do
        KER_VER=`echo $vv | awk -F "=" {'print $1'}`
        if [ "$KER_VER" = "$HOST_KERNEL_VERSION" ]; then
            SUPPORTED_KERNEL=1;
            break;
        fi
    done

    if [ $SUPPORTED_KERNEL -eq 0 ]; then
        _append_analysis 2 "$MSG_ERR_UNSUPPORTED_KERNEL" \
                                "$MSG_LABEL_KERNEL";
        return 1
    fi;
fi
        return 0
}

# function to check if the distribution is not supported.
check_for_unsupported_distributions () 
{
    FOUND_INVALID_DISTRIBUTION=0;
    if [ "$THIS_PLATFORM" = "Linux" ]; then
	# linux case is different. it has some supported distributions.
	# check for kernel versions is not made. 
    	FOUND_INVALID_DISTRIBUTION=1;


        if [ -f "$RELEASE_PATH" ]; then
    	    FOUND_INVALID_DISTRIBUTION=0

	    #NOTE: To add a new linux distro to supported list, update at both the places below.
    	    MSG_SUPPORTED_DISTRIBUTIONS="$LRH72 $AS3, $AS4, $ES3, $ES4, $WS3, $WS4, $EL5, $OEL5, $EL6, $OEL6, $CS4, $CS5, $CS6, $CS6_0, $SES9, $SES9x, $SES10 $SED11 $SES11 $Opensuse10 $Opensuse11 $LWR14, $LWR40, $LWR42, $LWR43, $LWR60, $LWR, $Ubuntu1204" 

    	    case $DISTRO_STRING in
    	    $AS3);;           # Redhat AS 3
     	    $AS4);;           # Redhat AS 4
    	    $ES3);;           # Redhat ES 3
    	    $ES4);;           # Redhat ES 4
    	    $WS3);;           # Redhat WS 3
    	    $WS4);;           # Redhat WS 4
    	    $EL5);;           # Redhat EL 5
    	    $OEL5);;          # Oracle EL 5
    	    $CS4);;           # CentOS 4
    	    $CS5);;           # CentOS 5
    	    $CS6);;           # CentOS 6
    	    $CS6_0);;         # CentOS 6
    	    $EL6);;           # Redhat EL 6
    	    $OEL6);;          # Oracle EL 6
    	    $SES9);;          # SLES 9
    	    $SES9x);;         # Suse 9.3 Pro
    	    $SES10);;         # SLES 10
    	    $SED11);;         # SLED 11
    	    $SES11);;         # SLES 11
    	    $LRH72);;         # Redhat 7.2
    	    $Opensuse10);;    # OpenSuse 10
    	    $Opensuse11);;    # OpenSuse 11
            $LWR14);;         # WindRiver 1.4
            $LWR40);;         # WindRiver 4.0
            $LWR42);;         # WindRiver 4.2
            $LWR43);;         # WindRiver 4.3
	    $LWR60);;         # WindRiver 6.0
            $LWR);;           # WindRiver 5.0 and onwards
    	    $Ubuntu1204);;    # Ubuntu 12
    	    *)FOUND_INVALID_DISTRIBUTION=1
    	    esac
    	fi

        if [ $FOUND_INVALID_DISTRIBUTION -eq 0 ]; then
	    check_for_supported_kernels;
            if [ $? -eq 1 ]; then
	    	return 1
	    fi
    	fi
    else

	MSG_SUPPORTED_DISTRIBUTIONS="$ANALYSIS_SUPPORTED_DISTRIBUTIONS"
	_contains "$DISTRO_STRING" \
	    	"$ANALYSIS_SUPPORTED_DISTRIBUTIONS"; 
    	[ $? -ne 1 ] && FOUND_INVALID_DISTRIBUTION=1;
    fi

    MSG_VALID_OS="List of supported OS distributions: \"$MSG_SUPPORTED_DISTRIBUTIONS\"."
    if [ $FOUND_INVALID_DISTRIBUTION -eq 1 ]; then
	_append_analysis 2 "$MSG_ERR_UNSUPPORTED_OS" \
	    			"$MSG_LABEL_OS_DISTRIBUTION";
 	_append_analysis 3 "$MSG_VALID_OS"
    fi;

    return $FOUND_INVALID_DISTRIBUTION; 
}

# Function to check if the available space meets min. requirement
check_for_available_disk_space () 
{
    DISKSPACE_CHECKED_FOR="$1"
    FOUND_LOW_DISK_SPACE=0;
    STR_DISKSPACE=`$CMDS_DISK_SPACE_INFO $SC_INSTALL_DIR`;
    STR_DISKSPACE_ROOT=`$CMDS_DISK_SPACE_INFO /`; 
    
    if [ "$THIS_PLATFORM" = "Linux" ]; then
	diff_factor=`echo $STR_DISKSPACE | tr '\n' ' ' | awk '{print $8}'`
	if [ "$diff_factor" = "on" ]; then
	    DISKSPACE_KB=`echo $STR_DISKSPACE | awk {'print $13'}`;
    	    DISKSPACE_ROOT_KB=`echo $STR_DISKSPACE_ROOT | awk {'print $13'}`;
	else
	    DISKSPACE_KB=`echo $STR_DISKSPACE | awk {'print $12'}`;
            DISKSPACE_ROOT_KB=`echo $STR_DISKSPACE_ROOT | awk {'print $12'}`;
        fi

	elif [ "$THIS_PLATFORM" = "SunOS" ]; then
	DISKSPACE_KB=`echo $STR_DISKSPACE | awk {'print $11'}`; 
	DISKSPACE_ROOT_KB=`echo $STR_DISKSPACE_ROOT | awk {'print $11'}`; 
    elif [ "$THIS_PLATFORM" = "HP-UX" ]; then
	DISKSPACE_KB=`echo $STR_DISKSPACE | awk {'print $11'}`; 
	DISKSPACE_ROOT_KB=`echo $STR_DISKSPACE_ROOT | awk {'print $11'}`; 
    elif [ "$THIS_PLATFORM" = "AIX" ]; then
	DISKSPACE_KB=`echo $STR_DISKSPACE | awk {'print $11'}`; 
	DISKSPACE_ROOT_KB=`echo $STR_DISKSPACE_ROOT | awk {'print $11'}`; 
    fi;

     #If disk space is checked for solidcore installation
     if [ "$DISKSPACE_CHECKED_FOR" = "sc_installation" ] ;then
        if [ $DISKSPACE_KB -le $ANALYSIS_REQD_DISK_SPACE_KB ]; then
	  DISKSPACE_MB=`expr $DISKSPACE_KB / 1024`;	
          _append_analysis 2 "$MSG_ERR_LOW_DISK_SPACE$DISKSPACE_MB MB" \
                              "$MSG_LABEL_DISK_SPACE";
          FOUND_LOW_DISK_SPACE=1;
        fi;

        if [ $DISKSPACE_ROOT_KB -le $ANALYSIS_REQD_DISK_SPACE_ROOT_KB ]; then
	  DISKSPACE_ROOT_MB=`expr $DISKSPACE_ROOT_KB / 1024`;
          _append_analysis 2 "$MSG_ERR_LOW_DISK_ROOT_SPACE$DISKSPACE_ROOT_MB MB" \
                                 "$MSG_LABEL_DISK_SPACE";
          FOUND_LOW_DISK_SPACE=1;
        fi

    else # disk space checked for scananalyzer log file
       
       if [ $DISKSPACE_KB -le $LOG_REQD_DISK_SPACE_KB ]; then
         DISKSPACE_MB=`expr $DISKSPACE_KB / 1024`;
         echo "$MSG_ERR_LOG_REQD_DISK_SPACE$DISKSPACE_MB MB ";
         echo " Please free required disk space and rerun the script"
         exit 1;
       fi
    fi; 
   return $FOUND_LOW_DISK_SPACE; 
}

# function to look for atleast one supported file system. 
check_for_supported_file_systems () 
{
    for k in $ALL_FILESYSTEMS; do 
	_contains "$k" "$ANALYSIS_SUPPORTED_FILESYSTEMS"; 
    	if [ $? -eq 1 ]; then 
		return 0;
	fi 
    done 

    _append_analysis 2 \
    	"$MSG_ERR_UNSUPPORTED_FILESYSTEM" "$MSG_LABEL_FILE_SYSTEMS";
    return 1;
}

# function to check for minimum memory requirement
check_for_physical_memory () 
{
    FOUND_LOW_PHYSICAL_RAM=0;
    if [ "$PHYSICAL_MEMORY_KB" ] ; then 
 	   if [ $PHYSICAL_MEMORY_KB -lt $ANALYSIS_REQD_RAM ]; then
		_append_analysis 2 \
                   "$MSG_ERR_LOW_PHYSICAL_MEMORY Required Memory:$ANALYSIS_REQD_RAM Kbytes, Available Memory:$PHYSICAL_MEMORY_KB Kbytes"\
                   "$MSG_LABEL_PHYSICAL_MEMORY";

		FOUND_LOW_PHYSICAL_RAM=1;
 	   fi 
    fi 
    return $FOUND_LOW_PHYSICAL_RAM;
}

# function to check for McAfee Solidifier installation and open port.
check_for_solidifier_installation () 
{
    FOUND_SOLIDIFIER_INSTALLED=0
    RETVAL=0

    $CMD_PACKAGE_QUERY $S3CONTROL >/dev/null 2>&1;
    [ $? -eq 0 ] && FOUND_SOLIDIFIER_INSTALLED=1;

    if [ $FOUND_SOLIDIFIER_INSTALLED -eq 1 ]; then
	_append_analysis 1 "$MSG_ERR_SOLIDIFIER_INSTALLED" \
		    			"$MSG_LABEL_SOLIDIFIER_INSTALLATION"
	SC_IS_SOLIDIFIER_INSTALLED=1
    fi

    return $RETVAL
}

# function to check for any required packages. 
check_for_prerequisite_packages () 
{
    FOUND_MISSING_PREREQUISITE_PKG=0;
    MISSING_PKGS="";
    for p in $ANALYSIS_PREREQUISITE_PKGS; do
	eval $CMDS_PACKAGES_LIST | grep $p >/dev/null 2>&1;
    	if [ $? -ne 0 ]; then 
	    MISSING_PKGS="$MISSING_PKGS$p ";
	    FOUND_MISSING_PREREQUISITE_PKG=1;
	fi 
    done; 

    if [ $FOUND_MISSING_PREREQUISITE_PKG -eq 1 ]; then 
	_append_analysis 2 \
	    "$MSG_ERR_REQUIRED_PKG_MISSING $MISSING_PKGS" \
  			"$MSG_LABEL_PREREQUISITE_PKGS";
    fi 
    
    return $FOUND_MISSING_PREREQUISITE_PKG;
}

# function to check for the any problematic packages.
check_for_problematic_packages () 
{
    FOUND_PROBLEMATIC_PKG=0;
    PROBLEMATIC_PKGS="";
    for c in $ANALYSIS_PROBLEMATIC_PKGS; do 
	eval $CMDS_PACKAGES_LIST | grep $c >/dev/null 2>&1;
    	if [ $? -eq 0 ]; then 
		PROBLEMATIC_PKGS="$PROBLEMATIC_PKGS$c ";
		FOUND_PROBLEMATIC_PKG=1;
	fi;   
    done; 

    if [ $FOUND_PROBLEMATIC_PKG -eq 1 ]; then 
	_append_analysis 2 \
	    "$MSG_ERR_PROBLEMATIC_PKG_FOUND $PROBLEMATIC_PKGS" \
	    		"$MSG_LABEL_PROBLEMATIC_PKGS";
    fi 
    
    return $FOUND_PROBLEMATIC_PKG;
}

# function for checking any problematic modules
check_for_problematic_modules () 
{
    # check for presnece of genkex on AIX.
    if [ $THIS_PLATFORM = "AIX" ]; then
	IS_GENKEX_PRESENT=0
	genkex > /dev/null 2>/dev/null
	if [ $? -eq 127 ]; then
	    IS_GENKEX_PRESENT=1
	    _append_analysis 2 \
    		"$MSG_ERR_GENKEX_MISSING" "$MSG_LABEL_PREREQUISITE_PKGS";
	fi
	return $IS_GENKEX_PRESENT
    fi

    # now check for problematic modules.
    
    FOUND_PROBLEMATIC_MODULE=0;
    FOUND_SOLIDIFIER_INSTALLED=0;

    $CMD_PACKAGE_QUERY $S3CONTROL >/dev/null 2>&1;
    [ $? -eq 0 ] && FOUND_SOLIDIFIER_INSTALLED=1;

    if [ $FOUND_SOLIDIFIER_INSTALLED -ne 1 ]; then 
 	   
	PROBLEMATIC_MODULES="";
	    for m in $ANALYSIS_PROBLEMARIC_MODULES; do
		eval $CMD_LOADED_MODULES | grep $m >/dev/null 2>&1; 
	    	if [ $? -eq 0 ]; then
			PROBLEMATIC_MODULES="$PROBLEMATIC_MODULES$m ";
			FOUND_PROBLEMATIC_MODULE=1;
		fi 
	    done 

 	   if [ $FOUND_PROBLEMATIC_MODULE -eq 1 ]; then
		_append_analysis 2 \
		    "$MSG_ERR_PROBLEMATIC_MODULE_FOUND $PROBLEMATIC_MODULES" \
		    		"$MSG_LABEL_PROBLEMATIC_MODULES";
	   fi
    fi 

    return $FOUND_PROBLEMATIC_MODULE;
}

# function for checking for some platform specifc data not covered in other fn.
check_for_platform_specific_misc_data () 
{
    FOUND_MISC_PROBLEM=0;

    if [ "$THIS_PLATFORM" = "SunOS" ]; then
	 # Solaris System /etc/system file 
	 $CMD_GET_ETC_SYSTEM > $OUT_ETC_SYSTEM; 
    	 STACK_PROTECTION_BIT=`eval $CMD_GET_ETC_SYSTEM|grep \
			-w noexec_user_stack|awk {'print $4'}`;
    	 if [ "$STACK_PROTECTION_BIT" = "1" ]; then
		_append_analysis 0 "$MSG_ERR_STACK_PROTECTION_ON" \
			  "$MSG_LABEL_STACK_PROTECTION_ON";
	 fi
    elif [ "$THIS_PLATFORM" = "AIX" ]; then
	 $CMD_GET_SYSLOG_CONF > $OUT_SYSLOG_CONF; 
    elif [ "$THIS_PLATFORM" = "HP-UX" ]; then
	if [ "`uname -r`" = "B.11.00" -o "`uname -r`" = "B.11.11" ]; then
		eval $CMD_NM_VERSION 2>&1 | grep "PA64" >/dev/null 2>&1;
		if [ $? -ne 0 ]; then
		    _append_analysis 2 \
			"$MSG_ERR_NM_VERSION_NON_HP" "$MSG_LABEL_NM_VERSION";
		    FOUND_MISC_PROBLEM=1;
		fi
	fi
    fi 

    return $FOUND_MISC_PROBLEM;
}

# check for any files on system that may conflict with McAfee Solidifier files
check_for_solidifier_conflicts () 
{
    ERR_FOUND=0;
    FOUND_SOLIDIFIER_INSTALLED=0;
    
    SC_CONFIG_FILE="/etc/mcafee/solidcore/solidcore.conf";

    if [ $THIS_ARCH = "ppc" ]; then
        if [ -f $SC_CONFIG_FILE ]; then
            FOUND_SOLIDIFIER_INSTALLED=1;
        fi
    else
        $CMD_PACKAGE_QUERY $S3CONTROL >/dev/null 2>&1;
        [ $? -eq 0 ] && FOUND_SOLIDIFIER_INSTALLED=1;
    fi

    if [ $FOUND_SOLIDIFIER_INSTALLED -eq 1 ]; then 
	return 1;
    fi 
    
    if [ ! -f $SC_CONFIG_FILE ]; then
	return 0;
    fi

    SC_ROOT_DIR=`cat $SC_CONFIG_FILE | grep InstallDir | awk {'print $3'}`
    SOLIDIFIER_CLI_BINARY_LINK="/usr/sbin/sadmin";

    if [ -d "$SC_ROOT_DIR" ]; then
	_append_analysis 1 "$MSG_ERR_SOLIDIFIER_CONFLICTS$SC_ROOT_DIR" \
	    	"$MSG_LABEL_SOLIDIFIER_CONFLICTS";
    	ERR_FOUND=1;
    fi 

    if [ -f "$SOLIDIFIER_CLI_BINARY_LINK" ]; then
	_append_analysis 1 \
	    "$MSG_ERR_SOLIDIFIER_CONFLICTS$SOLIDIFIER_CLI_BINARY_LINK" \
	    	"$MSG_LABEL_SOLIDIFIER_CONFLICTS";
    	ERR_FOUND=1;
    fi 

    if [ -f "$SC_CONFIG_FILE" ]; then
	_append_analysis 1 \
	    "$MSG_ERR_SOLIDIFIER_CONFLICTS$SC_CONFIG_FILE" \
	    	"$MSG_LABEL_SOLIDIFIER_CONFLICTS";
    	ERR_FOUND=1;
    fi; 

    return $ERR_FOUND;
}

# function to analyze the data that is collected 
sc_analyze_data_collected () 
{
    SOME_PROBLEM_FOUND=0;

    if [ "$QOPT" = 0 ]; then
	echo "Analyzing system for McAfee Solidifier installation..."    
    fi
    
    echo "	[ Analysis of the Collected Data ]">>$OUT_REPORT;
    echo "">>$OUT_REPORT;

    check_for_unsupported_distributions; 
    [ $? -eq 1 ] && SOME_PROBLEM_FOUND=1;

    if [ "$host_distro" = "LWR14" ] || [ "$host_distro" = "LWR40" ]; then
        SC_INSTALL_DIR="/opt"
    fi

    check_for_available_disk_space "sc_installation";
    [ $? -eq 1 ] && SOME_PROBLEM_FOUND=1;

    check_for_supported_file_systems;
    [ $? -eq 1 ] && SOME_PROBLEM_FOUND=1;

    check_for_physical_memory; 
    [ $? -eq 1 ] && SOME_PROBLEM_FOUND=1;
     
    check_for_prerequisite_packages; 
    [ $? -eq 1 ] && SOME_PROBLEM_FOUND=1;

    check_for_problematic_packages; 
    [ $? -eq 1 ] && SOME_PROBLEM_FOUND=1;
	
    check_for_problematic_modules; 
    [ $? -eq 1 ] && SOME_PROBLEM_FOUND=1;
 
    check_for_platform_specific_misc_data;
    [ $? -eq 1 ] && SOME_PROBLEM_FOUND=1;
 
    if [ $SOME_PROBLEM_FOUND -eq 1 ]; then
	sc_postgen_fops
	echo "Some problem found during analysis."
	echo "Check the file $OUT_REPORT for details" 
	sc_del_temp_files
        exit $SOME_PROBLEM_FOUND
    fi	

    check_for_solidifier_conflicts; 
}

# Main function for normal or pre execution.  
sc_normal_execution () 
{
    _common_capabilities;
}

# Main function for post execution.
#sc_post_execution ()
#{
#    $CMD_PACKAGE_QUERY $S3CONTROL>/dev/null
#    if [ $? -ne 0 ]; then
#	echo "McAfee Solidifier product is not installed.";
#	echo "Please install the product before executing post script."
#	exit $SC_ERR_SOLIDIFIER_NOT_INSTALLED;
#    else
#	sc_normal_execution;
#    fi
#}

# function to delete temp files.
sc_del_temp_files ()
{
    rm -f  $SC_TARGET_DIR/*.info

    # Delete the target dir if it is empty.
    FILE_COUNT=`ls -a $SC_TARGET_DIR | wc -l`
    if [ $FILE_COUNT = 2 ];then
	rm -rf $SC_TARGET_DIR
    fi
}

# cleanup function to be called in case of `Ctrl + C'
sc_cleanup ()
{
    echo "";
    [ -d "$SC_TARGET_DIR" ] && rm -rf $SC_TARGET_DIR;
    [ -f "$SC_TMPFILE" ] && rm -f $SC_TMPFILE;

    [ "$1" != "1" ] && exit 0;
}
trap "sc_cleanup" 2

# ****************************************************************************
# Script execution starts here.
# ****************************************************************************

argv="$*"
newargv=' '
for v in $argv; do
	if [ "$v" != "-q" ]; then
		newargv="$newargv $v"
	fi
done
_contains "-n" $@
if [ $? -eq 1 ]; then
    NO_TIME_STAMP=1
else
    NO_TIME_STAMP=0
fi

# parse the command line options
popt=0
bopt=0
QOPT=0
SC_IS_ARGUMENT_FILE_NAME=0
SC_INSTALL_DIR="/usr/local"

while [ $# -gt 0 ]
do
    case "$1" in
    -h|--help) 
	usage ;; 
   -v|--version)
        SC_SCANALYZER_VERSION="6.6.2"
	sc_version; exit 0; ;; 
   -b) 
	if [ $bopt -ne 0 ]; then
		echo "ERROR: -b option can not be used multiple times"
		exit $SC_ERR_TOO_MANY_ARGUMENTS;
	fi
 	bopt=1
 	if [ -z "$2" ]; then 
		echo "-b option requires a blacklist file as argument."
		exit -1
	else
		BLIST=$2;shift
	fi 
	;;
#    -p)
#        if [ $popt -ne 0 ]; then
#                echo "ERROR: -p option can not be used multiple times"
#                exit $SC_ERR_TOO_MANY_ARGUMENTS;
#        fi;
#        popt=1;
#        if [ -z "$2" ]; then
#                echo "-p option requires a pre or post argument."
#                echo "use -h option for 'help'"
#                exit 1
#        else
#	    if [ "$2" = "pre" ]; then
#		SC_PRE_MODE=1
#	    elif [ "$2" = "post" ]; then
#		SC_PRE_MODE=0
#	    else
#		echo "Invalid argument. $2 "
#		exit 1
#	    fi
#                RUNTIME=$2;shift
#        fi
#
#        ;;
   -o)
	if [ "$2" ]; then
		SC_IS_ARGUMENT_FILE_NAME=1
		USE_NAME="$2"
	else
  		SC_IS_ARGUMENT_FILE_NAME=0
		echo "ERROR: -o option needs a parameter to follow"
		exit $SC_ERR_TOO_LESS_ARGUMENTS;
	fi
	fopt=1
	FLIST=$2;shift;;

   -r)
   	if [ -d "$2" ]; then
	    	SC_INSTALL_DIR="$2"
	else
	    	echo "ERROR: specify a valid directory"
		exit $SC_ERR_TOO_LESS_ARGUEMENTS;
	fi
	shift;;
    	
   -d)
	if [ -z "$2" ] || [ -z "$3" ]; then
		echo "-cmp option requires the reports to compare as argument"
		echo "use -h for 'help'"
		exit 1
	fi
	SCAN_REPORT1=$2
	SCAN_REPORT2=$3
	sc_compare_reports
 	exit 0	
	;;

    -n) 
    	# keep this so that this option remains valid 
    	;;
    -q)
	 QOPT=1;
	 ;;

    *) echo "option \"$1\" is not supported"; sc_cleanup 1; usage 1;;
    esac
    shift 
done


define_variables
define_commands
check_for_available_disk_space "scananalyzer_log"

mkdir -p $SC_TARGET_DIR

if [ "$QOPT" = 0 ]; then
echo "+-----------------------------------------------------------------------+"
echo "				Starting ScAnalyzer"
echo "+-----------------------------------------------------------------------+"
fi

# validate for root user 
if [ $THIS_ARCH != "ppc" ]; then
    sc_validate_user
fi

# scan for blacklisted applications/installations
[ "$bopt" -eq "1" ] && scan_for_black_list $BLIST; 

check_for_solidifier_installation

sc_generate_report_header
if [ $SC_IS_SOLIDIFIER_INSTALLED -eq 0 ]; then
	sc_analyze_data_collected;
	echo "INFO: No problem found for McAfee Solidifier installation." >> $OUT_REPORT
fi

sc_normal_execution;

# remove the tool log file if it doesnt have anything in it.
[ ! -s "$LOGFILE" ] && rm -f $LOGFILE; 

# generate a single report 
sc_generate_report; 
sc_postgen_fops;

if [ $SC_REPORT_COMPRESSED -eq 0 ]; then
    echo "Collect the report $OUT_REPORT"
fi

sc_del_temp_files;

if [ "$QOPT" = 0 ]; then
echo "+-----------------------------------------------------------------------+"
fi
exit 0


