#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (c) 2016, Intel Corporation. All rights reserved.
# This file is licensed under the GPLv2 license.
# For the full content of this license, see the LICENSE.txt
# file at the top level of this source tree.

import cherrypy
import os
import pexpect
import codecs
import json
import rpm
import subprocess
from tools import logging_helper, data_ops, shell_ops, sysinfo_ops
import manage_config
import manage_repo
from manage_auth import require


# global variables used across multiple modules
constructed_packages_list = []  # only used by unit_test
constructed_packages_list_new = []  # only used by unit_test


def update_package_list():
    """ Grabs the curated.tar.gz file from download.01.org to update the curated package listings

    Returns:
        str: Json response with key 'status' and value 'success' or 'fail
    """
    log_helper = logging_helper.logging_helper.Logger()
    data_collector = sysinfo_ops.DataCollect()

    # Determine architecture and proper repository
    config = manage_config.read_config_file()
    developer_hub_url = config.get('DefaultRepo', 'base_repo')
    architecture, rcpl_version = data_collector.platform_details()
    developer_hub_url = developer_hub_url + rcpl_version + '/' + architecture
    curated_url = developer_hub_url + '/' + 'curated.xml.gz'
    local_path = '/tmp/curated.xml.gz'
    local_file = 'curated.txt'

    # Download and decompress the curated list
    # todo: this needs to return 'False' on timeout and give a json status of 'fail'
    shell_ops.run_command('timeout 5 wget %s -O %s' % (curated_url, local_path))
    data_ops.uncompress(local_path, local_file)
    build_package_database()

    # Remove tar file after use
    try:
        os.remove(local_path)
    except:  # todo: This needs to throw an error. Try 'except (OSError, IOError):'
        pass

    # From the UI if json == null then the response failed (timed out)
    response = ({
        'status': 'success'
    })
    response = json.dumps(response)
    log_helper.logger.debug("Finished updating package list: '%s'" % response)
    return response


def get_is_installed(process, package_name):
    """ Check if a specified package is installed.

    Args:
        process (Optional[subprocess]): (Optional) Previously opened shell subprocess.
                               None type object if a shell is not already open.
        package_name (str): Check if this package is installed

    Returns:
        tuple: 1st is True/False for installed or not. 2nd is dict with 'package' and 'installed' keys.
    """
    installed = False
    if process is None:
        if package_name in get_installed_packages(None):
            installed = True
    else:
        if package_name in get_installed_packages(process):
            installed = True

    response = {'package': package_name, 'installed': str(installed)}
    return installed, response


def get_installed_packages(process):
    """ Get a list of the installed packages from the smart --installed command

    Args:
        process (Optional[subprocess]): (Optional) shell subprocess. None type object if a shell is not already open.

    Returns:
        list: list of string.
    """
    if manage_config.use_new_get_installed_packages:
        my_list, my_dict = get_installed_packages_new()
        return my_list
    else:
        return get_installed_packages_original(process)


def get_installed_packages_original(process):
    """ Get a list of the installed packages from the smart --installed command

    Args:
        process (Optional[subprocess]): (Optional) shell subprocess. None type object if a shell is not already open.

    Returns:
        list: list of string.
    """
    if process is None:
        installed_packages = []
        result = shell_ops.run_command("smart query --installed --show-format=$name|")
        for line in result.split('|'):
            installed_packages.append(line)
        return installed_packages
    else:
        process.sendline('query --installed --show-format=$name|')
        process.expect('smart> ')
        return process.before.split('|')


def get_installed_packages_new():
    """ Get a list of the installed packages from rpm module

    Returns:
        tuple: 1st is list of installed package names. 2nd is dict.
    """
    dict_installed_packages = {}
    installed_packages = []
    log_helper = logging_helper.logging_helper.Logger()

    try:
        ts = rpm.TransactionSet()
        mi = ts.dbMatch()
    except Exception as e:
        log_helper.logger.error(str(e))
        return installed_packages, dict_installed_packages

    for h in mi:
        try:
            name = h['name']
            dict_installed_packages[name] = h['version'] + '-' + h['release']
            installed_packages.append(name)
        except Exception as e:
            log_helper.logger.error(str(e))
            continue
    return installed_packages, dict_installed_packages


def build_package_database():
    """ Parses curated and non-curated packages and places them into a json format.

    The json formatted string is saved to a file.
    """
    if manage_config.use_new_build_package_database:
        build_package_database_new()
    else:
        build_package_database_original()


def build_package_database_original():
    """ Parses curated and non-curated packages and places them into a json format.

    The json formatted string is saved to a file.
    """
    global constructed_packages_list
    constructed_packages_list = []

    log_helper = logging_helper.logging_helper.Logger()
    log_helper.logger.debug("Starting Build...")
   
    try:
        process = pexpect.spawn('smart --shell', timeout=240)
        process.expect('smart> ', timeout=240)
    except Exception as e:
        log_helper.logger.error(str(e))
        log_helper.logger.error("System is overloaded, can't create package list")
        return   

    data = []
    packages_added = []
    upgrade_list = []
    installed_packages = get_installed_packages(process)

    try:
        process.sendline('newer')
        process.expect('smart> ', timeout=240)
        upgrade_output = process.before
        if 'No interesting upgrades' not in upgrade_output:
            upgrade_output = upgrade_output[upgrade_output.rindex('---') + 3:]
            for line in upgrade_output.split('\n'):
                if len(line) < 5:
                    continue
                info = line.split('|')
                upgrade_info = {'name': info[0].strip(),
                                'installed_version': info[1].split(' ')[1],
                                'upgrade_version': info[2].split(' ')[1],
                                'upgrade_size': info[4].strip()}
                upgrade_list.append(upgrade_info)
            log_helper.logger.debug("Package upgrade list: '%s" % str(upgrade_list))
    except Exception as e:
        log_helper.logger.error(str(e))
        log_helper.logger.error("System is overloaded, could not get newer package list")
        pass
    
    # Get the info for curated packages
    filepath = os.path.dirname(os.path.realpath(__file__))
    with codecs.open(filepath + '/' + 'curated.txt', 'r') as my_file:
        text = json.loads(my_file.read())
        file_info = data_ops.byteify(text)
        for package in file_info:
            command = "query %s --show-format='$version|$group|'" % package['name']                 
            try:
                process.sendline(command) 
                process.expect('smart> ', timeout=240)
            except Exception as e:
                log_helper.logger.error(str(e))
                continue
            result = process.before
            result = result[result.index(command)+len(command):]
            sections = result.replace('\r\n', '').split('|')
            if len(sections) < 2:
                continue
            package['version'] = sections[0][:sections[0].index('@')]
            for item in upgrade_list:
                if package['name'] == item['name']:
                    package['version'] = item['installed_version']
            package['group'] = sections[1]
            package['curated'] = True
            package['installed'] = True if package['name'] in installed_packages else False

            packages_added.append(package['name'])
            data.append(package)

    # Get the info for non-curated packages
    for channel in manage_repo.list_repos():
        log_helper.logger.debug("Checking packages on channel: '%s'" % channel)
        command = "query --channel=%s --show-format='$name|$version|$summary|$group}'" % channel
        try:
            process.sendline(command)
            process.expect('smart> ', timeout=240)
        except Exception as e:
            log_helper.logger.error(str(e))
            continue
                
        result = process.before.split('}')
        result = result[1:]  # Skip the command in the result
        for line in result:
            sections = line.split('|')
            if len(sections) < 3:
                continue
            if 'query --channel' in sections[0]:
                continue
            package_name = sections[0].replace('\r\n', '')
            package_name = package_name.replace("'", "")
            if package_name in packages_added:
                continue  # Package is already curated

            # escape special JSON charater (") if any in description and summary
            sections[2] = sections[2].replace('"', '\\"')
            package = {
                'name': package_name,
                'version': sections[1][:sections[1].index('@')],
                'summary': sections[2],
                'group': sections[3],
                'image': 'packages.png',  # Default no icon
                'title': package_name.replace('-', ' ').title(),
                'installed': True if package_name in installed_packages else False,
                'upgrade_version': '',
                'curated': False,
                'depends': '',
                'bundle': '',
                'vertical': '',
                'service': '',
                'launch': '',
            }

            # Get the upgrade/install info
            for item in upgrade_list:
                if item['name'] == package_name:
                    package['upgrade_version'] = item['upgrade_version']
                    package['version'] = item['installed_version']
            package['installed'] = True if package_name in installed_packages else False

            data.append(package)

    constructed_packages_list = data

    # Output file with list of curated packages with additional info added
    with open('/tmp/' + 'json_packages.txt', 'w') as my_file:
        my_file.write(json.dumps(data))
    log_helper.logger.debug("Finished building package database. Output written to /tmp/json_packages.txt")
    process.close()


def build_package_database_new():
    """ Parses curated and non-curated packages and places them into a json format.

    The requirements of using this is to make sure that smart update is called to update cache,
    when add/remove repo is done.

    The json formatted string is saved to a file.
    """
    global constructed_packages_list_new
    constructed_packages_list_new = []
    data = []
    curated_packages = []
    curated_dict = {}
    upgrade_dict = {}
    query_result = ''
    packages_added_dict = {}

    log_helper = logging_helper.logging_helper.Logger()
    log_helper.logger.debug("Starting Build...")

    # Get the latest installed packages list
    my_list, my_dict = get_installed_packages_new()

    # Get the info for curated packages
    try:
        file_path = os.path.dirname(os.path.realpath(__file__))
        my_file = codecs.open(file_path + '/' + 'curated.txt', 'r')
        curated_packages = json.loads(my_file.read())  # list of json
        my_file.close()
    except Exception as e:
        log_helper.logger.error('Read curated.txt failed with ' + str(e))

    # Create a list of dict for curated packages, this can be used later..... dict key checking is
    # more efficient (due to hash table) than linear loop search
    for pc in curated_packages:
        try:
            curated_dict[pc['name']] = {'image': pc['image'], 'title': pc['title'],
                                        'summary': pc['summary'], 'url': pc['url'],
                                        'description': pc['description'], 'vertical': pc['vertical'],
                                        'service': pc['service'], 'launch': pc['launch']}
        except Exception as e:
            log_helper.logger.error(str(e) + ' for ' + pc['name'])
            continue

    # Get channel list
    list_channels_string = manage_repo.list_repos()
    list_query_args = []
    if list_channels_string:  # not empty
        for channel in list_channels_string:
            list_query_args.append('--channel=' + channel)
        list_query_args.append('--show-format=$name#myinfo#$version#myinfo#$summary#myinfo#$group#myline#')

        # Use Smart module directly to run smart

        # Prepare for args
        commands_list = ['newer', 'query']
        args_list = [[], list_query_args]

        # Run the commands
        p = subprocess.Popen(['python', 'smart_ops.py', str(commands_list), str(args_list)],
                             cwd='tools',
                             stdout=subprocess.PIPE,
                             stderr=subprocess.STDOUT)
        buffer_out = ""
        for line in iter(p.stdout.readline, ''):
            buffer_out += line
        if 'Error For Smart_ops.py' in buffer_out:
            log_helper.logger.error('Smart.ops.py running failed:  ' + str(buffer_out))
        else:
            results_list = buffer_out.split('#smart_opts_list#')
            if len(results_list) == 2:
                # Get upgrade list
                upgrade_output = results_list[0]
                if 'No interesting upgrades' not in upgrade_output and upgrade_output != '':
                    upgrade_output = upgrade_output[upgrade_output.rindex('---') + 3:]
                    for line in upgrade_output.split('\n'):
                        if len(line) < 5:
                            continue
                        info = line.split('|')
                        str_name = info[0].strip()
                        upgrade_dict[str_name] = {'name': str_name,
                                                  'installed_version': info[1].split(' ')[1],
                                                  'upgrade_version': info[2].split(' ')[1],
                                                  'upgrade_size': info[4].strip()}
                log_helper.logger.debug("Package upgrade list: '%s" % str(upgrade_dict))

                # Get packages
                query_result = results_list[1]
            else:
                log_helper.logger.error('Results do not have 2 items...' + str(len(results_list)))
    else:  # empty channel
        pass

    # loop through each package
    list_query_result = query_result.split('#myline#')
    for current_package in list_query_result:
        if current_package == '\n' or current_package == '\n\n' or current_package == '':  # safe guard the last entry
            continue
        else:
            package_info = current_package.split('#myinfo#')
            if not (len(package_info) == 4):
                log_helper.logger.error(current_package + " does not have current format to be parsed!")
                continue

        # get package information
        str_name = package_info[0]
        str_version = package_info[1]
        str_summary = package_info[2]
        str_group = package_info[3]

        # check if package is already in the dict
        already_added = (str_name in packages_added_dict)

        # check if package is in installed list
        installed = False
        install_version = ''
        if str_name in my_dict:
            installed = True
            install_version = my_dict[str_name]

        # check if package has upgrade/update or not
        has_upgrade = False
        if str_name in upgrade_dict:
            has_upgrade = True

        package = {'name': str_name,
                   'version': str_version[:str_version.index('@')],
                   'summary': str_summary,
                   'group': str_group,
                   'image': 'packages.png',  # Default no icon
                   'title': str_name.replace('-', ' ').title(),
                   'installed': installed,
                   'curated': False,
                   'vertical': '',
                   'service': '',
                   'launch': ''
                   }

        # check if package is in curated list
        if str_name in curated_dict:
            # print "Curated:  " + str_name + " installed: " + str(installed) + ' ' + install_version
            # Do not add duplicate ones
            if already_added:
                continue
            # Use the values in curated packages file
            curated_entry = curated_dict[str_name]
            package['curated'] = True
            package['image'] = curated_entry['image']
            package['title'] = curated_entry['title']
            package['summary'] = curated_entry['summary']
            package['url'] = curated_entry['url']
            package['description'] = curated_entry['description']
            package['vertical'] = curated_entry['vertical']
            package['service'] = curated_entry['service']
            package['launch'] = curated_entry['launch']
            if installed:
                package['version'] = install_version
                if has_upgrade:
                    package['upgrade_version'] = upgrade_dict[str_name]['upgrade_version']
            # Add this entry into the dict.
            packages_added_dict[str_name] = package
        else:
            # print "Non-curated:  " + str_name + " installed: " + str(installed) + ' ' + install_version
            # These fields are only for non-curated packages
            package['upgrade_version'] = ''
            package['depends'] = ''
            package['bundle'] = ''
            if already_added:  # Already an entry in the dict, only update if necessary.
                package_added = packages_added_dict[str_name]
                this_version_newer_than_recorded_one = data_ops.is_newer_version(package['version'],
                                                                                 package_added['version'])
                if not installed:  # Not installed, do not need to check upgrade.
                    if this_version_newer_than_recorded_one:
                        # Update entry
                        packages_added_dict[str_name]['version'] = package['version']
            else:  # Not in the dict yet. Add the entry to dict.
                if installed:  # Need to check upgrade
                    if has_upgrade:
                        package['upgrade_version'] = upgrade_dict[str_name]['upgrade_version']
                        package['version'] = upgrade_dict[str_name]['installed_version']
                # Add this entry into the dict.
                packages_added_dict[str_name] = package

    # Change dict to list
    for key in packages_added_dict:
        data.append(packages_added_dict[key])

    constructed_packages_list_new = data

    # Output file with list of curated packages with additional info added
    with open('/tmp/' + 'json_packages.txt', 'w') as my_file:
        my_file.write(json.dumps(data))
    log_helper.logger.debug("Finished building package database. Output written to /tmp/json_packages.txt")


def get_package_info(package_name):
    """ Get additional package info from the 'smart info' command.

    Args:
        package_name (str): String name of RPM

    Returns:
        str: In string format, dictionary with keys: summary, url, license, size, description, group, version.
    """
    log_helper = logging_helper.logging_helper.Logger()
    log_helper.logger.debug("Getting additional package info for %s" % package_name)
    command = "smart info " + package_name
    output = shell_ops.run_command(command)
    description = ''
    if output.count('Name:') > 1:
        # Multiple versions available. Narrow down smart info scope to get accurate info for the current version
        response = shell_ops.run_command("smart query --installed " + package_name + " --show-format=$version")
        version = response[response.index('[100%]')+6:response.index('@')].replace('\n', '')
        if 'not' in version:  # Workaround for "(not installed)" case
            version = 'Unknown'

        output = output[output.rindex(version):]

        if 'Name' in output:
            if output.index('Name') > output.index('Description'):
                # Additional entry after description
                description = output[output.rindex("Description:") + 14: output.index("Name")].replace('\n', '').strip()
        else:
            description = output[output.rindex("Description:") + 14:].replace('\n', '').strip()
    else:
        version = output[output.index("Version:") + 9: output.index("Priority:")].replace('\n', '')
        version = version[:version.index('@')]
        if 'not' in version:  # Workaround for "(not installed)" case
            version = 'Unknown'
        description = output[output.rindex("Description:") + 14:].replace('\n', '').strip()

    url = output[output.index("Reference URLs:") + 16: output.index("Flags:")].replace('\n', '')
    my_license = output[output.index("License:") + 9: output.index("Installed Size:")].replace('\n', '')
    size = output[output.index("Installed Size:") + 16: output.index("Reference URLs:")].replace('\n', '')
    group = output[output.index("Group:") + 7: output.index("License:")].replace('\n', '')
    summary = output[output.index("Summary:") + 9: output.index("Description:")].replace('\​r\n', '')

    # escape special JSON charater (") if any in description and summary
    summary = summary.replace('"', '\\"')
    description = description.replace('"', '\\"')

    package = {
        'url': url,
        'license': my_license,
        'size': size,
        'description': description,
        'summary': summary,
        'group': group,
        'version': version
    }
    log_helper.logger.debug("Returning package info: " + str(package))
    return json.dumps(package)


def get_data():
    """ Get the data from the build_package_database function.

    Returns:
        str: File contents if the file exists or None if it does not exist.
    """
    log_helper = logging_helper.logging_helper.Logger()
    try:
        my_file = open('/tmp/' + 'json_packages.txt', 'r')
        output = my_file.read().decode('string_escape')
        my_file.close()
        return output
    except IOError:
        log_helper.logger.debug("Database does not exist. Building database...")
        build_package_database()
        return None


def set_signature_verification_status(status):
    log_helper = logging_helper.logging_helper.Logger()
    log_helper.logger.debug("Smart config set to: rpm-check-signatures='%s'" % str(status))
    shell_ops.run_command("smart config --set rpm-check-signatures=" + str(status))


def package_transaction(command_type, package):
    """ Package management: handle smart calls

    Args:
        command_type (str): String 'install', 'remove', 'upgrade
        package (dict): String name of rpm package

    Returns:
        str: In string format, Json array with 'status' and 'error' (only if there was an error)
    """
    log_helper = logging_helper.logging_helper.Logger()
    command = ''
    pkg = ''

    # Halt unauthorized commands
    if command_type != "install" and command_type != "remove" \
            and command_type != "upgrade":
        return

    signature_status = ""

    try:
        signature_status = package['rpm']
        if signature_status == "untrusted":
            # untrusted RPM request
            # update smart config to install untrusted package
            set_signature_verification_status(False)
    except:
        pass

    if type(package) is dict:
        pkg = package['package']
        if pkg == "all":
            command = "smart " + command_type + " -y "
        else:
            command = "smart " + command_type + " -y " + pkg

    result = shell_ops.run_cmd_chk(command)
    log_helper.logger.debug("Ran command '%s' with returncode of '%s' and return of '%s'" % (command, result['returncode'], result['cmd_output']))
    response = parse_package_installation_result(pkg_name=pkg, result_dict=result)

    if signature_status == "untrusted":
        # reset signature verification
        set_signature_verification_status(True)

    if result['returncode'] == 0:
        # package list is updated. Recreate pacakge database
        build_package_database()

    log_helper.logger.debug('Return from package_transaction')
    return json.dumps(response)


def parse_package_installation_result(pkg_name, result_dict):
    """ Parse the result of Smart's package installation
    Args:
        pkg_name (str): package name
        result_dict (dict): the result of using shell_ops.run_cmd_chk to run smart package install.

    Returns:
        dict: dict of the result. The key is 'status'. If the 'status' is not 'success', then an extra key 'error'
              describes the error message.

    """
    response = ({'status': "success"})

    if ('returncode' in result_dict) and ('cmd_output' in result_dict):
        if result_dict['returncode']:
            if "error:" in result_dict['cmd_output']:
                # User clicked install/uninstall/upgrade then refreshed and page and hit it again
                if "Configuration is in readonly mode" in result_dict['cmd_output']:  # Smart shell already open
                    error = "For " + pkg_name + ", please wait for background processes to finish then try again."
                    status = "fail"
                elif "no package provides" in result_dict['cmd_output']:
                    error = "The dependencies for '" + pkg_name + "' could not be found."
                    status = "fail"
                elif "matches no packages" in result_dict['cmd_output']:
                    error = "The package '" + pkg_name + "' could not be found in any repositories that have been added. Please check your network configuration and repositories list on the Administration page."
                    status = "fail"
                elif "not customer signed" in result_dict['cmd_output']:
                    error = "The package '" + pkg_name + "' is untrusted. Do you want to install untrusted package?"
                    status = "untrusted"
                elif "package is not signed" in result_dict['cmd_output']:
                    error = "The package '" + pkg_name + "' is untrusted. Do you want to install untrusted package?"
                    status = "untrusted"
                else:
                    error = "For " + pkg_name + ", "
                    error += result_dict['cmd_output'][result_dict['cmd_output'].index("error:") + 7:].replace("\n", "")
                    status = "fail"
                response = ({
                    'status': status,
                    'error': error
                })

    return response


@require()
class Packages(object):
    exposed = True

    def GET(self, **kwargs):
        return get_data()

    def POST(self, **kwargs):
        return package_transaction("install", kwargs)

    def PUT(self, **kwargs):
        return package_transaction("upgrade", kwargs)

    def DELETE(self, **kwargs):
        return package_transaction("remove", kwargs)


@require()
class PackageInfo(object):
    exposed = True

    def GET(self, **kwargs):
        return get_package_info(kwargs['name'])
