# Copyright (c) 2016, Intel Corporation. All rights reserved.
# This file is licensed under the GPLv2 license.
# For the full content of this license, see the LICENSE.txt
# file at the top level of this source tree.

import cherrypy
import subprocess
import os
import json
from tools import logging_helper, network_ops, shell_ops, sysinfo_ops, rcpl
import manage_config
import manage_repo
import manage_os_update
import manage_package
from manage_auth import require


class Pro_Status(object):

    def __init__(self):
        self.config = manage_config.read_config_file()
        self.config_file = os.path.dirname(__file__) + '/' + 'developer_hub_config'
        self.pro_status = {'result': None}

    def enabled_state(self):
        status = self.config.get('ProRepo', 'enable_pro')
        self.pro_status['result'] = status 
        return self.pro_status 

    def upgrade_status(self):
        self.config.set('ProRepo', 'enable_pro', 'True')
        set_config = open(self.config_file, 'w')
        self.config.write(set_config)


class Pro_Repo(object):
    
    def __init__(self):
        self.__xml_file_url = str("Not Set")
        self.__xml_location = 'pro_directory.xml'
        self.__repo_number = 0
        self.__repo_list = []
        self.__current_repos = manage_repo.list_repos()
        self.__package_list = []
        self.__handler = None
        self.__rcpl_fname = str("pro_directory.xml")
        self.__log_helper = logging_helper.logging_helper.Logger()
        self.__network_checker = network_ops.NetworkCheck()
        self.__error = None
        self.__available_rcpls = None

    def remove_pro_repos(self, repo_list):
        try:
            removed = False
            for repo in repo_list:
                if 'Pro_Repo_' in repo:
                    self.__log_helper.logger.debug('Removing old repository: ' + str(repo))
                    shell_ops.run_command("smart channel --remove " + repo + " -y")
                    removed = True
            if removed:
                shell_ops.run_command("smart update")
        except:
            pass

    def processInfo(self, current_rcpl):
        try:
            self.__handler = rcpl.file_io.handler.RCPLINDEX_READER(self.__rcpl_fname, False, False, None)
            self.__handler.load_index()
            self.__available_rcpls = self.__handler.getData()
            repo_urls = self.__available_rcpls[current_rcpl]['files']
            return repo_urls
        except Exception as err:
            self.__error = err
            self.__log_helper.logger.error(str(self.__error))

    def update_repo(self, user_name, password, current_rcpl):

        result_dict = {}

        # Check if pro repositories are setup and remove
        self.remove_pro_repos(self.__current_repos)

        # Check RCPL version, source WR xml
        try:
            os.remove(self.__xml_location)
        except:
            pass

        net_con, _ = self.__network_checker.test_network_connection()
        if net_con['https_conn'] == 'False':
            result_dict['repo_update'] = '404'
            return result_dict
        
        config = manage_config.read_config_file()
        self.__xml_file_url = config.get('ProRepo', sysinfo_ops.arch)
        out = subprocess.check_output(["curl", "-u", "%s:%s" % (user_name, password), "%s" % self.__xml_file_url])
        if 'Username or password does not match' in out:
            result_dict['repo_update'] = 'Error 401 - Unauthorized: Access is denied due to invalid credentials.'
            return result_dict
        xml_file = open(self.__xml_location, "w")
        xml_file.write(out)
        xml_file.close()

        # Aquire list of repos
        repo_urls = self.processInfo(current_rcpl)

        # Add pro repositories and update
        try:
            for url in repo_urls:
                self.__repo_number += 1
                repo_name = "Pro_Repo_" + str(self.__repo_number)
                self.__log_helper.logger.debug('Adding new repository: ' + str(repo_name))
                user_name = user_name.replace("@", "%40")
                url = url.replace("://", "://%s:%s@" % (user_name, password))
                shell_ops.run_command("smart channel --add " + repo_name + " type=rpm-md baseurl=" + url + " -y")
                self.__repo_list.append(repo_name)
            channel_output = shell_ops.run_command("smart update")
            if 'error: 401' in channel_output:
                self.remove_pro_repos(repo_urls)
                result_dict['repo_update'] = '401'
                return result_dict
            result_dict['repo_update'] = 'complete'
            return result_dict
        except:
            result_dict['repo_update'] = 'Failed to Update'
            return result_dict
            
    def os_upgrade(self, user_name, password, pro_status, current_rcpl):
        result_dict = {}
        flex_repo_URL = []
        str_status_failure = 'fail'
        str_status_success = 'success'
        str_status_key = 'status'
        str_error_key = 'error'

        # Add Flex Repo
        flex_repo = manage_os_update.OS_UPDATER(sysinfo_ops.rcpl_version, sysinfo_ops.arch)
        flex_repo.getIndex()
        flex_repo.processInfo()
        flex_repo.getCurrent()
        flex_repo.updateCurrent('remove')
        flex_result = flex_repo.updateCurrent('add')
        if flex_result['status'] == 'failure':
            flex_repo.updateCurrent('remove')
            result_dict[str_status_key] = str_status_failure
            result_dict[str_error_key] = 'Failed adding Flex repositories.'
            return pro_status, result_dict

        # Update Repos
        temp_result = self.update_repo(user_name, password, current_rcpl)
        if temp_result['repo_update'] is '401' or temp_result['repo_update'] is '404' or temp_result['repo_update'] is 'Failed to Update':
            flex_repo.updateCurrent('remove')
            result_dict[str_status_key] = str_status_failure
            result_dict[str_error_key] = temp_result['repo_update']
            return pro_status, result_dict

        # Check available packages and build install list
        # Avoid using pexpect to run smart.
        list_query_args = []
        list_query_result = []
        for channel in self.__repo_list:
            list_query_args.append('--channel=' + channel)
        if list_query_args:  # non-empty list
            list_query_args.append('--show-format=$name#myline#')
            commands_list = ['query']
            args_list = [list_query_args]
            p = subprocess.Popen(['python', 'smart_ops.py', str(commands_list), str(args_list)],
                                 cwd='tools',
                                 stdout=subprocess.PIPE,
                                 stderr=subprocess.STDOUT)
            buffer_out = ""
            for line in iter(p.stdout.readline, ''):
                buffer_out += line
            if 'Error For Smart_ops.py' in buffer_out:
                flex_repo.updateCurrent('remove')
                self.__log_helper.logger.error('Smart.ops.py running failed:  ' + str(buffer_out))
                result_dict[str_status_key] = str_status_failure
                result_dict[str_error_key] = 'Failed building package list. ' + str(buffer_out)
                return pro_status, result_dict
            else:
                list_query_result = buffer_out.split('#myline#')
                if list_query_result:  # non-empty list
                    last_entry = list_query_result[-1]
                    if last_entry == '\n' or last_entry == '\n\n' or last_entry == '':  # safe guard the last entry
                        list_query_result.pop()  # remove the last entry

        # Remove debug, development and document RPMs from install list
        self.__package_list = list_query_result
        install_list = [i for i in self.__package_list if not ('-dbg' in i or '-dev' in i or '-doc' in i)]
        self.__log_helper.logger.debug('Install list: ' + str(install_list))
        str_error = ''
        has_failure = False
        for package in install_list:
            self.__log_helper.logger.debug('Installing ' + str(package))
            try:
                command = "smart install -y " + package
                result = shell_ops.run_cmd_chk(command)
                response = manage_package.parse_package_installation_result(pkg_name=package, result_dict=result)
                if 'status' in response:
                    if (response['status'] != 'success') and ('error' in response):
                        has_failure = True
                        str_error += response['error']
                        str_error += ' '
            except Exception as e:
                has_failure = True
                str_error += ("For " + package + ", " + str(e))
                str_error += ' '
                pass

        if has_failure:
            self.__log_helper.logger.error("Package installation failed:  " + str_error)
            result_dict[str_status_key] = str_status_failure
            result_dict[str_error_key] = str_error
        else:
            result_dict[str_status_key] = str_status_success
            result_dict['repo_update'] = str_status_success  # this line is just to make it work with old front end.
            self.__log_helper.logger.debug(str(result_dict))
            # Update config file
            # The following 3 lines need to be together.
            config = Pro_Status()
            config.upgrade_status()
            pro_status['result'] = True

        # Remove Flex Repos
        flex_repo.updateCurrent('remove')

        # Rebuild package database, now that we have pro repos.
        manage_package.build_package_database()

        # Update GUI
        return pro_status, result_dict


@require()
class EnablePro(object):
    exposed = True

    def __init__(self):
        self.__network_checker = network_ops.NetworkCheck()
        self.__pro_status = {'result': 'na'}

    def GET(self, **kwargs):
        if self.__pro_status['result'] == 'na':
            config = Pro_Status()
            self.__pro_status = config.enabled_state()
        return json.dumps(self.__pro_status)

    def POST(self, **kwargs):
        # Increased timeout value (12 hours) for quark
        cherrypy.response.timeout = 43200
        header_cl = cherrypy.request.headers['Content-Length']
        cl_content = cherrypy.request.body.read(int(header_cl))
        kwargs = json.loads(cl_content)
        net_test = self.__network_checker.test_network_connection()[0]
        if net_test['https_conn'] == 'False':
            return json.dumps(net_test)
        flex_to_pro = Pro_Repo()
        self.__pro_status, status = flex_to_pro.os_upgrade(kwargs['username'], kwargs['password'],
                                                           self.__pro_status, sysinfo_ops.rcpl_version)
        return json.dumps(status)

    def PUT(self, **kwargs):
        # Increased timeout value (12 hours) for quark
        cherrypy.response.timeout = 43200
        header_cl = cherrypy.request.headers['Content-Length']
        cl_content = cherrypy.request.body.read(int(header_cl))
        kwargs = json.loads(cl_content)
        net_test = self.__network_checker.test_network_connection()[0]
        if net_test['https_conn'] == 'False':
            return json.dumps(net_test)
        updater = manage_os_update.OS_UPDATER(sysinfo_ops.rcpl_version, sysinfo_ops.arch)
        updater.getIndex()
        updater.processInfo()
        _, update_to_version = updater.findNext()
        flex_to_pro = Pro_Repo()
        status = flex_to_pro.update_repo(kwargs['username'], kwargs['password'], str(update_to_version))
        return json.dumps(status)

