#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (c) 2016, Intel Corporation. All rights reserved.
# This file is licensed under the GPLv2 license.
# For the full content of this license, see the LICENSE.txt
# file at the top level of this source tree.

from tools import logging_helper, shell_ops, network_ops
from manage_auth import require
import json
import os
import manage_config
import manage_repo


def set_proxy_config(http_url, http_port, https_url, https_port, ftp_url,
                     ftp_port, socks_url, socks_port, no_proxy):
    """ Writes the new proxy configuration to the /etc/environment file without overriding other environment configurations.

    Args:
        http_url (str):
        http_port (str):
        https_url (str):
        https_port (str):
        ftp_url (str):
        ftp_port (str):
        socks_url (str):
        socks_port (str):
        no_proxy (str): comma separated

    Returns:
        str: Json response with key 'status' and value 'success' if no error was encountered.
    """
    log_helper = logging_helper.logging_helper.Logger()
    network_checker = network_ops.NetworkCheck()

    proxy_mass = ''
    http_proxy = ''
    https_proxy = ''
    if http_url != '' and http_port != '':
        http_proxy = http_url + ':' + http_port
        proxy_mass += 'http_proxy=' + http_proxy + '\n'
        os.environ["http_proxy"] = http_proxy
    else:
        os.environ["http_proxy"] = ''

    if https_url != '' and https_port != '':
        https_proxy = https_url + ':' + https_port
        proxy_mass += 'https_proxy=' + https_proxy + '\n'
        os.environ["https_proxy"] = https_proxy
    else:
        os.environ["https_proxy"] = ''

    if ftp_url != '' and ftp_port != '':
        ftp_proxy = ftp_url + ':' + ftp_port
        proxy_mass += 'ftp_proxy=' + ftp_proxy + '\n'
        os.environ["ftp_proxy"] = ftp_proxy
    else:
        os.environ["ftp_proxy"] = ''

    if socks_url != '' and socks_port != '':
        socks_proxy = socks_url + ':' + socks_port
        proxy_mass += 'socks_proxy=' + socks_proxy + '\n'
        os.environ["socks_proxy"] = socks_proxy
    else:
        os.environ["socks_proxy"] = ''

    if no_proxy != '':
        no_proxy = no_proxy
        proxy_mass += 'no_proxy=' + no_proxy + '\n'
        os.environ["no_proxy"] = no_proxy
    else:
        os.environ["no_proxy"] = ''

    # Add Java proxy support
    java_proxy = "_JAVA_OPTIONS='-Dhttp.proxyHost=%s -Dhttp.proxyPort=%s -Dhttps.proxyHost=%s -Dhttps.proxyPort=%s'" % (http_url, http_port, https_url, https_port)
    proxy_mass += java_proxy

    # Update proxy environment file
    update = open('/var/www/www-repo-gui/proxy_env', 'w+')
    update.write(proxy_mass)
    update.close()
    log_helper.logger.debug("New environment file: '%s'" % proxy_mass)

    # Confirm if NPM is installed and update proxy
    npm_check = os.path.isfile('/usr/bin/npm')
    if npm_check:
        if http_proxy != '':
            if '://' not in http_proxy:
                http_proxy = 'http://' + http_proxy
            shell_ops.run_command('npm config set proxy ' + http_proxy)
        else:
            shell_ops.run_command('npm config rm proxy')
        if https_proxy != '':
            if '://' not in https_proxy:
                https_proxy = 'https://' + https_proxy
            shell_ops.run_command('npm config set https-proxy ' + https_proxy)
        else:
            shell_ops.run_command('npm config rm https-proxy')

    # Set proxy for HDC
    if '://' in http_url:
        split_url = http_url.split('://')
        http_url = split_url[1]
    if http_url != '':
        manage_config.HDCSettings.set_proxy_settings_for_HDC("http", http_url, http_port)
    else:
        manage_config.HDCSettings.set_proxy_settings_for_HDC("none", "proxy.windriver.com", "3128")

    # Check network connections
    response = {'status': 'success', 'https_conn': 'False'}
    net_conn, net_resp = network_checker.test_network_connection(check_http=manage_config.network_check_http,
                                                                 no_rest_period=True)
    if net_conn['https_conn'] == "False" or net_conn['http_conn'] == 'False':
        log_helper.logger.debug("Proxy setting invalid: '%s'" % proxy_mass)
        manage_config.network_status['https_conn'] = net_conn['https_conn']
        manage_config.network_status['http_conn'] = net_conn['http_conn']
        return json.dumps(response)
    else:
        manage_config.network_status['https_conn'] = net_conn['https_conn']
        manage_config.network_status['http_conn'] = net_conn['http_conn']
        response['https_conn'] = 'True'

    # Update package list after proxy settings
    # And we do not need to check network again since we checked it already.
    manage_repo.update_channels(CheckNetworkAgain=False)

    # We do this at the end so that we don't run too many processes.
    # Restart Node-Red and WR-IOT-Agent for Proxy settings to take effect
    try:
        shell_ops.run_command('systemctl --no-block restart node-red-experience')
    except:
        log_helper.logger.debug("Unable to restart node-red service. Package may not be installed.")
        pass

    try:
        shell_ops.run_command('systemctl --no-block restart wr-iot-agent')
    except:
        log_helper.logger.debug("Unable to restart wr-iot-agent service. Package may not be installed.")
        pass

    log_helper.logger.debug("Finished setting proxy configuration.")
    return json.dumps(response)


def get_proxy_config():
    """ Reads the proxy configuration from the /etc/environment location

    Returns:
        str: String with all proxy information separated by line
    """
    log_helper = logging_helper.logging_helper.Logger()
    log_helper.logger.debug("Fetching proxy configuration...")
    http_url = http_port = https_url = https_port = ftp_url = ftp_port = socks_url = socks_port = no_proxy = ""
    with open("/var/www/www-repo-gui/proxy_env", "r") as my_file:
        content = my_file.read()
        for line in content.split("\n"):

            # Catch lines without content
            if len(line) < 13 or "#" in line:
                continue

            # no_proxy has no (url, port) pair to split
            if "no_proxy" in line:
                noproxy_split = line.split('=')
                no_proxy = noproxy_split[1]
                continue

            # Parse through sections for proxy types
            if '__JAVA_OPTIONS' not in line:
                sections = split_proxy_info(line)

            if "http_proxy" in line:
                http_url = sections[0]
                http_port = sections[1]
            if "https_proxy" in line:
                https_url = sections[0]
                https_port = sections[1]
            if "ftp_proxy" in line:
                ftp_url = sections[0]
                ftp_port = sections[1]
            if "socks_proxy" in line:
                socks_url = sections[0]
                socks_port = sections[1]

        response = ({
            'http_url': http_url,
            'http_port': http_port,
            'https_url': https_url,
            'https_port': https_port,
            'ftp_url': ftp_url,
            'ftp_port': ftp_port,
            'socks_url': socks_url,
            'socks_port': socks_port,
            'no_proxy': no_proxy
        })
        response = json.dumps(response)
        log_helper.logger.debug("Finished retrieving proxy configuration: '%s'" % response)
        return response


def split_proxy_info(string):
    """ Splits the proxy definition into 'url' and 'port'

    Args:
        string (str): String type with one line of proxy info

    Returns:
        tuple: url and port
    """
    pair = string.split('=')
    pair = pair[1].split(':')
    if len(pair) == 2:
        url = pair[0].strip('"')
        port = pair[1].strip('"')
        return url, port
    elif len(pair) == 3:
        url = pair[0].strip('"') + ':' + pair[1]
        port = pair[2].strip('"')
        return url, port


@require()
class Proxy(object):
    exposed = True

    def __init__(self):
        self.__network_checker = network_ops.NetworkCheck()

    def GET(self, **kwargs):
        if kwargs['request'] == "list":
            return get_proxy_config()
        elif kwargs['request'] == "test":
            return self.__network_checker.test_network_connection(check_http=manage_config.network_check_http)[1]

    def POST(self, **kwargs):
        return set_proxy_config(kwargs['http_url'], kwargs['http_port'], kwargs['https_url'],
                                kwargs['https_port'], kwargs['ftp_url'], kwargs['ftp_port'],
                                kwargs['socks_url'], kwargs['socks_port'], kwargs['no_proxy'])
